"use strict";
// tslint:disable: max-line-length
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const ssm = require("../lib");
module.exports = {
    'creating a String SSM Parameter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ssm.StringParameter(stack, 'Parameter', {
            allowedPattern: '.*',
            description: 'The value Foo',
            parameterName: 'FooParameter',
            stringValue: 'Foo',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            AllowedPattern: '.*',
            Description: 'The value Foo',
            Name: 'FooParameter',
            Type: 'String',
            Value: 'Foo',
        }));
        test.done();
    },
    'expect String SSM Parameter to have tier properly set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ssm.StringParameter(stack, 'Parameter', {
            allowedPattern: '.*',
            description: 'The value Foo',
            parameterName: 'FooParameter',
            stringValue: 'Foo',
            tier: ssm.ParameterTier.ADVANCED,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Tier: 'Advanced',
        }));
        test.done();
    },
    'String SSM Parameter rejects invalid values'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringParameter(stack, 'Parameter', { allowedPattern: '^Bar$', stringValue: 'FooBar' }), /does not match the specified allowedPattern/);
        test.done();
    },
    'String SSM Parameter allows unresolved tokens'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.doesNotThrow(() => {
            new ssm.StringParameter(stack, 'Parameter', {
                allowedPattern: '^Bar$',
                stringValue: cdk.Lazy.stringValue({ produce: () => 'Foo!' }),
            });
        });
        test.done();
    },
    'creating a StringList SSM Parameter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ssm.StringListParameter(stack, 'Parameter', {
            allowedPattern: '(Foo|Bar)',
            description: 'The values Foo and Bar',
            parameterName: 'FooParameter',
            stringListValue: ['Foo', 'Bar'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            AllowedPattern: '(Foo|Bar)',
            Description: 'The values Foo and Bar',
            Name: 'FooParameter',
            Type: 'StringList',
            Value: 'Foo,Bar',
        }));
        test.done();
    },
    'String SSM Parameter throws on long descriptions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ssm.StringParameter(stack, 'Parameter', {
                stringValue: 'Foo',
                description: '1024+ character long description: Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum.',
            });
        }, /Description cannot be longer than 1024 characters./);
        test.done();
    },
    'String SSM Parameter throws on long names'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ssm.StringParameter(stack, 'Parameter', {
                stringValue: 'Foo',
                parameterName: '2048+ character long name: Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum.',
            });
        }, /Name cannot be longer than 2048 characters./);
        test.done();
    },
    'StringList SSM Parameter throws on long descriptions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ssm.StringListParameter(stack, 'Parameter', {
                stringListValue: ['Foo', 'Bar'],
                description: '1024+ character long description: Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum.',
            });
        }, /Description cannot be longer than 1024 characters./);
        test.done();
    },
    'StringList SSM Parameter throws on long names'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ssm.StringListParameter(stack, 'Parameter', {
                stringListValue: ['Foo', 'Bar'],
                parameterName: '2048+ character long name: Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum.',
            });
        }, /Name cannot be longer than 2048 characters./);
        test.done();
    },
    'StringList SSM Parameter values cannot contain commas'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringListParameter(stack, 'Parameter', { stringListValue: ['Foo,Bar'] }), /cannot contain the ',' character/);
        test.done();
    },
    'StringList SSM Parameter rejects invalid values'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringListParameter(stack, 'Parameter', { allowedPattern: '^(Foo|Bar)$', stringListValue: ['Foo', 'FooBar'] }), /does not match the specified allowedPattern/);
        test.done();
    },
    'StringList SSM Parameter allows unresolved tokens'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.doesNotThrow(() => new ssm.StringListParameter(stack, 'Parameter', {
            allowedPattern: '^(Foo|Bar)$',
            stringListValue: ['Foo', cdk.Lazy.stringValue({ produce: () => 'Baz!' })],
        }));
        test.done();
    },
    'parameterArn is crafted correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const param = new ssm.StringParameter(stack, 'Parameter', { stringValue: 'Foo' });
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter/',
                    { Ref: 'Parameter9E1B4FBA' },
                ]],
        });
        test.done();
    },
    'parameterName that includes a "/" must be fully qualified (i.e. begin with "/") as well'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringParameter(stack, 'myParam', {
            stringValue: 'myValue',
            parameterName: 'path/to/parameter',
        }), /Parameter names must be fully qualified/);
        test.throws(() => new ssm.StringListParameter(stack, 'myParam2', {
            stringListValue: ['foo', 'bar'],
            parameterName: 'path/to/parameter2',
        }), /Parameter names must be fully qualified \(if they include \"\/\" they must also begin with a \"\/\"\)\: path\/to\/parameter2/);
        test.done();
    },
    'StringParameter.fromStringParameterName'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const param = ssm.StringParameter.fromStringParameterName(stack, 'MyParamName', 'MyParamName');
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter/MyParamName'
                ]],
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'String');
        test.deepEqual(stack.resolve(param.stringValue), { Ref: 'MyParamNameParameter' });
        assert_1.expect(stack).toMatch({
            Parameters: {
                MyParamNameParameter: {
                    Type: 'AWS::SSM::Parameter::Value<String>',
                    Default: 'MyParamName',
                },
            },
        });
        test.done();
    },
    'StringParameter.fromStringParameterAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const param = ssm.StringParameter.fromStringParameterAttributes(stack, 'MyParamName', {
            parameterName: 'MyParamName',
            version: 2,
        });
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter/MyParamName'
                ]],
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'String');
        test.deepEqual(stack.resolve(param.stringValue), '{{resolve:ssm:MyParamName:2}}');
        test.done();
    },
    'StringParameter.fromSecureStringParameterAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const param = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'MyParamName', {
            parameterName: 'MyParamName',
            version: 2,
        });
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter/MyParamName'
                ]],
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'SecureString');
        test.deepEqual(stack.resolve(param.stringValue), '{{resolve:ssm-secure:MyParamName:2}}');
        test.done();
    },
    'StringParameter.fromSecureStringParameterAttributes with encryption key creates the correct policy for grantRead'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = kms.Key.fromKeyArn(stack, 'CustomKey', 'arn:aws:kms:us-east-1:123456789012:key/xyz');
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // WHEN
        const param = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'MyParamName', {
            parameterName: 'MyParamName',
            version: 2,
            encryptionKey: key,
        });
        param.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'kms:Decrypt',
                        Effect: 'Allow',
                        Resource: 'arn:aws:kms:us-east-1:123456789012:key/xyz',
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/MyParamName',
                                ],
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'StringParameter.fromSecureStringParameterAttributes with encryption key creates the correct policy for grantWrite'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = kms.Key.fromKeyArn(stack, 'CustomKey', 'arn:aws:kms:us-east-1:123456789012:key/xyz');
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // WHEN
        const param = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'MyParamName', {
            parameterName: 'MyParamName',
            version: 2,
            encryptionKey: key,
        });
        param.grantWrite(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:kms:us-east-1:123456789012:key/xyz',
                    },
                    {
                        Action: 'ssm:PutParameter',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/MyParamName',
                                ],
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'StringListParameter.fromName'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const param = ssm.StringListParameter.fromStringListParameterName(stack, 'MyParamName', 'MyParamName');
        // THEN
        test.deepEqual(stack.resolve(param.parameterArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ssm:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':parameter/MyParamName'
                ]],
        });
        test.deepEqual(stack.resolve(param.parameterName), 'MyParamName');
        test.deepEqual(stack.resolve(param.parameterType), 'StringList');
        test.deepEqual(stack.resolve(param.stringListValue), { 'Fn::Split': [',', '{{resolve:ssm:MyParamName}}'] });
        test.done();
    },
    'fromLookup will use the SSM context provider to read value during synthesis'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'my-staq', { env: { region: 'us-east-1', account: '12344' } });
        // WHEN
        const value = ssm.StringParameter.valueFromLookup(stack, 'my-param-name');
        // THEN
        test.deepEqual(value, 'dummy-value-for-my-param-name');
        test.deepEqual(app.synth().manifest.missing, [
            {
                key: 'ssm:account=12344:parameterName=my-param-name:region=us-east-1',
                props: {
                    account: '12344',
                    region: 'us-east-1',
                    parameterName: 'my-param-name',
                },
                provider: 'ssm',
            },
        ]);
        test.done();
    },
    'valueForStringParameter': {
        'returns a token that represents the SSM parameter value'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const value = ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            // THEN
            assert_1.expect(stack).toMatch({
                Parameters: {
                    SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: 'AWS::SSM::Parameter::Value<String>',
                        Default: 'my-param-name',
                    },
                },
            });
            test.deepEqual(stack.resolve(value), { Ref: 'SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter' });
            test.done();
        },
        'de-dup based on parameter name'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name-2');
            ssm.StringParameter.valueForStringParameter(stack, 'my-param-name');
            // THEN
            assert_1.expect(stack).toMatch({
                Parameters: {
                    SsmParameterValuemyparamnameC96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: 'AWS::SSM::Parameter::Value<String>',
                        Default: 'my-param-name',
                    },
                    SsmParameterValuemyparamname2C96584B6F00A464EAD1953AFF4B05118Parameter: {
                        Type: 'AWS::SSM::Parameter::Value<String>',
                        Default: 'my-param-name-2',
                    },
                },
            });
            test.done();
        },
        'can query actual SSM Parameter Names, multiple times'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            ssm.StringParameter.valueForStringParameter(stack, '/my/param/name');
            ssm.StringParameter.valueForStringParameter(stack, '/my/param/name');
            test.done();
        },
    },
    'rendering of parameter arns'(test) {
        const stack = new cdk.Stack();
        const param = new cdk.CfnParameter(stack, 'param');
        const expectedA = { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/bam']] };
        const expectedB = { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'param' }]] };
        const expectedC = { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter', { Ref: 'param' }]] };
        let i = 0;
        // WHEN
        const case1 = ssm.StringParameter.fromStringParameterName(stack, `p${i++}`, 'bam');
        const case2 = ssm.StringParameter.fromStringParameterName(stack, `p${i++}`, '/bam');
        const case4 = ssm.StringParameter.fromStringParameterAttributes(stack, `p${i++}`, { parameterName: 'bam' });
        const case5 = ssm.StringParameter.fromStringParameterAttributes(stack, `p${i++}`, { parameterName: '/bam' });
        const case6 = ssm.StringParameter.fromStringParameterAttributes(stack, `p${i++}`, { parameterName: param.valueAsString, simpleName: true });
        const case7 = ssm.StringParameter.fromSecureStringParameterAttributes(stack, `p${i++}`, { parameterName: 'bam', version: 10 });
        const case8 = ssm.StringParameter.fromSecureStringParameterAttributes(stack, `p${i++}`, { parameterName: '/bam', version: 10 });
        const case9 = ssm.StringParameter.fromSecureStringParameterAttributes(stack, `p${i++}`, { parameterName: param.valueAsString, version: 10, simpleName: false });
        // auto-generated name is always generated as a "simple name" (not/a/path)
        const case10 = new ssm.StringParameter(stack, `p${i++}`, { stringValue: 'value' });
        // explicitly named physical name gives us a hint on how to render the ARN
        const case11 = new ssm.StringParameter(stack, `p${i++}`, { parameterName: '/foo/bar', stringValue: 'hello' });
        const case12 = new ssm.StringParameter(stack, `p${i++}`, { parameterName: 'simple-name', stringValue: 'hello' });
        const case13 = new ssm.StringListParameter(stack, `p${i++}`, { stringListValue: ['hello', 'world'] });
        const case14 = new ssm.StringListParameter(stack, `p${i++}`, { parameterName: '/not/simple', stringListValue: ['hello', 'world'] });
        const case15 = new ssm.StringListParameter(stack, `p${i++}`, { parameterName: 'simple', stringListValue: ['hello', 'world'] });
        // THEN
        test.deepEqual(stack.resolve(case1.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case2.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case4.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case5.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case6.parameterArn), expectedB);
        test.deepEqual(stack.resolve(case7.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case8.parameterArn), expectedA);
        test.deepEqual(stack.resolve(case9.parameterArn), expectedC);
        // new ssm.Parameters determine if "/" is needed based on the posture of `parameterName`.
        test.deepEqual(stack.resolve(case10.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'p81BB0F6FE' }]] });
        test.deepEqual(stack.resolve(case11.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter', { Ref: 'p97A508212' }]] });
        test.deepEqual(stack.resolve(case12.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'p107D6B8AB0' }]] });
        test.deepEqual(stack.resolve(case13.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'p118A9CB02C' }]] });
        test.deepEqual(stack.resolve(case14.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter', { Ref: 'p129BE4CE91' }]] });
        test.deepEqual(stack.resolve(case15.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'p1326A2AEC4' }]] });
        test.done();
    },
    'if parameterName is a token separator must be specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const param = new cdk.CfnParameter(stack, 'param');
        let i = 0;
        // WHEN
        const p1 = new ssm.StringParameter(stack, `p${i++}`, { parameterName: param.valueAsString, stringValue: 'foo', simpleName: true });
        const p2 = new ssm.StringParameter(stack, `p${i++}`, { parameterName: param.valueAsString, stringValue: 'foo', simpleName: false });
        const p3 = new ssm.StringListParameter(stack, `p${i++}`, { parameterName: param.valueAsString, stringListValue: ['foo'], simpleName: false });
        // THEN
        test.deepEqual(stack.resolve(p1.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter/', { Ref: 'p0B02A8F65' }]] });
        test.deepEqual(stack.resolve(p2.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter', { Ref: 'p1E43AD5AC' }]] });
        test.deepEqual(stack.resolve(p3.parameterArn), { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':ssm:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':parameter', { Ref: 'p2C1903AEB' }]] });
        test.done();
    },
    'fails if name is a token and no explicit separator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const param = new cdk.CfnParameter(stack, 'param');
        let i = 0;
        // THEN
        const expected = /Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly/;
        test.throws(() => ssm.StringParameter.fromStringParameterName(stack, `p${i++}`, param.valueAsString), expected);
        test.throws(() => ssm.StringParameter.fromSecureStringParameterAttributes(stack, `p${i++}`, { parameterName: param.valueAsString, version: 1 }), expected);
        test.throws(() => new ssm.StringParameter(stack, `p${i++}`, { parameterName: param.valueAsString, stringValue: 'foo' }), expected);
        test.throws(() => new ssm.StringParameter(stack, `p${i++}`, { parameterName: param.valueAsString, stringValue: 'foo' }), expected);
        test.done();
    },
    'fails if simpleName is wrong based on a concrete physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let i = 0;
        // THEN
        test.throws(() => ssm.StringParameter.fromStringParameterAttributes(stack, `p${i++}`, { parameterName: 'simple', simpleName: false }), /Parameter name "simple" is a simple name, but "simpleName" was explicitly set to false. Either omit it or set it to true/);
        test.throws(() => ssm.StringParameter.fromStringParameterAttributes(stack, `p${i++}`, { parameterName: '/foo/bar', simpleName: true }), /Parameter name "\/foo\/bar" is not a simple name, but "simpleName" was explicitly set to true. Either omit it or set it to false/);
        test.done();
    },
    'fails if parameterName is undefined and simpleName is "false"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new ssm.StringParameter(stack, 'p', { simpleName: false, stringValue: 'foo' }), /If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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