use std::collections::HashMap;

use dicom_core::VR;
use once_cell::sync::Lazy;

pub struct PrivateTagEntry<'a> {
    pub tag: u32,
    pub vr: VR,
    pub alias: &'a str,
}

type E = PrivateTagEntry<'static>;

pub struct PrivateCreatorEntry {
    pub creator: &'static str,
    pub entries: &'static [E],
}

type P = PrivateCreatorEntry;

type PrivateDictionary = HashMap<&'static str, HashMap<u32, &'static E>>;

fn init_private_dictionary() -> PrivateDictionary {
    let mut dict = HashMap::new();
    for entry in PRIVATE_ENTRIES {
        let mut creator_dict = HashMap::new();
        for private_entry in entry.entries {
            creator_dict.insert(private_entry.tag, private_entry);
        }
        dict.insert(entry.creator, creator_dict);
    }
    dict
}

pub static PRIVATE_DICT: Lazy<PrivateDictionary> = Lazy::new(init_private_dictionary);

#[rustfmt::skip]
pub(crate) const PRIVATE_ENTRIES: &[P] = &[
    P {
        creator: "AGFA PACS Archive Mirroring 1.0",
        entries: &[
            E { tag: 0x00311000, alias: "StudyStatus", vr: VR::CS },
            E { tag: 0x00311001, alias: "DateTimeVerified", vr: VR::UL },
        ],
    },
    P {
        creator: "CARDIO-D.R. 1.0",
        entries: &[
            E { tag: 0x00191030, alias: "MaximumImageFrameSize", vr: VR::UL },
        ],
    },
    P {
        creator: "CMR42 CIRCLECVI",
        entries: &[
            E { tag: 0x00251010, alias: "WorkspaceID", vr: VR::LO },
            E { tag: 0x00251020, alias: "WorkspaceTimeString", vr: VR::LO },
            E { tag: 0x00251030, alias: "WorkspaceStream", vr: VR::OB },
        ],
    },
    P {
        creator: "DCMTK_ANONYMIZER",
        entries: &[
            E { tag: 0x00091000, alias: "AnonymizerUIDMap", vr: VR::SQ },
            E { tag: 0x00091010, alias: "AnonymizerUIDKey", vr: VR::UI },
            E { tag: 0x00091020, alias: "AnonymizerUIDValue", vr: VR::UI },
            E { tag: 0x00091030, alias: "AnonymizerPatientIDMap", vr: VR::SQ },
            E { tag: 0x00091040, alias: "AnonymizerPatientIDKey", vr: VR::LO },
            E { tag: 0x00091050, alias: "AnonymizerPatientIDValue", vr: VR::LO },
        ],
    },
    P {
        creator: "DLX_EXAMS_01",
        entries: &[
            E { tag: 0x00151001, alias: "StenosisCalibrationRatio", vr: VR::DS },
            E { tag: 0x00151002, alias: "StenosisMagnification", vr: VR::DS },
            E { tag: 0x00151003, alias: "CardiacCalibrationRatio", vr: VR::DS },
        ],
    },
    P {
        creator: "FDMS 1.0",
        entries: &[
            E { tag: 0x00191060, alias: "RadiographersCode", vr: VR::SH },
            E { tag: 0x00291034, alias: "MagnificationReductionRatio", vr: VR::US },
        ],
    },
    P {
        creator: "Flywheel",
        entries: &[
            E { tag: 0x00211000, alias: "DICOM Send", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_ACQU_01",
        entries: &[
            E { tag: 0x00191002, alias: "NumberOfCellsInDetector", vr: VR::SL },
            E { tag: 0x0019100f, alias: "HorizontalFrameOfReference", vr: VR::DS },
            E { tag: 0x0019101b, alias: "LastScanLocation", vr: VR::DS },
            E { tag: 0x00191023, alias: "TableSpeed", vr: VR::DS },
            E { tag: 0x00191024, alias: "MidScanTime", vr: VR::DS },
            E { tag: 0x00191026, alias: "DegreesOfAzimuth", vr: VR::SL },
            E { tag: 0x00191027, alias: "GantryPeriod", vr: VR::DS },
            E { tag: 0x00191039, alias: "ScanFOVType", vr: VR::SS },
            E { tag: 0x00191043, alias: "TotalSegmentsRequested", vr: VR::SS },
            E { tag: 0x0019104a, alias: "TotalNumberOfRefChannels", vr: VR::SS },
            E { tag: 0x00191052, alias: "ReconPostProcessingFlag", vr: VR::SS },
            E { tag: 0x0019105e, alias: "NumberOfChannels1To512", vr: VR::SL },
            E { tag: 0x0019106a, alias: "DependantOnNumberOfViewsProcessed", vr: VR::SS },
            E { tag: 0x00191072, alias: "ZChannelAvgOverViews", vr: VR::DS },
            E { tag: 0x00191073, alias: "AvgOfLeftRefChannelsOverViews", vr: VR::DS },
            E { tag: 0x00191074, alias: "MaxLeftChannelOverViews", vr: VR::DS },
            E { tag: 0x00191075, alias: "AvgOfRightRefChannelsOverViews", vr: VR::DS },
            E { tag: 0x00191076, alias: "MaxRightChannelOverViews", vr: VR::DS },
            E { tag: 0x0019107e, alias: "NumberOfEchos", vr: VR::SS },
            E { tag: 0x0019108f, alias: "SwapPhaseFrequency", vr: VR::SS },
            E { tag: 0x00191091, alias: "PulseTime", vr: VR::DS },
            E { tag: 0x00191092, alias: "SliceOffsetOnFrequencyAxis", vr: VR::SL },
            E { tag: 0x00191093, alias: "CenterFrequency", vr: VR::DS },
            E { tag: 0x00191094, alias: "TransmitGain", vr: VR::SS },
            E { tag: 0x00191095, alias: "AnalogReceiverGain", vr: VR::SS },
            E { tag: 0x00191096, alias: "DigitalReceiverGain", vr: VR::SS },
            E { tag: 0x00191098, alias: "CenterFrequencyMethod", vr: VR::SS },
            E { tag: 0x0019109f, alias: "TransmittingCoil", vr: VR::SS },
            E { tag: 0x001910c1, alias: "SurfaceCoilIntensityCorrectionFlag", vr: VR::SS },
            E { tag: 0x001910cb, alias: "PrescribedFlowAxis", vr: VR::SS },
            E { tag: 0x001910d3, alias: "ProjectionAlgorithm", vr: VR::SH },
            E { tag: 0x001910d7, alias: "CardiacPhases", vr: VR::SS },
            E { tag: 0x001910d9, alias: "ConcatenatedSAT", vr: VR::DS },
            E { tag: 0x001910df, alias: "UserData23", vr: VR::DS },
            E { tag: 0x001910e0, alias: "UserData24", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_PARM_01",
        entries: &[
            E { tag: 0x00431008, alias: "RespiratoryRateInBPM", vr: VR::SS },
            E { tag: 0x0043100b, alias: "PeakRateOfChangeOfGradientField", vr: VR::DS },
            E { tag: 0x0043100c, alias: "LimitsInUnitsOfPercent", vr: VR::DS },
            E { tag: 0x00431013, alias: "ReconKernelParameters", vr: VR::SS },
            E { tag: 0x0043101e, alias: "DeltaStartTime", vr: VR::DS },
            E { tag: 0x00431026, alias: "NoViewsRefChannelsBlocked", vr: VR::US },
            E { tag: 0x00431028, alias: "UniqueImageIdentifier", vr: VR::OB },
            E { tag: 0x0043102d, alias: "StringSlopField1", vr: VR::SH },
            E { tag: 0x0043102f, alias: "RawDataType", vr: VR::SS },
            E { tag: 0x00431030, alias: "RawDataType", vr: VR::SS },
            E { tag: 0x00431031, alias: "RACoordOfTargetReconCentre", vr: VR::DS },
            E { tag: 0x00431032, alias: "RawDataType", vr: VR::SS },
            E { tag: 0x00431038, alias: "User25ToUser48", vr: VR::FL },
            E { tag: 0x00431039, alias: "SlopInteger6ToSlopInteger9", vr: VR::IS },
            E { tag: 0x00431060, alias: "SlopInteger10ToSlopInteger17", vr: VR::IS },
            E { tag: 0x0043106f, alias: "ScannerTableEntry", vr: VR::DS },
            E { tag: 0x00431074, alias: "NumberOfRestVolumes", vr: VR::US },
            E { tag: 0x00431075, alias: "NumberOfActiveVolumes", vr: VR::US },
            E { tag: 0x00431076, alias: "NumberOfDummyScans", vr: VR::US },
            E { tag: 0x0043107f, alias: "EDWIScaleFactor", vr: VR::DS },
            E { tag: 0x00431087, alias: "ScannerSoftwareVersionLongForm", vr: VR::UT },
        ],
    },
    P {
        creator: "GEMS_RELA_01",
        entries: &[
            E { tag: 0x00211056, alias: "IntegerSlop", vr: VR::SL },
            E { tag: 0x00211057, alias: "IntegerSlop", vr: VR::SL },
            E { tag: 0x00211058, alias: "IntegerSlop", vr: VR::SL },
            E { tag: 0x00211059, alias: "IntegerSlop", vr: VR::SL },
            E { tag: 0x0021105a, alias: "IntegerSlop", vr: VR::SL },
            E { tag: 0x0021105b, alias: "FloatSlop", vr: VR::DS },
            E { tag: 0x0021105c, alias: "FloatSlop", vr: VR::DS },
            E { tag: 0x0021105d, alias: "FloatSlop", vr: VR::DS },
            E { tag: 0x0021105e, alias: "FloatSlop", vr: VR::DS },
            E { tag: 0x0021105f, alias: "FloatSlop", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_SENO_02",
        entries: &[
            E { tag: 0x00451006, alias: "Angulation", vr: VR::DS },
            E { tag: 0x0045100d, alias: "ROIOriginXY", vr: VR::DS },
            E { tag: 0x00451011, alias: "ReceptorSizeCmXY", vr: VR::DS },
            E { tag: 0x00451012, alias: "ReceptorSizePixelsXY", vr: VR::IS },
            E { tag: 0x00451016, alias: "BinningFactorXY", vr: VR::IS },
            E { tag: 0x00451020, alias: "MeanOfRegionGrayLevels", vr: VR::DS },
            E { tag: 0x00451028, alias: "WindowingType", vr: VR::CS },
            E { tag: 0x0045102a, alias: "CrosshairCursorXCoordinates", vr: VR::IS },
            E { tag: 0x0045102b, alias: "CrosshairCursorYCoordinates", vr: VR::IS },
        ],
    },
    P {
        creator: "GEMS_SERS_01",
        entries: &[
            E { tag: 0x00251014, alias: "IndicatesNumberOfUpdatesToHeader", vr: VR::SL },
        ],
    },
    P {
        creator: "GEMS_STDY_01",
        entries: &[
            E { tag: 0x00231074, alias: "NumberOfUpdatesToHeader", vr: VR::SL },
            E { tag: 0x0023107d, alias: "IndicatesIfStudyHasCompleteInfo", vr: VR::SS },
        ],
    },
    P {
        creator: "GE_GENESIS_REV3.0",
        entries: &[
            E { tag: 0x0019108f, alias: "SwapPhaseFrequency", vr: VR::SS },
            E { tag: 0x001910a4, alias: "SATFatWaterBone", vr: VR::SS },
            E { tag: 0x001910c1, alias: "SurfaceCoilIntensityCorrectionFlag", vr: VR::SS },
            E { tag: 0x00431027, alias: "ScanPitchRatio", vr: VR::SH },
        ],
    },
    P {
        creator: "INTELERAD MEDICAL SYSTEMS",
        entries: &[
            E { tag: 0x00291001, alias: "ImageCompressionFraction", vr: VR::FD },
            E { tag: 0x00291002, alias: "ImageQuality", vr: VR::FD },
            E { tag: 0x00291003, alias: "ImageBytesTransferred", vr: VR::FD },
            E { tag: 0x00291010, alias: "J2cParameterType", vr: VR::SH },
            E { tag: 0x00291011, alias: "J2cPixelRepresentation", vr: VR::US },
            E { tag: 0x00291012, alias: "J2cBitsAllocated", vr: VR::US },
            E { tag: 0x00291013, alias: "J2cPixelShiftValue", vr: VR::US },
            E { tag: 0x00291014, alias: "J2cPlanarConfiguration", vr: VR::US },
            E { tag: 0x00291015, alias: "J2cRescaleIntercept", vr: VR::DS },
            E { tag: 0x00291020, alias: "PixelDataMD5SumPerFrame", vr: VR::LO },
            E { tag: 0x00291021, alias: "HistogramPercentileLabels", vr: VR::US },
            E { tag: 0x00291022, alias: "HistogramPercentileValues", vr: VR::FD },
            E { tag: 0x3f011006, alias: "OrderGroupNumber", vr: VR::LO },
            E { tag: 0x3f011007, alias: "StrippedPixelData", vr: VR::SH },
            E { tag: 0x3f011008, alias: "PendingMoveRequest", vr: VR::SH },
        ],
    },
    P {
        creator: "OCULUS Optikgeraete GmbH",
        entries: &[
            E { tag: 0x00291010, alias: "OriginalMeasuringData", vr: VR::OB },
            E { tag: 0x00291012, alias: "OriginalMeasuringDataLength", vr: VR::UL },
            E { tag: 0x00291020, alias: "OriginalMeasuringRawData", vr: VR::OB },
            E { tag: 0x00291022, alias: "OriginalMeasuringRawDataLength", vr: VR::UL },
        ],
    },
    P {
        creator: "Philips EV Imaging DD022",
        entries: &[
            E { tag: 0x20071012, alias: "VolumeSequenceCapture", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips Imaging DD 001",
        entries: &[
            E { tag: 0x20011005, alias: "GraphicAnnotationParentID", vr: VR::SS },
            E { tag: 0x20011009, alias: "ImagePrepulseDelay", vr: VR::FL },
            E { tag: 0x2001100c, alias: "ArrhythmiaRejection", vr: VR::CS },
            E { tag: 0x2001100e, alias: "CardiacCycled", vr: VR::CS },
            E { tag: 0x2001100f, alias: "CardiacGateWidth", vr: VR::SS },
            E { tag: 0x2001101e, alias: "MRSeriesReformatAccuracy", vr: VR::CS },
            E { tag: 0x2001103d, alias: "ContourFillColor", vr: VR::UL },
            E { tag: 0x2001103f, alias: "ZoomMode", vr: VR::CS },
            E { tag: 0x20011043, alias: "EllipsDisplShutMajorAxFrstEndPnt", vr: VR::IS },
            E { tag: 0x20011044, alias: "EllipsDisplShutMajorAxScndEndPnt", vr: VR::IS },
            E { tag: 0x20011045, alias: "EllipsDisplShutOtherAxFrstEndPnt", vr: VR::IS },
            E { tag: 0x20011050, alias: "GraphicMarkerType", vr: VR::LO },
            E { tag: 0x20011051, alias: "OverlayPlaneID", vr: VR::IS },
            E { tag: 0x20011052, alias: "ImagePresentationStateUID", vr: VR::UI },
            E { tag: 0x20011054, alias: "ContourFillTransparency", vr: VR::FL },
            E { tag: 0x20011058, alias: "ContrastTransferTaste", vr: VR::UL },
            E { tag: 0x20011061, alias: "SeriesTransmitted", vr: VR::CS },
            E { tag: 0x20011062, alias: "SeriesCommitted", vr: VR::CS },
            E { tag: 0x20011065, alias: "ROIOverlayPlane", vr: VR::SQ },
            E { tag: 0x20011067, alias: "LinearPresentationGLTrafoShapeSub", vr: VR::CS },
            E { tag: 0x20011068, alias: "LinearModalityGLTrafo", vr: VR::SQ },
            E { tag: 0x2001106a, alias: "SpatialTransformation", vr: VR::SQ },
            E { tag: 0x20011071, alias: "GraphicConstraint", vr: VR::CS },
            E { tag: 0x20011072, alias: "EllipsDisplShutOtherAxScndEndPnt", vr: VR::IS },
            E { tag: 0x20011076, alias: "MRNumberOfFrames", vr: VR::UI },
            E { tag: 0x2001107a, alias: "WindowRoundingFactor", vr: VR::FL },
            E { tag: 0x2001107c, alias: "FrameNumber", vr: VR::US },
            E { tag: 0x20011086, alias: "MRSeriesNrOfPhaseEncodingSteps", vr: VR::IS },
            E { tag: 0x2001108b, alias: "MRSeriesTransmittingCoil", vr: VR::SH },
            E { tag: 0x20011090, alias: "TextForegroundColor", vr: VR::LO },
            E { tag: 0x20011091, alias: "TextBackgroundColor", vr: VR::LO },
            E { tag: 0x20011092, alias: "TextShadowColor", vr: VR::LO },
            E { tag: 0x2001109c, alias: "GraphicAnnotationLabel", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 001",
        entries: &[
            E { tag: 0x20051000, alias: "MRImageAngulationAP", vr: VR::FL },
            E { tag: 0x20051001, alias: "MRImageAngulationFH", vr: VR::FL },
            E { tag: 0x20051002, alias: "MRImageAngulationRL", vr: VR::FL },
            E { tag: 0x20051003, alias: "ImageAnnotationCount", vr: VR::IS },
            E { tag: 0x20051004, alias: "MRImageDisplayOrientation", vr: VR::CS },
            E { tag: 0x20051007, alias: "ImageLineCount", vr: VR::IS },
            E { tag: 0x20051008, alias: "MRImageOffCentreAP", vr: VR::FL },
            E { tag: 0x20051009, alias: "MRImageOffCentreFH", vr: VR::FL },
            E { tag: 0x2005100a, alias: "MRImageOffCentreRL", vr: VR::FL },
            E { tag: 0x2005100b, alias: "MRMaxFP", vr: VR::FL },
            E { tag: 0x2005100c, alias: "MRMinFP", vr: VR::FL },
            E { tag: 0x2005100d, alias: "MRScaleIntercept", vr: VR::FL },
            E { tag: 0x2005100e, alias: "MRScaleSlope", vr: VR::FL },
            E { tag: 0x20051011, alias: "MRImageTypeMR", vr: VR::CS },
            E { tag: 0x20051012, alias: "MRCardiacGating", vr: VR::CS },
            E { tag: 0x20051013, alias: "MRSeriesDevelopmentMode", vr: VR::CS },
            E { tag: 0x20051014, alias: "MRSeriesDiffusion", vr: VR::CS },
            E { tag: 0x20051015, alias: "MRFatSaturation", vr: VR::CS },
            E { tag: 0x20051016, alias: "MRFlowCompensation", vr: VR::CS },
            E { tag: 0x20051017, alias: "MRFourierInterpolation", vr: VR::CS },
            E { tag: 0x20051018, alias: "MRHardcopyProtocol", vr: VR::LO },
            E { tag: 0x20051019, alias: "MRInverseReconstructed", vr: VR::CS },
            E { tag: 0x2005101a, alias: "MRLabelSyntax", vr: VR::SS },
            E { tag: 0x2005101b, alias: "MRMagnetiPrepared", vr: VR::CS },
            E { tag: 0x2005101c, alias: "MRMagnetTransferConst", vr: VR::CS },
            E { tag: 0x2005101d, alias: "MRMeasurementScanResolution", vr: VR::SS },
            E { tag: 0x2005101e, alias: "MIPProtocol", vr: VR::SH },
            E { tag: 0x2005101f, alias: "MPRProtocol", vr: VR::SH },
            E { tag: 0x20051021, alias: "MRNumberOfMixes", vr: VR::SS },
            E { tag: 0x20051022, alias: "MRNumberOfReferences", vr: VR::IS },
            E { tag: 0x20051023, alias: "MRNumberOfSlabs", vr: VR::SS },
            E { tag: 0x20051025, alias: "MRNumberOfVolumes", vr: VR::SS },
            E { tag: 0x20051026, alias: "MROverSampleingPhase", vr: VR::CS },
            E { tag: 0x20051027, alias: "MRPackageMode", vr: VR::CS },
            E { tag: 0x20051028, alias: "MRPartialFourierFrequency", vr: VR::CS },
            E { tag: 0x20051029, alias: "MRPartialFourierPhase", vr: VR::CS },
            E { tag: 0x2005102a, alias: "MRPatientReferenceID", vr: VR::IS },
            E { tag: 0x2005102b, alias: "MRPercentScanComplete", vr: VR::SS },
            E { tag: 0x2005102c, alias: "MRPhaseEncodedRecording", vr: VR::CS },
            E { tag: 0x2005102d, alias: "NumberOfStackSlices", vr: VR::SS },
            E { tag: 0x2005102e, alias: "MRPPGPPUGating", vr: VR::CS },
            E { tag: 0x2005102f, alias: "MRSpatialPresaturation", vr: VR::CS },
            E { tag: 0x20051031, alias: "MRRespiratoryGating", vr: VR::CS },
            E { tag: 0x20051032, alias: "SampleRepresentation", vr: VR::CS },
            E { tag: 0x20051034, alias: "MRSegmentedKSpace", vr: VR::CS },
            E { tag: 0x20051035, alias: "MRSeriesDataType", vr: VR::CS },
            E { tag: 0x20051036, alias: "MRSeriesIsCardiac", vr: VR::CS },
            E { tag: 0x20051037, alias: "MRSeriesIsSpectro", vr: VR::CS },
            E { tag: 0x20051038, alias: "MRSpoiled", vr: VR::CS },
            E { tag: 0x20051039, alias: "MRSteadyState", vr: VR::CS },
            E { tag: 0x2005103a, alias: "MRSubAnatomy", vr: VR::SH },
            E { tag: 0x2005103b, alias: "MRTimeReversedSteadyState", vr: VR::CS },
            E { tag: 0x2005103c, alias: "MRSeriesTone", vr: VR::CS },
            E { tag: 0x2005103d, alias: "MRNumberOfRRIntervalRanges", vr: VR::SS },
            E { tag: 0x2005103e, alias: "MRRRIntervalsDistribution", vr: VR::SL },
            E { tag: 0x2005103f, alias: "MRPlanScanAcquisitionNo", vr: VR::SL },
            E { tag: 0x20051040, alias: "MRChemicalShiftNo", vr: VR::SL },
            E { tag: 0x20051041, alias: "MRPlanScanDynamicScanNo", vr: VR::SL },
            E { tag: 0x20051042, alias: "MRPlanScanSurveyEchoNo", vr: VR::SL },
            E { tag: 0x20051043, alias: "MRPlanScanImageType", vr: VR::CS },
            E { tag: 0x20051044, alias: "MRPlanScanPhaseNo", vr: VR::SL },
            E { tag: 0x20051045, alias: "MRPlanScanReconstructionNo", vr: VR::SL },
            E { tag: 0x20051046, alias: "MRPlanScanScanningSequence", vr: VR::CS },
            E { tag: 0x20051047, alias: "MRPlanScanSliceNo", vr: VR::SL },
            E { tag: 0x20051048, alias: "MRReferenceAcquisitionNo", vr: VR::IS },
            E { tag: 0x20051049, alias: "MRReferenceChemicalShiftNo", vr: VR::IS },
            E { tag: 0x2005104a, alias: "MRReferenceDynamicScanNo", vr: VR::IS },
            E { tag: 0x2005104b, alias: "MRReferenceEchoNo", vr: VR::IS },
            E { tag: 0x2005104c, alias: "MRReferenceEntity", vr: VR::CS },
            E { tag: 0x2005104d, alias: "MRReferenceImageType", vr: VR::CS },
            E { tag: 0x2005104e, alias: "MRSlabFovRL", vr: VR::FL },
            E { tag: 0x2005104f, alias: "MRSlabOffcentreAP", vr: VR::FL },
            E { tag: 0x20051050, alias: "MRSlabOffcentreFH", vr: VR::FL },
            E { tag: 0x20051051, alias: "MRSlabOffcentreRL", vr: VR::FL },
            E { tag: 0x20051052, alias: "MRSlabType", vr: VR::CS },
            E { tag: 0x20051053, alias: "MRSlabViewAxis", vr: VR::CS },
            E { tag: 0x20051054, alias: "MRVolumeAngulationAP", vr: VR::FL },
            E { tag: 0x20051055, alias: "MRVolumeAngulationFH", vr: VR::FL },
            E { tag: 0x20051056, alias: "MRVolumeAngulationRL", vr: VR::FL },
            E { tag: 0x20051057, alias: "MRVolumeFovAP", vr: VR::FL },
            E { tag: 0x20051058, alias: "MRVolumeFovFH", vr: VR::FL },
            E { tag: 0x20051059, alias: "MRVolumeFovRL", vr: VR::FL },
            E { tag: 0x2005105a, alias: "MRVolumeOffcentreAP", vr: VR::FL },
            E { tag: 0x2005105b, alias: "MRVolumeOffcentreFH", vr: VR::FL },
            E { tag: 0x2005105c, alias: "MRVolumeOffcentreRL", vr: VR::FL },
            E { tag: 0x2005105d, alias: "MRVolumeType", vr: VR::CS },
            E { tag: 0x2005105e, alias: "MRVolumeViewAxis", vr: VR::CS },
            E { tag: 0x2005105f, alias: "MRStudyOrigin", vr: VR::CS },
            E { tag: 0x20051060, alias: "MRStudySequenceNumber", vr: VR::IS },
            E { tag: 0x20051061, alias: "MRImagePrepulseType", vr: VR::CS },
            E { tag: 0x20051063, alias: "MRfMRIStatusIndication", vr: VR::SS },
            E { tag: 0x20051064, alias: "MRReferencePhaseNo", vr: VR::IS },
            E { tag: 0x20051065, alias: "MRReferenceReconstructionNo", vr: VR::IS },
            E { tag: 0x20051066, alias: "MRReferenceScanningSequence", vr: VR::CS },
            E { tag: 0x20051067, alias: "MRReferenceSliceNo", vr: VR::IS },
            E { tag: 0x20051068, alias: "MRReferenceType", vr: VR::CS },
            E { tag: 0x20051069, alias: "MRSlabAngulationAP", vr: VR::FL },
            E { tag: 0x2005106a, alias: "MRSlabAngulationFH", vr: VR::FL },
            E { tag: 0x2005106b, alias: "MRSlabAngulationRL", vr: VR::FL },
            E { tag: 0x2005106c, alias: "MRSlabFovAP", vr: VR::FL },
            E { tag: 0x2005106d, alias: "MRSlabFovFH", vr: VR::FL },
            E { tag: 0x2005106e, alias: "MRImageScanningSequencePrivate", vr: VR::CS },
            E { tag: 0x2005106f, alias: "MRSeriesAcquisitionTypePrivate", vr: VR::CS },
            E { tag: 0x20051070, alias: "MRSeriesHardcopyProtocolEV", vr: VR::LO },
            E { tag: 0x20051071, alias: "MRStackAngulationAP", vr: VR::FL },
            E { tag: 0x20051072, alias: "MRStackAngulationFH", vr: VR::FL },
            E { tag: 0x20051073, alias: "MRStackAngulationRL", vr: VR::FL },
            E { tag: 0x20051074, alias: "MRStackFovAP", vr: VR::FL },
            E { tag: 0x20051075, alias: "MRStackFovFH", vr: VR::FL },
            E { tag: 0x20051076, alias: "MRStackFovRL", vr: VR::FL },
            E { tag: 0x20051078, alias: "MRStackOffcentreAP", vr: VR::FL },
            E { tag: 0x20051079, alias: "MRStackOffcentreFH", vr: VR::FL },
            E { tag: 0x2005107a, alias: "MRStackOffcentreRL", vr: VR::FL },
            E { tag: 0x2005107b, alias: "MRStackPreparationDirection", vr: VR::CS },
            E { tag: 0x2005107e, alias: "MRStackSliceDistance", vr: VR::FL },
            E { tag: 0x20051080, alias: "SeriesPlanScan", vr: VR::SQ },
            E { tag: 0x20051081, alias: "MRStackViewAxis", vr: VR::CS },
            E { tag: 0x20051084, alias: "SeriesReference", vr: VR::SQ },
            E { tag: 0x20051085, alias: "SeriesVolume", vr: VR::SQ },
            E { tag: 0x20051086, alias: "MRNumberOfGeometry", vr: VR::SS },
            E { tag: 0x20051087, alias: "MRNumberOfGeometrySlices", vr: VR::SL },
            E { tag: 0x20051088, alias: "MRGeomAngulationAP", vr: VR::FL },
            E { tag: 0x20051089, alias: "MRGeomAngulationFH", vr: VR::FL },
            E { tag: 0x2005108a, alias: "MRGeomAngulationRL", vr: VR::FL },
            E { tag: 0x2005108b, alias: "MRGeomFOVAP", vr: VR::FL },
            E { tag: 0x2005108c, alias: "MRGeomFOVFH", vr: VR::FL },
            E { tag: 0x2005108d, alias: "MRGeomFOVRL", vr: VR::FL },
            E { tag: 0x2005108e, alias: "MRGeomOffCentreAP", vr: VR::FL },
            E { tag: 0x2005108f, alias: "MRGeomOffCentreFH", vr: VR::FL },
            E { tag: 0x20051090, alias: "MRGeomOffCentreRL", vr: VR::FL },
            E { tag: 0x20051091, alias: "MRGeomPreparationDirect", vr: VR::CS },
            E { tag: 0x20051092, alias: "MRGeomRadialAngle", vr: VR::FL },
            E { tag: 0x20051093, alias: "MRGeomRadialAxis", vr: VR::CS },
            E { tag: 0x20051094, alias: "MRGeomSliceDistance", vr: VR::FL },
            E { tag: 0x20051095, alias: "MRGeomSliceNumber", vr: VR::SL },
            E { tag: 0x20051096, alias: "MRGeomType", vr: VR::CS },
            E { tag: 0x20051097, alias: "MRGeomViewAxis", vr: VR::CS },
            E { tag: 0x20051098, alias: "MRGeomColour", vr: VR::CS },
            E { tag: 0x20051099, alias: "MRGeomApplicationType", vr: VR::CS },
            E { tag: 0x2005109a, alias: "MRGeomId", vr: VR::SL },
            E { tag: 0x2005109b, alias: "MRGeomApplicationName", vr: VR::SH },
            E { tag: 0x2005109c, alias: "MRGeomLableName", vr: VR::SH },
            E { tag: 0x2005109d, alias: "MRGeomLineStyle", vr: VR::CS },
            E { tag: 0x2005109e, alias: "SeriesGeom", vr: VR::SQ },
            E { tag: 0x2005109f, alias: "MRSeriesSpectralSelectiveExcitationPulse", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 002",
        entries: &[
            E { tag: 0x20051015, alias: "MRUserName", vr: VR::LO },
            E { tag: 0x20051016, alias: "MRPassWord", vr: VR::LO },
            E { tag: 0x20051017, alias: "MRServerName", vr: VR::LO },
            E { tag: 0x20051018, alias: "MRDataBaseName", vr: VR::LO },
            E { tag: 0x20051019, alias: "MRRootName", vr: VR::LO },
            E { tag: 0x20051020, alias: "DMIApplicationName", vr: VR::LO },
            E { tag: 0x2005102d, alias: "MRRootId", vr: VR::LO },
            E { tag: 0x20051032, alias: "MRBlobDataObjectArray", vr: VR::SQ },
            E { tag: 0x20051034, alias: "SeriesTransactionUID", vr: VR::LT },
            E { tag: 0x20051035, alias: "ParentID", vr: VR::IS },
            E { tag: 0x20051036, alias: "ParentType", vr: VR::PN },
            E { tag: 0x20051037, alias: "MRBlobName", vr: VR::PN },
            E { tag: 0x20051038, alias: "MRApplicationName", vr: VR::PN },
            E { tag: 0x20051039, alias: "MRTypeName", vr: VR::PN },
            E { tag: 0x20051040, alias: "MRVersionStr", vr: VR::PN },
            E { tag: 0x20051041, alias: "MRCommentStr", vr: VR::PN },
            E { tag: 0x20051042, alias: "BlobInFile", vr: VR::CS },
            E { tag: 0x20051043, alias: "MRActualBlobSize", vr: VR::SL },
            E { tag: 0x20051044, alias: "MRBlobData", vr: VR::OW },
            E { tag: 0x20051045, alias: "BlobFilename", vr: VR::PN },
            E { tag: 0x20051046, alias: "BlobOffset", vr: VR::SL },
            E { tag: 0x20051047, alias: "MRBlobFlag", vr: VR::CS },
            E { tag: 0x20051099, alias: "MRNumberOfRequestExcerpts", vr: VR::UL },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 003",
        entries: &[
            E { tag: 0x20051000, alias: "MRNumberOfSOPCommon", vr: VR::UL },
            E { tag: 0x20051001, alias: "MRNoOfFilmConsumption", vr: VR::UL },
            E { tag: 0x20051013, alias: "MRNumberOfCodes", vr: VR::UL },
            E { tag: 0x20051034, alias: "MRNumberOfImagePerSeriesRef", vr: VR::SL },
            E { tag: 0x20051043, alias: "MRNoDateOfLastCalibration", vr: VR::SS },
            E { tag: 0x20051044, alias: "MRNoTimeOfLastCalibration", vr: VR::SS },
            E { tag: 0x20051045, alias: "MRNrOfSoftwareVersion", vr: VR::SS },
            E { tag: 0x20051047, alias: "MRNrOfPatientOtherNames", vr: VR::SS },
            E { tag: 0x20051048, alias: "MRNrOfReqRecipeOfResults", vr: VR::SS },
            E { tag: 0x20051049, alias: "MRNrOfSeriesOperatorsName", vr: VR::SS },
            E { tag: 0x20051050, alias: "MRNrOfSeriesPerfPhysiName", vr: VR::SS },
            E { tag: 0x20051051, alias: "MRNrOfStudyAdmittingDiagnosticDescr", vr: VR::SS },
            E { tag: 0x20051052, alias: "MRNrOfStudyPatientContrastAllergies", vr: VR::SS },
            E { tag: 0x20051053, alias: "MRNrOfStudyPatientMedicalAlerts", vr: VR::SS },
            E { tag: 0x20051054, alias: "MRNrOfStudyPhysiciansOfRecord", vr: VR::SS },
            E { tag: 0x20051055, alias: "MRNrOfStudyPhysiReadingStudy", vr: VR::SS },
            E { tag: 0x20051056, alias: "MRNrSCSoftwareVersions", vr: VR::SS },
            E { tag: 0x20051057, alias: "MRNrRunningAttributes", vr: VR::SS },
            E { tag: 0x20051070, alias: "SpectrumPixelData", vr: VR::OW },
            E { tag: 0x20051081, alias: "DefaultImageUID", vr: VR::UI },
            E { tag: 0x20051082, alias: "RunningAttributes", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 004",
        entries: &[
            E { tag: 0x20051000, alias: "MRSpectrumExtraNumber", vr: VR::SS },
            E { tag: 0x20051001, alias: "MRSpectrumKxCoordinate", vr: VR::SS },
            E { tag: 0x20051002, alias: "MRSpectrumKyCoordinate", vr: VR::SS },
            E { tag: 0x20051003, alias: "MRSpectrumLocationNumber", vr: VR::SS },
            E { tag: 0x20051004, alias: "MRSpectrumMixNumber", vr: VR::SS },
            E { tag: 0x20051005, alias: "MRSpectrumXCoordinate", vr: VR::SS },
            E { tag: 0x20051006, alias: "MRSpectrumYCoordinate", vr: VR::SS },
            E { tag: 0x20051007, alias: "MRSpectrumDCLevel", vr: VR::FL },
            E { tag: 0x20051008, alias: "MRSpectrumNoiseLevel", vr: VR::FL },
            E { tag: 0x20051009, alias: "MRSpectrumBeginTime", vr: VR::FL },
            E { tag: 0x20051010, alias: "MRSpectrumEchoTime", vr: VR::FL },
            E { tag: 0x20051012, alias: "SpectrumNumber", vr: VR::FL },
            E { tag: 0x20051013, alias: "MRSpectrumNumber", vr: VR::SS },
            E { tag: 0x20051014, alias: "MRSpectrumNumberOfAverages", vr: VR::SS },
            E { tag: 0x20051015, alias: "MRSpectrumNumberOfSamples", vr: VR::SS },
            E { tag: 0x20051016, alias: "MRSpectrumScanSequenceNumber", vr: VR::SS },
            E { tag: 0x20051017, alias: "MRSpectrumNumberOfPeaks", vr: VR::SS },
            E { tag: 0x20051018, alias: "MRSpectrumPeak", vr: VR::SQ },
            E { tag: 0x20051019, alias: "MRSpectrumPeakIntensity", vr: VR::FL },
            E { tag: 0x20051020, alias: "MRSpectrumPeakLabel", vr: VR::LO },
            E { tag: 0x20051021, alias: "MRSpectrumPeakPhase", vr: VR::FL },
            E { tag: 0x20051022, alias: "MRSpectrumPeakPosition", vr: VR::FL },
            E { tag: 0x20051023, alias: "MRSpectrumPeakType", vr: VR::CS },
            E { tag: 0x20051024, alias: "MRSpectrumPeakWidth", vr: VR::FL },
            E { tag: 0x20051025, alias: "MRSpectroSIB0Correction", vr: VR::CS },
            E { tag: 0x20051026, alias: "MRSpectroB0EchoTopPosition", vr: VR::FL },
            E { tag: 0x20051027, alias: "MRSpectroComplexComponent", vr: VR::CS },
            E { tag: 0x20051028, alias: "MRSpectroDataOrigin", vr: VR::CS },
            E { tag: 0x20051029, alias: "MRSpectroEchoTopPosition", vr: VR::FL },
            E { tag: 0x20051030, alias: "MRInPlaneTransforms", vr: VR::CS },
            E { tag: 0x20051031, alias: "MRNumberOfSpectraAcquired", vr: VR::SS },
            E { tag: 0x20051033, alias: "MRPhaseEncodingEchoTopPositions", vr: VR::FL },
            E { tag: 0x20051034, alias: "MRPhysicalQuantityForChemicalShift", vr: VR::CS },
            E { tag: 0x20051035, alias: "MRPhysicalQuantitySpatial", vr: VR::CS },
            E { tag: 0x20051036, alias: "MRReferenceFrequency", vr: VR::FL },
            E { tag: 0x20051037, alias: "MRSampleOffset", vr: VR::FL },
            E { tag: 0x20051038, alias: "MRSamplePitch", vr: VR::FL },
            E { tag: 0x20051039, alias: "MRSearchIntervalForPeaks", vr: VR::SS },
            E { tag: 0x20051040, alias: "MRSignalDomainForChemicalShift", vr: VR::CS },
            E { tag: 0x20051041, alias: "MRSignalDomainSpatial", vr: VR::CS },
            E { tag: 0x20051042, alias: "MRSignalType", vr: VR::CS },
            E { tag: 0x20051043, alias: "MRSpectroAdditionalRotations", vr: VR::CS },
            E { tag: 0x20051044, alias: "MRSpectroDisplayRanges", vr: VR::SS },
            E { tag: 0x20051045, alias: "MRSpectroEchoAcquisition", vr: VR::CS },
            E { tag: 0x20051046, alias: "MRSpectroFrequencyUnit", vr: VR::CS },
            E { tag: 0x20051047, alias: "MRSpectroGamma", vr: VR::FL },
            E { tag: 0x20051048, alias: "MRSpectroHiddenLineRemoval", vr: VR::CS },
            E { tag: 0x20051049, alias: "MRSpectroHorizontalShift", vr: VR::FL },
            E { tag: 0x20051050, alias: "MRSpectroHorizontalWindow", vr: VR::FL },
            E { tag: 0x20051051, alias: "MRSpectroNumberOfDisplayRanges", vr: VR::SS },
            E { tag: 0x20051052, alias: "MRSpectroNumberOfEchoPulses", vr: VR::SS },
            E { tag: 0x20051053, alias: "MRSpectroProcessingHistory", vr: VR::LO },
            E { tag: 0x20051054, alias: "MRSpectroScanType", vr: VR::CS },
            E { tag: 0x20051055, alias: "MRSpectroSICSIntervals", vr: VR::FL },
            E { tag: 0x20051056, alias: "MRSpectroSIMode", vr: VR::CS },
            E { tag: 0x20051057, alias: "MRSpectroSpectralBW", vr: VR::SS },
            E { tag: 0x20051058, alias: "MRSpectroTitleLine", vr: VR::LO },
            E { tag: 0x20051059, alias: "MRSpectroTurboEchoSpacing", vr: VR::FL },
            E { tag: 0x20051060, alias: "MRSpectroVerticalShift", vr: VR::FL },
            E { tag: 0x20051061, alias: "MRSpectroVerticalWindow", vr: VR::FL },
            E { tag: 0x20051062, alias: "MRSpectroOffset", vr: VR::FL },
            E { tag: 0x20051063, alias: "MRSpectrumPitch", vr: VR::FL },
            E { tag: 0x20051064, alias: "MRVolumeSelection", vr: VR::CS },
            E { tag: 0x20051070, alias: "MRNoMixesSpectro", vr: VR::SS },
            E { tag: 0x20051071, alias: "MRSeriesSPMix", vr: VR::SQ },
            E { tag: 0x20051072, alias: "SPMixTResolution", vr: VR::SS },
            E { tag: 0x20051073, alias: "SPMixKXResolution", vr: VR::SS },
            E { tag: 0x20051074, alias: "SPMixKYResolution", vr: VR::SS },
            E { tag: 0x20051075, alias: "SPMixFResolution", vr: VR::SS },
            E { tag: 0x20051076, alias: "SPMixXResolution", vr: VR::SS },
            E { tag: 0x20051077, alias: "SPMixYResolution", vr: VR::SS },
            E { tag: 0x20051078, alias: "SPMixNoSpectraIntended", vr: VR::SS },
            E { tag: 0x20051079, alias: "SPMixNoAverages", vr: VR::SS },
            E { tag: 0x20051080, alias: "MRSeriesNrOfMFImageObjects", vr: VR::SL },
            E { tag: 0x20051081, alias: "MRScanoGramSurveyNumberOfImages", vr: VR::IS },
            E { tag: 0x20051082, alias: "MRNumberOfProcedureCodes", vr: VR::UL },
            E { tag: 0x20051083, alias: "SortAttributes", vr: VR::CS },
            E { tag: 0x20051084, alias: "MRNrSortAttributes", vr: VR::SS },
            E { tag: 0x20051085, alias: "ImageDisplayDirection", vr: VR::CS },
            E { tag: 0x20051086, alias: "InsetScanogram", vr: VR::CS },
            E { tag: 0x20051087, alias: "MRDisplayLayoutNrColumns", vr: VR::SS },
            E { tag: 0x20051088, alias: "MRDisplayLayoutNrRows", vr: VR::SS },
            E { tag: 0x20051089, alias: "ViewingProtocol", vr: VR::SQ },
            E { tag: 0x20051090, alias: "MRStackCoilFunction", vr: VR::CS },
            E { tag: 0x20051091, alias: "PatientNameJobInParams", vr: VR::PN },
            E { tag: 0x20051092, alias: "MRGeolinkID", vr: VR::IS },
            E { tag: 0x20051093, alias: "MRStationNo", vr: VR::IS },
            E { tag: 0x20051094, alias: "ProcessingHistory", vr: VR::CS },
            E { tag: 0x20051095, alias: "ViewProcedureString", vr: VR::UI },
            E { tag: 0x20051096, alias: "MRFlowImagesPresent", vr: VR::CS },
            E { tag: 0x20051097, alias: "AnatomicRegCodeValue", vr: VR::LO },
            E { tag: 0x20051098, alias: "MRMobiviewEnabled", vr: VR::CS },
            E { tag: 0x20051099, alias: "MRIViewBoldEnabled", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 005",
        entries: &[
            E { tag: 0x20051000, alias: "MRVolumeViewEnabled", vr: VR::CS },
            E { tag: 0x20051001, alias: "MRNumberOfStudyReference", vr: VR::UL },
            E { tag: 0x20051003, alias: "MRNumberOfSPSCodes", vr: VR::UL },
            E { tag: 0x20051007, alias: "MRNrOfSpecificCharacterSet", vr: VR::SS },
            E { tag: 0x20051009, alias: "RescaleInterceptOriginal", vr: VR::DS },
            E { tag: 0x2005100a, alias: "RescaleSlopeOriginal", vr: VR::DS },
            E { tag: 0x2005100b, alias: "RescaleTypeOriginal", vr: VR::LO },
            E { tag: 0x2005100e, alias: "PrivateSharedSq", vr: VR::SQ },
            E { tag: 0x2005100f, alias: "PrivatePerFrameSq", vr: VR::SQ },
            E { tag: 0x20051010, alias: "MFConvTreatSpectorMixNo", vr: VR::IS },
            E { tag: 0x20051011, alias: "MFPrivateReferencedSOPInstanceUID", vr: VR::UI },
            E { tag: 0x20051012, alias: "MRImageDiffBValueNumber", vr: VR::IS },
            E { tag: 0x20051013, alias: "MRImageGradientOrientationNumber", vr: VR::IS },
            E { tag: 0x20051014, alias: "MRSeriesNrOfDiffBValues", vr: VR::SL },
            E { tag: 0x20051015, alias: "MRSeriesNrOfDiffGradOrients", vr: VR::SL },
            E { tag: 0x20051016, alias: "MRSeriesPlanMode", vr: VR::CS },
            E { tag: 0x20051017, alias: "DiffusionBMatrix", vr: VR::FD },
            E { tag: 0x20051018, alias: "PrivOperatingModeType", vr: VR::CS },
            E { tag: 0x20051019, alias: "PrivOperatingMode", vr: VR::CS },
            E { tag: 0x2005101a, alias: "MRFatSaturationTechnique", vr: VR::CS },
            E { tag: 0x2005101b, alias: "MRVersionNumberDeletedImages", vr: VR::IS },
            E { tag: 0x2005101c, alias: "MRVersionNumberDeletedSpectra", vr: VR::IS },
            E { tag: 0x2005101d, alias: "MRVersionNumberDeletedBlobsets", vr: VR::IS },
            E { tag: 0x2005101e, alias: "LUT1Offset", vr: VR::UL },
            E { tag: 0x2005101f, alias: "LUT1Range", vr: VR::UL },
            E { tag: 0x20051020, alias: "LUT1BeginColor", vr: VR::UL },
            E { tag: 0x20051021, alias: "LUT1EndColor", vr: VR::UL },
            E { tag: 0x20051022, alias: "LUT2Offset", vr: VR::UL },
            E { tag: 0x20051023, alias: "LUT2Range", vr: VR::UL },
            E { tag: 0x20051024, alias: "LUT2BeginColor", vr: VR::UL },
            E { tag: 0x20051025, alias: "LUT2EndColor", vr: VR::UL },
            E { tag: 0x20051026, alias: "ViewingHardcopyOnly", vr: VR::CS },
            E { tag: 0x20051027, alias: "PrivateEMR", vr: VR::SQ },
            E { tag: 0x20051028, alias: "MRSeriesNrOfLabelTypes", vr: VR::SL },
            E { tag: 0x20051029, alias: "MRImageLabelType", vr: VR::CS },
            E { tag: 0x2005102a, alias: "ExamPrintStatus", vr: VR::CS },
            E { tag: 0x2005102b, alias: "ExamExportStatus", vr: VR::CS },
            E { tag: 0x2005102c, alias: "ExamStorageCommitStatus", vr: VR::CS },
            E { tag: 0x2005102d, alias: "ExamMediaWriteStatus", vr: VR::CS },
            E { tag: 0x2005102e, alias: "MRSeriesDBdt", vr: VR::FL },
            E { tag: 0x2005102f, alias: "MRSeriesProtonSAR", vr: VR::FL },
            E { tag: 0x20051030, alias: "MRSeriesNonProtonSAR", vr: VR::FL },
            E { tag: 0x20051031, alias: "MRSeriesLocalSAR", vr: VR::FL },
            E { tag: 0x20051032, alias: "MRSeriesSafetyOverrideMode", vr: VR::CS },
            E { tag: 0x20051033, alias: "EVDVDJobInParamsDatetime", vr: VR::DT },
            E { tag: 0x20051034, alias: "DVDJobInParamsVolumeLabel", vr: VR::DT },
            E { tag: 0x20051035, alias: "SpectroExamcard", vr: VR::CS },
            E { tag: 0x20051036, alias: "MRRefSeriesInstanceUID", vr: VR::UI },
            E { tag: 0x20051037, alias: "ColorLUTType", vr: VR::CS },
            E { tag: 0x2005103b, alias: "MRIsCoilSurvey", vr: VR::CS },
            E { tag: 0x2005103c, alias: "MRStackTablePosLong", vr: VR::FL },
            E { tag: 0x2005103d, alias: "MRStackTablePosLat", vr: VR::FL },
            E { tag: 0x2005103e, alias: "MRStackPosteriorCoilPos", vr: VR::FL },
            E { tag: 0x2005103f, alias: "AIMDLimitsApplied", vr: VR::CS },
            E { tag: 0x20051040, alias: "AIMDHeadSARLimit", vr: VR::FL },
            E { tag: 0x20051041, alias: "AIMDWholeBodySARLimit", vr: VR::FL },
            E { tag: 0x20051042, alias: "AIMDB1RMSLimit", vr: VR::FL },
            E { tag: 0x20051043, alias: "AIMDdbDtLimit", vr: VR::FL },
            E { tag: 0x20051044, alias: "TFEFactor", vr: VR::IS },
            E { tag: 0x20051045, alias: "AttenuationCorrection", vr: VR::CS },
            E { tag: 0x20051046, alias: "FWHMShim", vr: VR::FL },
            E { tag: 0x20051047, alias: "PowerOptimization", vr: VR::FL },
            E { tag: 0x20051048, alias: "CoilQ", vr: VR::FL },
            E { tag: 0x20051049, alias: "ReceiverGain", vr: VR::FL },
            E { tag: 0x2005104a, alias: "DataWindowDuration", vr: VR::FL },
            E { tag: 0x2005104b, alias: "MixingTime", vr: VR::FL },
            E { tag: 0x2005104c, alias: "FirstEchoTime", vr: VR::FL },
            E { tag: 0x2005104d, alias: "IsB0Series", vr: VR::CS },
            E { tag: 0x2005104e, alias: "IsB1Series", vr: VR::CS },
            E { tag: 0x2005104f, alias: "VolumeSelect", vr: VR::CS },
            E { tag: 0x20051050, alias: "MRNrOfPatientOtherIDs", vr: VR::SS },
            E { tag: 0x20051051, alias: "PrivateSeriesNumber", vr: VR::IS },
            E { tag: 0x20051052, alias: "PrivateSeriesInstanceUID", vr: VR::UI },
            E { tag: 0x20051053, alias: "SplitSeriesJobParams", vr: VR::CS },
            E { tag: 0x20051054, alias: "PreferredDimensionForSplitting", vr: VR::SS },
            E { tag: 0x20051055, alias: "ImageVelocityEncodingDirection", vr: VR::FD },
            E { tag: 0x20051056, alias: "ContrastBolusNoInjections", vr: VR::SS },
            E { tag: 0x20051057, alias: "ContrastBolusAgentCode", vr: VR::LT },
            E { tag: 0x20051058, alias: "ContrastBolusAdminRouteCode", vr: VR::LT },
            E { tag: 0x20051059, alias: "ContrastBolusVolume", vr: VR::DS },
            E { tag: 0x2005105a, alias: "ContrastBolusIngredientConcentration", vr: VR::DS },
            E { tag: 0x2005105b, alias: "ContrastBolusDynamicNumber", vr: VR::IS },
            E { tag: 0x2005105c, alias: "SeriesBolusContrast", vr: VR::SQ },
            E { tag: 0x2005105d, alias: "ContrastBolusID", vr: VR::IS },
            E { tag: 0x20051060, alias: "LUTtoRGBJobParams", vr: VR::CS },
            E { tag: 0x20051092, alias: "SpecificEnergyDose", vr: VR::FL },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 006",
        entries: &[
            E { tag: 0x20051053, alias: "MREFrequency", vr: VR::FL },
            E { tag: 0x20051054, alias: "MREAmplitude", vr: VR::FL },
            E { tag: 0x20051055, alias: "MREMEGFrequency", vr: VR::FL },
            E { tag: 0x20051056, alias: "MREMEGPairs", vr: VR::FL },
            E { tag: 0x20051057, alias: "MREMEGDirection", vr: VR::CS },
            E { tag: 0x20051058, alias: "MREMEGAmplitude", vr: VR::FL },
            E { tag: 0x20051059, alias: "MRENumberofPhaseDelays", vr: VR::FL },
            E { tag: 0x20051060, alias: "MRENumberofMotionCycles", vr: VR::IS },
            E { tag: 0x20051061, alias: "MREMotionMegPhaseDelay", vr: VR::FL },
            E { tag: 0x20051062, alias: "MREInversionAlgorithmVersion", vr: VR::LT },
            E { tag: 0x20051063, alias: "SagittalSliceOrder", vr: VR::CS },
            E { tag: 0x20051064, alias: "CoronalSliceOrder", vr: VR::CS },
            E { tag: 0x20051065, alias: "TransversalSliceOrder", vr: VR::CS },
            E { tag: 0x20051066, alias: "SeriesOrientation", vr: VR::CS },
            E { tag: 0x20051067, alias: "MRStackReverse", vr: VR::IS },
            E { tag: 0x20051068, alias: "MREPhaseDelayNumber", vr: VR::IS },
            E { tag: 0x20051071, alias: "NumberOfInversionDelays", vr: VR::IS },
            E { tag: 0x20051072, alias: "InversionDelayTime", vr: VR::FL },
            E { tag: 0x20051073, alias: "InversionDelayNumber", vr: VR::IS },
            E { tag: 0x20051074, alias: "MaxDbDt", vr: VR::DS },
            E { tag: 0x20051075, alias: "MaxSAR", vr: VR::DS },
            E { tag: 0x20051076, alias: "SARType", vr: VR::LT },
            E { tag: 0x20051078, alias: "MetalImplantStatus", vr: VR::CS },
            E { tag: 0x20051079, alias: "OrientationMirrorFlip", vr: VR::CS },
            E { tag: 0x20051081, alias: "SAROperationMode", vr: VR::CS },
            E { tag: 0x20051082, alias: "SpatialGradient", vr: VR::IS },
            E { tag: 0x20051083, alias: "AdditionalConstraints", vr: VR::LT },
            E { tag: 0x20051087, alias: "PIIM_MR_STUDY_B1RMS", vr: VR::DS },
            E { tag: 0x20051092, alias: "ContrastInformationSequence", vr: VR::SQ },
            E { tag: 0x20051095, alias: "Diffusion2KDTI", vr: VR::CS },
            E { tag: 0x20051096, alias: "DiffusionOrder", vr: VR::IS },
            E { tag: 0x20051097, alias: "IsJEditingSeries ", vr: VR::CS },
            E { tag: 0x20051098, alias: "MRSpectrumEditingType", vr: VR::SS },
            E { tag: 0x20051099, alias: "MRSeriesNrOfDiffOrder", vr: VR::SL },
        ],
    },
    P {
        creator: "SIEMENS MED",
        entries: &[
            E { tag: 0x70011010, alias: "Dummy", vr: VR::LT },
            E { tag: 0x70031010, alias: "Header", vr: VR::LT },
            E { tag: 0x70051010, alias: "Dummy", vr: VR::LT },
        ],
    },
    P {
        creator: "SIEMENS MED BRAIN ORIENTATION DATA",
        entries: &[
            E { tag: 0x00271006, alias: "Brain Orientation Value", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED MEASUREMENT",
        entries: &[
            E { tag: 0x00271001, alias: "Percist Cylinder Position", vr: VR::DS },
            E { tag: 0x00271002, alias: "Percist Cylinder Axis", vr: VR::DS },
            E { tag: 0x00271003, alias: "Percist Cylinder Radius", vr: VR::DS },
            E { tag: 0x00271004, alias: "Isocontour Threshold", vr: VR::LT },
            E { tag: 0x00271005, alias: "Auto Created", vr: VR::LO },
            E { tag: 0x00271006, alias: "Finding Creation Mode", vr: VR::CS },
            E { tag: 0x00271007, alias: "Pet Segmentation Threshold", vr: VR::DS },
            E { tag: 0x00271008, alias: "Change Rate", vr: VR::DS },
            E { tag: 0x00271009, alias: "Volume Doubling Time", vr: VR::DS },
            E { tag: 0x00271010, alias: "Change Rates", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS MED MI",
        entries: &[
            E { tag: 0x00671001, alias: "MI Scan ID", vr: VR::LT },
            E { tag: 0x00671002, alias: "Scanner Console Generation", vr: VR::LO },
            E { tag: 0x00671003, alias: "Recon Parameters", vr: VR::OB },
            E { tag: 0x00671004, alias: "Group Reconstruction ID", vr: VR::LO },
            E { tag: 0x00671005, alias: "Device IVK", vr: VR::LO },
            E { tag: 0x00671014, alias: "Raw Data Description", vr: VR::LO },
            E { tag: 0x00671016, alias: "Raw Data Series Instance UIDs", vr: VR::UI },
            E { tag: 0x00671017, alias: "Raw Data Referenced Series Instance UIDs", vr: VR::UI },
        ],
    },
    P {
        creator: "SIEMENS MED ORIENTATION RESULT",
        entries: &[
            E { tag: 0x00271005, alias: "Cardiac Orientation Value", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED PT",
        entries: &[
            E { tag: 0x00711021, alias: "Reference To Registration", vr: VR::UI },
            E { tag: 0x00711022, alias: "Decay Correction DateTime", vr: VR::DT },
            E { tag: 0x00711023, alias: "Registration Matrix", vr: VR::US },
            E { tag: 0x00711024, alias: "Table Motion", vr: VR::CS },
            E { tag: 0x00711025, alias: "Lumped Constant", vr: VR::FD },
            E { tag: 0x00711026, alias: "Histogramming Method", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED PT MU MAP",
        entries: &[
            E { tag: 0x00711001, alias: "SOP Class of Source", vr: VR::UI },
            E { tag: 0x00711002, alias: "Related Mu Map Series", vr: VR::UI },
        ],
    },
    P {
        creator: "SIEMENS MED RTSTRUCT",
        entries: &[
            E { tag: 0x00631032, alias: "GTV Marker Position", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MI RWVM SUV",
        entries: &[
            E { tag: 0x00411001, alias: "SUV Decay Correction Method", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS PET DERIVED",
        entries: &[
            E { tag: 0x00751001, alias: "Volume Index", vr: VR::US },
            E { tag: 0x00751002, alias: "Time Slice Duration", vr: VR::IS },
            E { tag: 0x00751003, alias: "Frame Reference Time Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "SIEMENS RA GEN",
        entries: &[
            E { tag: 0x00111026, alias: "PtopTotalSkinDose", vr: VR::SL },
            E { tag: 0x0019101f, alias: "DefaultTableIsoCenterHeight", vr: VR::SS },
            E { tag: 0x001910a2, alias: "SceneNumber", vr: VR::SL },
            E { tag: 0x001910a4, alias: "AcquisitionMode", vr: VR::SS },
            E { tag: 0x001910a5, alias: "AcquisitonFrameRate", vr: VR::SS },
            E { tag: 0x001910a6, alias: "ECGFlag", vr: VR::SL },
            E { tag: 0x001910a7, alias: "AdditionalSceneData", vr: VR::SL },
            E { tag: 0x001910a8, alias: "FileCopyFlag", vr: VR::SL },
            E { tag: 0x001910a9, alias: "PhlebovisionFlag", vr: VR::SL },
            E { tag: 0x001910aa, alias: "Co2Flag", vr: VR::SL },
            E { tag: 0x001910ab, alias: "MaxSpeed", vr: VR::SS },
            E { tag: 0x001910ac, alias: "StepWidth", vr: VR::SS },
            E { tag: 0x001910ad, alias: "DigitalAcquisitionZoom", vr: VR::SL },
            E { tag: 0x001910ff, alias: "Internal", vr: VR::SS },
            E { tag: 0x00211027, alias: "PlaneBImagesExist", vr: VR::SS },
            E { tag: 0x00211028, alias: "NoOf2MBChunks", vr: VR::SS },
            E { tag: 0x00211040, alias: "ArchiveSWInternalVersion", vr: VR::SS },
        ],
    },
    P {
        creator: "SIEMENS RA PLANE A",
        entries: &[
            E { tag: 0x00111028, alias: "FluoroTimerA", vr: VR::SL },
            E { tag: 0x00111029, alias: "FluoroSkinDoseA", vr: VR::SL },
            E { tag: 0x0011102a, alias: "TotalSkinDoseA", vr: VR::SL },
            E { tag: 0x0011102b, alias: "FluoroDoseAreaProductA", vr: VR::SL },
            E { tag: 0x0011102c, alias: "TotalDoseAreaProductA", vr: VR::SL },
            E { tag: 0x00191015, alias: "OfflineUID", vr: VR::LT },
            E { tag: 0x00191018, alias: "Internal", vr: VR::SS },
            E { tag: 0x00191019, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101a, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101b, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101c, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101d, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101e, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019101f, alias: "Internal", vr: VR::SS },
            E { tag: 0x0019102a, alias: "AcquisitionDelay", vr: VR::SS },
            E { tag: 0x001910ae, alias: "IIToCoverDistance", vr: VR::SL },
            E { tag: 0x001910b0, alias: "LastFramePhase1", vr: VR::SS },
            E { tag: 0x001910b1, alias: "FrameRatePhase1", vr: VR::SS },
            E { tag: 0x001910b2, alias: "LastFramePhase2", vr: VR::SS },
            E { tag: 0x001910b3, alias: "FrameRatePhase2", vr: VR::SS },
            E { tag: 0x001910b4, alias: "LastFramePhase3", vr: VR::SS },
            E { tag: 0x001910b5, alias: "FrameRatePhase3", vr: VR::SS },
            E { tag: 0x001910b6, alias: "LastFramePhase4", vr: VR::SS },
            E { tag: 0x001910b7, alias: "FrameRatePhase4", vr: VR::SS },
            E { tag: 0x001910b8, alias: "GammaOfNativeImage", vr: VR::SS },
            E { tag: 0x001910b9, alias: "GammaOfTVSystem", vr: VR::SS },
            E { tag: 0x001910bb, alias: "PixelshiftX", vr: VR::SL },
            E { tag: 0x001910bc, alias: "PixelshiftY", vr: VR::SL },
            E { tag: 0x001910bd, alias: "MaskAverageFactor", vr: VR::SL },
            E { tag: 0x001910be, alias: "BlankingCircleFlag", vr: VR::SL },
            E { tag: 0x001910bf, alias: "CircleRowStart", vr: VR::SL },
            E { tag: 0x001910c0, alias: "CircleRowEnd", vr: VR::SL },
            E { tag: 0x001910c1, alias: "CircleColumnStart", vr: VR::SL },
            E { tag: 0x001910c2, alias: "CircleColumnEnd", vr: VR::SL },
            E { tag: 0x001910c3, alias: "CircleDiameter", vr: VR::SL },
            E { tag: 0x001910c4, alias: "RectangularCollimaterFlag", vr: VR::SL },
            E { tag: 0x001910c5, alias: "RectangleRowStart", vr: VR::SL },
            E { tag: 0x001910c6, alias: "RectangleRowEnd", vr: VR::SL },
            E { tag: 0x001910c7, alias: "RectangleColumnStart", vr: VR::SL },
            E { tag: 0x001910c8, alias: "RectangleColumnEnd", vr: VR::SL },
            E { tag: 0x001910c9, alias: "RectangleAngulation", vr: VR::SL },
            E { tag: 0x001910ca, alias: "IrisCollimatorFlag", vr: VR::SL },
            E { tag: 0x001910cb, alias: "IrisRowStart", vr: VR::SL },
            E { tag: 0x001910cc, alias: "IrisRowEnd", vr: VR::SL },
            E { tag: 0x001910cd, alias: "IrisColumnStart", vr: VR::SL },
            E { tag: 0x001910ce, alias: "IrisColumnEnd", vr: VR::SL },
            E { tag: 0x001910cf, alias: "IrisAngulation", vr: VR::SL },
            E { tag: 0x001910d1, alias: "NumberOfFramesPlane", vr: VR::SS },
            E { tag: 0x001910d2, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d3, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d4, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d5, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d6, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d7, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d8, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910d9, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910da, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910db, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910dc, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910dd, alias: "AnatomicBackground", vr: VR::SL },
            E { tag: 0x001910de, alias: "AutoWindowBase", vr: VR::SL },
            E { tag: 0x001910df, alias: "Internal", vr: VR::SS },
            E { tag: 0x001910e0, alias: "Internal", vr: VR::SL },
        ],
    },
    P {
        creator: "SIEMENS RA PLANE B",
        entries: &[
            E { tag: 0x00111028, alias: "FluoroTimerB", vr: VR::SL },
            E { tag: 0x00111029, alias: "FluoroSkinDoseB", vr: VR::SL },
            E { tag: 0x0011102a, alias: "TotalSkinDoseB", vr: VR::SL },
            E { tag: 0x0011102b, alias: "FluoroDoseAreaProductB", vr: VR::SL },
            E { tag: 0x0011102c, alias: "TotalDoseAreaProductB", vr: VR::SL },
        ],
    },
    P {
        creator: "SIENET",
        entries: &[
            E { tag: 0x00991002, alias: "DataObjectAttributes", vr: VR::UL },
        ],
    },
    P {
        creator: "SONOWAND AS",
        entries: &[
            E { tag: 0x01351010, alias: "UltrasoundScannerName", vr: VR::LO },
            E { tag: 0x01351011, alias: "TransducerSerial", vr: VR::LO },
            E { tag: 0x01351012, alias: "ProbeApplication", vr: VR::LO },
        ],
    },
    P {
        creator: "SPI-P-Private-DCI Release 1",
        entries: &[
            E { tag: 0x00191010, alias: "ECGTimeMapDataBitsAllocated", vr: VR::UN },
            E { tag: 0x00191011, alias: "ECGTimeMapDataBitsStored", vr: VR::UN },
            E { tag: 0x00191012, alias: "ECGTimeMapDataHighBit", vr: VR::UN },
            E { tag: 0x00191013, alias: "ECGTimeMapDataRepresentation", vr: VR::UN },
            E { tag: 0x00191014, alias: "ECGTimeMapDataSmallestDataValue", vr: VR::UN },
            E { tag: 0x00191015, alias: "ECGTimeMapDataLargestDataValue", vr: VR::UN },
            E { tag: 0x00191016, alias: "ECGTimeMapDataNumberOfDataValues", vr: VR::UN },
            E { tag: 0x00191017, alias: "ECGTimeMapData", vr: VR::UN },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1",
        entries: &[
            E { tag: 0x00291043, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291044, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "SVISION",
        entries: &[
            E { tag: 0x00251021, alias: "Unknown", vr: VR::UI },
        ],
    },
    // pydicom private dictionary
    P {
        creator: "1.2.840.113663.1",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "1.2.840.113681",
        entries: &[
            E { tag: 0x00191010, alias: "CR Image Params Common", vr: VR::ST },
            E { tag: 0x00191011, alias: "CR Image IP Params Single", vr: VR::ST },
            E { tag: 0x00191012, alias: "CR Image IP Params Left", vr: VR::ST },
            E { tag: 0x00191013, alias: "CR Image IP Params Right", vr: VR::ST },
        ],
    },
    P {
        creator: "1.2.840.113708.794.1.1.2.0",
        entries: &[
            E { tag: 0x00871010, alias: "Media Type", vr: VR::CS },
            E { tag: 0x00871020, alias: "Media Location", vr: VR::CS },
            E { tag: 0x00871030, alias: "Storage File ID", vr: VR::ST },
            E { tag: 0x00871040, alias: "Study or Image Size in MB", vr: VR::DS },
            E { tag: 0x00871050, alias: "Estimated Retrieve Time", vr: VR::IS },
        ],
    },
    P {
        creator: "2.16.840.1.114059.1.1.6.1.50.1",
        entries: &[
            E { tag: 0x00291020, alias: "Description", vr: VR::LT },
            E { tag: 0x00291021, alias: "Orientation", vr: VR::ST },
            E { tag: 0x00291022, alias: "Parameter 1", vr: VR::ST },
            E { tag: 0x00291023, alias: "Parameter 2", vr: VR::ST },
            E { tag: 0x00291024, alias: "Teeth", vr: VR::LO },
            E { tag: 0x00291025, alias: "Jaw", vr: VR::LO },
            E { tag: 0x00291026, alias: "Quadrant", vr: VR::LO },
            E { tag: 0x00291027, alias: "CRC", vr: VR::LO },
        ],
    },
    P {
        creator: "A.L.I. Technologies, Inc.",
        entries: &[
            E { tag: 0x37111001, alias: "Filename", vr: VR::LO },
            E { tag: 0x37111002, alias: "Data Blob of a Visit", vr: VR::OB },
            E { tag: 0x37111003, alias: "Revision Number", vr: VR::US },
            E { tag: 0x37111004, alias: "Unix Timestamp", vr: VR::UL },
            E { tag: 0x37111005, alias: "Bag ID", vr: VR::IS },
            E { tag: 0x3711100C, alias: "Original Study UID", vr: VR::UI },
            E { tag: 0x3711100D, alias: "Overlay Grayscale Value", vr: VR::US },
            E { tag: 0x3711100E, alias: "Anonymization Status", vr: VR::CS },
        ],
    },
    P {
        creator: "ACUSON",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091001, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091002, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091003, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091004, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091005, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091006, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091007, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091008, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00091009, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0009100A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009100B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009100C, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009100D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009100E, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009100F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091010, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091011, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091013, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091014, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00091015, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "ACUSON: 1.2.840.11386.1.0",
        entries: &[
            E { tag: 0x7FDF1000, alias: "Lossy Compression Ratio", vr: VR::IS },
            E { tag: 0x7FDF1001, alias: "Image Format", vr: VR::US },
            E { tag: 0x7FDF1002, alias: "Acuson Region Type", vr: VR::US },
            E { tag: 0x7FDF100B, alias: "Acuson Image Apex X", vr: VR::UL },
            E { tag: 0x7FDF100C, alias: "Acuson Image Apex Y", vr: VR::UL },
            E { tag: 0x7FDF100D, alias: "B-Color-On Flag", vr: VR::IS },
        ],
    },
    P {
        creator: "ACUSON:1.2.840.113680.1.0:0921",
        entries: &[
            E { tag: 0x00091020, alias: "View Name", vr: VR::UN },
            E { tag: 0x0009102A, alias: "View List", vr: VR::UN },
        ],
    },
    P {
        creator: "ACUSON:1.2.840.113680.1.0:7f10",
        entries: &[
            E { tag: 0x7FDF1000, alias: "Lossy Compression Ratio", vr: VR::UN },
            E { tag: 0x7FDF1001, alias: "Image Format", vr: VR::UN },
            E { tag: 0x7FDF1002, alias: "Acuson Region Type", vr: VR::UN },
            E { tag: 0x7FDF100B, alias: "Acuson Image Apex X", vr: VR::UN },
            E { tag: 0x7FDF100C, alias: "Acuson Image Apex Y", vr: VR::UN },
            E { tag: 0x7FDF100D, alias: "B-Color-On Flag", vr: VR::UN },
            E { tag: 0x7FDF100E, alias: "Acuson Mechanical Apex X", vr: VR::UN },
            E { tag: 0x7FDF100F, alias: "Acuson Mechanical Apex Y", vr: VR::UN },
            E { tag: 0x7FDF1010, alias: "Acquisition Type:", vr: VR::UN },
            E { tag: 0x7FDF1018, alias: "Transformation Matrix Sequence", vr: VR::UN },
            E { tag: 0x7FDF1020, alias: "Left angle", vr: VR::UN },
            E { tag: 0x7FDF1022, alias: "Right angle", vr: VR::UN },
            E { tag: 0x7FDF1024, alias: "Color Map Family", vr: VR::UN },
            E { tag: 0x7FDF1025, alias: "Full Colormap.", vr: VR::UN },
            E { tag: 0x7FDF1026, alias: "Color Invert", vr: VR::UN },
            E { tag: 0x7FDF1027, alias: "Color Baseline", vr: VR::UN },
            E { tag: 0x7FDF1028, alias: "CD Color Mix Points X1", vr: VR::UN },
            E { tag: 0x7FDF1029, alias: "CD Color Mix Points Y1", vr: VR::UN },
            E { tag: 0x7FDF102A, alias: "CD Color Mix Points X2", vr: VR::UN },
            E { tag: 0x7FDF102B, alias: "CD Color Mix Points Y2", vr: VR::UN },
            E { tag: 0x7FDF102C, alias: "Color Accent", vr: VR::UN },
            E { tag: 0x7FDF1030, alias: "Persistence SQ", vr: VR::UN },
            E { tag: 0x7FDF1031, alias: "Persistence Mode", vr: VR::UN },
            E { tag: 0x7FDF1032, alias: "Persistence Coefficient Mode", vr: VR::UN },
            E { tag: 0x7FDF1033, alias: "Alpha coefficient", vr: VR::UN },
            E { tag: 0x7FDF1034, alias: "Gamma coefficient", vr: VR::UN },
            E { tag: 0x7FDF1035, alias: "Persistence Time Flag", vr: VR::UN },
            E { tag: 0x7FDF1036, alias: "Persistence adaptive flag", vr: VR::UN },
            E { tag: 0x7FDF1037, alias: "Persistence Frame Rate", vr: VR::UN },
            E { tag: 0x7FDF1038, alias: "Persistence ID", vr: VR::UN },
            E { tag: 0x7FDF1040, alias: "Observation Date Time SQ", vr: VR::UN },
            E { tag: 0x7FDF1050, alias: "Capture Type Name", vr: VR::UN },
            E { tag: 0x7FDF1052, alias: "Capture Type Number", vr: VR::UN },
            E { tag: 0x7FDF1054, alias: "Number of Capture Types", vr: VR::UN },
            E { tag: 0x7FDF1060, alias: "CD Steering Angle", vr: VR::UN },
            E { tag: 0x7FDF1061, alias: "CD PRI", vr: VR::UN },
            E { tag: 0x7FDF1062, alias: "CD Dynamic Range", vr: VR::UN },
            E { tag: 0x7FDF1063, alias: "CD Velocity Scale Min", vr: VR::UN },
            E { tag: 0x7FDF1064, alias: "CD Velocity Scale Max", vr: VR::UN },
            E { tag: 0x7FDF1065, alias: "CD Color Mode", vr: VR::UN },
            E { tag: 0x7FDF1066, alias: "CD Frequency", vr: VR::UN },
            E { tag: 0x7FDF1067, alias: "CD Balance", vr: VR::UN },
            E { tag: 0x7FDF1068, alias: "CD Delta", vr: VR::UN },
            E { tag: 0x7FDF1069, alias: "CD Pan Box Min X0", vr: VR::UN },
            E { tag: 0x7FDF106A, alias: "CD Pan Box Min Y0", vr: VR::UN },
            E { tag: 0x7FDF106B, alias: "CD Pan Box Min X1", vr: VR::UN },
            E { tag: 0x7FDF106C, alias: "CD Pan Box Min Y1", vr: VR::UN },
            E { tag: 0x7FDF106D, alias: "CPS Map Type", vr: VR::UN },
            E { tag: 0x7FDF106E, alias: "CPS Map Data", vr: VR::UN },
            E { tag: 0x7FDF106F, alias: "CPS Balance Setting", vr: VR::UN },
            E { tag: 0x7FDF1070, alias: "3DCard Step Angle", vr: VR::UN },
            E { tag: 0x7FDF1071, alias: "3DCard Xdcr Angle", vr: VR::UN },
            E { tag: 0x7FDF1072, alias: "B-mode Frequency", vr: VR::UN },
            E { tag: 0x7FDF1073, alias: "B-mode Dynamic Range", vr: VR::UN },
            E { tag: 0x7FDF1074, alias: "B-mode Frame Rate", vr: VR::UN },
            E { tag: 0x7FDF1075, alias: "B-mode Space Time", vr: VR::UN },
            E { tag: 0x7FDF1076, alias: "B-mode Persistence", vr: VR::UN },
            E { tag: 0x7FDF1077, alias: "B-mode Display Depth Start", vr: VR::UN },
            E { tag: 0x7FDF1078, alias: "B-mode Display Depth End", vr: VR::UN },
            E { tag: 0x7FDF1079, alias: "B-mode Res Mode", vr: VR::UN },
            E { tag: 0x7FDF107A, alias: "B-mode Preset Application", vr: VR::UN },
            E { tag: 0x7FDF107B, alias: "Image Spec Name", vr: VR::UN },
            E { tag: 0x7FDF107C, alias: "B Preset Image Look", vr: VR::UN },
            E { tag: 0x7FDF107D, alias: "B-mode Post Processing", vr: VR::UN },
            E { tag: 0x7FDF107E, alias: "B Edge", vr: VR::UN },
            E { tag: 0x7FDF107F, alias: "B Delta", vr: VR::UN },
            E { tag: 0x7FDF1080, alias: "B-mode 1D Post Processing Curve", vr: VR::UN },
            E { tag: 0x7FDF1081, alias: "B-mode Delta (ECRI) Map Diagonal", vr: VR::UN },
            E { tag: 0x7FDF1082, alias: "Bytes Per Timestamp", vr: VR::UN },
            E { tag: 0x7FDF1083, alias: "Microseconds in unit timestamp", vr: VR::UN },
            E { tag: 0x7FDF1084, alias: "Start Stopwatch Timestamp", vr: VR::UN },
            E { tag: 0x7FDF1085, alias: "Acoustic Frame Timestamp", vr: VR::UN },
            E { tag: 0x7FDF1086, alias: "R-Wave Timestamp", vr: VR::UN },
            E { tag: 0x7FDF1087, alias: "Last Destruction Timestamp", vr: VR::UN },
            E { tag: 0x7FDF1088, alias: "Pixels Per Second", vr: VR::UN },
            E { tag: 0x7FDF1089, alias: "ECG Reference Timestamp", vr: VR::UN },
            E { tag: 0x7FDF108A, alias: "ECG Sampling Interval (milliseconds)", vr: VR::UN },
            E { tag: 0x7FDF108B, alias: "ECG Sample Count", vr: VR::UN },
            E { tag: 0x7FDF108C, alias: "ECG Sample Size", vr: VR::UN },
            E { tag: 0x7FDF108D, alias: "ECG Data Value", vr: VR::UN },
            E { tag: 0x7FDF108E, alias: "Contrast/Active Image Indicator", vr: VR::UN },
            E { tag: 0x7FDF108F, alias: "Live Dual Mode Indicator", vr: VR::UN },
            E { tag: 0x7FDF1090, alias: "3DCard Clipset ID", vr: VR::UN },
            E { tag: 0x7FDF1091, alias: "3DCard HRWave Min", vr: VR::UN },
            E { tag: 0x7FDF1092, alias: "3DCard HRWave Max", vr: VR::UN },
            E { tag: 0x7FDF1093, alias: "Perspective Capture Type", vr: VR::UN },
            E { tag: 0x7FDF10F1, alias: "Trigger Mask.", vr: VR::UN },
            E { tag: 0x7FDF10F2, alias: "Study Directory", vr: VR::UN },
            E { tag: 0x7FDF10F3, alias: "Last Modify Date", vr: VR::UN },
            E { tag: 0x7FDF10F4, alias: "Last Modify Time", vr: VR::UN },
            E { tag: 0x7FDF10F5, alias: "Teaching Study", vr: VR::UN },
            E { tag: 0x7FDF10F6, alias: "Series Base UID", vr: VR::UN },
        ],
    },
    P {
        creator: "ACUSON:1.2.840.113680.1.0:7ffe",
        entries: &[
            E { tag: 0x7FDF1000, alias: "Data Padding", vr: VR::UN },
        ],
    },
    P {
        creator: "ADAC_IMG",
        entries: &[
            E { tag: 0x00191002, alias: "Ver200 ADAC Pegasys File Size", vr: VR::IS },
            E { tag: 0x00191010, alias: "ADAC Header Signature", vr: VR::LO },
            E { tag: 0x00191011, alias: "Number of ADAC Headers", vr: VR::US },
            E { tag: 0x00191012, alias: "ADAC Header/Image Sizes", vr: VR::IS },
            E { tag: 0x00191020, alias: "ADAC Pegasys Headers", vr: VR::OB },
            E { tag: 0x00191021, alias: "Ver200 Number of ADAC Headers", vr: VR::US },
            E { tag: 0x00191041, alias: "Ver200 ADAC Header/Image Size", vr: VR::IS },
            E { tag: 0x00191061, alias: "Ver200 ADAC Pegasys Headers", vr: VR::OB },
            E { tag: 0x70431000, alias: "Cardiac Stress State", vr: VR::SH },
            E { tag: 0x70431010, alias: "Philips NM Private Group", vr: VR::LO },
        ],
    },
    P {
        creator: "AEGIS_DICOM_2.00",
        entries: &[
            E { tag: 0x00031000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00051000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291000, alias: "Unknown", vr: VR::US },
            E { tag: 0x13691000, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "AGFA",
        entries: &[
            E { tag: 0x00091010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091011, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091013, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091014, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091015, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191010, alias: "Private Identification Code", vr: VR::SH },
            E { tag: 0x00191011, alias: "Identification Data (Note 2)", vr: VR::LO },
            E { tag: 0x00191013, alias: "Sensitometry Name", vr: VR::LO },
            E { tag: 0x00191014, alias: "Window/Level List (Note 3)", vr: VR::ST },
            E { tag: 0x00191015, alias: "Dose Monitoring List", vr: VR::LO },
            E { tag: 0x00191016, alias: "Other Info (Note 5)", vr: VR::LO },
            E { tag: 0x0019101A, alias: "Clipped Exposure Deviation", vr: VR::LO },
            E { tag: 0x0019101B, alias: "Logarithmic PLT Full Scale", vr: VR::LO },
            E { tag: 0x00191060, alias: "Total number of series", vr: VR::US },
            E { tag: 0x00191061, alias: "Session Number", vr: VR::SH },
            E { tag: 0x00191062, alias: "ID Station name", vr: VR::SH },
            E { tag: 0x00191065, alias: "Number of images in study to be transmitted (only sent with autoverify: on)", vr: VR::US },
            E { tag: 0x00191070, alias: "Total number of images", vr: VR::US },
            E { tag: 0x00191080, alias: "Geometrical Transformations", vr: VR::ST },
            E { tag: 0x00191081, alias: "Roam Origin", vr: VR::ST },
            E { tag: 0x00191082, alias: "Zoom factor", vr: VR::US },
            E { tag: 0x00191093, alias: "Status", vr: VR::CS },
        ],
    },
    P {
        creator: "AGFA PACS Archive Mirroring 1.0",
        entries: &[
            E { tag: 0x00311000, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00311001, alias: "Unknown", vr: VR::UL },
        ],
    },
    P {
        creator: "AGFA-AG_HPState",
        entries: &[
            E { tag: 0x00711018, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00711019, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0071101A, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0071101C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0071101E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00711020, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00711021, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00711022, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00711023, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00711024, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00731023, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00731024, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00731028, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00731080, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00751010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00871001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00871002, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "AGFA_ADC_Compact",
        entries: &[
            E { tag: 0x00191005, alias: "Data stream from cassette", vr: VR::ST },
            E { tag: 0x00191010, alias: "Private Identification Code", vr: VR::LO },
            E { tag: 0x00191030, alias: "Set of destination types", vr: VR::ST },
            E { tag: 0x00191040, alias: "Set of destination Ids", vr: VR::ST },
            E { tag: 0x00191050, alias: "Set of processing codes", vr: VR::ST },
            E { tag: 0x00191060, alias: "Number of series in study", vr: VR::US },
            E { tag: 0x00191061, alias: "Session Number", vr: VR::US },
            E { tag: 0x00191062, alias: "ID station name", vr: VR::SH },
            E { tag: 0x00191070, alias: "Number of images in series", vr: VR::US },
            E { tag: 0x00191071, alias: "Break condition", vr: VR::US },
            E { tag: 0x00191072, alias: "Wait (or Hold) flag", vr: VR::US },
            E { tag: 0x00191073, alias: "ScanRes flag", vr: VR::US },
            E { tag: 0x00191074, alias: "Operation code", vr: VR::SH },
            E { tag: 0x00191095, alias: "Image quality", vr: VR::CS },
        ],
    },
    P {
        creator: "ALOKA:1.2.392.200039.103.2",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00091004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091006, alias: "Unknown", vr: VR::US },
            E { tag: 0x0009100A, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00091020, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00091022, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00091024, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00091026, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00091028, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0009102A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00091030, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00091032, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00091034, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019100C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019100E, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191018, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019101A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00191046, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00191052, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191054, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191056, alias: "Unknown", vr: VR::FD },
        ],
    },
    P {
        creator: "AMI Annotations_01",
        entries: &[
            E { tag: 0x31011010, alias: "AMI Annotation Sequence (RET)", vr: VR::SQ },
        ],
    },
    P {
        creator: "AMI Annotations_02",
        entries: &[
            E { tag: 0x31011020, alias: "AMI Annotation Sequence (RET)", vr: VR::SQ },
        ],
    },
    P {
        creator: "AMI ImageContextExt_01",
        entries: &[
            E { tag: 0x310710A0, alias: "AMI Window Function (RET)", vr: VR::CS },
            E { tag: 0x310710B0, alias: "AMI Window Slope (RET)", vr: VR::DS },
        ],
    },
    P {
        creator: "AMI ImageContext_01",
        entries: &[
            E { tag: 0x31091010, alias: "AMI Window Invert (RET)", vr: VR::CS },
            E { tag: 0x31091020, alias: "AMI Window Center (RET)", vr: VR::IS },
            E { tag: 0x31091030, alias: "AMI Window Widith (RET)", vr: VR::IS },
            E { tag: 0x31091040, alias: "AMI Pixel Aspect Ratio Swap (RET)", vr: VR::CS },
            E { tag: 0x31091050, alias: "AMI Enable Averaging (RET)", vr: VR::CS },
            E { tag: 0x31091060, alias: "AMI Quality (RET)", vr: VR::CS },
            E { tag: 0x31091070, alias: "AMI Viewport Annotation Level (RET)", vr: VR::CS },
            E { tag: 0x31091080, alias: "AMI Show Image Annotation (RET)", vr: VR::CS },
            E { tag: 0x31091090, alias: "AMI Show Image Overlay (RET)", vr: VR::CS },
        ],
    },
    P {
        creator: "AMI ImageTransform_01",
        entries: &[
            E { tag: 0x31071010, alias: "AMI Transformation Matrix (RET)", vr: VR::DS },
            E { tag: 0x31071020, alias: "AMI Center Offset (RET)", vr: VR::DS },
            E { tag: 0x31071030, alias: "AMI Magnification (RET)", vr: VR::DS },
            E { tag: 0x31071040, alias: "AMI Magnification Type (RET)", vr: VR::CS },
            E { tag: 0x31071050, alias: "AMI Displayed Area (RET)", vr: VR::DS },
            E { tag: 0x31071060, alias: "AMI Calibration Factor (RET)", vr: VR::DS },
        ],
    },
    P {
        creator: "AMI Sequence AnnotElements_01",
        entries: &[
            E { tag: 0x31051010, alias: "AMI Annotation Element Position", vr: VR::DS },
            E { tag: 0x31051020, alias: "AMI Annotation Element Text", vr: VR::LT },
        ],
    },
    P {
        creator: "AMI Sequence Annotations_01",
        entries: &[
            E { tag: 0x31031010, alias: "AMI Annotation Sequence (RET)", vr: VR::CS },
            E { tag: 0x31031020, alias: "AMI Annotation UID (RET)", vr: VR::UI },
            E { tag: 0x31031030, alias: "AMI Annotation Color (RET)", vr: VR::US },
            E { tag: 0x31031040, alias: "FontSize", vr: VR::FD },
            E { tag: 0x31031050, alias: "AMI Annotation Line Style (RET)", vr: VR::CS },
            E { tag: 0x31031060, alias: "AMI Annotation Elements (RET)", vr: VR::SQ },
            E { tag: 0x31031070, alias: "AMI Annotation Label (RET)", vr: VR::SH },
            E { tag: 0x31031080, alias: "AMI Annotation Creator (RET)", vr: VR::PN },
            E { tag: 0x31031090, alias: "AMI Annotation Modifiers (RET)", vr: VR::PN },
            E { tag: 0x310310A0, alias: "AMI Annotation Creation Date (RET)", vr: VR::DA },
            E { tag: 0x310310B0, alias: "AMI Annotation Creation Time (RET)", vr: VR::TM },
            E { tag: 0x310310C0, alias: "AMI Annotation Modification Dates (RET)", vr: VR::DA },
            E { tag: 0x310310D0, alias: "AMI Annotation Modification Times (RET)", vr: VR::TM },
            E { tag: 0x310310E0, alias: "AMI Annotation Frame Number (RET)", vr: VR::US },
        ],
    },
    P {
        creator: "AMI Sequence Annotations_02",
        entries: &[
            E { tag: 0x31031010, alias: "AMI Annotation Sequence (RET)", vr: VR::CS },
            E { tag: 0x31031020, alias: "AMI Annotation UID (RET)", vr: VR::UI },
            E { tag: 0x31031030, alias: "AMI Annotation Color (RET)", vr: VR::US },
            E { tag: 0x31031050, alias: "AMI Annotation Line Style (RET)", vr: VR::CS },
            E { tag: 0x31031060, alias: "AMI Annotation Elements (RET)", vr: VR::SQ },
            E { tag: 0x31031070, alias: "AMI Annotation Label (RET)", vr: VR::SH },
            E { tag: 0x31031080, alias: "AMI Annotation Creator (RET)", vr: VR::PN },
            E { tag: 0x31031090, alias: "AMI Annotation Modifiers (RET)", vr: VR::PN },
            E { tag: 0x310310A0, alias: "AMI Annotation Creation Date (RET)", vr: VR::DA },
            E { tag: 0x310310B0, alias: "AMI Annotation Creation Time (RET)", vr: VR::TM },
            E { tag: 0x310310C0, alias: "AMI Annotation Modification Dates (RET)", vr: VR::DA },
            E { tag: 0x310310D0, alias: "AMI Annotation Modification Times (RET)", vr: VR::TM },
            E { tag: 0x310310E0, alias: "AMI Annotation Frame Number (RET)", vr: VR::US },
        ],
    },
    P {
        creator: "AMI StudyExtensions_01",
        entries: &[
            E { tag: 0x31111001, alias: "AMI Last Released Annot Label (RET)", vr: VR::UL },
        ],
    },
    P {
        creator: "AMICAS0",
        entries: &[
            E { tag: 0x00231001, alias: "", vr: VR::UI },
            E { tag: 0x00231008, alias: "", vr: VR::US },
            E { tag: 0x00231010, alias: "", vr: VR::US },
            E { tag: 0x00231016, alias: "", vr: VR::SL },
        ],
    },
    P {
        creator: "APEX_PRIVATE",
        entries: &[
            E { tag: 0x00271010, alias: "Private Creator", vr: VR::LO },
            E { tag: 0x00271011, alias: "Bed Position", vr: VR::DS },
        ],
    },
    P {
        creator: "ATL HDI V1.0",
        entries: &[
            E { tag: 0x00091000, alias: "Private", vr: VR::UN },
            E { tag: 0x00091010, alias: "Private", vr: VR::UN },
            E { tag: 0x00091020, alias: "Private", vr: VR::UN },
            E { tag: 0x00091030, alias: "Private", vr: VR::UN },
            E { tag: 0x00091040, alias: "Private", vr: VR::UN },
            E { tag: 0x00091050, alias: "Private", vr: VR::UN },
            E { tag: 0x00091060, alias: "Private", vr: VR::UN },
            E { tag: 0x00091070, alias: "Private", vr: VR::UN },
            E { tag: 0x00091080, alias: "Private", vr: VR::UN },
            E { tag: 0x00091090, alias: "Private", vr: VR::UN },
            E { tag: 0x00091091, alias: "Private", vr: VR::UN },
            E { tag: 0x00291030, alias: "Loop Mode", vr: VR::UN },
            E { tag: 0x00291031, alias: "Trigger mode", vr: VR::UN },
            E { tag: 0x00291032, alias: "Number of Loops", vr: VR::UN },
            E { tag: 0x00291033, alias: "Loop Indexes", vr: VR::UN },
            E { tag: 0x00291034, alias: "Loop Heart Rates", vr: VR::UN },
            E { tag: 0x00291035, alias: "Medications", vr: VR::UN },
        ],
    },
    P {
        creator: "ATL PRIVATE TAGS",
        entries: &[
            E { tag: 0x00291030, alias: "Loop Mode", vr: VR::UL },
            E { tag: 0x00291031, alias: "Trigger mode", vr: VR::UL },
            E { tag: 0x00291032, alias: "Number of Loops", vr: VR::UL },
            E { tag: 0x00291033, alias: "Loop Indexes", vr: VR::DS },
            E { tag: 0x00291034, alias: "Loop Heart Rates", vr: VR::DS },
            E { tag: 0x00291035, alias: "Medications", vr: VR::LO },
        ],
    },
    P {
        creator: "Acuson X500",
        entries: &[
            E { tag: 0x00091020, alias: "(a)View Name", vr: VR::UN },
            E { tag: 0x0009102A, alias: "View List", vr: VR::UN },
            E { tag: 0x00111010, alias: "Siemens Medical", vr: VR::UN },
            E { tag: 0x00111011, alias: "DIMAQ Software", vr: VR::UN },
            E { tag: 0x00111020, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00111021, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00131010, alias: "Siemens Medical", vr: VR::UN },
            E { tag: 0x00131011, alias: "DIMAQ Software", vr: VR::UN },
            E { tag: 0x00131020, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00151010, alias: "Siemens Medical", vr: VR::UN },
            E { tag: 0x00151011, alias: "DIMAQ Software", vr: VR::UN },
            E { tag: 0x00151020, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00171010, alias: "Siemens Medical", vr: VR::UN },
            E { tag: 0x00171011, alias: "DIMAQ Software", vr: VR::UN },
            E { tag: 0x00171020, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00191020, alias: "Import Structured", vr: VR::UN },
        ],
    },
    P {
        creator: "Agfa ADC NX",
        entries: &[
            E { tag: 0x00191009, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001910F5, alias: "Cassette Orientation", vr: VR::CS },
            E { tag: 0x001910F6, alias: "Plate Sensitivity", vr: VR::DS },
            E { tag: 0x001910F7, alias: "Plate Erasability", vr: VR::DS },
            E { tag: 0x001910F8, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910FE, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "AgilityRuntime",
        entries: &[
            E { tag: 0x00291011, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291012, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291013, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291014, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029101F, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "Applicare/Centricity Radiology Web/Version 1.0",
        entries: &[
            E { tag: 0x41091001, alias: "Mammography Laterality", vr: VR::SH },
            E { tag: 0x41091002, alias: "Mammography View Name", vr: VR::SH },
            E { tag: 0x41091003, alias: "Mammography View Modifier", vr: VR::SH },
        ],
    },
    P {
        creator: "Applicare/Centricity Radiology Web/Version 2.0",
        entries: &[
            E { tag: 0x41111001, alias: "Secondary Spine Label", vr: VR::CS },
            E { tag: 0x41111002, alias: "Additional tags for Presentation State", vr: VR::IS },
        ],
    },
    P {
        creator: "Applicare/Print/Version 5.1",
        entries: &[
            E { tag: 0x41011001, alias: "<internal", vr: VR::UL },
            E { tag: 0x41011002, alias: "<internal", vr: VR::SQ },
            E { tag: 0x41011003, alias: "<internal", vr: VR::LO },
            E { tag: 0x41011004, alias: "<internal", vr: VR::UL },
            E { tag: 0x41011005, alias: "<internal", vr: VR::FD },
            E { tag: 0x41011006, alias: "<internal", vr: VR::US },
            E { tag: 0x41011007, alias: "<internal", vr: VR::US },
            E { tag: 0x41011008, alias: "<internal", vr: VR::US },
            E { tag: 0x41011009, alias: "<internal", vr: VR::US },
        ],
    },
    P {
        creator: "Applicare/RadStore/Version 1.0",
        entries: &[
            E { tag: 0x31131001, alias: "OBSOLETE", vr: VR::SL },
            E { tag: 0x31131002, alias: "Id1", vr: VR::SL },
            E { tag: 0x31131003, alias: "Id2", vr: VR::SL },
            E { tag: 0x31131004, alias: "Id3", vr: VR::SL },
            E { tag: 0x31131011, alias: "OBSOLETE", vr: VR::LO },
            E { tag: 0x31131012, alias: "State", vr: VR::CS },
            E { tag: 0x31131013, alias: "DateLastModified", vr: VR::DT },
            E { tag: 0x31131014, alias: "DateLastAccessed", vr: VR::DT },
            E { tag: 0x31131015, alias: "OBSOLETE", vr: VR::CS },
            E { tag: 0x31131016, alias: "ByteSize", vr: VR::FD },
            E { tag: 0x31131017, alias: "LibraryId", vr: VR::LO },
            E { tag: 0x31131018, alias: "Pathnames", vr: VR::LO },
            E { tag: 0x31131019, alias: "DriverPath", vr: VR::LO },
            E { tag: 0x3113101A, alias: "Source", vr: VR::LO },
            E { tag: 0x3113101B, alias: "Destination", vr: VR::LO },
            E { tag: 0x3113101C, alias: "MediumId", vr: VR::SL },
            E { tag: 0x3113101D, alias: "ArchiveId", vr: VR::LO },
            E { tag: 0x3113101E, alias: "Origin", vr: VR::LO },
            E { tag: 0x31131021, alias: "Version", vr: VR::SL },
            E { tag: 0x31131022, alias: "OBSOLETE", vr: VR::SL },
            E { tag: 0x31131023, alias: "InstanceFileLocation", vr: VR::ST },
            E { tag: 0x31131031, alias: "OBSOLETE", vr: VR::IS },
            E { tag: 0x31131032, alias: "OBSOLETE", vr: VR::IS },
            E { tag: 0x31131033, alias: "OBSOLETE", vr: VR::IS },
            E { tag: 0x31131035, alias: "ImageMediumLocation", vr: VR::LO },
            E { tag: 0x31131036, alias: "ImageMediumLabel", vr: VR::LO },
            E { tag: 0x31131037, alias: "ImageMediumState", vr: VR::CS },
            E { tag: 0x31131038, alias: "SeriesMediumLocation", vr: VR::LO },
            E { tag: 0x31131039, alias: "SeriesMediumLabel", vr: VR::LO },
            E { tag: 0x3113103A, alias: "SeriesMediumState", vr: VR::CS },
            E { tag: 0x3113103B, alias: "StudyMediumLocation", vr: VR::LO },
            E { tag: 0x3113103C, alias: "StudyMediumLabel", vr: VR::LO },
            E { tag: 0x3113103D, alias: "StudyMediumState", vr: VR::CS },
            E { tag: 0x31131052, alias: "StudyState", vr: VR::CS },
            E { tag: 0x31131053, alias: "SeriesState", vr: VR::CS },
            E { tag: 0x31131055, alias: "ImageStateText", vr: VR::CS },
            E { tag: 0x31131056, alias: "SeriesStateText", vr: VR::CS },
            E { tag: 0x31131057, alias: "StudyStateText", vr: VR::CS },
            E { tag: 0x31131060, alias: "Expiration", vr: VR::DT },
            E { tag: 0x31131069, alias: "DeletedTags", vr: VR::AT },
        ],
    },
    P {
        creator: "Applicare/RadWorks/Version 5.0",
        entries: &[
            E { tag: 0x31091001, alias: "<internal", vr: VR::ST },
            E { tag: 0x31091002, alias: "NEW/SEEN Status", vr: VR::SH },
            E { tag: 0x31091003, alias: "Delete Lock", vr: VR::CS },
            E { tag: 0x31091004, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091005, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091006, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091007, alias: "<internal", vr: VR::UL },
            E { tag: 0x31091008, alias: "Receive Origin", vr: VR::LO },
            E { tag: 0x31091009, alias: "Folder", vr: VR::LO },
            E { tag: 0x3109100A, alias: "Receive Date", vr: VR::DA },
            E { tag: 0x3109100B, alias: "Receive Time", vr: VR::TM },
            E { tag: 0x3109100C, alias: "Prior", vr: VR::CS },
            E { tag: 0x3109100D, alias: "STAT Study", vr: VR::CS },
            E { tag: 0x3109100E, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091010, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091011, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091012, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091013, alias: "<internal", vr: VR::DA },
            E { tag: 0x31091014, alias: "<internal", vr: VR::TM },
            E { tag: 0x31091019, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091020, alias: "<internal", vr: VR::UI },
            E { tag: 0x31091021, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091022, alias: "<internal", vr: VR::AE },
            E { tag: 0x31091023, alias: "<internal", vr: VR::US },
            E { tag: 0x31091024, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091025, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091026, alias: "<internal", vr: VR::UI },
            E { tag: 0x31091027, alias: "<internal", vr: VR::SQ },
            E { tag: 0x31091028, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091029, alias: "<internal", vr: VR::LO },
            E { tag: 0x3109102A, alias: "Send Flag", vr: VR::LO },
            E { tag: 0x3109102B, alias: "Print Flag", vr: VR::LO },
            E { tag: 0x3109102C, alias: "Archive Flag", vr: VR::LO },
            E { tag: 0x31091030, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091031, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091032, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091033, alias: "Request Storage Commitment", vr: VR::CS },
            E { tag: 0x31091034, alias: "Requested Compression", vr: VR::CS },
            E { tag: 0x31091035, alias: "<internal", vr: VR::SQ },
            E { tag: 0x31091036, alias: "<internal", vr: VR::LO },
            E { tag: 0x31091037, alias: "<internal", vr: VR::UI },
            E { tag: 0x31091038, alias: "<internal", vr: VR::SH },
            E { tag: 0x31091039, alias: "<internal", vr: VR::SH },
            E { tag: 0x31091040, alias: "<internal", vr: VR::IS },
            E { tag: 0x31091041, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091042, alias: "<internal", vr: VR::CS },
            E { tag: 0x31091043, alias: "Archive Status", vr: VR::CS },
            E { tag: 0x310910EE, alias: "<internal", vr: VR::UI },
            E { tag: 0x310910EF, alias: "<internal", vr: VR::CS },
        ],
    },
    P {
        creator: "Applicare/RadWorks/Version 6.0",
        entries: &[
            E { tag: 0x41031001, alias: "Internal: non-existent tags", vr: VR::AT },
            E { tag: 0x41031002, alias: "Internal: non-existent objects", vr: VR::UI },
            E { tag: 0x41051001, alias: "Annotation Type", vr: VR::CS },
            E { tag: 0x41051002, alias: "Annotation Value", vr: VR::DS },
            E { tag: 0x41051003, alias: "Cutline Image UID", vr: VR::UI },
            E { tag: 0x41051004, alias: "Cutline Set UID", vr: VR::UI },
            E { tag: 0x41051005, alias: "Annotation Color", vr: VR::US },
            E { tag: 0x41051006, alias: "Annotation Line Style", vr: VR::CS },
            E { tag: 0x41051007, alias: "Annotation Label", vr: VR::SH },
            E { tag: 0x41051008, alias: "Annotation Creator", vr: VR::PN },
            E { tag: 0x41051009, alias: "Annotation Creation Date", vr: VR::DA },
            E { tag: 0x4105100A, alias: "Annotation Creation Time", vr: VR::TM },
            E { tag: 0x4105100B, alias: "Annotation Modification Sequence", vr: VR::SQ },
            E { tag: 0x4105100C, alias: "Annotation Modifier", vr: VR::PN },
            E { tag: 0x4105100D, alias: "Annotation Modification Date", vr: VR::DA },
            E { tag: 0x4105100E, alias: "Annotation Modification Time", vr: VR::TM },
            E { tag: 0x4105100F, alias: "Palette Color LUT Name", vr: VR::LO },
            E { tag: 0x41051010, alias: "Annotation Number", vr: VR::US },
            E { tag: 0x41071001, alias: "Requested Palette Color LUT", vr: VR::SQ },
        ],
    },
    P {
        creator: "Applicare/RadWorks/Version 6.0/Summary",
        entries: &[
            E { tag: 0x31091011, alias: "Receive Origin Description", vr: VR::ST },
            E { tag: 0x31091012, alias: "Receive Origin Description", vr: VR::ST },
        ],
    },
    P {
        creator: "Applicare/Workflow/Version 1.0",
        entries: &[
            E { tag: 0x31131001, alias: "<internal", vr: VR::CS },
            E { tag: 0x31131010, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131011, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131012, alias: "<internal", vr: VR::LO },
            E { tag: 0x31131015, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131016, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131017, alias: "<internal", vr: VR::LO },
            E { tag: 0x31131020, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131021, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131022, alias: "<internal", vr: VR::LO },
            E { tag: 0x31131025, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131026, alias: "<internal", vr: VR::SH },
            E { tag: 0x31131027, alias: "<internal", vr: VR::LO },
            E { tag: 0x31131030, alias: "<internal", vr: VR::UI },
            E { tag: 0x31131031, alias: "<internal", vr: VR::UI },
            E { tag: 0x311310E0, alias: "<internal", vr: VR::CS },
            E { tag: 0x311310E1, alias: "<internal", vr: VR::CS },
            E { tag: 0x311310E2, alias: "<internal", vr: VR::CS },
        ],
    },
    P {
        creator: "BRIT Systems, Inc.",
        entries: &[
            E { tag: 0x00211000, alias: "Person Information Sequence", vr: VR::SQ },
            E { tag: 0x00211001, alias: "Person ID", vr: VR::LO },
            E { tag: 0x00211002, alias: "Person Name", vr: VR::PN },
            E { tag: 0x00211003, alias: "Person Role", vr: VR::LO },
            E { tag: 0x00211004, alias: "Person Home Phone", vr: VR::SH },
            E { tag: 0x00211005, alias: "Person Work Phone", vr: VR::SH },
            E { tag: 0x00211006, alias: "Person Cell Phone", vr: VR::SH },
            E { tag: 0x00211007, alias: "Person Pager Phone", vr: VR::SH },
            E { tag: 0x00211008, alias: "Person Fax Phone", vr: VR::SH },
            E { tag: 0x00211009, alias: "Person EMail", vr: VR::LO },
            E { tag: 0x0021100A, alias: "Person Address", vr: VR::ST },
            E { tag: 0x0021100B, alias: "Person Password", vr: VR::LO },
            E { tag: 0x0021100C, alias: "Person Emergency Phone", vr: VR::SH },
            E { tag: 0x0021100D, alias: "Physician ID", vr: VR::LO },
            E { tag: 0x00211011, alias: "Original Patient ID", vr: VR::LO },
            E { tag: 0x00211012, alias: "Original Study Instance UID", vr: VR::UI },
            E { tag: 0x00211013, alias: "Original Series Instance UID", vr: VR::UI },
            E { tag: 0x00211014, alias: "Master Accession Number", vr: VR::LO },
            E { tag: 0x00211015, alias: "Order Category", vr: VR::LO },
            E { tag: 0x00211016, alias: "Patient ICN", vr: VR::LO },
            E { tag: 0x00211017, alias: "Patient DFS", vr: VR::LO },
            E { tag: 0x00211018, alias: "Patient Class", vr: VR::LO },
            E { tag: 0x00211019, alias: "Patient Type", vr: VR::LO },
            E { tag: 0x0021101F, alias: "Generic String", vr: VR::LT },
            E { tag: 0x00211020, alias: "QC Study Assigned By", vr: VR::LO },
            E { tag: 0x00211021, alias: "QC Study Split By", vr: VR::LO },
            E { tag: 0x00211022, alias: "QC Study Moved By", vr: VR::LO },
            E { tag: 0x00211023, alias: "QC Study Edited By", vr: VR::LO },
            E { tag: 0x00211024, alias: "QC Series Split By", vr: VR::LO },
            E { tag: 0x00211025, alias: "QC Series Moved By", vr: VR::LO },
            E { tag: 0x00211026, alias: "QC Series Edited By", vr: VR::LO },
            E { tag: 0x00211027, alias: "QC Image Moved By", vr: VR::LO },
            E { tag: 0x00211028, alias: "QC Image Edited By", vr: VR::LO },
            E { tag: 0x00211030, alias: "QC Done Time", vr: VR::LO },
            E { tag: 0x00211031, alias: "QC Last Modification Time", vr: VR::LO },
            E { tag: 0x00211032, alias: "QC Image Accepted By", vr: VR::LO },
            E { tag: 0x00211033, alias: "QC Image Rejected By", vr: VR::LO },
            E { tag: 0x00211034, alias: "QC Done Date", vr: VR::DA },
            E { tag: 0x00211050, alias: "QC Deletion Requested", vr: VR::LO },
            E { tag: 0x00211090, alias: "Original Sender AE Title", vr: VR::AE },
            E { tag: 0x00211091, alias: "Software Title", vr: VR::LO },
            E { tag: 0x00211092, alias: "Software Version", vr: VR::SH },
            E { tag: 0x00211093, alias: "Serial Number", vr: VR::LO },
            E { tag: 0x002110A0, alias: "Object Action Sequence", vr: VR::SQ },
            E { tag: 0x002110A1, alias: "Object Action", vr: VR::ST },
            E { tag: 0x002110A2, alias: "Object Action Date", vr: VR::DA },
            E { tag: 0x002110A3, alias: "Object Action Time", vr: VR::TM },
            E { tag: 0x002110A5, alias: "Local AE Title", vr: VR::AE },
            E { tag: 0x002110A6, alias: "Local IP Address", vr: VR::SH },
            E { tag: 0x002110A7, alias: "Remote AE Title", vr: VR::AE },
            E { tag: 0x002110A8, alias: "Remote IP Address", vr: VR::SH },
        ],
    },
    P {
        creator: "BioPri",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091001, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091002, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091009, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091010, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "BioPri3D",
        entries: &[
            E { tag: 0x00111010, alias: "Private Segmentation Data", vr: VR::UN },
            E { tag: 0x00631010, alias: "Private Creator", vr: VR::UN },
        ],
    },
    P {
        creator: "BrainLAB_BeamProfile",
        entries: &[
            E { tag: 0x34111001, alias: "Beam Profile Sequence", vr: VR::SQ },
            E { tag: 0x34111002, alias: "Beam Profile Number", vr: VR::IS },
            E { tag: 0x34111003, alias: "Beam Parameter Sequence", vr: VR::SQ },
            E { tag: 0x34111004, alias: "Parameter Description", vr: VR::UT },
            E { tag: 0x34111005, alias: "Parameter Data", vr: VR::OB },
            E { tag: 0x34111006, alias: "Referenced Beam Profile Number", vr: VR::IS },
        ],
    },
    P {
        creator: "BrainLAB_Conversion",
        entries: &[
            E { tag: 0x00091001, alias: "Export Platform Name", vr: VR::LO },
            E { tag: 0x00091002, alias: "Export Platform Data", vr: VR::OB },
        ],
    },
    P {
        creator: "BrainLAB_PatientSetup",
        entries: &[
            E { tag: 0x32731000, alias: "Isocenter Position", vr: VR::DS },
            E { tag: 0x32731001, alias: "Patient Position", vr: VR::CS },
        ],
    },
    P {
        creator: "CAMTRONICS",
        entries: &[
            E { tag: 0x00291010, alias: "Commentline", vr: VR::LT },
            E { tag: 0x00291020, alias: "Edge Enhancement Coefficient", vr: VR::DS },
            E { tag: 0x00291050, alias: "Scene Text", vr: VR::LT },
            E { tag: 0x00291060, alias: "Image Text", vr: VR::LT },
            E { tag: 0x00291070, alias: "Pixel Shift Horizontal", vr: VR::IS },
            E { tag: 0x00291080, alias: "Pixel Shift Vertical", vr: VR::IS },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "CAMTRONICS IP",
        entries: &[
            E { tag: 0x00291010, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291040, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "CANON_MEC_MR3",
        entries: &[
            E { tag: 0x00290001, alias: "Other Private Data", vr: VR::SQ },
            E { tag: 0x00290089, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290090, alias: "Reversed EVRLE DataSet", vr: VR::OB },
            E { tag: 0x700D0000, alias: "Scale Factor", vr: VR::DS },
            E { tag: 0x700D0005, alias: "FOV", vr: VR::DS },
            E { tag: 0x700D000C, alias: "Receiver Gain Correction Check Flag", vr: VR::CS },
            E { tag: 0x700D0020, alias: "Identification Flag of 3D GDC", vr: VR::SH },
        ],
    },
    P {
        creator: "CANON_MEC_MR3^10",
        entries: &[
            E { tag: 0x00090002, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00110003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110004, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00110005, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110006, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110007, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00110008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0011000C, alias: "Unknown", vr: VR::SS },
            E { tag: 0x0011000E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190015, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190017, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190027, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019002F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::OF },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190033, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190039, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190045, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190047, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190049, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190050, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190051, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190054, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190056, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190065, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190071, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190073, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190085, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190088, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190089, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019008D, alias: "Unknown", vr: VR::TM },
            E { tag: 0x0019008E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190096, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CA, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900CB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CC, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D9, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900DA, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900DC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EA, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900EB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::FD },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210008, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00210009, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021000A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021000B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0021000F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210012, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00210014, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210015, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210018, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021001D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0021001F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210022, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210023, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00290001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290005, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00290006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029001C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290021, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0029004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290054, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290067, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00290068, alias: "Unknown", vr: VR::SS },
            E { tag: 0x0029006E, alias: "Unknown", vr: VR::OB },
            E { tag: 0x700D0010, alias: "2nd Flip Angle [degree]", vr: VR::DS },
            E { tag: 0x700D0011, alias: "Acquisition Inner Matrix", vr: VR::US },
            E { tag: 0x700D0012, alias: "MP2RAGE Flag", vr: VR::US },
            E { tag: 0x700D0013, alias: "Inversion efficiency of inversion recovery pulse", vr: VR::FL },
            E { tag: 0x700D0014, alias: "Number of dummy shot", vr: VR::SL },
            E { tag: 0x700D0015, alias: "FFE total repetition time[s]", vr: VR::FL },
            E { tag: 0x700D0016, alias: "PAS Name", vr: VR::LO },
            E { tag: 0x700D0017, alias: "Intended Processing", vr: VR::LT },
            E { tag: 0x700D0018, alias: "Scanned Orientation IDs", vr: VR::SS },
            E { tag: 0x700D0019, alias: "PAS Reproduct Information", vr: VR::OB },
            E { tag: 0x700D001A, alias: "ASTAR Inversion Time", vr: VR::DS },
            E { tag: 0x700D001B, alias: "Saturation Recovery Time", vr: VR::DS },
            E { tag: 0xE2010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE3010000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010001, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010003, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010004, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010005, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010007, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010008, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010012, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010013, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010020, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010021, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010022, alias: "Unknown", vr: VR::LO },
            E { tag: 0xE4010023, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010024, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xEE010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xEE010001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xEE010002, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "CANON_MEC_MR3^11",
        entries: &[
            E { tag: 0x00190001, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019001A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190024, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190025, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190037, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190039, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190041, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00190049, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190065, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190066, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190068, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190069, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019006E, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190071, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190072, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190073, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190077, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190079, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190080, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190083, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190094, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B3, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B9, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F7, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "CANON_MEC_MR3^12",
        entries: &[
            E { tag: 0x00190000, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190079, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190080, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190088, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190089, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008B, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190094, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190095, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190097, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190098, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190099, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019009A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B3, alias: "Unknown", vr: VR::FD },
            E { tag: 0x001900C2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SS },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D7, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DE, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DF, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E3, alias: "Unknown", vr: VR::TM },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EB, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "CANON_MEC_MR3^13",
        entries: &[
            E { tag: 0x00190002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000F, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190022, alias: "Unknown", vr: VR::SL },
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4030001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030002, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030003, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "CARDIO-D.R. 1.0",
        entries: &[
            E { tag: 0x00091000, alias: "File Location", vr: VR::UL },
            E { tag: 0x00091001, alias: "File Size", vr: VR::UL },
            E { tag: 0x00091040, alias: "Alternate Image Sequence", vr: VR::SQ },
            E { tag: 0x00191000, alias: "Image Blanking Shape", vr: VR::CS },
            E { tag: 0x00191002, alias: "Image Blanking Left Vertical Edge", vr: VR::IS },
            E { tag: 0x00191004, alias: "Image Blanking Right Vertical Edge", vr: VR::IS },
            E { tag: 0x00191006, alias: "Image Blanking Upper Horizontal Edge", vr: VR::IS },
            E { tag: 0x00191008, alias: "Image Blanking Lower Horizontal Edge", vr: VR::IS },
            E { tag: 0x00191010, alias: "Center Of Circular Image Blanking", vr: VR::IS },
            E { tag: 0x00191012, alias: "Radius Of Circular Image Blanking", vr: VR::IS },
            E { tag: 0x00191030, alias: "Maximum Frame Size", vr: VR::UL },
            E { tag: 0x00211013, alias: "Image Sequence Number", vr: VR::IS },
            E { tag: 0x00291000, alias: "Standard Edge Enhancement Sequence", vr: VR::SQ },
            E { tag: 0x00291001, alias: "Convolution Kernel Size", vr: VR::US },
            E { tag: 0x00291002, alias: "Convolution Kernel Coefficients", vr: VR::US },
            E { tag: 0x00291003, alias: "Edge Enhancement Gain", vr: VR::FL },
        ],
    },
    P {
        creator: "CARDIO-SMS 1.0",
        entries: &[
            E { tag: 0x00091002, alias: "Private Data", vr: VR::OB },
            E { tag: 0x00091003, alias: "Private Data", vr: VR::OB },
            E { tag: 0x00091004, alias: "Private Data", vr: VR::OB },
            E { tag: 0x00091005, alias: "Private Data", vr: VR::OB },
            E { tag: 0x00091006, alias: "Private Data", vr: VR::OB },
            E { tag: 0x00091008, alias: "ACOM flags", vr: VR::OB },
            E { tag: 0x0009100A, alias: "Private Data", vr: VR::UN },
            E { tag: 0x0009100B, alias: "Private Data", vr: VR::UN },
            E { tag: 0x0009100C, alias: "Private Data", vr: VR::UN },
            E { tag: 0x00291080, alias: "Private Data", vr: VR::SQ },
            E { tag: 0x00311001, alias: "Private Data stream", vr: VR::UN },
        ],
    },
    P {
        creator: "CEMAX-ICON",
        entries: &[
            E { tag: 0x00191000, alias: "CR Processing Parameters", vr: VR::LT },
            E { tag: 0x00191010, alias: "CR Exposure Menu Code", vr: VR::LO },
            E { tag: 0x00191020, alias: "CR Exposure Menu String", vr: VR::LO },
            E { tag: 0x00191030, alias: "CR EDR Mode", vr: VR::LO },
            E { tag: 0x00191040, alias: "CR Latitude", vr: VR::LO },
            E { tag: 0x00191050, alias: "CR Group Number", vr: VR::LO },
            E { tag: 0x00191060, alias: "CR Equipment ID", vr: VR::LO },
            E { tag: 0x00191070, alias: "CR Image Serial Number", vr: VR::LO },
            E { tag: 0x00191080, alias: "CR Bar Code Number", vr: VR::LO },
            E { tag: 0x00191090, alias: "CR Film Output Exposure", vr: VR::LO },
            E { tag: 0x00291000, alias: "Key Image", vr: VR::LO },
            E { tag: 0x00291010, alias: "Note", vr: VR::LT },
            E { tag: 0x00291020, alias: "Annotation Sequence", vr: VR::SQ },
            E { tag: 0x00291030, alias: "X Start Position", vr: VR::IS },
            E { tag: 0x00291040, alias: "Y Start Position", vr: VR::IS },
            E { tag: 0x00291050, alias: "X End Position", vr: VR::IS },
            E { tag: 0x00291060, alias: "Y End Position", vr: VR::IS },
            E { tag: 0x00291070, alias: "Annotation Text", vr: VR::LO },
            E { tag: 0x00291080, alias: "Which Marker", vr: VR::IS },
            E { tag: 0x00291090, alias: "Annotation Sequence", vr: VR::SQ },
            E { tag: 0x00291091, alias: "X Coordinates", vr: VR::IS },
            E { tag: 0x00291092, alias: "Y Coordinates", vr: VR::IS },
            E { tag: 0x00291093, alias: "Annotation Type", vr: VR::IS },
            E { tag: 0x00291094, alias: "Number of points", vr: VR::IS },
            E { tag: 0x00291095, alias: "Fill Type", vr: VR::IS },
            E { tag: 0x30331000, alias: "Study Priority Status", vr: VR::CS },
        ],
    },
    P {
        creator: "Canon Inc.",
        entries: &[
            E { tag: 0x00191010, alias: "Canon Internal Data 1", vr: VR::OB },
            E { tag: 0x00191013, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191015, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191016, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191017, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191018, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191019, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019101A, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019101B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019101C, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019101E, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019101F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191021, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191060, alias: "Performed number of series", vr: VR::US },
            E { tag: 0x00191070, alias: "Performed number of images", vr: VR::US },
            E { tag: 0x0019107F, alias: "Canon Internal Data 111", vr: VR::OB },
        ],
    },
    P {
        creator: "DIDI TO PCR 1.1",
        entries: &[
            E { tag: 0x00191000, alias: "Post Mode String", vr: VR::LT },
            E { tag: 0x00191001, alias: "Post Data", vr: VR::LT },
            E { tag: 0x00191010, alias: "Image Header", vr: VR::LT },
            E { tag: 0x00191022, alias: "Route AET", vr: VR::LO },
            E { tag: 0x00191023, alias: "PCR Print Scale", vr: VR::DS },
            E { tag: 0x00191024, alias: "PCR Print Job End", vr: VR::ST },
            E { tag: 0x00191025, alias: "PCR No Film Copies", vr: VR::IS },
            E { tag: 0x00191026, alias: "PCR Film Layout Position", vr: VR::IS },
            E { tag: 0x00191027, alias: "PCR Print Report Name", vr: VR::ST },
            E { tag: 0x00191070, alias: "RAD Protocol Printer", vr: VR::ST },
            E { tag: 0x00191071, alias: "RAD Protocol Medium", vr: VR::ST },
            E { tag: 0x00191080, alias: "Original Filename", vr: VR::LO },
            E { tag: 0x00191081, alias: "Filter Type", vr: VR::SH },
            E { tag: 0x00191082, alias: "Stitching", vr: VR::LT },
            E { tag: 0x00191089, alias: "Exposure Index", vr: VR::IS },
            E { tag: 0x0019108A, alias: "Collimator X", vr: VR::IS },
            E { tag: 0x0019108B, alias: "Collimator Y", vr: VR::IS },
            E { tag: 0x0019108C, alias: "Print Marker", vr: VR::LO },
            E { tag: 0x0019108D, alias: "RGDV Name", vr: VR::LO },
            E { tag: 0x0019108E, alias: "Acqd Sensitivity", vr: VR::LO },
            E { tag: 0x0019108F, alias: "Processing Category", vr: VR::LO },
            E { tag: 0x00191090, alias: "Unprocessed Flag", vr: VR::LO },
            E { tag: 0x00191091, alias: "Key Values", vr: VR::DS },
            E { tag: 0x00191092, alias: "Destination Postprocessing Function", vr: VR::LO },
            E { tag: 0x001910A0, alias: "Version", vr: VR::LO },
            E { tag: 0x001910A1, alias: "Ranging Mode", vr: VR::LO },
            E { tag: 0x001910A2, alias: "Abdomen Brightness", vr: VR::DS },
            E { tag: 0x001910A3, alias: "Fixed Brightness", vr: VR::DS },
            E { tag: 0x001910A4, alias: "Detail Contrast", vr: VR::DS },
            E { tag: 0x001910A5, alias: "Contrast Balance", vr: VR::DA },
            E { tag: 0x001910A6, alias: "Structure Boost", vr: VR::DS },
            E { tag: 0x001910A7, alias: "Structure Preference", vr: VR::DS },
            E { tag: 0x001910A8, alias: "Noise Robustness", vr: VR::DS },
            E { tag: 0x001910A9, alias: "Noise Dose Limit", vr: VR::DS },
            E { tag: 0x001910AA, alias: "Noise Dose Step", vr: VR::DS },
            E { tag: 0x001910AB, alias: "Noise Frequency Limit", vr: VR::DS },
            E { tag: 0x001910AC, alias: "Weak Contrast Limit", vr: VR::DS },
            E { tag: 0x001910AD, alias: "Strong Contrast Limit", vr: VR::DS },
            E { tag: 0x001910AE, alias: "Structure Boost Offset", vr: VR::DS },
            E { tag: 0x001910AF, alias: "Smooth Gain", vr: VR::LO },
            E { tag: 0x001910B0, alias: "Measure Field 1", vr: VR::LO },
            E { tag: 0x001910B1, alias: "Measure Field 2", vr: VR::LO },
            E { tag: 0x001910B2, alias: "Key Percentile 1", vr: VR::IS },
            E { tag: 0x001910B3, alias: "Key Percentile 2", vr: VR::IS },
            E { tag: 0x001910B4, alias: "Density LUT", vr: VR::IS },
            E { tag: 0x001910B5, alias: "Brightness", vr: VR::DS },
            E { tag: 0x001910B6, alias: "Gamma", vr: VR::DS },
            E { tag: 0x00891010, alias: "Stamp Image Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "DIGISCAN IMAGE",
        entries: &[
            E { tag: 0x00291031, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291033, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291034, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "DLX_ANNOT_01",
        entries: &[
            E { tag: 0x70101004, alias: "Text_annotation", vr: VR::ST },
            E { tag: 0x70101005, alias: "Box", vr: VR::IS },
            E { tag: 0x70101007, alias: "Arrow end", vr: VR::IS },
        ],
    },
    P {
        creator: "DLX_EXAMS_01",
        entries: &[
            E { tag: 0x00151001, alias: "Stenos_calibr_ratio;", vr: VR::DS },
            E { tag: 0x00151002, alias: "Stenos_magnification;", vr: VR::DS },
            E { tag: 0x00151003, alias: "Cardiac_calibr_ratio;", vr: VR::DS },
        ],
    },
    P {
        creator: "DLX_LKUP_01",
        entries: &[
            E { tag: 0x60101001, alias: "Gray Palette color lookup table descriptor", vr: VR::LT },
            E { tag: 0x60101002, alias: "Gray Palette color lookup table data", vr: VR::LT },
        ],
    },
    P {
        creator: "DLX_PATNT_01",
        entries: &[
            E { tag: 0x00111001, alias: "patient DOB", vr: VR::LT },
        ],
    },
    P {
        creator: "DLX_SERIE_01",
        entries: &[
            E { tag: 0x00191001, alias: "Angle Value 1", vr: VR::DS },
            E { tag: 0x00191002, alias: "Angle Value 2", vr: VR::DS },
            E { tag: 0x00191003, alias: "Angle Value 3", vr: VR::DS },
            E { tag: 0x00191004, alias: "Angle Label 1", vr: VR::CS },
            E { tag: 0x00191005, alias: "Angle Label 2", vr: VR::CS },
            E { tag: 0x00191006, alias: "Angle Label 3", vr: VR::CS },
            E { tag: 0x00191007, alias: "Adx Procedure Name", vr: VR::ST },
            E { tag: 0x00191008, alias: "Adx Exam Name", vr: VR::ST },
            E { tag: 0x00191009, alias: "Adx Patient Size", vr: VR::SH },
            E { tag: 0x0019100A, alias: "Adx Record View", vr: VR::IS },
            E { tag: 0x00191010, alias: "Adx Injector Delay", vr: VR::DS },
            E { tag: 0x00191011, alias: "Adx Auto Inject", vr: VR::CS },
            E { tag: 0x00191014, alias: "Adx Acq Mode", vr: VR::IS },
            E { tag: 0x00191015, alias: "Adx Camera Rotation Enable", vr: VR::CS },
            E { tag: 0x00191016, alias: "Adx Reverse Sweep", vr: VR::CS },
            E { tag: 0x00191017, alias: "User Spatial Filter Strength", vr: VR::IS },
            E { tag: 0x00191018, alias: "User Zoom Factor", vr: VR::IS },
            E { tag: 0x00191019, alias: "X Zoom", vr: VR::IS },
            E { tag: 0x0019101A, alias: "Y Zoom", vr: VR::IS },
            E { tag: 0x0019101B, alias: "Adx Focus", vr: VR::DS },
            E { tag: 0x0019101C, alias: "Adx Dose", vr: VR::CS },
            E { tag: 0x0019101D, alias: "Side Mark", vr: VR::IS },
            E { tag: 0x0019101E, alias: "Percent Landscape", vr: VR::IS },
            E { tag: 0x0019101F, alias: "Adx Exposure Duration", vr: VR::DS },
            E { tag: 0x00191020, alias: "Ip Address", vr: VR::SH },
            E { tag: 0x00191021, alias: "Table Vertical Position", vr: VR::DS },
            E { tag: 0x00191022, alias: "Table Longitudinal Position", vr: VR::DS },
            E { tag: 0x00191023, alias: "Table Lateral Position", vr: VR::DS },
            E { tag: 0x00191024, alias: "Lambda cm Pincushion Distortion", vr: VR::DS },
            E { tag: 0x00191025, alias: "Slope LV Regression", vr: VR::DS },
            E { tag: 0x00191026, alias: "Intercept LV Regression", vr: VR::DS },
            E { tag: 0x00191027, alias: "Image chain FWHM psf mm min", vr: VR::DS },
            E { tag: 0x00191028, alias: "Image chain FWHM psf mm max", vr: VR::DS },
        ],
    },
    P {
        creator: "DL_INTERNAL_USE",
        entries: &[
            E { tag: 0x0015108F, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "DR Systems, Inc.",
        entries: &[
            E { tag: 0x44531001, alias: "DR Exam ID", vr: VR::LO },
            E { tag: 0x44531002, alias: "DR Image Type", vr: VR::LO },
            E { tag: 0x44531004, alias: "DR File Type", vr: VR::LO },
            E { tag: 0x44531005, alias: "DR File Suffix", vr: VR::LO },
            E { tag: 0x4453100A, alias: "DR Annotation Type", vr: VR::UN },
            E { tag: 0x4453100C, alias: "DR Original Instance UID Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "DicomUtils 20100512",
        entries: &[
            E { tag: 0x00090001, alias: "DICOMUTILS_PROCESSING_METHODS", vr: VR::SQ },
        ],
    },
    P {
        creator: "ELGEMS_XPERT",
        entries: &[
            E { tag: 0x7FA11000, alias: "XPert Composite Report Item buffer", vr: VR::OB },
            E { tag: 0x7FA11010, alias: "Xpert Composite Report Item Identificator", vr: VR::OB },
        ],
    },
    P {
        creator: "ELSCINT1",
        entries: &[
            E { tag: 0x00031001, alias: "Offset List Structure", vr: VR::OW },
            E { tag: 0x00E10046, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00E100E1, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00E100EC, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00E11021, alias: "DLP", vr: VR::DS },
            E { tag: 0x00E11037, alias: "Total Saving Dose", vr: VR::DS },
            E { tag: 0x00E1103E, alias: "Split Is Dual Surview", vr: VR::IS },
            E { tag: 0x00E11050, alias: "Acquisition Duration", vr: VR::DS },
            E { tag: 0x00E110C4, alias: "Abs Bed Pos", vr: VR::DS },
            E { tag: 0x00E11001, alias: "Data Dictionary Version", vr: VR::US },
            E { tag: 0x00E11005, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E11006, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E11007, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E11014, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E11018, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00E11021, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00E11022, alias: "Presentation Relative Center", vr: VR::DS },
            E { tag: 0x00E11023, alias: "Presentation Relative Part", vr: VR::DS },
            E { tag: 0x00E11024, alias: "Presentation Horizontal Invert", vr: VR::CS },
            E { tag: 0x00E11025, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E1102A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00E11030, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00E11031, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E11032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00E11037, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00E11039, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00E1103F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E11040, alias: "Image Label", vr: VR::SH },
            E { tag: 0x00E11041, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00E11042, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00E11043, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E11050, alias: "Acquisition Duration", vr: VR::DS },
            E { tag: 0x00E11051, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00E11060, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E11061, alias: "Protocol File Name", vr: VR::LO },
            E { tag: 0x00E11062, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00E11063, alias: "Patient Language", vr: VR::SH },
            E { tag: 0x00E11065, alias: "Patient Data Modification Date", vr: VR::LO },
            E { tag: 0x00E1106A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E1106B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00E110A0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00E110C2, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00E31000, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00E31018, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00E3101F, alias: "Unknown", vr: VR::OB },
            E { tag: 0x01E10040, alias: "Unknown", vr: VR::UN },
            E { tag: 0x01E11017, alias: "ECG Reference UID", vr: VR::UI },
            E { tag: 0x01E11026, alias: "Phantom Type", vr: VR::CS },
            E { tag: 0x01E11018, alias: "Unknown", vr: VR::OB },
            E { tag: 0x01E11021, alias: "Unknown", vr: VR::ST },
            E { tag: 0x01E11034, alias: "Unknown", vr: VR::IS },
            E { tag: 0x01F1000F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x01F11001, alias: "Acquisition Type", vr: VR::CS },
            E { tag: 0x01F11002, alias: "Resolution", vr: VR::CS },
            E { tag: 0x01F11004, alias: "Angular Sampling", vr: VR::CS },
            E { tag: 0x01F11008, alias: "Scan Length", vr: VR::DS },
            E { tag: 0x01F1100C, alias: "Scanner Relative Center", vr: VR::DS },
            E { tag: 0x01F1100E, alias: "Recon Enhancement", vr: VR::FL },
            E { tag: 0x01F11026, alias: "Pitch", vr: VR::FD },
            E { tag: 0x01F11027, alias: "Rotation Time", vr: VR::DS },
            E { tag: 0x01F11028, alias: "Table Increment", vr: VR::DS },
            E { tag: 0x01F11032, alias: "View Convention", vr: VR::CS },
            E { tag: 0x01F11033, alias: "Cycle Time", vr: VR::DS },
            E { tag: 0x01F11041, alias: "Gating Delay", vr: VR::LO },
            E { tag: 0x01F11045, alias: "Initial Heart Rate", vr: VR::IS },
            E { tag: 0x01F11049, alias: "Planned mAs", vr: VR::DS },
            E { tag: 0x01F1104B, alias: "Collimation", vr: VR::SH },
            E { tag: 0x01F1104C, alias: "DOSE Right DOM", vr: VR::SH },
            E { tag: 0x01F1104D, alias: "Adaptive Filter", vr: VR::SH },
            E { tag: 0x01F1104E, alias: "Scan Type", vr: VR::SH },
            E { tag: 0x01F11001, alias: "Acquisition Type", vr: VR::CS },
            E { tag: 0x01F11002, alias: "Unknown", vr: VR::CS },
            E { tag: 0x01F11003, alias: "Concurrent Slices Generation", vr: VR::CS },
            E { tag: 0x01F11004, alias: "Angular Sampling Density", vr: VR::CS },
            E { tag: 0x01F11005, alias: "Reconstruction Arc", vr: VR::DS },
            E { tag: 0x01F11006, alias: "Unknown", vr: VR::DS },
            E { tag: 0x01F11007, alias: "Table Velocity", vr: VR::DS },
            E { tag: 0x01F11008, alias: "Acquisition Length", vr: VR::DS },
            E { tag: 0x01F1100A, alias: "Edge Enhancement Weight", vr: VR::US },
            E { tag: 0x01F1100B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x01F1100C, alias: "Scanner Relative Center", vr: VR::DS },
            E { tag: 0x01F1100D, alias: "Rotation Angle", vr: VR::DS },
            E { tag: 0x01F1100E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F11026, alias: "Pitch", vr: VR::DS },
            E { tag: 0x01F11027, alias: "Rotation Time", vr: VR::DS },
            E { tag: 0x01F11028, alias: "Unknown", vr: VR::DS },
            E { tag: 0x01F11030, alias: "Unknown", vr: VR::US },
            E { tag: 0x01F11032, alias: "Image View Convention", vr: VR::CS },
            E { tag: 0x01F11033, alias: "Unknown", vr: VR::DS },
            E { tag: 0x01F11036, alias: "Unknown", vr: VR::CS },
            E { tag: 0x01F11037, alias: "Unknown", vr: VR::DS },
            E { tag: 0x01F11038, alias: "Unknown", vr: VR::LO },
            E { tag: 0x01F11039, alias: "Unknown", vr: VR::LO },
            E { tag: 0x01F11040, alias: "Unknown", vr: VR::CS },
            E { tag: 0x01F11042, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F11043, alias: "Unknown", vr: VR::LO },
            E { tag: 0x01F11044, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F11045, alias: "Unknown", vr: VR::IS },
            E { tag: 0x01F11046, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F11047, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F11049, alias: "Unknown", vr: VR::DS },
            E { tag: 0x01F1104A, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F1104B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F1104C, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F1104D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F1104E, alias: "Unknown", vr: VR::LO },
            E { tag: 0x01F11053, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F31001, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x01F31002, alias: "Unknown", vr: VR::SS },
            E { tag: 0x01F31003, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31004, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31011, alias: "PS Sequence", vr: VR::SQ },
            E { tag: 0x01F31012, alias: "Unknown", vr: VR::SS },
            E { tag: 0x01F31013, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31014, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31015, alias: "Unknown", vr: VR::US },
            E { tag: 0x01F31016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31017, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31018, alias: "Unknown", vr: VR::SH },
            E { tag: 0x01F31019, alias: "Unknown", vr: VR::FL },
            E { tag: 0x01F31023, alias: "Unknown", vr: VR::US },
            E { tag: 0x01F31024, alias: "Unknown", vr: VR::IS },
            E { tag: 0x01F70097, alias: "Unknown", vr: VR::UN },
            E { tag: 0x01F70099, alias: "Unknown", vr: VR::UN },
            E { tag: 0x01F7009B, alias: "iDose Level", vr: VR::IS },
            E { tag: 0x01F7109B, alias: "iDose Level", vr: VR::IS },
            E { tag: 0x01F71010, alias: "Unknown", vr: VR::OB },
            E { tag: 0x01F71011, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71013, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71014, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71015, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71016, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71017, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71018, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71019, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7101A, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7101B, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7101C, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7101E, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7101F, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71022, alias: "Unknown", vr: VR::UI },
            E { tag: 0x01F71023, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71025, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71026, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71027, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71028, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71029, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7102B, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7102C, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7102D, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F7102E, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71030, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71031, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71070, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71073, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71074, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F71075, alias: "Unknown", vr: VR::OW },
            E { tag: 0x01F91001, alias: "Mar Filter", vr: VR::LO },
            E { tag: 0x01F91002, alias: "Recon Increment", vr: VR::DS },
            E { tag: 0x01F91003, alias: "CTDIw", vr: VR::DS },
            E { tag: 0x01F91004, alias: "Couch Direction", vr: VR::IS },
            E { tag: 0x01F91005, alias: "Series No In Acquisition", vr: VR::IS },
            E { tag: 0x01F91007, alias: "Dose Right ACS", vr: VR::SH },
            E { tag: 0x01F91008, alias: "Left DMS Tmp Diff", vr: VR::DS },
            E { tag: 0x01F91009, alias: "Right DMS Tmp Diff", vr: VR::DS },
            E { tag: 0x01F91010, alias: "Dose Right Noise", vr: VR::LO },
            E { tag: 0x01F91011, alias: "Zero Position", vr: VR::DS },
            E { tag: 0x01F91012, alias: "Show Couch Position", vr: VR::DS },
            E { tag: 0x01F91013, alias: "Recon Mode", vr: VR::IS },
            E { tag: 0x01F91014, alias: "Water Size", vr: VR::DS },
            E { tag: 0x01F91015, alias: "Digital Tilt", vr: VR::DS },
            E { tag: 0x01F91016, alias: "Scan Arc", vr: VR::DS },
            E { tag: 0x06011000, alias: "Implementation Version", vr: VR::SH },
            E { tag: 0x06011020, alias: "Relative Table Position", vr: VR::DS },
            E { tag: 0x06011021, alias: "Relative Table Height", vr: VR::DS },
            E { tag: 0x06011030, alias: "Surview Direction", vr: VR::SH },
            E { tag: 0x06011031, alias: "Surview Length", vr: VR::DS },
            E { tag: 0x06011050, alias: "Image View Type", vr: VR::SH },
            E { tag: 0x06011070, alias: "Batch Number", vr: VR::DS },
            E { tag: 0x06011071, alias: "Batch Size", vr: VR::DS },
            E { tag: 0x06011072, alias: "Batch Slice Number", vr: VR::DS },
            E { tag: 0x07A11001, alias: "number of series in study", vr: VR::UL },
            E { tag: 0x07A11002, alias: "number of images in series", vr: VR::UL },
            E { tag: 0x07A11003, alias: "last update time", vr: VR::UL },
            E { tag: 0x07A11004, alias: "last update date", vr: VR::UL },
            E { tag: 0x07A11007, alias: "Unknown", vr: VR::US },
            E { tag: 0x07A11008, alias: "Unknown", vr: VR::DS },
            E { tag: 0x07A11009, alias: "Unknown", vr: VR::OW },
            E { tag: 0x07A1100A, alias: "Tamar Compressed Pixel Data", vr: VR::OB },
            E { tag: 0x07A1100C, alias: "Unknown", vr: VR::US },
            E { tag: 0x07A11010, alias: "Tamar Software Version", vr: VR::LO },
            E { tag: 0x07A11011, alias: "Tamar Compression Type", vr: VR::CS },
            E { tag: 0x07A11012, alias: "Unknown", vr: VR::FL },
            E { tag: 0x07A11013, alias: "Unknown", vr: VR::UL },
            E { tag: 0x07A11014, alias: "protection flag", vr: VR::LO },
            E { tag: 0x07A11016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x07A11018, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x07A11019, alias: "Unknown", vr: VR::FL },
            E { tag: 0x07A1101C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x07A1102A, alias: "Tamar Study Status", vr: VR::CS },
            E { tag: 0x07A1102B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A11036, alias: "Tamar Source Ae", vr: VR::AE },
            E { tag: 0x07A11040, alias: "Tamar Study Body Part", vr: VR::CS },
            E { tag: 0x07A11043, alias: "Unknown", vr: VR::IS },
            E { tag: 0x07A11050, alias: "Tamar Site Id", vr: VR::US },
            E { tag: 0x07A11056, alias: "Unknown", vr: VR::US },
            E { tag: 0x07A1105A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A11075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x07A11085, alias: "Tamar Translate Flags", vr: VR::UL },
            E { tag: 0x07A1109F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A31001, alias: "Tamar Exe Software Version", vr: VR::LO },
            E { tag: 0x07A31003, alias: "Tamar Study Has Sticky Note", vr: VR::CS },
            E { tag: 0x07A31005, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A31006, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A31010, alias: "Tamar Original Curve Desc", vr: VR::LO },
            E { tag: 0x07A31034, alias: "Tamar Study Age", vr: VR::SH },
            E { tag: 0x07A31043, alias: "Unknown", vr: VR::DS },
            E { tag: 0x07A31055, alias: "Unknown", vr: VR::SH },
            E { tag: 0x07A31061, alias: "Unknown", vr: VR::LT },
            E { tag: 0x07A31062, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x07A31063, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x07A31064, alias: "Unknown", vr: VR::IS },
            E { tag: 0x07A31066, alias: "Unknown", vr: VR::IS },
            E { tag: 0x07A31099, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A3109C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x07A3109F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x50011070, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x50011071, alias: "Unknown", vr: VR::SH },
            E { tag: 0x50011080, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x50011081, alias: "Unknown", vr: VR::SH },
            E { tag: 0x50011082, alias: "Unknown", vr: VR::US },
            E { tag: 0x50011083, alias: "Unknown", vr: VR::FL },
            E { tag: 0x50011084, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x60011010, alias: "Text Overlay Flag", vr: VR::CS },
            E { tag: 0x7FDF10F0, alias: "Unknown", vr: VR::OB },
            E { tag: 0x7FDF10FF, alias: "Unknown", vr: VR::SH },
        ],
    },
    P {
        creator: "EMAGEON JPEG2K INFO",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00091001, alias: "Unknown", vr: VR::DT },
        ],
    },
    P {
        creator: "EMAGEON STUDY HOME",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091001, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Eclipse 60",
        entries: &[
            E { tag: 0x00291030, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291031, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291032, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291033, alias: "Private data", vr: VR::DS },
            E { tag: 0x00291034, alias: "Private data", vr: VR::DS },
            E { tag: 0x00291036, alias: "Private data", vr: VR::SL },
            E { tag: 0x00291040, alias: "Private data", vr: VR::CS },
            E { tag: 0x00291050, alias: "Private data", vr: VR::CS },
            E { tag: 0x00291051, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291052, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291053, alias: "Private data", vr: VR::DS },
            E { tag: 0x00291054, alias: "Private data", vr: VR::UL },
            E { tag: 0x00291055, alias: "Private data", vr: VR::FL },
            E { tag: 0x00291056, alias: "Private data", vr: VR::US },
            E { tag: 0x00291060, alias: "Private data", vr: VR::LO },
            E { tag: 0x00311030, alias: "Private data", vr: VR::UL },
            E { tag: 0x00311031, alias: "Private data", vr: VR::UL },
            E { tag: 0x00311032, alias: "Private data", vr: VR::LO },
            E { tag: 0x00311033, alias: "Private data", vr: VR::UL },
            E { tag: 0x00311034, alias: "Private data", vr: VR::UL },
            E { tag: 0x00311035, alias: "Private data", vr: VR::UL },
            E { tag: 0x200D1001, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1006, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1008, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1009, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D100A, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D100B, alias: "Private data", vr: VR::OB },
            E { tag: 0x200D100C, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D100D, alias: "Private data", vr: VR::UL },
            E { tag: 0x200D100E, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D100F, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1010, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1011, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1012, alias: "Private data", vr: VR::OB },
            E { tag: 0x200D1013, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1014, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1015, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D101A, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D101B, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1020, alias: "Private data", vr: VR::SQ },
            E { tag: 0x200D1027, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1028, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1030, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1031, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1032, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1033, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1034, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1035, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1036, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1037, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1038, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1039, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D103A, alias: "Private data", vr: VR::LO },
            E { tag: 0x200D1050, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D1051, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D1052, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1053, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D1054, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D1055, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1056, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1057, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D1058, alias: "Private data", vr: VR::IS },
            E { tag: 0x200D1059, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D105A, alias: "Private data", vr: VR::FD },
            E { tag: 0x200D105D, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D105E, alias: "Private data", vr: VR::CS },
            E { tag: 0x200D10FF, alias: "Private data", vr: VR::IS },
            E { tag: 0x77771001, alias: "Private data", vr: VR::DA },
            E { tag: 0x77771002, alias: "Private data", vr: VR::CS },
            E { tag: 0x77771003, alias: "Private data", vr: VR::LO },
            E { tag: 0x77771010, alias: "Private data", vr: VR::LO },
            E { tag: 0x77771014, alias: "Private data", vr: VR::LT },
        ],
    },
    P {
        creator: "FDMS 1.0",
        entries: &[
            E { tag: 0x00091004, alias: "Image Control Unit", vr: VR::SH },
            E { tag: 0x00091005, alias: "Image UID", vr: VR::OW },
            E { tag: 0x00091006, alias: "Route Image UID", vr: VR::OW },
            E { tag: 0x00091008, alias: "Image Display Information Version No.", vr: VR::UL },
            E { tag: 0x00091009, alias: "Patient Information Version No.", vr: VR::UL },
            E { tag: 0x0009100C, alias: "Film UID", vr: VR::OW },
            E { tag: 0x00091010, alias: "Exposure Unit Type Code", vr: VR::CS },
            E { tag: 0x00091080, alias: "Kanji Hospital Name", vr: VR::LO },
            E { tag: 0x00091090, alias: "Distribution Code", vr: VR::ST },
            E { tag: 0x00091092, alias: "Kanji Department Name", vr: VR::SH },
            E { tag: 0x000910F0, alias: "Blackening Process Flag", vr: VR::CS },
            E { tag: 0x000910F1, alias: "Processing Information Flag", vr: VR::ST },
            E { tag: 0x00191015, alias: "Kanji Body Part for Exposure", vr: VR::LO },
            E { tag: 0x00191030, alias: "Menu Character String", vr: VR::LO },
            E { tag: 0x00191032, alias: "Kanji Menu Name", vr: VR::LO },
            E { tag: 0x00191040, alias: "Image Processing Type", vr: VR::CS },
            E { tag: 0x00191050, alias: "EDR Mode", vr: VR::CS },
            E { tag: 0x00191060, alias: "Radiographer's Code", vr: VR::SH },
            E { tag: 0x00191070, alias: "Split Exposure Format", vr: VR::IS },
            E { tag: 0x00191071, alias: "No. of Split Exposure Frames", vr: VR::IS },
            E { tag: 0x00191080, alias: "Reading Position Specification", vr: VR::IS },
            E { tag: 0x00191081, alias: "Reading Sensitivity Center", vr: VR::IS },
            E { tag: 0x00191090, alias: "Film Annotation Character String 1", vr: VR::SH },
            E { tag: 0x00191091, alias: "Film Annotation Character String 2", vr: VR::SH },
            E { tag: 0x001910A0, alias: "Exposure Status Sequence", vr: VR::SQ },
            E { tag: 0x001910A1, alias: "Exposure Status", vr: VR::CS },
            E { tag: 0x001910A2, alias: "Exposure Kind", vr: VR::CS },
            E { tag: 0x00211010, alias: "FCR Image ID", vr: VR::CS },
            E { tag: 0x00211030, alias: "Set No.", vr: VR::CS },
            E { tag: 0x00211040, alias: "Image No. in the Set", vr: VR::IS },
            E { tag: 0x00211050, alias: "Pair Processing Information", vr: VR::CS },
            E { tag: 0x00211070, alias: "Film Number within the Series", vr: VR::IS },
            E { tag: 0x00211080, alias: "Equipment Type-Specific Information", vr: VR::OB },
            E { tag: 0x00211090, alias: "LUT Number", vr: VR::CS },
            E { tag: 0x00231010, alias: "Left-image Processing Parameter Sequence", vr: VR::SQ },
            E { tag: 0x00231020, alias: "Right-image Processing Parameter Sequence", vr: VR::SQ },
            E { tag: 0x00231030, alias: "Single-image Processing Parameter Sequence", vr: VR::SQ },
            E { tag: 0x00251010, alias: "Relative Light Emission Amount Sk", vr: VR::US },
            E { tag: 0x00251011, alias: "Term of Correction for Each IP Type St", vr: VR::US },
            E { tag: 0x00251012, alias: "Reading Gain Gp", vr: VR::US },
            E { tag: 0x00251013, alias: "Latitude", vr: VR::US },
            E { tag: 0x00251015, alias: "Image Processing Selection Flag", vr: VR::CS },
            E { tag: 0x00251016, alias: "Min", vr: VR::US },
            E { tag: 0x00251017, alias: "Max", vr: VR::US },
            E { tag: 0x00251020, alias: "Sensitivity Shift (S-SHFT)", vr: VR::US },
            E { tag: 0x00251021, alias: "Contrast Shift (C-SHIFT)", vr: VR::US },
            E { tag: 0x00251030, alias: "GT", vr: VR::US },
            E { tag: 0x00251031, alias: "GA", vr: VR::SS },
            E { tag: 0x00251032, alias: "GC", vr: VR::US },
            E { tag: 0x00251033, alias: "GS", vr: VR::SS },
            E { tag: 0x00251034, alias: "tan θ", vr: VR::SS },
            E { tag: 0x00251040, alias: "RT", vr: VR::US },
            E { tag: 0x00251041, alias: "RN", vr: VR::US },
            E { tag: 0x00251042, alias: "RE", vr: VR::US },
            E { tag: 0x00251043, alias: "N", vr: VR::US },
            E { tag: 0x00251050, alias: "DRT", vr: VR::US },
            E { tag: 0x00251051, alias: "DRN", vr: VR::US },
            E { tag: 0x00251052, alias: "DRE", vr: VR::US },
            E { tag: 0x00251053, alias: "DN", vr: VR::US },
            E { tag: 0x00251060, alias: "ORN", vr: VR::US },
            E { tag: 0x00251061, alias: "ORE", vr: VR::US },
            E { tag: 0x00251062, alias: "ON", vr: VR::US },
            E { tag: 0x00251063, alias: "ORD", vr: VR::CS },
            E { tag: 0x00251070, alias: "MDT", vr: VR::US },
            E { tag: 0x00251071, alias: "MDB", vr: VR::US },
            E { tag: 0x00251072, alias: "MDE", vr: VR::US },
            E { tag: 0x00251073, alias: "MFP-DRC Suppression Characteristics Parameters", vr: VR::US },
            E { tag: 0x00251074, alias: "MFP-DRC Frequency Characteristics Parameters", vr: VR::US },
            E { tag: 0x00251080, alias: "MRT", vr: VR::US },
            E { tag: 0x00251081, alias: "MRB", vr: VR::US },
            E { tag: 0x00251082, alias: "MRE", vr: VR::US },
            E { tag: 0x00251083, alias: "MFP-USM Suppression Characteristics Parameters", vr: VR::US },
            E { tag: 0x00251084, alias: "MFP-USM Frequency Characteristics Parameters", vr: VR::US },
            E { tag: 0x00251090, alias: "PTE (α edge)", vr: VR::US },
            E { tag: 0x00251091, alias: "PTR (α calc)", vr: VR::US },
            E { tag: 0x00251092, alias: "PRN", vr: VR::US },
            E { tag: 0x00251093, alias: "PRE", vr: VR::US },
            E { tag: 0x00251094, alias: "PEM Sstd", vr: VR::US },
            E { tag: 0x00251095, alias: "PEM Lstd", vr: VR::US },
            E { tag: 0x00251096, alias: "PEM Unsharp Mask Value N", vr: VR::CS },
            E { tag: 0x002510A0, alias: "MFP GT", vr: VR::US },
            E { tag: 0x002510A1, alias: "MFP GA", vr: VR::SS },
            E { tag: 0x002510A2, alias: "MFP GC", vr: VR::US },
            E { tag: 0x002510A3, alias: "MFP GS", vr: VR::SS },
            E { tag: 0x00271010, alias: "γ Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271020, alias: "β Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271030, alias: "DRC Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271040, alias: "MFP-DRC Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271050, alias: "MFP-USM Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271060, alias: "PEM α Edge Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271070, alias: "PEM α Calc Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x00271080, alias: "MFP γ Raw Data Table Sequence", vr: VR::SQ },
            E { tag: 0x002710A0, alias: "No. of Data", vr: VR::IS },
            E { tag: 0x002710A1, alias: "Data Input Type", vr: VR::CS },
            E { tag: 0x002710A2, alias: "Data Output Type", vr: VR::CS },
            E { tag: 0x002710A3, alias: "Raw Data", vr: VR::US },
            E { tag: 0x00291020, alias: "Image Scanning Direction", vr: VR::CS },
            E { tag: 0x00291025, alias: "Image Rotation/Reversal Information", vr: VR::CS },
            E { tag: 0x00291030, alias: "Extended Reading Size Value", vr: VR::CS },
            E { tag: 0x00291034, alias: "Mag./Reduc. Ratio", vr: VR::US },
            E { tag: 0x00291044, alias: "Line Density Code", vr: VR::CS },
            E { tag: 0x00291050, alias: "Data Compression Code", vr: VR::CS },
            E { tag: 0x00321032, alias: "Requesting Physician", vr: VR::PN },
            E { tag: 0x00321033, alias: "Requesting Service", vr: VR::LO },
            E { tag: 0x20111000, alias: "Trim Density", vr: VR::CS },
            E { tag: 0x20111001, alias: "Trim Width", vr: VR::IS },
            E { tag: 0x20111002, alias: "Image Mag./Reduc. Range", vr: VR::CS },
            E { tag: 0x20111010, alias: "Image Display Format", vr: VR::CS },
            E { tag: 0x20111011, alias: "Image Position Specifying Flag", vr: VR::CS },
            E { tag: 0x20111020, alias: "Interpolation A-VRS System Format", vr: VR::OW },
            E { tag: 0x50F11006, alias: "Energy Subtraction Param.", vr: VR::CS },
            E { tag: 0x50F11007, alias: "Subtraction Registration Result", vr: VR::CS },
            E { tag: 0x50F11008, alias: "Energy Subtraction Param. 2", vr: VR::CS },
            E { tag: 0x50F11009, alias: "Afin Conversion Coefficient", vr: VR::SL },
            E { tag: 0x50F1100A, alias: "FNC Parameters", vr: VR::SH },
            E { tag: 0x50F11010, alias: "Film Output Format", vr: VR::CS },
            E { tag: 0x50F11020, alias: "Image Processing Modification Flag", vr: VR::CS },
        ],
    },
    P {
        creator: "FFP DATA",
        entries: &[
            E { tag: 0x00091001, alias: "CR Header Information", vr: VR::UN },
        ],
    },
    P {
        creator: "GDCM CONFORMANCE TESTS",
        entries: &[
            E { tag: 0x4D4D1001, alias: "Array Container Sequence", vr: VR::SQ },
            E { tag: 0x4D4D1002, alias: "0-Filled Array", vr: VR::OB },
        ],
    },
    P {
        creator: "GE ??? From Adantage Review CS",
        entries: &[
            E { tag: 0x00191030, alias: "CR EDR Mode", vr: VR::LO },
            E { tag: 0x00191040, alias: "CR Latitude", vr: VR::LO },
            E { tag: 0x00191050, alias: "CR Group Number", vr: VR::LO },
            E { tag: 0x00191070, alias: "CR Image Serial Number", vr: VR::LO },
            E { tag: 0x00191080, alias: "CR Bar Code Number", vr: VR::LO },
            E { tag: 0x00191090, alias: "CR Film Output Exposures", vr: VR::LO },
        ],
    },
    P {
        creator: "GE LUT Asymmetry Parameter",
        entries: &[
            E { tag: 0x00451067, alias: "LUT Asymmetry", vr: VR::DS },
        ],
    },
    P {
        creator: "GEHC_CT_ADVAPP_001",
        entries: &[
            E { tag: 0x00531020, alias: "ShuttleFlag", vr: VR::IS },
            E { tag: 0x00531021, alias: "TableSpeedNotReachesTargetFlag", vr: VR::IS },
            E { tag: 0x00531040, alias: "IterativeReconAnnotation", vr: VR::SH },
            E { tag: 0x00531041, alias: "IterativeReconMode", vr: VR::SH },
            E { tag: 0x00531042, alias: "IterativeReconConfiguration", vr: VR::LO },
            E { tag: 0x00531043, alias: "IterativeReconLevel", vr: VR::SH },
            E { tag: 0x00531060, alias: "reconFlipRotateAnno", vr: VR::SH },
            E { tag: 0x00531061, alias: "highResolutionFlag", vr: VR::SH },
            E { tag: 0x00531062, alias: "RespiratoryFlag", vr: VR::SH },
        ],
    },
    P {
        creator: "GEIIS",
        entries: &[
            E { tag: 0x00091010, alias: "GE IIS Thumbnail Sequence", vr: VR::SQ },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291010, alias: "Shift Count", vr: VR::UL },
            E { tag: 0x00291012, alias: "Offset", vr: VR::UL },
            E { tag: 0x00291014, alias: "Actual Frame Number", vr: VR::UL },
            E { tag: 0x004B1013, alias: "Unknown", vr: VR::IS },
            E { tag: 0x004B1015, alias: "Unknown", vr: VR::LT },
            E { tag: 0x09051030, alias: "Assigning Authority For Patient ID", vr: VR::LO },
            E { tag: 0x09071010, alias: "Original Study Instance UID", vr: VR::UI },
            E { tag: 0x09071020, alias: "Original Series Instance UID", vr: VR::UI },
            E { tag: 0x09071030, alias: "Original SOP Instance UID", vr: VR::UI },
            E { tag: 0x7FD11010, alias: "GE IIS Compression ID", vr: VR::UL },
            E { tag: 0x7FD11020, alias: "GE IIS Multiframe Offsets", vr: VR::UL },
            E { tag: 0x7FD11030, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FD11040, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FD11050, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FD11060, alias: "Unknown", vr: VR::UL },
        ],
    },
    P {
        creator: "GEIIS PACS",
        entries: &[
            E { tag: 0x09031010, alias: "Reject Image Flag", vr: VR::US },
            E { tag: 0x09031011, alias: "Significant Flag", vr: VR::US },
            E { tag: 0x09031012, alias: "Confidential Flag", vr: VR::US },
            E { tag: 0x09031020, alias: "Unknown", vr: VR::CS },
            E { tag: 0x09071021, alias: "GEIIS_PREFETCH_ALGORITHM", vr: VR::US },
            E { tag: 0x09071022, alias: "GEIIS_LIMIT_RECENT_STUDIES", vr: VR::US },
            E { tag: 0x09071023, alias: "GEIIS_LIMIT_OLDEST_STUDIES", vr: VR::US },
            E { tag: 0x09071024, alias: "GEIIS_LIMIT_RECENT_MONTHS", vr: VR::US },
            E { tag: 0x09071031, alias: "GEIIS_EXCLUDE_STUDY_UIDS", vr: VR::UI },
        ],
    },
    P {
        creator: "GEMS-IT/Centricity RA600/7.0",
        entries: &[
            E { tag: 0x41131010, alias: "Number of images in study", vr: VR::UI },
        ],
    },
    P {
        creator: "GEMS_0039",
        entries: &[
            E { tag: 0x00391095, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_3DSTATE_001",
        entries: &[
            E { tag: 0x004710D6, alias: "General Description", vr: VR::ST },
            E { tag: 0x004710D7, alias: "TDRT", vr: VR::CS },
            E { tag: 0x004710D8, alias: "NVRP", vr: VR::US },
            E { tag: 0x004710D9, alias: "CVRPN", vr: VR::US },
            E { tag: 0x004710DA, alias: "Volume Rendering Presets Sequence", vr: VR::SQ },
            E { tag: 0x004710DB, alias: "Preset Name", vr: VR::LO },
            E { tag: 0x004710DC, alias: "Opacity Curve X", vr: VR::SS },
            E { tag: 0x004710DD, alias: "Opacity Curve Y", vr: VR::FL },
            E { tag: 0x004710DE, alias: "NOCP", vr: VR::US },
            E { tag: 0x004710DF, alias: "Color Curve X", vr: VR::SS },
            E { tag: 0x004710E1, alias: "NCCP", vr: VR::US },
            E { tag: 0x004710E2, alias: "GSA", vr: VR::CS },
            E { tag: 0x004710E3, alias: "VRSF", vr: VR::CS },
            E { tag: 0x004710E4, alias: "AF", vr: VR::FL },
            E { tag: 0x004710E5, alias: "DF", vr: VR::FL },
            E { tag: 0x004710E6, alias: "SCF", vr: VR::FL },
            E { tag: 0x004710E7, alias: "SPF", vr: VR::FL },
            E { tag: 0x004710E8, alias: "Orthogonal Clipping Planes", vr: VR::FL },
            E { tag: 0x004710E9, alias: "CP", vr: VR::FL },
            E { tag: 0x004710EA, alias: "CFP", vr: VR::DS },
            E { tag: 0x004710EB, alias: "CVU", vr: VR::DS },
            E { tag: 0x004710EC, alias: "RFOV", vr: VR::FL },
            E { tag: 0x004710ED, alias: "PPRP", vr: VR::CS },
            E { tag: 0x004710EE, alias: "3DWW", vr: VR::DS },
            E { tag: 0x004710EF, alias: "3DWL", vr: VR::DS },
            E { tag: 0x004710F0, alias: "BBV", vr: VR::CS },
            E { tag: 0x004710F1, alias: "ERF", vr: VR::CS },
            E { tag: 0x004710F2, alias: "TDRMS", vr: VR::US },
            E { tag: 0x004710F3, alias: "TDSSS", vr: VR::FL },
        ],
    },
    P {
        creator: "GEMS_ACQU_01",
        entries: &[
            E { tag: 0x00091024, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00091025, alias: "Unknown", vr: VR::US },
            E { tag: 0x0009103E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0009103F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091043, alias: "Unknown", vr: VR::US },
            E { tag: 0x000910F8, alias: "Unknown", vr: VR::US },
            E { tag: 0x000910FB, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00191001, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191002, alias: "Detector Channel", vr: VR::SL },
            E { tag: 0x00191003, alias: "Cell number at Theta", vr: VR::DS },
            E { tag: 0x00191004, alias: "Cell spacing", vr: VR::DS },
            E { tag: 0x00191005, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191006, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019100E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019100F, alias: "Horiz. Frame of ref.", vr: VR::DS },
            E { tag: 0x00191011, alias: "Series contrast", vr: VR::SS },
            E { tag: 0x00191012, alias: "Last pseq", vr: VR::SS },
            E { tag: 0x00191013, alias: "Start number for baseline", vr: VR::SS },
            E { tag: 0x00191014, alias: "End number for baseline", vr: VR::SS },
            E { tag: 0x00191015, alias: "Start number for enhanced scans", vr: VR::SS },
            E { tag: 0x00191016, alias: "End number for enhanced scans", vr: VR::SS },
            E { tag: 0x00191017, alias: "Series plane", vr: VR::SS },
            E { tag: 0x00191018, alias: "First scan ras", vr: VR::LO },
            E { tag: 0x00191019, alias: "First scan location", vr: VR::DS },
            E { tag: 0x0019101A, alias: "Last scan ras", vr: VR::LO },
            E { tag: 0x0019101B, alias: "Last scan loc", vr: VR::DS },
            E { tag: 0x0019101E, alias: "Display field of view", vr: VR::DS },
            E { tag: 0x00191020, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191022, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191023, alias: "Table Speed [mm/rotation]", vr: VR::DS },
            E { tag: 0x00191024, alias: "Mid Scan Time [sec]", vr: VR::DS },
            E { tag: 0x00191025, alias: "Mid scan flag", vr: VR::SS },
            E { tag: 0x00191026, alias: "Tube Azimuth [degree]", vr: VR::SL },
            E { tag: 0x00191027, alias: "Rotation Speed [msec]", vr: VR::DS },
            E { tag: 0x0019102A, alias: "x-ray On position", vr: VR::DS },
            E { tag: 0x0019102B, alias: "x-ray Off position", vr: VR::DS },
            E { tag: 0x0019102C, alias: "Number of triggers", vr: VR::SL },
            E { tag: 0x0019102D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019102E, alias: "Angle of first view", vr: VR::DS },
            E { tag: 0x0019102F, alias: "Trigger frequency", vr: VR::DS },
            E { tag: 0x00191039, alias: "SFOV Type", vr: VR::SS },
            E { tag: 0x0019103A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019103B, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0019103C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019103E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019103F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191040, alias: "Stat recon flag", vr: VR::SS },
            E { tag: 0x00191041, alias: "Compute type", vr: VR::SS },
            E { tag: 0x00191042, alias: "Segment Number", vr: VR::SS },
            E { tag: 0x00191043, alias: "Total Segments Required", vr: VR::SS },
            E { tag: 0x00191044, alias: "Interscan delay", vr: VR::DS },
            E { tag: 0x00191047, alias: "View compression factor", vr: VR::SS },
            E { tag: 0x00191048, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191049, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019104A, alias: "Total no. of ref channels", vr: VR::SS },
            E { tag: 0x0019104B, alias: "Data size for scan data", vr: VR::SL },
            E { tag: 0x00191052, alias: "Recon post proc. Flag", vr: VR::SS },
            E { tag: 0x00191054, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191057, alias: "CT water number", vr: VR::SS },
            E { tag: 0x00191058, alias: "CT bone number", vr: VR::SS },
            E { tag: 0x0019105A, alias: "Acquisition Duration", vr: VR::FL },
            E { tag: 0x0019105D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019105E, alias: "Number of channels (1...512)", vr: VR::SL },
            E { tag: 0x0019105F, alias: "Increment between channels", vr: VR::SL },
            E { tag: 0x00191060, alias: "Starting view", vr: VR::SL },
            E { tag: 0x00191061, alias: "Number of views", vr: VR::SL },
            E { tag: 0x00191062, alias: "Increment between views", vr: VR::SL },
            E { tag: 0x0019106A, alias: "Dependent on #views processed", vr: VR::SS },
            E { tag: 0x0019106B, alias: "Field of view in detector cells", vr: VR::SS },
            E { tag: 0x00191070, alias: "Value of back projection button", vr: VR::SS },
            E { tag: 0x00191071, alias: "Set if fatq estimates were used", vr: VR::SS },
            E { tag: 0x00191072, alias: "Z chan avg over views", vr: VR::DS },
            E { tag: 0x00191073, alias: "Avg of left ref chans over views", vr: VR::DS },
            E { tag: 0x00191074, alias: "Max left chan over views", vr: VR::DS },
            E { tag: 0x00191075, alias: "Avg of right ref chans over views", vr: VR::DS },
            E { tag: 0x00191076, alias: "Max right chan over views", vr: VR::DS },
            E { tag: 0x0019107D, alias: "Second echo", vr: VR::DS },
            E { tag: 0x0019107E, alias: "Number of echoes", vr: VR::SS },
            E { tag: 0x0019107F, alias: "Table delta", vr: VR::DS },
            E { tag: 0x00191081, alias: "Contiguous", vr: VR::SS },
            E { tag: 0x00191082, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191083, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191084, alias: "Peak SAR", vr: VR::DS },
            E { tag: 0x00191085, alias: "Monitor SAR", vr: VR::SS },
            E { tag: 0x00191086, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191087, alias: "Cardiac repetition time", vr: VR::DS },
            E { tag: 0x00191088, alias: "Images per cardiac cycle", vr: VR::SS },
            E { tag: 0x0019108A, alias: "Actual receive gain analog", vr: VR::SS },
            E { tag: 0x0019108B, alias: "Actual receive gain digital", vr: VR::SS },
            E { tag: 0x0019108D, alias: "Delay after trigger", vr: VR::DS },
            E { tag: 0x0019108F, alias: "Swappf", vr: VR::SS },
            E { tag: 0x00191090, alias: "Pause Interval", vr: VR::SS },
            E { tag: 0x00191091, alias: "Pause Time", vr: VR::DS },
            E { tag: 0x00191092, alias: "Slice offset on freq axis", vr: VR::SL },
            E { tag: 0x00191093, alias: "Auto Prescan Center Frequency", vr: VR::DS },
            E { tag: 0x00191094, alias: "Auto Prescan Transmit Gain", vr: VR::SS },
            E { tag: 0x00191095, alias: "Auto Prescan Analog receiver gain", vr: VR::SS },
            E { tag: 0x00191096, alias: "Auto Prescan Digital receiver gain", vr: VR::SS },
            E { tag: 0x00191097, alias: "Bitmap defining CVs", vr: VR::SL },
            E { tag: 0x00191098, alias: "Center freq. Method", vr: VR::SS },
            E { tag: 0x00191099, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019109B, alias: "Pulse Sequence Mode", vr: VR::SS },
            E { tag: 0x0019109C, alias: "Pulse Sequence Name", vr: VR::LO },
            E { tag: 0x0019109D, alias: "Pulse Sequence Date", vr: VR::DT },
            E { tag: 0x0019109E, alias: "Internal Pulse Sequence Name", vr: VR::LO },
            E { tag: 0x0019109F, alias: "Transmitting Coil Type", vr: VR::SS },
            E { tag: 0x001910A0, alias: "Surface Coil Type", vr: VR::SS },
            E { tag: 0x001910A1, alias: "Extremity Coil flag", vr: VR::SS },
            E { tag: 0x001910A2, alias: "Raw data run number", vr: VR::SL },
            E { tag: 0x001910A3, alias: "Calibrated Field strength", vr: VR::UL },
            E { tag: 0x001910A4, alias: "SAT fat/water/bone", vr: VR::SS },
            E { tag: 0x001910A5, alias: "Receive bandwidth", vr: VR::DS },
            E { tag: 0x001910A7, alias: "User data 0", vr: VR::DS },
            E { tag: 0x001910A8, alias: "User data 1", vr: VR::DS },
            E { tag: 0x001910A9, alias: "User data 2", vr: VR::DS },
            E { tag: 0x001910AA, alias: "User data 3", vr: VR::DS },
            E { tag: 0x001910AB, alias: "User data 4", vr: VR::DS },
            E { tag: 0x001910AC, alias: "User data 5", vr: VR::DS },
            E { tag: 0x001910AD, alias: "User data 6", vr: VR::DS },
            E { tag: 0x001910AE, alias: "User data 7", vr: VR::DS },
            E { tag: 0x001910AF, alias: "User data 8", vr: VR::DS },
            E { tag: 0x001910B0, alias: "User data 9", vr: VR::DS },
            E { tag: 0x001910B1, alias: "User data 10", vr: VR::DS },
            E { tag: 0x001910B2, alias: "User data 11", vr: VR::DS },
            E { tag: 0x001910B3, alias: "User data 12", vr: VR::DS },
            E { tag: 0x001910B4, alias: "User data 13", vr: VR::DS },
            E { tag: 0x001910B5, alias: "User data 14", vr: VR::DS },
            E { tag: 0x001910B6, alias: "User data 15", vr: VR::DS },
            E { tag: 0x001910B7, alias: "User data 16", vr: VR::DS },
            E { tag: 0x001910B8, alias: "User data 17", vr: VR::DS },
            E { tag: 0x001910B9, alias: "User data 18", vr: VR::DS },
            E { tag: 0x001910BA, alias: "User data 19", vr: VR::DS },
            E { tag: 0x001910BB, alias: "User data 20", vr: VR::DS },
            E { tag: 0x001910BC, alias: "User data 21", vr: VR::DS },
            E { tag: 0x001910BD, alias: "User data 22", vr: VR::DS },
            E { tag: 0x001910BE, alias: "Projection angle", vr: VR::DS },
            E { tag: 0x001910C0, alias: "Saturation planes", vr: VR::SS },
            E { tag: 0x001910C1, alias: "Surface coil intensity", vr: VR::SS },
            E { tag: 0x001910C2, alias: "SAT location R", vr: VR::SS },
            E { tag: 0x001910C3, alias: "SAT location L", vr: VR::SS },
            E { tag: 0x001910C4, alias: "SAT location A", vr: VR::SS },
            E { tag: 0x001910C5, alias: "SAT location P", vr: VR::SS },
            E { tag: 0x001910C6, alias: "SAT location H", vr: VR::SS },
            E { tag: 0x001910C7, alias: "SAT location F", vr: VR::SS },
            E { tag: 0x001910C8, alias: "SAT thickness R/L", vr: VR::SS },
            E { tag: 0x001910C9, alias: "SAT thickness A/P", vr: VR::SS },
            E { tag: 0x001910CA, alias: "SAT thickness H/F", vr: VR::SS },
            E { tag: 0x001910CB, alias: "Phase Contrast flow axis", vr: VR::SS },
            E { tag: 0x001910CC, alias: "Velocity encoding", vr: VR::SS },
            E { tag: 0x001910CD, alias: "Thickness disclaimer", vr: VR::SS },
            E { tag: 0x001910CE, alias: "Prescan type", vr: VR::SS },
            E { tag: 0x001910CF, alias: "Prescan status", vr: VR::SS },
            E { tag: 0x001910D0, alias: "Raw data type", vr: VR::SH },
            E { tag: 0x001910D2, alias: "Projection Algorithm", vr: VR::SS },
            E { tag: 0x001910D3, alias: "Projection Algorithm Name", vr: VR::SH },
            E { tag: 0x001910D4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910D5, alias: "Fractional echo", vr: VR::SS },
            E { tag: 0x001910D6, alias: "Prep pulse", vr: VR::SS },
            E { tag: 0x001910D7, alias: "Cardiac phase number", vr: VR::SS },
            E { tag: 0x001910D8, alias: "Variable echoflag", vr: VR::SS },
            E { tag: 0x001910D9, alias: "Concatenated SAT {# DTI Diffusion Dir., release 9.0 & below}", vr: VR::DS },
            E { tag: 0x001910DA, alias: "Reference channel used", vr: VR::SS },
            E { tag: 0x001910DB, alias: "Back projector coefficient", vr: VR::DS },
            E { tag: 0x001910DC, alias: "Primary speed correction used", vr: VR::SS },
            E { tag: 0x001910DD, alias: "Overrange correction used", vr: VR::SS },
            E { tag: 0x001910DE, alias: "Dynamic Z alpha value", vr: VR::DS },
            E { tag: 0x001910DF, alias: "User data 23 {# DTI Diffusion Dir., release 9.0 & below}", vr: VR::DS },
            E { tag: 0x001910E0, alias: "User data 24 {# DTI Diffusion Dir., release 10.0 & above}", vr: VR::DS },
            E { tag: 0x001910E1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910E2, alias: "Velocity Encode Scale", vr: VR::DS },
            E { tag: 0x001910E3, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001910E4, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001910E5, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910E6, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910E8, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910E9, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910EB, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910F0, alias: "Unknown", vr: VR::UN },
            E { tag: 0x001910F1, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001910F2, alias: "Fast phases", vr: VR::SS },
            E { tag: 0x001910F3, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001910F4, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001910F9, alias: "Transmit gain", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_ACRQA_1.0 BLOCK1",
        entries: &[
            E { tag: 0x00231000, alias: "CR Exposure Menu Code", vr: VR::LO },
            E { tag: 0x00231010, alias: "CR Exposure Menu String", vr: VR::LO },
            E { tag: 0x00231020, alias: "CR EDR Mode", vr: VR::LO },
            E { tag: 0x00231030, alias: "CR Latitude", vr: VR::LO },
            E { tag: 0x00231040, alias: "CR Group Number", vr: VR::LO },
            E { tag: 0x00231050, alias: "CR Image Serial Number", vr: VR::US },
            E { tag: 0x00231060, alias: "CR Bar Code Number", vr: VR::LO },
            E { tag: 0x00231070, alias: "CR Film Output Exposure", vr: VR::LO },
            E { tag: 0x00231080, alias: "CR Film Format", vr: VR::LO },
            E { tag: 0x00231090, alias: "CR S-Shift String", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_ACRQA_1.0 BLOCK2",
        entries: &[
            E { tag: 0x00231000, alias: "CR S-Shift", vr: VR::US },
            E { tag: 0x00231010, alias: "CR C-Shift", vr: VR::DS },
            E { tag: 0x00231020, alias: "CR GT", vr: VR::DS },
            E { tag: 0x00231030, alias: "CR GA", vr: VR::DS },
            E { tag: 0x00231040, alias: "CR GC", vr: VR::DS },
            E { tag: 0x00231050, alias: "CR GS", vr: VR::DS },
            E { tag: 0x00231060, alias: "CR RT", vr: VR::DS },
            E { tag: 0x00231070, alias: "CR RE", vr: VR::DS },
            E { tag: 0x00231080, alias: "CR RN", vr: VR::US },
            E { tag: 0x00231090, alias: "CR DRT", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_ACRQA_1.0 BLOCK3",
        entries: &[
            E { tag: 0x00231000, alias: "CR DRE", vr: VR::DS },
            E { tag: 0x00231010, alias: "CR DRN", vr: VR::US },
            E { tag: 0x00231020, alias: "CR ORE", vr: VR::DS },
            E { tag: 0x00231030, alias: "CR ORN", vr: VR::US },
            E { tag: 0x00231040, alias: "CR ORD", vr: VR::US },
            E { tag: 0x00231050, alias: "CR Cassette Size", vr: VR::LO },
            E { tag: 0x00231060, alias: "CR Machine ID", vr: VR::LO },
            E { tag: 0x00231070, alias: "CR Machine Type", vr: VR::LO },
            E { tag: 0x00231080, alias: "CR Technician Code", vr: VR::LO },
            E { tag: 0x00231090, alias: "CR ES Parameters", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_ACRQA_2.0 BLOCK1",
        entries: &[
            E { tag: 0x00231000, alias: "CR Exposure Menu Code", vr: VR::LO },
            E { tag: 0x00231010, alias: "CR Exposure Menu String", vr: VR::LO },
            E { tag: 0x00231020, alias: "CR EDR Mode", vr: VR::LO },
            E { tag: 0x00231030, alias: "CR Latitude", vr: VR::LO },
            E { tag: 0x00231040, alias: "CR Group Number", vr: VR::LO },
            E { tag: 0x00231050, alias: "CR Image Serial Number", vr: VR::US },
            E { tag: 0x00231060, alias: "CR Bar Code Number", vr: VR::LO },
            E { tag: 0x00231070, alias: "CR Film Output Exposure", vr: VR::LO },
            E { tag: 0x00231080, alias: "CR Film Format", vr: VR::LO },
            E { tag: 0x00231090, alias: "CR S Shift String", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_ACRQA_2.0 BLOCK2",
        entries: &[
            E { tag: 0x00231000, alias: "CR S Shift", vr: VR::US },
            E { tag: 0x00231010, alias: "CR C Shift", vr: VR::LO },
            E { tag: 0x00231020, alias: "CR GT", vr: VR::LO },
            E { tag: 0x00231030, alias: "CR GA", vr: VR::DS },
            E { tag: 0x00231040, alias: "CR GC", vr: VR::DS },
            E { tag: 0x00231050, alias: "CR GS", vr: VR::DS },
            E { tag: 0x00231060, alias: "CR RT", vr: VR::LO },
            E { tag: 0x00231070, alias: "CR RE", vr: VR::DS },
            E { tag: 0x00231080, alias: "CR RN", vr: VR::US },
            E { tag: 0x00231090, alias: "CR DRT", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_ACRQA_2.0 BLOCK3",
        entries: &[
            E { tag: 0x00231000, alias: "CR DRE", vr: VR::DS },
            E { tag: 0x00231010, alias: "CR DRN", vr: VR::US },
            E { tag: 0x00231020, alias: "CR ORE", vr: VR::DS },
            E { tag: 0x00231030, alias: "CR ORN", vr: VR::US },
            E { tag: 0x00231040, alias: "CR ORD", vr: VR::US },
            E { tag: 0x00231050, alias: "CR Cassette Size", vr: VR::LO },
            E { tag: 0x00231060, alias: "CR Machine ID", vr: VR::LO },
            E { tag: 0x00231070, alias: "CR Machine Type", vr: VR::LO },
            E { tag: 0x00231080, alias: "CR Technician Code", vr: VR::LO },
            E { tag: 0x00231090, alias: "CR Energy Subtraction Parameters", vr: VR::LO },
            E { tag: 0x002310F0, alias: "CR Distribution Code", vr: VR::LO },
            E { tag: 0x002310FF, alias: "CR Shutters Applied", vr: VR::US },
        ],
    },
    P {
        creator: "GEMS_ADWSoft_3D1",
        entries: &[
            E { tag: 0x00471001, alias: "Reconstruction Parameters Sequence", vr: VR::SQ },
            E { tag: 0x00471049, alias: "Volume Color", vr: VR::UL },
            E { tag: 0x00471050, alias: "Volume Voxel Count", vr: VR::UL },
            E { tag: 0x00471051, alias: "Volume Segment Count", vr: VR::UL },
            E { tag: 0x00471053, alias: "Volume Slice Size", vr: VR::US },
            E { tag: 0x00471054, alias: "Volume Slice Count", vr: VR::US },
            E { tag: 0x00471055, alias: "Volume Threshold Value", vr: VR::SL },
            E { tag: 0x00471057, alias: "Volume Voxel Ratio", vr: VR::DS },
            E { tag: 0x00471058, alias: "Volume Voxel Size", vr: VR::DS },
            E { tag: 0x00471059, alias: "Volume Z Position Size", vr: VR::US },
            E { tag: 0x00471060, alias: "Volume Base Line", vr: VR::DS },
            E { tag: 0x00471061, alias: "Volume Center Point", vr: VR::DS },
            E { tag: 0x00471063, alias: "Volume Skew Base", vr: VR::SL },
            E { tag: 0x00471064, alias: "Volume Registration Transform Rotation Matrix", vr: VR::DS },
            E { tag: 0x00471065, alias: "Volume Registration Transform Translation Vector", vr: VR::DS },
            E { tag: 0x00471066, alias: "Volume Tilt", vr: VR::DS },
            E { tag: 0x00471070, alias: "KPV List", vr: VR::DS },
            E { tag: 0x00471071, alias: "X-Ray Tube Current List", vr: VR::IS },
            E { tag: 0x00471072, alias: "Exposure List", vr: VR::IS },
            E { tag: 0x00471080, alias: "Acquisition DLX Identifier", vr: VR::LO },
            E { tag: 0x00471081, alias: "Acquisition DLX 2D Series Count", vr: VR::IS },
            E { tag: 0x00471085, alias: "Acquisition DLX 2D Series Sequence", vr: VR::SQ },
            E { tag: 0x00471089, alias: "Contrast Agent Volume List", vr: VR::DS },
            E { tag: 0x0047108A, alias: "Number Of Injections", vr: VR::US },
            E { tag: 0x0047108B, alias: "Frame Count", vr: VR::US },
            E { tag: 0x00471091, alias: "XA 3D Reconstruction Algorithm Name", vr: VR::LO },
            E { tag: 0x00471092, alias: "XA 3D Reconstruction Algorithm Version", vr: VR::CS },
            E { tag: 0x00471093, alias: "DLX Calibration Date", vr: VR::DA },
            E { tag: 0x00471094, alias: "DLX Calibration Time", vr: VR::TM },
            E { tag: 0x00471095, alias: "DLX Calibration Status", vr: VR::CS },
            E { tag: 0x00471096, alias: "Used Frames", vr: VR::IS },
            E { tag: 0x00471098, alias: "Transform Count", vr: VR::US },
            E { tag: 0x00471099, alias: "Transform Sequence", vr: VR::SQ },
            E { tag: 0x0047109A, alias: "Transform Rotation Matrix", vr: VR::DS },
            E { tag: 0x0047109B, alias: "Transform Translation Vector", vr: VR::DS },
            E { tag: 0x0047109C, alias: "Transform Label", vr: VR::LO },
            E { tag: 0x004710B0, alias: "Wireframe List", vr: VR::SQ },
            E { tag: 0x004710B1, alias: "Wireframe Count", vr: VR::US },
            E { tag: 0x004710B2, alias: "Location System", vr: VR::US },
            E { tag: 0x004710B5, alias: "Wireframe Name", vr: VR::LO },
            E { tag: 0x004710B6, alias: "Wireframe Group Name", vr: VR::LO },
            E { tag: 0x004710B7, alias: "Wireframe Color", vr: VR::LO },
            E { tag: 0x004710B8, alias: "Wireframe Attributes", vr: VR::SL },
            E { tag: 0x004710B9, alias: "Wireframe Point Count", vr: VR::SL },
            E { tag: 0x004710BA, alias: "Wireframe Timestamp", vr: VR::SL },
            E { tag: 0x004710BB, alias: "Wireframe Point List", vr: VR::SQ },
            E { tag: 0x004710BC, alias: "Wireframe Points Coordinates", vr: VR::DS },
            E { tag: 0x004710C0, alias: "Volume Upper Left High Corner RAS", vr: VR::DS },
            E { tag: 0x004710C1, alias: "Volume Slice To RAS Rotation Matrix", vr: VR::DS },
            E { tag: 0x004710C2, alias: "Volume Upper Left High Corner TLOC", vr: VR::DS },
            E { tag: 0x004710D1, alias: "Volume Segment List", vr: VR::OB },
            E { tag: 0x004710D2, alias: "Volume Gradient List", vr: VR::OB },
            E { tag: 0x004710D3, alias: "Volume Density List", vr: VR::OB },
            E { tag: 0x004710D4, alias: "Volume Z Position List", vr: VR::OB },
            E { tag: 0x004710D5, alias: "Volume Original Index List", vr: VR::OB },
            E { tag: 0x004710F4, alias: "Volume Name(s)", vr: VR::LO },
            E { tag: 0x004710F5, alias: "Min original density", vr: VR::DS },
            E { tag: 0x004710F6, alias: "Max original density", vr: VR::DS },
            E { tag: 0x004710F7, alias: "Min Converted Density", vr: VR::DS },
            E { tag: 0x004710F8, alias: "Max Converted Density", vr: VR::DS },
            E { tag: 0x004710F9, alias: "Protocol Film Name", vr: VR::LO },
            E { tag: 0x004710FA, alias: "Protocol Resolution", vr: VR::US },
            E { tag: 0x004710FB, alias: "Phase Number (percent)", vr: VR::US },
            E { tag: 0x004710FC, alias: "Volume midscan times list", vr: VR::OB },
            E { tag: 0x004710FD, alias: "Volume Registered Phases List", vr: VR::OB },
            E { tag: 0x004710FE, alias: "Protocol Name", vr: VR::LO },
            E { tag: 0x004710FF, alias: "Protocol Title", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_ADWSoft_3D2",
        entries: &[
            E { tag: 0x00571001, alias: "Cardiac Reconstruction Algorithm List", vr: VR::OB },
            E { tag: 0x00571002, alias: "Average Heart Rate for Image List", vr: VR::OB },
            E { tag: 0x00571003, alias: "Temporal Resolution List", vr: VR::OB },
            E { tag: 0x00571004, alias: "Layout View Preset", vr: VR::LT },
        ],
    },
    P {
        creator: "GEMS_ADWSoft_DPO",
        entries: &[
            E { tag: 0x00391080, alias: "Private Entity Number", vr: VR::IS },
            E { tag: 0x00391085, alias: "Private Entity Date", vr: VR::DA },
            E { tag: 0x00391090, alias: "Private Entity Time", vr: VR::TM },
            E { tag: 0x00391095, alias: "Private Entity Launch Command", vr: VR::LO },
            E { tag: 0x003910AA, alias: "Private Entity Type", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_ADWSoft_DPO1",
        entries: &[
            E { tag: 0x00391080, alias: "Private Entity Number", vr: VR::IS },
            E { tag: 0x00391085, alias: "Private Entity Date", vr: VR::DA },
            E { tag: 0x00391090, alias: "Private Entity Time", vr: VR::TM },
            E { tag: 0x00391095, alias: "Private Entity Launch Command", vr: VR::LO },
            E { tag: 0x003910AA, alias: "Private Entity Type", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_AWSOFT_CD1",
        entries: &[
            E { tag: 0x00391065, alias: "Reference to Study UID", vr: VR::UI },
            E { tag: 0x00391070, alias: "Reference to Series UID", vr: VR::UI },
            E { tag: 0x00391075, alias: "Reference to Original Instance", vr: VR::IS },
            E { tag: 0x00391080, alias: "DPO Number", vr: VR::IS },
            E { tag: 0x00391085, alias: "DPO Date", vr: VR::DA },
            E { tag: 0x00391090, alias: "DPO Time", vr: VR::TM },
            E { tag: 0x00391095, alias: "DPO Invocation String", vr: VR::LO },
            E { tag: 0x003910AA, alias: "DPO type", vr: VR::CS },
            E { tag: 0x003910FF, alias: "DPO data", vr: VR::OB },
        ],
    },
    P {
        creator: "GEMS_AWSoft_SB1",
        entries: &[
            E { tag: 0x00391050, alias: "Reference to Study UID", vr: VR::UI },
            E { tag: 0x00391051, alias: "Reference to Series UID", vr: VR::UI },
            E { tag: 0x00391052, alias: "Reference to Original Instance Number", vr: VR::IS },
            E { tag: 0x00391095, alias: "Private Entity Launch Command", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_CTHD_01",
        entries: &[
            E { tag: 0x00331002, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "GEMS_CT_CARDIAC_001",
        entries: &[
            E { tag: 0x00491001, alias: "CT Cardiac Sequence", vr: VR::SQ },
            E { tag: 0x00491002, alias: "HeartRateAtConfirm", vr: VR::CS },
            E { tag: 0x00491003, alias: "AvgHeartRatePriorToConfirm", vr: VR::FL },
            E { tag: 0x00491004, alias: "MinHeartRatePriorToConfirm", vr: VR::CS },
            E { tag: 0x00491005, alias: "MaxHeartRatePriorToConfirm", vr: VR::CS },
            E { tag: 0x00491006, alias: "StdDevHeartRatePriorToConfirm", vr: VR::FL },
            E { tag: 0x00491007, alias: "NumHeartRateSamplesPriorToConfirm", vr: VR::US },
            E { tag: 0x00491008, alias: "AutoHeartRateDetectPredict", vr: VR::CS },
            E { tag: 0x00491009, alias: "SystemOptimizedHeartRate", vr: VR::CS },
            E { tag: 0x0049100A, alias: "EkgMonitorType", vr: VR::ST },
            E { tag: 0x0049100B, alias: "NumReconSectors", vr: VR::CS },
            E { tag: 0x0049100C, alias: "RpeakTimeStamps", vr: VR::FL },
        ],
    },
    P {
        creator: "GEMS_CT_HINO_01",
        entries: &[
            E { tag: 0x004B1001, alias: "Beam Thickess", vr: VR::DS },
            E { tag: 0x004B1002, alias: "R Time", vr: VR::DS },
            E { tag: 0x004B1003, alias: "HBC Number", vr: VR::IS },
        ],
    },
    P {
        creator: "GEMS_CT_VES_01",
        entries: &[
            E { tag: 0x00511001, alias: "CTVESSequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "GEMS_DLX_DOSE_01",
        entries: &[
            E { tag: 0x00271001, alias: "Private Radiation Dose Sequence", vr: VR::SQ },
            E { tag: 0x00271002, alias: "Run Number", vr: VR::IS },
            E { tag: 0x00271003, alias: "Run Time", vr: VR::TM },
            E { tag: 0x00271004, alias: "Number of Frames", vr: VR::IS },
            E { tag: 0x00271005, alias: "Frames per Second", vr: VR::DS },
            E { tag: 0x00271006, alias: "Plane", vr: VR::CS },
            E { tag: 0x00271007, alias: "KV", vr: VR::DS },
            E { tag: 0x00271008, alias: "MA", vr: VR::DS },
            E { tag: 0x00271009, alias: "Mas", vr: VR::DS },
            E { tag: 0x00271010, alias: "Ms", vr: VR::DS },
            E { tag: 0x00271011, alias: "Angulation", vr: VR::DS },
            E { tag: 0x00271012, alias: "Rotation", vr: VR::DS },
            E { tag: 0x00271013, alias: "Focal Distance", vr: VR::DS },
            E { tag: 0x00271014, alias: "Image Intensifier Mode", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_DL_FRAME_01",
        entries: &[
            E { tag: 0x00251002, alias: "Frame Id", vr: VR::IS },
            E { tag: 0x00251003, alias: "Distance Source To Detector", vr: VR::DS },
            E { tag: 0x00251004, alias: "Distance Source To Patient", vr: VR::DS },
            E { tag: 0x00251005, alias: "Distance Source To Skin", vr: VR::DS },
            E { tag: 0x00251006, alias: "Positioner Primary Angle", vr: VR::DS },
            E { tag: 0x00251007, alias: "Positioner Secondary Angle", vr: VR::DS },
            E { tag: 0x00251008, alias: "Beam Orientation", vr: VR::IS },
            E { tag: 0x00251009, alias: "Larm Angle", vr: VR::DS },
            E { tag: 0x0025100A, alias: "Frame Sequence", vr: VR::SQ },
            E { tag: 0x0025100B, alias: "Target Cnr", vr: VR::DS },
            E { tag: 0x00251010, alias: "Pivot Angle", vr: VR::DS },
            E { tag: 0x0025101A, alias: "Arc Angle", vr: VR::DS },
            E { tag: 0x0025101B, alias: "Table Vertical Position", vr: VR::DS },
            E { tag: 0x0025101C, alias: "Table Longitudinal Position", vr: VR::DS },
            E { tag: 0x0025101D, alias: "Table Lateral Position", vr: VR::DS },
            E { tag: 0x0025101E, alias: "Beam Cover Area", vr: VR::IS },
            E { tag: 0x0025101F, alias: "Kvp Actual", vr: VR::DS },
            E { tag: 0x00251020, alias: "Mas Actual", vr: VR::DS },
            E { tag: 0x00251021, alias: "Pw Actual", vr: VR::DS },
            E { tag: 0x00251022, alias: "Kvp Commanded", vr: VR::DS },
            E { tag: 0x00251023, alias: "Mas Commanded", vr: VR::DS },
            E { tag: 0x00251024, alias: "Pw Commanded", vr: VR::DS },
            E { tag: 0x00251025, alias: "Grid", vr: VR::CS },
            E { tag: 0x00251026, alias: "Sensor Feedback", vr: VR::DS },
            E { tag: 0x00251027, alias: "Target Entrance Dose", vr: VR::DS },
            E { tag: 0x00251028, alias: "Cnr Commanded", vr: VR::DS },
            E { tag: 0x00251029, alias: "Contrast Commanded", vr: VR::DS },
            E { tag: 0x0025102A, alias: "Ept Actual", vr: VR::DS },
            E { tag: 0x0025102B, alias: "Spectral Filter Znb", vr: VR::IS },
            E { tag: 0x0025102C, alias: "Spectral Filter Weight", vr: VR::DS },
            E { tag: 0x0025102D, alias: "Spectral Filter Density", vr: VR::DS },
            E { tag: 0x0025102E, alias: "Spectral Filter Thickness", vr: VR::IS },
            E { tag: 0x0025102F, alias: "Spectral Filter Status", vr: VR::IS },
            E { tag: 0x00251030, alias: "Fov Dimension", vr: VR::IS },
            E { tag: 0x00251033, alias: "Fov Origin", vr: VR::IS },
            E { tag: 0x00251034, alias: "Collimator Left Vertical Edge", vr: VR::IS },
            E { tag: 0x00251035, alias: "Collimator Right Vertical Edge", vr: VR::IS },
            E { tag: 0x00251036, alias: "Collimator Up Horizontal Edge", vr: VR::IS },
            E { tag: 0x00251037, alias: "Collimator Low Horizontal Edge", vr: VR::IS },
            E { tag: 0x00251038, alias: "Vertices Polygonal Collimator", vr: VR::IS },
            E { tag: 0x00251039, alias: "Contour Filter Distance", vr: VR::IS },
            E { tag: 0x0025103A, alias: "Contour Filter Angle", vr: VR::UL },
            E { tag: 0x0025103B, alias: "Table Rotation Status", vr: VR::CS },
            E { tag: 0x0025103C, alias: "Internal Label Frame", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_DL_IMG_01",
        entries: &[
            E { tag: 0x0019100B, alias: "Fov Dimension Double", vr: VR::DS },
            E { tag: 0x0019100C, alias: "LV Diastolic contour", vr: VR::FL },
            E { tag: 0x0019100D, alias: "LV Systolic contour", vr: VR::FL },
            E { tag: 0x0019102B, alias: "Distance to table top", vr: VR::FL },
            E { tag: 0x00191030, alias: "Image File Name", vr: VR::LO },
            E { tag: 0x00191031, alias: "Default Spatial Filter Family", vr: VR::IS },
            E { tag: 0x00191032, alias: "Default Spatial Filter Strength", vr: VR::IS },
            E { tag: 0x00191033, alias: "Min Saturation Dose", vr: VR::DS },
            E { tag: 0x00191034, alias: "Detector Gain", vr: VR::DS },
            E { tag: 0x00191035, alias: "Patient Dose Limit", vr: VR::DS },
            E { tag: 0x00191036, alias: "Preproc Image Rate Max", vr: VR::DS },
            E { tag: 0x00191037, alias: "Sensor Roi Shape", vr: VR::CS },
            E { tag: 0x00191038, alias: "Sensor Roi x Position", vr: VR::DS },
            E { tag: 0x00191039, alias: "Sensor Roi y Position", vr: VR::DS },
            E { tag: 0x0019103A, alias: "Sensor Roi x Size", vr: VR::DS },
            E { tag: 0x0019103B, alias: "Sensor Roi y Size", vr: VR::DS },
            E { tag: 0x0019103D, alias: "Noise Sensitivity", vr: VR::DS },
            E { tag: 0x0019103E, alias: "Sharp Sensitivity", vr: VR::DS },
            E { tag: 0x0019103F, alias: "Contrast Sensitivity", vr: VR::DS },
            E { tag: 0x00191040, alias: "Lag Sensitivity", vr: VR::DS },
            E { tag: 0x00191041, alias: "Tube", vr: VR::CS },
            E { tag: 0x00191042, alias: "Detector Size Rows", vr: VR::US },
            E { tag: 0x00191043, alias: "Detector Size Columns", vr: VR::US },
            E { tag: 0x00191044, alias: "Min Object Size", vr: VR::DS },
            E { tag: 0x00191045, alias: "Max Object Size", vr: VR::DS },
            E { tag: 0x00191046, alias: "Max Object Speed", vr: VR::DS },
            E { tag: 0x00191047, alias: "Object Back Motion", vr: VR::CS },
            E { tag: 0x00191048, alias: "Exposure Trajectory Family", vr: VR::UL },
            E { tag: 0x00191049, alias: "Window Time Duration", vr: VR::DS },
            E { tag: 0x0019104A, alias: "Positioner Angle Display Mode", vr: VR::CS },
            E { tag: 0x0019104B, alias: "Detector Origin", vr: VR::IS },
            E { tag: 0x0019104C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019104E, alias: "Default Brightness Contrast", vr: VR::DS },
            E { tag: 0x0019104F, alias: "User Brightness Contrast", vr: VR::DS },
            E { tag: 0x00191050, alias: "Source Series Number", vr: VR::IS },
            E { tag: 0x00191051, alias: "Source Image Number", vr: VR::IS },
            E { tag: 0x00191052, alias: "Source Frame Number", vr: VR::IS },
            E { tag: 0x00191053, alias: "Source Series Item Id", vr: VR::UI },
            E { tag: 0x00191054, alias: "Source Image Item Id", vr: VR::UI },
            E { tag: 0x00191055, alias: "Source Frame Item Id", vr: VR::UI },
            E { tag: 0x00191080, alias: "Image Dose", vr: VR::DS },
            E { tag: 0x00191081, alias: "Calibration Frame", vr: VR::US },
            E { tag: 0x00191082, alias: "Calibration Object", vr: VR::CS },
            E { tag: 0x00191083, alias: "Calibration Object Size mm", vr: VR::DS },
            E { tag: 0x00191084, alias: "Calibration Factor", vr: VR::FL },
            E { tag: 0x00191085, alias: "Calibration Date", vr: VR::DA },
            E { tag: 0x00191086, alias: "Calibration Time", vr: VR::TM },
            E { tag: 0x00191087, alias: "Calibration Accuracy", vr: VR::US },
            E { tag: 0x00191088, alias: "Calibration Extended", vr: VR::CS },
            E { tag: 0x00191089, alias: "Calibration Image Original", vr: VR::US },
            E { tag: 0x0019108A, alias: "Calibration Frame Original", vr: VR::US },
            E { tag: 0x0019108B, alias: "Calibration Number Of Points Uif", vr: VR::US },
            E { tag: 0x0019108C, alias: "Calibration Points Row", vr: VR::US },
            E { tag: 0x0019108D, alias: "Calibration Points Column", vr: VR::US },
            E { tag: 0x0019108E, alias: "Calibration Magnification Ratio", vr: VR::FL },
            E { tag: 0x0019108F, alias: "Calibration Sw Version", vr: VR::LO },
            E { tag: 0x00191090, alias: "Extend Calibration Sw Version", vr: VR::LO },
            E { tag: 0x00191091, alias: "Calibration Return Code", vr: VR::IS },
            E { tag: 0x00191092, alias: "Detector Rotation Angle", vr: VR::DS },
            E { tag: 0x00191093, alias: "Spatial Change", vr: VR::CS },
            E { tag: 0x00191094, alias: "Inconsistent Flag", vr: VR::CS },
            E { tag: 0x00191095, alias: "Image Sweep", vr: VR::CS },
            E { tag: 0x00191096, alias: "Internal Label Image", vr: VR::CS },
            E { tag: 0x00191097, alias: "Angle 1 Increment", vr: VR::DS },
            E { tag: 0x00191098, alias: "Angle 2 Increment", vr: VR::DS },
            E { tag: 0x00191099, alias: "Angle 3 Increment", vr: VR::DS },
            E { tag: 0x0019109A, alias: "Sensor Feedback", vr: VR::DS },
            E { tag: 0x0019109B, alias: "Grid", vr: VR::CS },
            E { tag: 0x0019109C, alias: "Default Mask Pixel Shift", vr: VR::FL },
            E { tag: 0x0019109D, alias: "Applicable Review Mode", vr: VR::CS },
            E { tag: 0x0019109E, alias: "Log Lut Control Points", vr: VR::DS },
            E { tag: 0x0019109F, alias: "Exp Lut Control Points", vr: VR::DS },
            E { tag: 0x001910A0, alias: "ABD Value", vr: VR::DS },
            E { tag: 0x001910A1, alias: "Sub Window Center", vr: VR::DS },
            E { tag: 0x001910A2, alias: "Sub Window Width", vr: VR::DS },
            E { tag: 0x001910A3, alias: "Image Rotation", vr: VR::DS },
            E { tag: 0x001910A4, alias: "Auto Injection Enabled", vr: VR::CS },
            E { tag: 0x001910A5, alias: "Injection Phase", vr: VR::CS },
            E { tag: 0x001910A6, alias: "Injection Delay", vr: VR::DS },
            E { tag: 0x001910A7, alias: "Reference Injection Frame Number", vr: VR::IS },
            E { tag: 0x001910A8, alias: "Injection Duration", vr: VR::DS },
            E { tag: 0x001910A9, alias: "EPT", vr: VR::DS },
            E { tag: 0x001910AA, alias: "Can Downscan 512", vr: VR::CS },
            E { tag: 0x001910AB, alias: "Current Spatial Filter Strength", vr: VR::IS },
            E { tag: 0x001910AC, alias: "Brightness Sensitivity", vr: VR::DS },
            E { tag: 0x001910AD, alias: "Exp Lut NOSUB Control Points", vr: VR::DS },
            E { tag: 0x001910AE, alias: "SUB operator LUTs names", vr: VR::LO },
            E { tag: 0x001910AF, alias: "kVp actual vector", vr: VR::DS },
            E { tag: 0x001910B0, alias: "mAs actual vector", vr: VR::DS },
            E { tag: 0x001910B1, alias: "Acquisition Mode Description", vr: VR::LO },
            E { tag: 0x001910B2, alias: "Acquisition Mode Display Label", vr: VR::LO },
            E { tag: 0x001910B3, alias: "Acquisition Protocol User Name", vr: VR::LO },
            E { tag: 0x001910B8, alias: "Recommended display frame rate float", vr: VR::FL },
            E { tag: 0x001910B9, alias: "ABD Vector", vr: VR::FL },
            E { tag: 0x001910BA, alias: "Acquisition region", vr: VR::CS },
            E { tag: 0x001910BB, alias: "Acquisition SUB mode", vr: VR::CS },
            E { tag: 0x001910BC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001910BD, alias: "Table rotation status vector", vr: VR::CS },
            E { tag: 0x001910BE, alias: "Source to image distance per frame vector", vr: VR::FL },
            E { tag: 0x001910C2, alias: "pw actual vector", vr: VR::DS },
            E { tag: 0x001910C4, alias: "Spectral filter thickness", vr: VR::IS },
            E { tag: 0x001910C5, alias: "Preselected pivot rotation speed", vr: VR::FL },
            E { tag: 0x001910C7, alias: "Patient position per image", vr: VR::CS },
            E { tag: 0x001910C8, alias: "3D structure of interest", vr: VR::CS },
            E { tag: 0x001910C9, alias: "3D calibration out of date flag", vr: VR::CS },
            E { tag: 0x001910CA, alias: "3Dspin expected number of frames", vr: VR::IS },
            E { tag: 0x001910D4, alias: "Detection gain value", vr: VR::FL },
            E { tag: 0x001910D5, alias: "mR mAs calibration value", vr: VR::FL },
            E { tag: 0x001910DC, alias: "DRM LUT file name", vr: VR::LO },
            E { tag: 0x001910DD, alias: "DRM Strength", vr: VR::DS },
            E { tag: 0x001910DE, alias: "Acquisition Plane", vr: VR::CS },
            E { tag: 0x001910DF, alias: "LP off longitudinal position Z", vr: VR::FL },
            E { tag: 0x001910E0, alias: "DAP of current record", vr: VR::FL },
            E { tag: 0x001910E1, alias: "Pivot Lateral Angle", vr: VR::FL },
            E { tag: 0x001910E2, alias: "Carm Lateral Angle", vr: VR::FL },
            E { tag: 0x001910E3, alias: "Pivot Lateral Angle increment", vr: VR::FL },
            E { tag: 0x001910E4, alias: "Carm Lateral Angle increment", vr: VR::FL },
            E { tag: 0x001910E7, alias: "LP off long pos Z first frame", vr: VR::FL },
            E { tag: 0x001910E8, alias: "LP off long pos Z increment", vr: VR::FL },
            E { tag: 0x001910E9, alias: "Source to detector distance per frame vector", vr: VR::FL },
            E { tag: 0x001910EA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001910EB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001910EC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001910ED, alias: "Unknown", vr: VR::FL },
        ],
    },
    P {
        creator: "GEMS_DL_PATNT_01",
        entries: &[
            E { tag: 0x00111080, alias: "Patient Instance Uid", vr: VR::UI },
            E { tag: 0x00111081, alias: "Last Study Number", vr: VR::IS },
            E { tag: 0x00111082, alias: "Patient Repaired", vr: VR::CS },
            E { tag: 0x00111083, alias: "Lock Demographics", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_DL_SERIES",
        entries: &[
            E { tag: 0x00151087, alias: "Number of images", vr: VR::IS },
            E { tag: 0x0019100B, alias: "fov dimension double", vr: VR::DS },
            E { tag: 0x00191031, alias: "default spatial filter family", vr: VR::IS },
            E { tag: 0x00191032, alias: "default spatial filter strength", vr: VR::IS },
            E { tag: 0x0019104B, alias: "Detector origin", vr: VR::IS },
            E { tag: 0x0019104E, alias: "def_bright_contr", vr: VR::DS },
            E { tag: 0x0019104F, alias: "user_bright_contr", vr: VR::DS },
            E { tag: 0x00191081, alias: "Calibration frame", vr: VR::US },
            E { tag: 0x00191082, alias: "Calibration object", vr: VR::CS },
            E { tag: 0x00191083, alias: "Calibration object size mm", vr: VR::DS },
            E { tag: 0x00191084, alias: "Calibration factor", vr: VR::FL },
            E { tag: 0x00191085, alias: "Calibration date", vr: VR::DA },
            E { tag: 0x00191086, alias: "Calibration time", vr: VR::TM },
            E { tag: 0x00191087, alias: "Calibration accuracy", vr: VR::US },
            E { tag: 0x00191088, alias: "Calibration extended", vr: VR::CS },
            E { tag: 0x00191089, alias: "Calibration image original", vr: VR::US },
            E { tag: 0x0019108A, alias: "Calibration frame original", vr: VR::US },
            E { tag: 0x0019108B, alias: "Calibration nb points uif", vr: VR::US },
            E { tag: 0x0019108C, alias: "Calibration points row", vr: VR::US },
            E { tag: 0x0019108D, alias: "Calibration points column", vr: VR::US },
            E { tag: 0x0019108E, alias: "Calibration mag ratio", vr: VR::FL },
            E { tag: 0x0019108F, alias: "Calibration sw version", vr: VR::LO },
            E { tag: 0x00191090, alias: "Extend calib sw version", vr: VR::LO },
            E { tag: 0x00191091, alias: "Calibration return code", vr: VR::IS },
            E { tag: 0x00191092, alias: "detector_rot_angle", vr: VR::DS },
            E { tag: 0x00191093, alias: "Spatial change", vr: VR::CS },
            E { tag: 0x00191095, alias: "Image sweep", vr: VR::CS },
            E { tag: 0x00191097, alias: "Angle 1 increment", vr: VR::DS },
            E { tag: 0x00191098, alias: "Angle 2 increment", vr: VR::DS },
            E { tag: 0x00191099, alias: "Angle 3 increment", vr: VR::DS },
            E { tag: 0x0019109A, alias: "sensor feedback", vr: VR::DS },
            E { tag: 0x0019109D, alias: "applicable review mode", vr: VR::CS },
            E { tag: 0x0019109E, alias: "log LUT control points", vr: VR::DS },
            E { tag: 0x0019109F, alias: "exp LUT SUB control points", vr: VR::DS },
            E { tag: 0x001910A0, alias: "ABD value", vr: VR::DS },
            E { tag: 0x001910A1, alias: "Sub window center", vr: VR::DS },
            E { tag: 0x001910A2, alias: "Sub window width", vr: VR::DS },
            E { tag: 0x001910A4, alias: "Auto injection enabled", vr: VR::CS },
            E { tag: 0x001910A5, alias: "injection phase", vr: VR::CS },
            E { tag: 0x001910A6, alias: "injection delay", vr: VR::DS },
            E { tag: 0x001910A7, alias: "reference injection frame number", vr: VR::IS },
            E { tag: 0x001910A9, alias: "EPT", vr: VR::DS },
            E { tag: 0x001910AA, alias: "can downscan 512", vr: VR::CS },
            E { tag: 0x001910AB, alias: "current spatial filter strength", vr: VR::IS },
            E { tag: 0x001910AD, alias: "exp LUT NOSUB control points", vr: VR::DS },
            E { tag: 0x001910AE, alias: "SUB operator LUTs names", vr: VR::LO },
            E { tag: 0x001910C4, alias: "Default spatial filter family", vr: VR::IS },
            E { tag: 0x001910C5, alias: "Preselected pivot rotation speed", vr: VR::FL },
            E { tag: 0x001910C7, alias: "Patient position per image", vr: VR::CS },
            E { tag: 0x001910C8, alias: "3D structure of interest", vr: VR::CS },
            E { tag: 0x001910C9, alias: "3D calibration out of date flag", vr: VR::CS },
            E { tag: 0x001910CA, alias: "3Dspin expected number of frames", vr: VR::IS },
            E { tag: 0x001910D4, alias: "Detection gain value", vr: VR::FL },
            E { tag: 0x001910D5, alias: "mR mAs calibration value", vr: VR::FL },
            E { tag: 0x001910DC, alias: "DRM LUT file name", vr: VR::LO },
            E { tag: 0x001910DD, alias: "DRM Strength", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_DL_SERIES_01",
        entries: &[
            E { tag: 0x00151085, alias: "Series File Name", vr: VR::LO },
            E { tag: 0x00151087, alias: "Number Of Images", vr: VR::IS },
            E { tag: 0x0015108C, alias: "Sent Flag", vr: VR::CS },
            E { tag: 0x0015108D, alias: "Item Locked", vr: VR::US },
            E { tag: 0x0019104C, alias: "internal label", vr: VR::CS },
            E { tag: 0x0019104D, alias: "browser hide", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_DL_STUDY_01",
        entries: &[
            E { tag: 0x00151080, alias: "Study Dose", vr: VR::DS },
            E { tag: 0x00151081, alias: "Study Total Dap", vr: VR::DS },
            E { tag: 0x00151082, alias: "Study Fluoro Dap", vr: VR::DS },
            E { tag: 0x00151083, alias: "Study Fluoro Time", vr: VR::IS },
            E { tag: 0x00151084, alias: "Study Record Dap", vr: VR::DS },
            E { tag: 0x00151085, alias: "Study Record Time", vr: VR::IS },
            E { tag: 0x00151086, alias: "Last Xa Number", vr: VR::IS },
            E { tag: 0x00151088, alias: "Default Operator Name", vr: VR::PN },
            E { tag: 0x00151089, alias: "Default Performing Physician Name", vr: VR::PN },
            E { tag: 0x0015108A, alias: "Default Patient Orientation", vr: VR::CS },
            E { tag: 0x0015108B, alias: "Last Sc Number", vr: VR::IS },
            E { tag: 0x0015108E, alias: "Common Series Instance Uid", vr: VR::UI },
            E { tag: 0x0015108F, alias: "Study Number", vr: VR::IS },
            E { tag: 0x00151092, alias: "Study dose Frontal", vr: VR::FL },
            E { tag: 0x00151093, alias: "Study total dap Frontal", vr: VR::FL },
            E { tag: 0x00151094, alias: "Study fluoro dap frontal", vr: VR::FL },
            E { tag: 0x00151095, alias: "Study fluoro time frontal", vr: VR::IS },
            E { tag: 0x00151096, alias: "Study record dap frontal", vr: VR::FL },
            E { tag: 0x00151097, alias: "Study record time frontal", vr: VR::IS },
            E { tag: 0x00151098, alias: "Study dose lateral", vr: VR::FL },
            E { tag: 0x00151099, alias: "Study total dap lateral", vr: VR::FL },
            E { tag: 0x0015109A, alias: "Study fluoro dap lateral", vr: VR::FL },
            E { tag: 0x0015109B, alias: "Study fluoro time lateral", vr: VR::IS },
            E { tag: 0x0015109C, alias: "Study record dap leteral", vr: VR::FL },
            E { tag: 0x0015109D, alias: "Study record time lateral", vr: VR::IS },
        ],
    },
    P {
        creator: "GEMS_DRS_1",
        entries: &[
            E { tag: 0x00371010, alias: "Referring Department", vr: VR::LO },
            E { tag: 0x00371020, alias: "Screen Number", vr: VR::US },
            E { tag: 0x00371040, alias: "Left Orientation", vr: VR::SH },
            E { tag: 0x00371042, alias: "Right Orientation", vr: VR::SH },
            E { tag: 0x00371050, alias: "Inversion", vr: VR::CS },
            E { tag: 0x00371060, alias: "DSA", vr: VR::US },
        ],
    },
    P {
        creator: "GEMS_FALCON_03",
        entries: &[
            E { tag: 0x00451055, alias: "A_Coefficients", vr: VR::DS },
            E { tag: 0x00451062, alias: "User Window Center", vr: VR::IS },
            E { tag: 0x00451063, alias: "User Window Width", vr: VR::IS },
            E { tag: 0x00451065, alias: "Requested Detector Entrance Dose", vr: VR::IS },
            E { tag: 0x00451067, alias: "VOI LUT Asymmetry parameter beta", vr: VR::DS },
            E { tag: 0x00451069, alias: "Collimator rotation", vr: VR::IS },
            E { tag: 0x00451072, alias: "Collimator Width", vr: VR::DS },
            E { tag: 0x00451073, alias: "Collimator Height", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_FUNCTOOL_01",
        entries: &[
            E { tag: 0x00511001, alias: "Functional Proc Group Name", vr: VR::LO },
            E { tag: 0x00511002, alias: "Functional Processing Name", vr: VR::LO },
            E { tag: 0x00511003, alias: "Bias of Functional Image", vr: VR::SL },
            E { tag: 0x00511004, alias: "Scale of Functional Image", vr: VR::FL },
            E { tag: 0x00511005, alias: "Length of Parameters String", vr: VR::SL },
            E { tag: 0x00511006, alias: "Store Parameters string, delimited by character ESC=0x1B (27)", vr: VR::LT },
            E { tag: 0x00511007, alias: "Functional Image Version", vr: VR::LO },
            E { tag: 0x00511008, alias: "Store Color Ramp", vr: VR::SL },
            E { tag: 0x00511009, alias: "Store Width of Functional Image", vr: VR::SL },
            E { tag: 0x0051100A, alias: "Store level of Functional Image", vr: VR::SL },
            E { tag: 0x0051100B, alias: "Store B-Value with Functional Image", vr: VR::FL },
            E { tag: 0x0051100C, alias: "Analysis Package", vr: VR::LO },
            E { tag: 0x0051100E, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "GEMS_GDXE_ATHENAV2_INTERNAL_USE",
        entries: &[
            E { tag: 0x7FDF1010, alias: "PPS Stream", vr: VR::LT },
            E { tag: 0x7FDF1011, alias: "Pixel Data References (temporary)", vr: VR::LT },
            E { tag: 0x7FDF1020, alias: "Auto Push Tag", vr: VR::SS },
            E { tag: 0x7FDF1025, alias: "PPS Status", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_GDXE_FALCON_04",
        entries: &[
            E { tag: 0x00111003, alias: "Processed Series UID", vr: VR::UI },
            E { tag: 0x00111004, alias: "Acquisition Type", vr: VR::CS },
            E { tag: 0x00111005, alias: "Acquisition UID", vr: VR::UI },
            E { tag: 0x00111006, alias: "Image Dose", vr: VR::DS },
            E { tag: 0x00111007, alias: "Study Dose", vr: VR::FL },
            E { tag: 0x00111008, alias: "Study DAP", vr: VR::FL },
            E { tag: 0x00111009, alias: "Non-Digital Exposures", vr: VR::SL },
            E { tag: 0x00111010, alias: "Total Exposures", vr: VR::SL },
            E { tag: 0x00111011, alias: "ROI", vr: VR::LT },
            E { tag: 0x00111012, alias: "Patient Size String", vr: VR::LT },
            E { tag: 0x00111013, alias: "SPS UID", vr: VR::UI },
            E { tag: 0x00111014, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00111015, alias: "Detector ARC Gain", vr: VR::DS },
            E { tag: 0x00111016, alias: "Processing Debug Info", vr: VR::LT },
            E { tag: 0x00111017, alias: "Override mode", vr: VR::CS },
            E { tag: 0x00111019, alias: "Film Speed Selection", vr: VR::DS },
            E { tag: 0x00111027, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111028, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111029, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111030, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111031, alias: "Detected Field of View", vr: VR::IS },
            E { tag: 0x00111032, alias: "Adjusted Field of View", vr: VR::IS },
            E { tag: 0x00111033, alias: "Detected Exposure Index", vr: VR::DS },
            E { tag: 0x00111034, alias: "Compensated Detector Exposure", vr: VR::DS },
            E { tag: 0x00111035, alias: "Uncompensated Detector Exposure", vr: VR::DS },
            E { tag: 0x00111036, alias: "Median Anatomy Count Value", vr: VR::DS },
            E { tag: 0x00111037, alias: "DEI lower and upper limit values", vr: VR::DS },
            E { tag: 0x00111038, alias: "Shift Vector for Pasting", vr: VR::SL },
            E { tag: 0x00111039, alias: "Image Number in Pasting", vr: VR::CS },
            E { tag: 0x00111040, alias: "Pasting Overlap", vr: VR::SL },
            E { tag: 0x00111041, alias: "Sub-image Collimator Vertices", vr: VR::IS },
            E { tag: 0x00111042, alias: "View IP", vr: VR::LO },
            E { tag: 0x00111043, alias: "Key-stone Coordinates", vr: VR::IS },
            E { tag: 0x00111044, alias: "Receptor Type", vr: VR::CS },
            E { tag: 0x00111046, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00111047, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_GENIE_1",
        entries: &[
            E { tag: 0x00091001, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00091010, alias: "Study Name", vr: VR::LO },
            E { tag: 0x00091011, alias: "Study Flags", vr: VR::SL },
            E { tag: 0x00091012, alias: "Study Type", vr: VR::SL },
            E { tag: 0x0009101E, alias: "Dataset UID", vr: VR::UI },
            E { tag: 0x00091020, alias: "Series Object Name", vr: VR::LO },
            E { tag: 0x00091021, alias: "Series Flags", vr: VR::SL },
            E { tag: 0x00091022, alias: "User Orientation", vr: VR::SH },
            E { tag: 0x00091023, alias: "Initiation Type", vr: VR::SL },
            E { tag: 0x00091024, alias: "Initiation Delay", vr: VR::SL },
            E { tag: 0x00091025, alias: "Initiation Count Rate", vr: VR::SL },
            E { tag: 0x00091026, alias: "Number Energy Sets", vr: VR::SL },
            E { tag: 0x00091027, alias: "Number Detectors", vr: VR::SL },
            E { tag: 0x00091028, alias: "Number RR Windows", vr: VR::SL },
            E { tag: 0x00091029, alias: "Number MG Time Slots", vr: VR::SL },
            E { tag: 0x0009102A, alias: "Number View Sets", vr: VR::SL },
            E { tag: 0x0009102B, alias: "Trigger History UID", vr: VR::LO },
            E { tag: 0x0009102C, alias: "Series Comments", vr: VR::LO },
            E { tag: 0x0009102D, alias: "Track Beat Average", vr: VR::SL },
            E { tag: 0x0009102E, alias: "Distance Prescribed", vr: VR::FD },
            E { tag: 0x0009102F, alias: "Table Direction", vr: VR::SL },
            E { tag: 0x00091030, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091033, alias: "Rotational Continuous Speed", vr: VR::FD },
            E { tag: 0x00091034, alias: "Gantry Motion Type (retired)", vr: VR::SL },
            E { tag: 0x00091035, alias: "Gantry Locus Type", vr: VR::SL },
            E { tag: 0x00091037, alias: "Starting Heart Rate", vr: VR::SL },
            E { tag: 0x00091038, alias: "RR Window Width", vr: VR::SL },
            E { tag: 0x00091039, alias: "RR Window Offset", vr: VR::SL },
            E { tag: 0x0009103A, alias: "Percent Cycle Imaged", vr: VR::SL },
            E { tag: 0x00091040, alias: "Patient Object Name", vr: VR::PN },
            E { tag: 0x00091041, alias: "Patient Flags", vr: VR::SL },
            E { tag: 0x00091042, alias: "Patient Creation Date", vr: VR::DA },
            E { tag: 0x00091043, alias: "Patient Creation Time", vr: VR::TM },
            E { tag: 0x00091044, alias: "Num Views Acquired (retired)", vr: VR::SL },
            E { tag: 0x00091045, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0011100A, alias: "Series Type", vr: VR::SL },
            E { tag: 0x0011100B, alias: "Effective Series Duration", vr: VR::SL },
            E { tag: 0x0011100C, alias: "Num Beats", vr: VR::SL },
            E { tag: 0x0011100D, alias: "Radio Nuclide Name", vr: VR::LO },
            E { tag: 0x00111010, alias: "Dataset Object Name", vr: VR::LO },
            E { tag: 0x00111011, alias: "Dataset Modified", vr: VR::IS },
            E { tag: 0x00111012, alias: "Dataset Name", vr: VR::LO },
            E { tag: 0x00111013, alias: "Dataset Type", vr: VR::SL },
            E { tag: 0x00111014, alias: "Completion Time", vr: VR::SH },
            E { tag: 0x00111015, alias: "Detector Number", vr: VR::SL },
            E { tag: 0x00111016, alias: "Energy Number", vr: VR::SL },
            E { tag: 0x00111017, alias: "RR Interval Window Number", vr: VR::SL },
            E { tag: 0x00111018, alias: "MG Bin Number", vr: VR::SL },
            E { tag: 0x00111019, alias: "Radius Of Rotation", vr: VR::FD },
            E { tag: 0x0011101A, alias: "Detector Count Zone", vr: VR::SL },
            E { tag: 0x0011101B, alias: "Num Energy Windows", vr: VR::SL },
            E { tag: 0x0011101C, alias: "Energy Offset", vr: VR::SL },
            E { tag: 0x0011101D, alias: "Energy Range", vr: VR::SL },
            E { tag: 0x0011101E, alias: "Energy Width (retired)", vr: VR::SL },
            E { tag: 0x0011101F, alias: "Image Orientation", vr: VR::SL },
            E { tag: 0x00111021, alias: "Acq Zoom (retired)", vr: VR::DS },
            E { tag: 0x00111022, alias: "Acq Pan (retired)", vr: VR::DS },
            E { tag: 0x00111023, alias: "Use FOV Mask", vr: VR::SL },
            E { tag: 0x00111024, alias: "FOV Mask Y Cutoff Distance", vr: VR::SL },
            E { tag: 0x00111025, alias: "FOV Mask Cutoff Angle", vr: VR::SL },
            E { tag: 0x00111026, alias: "Table Orientation", vr: VR::SL },
            E { tag: 0x00111027, alias: "ROI Top Left", vr: VR::SL },
            E { tag: 0x00111028, alias: "ROI Bottom Right", vr: VR::SL },
            E { tag: 0x00111029, alias: "Uniformity Mean", vr: VR::SL },
            E { tag: 0x0011102A, alias: "Phase Duration (retired)", vr: VR::FD },
            E { tag: 0x0011102C, alias: "View X Adjustment", vr: VR::FD },
            E { tag: 0x0011102D, alias: "View Y Adjustment", vr: VR::FD },
            E { tag: 0x0011102E, alias: "Pixel Overflow Flag", vr: VR::SL },
            E { tag: 0x0011102F, alias: "Overflow Level", vr: VR::SL },
            E { tag: 0x00111030, alias: "Picture Object Name", vr: VR::LO },
            E { tag: 0x00111031, alias: "Acquisition Parent UID", vr: VR::LO },
            E { tag: 0x00111032, alias: "Processing Parent UID", vr: VR::LO },
            E { tag: 0x00111033, alias: "Energy Correct Name", vr: VR::LO },
            E { tag: 0x00111034, alias: "Spatial Correct Name", vr: VR::LO },
            E { tag: 0x00111035, alias: "Tuning Calib Name", vr: VR::LO },
            E { tag: 0x00111036, alias: "Uniformity Correct Name", vr: VR::LO },
            E { tag: 0x00111037, alias: "Acquisition Specific Correction Name", vr: VR::LT },
            E { tag: 0x00111038, alias: "Byte Order", vr: VR::SL },
            E { tag: 0x00111039, alias: "Compression Type", vr: VR::SL },
            E { tag: 0x0011103A, alias: "Picture Format", vr: VR::SL },
            E { tag: 0x0011103B, alias: "Pixel Scale", vr: VR::FD },
            E { tag: 0x0011103C, alias: "Pixel Offset", vr: VR::FD },
            E { tag: 0x0011103D, alias: "Energy Peak (retired)", vr: VR::SL },
            E { tag: 0x0011103E, alias: "FOV Shape", vr: VR::SL },
            E { tag: 0x0011103F, alias: "Dataset Flags", vr: VR::SL },
            E { tag: 0x00111040, alias: "Viewing Object Name", vr: VR::LO },
            E { tag: 0x00111041, alias: "Orientation Angle", vr: VR::SL },
            E { tag: 0x00111042, alias: "Rotation Angle", vr: VR::FD },
            E { tag: 0x00111043, alias: "Window Inverse Flag", vr: VR::SL },
            E { tag: 0x00111044, alias: "Threshold Center", vr: VR::FD },
            E { tag: 0x00111045, alias: "Threshold Width", vr: VR::FD },
            E { tag: 0x00111046, alias: "Interpolation Type", vr: VR::SL },
            E { tag: 0x00111050, alias: "Where Object Name", vr: VR::LO },
            E { tag: 0x00111055, alias: "Period", vr: VR::FD },
            E { tag: 0x00111056, alias: "Elapsed Time", vr: VR::FD },
            E { tag: 0x00111057, alias: "FOV", vr: VR::FD },
            E { tag: 0x00111061, alias: "Image Size", vr: VR::SL },
            E { tag: 0x00111062, alias: "Linear FOV", vr: VR::FD },
            E { tag: 0x00111063, alias: "Spatial Offset", vr: VR::FD },
            E { tag: 0x00111064, alias: "Spatial Orientation", vr: VR::FD },
            E { tag: 0x00111065, alias: "Reference Dataset UID", vr: VR::LO },
            E { tag: 0x00111066, alias: "Starcam Reference Dataset", vr: VR::SH },
            E { tag: 0x00111067, alias: "Reference Frame Number", vr: VR::SL },
            E { tag: 0x00111068, alias: "Cursor Length", vr: VR::SL },
            E { tag: 0x00111069, alias: "Number of Cursors", vr: VR::SL },
            E { tag: 0x0011106A, alias: "Cursor Coordinates", vr: VR::SL },
            E { tag: 0x0011106B, alias: "Recon Options Flag", vr: VR::SL },
            E { tag: 0x0011106C, alias: "Motion Threshold", vr: VR::FD },
            E { tag: 0x0011106D, alias: "Motion Curve UID", vr: VR::UI },
            E { tag: 0x0011106E, alias: "Recon Type", vr: VR::SL },
            E { tag: 0x0011106F, alias: "Pre Filter Type", vr: VR::SL },
            E { tag: 0x00111071, alias: "Back Proj Filter Type", vr: VR::SL },
            E { tag: 0x00111072, alias: "Recon Arc", vr: VR::SL },
            E { tag: 0x00111073, alias: "Recon Pan AP Offset", vr: VR::FD },
            E { tag: 0x00111074, alias: "Recon Pan LR Offset", vr: VR::FD },
            E { tag: 0x00111075, alias: "Recon Area", vr: VR::FD },
            E { tag: 0x00111076, alias: "Start View", vr: VR::SL },
            E { tag: 0x00111077, alias: "Attenuation Type", vr: VR::SL },
            E { tag: 0x00111078, alias: "Dual Energy Processing", vr: VR::SL },
            E { tag: 0x00111079, alias: "Pre Filter Param", vr: VR::SH },
            E { tag: 0x0011107A, alias: "Pre Filter Param 2", vr: VR::SH },
            E { tag: 0x0011107B, alias: "Back Proj Filter Param", vr: VR::SH },
            E { tag: 0x0011107C, alias: "Back Proj Filter Param 2", vr: VR::SH },
            E { tag: 0x0011107D, alias: "Attenuation Coef", vr: VR::SH },
            E { tag: 0x0011107E, alias: "Ref Slice Width", vr: VR::SL },
            E { tag: 0x0011107F, alias: "Ref Trans Pixel Volume", vr: VR::FD },
            E { tag: 0x00111081, alias: "Attenuation Threshold", vr: VR::SH },
            E { tag: 0x00111082, alias: "Interpolation Distance", vr: VR::FD },
            E { tag: 0x00111083, alias: "Interpolation Center X", vr: VR::FD },
            E { tag: 0x00111084, alias: "Interpolation Center Y", vr: VR::FD },
            E { tag: 0x00111085, alias: "Quant Filter Flag", vr: VR::SL },
            E { tag: 0x00111086, alias: "Head Conversion", vr: VR::SL },
            E { tag: 0x00111087, alias: "Slice Width Pixels", vr: VR::SL },
            E { tag: 0x00111088, alias: "Rfmtr Trans Ref", vr: VR::SL },
            E { tag: 0x00111089, alias: "Rfmtr Trans Ref mm", vr: VR::FD },
            E { tag: 0x0011108A, alias: "Two Line Trans Ref", vr: VR::SL },
            E { tag: 0x0011108B, alias: "Three-D Zero", vr: VR::SL },
            E { tag: 0x0011108C, alias: "Three-D Zero Length", vr: VR::SL },
            E { tag: 0x0011108D, alias: "Three-D Zero In", vr: VR::SL },
            E { tag: 0x00131010, alias: "Digital FOV", vr: VR::FD },
            E { tag: 0x00131011, alias: "Source Translator", vr: VR::SL },
            E { tag: 0x00131012, alias: "RAL Flags", vr: VR::UL },
            E { tag: 0x00131013, alias: "Xeleris Frame Sequence", vr: VR::SQ },
            E { tag: 0x00131014, alias: "Original Image Num", vr: VR::SL },
            E { tag: 0x00131015, alias: "Fscalar", vr: VR::FD },
            E { tag: 0x00131016, alias: "AutoTrack Peak", vr: VR::SL },
            E { tag: 0x00131017, alias: "AutoTrack Width", vr: VR::SL },
            E { tag: 0x00131018, alias: "Transmission Scan Time", vr: VR::FD },
            E { tag: 0x00131019, alias: "Transmission Mask Width", vr: VR::FD },
            E { tag: 0x0013101A, alias: "Copper Attenuator Thickness", vr: VR::FD },
            E { tag: 0x0013101B, alias: "Det Ang Separation", vr: VR::FD },
            E { tag: 0x0013101C, alias: "Axial Acceptance Angle", vr: VR::SL },
            E { tag: 0x0013101D, alias: "Theta Acceptance Value", vr: VR::SL },
            E { tag: 0x0013101E, alias: "Tomo View Offset", vr: VR::FD },
            E { tag: 0x00131020, alias: "Accepted Beats Time", vr: VR::FD },
            E { tag: 0x00131021, alias: "Threshold", vr: VR::FD },
            E { tag: 0x00131022, alias: "Linear Depth", vr: VR::FD },
            E { tag: 0x00131023, alias: "Unif Date Time", vr: VR::LO },
            E { tag: 0x00131024, alias: "Series Accepted Beats", vr: VR::SL },
            E { tag: 0x00131025, alias: "Series Rejected Beats", vr: VR::SL },
            E { tag: 0x00131026, alias: "Study Comments", vr: VR::LT },
            E { tag: 0x00151010, alias: "Frame Termination Condition", vr: VR::SL },
            E { tag: 0x00151011, alias: "Frame Termination Value", vr: VR::SL },
            E { tag: 0x00151012, alias: "Num ECT Phases", vr: VR::SL },
            E { tag: 0x00151013, alias: "Num WB Scans", vr: VR::SL },
            E { tag: 0x00151014, alias: "ECT Phase Num", vr: VR::SL },
            E { tag: 0x00151015, alias: "WB Scan Num", vr: VR::SL },
            E { tag: 0x00151016, alias: "Comb Head Number", vr: VR::SL },
            E { tag: 0x00151017, alias: "Preceding Beat", vr: VR::UL },
            E { tag: 0x0019105F, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00331007, alias: "Orig SOP Instance UID", vr: VR::SQ },
            E { tag: 0x00331008, alias: "Xeleris Data Object Type", vr: VR::CS },
            E { tag: 0x00331010, alias: "Modified", vr: VR::SL },
            E { tag: 0x00331011, alias: "Name", vr: VR::LO },
            E { tag: 0x00331016, alias: "ProtocolDataUID", vr: VR::LO },
            E { tag: 0x00331017, alias: "Date", vr: VR::SH },
            E { tag: 0x00331018, alias: "Time", vr: VR::SH },
            E { tag: 0x00331019, alias: "ProtocoldataFlags", vr: VR::UL },
            E { tag: 0x0033101A, alias: "ProtocolName", vr: VR::UL },
            E { tag: 0x0033101B, alias: "RelevantDataUID", vr: VR::LO },
            E { tag: 0x0033101C, alias: "BulkData", vr: VR::LO },
            E { tag: 0x0033101D, alias: "IntData", vr: VR::SL },
            E { tag: 0x0033101E, alias: "DoubleData", vr: VR::FD },
            E { tag: 0x0033101F, alias: "StringData", vr: VR::OB },
            E { tag: 0x00331020, alias: "BulkDataFormat", vr: VR::LT },
            E { tag: 0x00331021, alias: "IntDataFormat", vr: VR::LT },
            E { tag: 0x00331022, alias: "DoubleDataFormat", vr: VR::LT },
            E { tag: 0x00331023, alias: "StringDataFormat", vr: VR::LT },
            E { tag: 0x00331024, alias: "Description", vr: VR::LT },
            E { tag: 0x00331030, alias: "Allocate trigger buffer", vr: VR::UL },
            E { tag: 0x00331033, alias: "Number of triggers", vr: VR::UL },
            E { tag: 0x00331034, alias: "Trigger size", vr: VR::UL },
            E { tag: 0x00331035, alias: "Trigger data size", vr: VR::UL },
            E { tag: 0x00331036, alias: "Trigger data", vr: VR::OB },
            E { tag: 0x00351001, alias: "Start Angle", vr: VR::FD },
            E { tag: 0x00551012, alias: "Xeleris Energy Window Information Sequence", vr: VR::SQ },
            E { tag: 0x00551013, alias: "Xeleris Energy Window Range Sequence", vr: VR::SQ },
            E { tag: 0x00551022, alias: "Xeleris Detector Information Sequence", vr: VR::SQ },
            E { tag: 0x00551052, alias: "Xeleris Rotation Information Sequence", vr: VR::SQ },
            E { tag: 0x00551062, alias: "Xeleris Gated Information Sequence", vr: VR::SQ },
            E { tag: 0x00551063, alias: "Xeleris Data Information Sequence", vr: VR::SQ },
            E { tag: 0x00551064, alias: "SDODoubleData SQ", vr: VR::SQ },
            E { tag: 0x00551065, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x50011001, alias: "Modified", vr: VR::SL },
            E { tag: 0x50011002, alias: "Name", vr: VR::LO },
            E { tag: 0x50011003, alias: "Cid", vr: VR::SL },
            E { tag: 0x50011004, alias: "Srid", vr: VR::SL },
            E { tag: 0x50011005, alias: "SOPClassUID", vr: VR::LO },
            E { tag: 0x50011006, alias: "SOPInstanceUID", vr: VR::LO },
            E { tag: 0x50011007, alias: "CurveType", vr: VR::SL },
            E { tag: 0x50011008, alias: "GraphType", vr: VR::SL },
            E { tag: 0x50011009, alias: "Legend", vr: VR::LO },
            E { tag: 0x5001100A, alias: "XUnits", vr: VR::LO },
            E { tag: 0x5001100B, alias: "YUnits", vr: VR::LO },
            E { tag: 0x5001100C, alias: "Edit", vr: VR::SL },
            E { tag: 0x5001100D, alias: "Suspend", vr: VR::SL },
            E { tag: 0x5001100E, alias: "StyleLine", vr: VR::SL },
            E { tag: 0x5001100F, alias: "StyleFill", vr: VR::SL },
            E { tag: 0x50011010, alias: "StyleColour", vr: VR::LO },
            E { tag: 0x50011011, alias: "StyleWidth", vr: VR::SL },
            E { tag: 0x50011012, alias: "StylePoint", vr: VR::SL },
            E { tag: 0x50011013, alias: "StylePColour", vr: VR::LO },
            E { tag: 0x50011014, alias: "StylePSize", vr: VR::SL },
            E { tag: 0x50011015, alias: "Segments", vr: VR::SL },
            E { tag: 0x50011016, alias: "SegType", vr: VR::SL },
            E { tag: 0x50011017, alias: "SegStart", vr: VR::FD },
            E { tag: 0x50011018, alias: "SegEnd", vr: VR::FD },
            E { tag: 0x50011019, alias: "SegStyleLine", vr: VR::SL },
            E { tag: 0x5001101A, alias: "SegStyleFill", vr: VR::SL },
            E { tag: 0x5001101B, alias: "SegStyleColour", vr: VR::LO },
            E { tag: 0x5001101C, alias: "SegStyleWidth", vr: VR::SL },
            E { tag: 0x5001101D, alias: "SegStylePoint", vr: VR::SL },
            E { tag: 0x5001101E, alias: "SegStylePColour", vr: VR::SL },
            E { tag: 0x5001101F, alias: "SegStylePSize", vr: VR::SL },
            E { tag: 0x50011020, alias: "SegName", vr: VR::LO },
            E { tag: 0x50011021, alias: "SegAllowDirInt", vr: VR::SL },
            E { tag: 0x50011022, alias: "TextAnnots", vr: VR::SL },
            E { tag: 0x50011023, alias: "TxtX", vr: VR::FD },
            E { tag: 0x50011024, alias: "TxtY", vr: VR::FD },
            E { tag: 0x50011025, alias: "TxtText", vr: VR::LO },
            E { tag: 0x50011026, alias: "TxtName", vr: VR::LO },
            E { tag: 0x50011030, alias: "ROIName", vr: VR::LO },
            E { tag: 0x50011031, alias: "DerivedFromImageUID", vr: VR::LO },
            E { tag: 0x50011032, alias: "DerivedFromImages", vr: VR::SL },
            E { tag: 0x50011033, alias: "CurveFlags", vr: VR::UL },
            E { tag: 0x50011034, alias: "CurveName", vr: VR::LO },
            E { tag: 0x50011035, alias: "DatasetName", vr: VR::LO },
            E { tag: 0x50011036, alias: "CurveUID", vr: VR::LO },
            E { tag: 0x50011037, alias: "ROIArea", vr: VR::FD },
            E { tag: 0x50011038, alias: "Modified", vr: VR::SL },
            E { tag: 0x50011039, alias: "Name", vr: VR::LO },
            E { tag: 0x5001103A, alias: "Software version", vr: VR::LO },
            E { tag: 0x5001103B, alias: "StartDate", vr: VR::SH },
            E { tag: 0x5001103C, alias: "CompletionDate", vr: VR::SH },
            E { tag: 0x5001103D, alias: "DetectorName", vr: VR::LO },
            E { tag: 0x50011041, alias: "Modified", vr: VR::SL },
            E { tag: 0x50011042, alias: "Name", vr: VR::LO },
            E { tag: 0x50011043, alias: "Name", vr: VR::SL },
            E { tag: 0x50011044, alias: "Name", vr: VR::SL },
            E { tag: 0x50011045, alias: "SOPClassUID", vr: VR::LO },
            E { tag: 0x50011046, alias: "SOPInstanceUID", vr: VR::LO },
            E { tag: 0x50011047, alias: "NormalColor", vr: VR::LO },
            E { tag: 0x50011048, alias: "NameFont", vr: VR::LT },
            E { tag: 0x50011049, alias: "FillPattern", vr: VR::SL },
            E { tag: 0x5001104A, alias: "LineStyle", vr: VR::SL },
            E { tag: 0x5001104B, alias: "LineDashLength", vr: VR::SL },
            E { tag: 0x5001104C, alias: "LineThickness", vr: VR::SL },
            E { tag: 0x5001104D, alias: "Interactivity", vr: VR::SL },
            E { tag: 0x5001104E, alias: "NamePos", vr: VR::SL },
            E { tag: 0x5001104F, alias: "NameDisplay", vr: VR::SL },
            E { tag: 0x50011050, alias: "Label", vr: VR::LO },
            E { tag: 0x50011051, alias: "BpSeg", vr: VR::SL },
            E { tag: 0x50011052, alias: "BpSegPairs", vr: VR::US },
            E { tag: 0x50011053, alias: "SeedSpace", vr: VR::SL },
            E { tag: 0x50011054, alias: "Seeds", vr: VR::FD },
            E { tag: 0x50011055, alias: "Shape", vr: VR::SL },
            E { tag: 0x50011056, alias: "ShapeTilt", vr: VR::FD },
            E { tag: 0x50011059, alias: "ShapePtsSpace", vr: VR::SL },
            E { tag: 0x5001105A, alias: "ShapeCtrlPtsCounts", vr: VR::SL },
            E { tag: 0x5001105B, alias: "ShapeCtrlPts", vr: VR::FD },
            E { tag: 0x5001105C, alias: "ShapeCPSpace", vr: VR::SL },
            E { tag: 0x5001105D, alias: "ROIFlags", vr: VR::SL },
            E { tag: 0x5001105E, alias: "FrameNumber", vr: VR::SL },
            E { tag: 0x5001105F, alias: "Id", vr: VR::SL },
            E { tag: 0x50011060, alias: "DatasetROIMapping", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_GNHD_01",
        entries: &[
            E { tag: 0x00331001, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00331002, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "GEMS_HELIOS_01",
        entries: &[
            E { tag: 0x00451001, alias: "Number of Macro Rows in Detector", vr: VR::SS },
            E { tag: 0x00451002, alias: "Macro width at ISO Center", vr: VR::FL },
            E { tag: 0x00451003, alias: "DAS type", vr: VR::SS },
            E { tag: 0x00451004, alias: "DAS gain", vr: VR::SS },
            E { tag: 0x00451005, alias: "DAS Temperature", vr: VR::SS },
            E { tag: 0x00451006, alias: "Table Direction", vr: VR::CS },
            E { tag: 0x00451007, alias: "Z smoothing Factor", vr: VR::FL },
            E { tag: 0x00451008, alias: "View Weighting Mode", vr: VR::SS },
            E { tag: 0x00451009, alias: "Sigma Row number", vr: VR::SS },
            E { tag: 0x0045100A, alias: "Minimum DAS value", vr: VR::FL },
            E { tag: 0x0045100B, alias: "Maximum Offset Value", vr: VR::FL },
            E { tag: 0x0045100C, alias: "Number of Views shifted", vr: VR::SS },
            E { tag: 0x0045100D, alias: "Z tracking Flag", vr: VR::SS },
            E { tag: 0x0045100E, alias: "Mean Z error", vr: VR::FL },
            E { tag: 0x0045100F, alias: "Z tracking Error", vr: VR::FL },
            E { tag: 0x00451010, alias: "Start View 2A", vr: VR::SS },
            E { tag: 0x00451011, alias: "Number of Views 2A", vr: VR::SS },
            E { tag: 0x00451012, alias: "Start View 1A", vr: VR::SS },
            E { tag: 0x00451013, alias: "Sigma Mode", vr: VR::SS },
            E { tag: 0x00451014, alias: "Number of Views 1A", vr: VR::SS },
            E { tag: 0x00451015, alias: "Start View 2B", vr: VR::SS },
            E { tag: 0x00451016, alias: "Number Views 2B", vr: VR::SS },
            E { tag: 0x00451017, alias: "Start View 1B", vr: VR::SS },
            E { tag: 0x00451018, alias: "Number of Views 1B", vr: VR::SS },
            E { tag: 0x00451019, alias: "Air Calibration Date", vr: VR::SS },
            E { tag: 0x0045101A, alias: "Air Calibration Time", vr: VR::SS },
            E { tag: 0x0045101B, alias: "Phantom Calibration Date", vr: VR::SS },
            E { tag: 0x0045101C, alias: "Phantom Calibration Time", vr: VR::SS },
            E { tag: 0x0045101D, alias: "Z Slope Calibration Date", vr: VR::SS },
            E { tag: 0x0045101E, alias: "Z Slope Calibration Time", vr: VR::SS },
            E { tag: 0x0045101F, alias: "Cross Talk Calibration Date", vr: VR::SS },
            E { tag: 0x00451020, alias: "Cross Talk Calibration Time", vr: VR::SS },
            E { tag: 0x00451021, alias: "Iterbone Flag", vr: VR::SS },
            E { tag: 0x00451022, alias: "Peristaltic Flag", vr: VR::SS },
            E { tag: 0x00451030, alias: "CardiacReconAlgorithm", vr: VR::CS },
            E { tag: 0x00451031, alias: "AvgHeartRateForImage", vr: VR::CS },
            E { tag: 0x00451032, alias: "Temporal Resolution", vr: VR::FL },
            E { tag: 0x00451033, alias: "PctRpeakDelay", vr: VR::CS },
            E { tag: 0x00451034, alias: "ActualPctRpeakDelay", vr: VR::CS },
            E { tag: 0x00451036, alias: "EkgFullMaStartPhase", vr: VR::CS },
            E { tag: 0x00451037, alias: "EkgFullMaEndPhase", vr: VR::CS },
            E { tag: 0x00451038, alias: "EkgModulationMaxMa", vr: VR::CS },
            E { tag: 0x00451039, alias: "EkgModulationMinMa", vr: VR::CS },
            E { tag: 0x0045103B, alias: "Noise Reduction Image Filter Description", vr: VR::LO },
            E { tag: 0x0045103F, alias: "RPeakTimeDelay", vr: VR::IS },
            E { tag: 0x00451044, alias: "ActualRPeakTimeDelay", vr: VR::IS },
            E { tag: 0x00451045, alias: "CardiacScanOptions", vr: VR::ST },
            E { tag: 0x00451050, alias: "Temporal Center View Angle", vr: VR::FD },
            E { tag: 0x00451051, alias: "Recon Center View Angle", vr: VR::FD },
            E { tag: 0x00451052, alias: "WideCone Masking", vr: VR::CS },
            E { tag: 0x00451053, alias: "WideCone Corner Blending Radius", vr: VR::FD },
            E { tag: 0x00451054, alias: "WideCone Corner Blending Radius Offset", vr: VR::FD },
            E { tag: 0x00451055, alias: "Internal Recon Algorithm", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_HINO_CT_01",
        entries: &[
            E { tag: 0x004B1001, alias: "Beam Thickness", vr: VR::DS },
            E { tag: 0x004B1002, alias: "R Time", vr: VR::DS },
            E { tag: 0x004B1003, alias: "HBC number", vr: VR::IS },
        ],
    },
    P {
        creator: "GEMS_IDEN_01",
        entries: &[
            E { tag: 0x00091010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00091001, alias: "Full fidelity", vr: VR::LO },
            E { tag: 0x00091002, alias: "Suite id", vr: VR::SH },
            E { tag: 0x00091004, alias: "Product id", vr: VR::SH },
            E { tag: 0x00091017, alias: "Series Description", vr: VR::LT },
            E { tag: 0x0009101A, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091020, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091027, alias: "Image actual date", vr: VR::SL },
            E { tag: 0x0009102F, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00091030, alias: "Service id", vr: VR::SH },
            E { tag: 0x00091031, alias: "Mobile location number", vr: VR::SH },
            E { tag: 0x000910E2, alias: "Image Type (ORIGINAL\\PROSPECTIVE ...)", vr: VR::LT },
            E { tag: 0x000910E3, alias: "Equipment UID", vr: VR::UI },
            E { tag: 0x000910E6, alias: "Genesis Version - now", vr: VR::SH },
            E { tag: 0x000910E7, alias: "Exam Record checksum", vr: VR::UL },
            E { tag: 0x000910E8, alias: "Series Suite Id", vr: VR::SH },
            E { tag: 0x000910E9, alias: "Actual series data time stamp", vr: VR::SL },
        ],
    },
    P {
        creator: "GEMS_IMAG_01",
        entries: &[
            E { tag: 0x00271010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00271006, alias: "Image archive flag", vr: VR::SL },
            E { tag: 0x00271010, alias: "Scout Type", vr: VR::SS },
            E { tag: 0x0027101C, alias: "Vma mamp", vr: VR::SL },
            E { tag: 0x0027101D, alias: "Vma phase", vr: VR::SS },
            E { tag: 0x0027101E, alias: "Vma mod", vr: VR::SL },
            E { tag: 0x0027101F, alias: "Vma clip", vr: VR::SL },
            E { tag: 0x00271020, alias: "Smart scan ON/OFF flag", vr: VR::SS },
            E { tag: 0x00271030, alias: "Foreign Image Revision", vr: VR::SH },
            E { tag: 0x00271031, alias: "Imaging Mode", vr: VR::SS },
            E { tag: 0x00271032, alias: "Pulse Sequence", vr: VR::SS },
            E { tag: 0x00271033, alias: "Imaging Options", vr: VR::SL },
            E { tag: 0x00271035, alias: "Plane Type", vr: VR::SS },
            E { tag: 0x00271036, alias: "Oblique Plane", vr: VR::SL },
            E { tag: 0x00271040, alias: "RAS letter of image location", vr: VR::SH },
            E { tag: 0x00271041, alias: "Image location", vr: VR::FL },
            E { tag: 0x00271042, alias: "Center R coord of plane image", vr: VR::FL },
            E { tag: 0x00271043, alias: "Center A coord of plane image", vr: VR::FL },
            E { tag: 0x00271044, alias: "Center S coord of plane image", vr: VR::FL },
            E { tag: 0x00271045, alias: "Normal R coord", vr: VR::FL },
            E { tag: 0x00271046, alias: "Normal A coord", vr: VR::FL },
            E { tag: 0x00271047, alias: "Normal S coord", vr: VR::FL },
            E { tag: 0x00271048, alias: "R Coord of Top Right Corner", vr: VR::FL },
            E { tag: 0x00271049, alias: "A Coord of Top Right Corner", vr: VR::FL },
            E { tag: 0x0027104A, alias: "S Coord of Top Right Corner", vr: VR::FL },
            E { tag: 0x0027104B, alias: "R Coord of Bottom Right Corner", vr: VR::FL },
            E { tag: 0x0027104C, alias: "A Coord of Bottom Right Corner", vr: VR::FL },
            E { tag: 0x0027104D, alias: "S Coord of Bottom Right Corner", vr: VR::FL },
            E { tag: 0x00271050, alias: "Scan Start Location", vr: VR::FL },
            E { tag: 0x00271051, alias: "Scan End Location", vr: VR::FL },
            E { tag: 0x00271052, alias: "RAS letter for side of image", vr: VR::SH },
            E { tag: 0x00271053, alias: "RAS letter for anterior/posterior", vr: VR::SH },
            E { tag: 0x00271054, alias: "RAS letter for scout start loc", vr: VR::SH },
            E { tag: 0x00271055, alias: "RAS letter for scout end loc", vr: VR::SH },
            E { tag: 0x00271060, alias: "Image dimension - X", vr: VR::FL },
            E { tag: 0x00271061, alias: "Image dimension - Y", vr: VR::FL },
            E { tag: 0x00271062, alias: "Number of Excitations", vr: VR::FL },
            E { tag: 0x00291015, alias: "Lower range of Pixels1", vr: VR::SL },
            E { tag: 0x00291016, alias: "Lower range of Pixels1", vr: VR::SL },
            E { tag: 0x00291017, alias: "Lower range of Pixels2", vr: VR::SL },
            E { tag: 0x00291018, alias: "Upper range of Pixels2", vr: VR::SL },
            E { tag: 0x0029101A, alias: "Len of tot hdr in bytes No longer supported", vr: VR::SL },
            E { tag: 0x00291026, alias: "Version of the hdr struct", vr: VR::SS },
            E { tag: 0x00291034, alias: "Advantage comp. Overflow", vr: VR::SL },
            E { tag: 0x00291035, alias: "Advantage comp. Underflow", vr: VR::SL },
        ],
    },
    P {
        creator: "GEMS_IMPS_01",
        entries: &[
            E { tag: 0x00291010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00291004, alias: "Lower range of Pixels1", vr: VR::SL },
            E { tag: 0x00291005, alias: "Lower range of Pixels1", vr: VR::DS },
            E { tag: 0x00291006, alias: "Lower range of Pixels1", vr: VR::DS },
            E { tag: 0x00291007, alias: "Lower range of Pixels1", vr: VR::SL },
            E { tag: 0x00291008, alias: "Lower range of Pixels1", vr: VR::SH },
            E { tag: 0x00291009, alias: "Lower range of Pixels1", vr: VR::SH },
            E { tag: 0x0029100A, alias: "Lower range of Pixels1", vr: VR::SS },
            E { tag: 0x00291015, alias: "Lower range of Pixels1", vr: VR::SL },
            E { tag: 0x00291016, alias: "Upper range of Pixels1", vr: VR::SL },
            E { tag: 0x00291017, alias: "Lower range of Pixels2", vr: VR::SL },
            E { tag: 0x00291018, alias: "Upper range of Pixels2", vr: VR::SL },
            E { tag: 0x0029101A, alias: "Len of tot hdr in bytes", vr: VR::SL },
            E { tag: 0x00291026, alias: "Version of the hdr struct", vr: VR::SS },
            E { tag: 0x00291034, alias: "Advantage comp. Overflow", vr: VR::SL },
            E { tag: 0x00291035, alias: "Advantage comp. Underflow", vr: VR::SL },
            E { tag: 0x00431013, alias: "Decon kernel parameters", vr: VR::SS },
        ],
    },
    P {
        creator: "GEMS_IQTB_IDEN_47",
        entries: &[
            E { tag: 0x00471001, alias: "Image Path", vr: VR::SH },
            E { tag: 0x00471002, alias: "Image Date/Time", vr: VR::UL },
            E { tag: 0x00471003, alias: "Queue Sequence Number", vr: VR::UL },
            E { tag: 0x00471004, alias: "Recon Sequence ID", vr: VR::UL },
        ],
    },
    P {
        creator: "GEMS_IT_US_REPORT",
        entries: &[
            E { tag: 0x00451011, alias: "Vivid excel file", vr: VR::OW },
            E { tag: 0x00451012, alias: "Vivid CHM file", vr: VR::OW },
            E { tag: 0x00451013, alias: "Vivid PDF file", vr: VR::OW },
        ],
    },
    P {
        creator: "GEMS_LUNAR_RAW",
        entries: &[
            E { tag: 0x70031001, alias: "enCORE File Name", vr: VR::ST },
            E { tag: 0x70031002, alias: "enCORE File Data", vr: VR::OB },
            E { tag: 0x70031003, alias: "enCORE File Length", vr: VR::UL },
            E { tag: 0x70031004, alias: "enCORE File Modified Time", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_MR_RAW_01",
        entries: &[
            E { tag: 0x70011001, alias: "rdb_hdr_rec", vr: VR::OB },
            E { tag: 0x70011002, alias: "rdb_hdr_per_pass_tab", vr: VR::OB },
            E { tag: 0x70011003, alias: "rdb_hdr_unlock_raw", vr: VR::OB },
            E { tag: 0x70011004, alias: "rdb_hdr_data_acq_tab", vr: VR::OB },
            E { tag: 0x70011005, alias: "rdb_hdr_nex_tab", vr: VR::OB },
            E { tag: 0x70011006, alias: "rdb_hdr_nex_abort_tab", vr: VR::OB },
            E { tag: 0x70011007, alias: "rdb_hdr_tool", vr: VR::OB },
            E { tag: 0x70011008, alias: "rdb_raw_data", vr: VR::OB },
            E { tag: 0x70011009, alias: "SSP save", vr: VR::OB },
            E { tag: 0x7001100A, alias: "UDA save", vr: VR::OB },
            E { tag: 0x7001100B, alias: "rdb_chemsat_data", vr: VR::OB },
        ],
    },
    P {
        creator: "GEMS_PARM_01",
        entries: &[
            E { tag: 0x00431010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00431001, alias: "Bitmap of prescan options", vr: VR::SS },
            E { tag: 0x00431002, alias: "Gradient offset in X", vr: VR::SS },
            E { tag: 0x00431003, alias: "Gradient offset in Y", vr: VR::SS },
            E { tag: 0x00431004, alias: "Gradient offset in Z", vr: VR::SS },
            E { tag: 0x00431005, alias: "Img is original or unoriginal", vr: VR::SS },
            E { tag: 0x00431006, alias: "Number of EPI shots", vr: VR::SS },
            E { tag: 0x00431007, alias: "Views per segment", vr: VR::SS },
            E { tag: 0x00431008, alias: "Respiratory rate, bpm", vr: VR::SS },
            E { tag: 0x00431009, alias: "Respiratory trigger point", vr: VR::SS },
            E { tag: 0x0043100A, alias: "Type of receiver used", vr: VR::SS },
            E { tag: 0x0043100B, alias: "DB/dt Peak rate of change of gradient field", vr: VR::DS },
            E { tag: 0x0043100C, alias: "dB/dt Limits in units of percent", vr: VR::DS },
            E { tag: 0x0043100D, alias: "PSD estimated limit", vr: VR::DS },
            E { tag: 0x0043100E, alias: "PSD estimated limit in tesla per second", vr: VR::DS },
            E { tag: 0x0043100F, alias: "Saravghead", vr: VR::DS },
            E { tag: 0x00431010, alias: "Window value", vr: VR::US },
            E { tag: 0x00431011, alias: "Total input views", vr: VR::US },
            E { tag: 0x00431012, alias: "X-ray chain", vr: VR::SS },
            E { tag: 0x00431013, alias: "Decon kernel parameters", vr: VR::SS },
            E { tag: 0x00431014, alias: "Calibration parameters", vr: VR::SS },
            E { tag: 0x00431015, alias: "Total output views", vr: VR::SS },
            E { tag: 0x00431016, alias: "Number of overranges", vr: VR::SS },
            E { tag: 0x00431017, alias: "IBH image scale factors", vr: VR::DS },
            E { tag: 0x00431018, alias: "BBH coefficients", vr: VR::DS },
            E { tag: 0x00431019, alias: "Number of BBH chains to blend", vr: VR::SS },
            E { tag: 0x0043101A, alias: "Starting channel number", vr: VR::SL },
            E { tag: 0x0043101B, alias: "Ppscan parameters", vr: VR::SS },
            E { tag: 0x0043101C, alias: "GE image integrity", vr: VR::SS },
            E { tag: 0x0043101D, alias: "Level value", vr: VR::SS },
            E { tag: 0x0043101E, alias: "Delta Start Time [msec]", vr: VR::DS },
            E { tag: 0x0043101F, alias: "Max overranges in a view", vr: VR::SL },
            E { tag: 0x00431020, alias: "Avg overranges all views", vr: VR::DS },
            E { tag: 0x00431021, alias: "Corrected after glow terms", vr: VR::SS },
            E { tag: 0x00431025, alias: "Reference channels", vr: VR::SS },
            E { tag: 0x00431026, alias: "No views ref chans blocked", vr: VR::US },
            E { tag: 0x00431027, alias: "Scan Pitch Ratio", vr: VR::SH },
            E { tag: 0x00431028, alias: "Unique image iden", vr: VR::OB },
            E { tag: 0x00431029, alias: "Histogram tables", vr: VR::OB },
            E { tag: 0x0043102A, alias: "User defined data", vr: VR::OB },
            E { tag: 0x0043102B, alias: "Private Scan Options", vr: VR::SS },
            E { tag: 0x0043102C, alias: "Effective echo spacing", vr: VR::SS },
            E { tag: 0x0043102D, alias: "Filter Mode (String slop field 1 in legacy GE MR", vr: VR::SH },
            E { tag: 0x0043102E, alias: "String slop field 2", vr: VR::SH },
            E { tag: 0x0043102F, alias: "Image Type (real, imaginary, phase, magnitude)", vr: VR::SS },
            E { tag: 0x00431030, alias: "Vas collapse flag", vr: VR::SS },
            E { tag: 0x00431031, alias: "Recon Center Coordinates", vr: VR::DS },
            E { tag: 0x00431032, alias: "Vas flags", vr: VR::SS },
            E { tag: 0x00431033, alias: "Neg_scanspacing", vr: VR::FL },
            E { tag: 0x00431034, alias: "Offset Frequency", vr: VR::IS },
            E { tag: 0x00431035, alias: "User_usage_tag", vr: VR::UL },
            E { tag: 0x00431036, alias: "User_fill_map_MSW", vr: VR::UL },
            E { tag: 0x00431037, alias: "User_fill_map_LSW", vr: VR::UL },
            E { tag: 0x00431038, alias: "User data 25...User data 48 {User48=Effective Resolution for spiral}", vr: VR::FL },
            E { tag: 0x00431039, alias: "Slop_int_6... slop_int_9", vr: VR::IS },
            E { tag: 0x00431040, alias: "Trigger on position", vr: VR::FL },
            E { tag: 0x00431041, alias: "Degree of rotation", vr: VR::FL },
            E { tag: 0x00431042, alias: "DAS trigger source", vr: VR::SL },
            E { tag: 0x00431043, alias: "DAS fpa gain", vr: VR::SL },
            E { tag: 0x00431044, alias: "DAS output source", vr: VR::SL },
            E { tag: 0x00431045, alias: "DAS ad input", vr: VR::SL },
            E { tag: 0x00431046, alias: "DAS cal mode", vr: VR::SL },
            E { tag: 0x00431047, alias: "DAS cal frequency", vr: VR::SL },
            E { tag: 0x00431048, alias: "DAS reg xm", vr: VR::SL },
            E { tag: 0x00431049, alias: "DAS auto zero", vr: VR::SL },
            E { tag: 0x0043104A, alias: "Starting channel of view", vr: VR::SS },
            E { tag: 0x0043104B, alias: "DAS xm pattern", vr: VR::SL },
            E { tag: 0x0043104C, alias: "TGGC trigger mode", vr: VR::SS },
            E { tag: 0x0043104D, alias: "Start scan to X-ray on delay", vr: VR::FL },
            E { tag: 0x0043104E, alias: "Duration of X-ray on", vr: VR::FL },
            E { tag: 0x00431060, alias: "Slop_int_10...slop_int_17", vr: VR::IS },
            E { tag: 0x00431061, alias: "Scanner Study Entity UID", vr: VR::UI },
            E { tag: 0x00431062, alias: "Scanner Study ID", vr: VR::SH },
            E { tag: 0x00431063, alias: "Raw Data ID", vr: VR::SH },
            E { tag: 0x00431064, alias: "Image Filter", vr: VR::CS },
            E { tag: 0x00431065, alias: "Motion Correction Indicator", vr: VR::US },
            E { tag: 0x00431066, alias: "Helical Correction Indicator", vr: VR::US },
            E { tag: 0x00431067, alias: "IBO Correction Indicator", vr: VR::US },
            E { tag: 0x00431068, alias: "XT Correction Indicator", vr: VR::US },
            E { tag: 0x00431069, alias: "Q-cal Correction Indicator", vr: VR::US },
            E { tag: 0x0043106A, alias: "AV Correction Indicator", vr: VR::US },
            E { tag: 0x0043106B, alias: "L-MDK Correction Indicator", vr: VR::US },
            E { tag: 0x0043106C, alias: "Detector Row", vr: VR::IS },
            E { tag: 0x0043106D, alias: "Area Size", vr: VR::IS },
            E { tag: 0x0043106E, alias: "Auto mA Mode", vr: VR::SH },
            E { tag: 0x0043106F, alias: "Scanner Table Entry (single gradient coil systems only)/Scanner Table Entry + Gradient Coil Selected", vr: VR::DS },
            E { tag: 0x00431070, alias: "Paradigm Name", vr: VR::LO },
            E { tag: 0x00431071, alias: "Paradigm Description", vr: VR::ST },
            E { tag: 0x00431072, alias: "Paradigm UID", vr: VR::UI },
            E { tag: 0x00431073, alias: "Experiment Type", vr: VR::US },
            E { tag: 0x00431074, alias: "#rest volumes", vr: VR::US },
            E { tag: 0x00431075, alias: "#active volumes", vr: VR::US },
            E { tag: 0x00431076, alias: "#dummy scans", vr: VR::US },
            E { tag: 0x00431077, alias: "Application Name", vr: VR::SH },
            E { tag: 0x00431078, alias: "Application Version", vr: VR::SH },
            E { tag: 0x00431079, alias: "Slices Per Volume", vr: VR::US },
            E { tag: 0x0043107A, alias: "Expected Time Points", vr: VR::US },
            E { tag: 0x0043107B, alias: "Regressor Values", vr: VR::FL },
            E { tag: 0x0043107C, alias: "Delay after slice group", vr: VR::FL },
            E { tag: 0x0043107D, alias: "Recon mode flag word", vr: VR::US },
            E { tag: 0x0043107E, alias: "PACC specific information", vr: VR::LO },
            E { tag: 0x0043107F, alias: "Reserved", vr: VR::DS },
            E { tag: 0x00431080, alias: "Coil ID Data", vr: VR::LO },
            E { tag: 0x00431081, alias: "GE Coil Name", vr: VR::LO },
            E { tag: 0x00431082, alias: "System Configuration Information", vr: VR::LO },
            E { tag: 0x00431083, alias: "Asset R Factors", vr: VR::DS },
            E { tag: 0x00431084, alias: "Additional Asset Data", vr: VR::LO },
            E { tag: 0x00431085, alias: "Debug Data (text format)", vr: VR::UT },
            E { tag: 0x00431086, alias: "Debug Data (binary format)", vr: VR::OB },
            E { tag: 0x00431087, alias: "Reserved", vr: VR::UT },
            E { tag: 0x00431088, alias: "PURE Acquisition Calibration Series UID", vr: VR::UI },
            E { tag: 0x00431089, alias: "Governing Body, dB/dt, and SAR definition", vr: VR::LO },
            E { tag: 0x0043108A, alias: "Private In-Plane Phase Encoding Direction", vr: VR::CS },
            E { tag: 0x0043108B, alias: "FMRI Binary Data Block", vr: VR::OB },
            E { tag: 0x0043108C, alias: "Voxel Location", vr: VR::DS },
            E { tag: 0x0043108D, alias: "SAT Band Locations", vr: VR::DS },
            E { tag: 0x0043108E, alias: "Spectro Prescan Values", vr: VR::DS },
            E { tag: 0x0043108F, alias: "Spectro Parameters", vr: VR::DS },
            E { tag: 0x00431090, alias: "SAR Definition", vr: VR::LO },
            E { tag: 0x00431091, alias: "SAR value", vr: VR::DS },
            E { tag: 0x00431092, alias: "Image Error Text", vr: VR::LO },
            E { tag: 0x00431093, alias: "Spectro Quantitation Values", vr: VR::DS },
            E { tag: 0x00431094, alias: "Spectro Ratio Values", vr: VR::DS },
            E { tag: 0x00431095, alias: "Prescan Reuse String", vr: VR::LO },
            E { tag: 0x00431096, alias: "Content Qualification", vr: VR::CS },
            E { tag: 0x00431097, alias: "Image Filtering Parameters", vr: VR::LO },
            E { tag: 0x00431098, alias: "ASSET Acquisition Calibration Series UID", vr: VR::UI },
            E { tag: 0x00431099, alias: "Extended Options", vr: VR::LO },
            E { tag: 0x0043109A, alias: "Rx Stack Identification", vr: VR::IS },
            E { tag: 0x0043109B, alias: "NPW factor", vr: VR::DS },
            E { tag: 0x0043109C, alias: "Research Tag 1", vr: VR::OB },
            E { tag: 0x0043109D, alias: "Research Tag 2", vr: VR::OB },
            E { tag: 0x0043109E, alias: "Research Tag 3", vr: VR::OB },
            E { tag: 0x0043109F, alias: "Research Tag 4", vr: VR::OB },
            E { tag: 0x004310A0, alias: "Spectroscopy Pixel Sequence", vr: VR::SQ },
            E { tag: 0x004310A1, alias: "Spectroscopy Default Display Sequence", vr: VR::SQ },
            E { tag: 0x004310A2, alias: "MEG Data", vr: VR::DS },
            E { tag: 0x004310A3, alias: "ASL Contrast technique", vr: VR::CS },
            E { tag: 0x004310A4, alias: "Detailed text for ASL labeling technique", vr: VR::LO },
            E { tag: 0x004310A5, alias: "Duration of the label or control pulse", vr: VR::IS },
            E { tag: 0x0051100A, alias: "Store level of Functional Image", vr: VR::SL },
            E { tag: 0x0051100B, alias: "Store B-Value with Functional Image", vr: VR::FL },
            E { tag: 0x20011010, alias: "DICOM Implementation UID", vr: VR::UI },
            E { tag: 0x20011011, alias: "DICOM Implementation Version", vr: VR::SH },
            E { tag: 0x20011012, alias: "Within-DICOM-Implementation SOP Instance UID", vr: VR::UI },
            E { tag: 0x20011013, alias: "Application Name", vr: VR::SH },
            E { tag: 0x20011014, alias: "Application Version", vr: VR::SH },
            E { tag: 0x20011015, alias: "Compatibility Version", vr: VR::SH },
            E { tag: 0x20011021, alias: "Referenced Series UID", vr: VR::UI },
            E { tag: 0x20011031, alias: "Number of Objects Averaged", vr: VR::US },
            E { tag: 0x20011041, alias: "Number of Expected Time Points", vr: VR::US },
            E { tag: 0x20011051, alias: "Number of Slices Per Volume", vr: VR::US },
            E { tag: 0x20011060, alias: "BW Image Type", vr: VR::US },
            E { tag: 0x20011061, alias: "Experiment Type", vr: VR::US },
            E { tag: 0x20011071, alias: "Paradigm UID", vr: VR::UI },
            E { tag: 0x20011072, alias: "Paradigm Name", vr: VR::LO },
            E { tag: 0x20011073, alias: "Paradigm Description", vr: VR::ST },
            E { tag: 0x20011080, alias: "Contrast", vr: VR::OB },
            E { tag: 0x20011081, alias: "Regressor Values", vr: VR::FL },
            E { tag: 0x20011086, alias: "Number of Degrees of Freedom", vr: VR::US },
            E { tag: 0x2001108A, alias: "Z Threshold", vr: VR::FL },
            E { tag: 0x2001108B, alias: "p Threshold", vr: VR::FL },
            E { tag: 0x20011090, alias: "Processing parameters", vr: VR::OB },
            E { tag: 0x20011091, alias: "Motion Plot", vr: VR::OB },
            E { tag: 0x20011092, alias: "ROIs", vr: VR::OB },
            E { tag: 0x20011093, alias: "Tracts", vr: VR::OB },
            E { tag: 0x20011094, alias: "Report", vr: VR::OB },
            E { tag: 0x20011095, alias: "Response Data", vr: VR::OB },
            E { tag: 0x200110A0, alias: "Motion Parameters", vr: VR::FL },
            E { tag: 0x200110A1, alias: "Registration Parameters", vr: VR::FL },
            E { tag: 0x200110A2, alias: "Subject Data", vr: VR::FL },
            E { tag: 0x200110B0, alias: "DTI Parameters", vr: VR::OB },
            E { tag: 0x200110C0, alias: "Paradigm Info", vr: VR::OB },
        ],
    },
    P {
        creator: "GEMS_PATI_01",
        entries: &[
            E { tag: 0x00111010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00111010, alias: "Patient Status", vr: VR::SS },
        ],
    },
    P {
        creator: "GEMS_PETD_01",
        entries: &[
            E { tag: 0x00091001, alias: "GE Discovery PET Implementation Version Name", vr: VR::LO },
            E { tag: 0x00091002, alias: "PET patient_id", vr: VR::LO },
            E { tag: 0x00091003, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x00091004, alias: "GE Advance Patient.software_version", vr: VR::SH },
            E { tag: 0x00091005, alias: "PET patient_datetime", vr: VR::DT },
            E { tag: 0x00091006, alias: "PET type", vr: VR::SL },
            E { tag: 0x00091007, alias: "PET exam_id", vr: VR::UI },
            E { tag: 0x00091008, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x00091009, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x0009100A, alias: "PET scan_id", vr: VR::UI },
            E { tag: 0x0009100B, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x0009100C, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x0009100D, alias: "PET scan_datetime", vr: VR::DT },
            E { tag: 0x0009100E, alias: "PET scan_ready", vr: VR::DT },
            E { tag: 0x0009100F, alias: "PET scan_description", vr: VR::ST },
            E { tag: 0x00091010, alias: "PET hospital_name", vr: VR::LO },
            E { tag: 0x00091011, alias: "PET scanner_desc", vr: VR::LO },
            E { tag: 0x00091012, alias: "PET manufacturer", vr: VR::LO },
            E { tag: 0x00091013, alias: "PET for_identifier", vr: VR::UI },
            E { tag: 0x00091014, alias: "PET landmark_name", vr: VR::LO },
            E { tag: 0x00091015, alias: "PET landmark_abbrev", vr: VR::SH },
            E { tag: 0x00091016, alias: "PET patient_position", vr: VR::SL },
            E { tag: 0x00091017, alias: "PET scan_perspective", vr: VR::SL },
            E { tag: 0x00091018, alias: "PET scan_type", vr: VR::SL },
            E { tag: 0x00091019, alias: "PET scan_mode", vr: VR::SL },
            E { tag: 0x0009101A, alias: "PET start_condition", vr: VR::SL },
            E { tag: 0x0009101B, alias: "PET start_cond_data", vr: VR::SL },
            E { tag: 0x0009101C, alias: "PET sel_stop_cond", vr: VR::SL },
            E { tag: 0x0009101D, alias: "PET sel_stop_cond_data", vr: VR::SL },
            E { tag: 0x0009101E, alias: "PET collect_deadtime", vr: VR::SL },
            E { tag: 0x0009101F, alias: "PET collect_singles", vr: VR::SL },
            E { tag: 0x00091020, alias: "PET collect_countrate", vr: VR::SL },
            E { tag: 0x00091021, alias: "PET countrate_period", vr: VR::SL },
            E { tag: 0x00091022, alias: "PET delayed_events", vr: VR::SL },
            E { tag: 0x00091023, alias: "PET delayed_bias", vr: VR::SL },
            E { tag: 0x00091024, alias: "PET word_size", vr: VR::SL },
            E { tag: 0x00091025, alias: "PET axial_acceptance", vr: VR::SL },
            E { tag: 0x00091026, alias: "PET axial_angle_3d", vr: VR::SL },
            E { tag: 0x00091027, alias: "PET theta_compression", vr: VR::SL },
            E { tag: 0x00091028, alias: "PET axial_compression", vr: VR::SL },
            E { tag: 0x00091029, alias: "PET gantry_tilt_angle", vr: VR::FL },
            E { tag: 0x0009102A, alias: "PET collimation", vr: VR::SL },
            E { tag: 0x0009102B, alias: "PET scan_fov", vr: VR::SL },
            E { tag: 0x0009102C, alias: "PET axial_fov", vr: VR::SL },
            E { tag: 0x0009102D, alias: "PET event_separation", vr: VR::SL },
            E { tag: 0x0009102E, alias: "PET mask_width", vr: VR::SL },
            E { tag: 0x0009102F, alias: "PET binning_mode", vr: VR::SL },
            E { tag: 0x00091030, alias: "PET trig_rej_method", vr: VR::SL },
            E { tag: 0x00091031, alias: "PET number_for_reject", vr: VR::SL },
            E { tag: 0x00091032, alias: "PET lower_reject_limit", vr: VR::SL },
            E { tag: 0x00091033, alias: "PET upper_reject_limit", vr: VR::SL },
            E { tag: 0x00091034, alias: "PET triggers_acquired", vr: VR::SL },
            E { tag: 0x00091035, alias: "PET triggers_rejected", vr: VR::SL },
            E { tag: 0x00091036, alias: "PET tracer_name", vr: VR::LO },
            E { tag: 0x00091037, alias: "PET batch_description", vr: VR::LO },
            E { tag: 0x00091038, alias: "PET tracer_activity", vr: VR::FL },
            E { tag: 0x00091039, alias: "PET meas_datetime", vr: VR::DT },
            E { tag: 0x0009103A, alias: "PET pre_inj_volume", vr: VR::FL },
            E { tag: 0x0009103B, alias: "PET admin_datetime", vr: VR::DT },
            E { tag: 0x0009103C, alias: "PET post_inj_activity", vr: VR::FL },
            E { tag: 0x0009103D, alias: "PET post_inj_datetime", vr: VR::DT },
            E { tag: 0x0009103E, alias: "PET radionuclide_name", vr: VR::SH },
            E { tag: 0x0009103F, alias: "PET half_life", vr: VR::FL },
            E { tag: 0x00091040, alias: "PET positron_fraction", vr: VR::FL },
            E { tag: 0x00091041, alias: "PET source1_holder", vr: VR::SL },
            E { tag: 0x00091042, alias: "PET source1_activity", vr: VR::FL },
            E { tag: 0x00091043, alias: "PET source1_meas_dt", vr: VR::DT },
            E { tag: 0x00091044, alias: "PET source1_radnuclide", vr: VR::SH },
            E { tag: 0x00091045, alias: "PET source1_half_life", vr: VR::FL },
            E { tag: 0x00091046, alias: "PET source2_holder", vr: VR::SL },
            E { tag: 0x00091047, alias: "PET source2_activity", vr: VR::FL },
            E { tag: 0x00091048, alias: "PET source2_meas_dt", vr: VR::DT },
            E { tag: 0x00091049, alias: "PET source2_radnuclide", vr: VR::SH },
            E { tag: 0x0009104A, alias: "PET source2_half_life", vr: VR::FL },
            E { tag: 0x0009104B, alias: "PET source_speed", vr: VR::SL },
            E { tag: 0x0009104C, alias: "PET source_location", vr: VR::FL },
            E { tag: 0x0009104D, alias: "PET emission_present", vr: VR::SL },
            E { tag: 0x0009104E, alias: "PET lower_axial_acc", vr: VR::SL },
            E { tag: 0x0009104F, alias: "PET upper_axial_acc", vr: VR::SL },
            E { tag: 0x00091050, alias: "PET lower_coinc_limit", vr: VR::SL },
            E { tag: 0x00091051, alias: "PET upper_coinc_limit", vr: VR::SL },
            E { tag: 0x00091052, alias: "PET coinc_delay_offset", vr: VR::SL },
            E { tag: 0x00091053, alias: "PET coinc_output_mode", vr: VR::SL },
            E { tag: 0x00091054, alias: "PET upper_energy_limit", vr: VR::SL },
            E { tag: 0x00091055, alias: "PET lower_energy_limit", vr: VR::SL },
            E { tag: 0x00091056, alias: "PET normal_cal_id", vr: VR::UI },
            E { tag: 0x00091057, alias: "PET normal_2d_cal_id", vr: VR::UI },
            E { tag: 0x00091058, alias: "PET blank_cal_id", vr: VR::UI },
            E { tag: 0x00091059, alias: "PET wc_cal_id", vr: VR::UI },
            E { tag: 0x0009105A, alias: "PET derived", vr: VR::SL },
            E { tag: 0x0009105B, alias: "PET contrast_agent", vr: VR::LO },
            E { tag: 0x0009105C, alias: "PET frame_id", vr: VR::UI },
            E { tag: 0x0009105D, alias: "PET scan_id", vr: VR::UI },
            E { tag: 0x0009105E, alias: "PET exam_id", vr: VR::UI },
            E { tag: 0x0009105F, alias: "PET patient_id", vr: VR::LO },
            E { tag: 0x00091060, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x00091061, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x00091062, alias: "PET where_is_frame", vr: VR::ST },
            E { tag: 0x00091063, alias: "PET frame_size", vr: VR::SL },
            E { tag: 0x00091064, alias: "PET file_exists", vr: VR::SL },
            E { tag: 0x00091065, alias: "PET patient_entry", vr: VR::SL },
            E { tag: 0x00091066, alias: "PET table_height", vr: VR::FL },
            E { tag: 0x00091067, alias: "PET table_z_position", vr: VR::FL },
            E { tag: 0x00091068, alias: "PET landmark_datetime", vr: VR::DT },
            E { tag: 0x00091069, alias: "PET slice_count", vr: VR::SL },
            E { tag: 0x0009106A, alias: "PET start_location", vr: VR::FL },
            E { tag: 0x0009106B, alias: "PET acq_delay", vr: VR::SL },
            E { tag: 0x0009106C, alias: "PET acq_start", vr: VR::DT },
            E { tag: 0x0009106D, alias: "PET acq_duration", vr: VR::SL },
            E { tag: 0x0009106E, alias: "PET acq_bin_dur", vr: VR::SL },
            E { tag: 0x0009106F, alias: "PET acq_bin_start", vr: VR::SL },
            E { tag: 0x00091070, alias: "PET actual_stop_cond", vr: VR::SL },
            E { tag: 0x00091071, alias: "PET total_prompts", vr: VR::FD },
            E { tag: 0x00091072, alias: "PET total_delays", vr: VR::FD },
            E { tag: 0x00091073, alias: "PET frame_valid", vr: VR::SL },
            E { tag: 0x00091074, alias: "PET validity_info", vr: VR::SL },
            E { tag: 0x00091075, alias: "PET archived", vr: VR::SL },
            E { tag: 0x00091076, alias: "PET compression", vr: VR::SL },
            E { tag: 0x00091077, alias: "PET uncompressed_size", vr: VR::SL },
            E { tag: 0x00091078, alias: "PET accum_bin_dur", vr: VR::SL },
            E { tag: 0x00091079, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x0009107A, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x0009107B, alias: "PET is_datetime", vr: VR::DT },
            E { tag: 0x0009107C, alias: "PET is_source", vr: VR::SL },
            E { tag: 0x0009107D, alias: "PET is_contents", vr: VR::SL },
            E { tag: 0x0009107E, alias: "PET is_type", vr: VR::SL },
            E { tag: 0x0009107F, alias: "PET is_reference", vr: VR::DS },
            E { tag: 0x00091080, alias: "PET multi_patient", vr: VR::SL },
            E { tag: 0x00091081, alias: "PET number_of_normals", vr: VR::SL },
            E { tag: 0x00091082, alias: "PET color_map_id", vr: VR::UI },
            E { tag: 0x00091083, alias: "PET window_level_type", vr: VR::SL },
            E { tag: 0x00091084, alias: "PET rotate", vr: VR::FL },
            E { tag: 0x00091085, alias: "PET flip", vr: VR::SL },
            E { tag: 0x00091086, alias: "PET zoom", vr: VR::FL },
            E { tag: 0x00091087, alias: "PET pan_x", vr: VR::SL },
            E { tag: 0x00091088, alias: "PET pan_y", vr: VR::SL },
            E { tag: 0x00091089, alias: "PET window_level_min", vr: VR::FL },
            E { tag: 0x0009108A, alias: "PET window_level_max", vr: VR::FL },
            E { tag: 0x0009108B, alias: "PET recon_method", vr: VR::SL },
            E { tag: 0x0009108C, alias: "PET attenuation", vr: VR::SL },
            E { tag: 0x0009108D, alias: "PET atten_coefficient", vr: VR::FL },
            E { tag: 0x0009108E, alias: "PET bp_filter", vr: VR::SL },
            E { tag: 0x0009108F, alias: "PET bp_filter_cutoff", vr: VR::FL },
            E { tag: 0x00091090, alias: "PET bp_filter_order", vr: VR::SL },
            E { tag: 0x00091091, alias: "PET bp_center_l", vr: VR::FL },
            E { tag: 0x00091092, alias: "PET bp_center_p", vr: VR::FL },
            E { tag: 0x00091093, alias: "PET atten_smooth", vr: VR::SL },
            E { tag: 0x00091094, alias: "PET atten_smooth_param", vr: VR::SL },
            E { tag: 0x00091095, alias: "PET angle_smooth_param", vr: VR::SL },
            E { tag: 0x00091096, alias: "PET wellcountercal_id", vr: VR::UI },
            E { tag: 0x00091097, alias: "PET trans_scan_id", vr: VR::UI },
            E { tag: 0x00091098, alias: "PET norm_cal_id", vr: VR::UI },
            E { tag: 0x00091099, alias: "PET blnk_cal_id", vr: VR::UI },
            E { tag: 0x0009109A, alias: "PET cac_edge_threshold", vr: VR::FL },
            E { tag: 0x0009109B, alias: "PET cac_skull_offset", vr: VR::FL },
            E { tag: 0x0009109C, alias: "PET emiss_sub_id", vr: VR::UI },
            E { tag: 0x0009109D, alias: "PET radial_filter_3d", vr: VR::SL },
            E { tag: 0x0009109E, alias: "PET radial_cutoff_3d", vr: VR::FL },
            E { tag: 0x0009109F, alias: "PET axial_filter_3d", vr: VR::SL },
            E { tag: 0x000910A0, alias: "PET axial_cutoff_3d", vr: VR::FL },
            E { tag: 0x000910A1, alias: "PET axial_start", vr: VR::FL },
            E { tag: 0x000910A2, alias: "PET axial_spacing", vr: VR::FL },
            E { tag: 0x000910A3, alias: "PET axial_angles_used", vr: VR::SL },
            E { tag: 0x000910A4, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x000910A5, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x000910A6, alias: "PET slice_number", vr: VR::SL },
            E { tag: 0x000910A7, alias: "PET total_counts", vr: VR::FL },
            E { tag: 0x000910A8, alias: "PET other_atts", vr: VR::OB },
            E { tag: 0x000910A9, alias: "PET other_atts_size", vr: VR::SL },
            E { tag: 0x000910AA, alias: "PET archived", vr: VR::SL },
            E { tag: 0x000910AB, alias: "PET bp_center_x", vr: VR::FL },
            E { tag: 0x000910AC, alias: "PET bp_center_y", vr: VR::FL },
            E { tag: 0x000910AD, alias: "PET trans_frame_id", vr: VR::UI },
            E { tag: 0x000910AE, alias: "PET tpluse_frame_id", vr: VR::UI },
            E { tag: 0x000910B1, alias: "PET profile_spacing", vr: VR::FL },
            E { tag: 0x000910B2, alias: "PET ir_num_iterations", vr: VR::SL },
            E { tag: 0x000910B3, alias: "PET ir_num_subsets", vr: VR::SL },
            E { tag: 0x000910B4, alias: "PET ir_recon_fov", vr: VR::FL },
            E { tag: 0x000910B5, alias: "PET ir_corr_model", vr: VR::SL },
            E { tag: 0x000910B6, alias: "PET ir_loop_filter", vr: VR::SL },
            E { tag: 0x000910B7, alias: "PET ir_pre_filt_parm", vr: VR::FL },
            E { tag: 0x000910B8, alias: "PET ir_loop_filt_parm", vr: VR::SL },
            E { tag: 0x000910B9, alias: "PET response_filt_parm", vr: VR::FL },
            E { tag: 0x000910BA, alias: "PET post_filter", vr: VR::SL },
            E { tag: 0x000910BB, alias: "PET post_filt_parm", vr: VR::FL },
            E { tag: 0x000910BC, alias: "PET ir_regularize", vr: VR::SL },
            E { tag: 0x000910BD, alias: "PET regularize_parm", vr: VR::FL },
            E { tag: 0x000910BE, alias: "PET ac_bp_filter", vr: VR::SL },
            E { tag: 0x000910BF, alias: "PET ac_bp_filt_cut_off", vr: VR::FL },
            E { tag: 0x000910C0, alias: "PET ac_bp_filt_order", vr: VR::SL },
            E { tag: 0x000910C1, alias: "PET ac_img_smooth", vr: VR::SL },
            E { tag: 0x000910C2, alias: "PET ac_img_smooth_parm", vr: VR::FL },
            E { tag: 0x000910C3, alias: "PET scatter_method", vr: VR::SL },
            E { tag: 0x000910C4, alias: "PET scatter_num_iter", vr: VR::SL },
            E { tag: 0x000910C5, alias: "PET scatter_parm", vr: VR::FL },
            E { tag: 0x000910C6, alias: "PET seg_qc_parm", vr: VR::FL },
            E { tag: 0x000910C7, alias: "PET overlap", vr: VR::SL },
            E { tag: 0x000910C8, alias: "PET ovlp_frm_id", vr: VR::UI },
            E { tag: 0x000910C9, alias: "PET ovlp_trans_frm_id", vr: VR::UI },
            E { tag: 0x000910CA, alias: "PET ovlp_tpulse_frm_id", vr: VR::UI },
            E { tag: 0x000910CB, alias: "PET vqc_x_axis_trans", vr: VR::FL },
            E { tag: 0x000910CC, alias: "PET vqc_x_axis_tilt", vr: VR::FL },
            E { tag: 0x000910CD, alias: "PET vqc_y_axis_trans", vr: VR::FL },
            E { tag: 0x000910CE, alias: "PET vqc_y_axis_swivel", vr: VR::FL },
            E { tag: 0x000910CF, alias: "PET vqc_z_axis_trans", vr: VR::FL },
            E { tag: 0x000910D0, alias: "PET vqc_z_axis_roll", vr: VR::FL },
            E { tag: 0x000910D1, alias: "PET ctac_conv_scale", vr: VR::LO },
            E { tag: 0x000910D2, alias: "PET image_set_id", vr: VR::UI },
            E { tag: 0x000910D3, alias: "PET constrast_route", vr: VR::SL },
            E { tag: 0x000910D4, alias: "PET ctac_conv_scale", vr: VR::LO },
            E { tag: 0x000910D5, alias: "PET loop_filter_parm", vr: VR::FL },
            E { tag: 0x000910D6, alias: "PET image_one_loc", vr: VR::FL },
            E { tag: 0x000910D7, alias: "PET image_index_loc", vr: VR::FL },
            E { tag: 0x000910D8, alias: "PET frame_number", vr: VR::SL },
            E { tag: 0x000910D9, alias: "PET list_file_exists", vr: VR::SL },
            E { tag: 0x000910DA, alias: "PET where_is_list_frame", vr: VR::ST },
            E { tag: 0x000910DB, alias: "PET ir_z_filter_flag", vr: VR::SL },
            E { tag: 0x000910DC, alias: "PET ir_z_filter_ratio", vr: VR::FL },
            E { tag: 0x000910DD, alias: "PET num_of_rr_interval", vr: VR::US },
            E { tag: 0x000910DE, alias: "PET num_of_time_slots", vr: VR::US },
            E { tag: 0x000910DF, alias: "PET num_of_slices", vr: VR::US },
            E { tag: 0x000910E0, alias: "PET num_of_time_slices", vr: VR::US },
            E { tag: 0x000910E1, alias: "PET unlisted_scan", vr: VR::SL },
            E { tag: 0x000910E2, alias: "PET rest_stress", vr: VR::SL },
            E { tag: 0x000910E3, alias: "PET phase percentage", vr: VR::FL },
            E { tag: 0x000910E4, alias: "Recon Protocol", vr: VR::ST },
            E { tag: 0x000910E5, alias: "PET left shift", vr: VR::FL },
            E { tag: 0x000910E6, alias: "PET posterior shift", vr: VR::FL },
            E { tag: 0x000910E7, alias: "PET superior shift", vr: VR::FL },
            E { tag: 0x000910E8, alias: "PET acq_bin_num", vr: VR::SL },
            E { tag: 0x000910E9, alias: "PET acq_bin_dur_percent", vr: VR::FL },
            E { tag: 0x000910EA, alias: "3D Filter flag", vr: VR::SL },
            E { tag: 0x000910EB, alias: "3D Filter cutoff", vr: VR::FL },
            E { tag: 0x000910EC, alias: "3D Filter order", vr: VR::SL },
            E { tag: 0x000910F0, alias: "Reformat group", vr: VR::UI },
            E { tag: 0x00111001, alias: "GE Advance ROI Sequence", vr: VR::SQ },
            E { tag: 0x00111002, alias: "GE Advance ROI.roi_id", vr: VR::UI },
            E { tag: 0x00111003, alias: "GE Advance ROI.image_id", vr: VR::UI },
            E { tag: 0x00111004, alias: "GE Advance ROI.compatible_version", vr: VR::SH },
            E { tag: 0x00111005, alias: "GE Advance ROI.software_version", vr: VR::SH },
            E { tag: 0x00111006, alias: "GE Advance ROI.roi_name", vr: VR::LO },
            E { tag: 0x00111007, alias: "GE Advance ROI.roi_datetime", vr: VR::DT },
            E { tag: 0x00111008, alias: "GE Advance ROI.roi_type", vr: VR::SL },
            E { tag: 0x00111009, alias: "GE Advance ROI.center_x", vr: VR::FL },
            E { tag: 0x0011100A, alias: "GE Advance ROI.center_y", vr: VR::FL },
            E { tag: 0x0011100B, alias: "GE Advance ROI.width", vr: VR::FL },
            E { tag: 0x0011100C, alias: "GE Advance ROI.height", vr: VR::FL },
            E { tag: 0x0011100D, alias: "GE Advance ROI.angle", vr: VR::FL },
            E { tag: 0x0011100E, alias: "GE Advance ROI.number_of_points", vr: VR::SL },
            E { tag: 0x0011100F, alias: "GE Advance ROI.roi_data", vr: VR::OB },
            E { tag: 0x00111010, alias: "GE Advance ROI.roi_size", vr: VR::SL },
            E { tag: 0x00111011, alias: "GE Advance ROI.color", vr: VR::LO },
            E { tag: 0x00111012, alias: "GE Advance ROI.line_type", vr: VR::SL },
            E { tag: 0x00111013, alias: "GE Advance ROI.line_width", vr: VR::SL },
            E { tag: 0x00111014, alias: "GE Advance ROI.roi_number", vr: VR::SL },
            E { tag: 0x00111015, alias: "GE Advance ROI.convex", vr: VR::SL },
            E { tag: 0x00111016, alias: "GE Advance ROI.atten_corr_flag", vr: VR::SL },
            E { tag: 0x00111018, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00131001, alias: "GE Advance Annotation Sequence", vr: VR::SQ },
            E { tag: 0x00131002, alias: "GE Advance Annotation.annotation_id", vr: VR::UI },
            E { tag: 0x00131003, alias: "GE Advance Annotation.image_id", vr: VR::UI },
            E { tag: 0x00131004, alias: "GE Advance Annotation.compatible_version", vr: VR::SH },
            E { tag: 0x00131005, alias: "GE Advance Annotation.software_version", vr: VR::SH },
            E { tag: 0x00131006, alias: "GE Advance Annotation.type", vr: VR::SL },
            E { tag: 0x00131007, alias: "GE Advance Annotation.font_name", vr: VR::LO },
            E { tag: 0x00131008, alias: "GE Advance Annotation.font_size", vr: VR::SH },
            E { tag: 0x00131009, alias: "GE Advance Annotation.foreground_color", vr: VR::LO },
            E { tag: 0x0013100A, alias: "GE Advance Annotation.background_color", vr: VR::LO },
            E { tag: 0x0013100B, alias: "GE Advance Annotation.coordinate_system", vr: VR::SL },
            E { tag: 0x0013100C, alias: "GE Advance Annotation.start_x", vr: VR::FL },
            E { tag: 0x0013100D, alias: "GE Advance Annotation.start_y", vr: VR::FL },
            E { tag: 0x0013100E, alias: "GE Advance Annotation.end_x", vr: VR::FL },
            E { tag: 0x0013100F, alias: "GE Advance Annotation.end_y", vr: VR::FL },
            E { tag: 0x00131010, alias: "GE Advance Annotation.start_symbol", vr: VR::SL },
            E { tag: 0x00131011, alias: "GE Advance Annotation.end_symbol", vr: VR::SL },
            E { tag: 0x00131012, alias: "GE Advance Annotation.annotation_data", vr: VR::OB },
            E { tag: 0x00131013, alias: "GE Advance Annotation.annotation_size", vr: VR::SL },
            E { tag: 0x00131014, alias: "GE Advance Annotation.label_id", vr: VR::LO },
            E { tag: 0x00171001, alias: "PET correction_cal_id", vr: VR::UI },
            E { tag: 0x00171002, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x00171003, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x00171004, alias: "PET cal_datetime", vr: VR::DT },
            E { tag: 0x00171005, alias: "PET cal_description", vr: VR::LO },
            E { tag: 0x00171006, alias: "PET cal_type", vr: VR::SL },
            E { tag: 0x00171007, alias: "PET where_is_corr", vr: VR::ST },
            E { tag: 0x00171008, alias: "PET corr_file_size", vr: VR::SL },
            E { tag: 0x00171009, alias: "PET scan_id", vr: VR::LO },
            E { tag: 0x0017100A, alias: "PET scan_datetime", vr: VR::DT },
            E { tag: 0x0017100B, alias: "PET norm_2d_cal_id", vr: VR::LO },
            E { tag: 0x0017100C, alias: "PET hosp_identifier", vr: VR::SH },
            E { tag: 0x0017100D, alias: "PET archived", vr: VR::SL },
            E { tag: 0x00191001, alias: "PET wc_cal_id", vr: VR::UI },
            E { tag: 0x00191002, alias: "PET compatible_version", vr: VR::SH },
            E { tag: 0x00191003, alias: "PET software_version", vr: VR::SH },
            E { tag: 0x00191004, alias: "PET cal_datetime", vr: VR::DT },
            E { tag: 0x00191005, alias: "PET cal_type", vr: VR::SL },
            E { tag: 0x00191006, alias: "PET cal_description", vr: VR::LO },
            E { tag: 0x00191007, alias: "PET cal_hardware", vr: VR::LO },
            E { tag: 0x00191008, alias: "PET coefficients", vr: VR::OB },
            E { tag: 0x00191009, alias: "PET activity_factor_hr", vr: VR::FL },
            E { tag: 0x0019100A, alias: "PET activity_factor_hs", vr: VR::FL },
            E { tag: 0x0019100B, alias: "PET activity_factor_3d", vr: VR::FL },
            E { tag: 0x0019100C, alias: "PET scan_id", vr: VR::LO },
            E { tag: 0x0019100D, alias: "PET scan_datetime", vr: VR::DT },
            E { tag: 0x0019100E, alias: "PET hosp_identifier", vr: VR::SH },
            E { tag: 0x0019100F, alias: "PET meas_activity", vr: VR::FL },
            E { tag: 0x00191010, alias: "PET meas_datetime", vr: VR::DT },
            E { tag: 0x00191011, alias: "PET axial_filter_3d", vr: VR::SL },
            E { tag: 0x00191012, alias: "PET axial_cutoff_3d", vr: VR::FL },
            E { tag: 0x00191013, alias: "PET default_flag", vr: VR::SL },
            E { tag: 0x00191014, alias: "PET archived", vr: VR::SL },
            E { tag: 0x00191015, alias: "PET wc_cal_rec_method", vr: VR::SL },
            E { tag: 0x00191016, alias: "PET activity_factor_2d", vr: VR::SL },
            E { tag: 0x00191017, alias: "PET isotope", vr: VR::SL },
            E { tag: 0x00211001, alias: "PET raw_data_type", vr: VR::US },
            E { tag: 0x00211002, alias: "PET raw_data_size", vr: VR::UL },
            E { tag: 0x00231001, alias: "raw_data_blob", vr: VR::OB },
            E { tag: 0x00231002, alias: "PET raw_data_blob", vr: VR::OB },
            E { tag: 0x50011001, alias: "GE Advance Curve.curve_id", vr: VR::UI },
            E { tag: 0x50011002, alias: "GE Advance Curve.compatible_version", vr: VR::SH },
            E { tag: 0x50011003, alias: "GE Advance Curve.software_version", vr: VR::SH },
            E { tag: 0x50011004, alias: "GE Advance Curve.statistics_type", vr: VR::SL },
            E { tag: 0x50011005, alias: "GE Advance Curve.how_derived", vr: VR::LT },
            E { tag: 0x50011006, alias: "GE Advance Curve.how_derived_size", vr: VR::SL },
            E { tag: 0x50011007, alias: "GE Advance Curve.multi_patient", vr: VR::SL },
            E { tag: 0x50011008, alias: "GE Advance Curve.deadtime", vr: VR::SL },
            E { tag: 0x50031001, alias: "GE Advance Graph Sequence", vr: VR::SQ },
            E { tag: 0x50031002, alias: "GE Advance Graph.graph_id", vr: VR::UI },
            E { tag: 0x50031003, alias: "GE Advance Graph.compatible_version", vr: VR::SH },
            E { tag: 0x50031004, alias: "GE Advance Graph.software_version", vr: VR::SH },
            E { tag: 0x50031005, alias: "GE Advance Graph.title", vr: VR::LO },
            E { tag: 0x50031006, alias: "GE Advance Graph.graph_datetime", vr: VR::DT },
            E { tag: 0x50031007, alias: "GE Advance Graph.graph_description", vr: VR::ST },
            E { tag: 0x50031008, alias: "GE Advance Graph.title_font_name", vr: VR::LO },
            E { tag: 0x50031009, alias: "GE Advance Graph.title_font_size", vr: VR::SH },
            E { tag: 0x5003100A, alias: "GE Advance Graph.footer", vr: VR::LO },
            E { tag: 0x5003100B, alias: "GE Advance Graph.footer_font_size", vr: VR::SH },
            E { tag: 0x5003100C, alias: "GE Advance Graph.foreground_color", vr: VR::LO },
            E { tag: 0x5003100D, alias: "GE Advance Graph.background_color", vr: VR::LO },
            E { tag: 0x5003100E, alias: "GE Advance Graph.graph_border", vr: VR::SL },
            E { tag: 0x5003100F, alias: "GE Advance Graph.graph_width", vr: VR::SL },
            E { tag: 0x50031010, alias: "GE Advance Graph.graph_height", vr: VR::SL },
            E { tag: 0x50031011, alias: "GE Advance Graph.grid", vr: VR::SL },
            E { tag: 0x50031012, alias: "GE Advance Graph.label_font_name", vr: VR::LO },
            E { tag: 0x50031013, alias: "GE Advance Graph.label_font_size", vr: VR::SH },
            E { tag: 0x50031014, alias: "GE Advance Graph.axes_color", vr: VR::LO },
            E { tag: 0x50031015, alias: "GE Advance Graph.x_axis_label", vr: VR::LO },
            E { tag: 0x50031016, alias: "GE Advance Graph.x_axis_units", vr: VR::SL },
            E { tag: 0x50031017, alias: "GE Advance Graph.x_major_tics", vr: VR::FL },
            E { tag: 0x50031018, alias: "GE Advance Graph.x_axis_min", vr: VR::FL },
            E { tag: 0x50031019, alias: "GE Advance Graph.x_axis_max", vr: VR::FL },
            E { tag: 0x5003101A, alias: "GE Advance Graph.y_axis_label", vr: VR::LO },
            E { tag: 0x5003101B, alias: "GE Advance Graph.y_axis_units", vr: VR::SL },
            E { tag: 0x5003101C, alias: "GE Advance Graph.y_major_tics", vr: VR::FL },
            E { tag: 0x5003101D, alias: "GE Advance Graph.y_axis_min", vr: VR::FL },
            E { tag: 0x5003101E, alias: "GE Advance Graph.y_axis_max", vr: VR::FL },
            E { tag: 0x5003101F, alias: "GE Advance Graph.legend_font_name", vr: VR::LO },
            E { tag: 0x50031020, alias: "GE Advance Graph.legend_font_size", vr: VR::SH },
            E { tag: 0x50031021, alias: "GE Advance Graph.legend_location_x", vr: VR::SL },
            E { tag: 0x50031022, alias: "GE Advance Graph.legend_location_y", vr: VR::SL },
            E { tag: 0x50031023, alias: "GE Advance Graph.legend_width", vr: VR::SL },
            E { tag: 0x50031024, alias: "GE Advance Graph.legend_height", vr: VR::SL },
            E { tag: 0x50031025, alias: "GE Advance Graph.legend_border", vr: VR::SL },
            E { tag: 0x50031026, alias: "GE Advance Graph.multi_patient", vr: VR::SL },
            E { tag: 0x50051001, alias: "GE Advance CurvePresentation Sequence", vr: VR::SQ },
            E { tag: 0x50051002, alias: "GE Advance CurvePresentation.curvepresent_id", vr: VR::UI },
            E { tag: 0x50051003, alias: "GE Advance CurvePresentation.graph_id", vr: VR::UI },
            E { tag: 0x50051004, alias: "GE Advance CurvePresentation.curve_id", vr: VR::UI },
            E { tag: 0x50051005, alias: "GE Advance CurvePresentation.compatible_version", vr: VR::SH },
            E { tag: 0x50051006, alias: "GE Advance CurvePresentation.software_version", vr: VR::SH },
            E { tag: 0x50051007, alias: "GE Advance CurvePresentation.curve_label", vr: VR::LO },
            E { tag: 0x50051008, alias: "GE Advance CurvePresentation.color", vr: VR::LO },
            E { tag: 0x50051009, alias: "GE Advance CurvePresentation.line_type", vr: VR::SL },
            E { tag: 0x5005100A, alias: "GE Advance CurvePresentation.line_width", vr: VR::SL },
            E { tag: 0x5005100B, alias: "GE Advance CurvePresentation.point_symbol", vr: VR::SL },
            E { tag: 0x5005100C, alias: "GE Advance CurvePresentation.point_symbol_dim", vr: VR::SL },
            E { tag: 0x5005100D, alias: "GE Advance CurvePresentation.point_color", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_QVA_PHOTO_01",
        entries: &[
            E { tag: 0x00091060, alias: "Dodge End Diastolic Volume ml", vr: VR::FL },
            E { tag: 0x00091061, alias: "Dodge End Systolic Volume ml", vr: VR::FL },
            E { tag: 0x00091062, alias: "Dodge Stroke Volume ml", vr: VR::FL },
            E { tag: 0x00091063, alias: "Dodge Ejection Fraction", vr: VR::IS },
            E { tag: 0x00091064, alias: "Simpson End Diastolic Volume ml", vr: VR::FL },
            E { tag: 0x00091065, alias: "Simpson End Systolic Volume ml", vr: VR::FL },
            E { tag: 0x00091066, alias: "Simpson Stroke Volume ml", vr: VR::FL },
            E { tag: 0x00091067, alias: "Simpson Ejection Fraction", vr: VR::IS },
            E { tag: 0x00091068, alias: "CFX Single Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x00091069, alias: "CFX Single Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x0009106A, alias: "CFX Single Total LV contour Percent", vr: VR::IS },
            E { tag: 0x0009106B, alias: "CFX Multiple Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x0009106C, alias: "CFX Multiple Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x0009106D, alias: "CFX Multiple Total LV contour Percent", vr: VR::IS },
            E { tag: 0x0009106E, alias: "RCA Single Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x0009106F, alias: "RCA Single Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x00091070, alias: "RCA Single Total LV contour Percent", vr: VR::IS },
            E { tag: 0x00091071, alias: "RCA Multiple Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x00091072, alias: "RCA Multiple Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x00091073, alias: "RCA Multiple Total LV contour Percent", vr: VR::IS },
            E { tag: 0x00091074, alias: "LAD Single Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x00091075, alias: "LAD Single Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x00091076, alias: "LAD Single Total LV contour Percent", vr: VR::IS },
            E { tag: 0x00091077, alias: "LAD Multiple Hypokinesia in Region", vr: VR::FL },
            E { tag: 0x00091078, alias: "LAD Multiple Hyperkinesia in Opposite Region", vr: VR::FL },
            E { tag: 0x00091079, alias: "LAD Multiple Total LV contour Percent", vr: VR::IS },
            E { tag: 0x0009107A, alias: "Dodge End Diastolic Volume ml/m2", vr: VR::FL },
            E { tag: 0x0009107C, alias: "Dodge End Systolic Volume ml/m2", vr: VR::FL },
            E { tag: 0x0009107E, alias: "Dodge Stroke Volume ml/m2", vr: VR::FL },
            E { tag: 0x00091080, alias: "Simpson End Diastolic Volume ml/m2", vr: VR::FL },
            E { tag: 0x00091082, alias: "Simpson End Systolic Volume ml/m2", vr: VR::FL },
            E { tag: 0x00091084, alias: "Simpson Stroke Volume ml/m2", vr: VR::FL },
        ],
    },
    P {
        creator: "GEMS_RELA_01",
        entries: &[
            E { tag: 0x00211010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00211003, alias: "Series from which Prescribed", vr: VR::SS },
            E { tag: 0x00211005, alias: "Genesis Version - now", vr: VR::SH },
            E { tag: 0x00211007, alias: "Series Record checksum", vr: VR::UL },
            E { tag: 0x00211015, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211016, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00211018, alias: "Genesis version - Now", vr: VR::SH },
            E { tag: 0x00211019, alias: "Acq recon record checksum", vr: VR::UL },
            E { tag: 0x00211020, alias: "Table start location", vr: VR::DS },
            E { tag: 0x00211035, alias: "Series from which prescribed", vr: VR::SS },
            E { tag: 0x00211036, alias: "Image from which prescribed", vr: VR::SS },
            E { tag: 0x00211037, alias: "Screen Format", vr: VR::SS },
            E { tag: 0x0021104A, alias: "Anatomical reference for scout", vr: VR::LO },
            E { tag: 0x0021104E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021104F, alias: "Locations in acquisition", vr: VR::SS },
            E { tag: 0x00211050, alias: "Graphically prescribed", vr: VR::SS },
            E { tag: 0x00211051, alias: "Rotation from source x rot", vr: VR::DS },
            E { tag: 0x00211052, alias: "Rotation from source y rot", vr: VR::DS },
            E { tag: 0x00211053, alias: "Rotation from source z rot", vr: VR::DS },
            E { tag: 0x00211054, alias: "Image position", vr: VR::SH },
            E { tag: 0x00211055, alias: "Image orientation", vr: VR::SH },
            E { tag: 0x00211056, alias: "Num 3D slabs", vr: VR::SL },
            E { tag: 0x00211057, alias: "Locs per 3D slab", vr: VR::SL },
            E { tag: 0x00211058, alias: "Overlaps", vr: VR::SL },
            E { tag: 0x00211059, alias: "Image Filtering 0.5/0.2T", vr: VR::SL },
            E { tag: 0x0021105A, alias: "Diffusion direction", vr: VR::SL },
            E { tag: 0x0021105B, alias: "Tagging Flip Angle", vr: VR::DS },
            E { tag: 0x0021105C, alias: "Tagging Orientation", vr: VR::DS },
            E { tag: 0x0021105D, alias: "Tag Spacing", vr: VR::DS },
            E { tag: 0x0021105E, alias: "RTIA_timer", vr: VR::DS },
            E { tag: 0x0021105F, alias: "Fps", vr: VR::DS },
            E { tag: 0x00211070, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00211071, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00211081, alias: "Auto window/level alpha", vr: VR::DS },
            E { tag: 0x00211082, alias: "Auto window/level beta", vr: VR::DS },
            E { tag: 0x00211083, alias: "Auto window/level window", vr: VR::DS },
            E { tag: 0x00211084, alias: "Auto window/level level", vr: VR::DS },
            E { tag: 0x00211090, alias: "Tube focal spot position", vr: VR::SS },
            E { tag: 0x00211091, alias: "Biopsy position", vr: VR::SS },
            E { tag: 0x00211092, alias: "Biopsy T location", vr: VR::FL },
            E { tag: 0x00211093, alias: "Biopsy ref location", vr: VR::FL },
        ],
    },
    P {
        creator: "GEMS_RTEN_01",
        entries: &[
            E { tag: 0x02491011, alias: "Plan Creation Date", vr: VR::DA },
            E { tag: 0x02491013, alias: "Plan Creation Time", vr: VR::TM },
            E { tag: 0x02491014, alias: "Operator Name", vr: VR::PN },
            E { tag: 0x02491016, alias: "Plan Comment", vr: VR::LO },
            E { tag: 0x02491018, alias: "Plan Image Sequence", vr: VR::SQ },
            E { tag: 0x0249101A, alias: "Exam/Series/Image Identifier", vr: VR::LO },
            E { tag: 0x0249101B, alias: "Additional Image Identifier", vr: VR::IS },
            E { tag: 0x0249101C, alias: "Build Resolution", vr: VR::CS },
            E { tag: 0x02491020, alias: "Structure Sequence", vr: VR::SQ },
            E { tag: 0x02491022, alias: "Structure Name", vr: VR::SH },
            E { tag: 0x02491024, alias: "Structure Type", vr: VR::CS },
            E { tag: 0x02491028, alias: "Structure Color", vr: VR::CS },
            E { tag: 0x02491030, alias: "Slab Sequence", vr: VR::SQ },
            E { tag: 0x02491032, alias: "Slab Image Sequence", vr: VR::SQ },
            E { tag: 0x02491036, alias: "Z Plus Thickness", vr: VR::DS },
            E { tag: 0x02491038, alias: "Z Minus Thickness", vr: VR::DS },
            E { tag: 0x0249103A, alias: "Slab Shape", vr: VR::DS },
            E { tag: 0x02491040, alias: "Marker Sequence", vr: VR::SQ },
            E { tag: 0x02491042, alias: "Marker Name", vr: VR::SH },
            E { tag: 0x02491044, alias: "Marker Color", vr: VR::CS },
            E { tag: 0x02491046, alias: "Marker Position", vr: VR::DS },
            E { tag: 0x02491050, alias: "Beam Group Sequence", vr: VR::SQ },
            E { tag: 0x02491051, alias: "Group Name", vr: VR::SH },
            E { tag: 0x02491052, alias: "Group Properties", vr: VR::CS },
            E { tag: 0x02491054, alias: "Beam Sequence", vr: VR::SQ },
            E { tag: 0x02491060, alias: "Machine Reference Sequence", vr: VR::SQ },
            E { tag: 0x02491062, alias: "Machine Name", vr: VR::SH },
            E { tag: 0x02491064, alias: "Beam Name", vr: VR::SH },
            E { tag: 0x02491066, alias: "Particle Type", vr: VR::CS },
            E { tag: 0x02491068, alias: "Nominal Particle Energy", vr: VR::DS },
            E { tag: 0x02491070, alias: "Block Sequence", vr: VR::SQ },
            E { tag: 0x02491072, alias: "Block Name", vr: VR::LO },
            E { tag: 0x02491074, alias: "Block Shape", vr: VR::DS },
            E { tag: 0x02491080, alias: "Cutout Sequence", vr: VR::SQ },
            E { tag: 0x02491082, alias: "Cutout Name", vr: VR::LO },
            E { tag: 0x02491084, alias: "Cutout Shape", vr: VR::DS },
            E { tag: 0x02491090, alias: "Collimator Name", vr: VR::SH },
            E { tag: 0x02491092, alias: "Collimator Type", vr: VR::CS },
            E { tag: 0x024910A0, alias: "Dynamic Segment Sequence", vr: VR::SQ },
            E { tag: 0x024910A2, alias: "Table Angle", vr: VR::DS },
            E { tag: 0x024910A4, alias: "Gantry Angle", vr: VR::DS },
            E { tag: 0x024910A6, alias: "Collimator Angle", vr: VR::DS },
            E { tag: 0x024910A8, alias: "Isocenter Position", vr: VR::DS },
            E { tag: 0x024910A9, alias: "Source-Surface Distance", vr: VR::DS },
            E { tag: 0x024910AA, alias: "X Symmetric Opening", vr: VR::DS },
            E { tag: 0x024910AC, alias: "Y Symmetric Opening", vr: VR::DS },
            E { tag: 0x024910AE, alias: "X Positive Jaw Position", vr: VR::DS },
            E { tag: 0x024910B0, alias: "X Negative Jaw Position", vr: VR::DS },
            E { tag: 0x024910B2, alias: "Y Positive Jaw Position", vr: VR::DS },
            E { tag: 0x024910B4, alias: "Y Negative Jaw Position", vr: VR::DS },
            E { tag: 0x024910B6, alias: "Leaf Positions", vr: VR::DS },
            E { tag: 0x024910D0, alias: "Treatment Position", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_SENO_02",
        entries: &[
            E { tag: 0x00451004, alias: "AES", vr: VR::CS },
            E { tag: 0x00451006, alias: "Stereo angle", vr: VR::DS },
            E { tag: 0x00451009, alias: "Real Magnification Factor", vr: VR::DS },
            E { tag: 0x0045100B, alias: "Senograph Type", vr: VR::CS },
            E { tag: 0x0045100C, alias: "Integration Time", vr: VR::DS },
            E { tag: 0x0045100D, alias: "ROI Origin X and Y", vr: VR::DS },
            E { tag: 0x00451011, alias: "Receptor Size cm X and Y", vr: VR::DS },
            E { tag: 0x00451012, alias: "Receptor Size Pixels X and Y", vr: VR::IS },
            E { tag: 0x00451013, alias: "Screen", vr: VR::ST },
            E { tag: 0x00451014, alias: "Pixel Pitch Microns", vr: VR::DS },
            E { tag: 0x00451015, alias: "Pixel Depth Bits", vr: VR::IS },
            E { tag: 0x00451016, alias: "Binning Factor X and Y", vr: VR::IS },
            E { tag: 0x0045101B, alias: "Clinical View", vr: VR::LO },
            E { tag: 0x0045101D, alias: "Mean Of Raw Gray Levels", vr: VR::DS },
            E { tag: 0x0045101E, alias: "Mean Of Offset Gray Levels", vr: VR::DS },
            E { tag: 0x0045101F, alias: "Mean Of Corrected Gray Levels", vr: VR::DS },
            E { tag: 0x00451020, alias: "Estimated Anat mean", vr: VR::DS },
            E { tag: 0x00451021, alias: "Mean Of Log Region Gray Levels", vr: VR::DS },
            E { tag: 0x00451022, alias: "Standard Deviation Of Raw Gray Levels", vr: VR::DS },
            E { tag: 0x00451023, alias: "Standard Deviation Of Corrected Gray Levels", vr: VR::DS },
            E { tag: 0x00451024, alias: "Standard Deviation Of Region Gray Levels", vr: VR::DS },
            E { tag: 0x00451025, alias: "Standard Deviation Of Log Region Gray Levels", vr: VR::DS },
            E { tag: 0x00451026, alias: "MAO Buffer", vr: VR::OB },
            E { tag: 0x00451027, alias: "Set Number", vr: VR::IS },
            E { tag: 0x00451028, alias: "WindowingType (LINEAR or GAMMA)", vr: VR::CS },
            E { tag: 0x00451029, alias: "Windowing parameters", vr: VR::DS },
            E { tag: 0x0045102A, alias: "2DLocX", vr: VR::IS },
            E { tag: 0x0045102B, alias: "2DLocY", vr: VR::IS },
            E { tag: 0x00451039, alias: "Vignette Rows", vr: VR::US },
            E { tag: 0x0045103A, alias: "Vignette Columns", vr: VR::US },
            E { tag: 0x0045103B, alias: "Vignette Bits Allocated", vr: VR::US },
            E { tag: 0x0045103C, alias: "Vignette Bits Stored", vr: VR::US },
            E { tag: 0x0045103D, alias: "Vignette High Bit", vr: VR::US },
            E { tag: 0x0045103E, alias: "Vignette Pixel Representation", vr: VR::US },
            E { tag: 0x0045103F, alias: "Vignette Pixel Data", vr: VR::OB },
            E { tag: 0x00451049, alias: "Radiological Thickness", vr: VR::DS },
            E { tag: 0x00451050, alias: "SC Instance UID", vr: VR::UI },
            E { tag: 0x00451051, alias: "SC series UID", vr: VR::UI },
            E { tag: 0x00451052, alias: "Raw Diagnostic Low", vr: VR::IS },
            E { tag: 0x00451053, alias: "Raw Diagnostic High", vr: VR::IS },
            E { tag: 0x00451054, alias: "Exponent", vr: VR::DS },
            E { tag: 0x00451055, alias: "A Coefficients", vr: VR::IS },
            E { tag: 0x00451056, alias: "Noise Reduction Sensitivity", vr: VR::DS },
            E { tag: 0x00451057, alias: "Noise Reduction Threshold", vr: VR::DS },
            E { tag: 0x00451058, alias: "mu", vr: VR::DS },
            E { tag: 0x00451059, alias: "Threshold", vr: VR::IS },
            E { tag: 0x00451060, alias: "Breast ROI X", vr: VR::IS },
            E { tag: 0x00451061, alias: "Breast ROI Y", vr: VR::IS },
            E { tag: 0x00451062, alias: "User Window Center", vr: VR::IS },
            E { tag: 0x00451063, alias: "User Window Width", vr: VR::IS },
            E { tag: 0x00451064, alias: "Segm Threshold", vr: VR::IS },
            E { tag: 0x00451065, alias: "Detector Entrance Dose", vr: VR::IS },
            E { tag: 0x00451066, alias: "Asymmetrical Collimation Information", vr: VR::IS },
            E { tag: 0x00451071, alias: "STX buffer", vr: VR::OB },
            E { tag: 0x00451072, alias: "Image Crop point", vr: VR::DS },
        ],
    },
    P {
        creator: "GEMS_SERS_01",
        entries: &[
            E { tag: 0x00251010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00251006, alias: "Last pulse sequence used", vr: VR::SS },
            E { tag: 0x00251007, alias: "Images in Series", vr: VR::SL },
            E { tag: 0x00251010, alias: "Landmark Counter", vr: VR::SL },
            E { tag: 0x00251011, alias: "Number of Acquisitions", vr: VR::SS },
            E { tag: 0x00251014, alias: "Indicates no. of updates to header", vr: VR::SL },
            E { tag: 0x00251017, alias: "Series Complete Flag", vr: VR::SL },
            E { tag: 0x00251018, alias: "Number of images archived", vr: VR::SL },
            E { tag: 0x00251019, alias: "Last image number used", vr: VR::SL },
            E { tag: 0x0025101A, alias: "Primary Receiver Suite and Host", vr: VR::SH },
            E { tag: 0x0025101B, alias: "Protocol Data Block (compressed)", vr: VR::OB },
        ],
    },
    P {
        creator: "GEMS_STDY_01",
        entries: &[
            E { tag: 0x00231010, alias: "Private Element Creator", vr: VR::LO },
            E { tag: 0x00231001, alias: "Number of series in Study", vr: VR::SL },
            E { tag: 0x00231002, alias: "Number of unarchived Series", vr: VR::SL },
            E { tag: 0x00231010, alias: "Reference image field", vr: VR::SS },
            E { tag: 0x00231050, alias: "Summary image", vr: VR::SS },
            E { tag: 0x00231070, alias: "Start time(secs) in first axial", vr: VR::FD },
            E { tag: 0x00231074, alias: "No. of updates to header", vr: VR::SL },
            E { tag: 0x0023107D, alias: "Indicates study has complete info (DICOM/genesis)", vr: VR::SS },
        ],
    },
    P {
        creator: "GEMS_Ultrasound_ImageGroup_001",
        entries: &[
            E { tag: 0x60031010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x60031011, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "GEMS_Ultrasound_MovieGroup_001",
        entries: &[
            E { tag: 0x7FE11001, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7FE11003, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FE11008, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11012, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7FE11018, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11020, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11024, alias: "Unknown", vr: VR::SH },
            E { tag: 0x7FE11026, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11036, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11037, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FE11043, alias: "Unknown", vr: VR::OB },
            E { tag: 0x7FE11048, alias: "Unknown", vr: VR::FD },
            E { tag: 0x7FE11049, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FE11051, alias: "Unknown", vr: VR::FL },
            E { tag: 0x7FE11052, alias: "Unknown", vr: VR::FD },
            E { tag: 0x7FE11053, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FE11054, alias: "Unknown", vr: VR::SL },
            E { tag: 0x7FE11055, alias: "Unknown", vr: VR::OB },
            E { tag: 0x7FE11057, alias: "Unknown", vr: VR::LT },
            E { tag: 0x7FE11060, alias: "Unknown", vr: VR::OB },
            E { tag: 0x7FE11061, alias: "Unknown", vr: VR::OW },
            E { tag: 0x7FE11070, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11071, alias: "Unknown", vr: VR::UL },
            E { tag: 0x7FE11072, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7FE11073, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11074, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7FE11075, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11077, alias: "Unknown", vr: VR::FD },
            E { tag: 0x7FE11079, alias: "Unknown", vr: VR::SL },
            E { tag: 0x7FE11083, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11084, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7FE11085, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7FE11086, alias: "Unknown", vr: VR::SL },
            E { tag: 0x7FE11087, alias: "Unknown", vr: VR::FD },
            E { tag: 0x7FE11088, alias: "Unknown", vr: VR::FD },
        ],
    },
    P {
        creator: "GEMS_VXTLSTATE_001",
        entries: &[
            E { tag: 0x00571005, alias: "Volumes_info", vr: VR::SQ },
            E { tag: 0x00571006, alias: "Image_UIDs", vr: VR::SQ },
            E { tag: 0x00571007, alias: "SUV_ScanTime", vr: VR::DT },
            E { tag: 0x00571008, alias: "SUV_AdministredTime", vr: VR::DT },
            E { tag: 0x00571009, alias: "SUV_MeasuredTime", vr: VR::DT },
            E { tag: 0x00571010, alias: "SUV_PostInjectedTime", vr: VR::DT },
            E { tag: 0x00571011, alias: "SUV_TracerActivity", vr: VR::FL },
            E { tag: 0x00571012, alias: "SUV_PostInjectedActivity", vr: VR::FL },
            E { tag: 0x00571013, alias: "SUV_HalfLife", vr: VR::FL },
            E { tag: 0x00571014, alias: "SState_Version", vr: VR::LO },
            E { tag: 0x00571015, alias: "SegList_Seq", vr: VR::SQ },
            E { tag: 0x00571016, alias: "SegList_Count", vr: VR::IS },
            E { tag: 0x00571017, alias: "SegList_List", vr: VR::OB },
            E { tag: 0x00571018, alias: "SegList_Name", vr: VR::LO },
            E { tag: 0x00571019, alias: "Volume_Filename", vr: VR::LO },
            E { tag: 0x00571020, alias: "SegList_Label", vr: VR::LO },
            E { tag: 0x00571021, alias: "Bookmark_Seq", vr: VR::SQ },
            E { tag: 0x00571022, alias: "Bookmark", vr: VR::LT },
            E { tag: 0x00571023, alias: "Cursor_position", vr: VR::FL },
            E { tag: 0x00571024, alias: "Color_Value_Field", vr: VR::UL },
            E { tag: 0x00571025, alias: "Count", vr: VR::IS },
            E { tag: 0x00571026, alias: "Views_info", vr: VR::SQ },
            E { tag: 0x00571027, alias: "View_Slot", vr: VR::LT },
            E { tag: 0x00571028, alias: "Wireframe_Size", vr: VR::IS },
            E { tag: 0x00571029, alias: "Wireframe", vr: VR::UT },
            E { tag: 0x00571030, alias: "Annotation_Size", vr: VR::IS },
            E { tag: 0x00571031, alias: "Annotation", vr: VR::LT },
            E { tag: 0x00571032, alias: "Slider_State", vr: VR::IS },
            E { tag: 0x00571033, alias: "Proto_Name", vr: VR::LO },
            E { tag: 0x00571034, alias: "Proto_Title", vr: VR::LO },
            E { tag: 0x00571035, alias: "Proto_FilmName", vr: VR::LO },
            E { tag: 0x00571036, alias: "Proto_Step", vr: VR::LO },
            E { tag: 0x00571038, alias: "SegList_Slots", vr: VR::LT },
            E { tag: 0x00571039, alias: "VT_Points_Size", vr: VR::IS },
            E { tag: 0x00571040, alias: "VT_Points", vr: VR::LT },
            E { tag: 0x00571041, alias: "VT_Meas_Size", vr: VR::IS },
            E { tag: 0x00571042, alias: "VT_Meas", vr: VR::LT },
            E { tag: 0x00571043, alias: "VT_Tree_Size", vr: VR::IS },
            E { tag: 0x00571044, alias: "VT_Tree", vr: VR::UT },
            E { tag: 0x00571046, alias: "VT_Preset_Size", vr: VR::IS },
            E { tag: 0x00571047, alias: "VT_Preset", vr: VR::LT },
            E { tag: 0x00571048, alias: "VT_State_Size", vr: VR::IS },
            E { tag: 0x00571049, alias: "VT_State", vr: VR::LT },
            E { tag: 0x00571050, alias: "Preferences_Size", vr: VR::IS },
            E { tag: 0x00571051, alias: "Preferences", vr: VR::LT },
            E { tag: 0x00571052, alias: "Volume_ZComb_Filter", vr: VR::IS },
            E { tag: 0x00571053, alias: "HTML_Page_Size", vr: VR::IS },
            E { tag: 0x00571054, alias: "HTML_Page", vr: VR::LT },
            E { tag: 0x00571055, alias: "Camera_Position", vr: VR::FL },
            E { tag: 0x00571056, alias: "Slider_Size", vr: VR::IS },
            E { tag: 0x00571057, alias: "Slider", vr: VR::LT },
            E { tag: 0x00571058, alias: "SegList_Pretty_Name", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_VXTL_USERDATA_01",
        entries: &[
            E { tag: 0x00471011, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "GEMS_XELPRV_01",
        entries: &[
            E { tag: 0x00331008, alias: "Object Type", vr: VR::CS },
            E { tag: 0x00331010, alias: "Modified", vr: VR::SL },
            E { tag: 0x00331011, alias: "Name", vr: VR::LO },
            E { tag: 0x00331014, alias: "StudyId", vr: VR::LO },
            E { tag: 0x00331016, alias: "ProtocolDataUID", vr: VR::LO },
            E { tag: 0x00331017, alias: "Date", vr: VR::SH },
            E { tag: 0x00331018, alias: "Time", vr: VR::SH },
            E { tag: 0x00331019, alias: "ProtocoldataFlags", vr: VR::UL },
            E { tag: 0x0033101A, alias: "ProtocolName", vr: VR::LO },
            E { tag: 0x0033101B, alias: "StudyId", vr: VR::LO },
            E { tag: 0x0033101C, alias: "BulkData", vr: VR::OB },
            E { tag: 0x0033101D, alias: "IntData", vr: VR::SL },
            E { tag: 0x0033101E, alias: "DoubleData", vr: VR::FD },
            E { tag: 0x0033101F, alias: "StringData", vr: VR::OB },
            E { tag: 0x00331020, alias: "BulkDataFormat", vr: VR::OB },
            E { tag: 0x00331021, alias: "IntDataFormat", vr: VR::OB },
            E { tag: 0x00331022, alias: "DoubleDataFormat", vr: VR::OB },
            E { tag: 0x00331023, alias: "StringDataFormat", vr: VR::OB },
            E { tag: 0x00331024, alias: "Description", vr: VR::LT },
            E { tag: 0x00331028, alias: "RTName", vr: VR::LO },
            E { tag: 0x00331029, alias: "RTSpecification", vr: VR::LT },
            E { tag: 0x0033102A, alias: "ReviewTemplatesFlags", vr: VR::UL },
            E { tag: 0x0033102B, alias: "DataValidationSpec", vr: VR::LT },
            E { tag: 0x0033102C, alias: "Description", vr: VR::LT },
            E { tag: 0x0033102D, alias: "IconDescription", vr: VR::LT },
            E { tag: 0x00331050, alias: "ProtocoldataSQ", vr: VR::SQ },
            E { tag: 0x00331051, alias: "PDOSOPClassUID", vr: VR::UI },
            E { tag: 0x00331052, alias: "PDOSOPInstanceUID", vr: VR::UI },
            E { tag: 0x00331060, alias: "ReviewtemplateSQ", vr: VR::SQ },
            E { tag: 0x00331061, alias: "RTOSOPClassUID", vr: VR::UI },
            E { tag: 0x00331062, alias: "RTOSOPInstanceUID", vr: VR::UI },
            E { tag: 0x00331070, alias: "SeriesdataSQ", vr: VR::SQ },
            E { tag: 0x00331071, alias: "SDOSOPClassUID", vr: VR::UI },
            E { tag: 0x00331072, alias: "SDOSOPInstanceUID", vr: VR::UI },
            E { tag: 0x00331073, alias: "SDODoubleDataSQ", vr: VR::SQ },
            E { tag: 0x00571001, alias: "ROISQ", vr: VR::SQ },
            E { tag: 0x00571002, alias: "SOPClassUID", vr: VR::UI },
            E { tag: 0x00571003, alias: "SOPInstanceUID", vr: VR::UI },
            E { tag: 0x00571010, alias: "ROINumber", vr: VR::IS },
            E { tag: 0x00571011, alias: "Dimensions", vr: VR::UI },
            E { tag: 0x00571012, alias: "Points", vr: VR::UI },
            E { tag: 0x00571013, alias: "Type", vr: VR::UI },
            E { tag: 0x00571014, alias: "Description", vr: VR::UI },
            E { tag: 0x00571015, alias: "Data value Representation", vr: VR::UI },
            E { tag: 0x00571016, alias: "Label", vr: VR::UI },
            E { tag: 0x00571017, alias: "Data", vr: VR::UI },
            E { tag: 0x00571041, alias: "Modified", vr: VR::SL },
            E { tag: 0x00571042, alias: "Name", vr: VR::LO },
            E { tag: 0x00571043, alias: "Rid", vr: VR::SL },
            E { tag: 0x00571045, alias: "SOPClassUID", vr: VR::LO },
            E { tag: 0x00571046, alias: "SOPInstanceUID", vr: VR::LO },
            E { tag: 0x00571047, alias: "Normal Colour", vr: VR::LO },
            E { tag: 0x00571048, alias: "NameFont", vr: VR::LT },
            E { tag: 0x00571049, alias: "FillPattern", vr: VR::SL },
            E { tag: 0x0057104A, alias: "LineStyle", vr: VR::SL },
            E { tag: 0x0057104B, alias: "LineDashLength", vr: VR::SL },
            E { tag: 0x0057104C, alias: "LineThickness", vr: VR::SL },
            E { tag: 0x0057104D, alias: "Interactivity", vr: VR::SL },
            E { tag: 0x0057104E, alias: "NamePos", vr: VR::SL },
            E { tag: 0x0057104F, alias: "NameDisplay", vr: VR::SL },
            E { tag: 0x00571050, alias: "Label", vr: VR::LO },
            E { tag: 0x00571051, alias: "BpSeg", vr: VR::SL },
            E { tag: 0x00571052, alias: "BpSegpairs", vr: VR::US },
            E { tag: 0x00571053, alias: "SeedSpace", vr: VR::SL },
            E { tag: 0x00571054, alias: "Seeds", vr: VR::FD },
            E { tag: 0x00571055, alias: "Shape", vr: VR::SL },
            E { tag: 0x00571056, alias: "ShapeTilt", vr: VR::FD },
            E { tag: 0x00571059, alias: "ShapePtsSpace", vr: VR::SL },
            E { tag: 0x0057105A, alias: "ShapeCtrlPtsCount", vr: VR::SL },
            E { tag: 0x0057105B, alias: "Shap CtrlPts", vr: VR::FD },
            E { tag: 0x0057105C, alias: "ShapeCPSpace", vr: VR::SL },
            E { tag: 0x0057105D, alias: "ROIFlags", vr: VR::UL },
            E { tag: 0x0057105E, alias: "FrameNumber", vr: VR::UL },
            E { tag: 0x00571060, alias: "DatasetROIMapping", vr: VR::LO },
        ],
    },
    P {
        creator: "GEMS_XR3DCAL_01",
        entries: &[
            E { tag: 0x00211001, alias: "3Dcal image rows", vr: VR::IS },
            E { tag: 0x00211002, alias: "3Dcal image columns", vr: VR::IS },
            E { tag: 0x00211003, alias: "3Dcal field of view", vr: VR::FL },
            E { tag: 0x00211004, alias: "3Dcal acquisition date", vr: VR::DA },
            E { tag: 0x00211005, alias: "3Dcal acquisition time", vr: VR::TM },
            E { tag: 0x00211006, alias: "3Dcal calibration processing date", vr: VR::DA },
            E { tag: 0x00211007, alias: "3Dcal calibration processing time", vr: VR::TM },
            E { tag: 0x00211008, alias: "3Dcal L arm angle", vr: VR::FL },
            E { tag: 0x00211009, alias: "3Dcal Pivot angle vector", vr: VR::FL },
            E { tag: 0x0021100A, alias: "3Dcal C arm angle", vr: VR::FL },
            E { tag: 0x0021100B, alias: "3Dcal matrix sequence", vr: VR::SQ },
            E { tag: 0x0021100C, alias: "3Dcal matrix elements", vr: VR::LO },
            E { tag: 0x0021100D, alias: "3Dcal algorithm version", vr: VR::LO },
            E { tag: 0x0021100E, alias: "3Dcal 3D frame unit size", vr: VR::FL },
            E { tag: 0x0021100F, alias: "3Dcal calibration mode", vr: VR::LO },
            E { tag: 0x00211010, alias: "3Dcal image frame origin row", vr: VR::FL },
            E { tag: 0x00211011, alias: "3Dcal image frame origin column", vr: VR::FL },
            E { tag: 0x00211012, alias: "3Dcal positioner pivot rotation speed", vr: VR::IS },
            E { tag: 0x00211013, alias: "3Dcal number of images", vr: VR::IS },
            E { tag: 0x00211014, alias: "3Dcal Instance UID", vr: VR::UI },
            E { tag: 0x00211015, alias: "3Dcal image pixel spacing", vr: VR::FL },
            E { tag: 0x00211016, alias: "3Dcal centering mode", vr: VR::CS },
        ],
    },
    P {
        creator: "GEMS_YMHD_01",
        entries: &[
            E { tag: 0x00331005, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00331006, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "GE_GENESIS_REV3.0",
        entries: &[
            E { tag: 0x00191039, alias: "Axial Type", vr: VR::SS },
            E { tag: 0x0019108F, alias: "Swap Phase / Frequency Axis", vr: VR::SS },
            E { tag: 0x0019109C, alias: "Pulse Sequence Name", vr: VR::SS },
            E { tag: 0x0019109F, alias: "Coil Type", vr: VR::SS },
            E { tag: 0x001910A4, alias: "SAT fat/water/none", vr: VR::SS },
            E { tag: 0x001910C0, alias: "Bitmap of SAT Selections", vr: VR::SS },
            E { tag: 0x001910C1, alias: "Surfacel Coil Iintensity Correction Flag", vr: VR::SS },
            E { tag: 0x001910CB, alias: "Phase Contrast Flow Axis", vr: VR::SS },
            E { tag: 0x001910CC, alias: "Phase Contrast Velocity Encoding", vr: VR::SS },
            E { tag: 0x001910D5, alias: "Fractional Echo", vr: VR::SS },
            E { tag: 0x001910D8, alias: "Variable Echo Flag", vr: VR::SS },
            E { tag: 0x001910D9, alias: "Concatenated Sat", vr: VR::DS },
            E { tag: 0x001910F2, alias: "Number of Phases", vr: VR::SS },
            E { tag: 0x0043101E, alias: "Delta Start Time", vr: VR::DS },
            E { tag: 0x00431027, alias: "Pitch Ratio", vr: VR::SH },
        ],
    },
    P {
        creator: "GE_GROUP",
        entries: &[
            E { tag: 0x60051010, alias: "Unknown", vr: VR::UT },
        ],
    },
    P {
        creator: "HMC - CT - ID",
        entries: &[
            E { tag: 0x00091000, alias: "Image ID Information (e.g. Patient Name, Patient ID)", vr: VR::UN },
            E { tag: 0x00091001, alias: "Image ID Information (e.g. Patient Comment)", vr: VR::UN },
        ],
    },
    P {
        creator: "HOLOGIC",
        entries: &[
            E { tag: 0x00111000, alias: "Hx Questionnaire", vr: VR::LO },
            E { tag: 0x00211001, alias: "Image Analysis Data in XML", vr: VR::LT },
            E { tag: 0x00231000, alias: "Encoding Scheme Version", vr: VR::LO },
            E { tag: 0x00231001, alias: "P File Name", vr: VR::LO },
            E { tag: 0x00231002, alias: "P File Data", vr: VR::OB },
            E { tag: 0x00231003, alias: "P File Length", vr: VR::UL },
            E { tag: 0x00231004, alias: "R File Data", vr: VR::OB },
            E { tag: 0x00231005, alias: "R File Length", vr: VR::UL },
        ],
    },
    P {
        creator: "HOLOGIC, Inc.",
        entries: &[
            E { tag: 0x00191006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191007, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191016, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191027, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00191028, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191029, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191030, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191033, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191034, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191035, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191037, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191041, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191042, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191043, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191044, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191045, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191046, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191085, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191090, alias: "Unknown", vr: VR::DS },
            E { tag: 0x7E011001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7E011002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7E011010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7E011011, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7E011012, alias: "Unknown", vr: VR::OB },
            E { tag: 0x7F011001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x7F011002, alias: "Unknown", vr: VR::SH },
            E { tag: 0x7F011010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7F011012, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "Hologic",
        entries: &[
            E { tag: 0x00111000, alias: "Hx Questionnaire", vr: VR::LO },
            E { tag: 0x00211001, alias: "Image Analysis Data in XML", vr: VR::LT },
            E { tag: 0x00231000, alias: "Encoding Scheme Version", vr: VR::LO },
            E { tag: 0x00231001, alias: "P File Name", vr: VR::LO },
            E { tag: 0x00231002, alias: "P File Data", vr: VR::OB },
            E { tag: 0x00231003, alias: "P File Length", vr: VR::UL },
            E { tag: 0x00231004, alias: "R File Data", vr: VR::OB },
            E { tag: 0x00231005, alias: "R File Length", vr: VR::UL },
        ],
    },
    P {
        creator: "IDEXX",
        entries: &[
            E { tag: 0x00111000, alias: "Breed Name", vr: VR::LO },
            E { tag: 0x00111001, alias: "Species Name", vr: VR::LO },
            E { tag: 0x00111002, alias: "Owner", vr: VR::PN },
        ],
    },
    P {
        creator: "IMPAC",
        entries: &[
            E { tag: 0x300B1002, alias: "Unknown", vr: VR::FL },
            E { tag: 0x300B1004, alias: "Unknown", vr: VR::FL },
            E { tag: 0x300B100E, alias: "Unknown", vr: VR::FL },
        ],
    },
    P {
        creator: "IMS s.r.l. Biopsy Private Code",
        entries: &[
            E { tag: 0x12691001, alias: "Biopsy Image", vr: VR::IS },
            E { tag: 0x12691010, alias: "Biopsy Markers X", vr: VR::IS },
            E { tag: 0x12691011, alias: "Biopsy Markers Y", vr: VR::IS },
            E { tag: 0x12691012, alias: "Biopsy Markers Number", vr: VR::IS },
            E { tag: 0x12691020, alias: "Biopsy Area Left Border", vr: VR::IS },
            E { tag: 0x12691021, alias: "Biopsy Area Right Border", vr: VR::IS },
            E { tag: 0x12691022, alias: "Biopsy Area Top Border", vr: VR::IS },
            E { tag: 0x12691023, alias: "Biopsy Area Bottom Border", vr: VR::IS },
        ],
    },
    P {
        creator: "IMS s.r.l. Mammography Private Code",
        entries: &[
            E { tag: 0x12711001, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711002, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711010, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711011, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711012, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711013, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711020, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711021, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711022, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711038, alias: "Unknown", vr: VR::LO },
            E { tag: 0x12711039, alias: "Unknown", vr: VR::US },
            E { tag: 0x12711040, alias: "Unknown", vr: VR::US },
            E { tag: 0x12711041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x12711042, alias: "Unknown", vr: VR::LT },
            E { tag: 0x12711045, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711046, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711047, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711048, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711049, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711050, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711051, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711052, alias: "Unknown", vr: VR::IS },
            E { tag: 0x12711053, alias: "Unknown", vr: VR::LO },
            E { tag: 0x12711054, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "INFINITT_FMX",
        entries: &[
            E { tag: 0x00151010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00151011, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "INTEGRIS 1.0",
        entries: &[
            E { tag: 0x00091008, alias: "Exposure Channel", vr: VR::CS },
            E { tag: 0x00091032, alias: "Exposure Start Time", vr: VR::TM },
            E { tag: 0x00191000, alias: "APR Name", vr: VR::LO },
            E { tag: 0x00191040, alias: "Frame Rate", vr: VR::DS },
            E { tag: 0x00211012, alias: "Exposure Number", vr: VR::IS },
            E { tag: 0x00291000, alias: "Edge Enhancement Sequence", vr: VR::SQ },
            E { tag: 0x00291001, alias: "Convolution Kernel Size", vr: VR::US },
            E { tag: 0x00291002, alias: "Convolution Kernel Coefficients", vr: VR::US },
            E { tag: 0x00291003, alias: "Edge Enhancement Gain", vr: VR::FL },
            E { tag: 0x00291008, alias: "Number of Exposure Results", vr: VR::IS },
            E { tag: 0x00411020, alias: "Accumulated Fluoroscopy Dose", vr: VR::DS },
            E { tag: 0x00411030, alias: "Accumulated Exposure Dose", vr: VR::DS },
            E { tag: 0x00411040, alias: "Total Dose", vr: VR::DS },
            E { tag: 0x00411041, alias: "Total Number of Frames", vr: VR::US },
            E { tag: 0x00411050, alias: "Exposure Information Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "INTELERAD MEDICAL SYSTEMS",
        entries: &[
            E { tag: 0x3F011001, alias: "Institution Code", vr: VR::LO },
            E { tag: 0x3F011002, alias: "Routed Transfer AE", vr: VR::LO },
            E { tag: 0x3F011003, alias: "Source AE", vr: VR::LO },
            E { tag: 0x3F011004, alias: "Deferred Validation", vr: VR::LO },
            E { tag: 0x3F011005, alias: "Series Owner", vr: VR::LO },
        ],
    },
    P {
        creator: "ISG Image",
        entries: &[
            E { tag: 0x00651011, alias: "Unknown", vr: VR::UL },
        ],
    },
    P {
        creator: "ISG shadow",
        entries: &[
            E { tag: 0x00291070, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291080, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "ISI",
        entries: &[
            E { tag: 0x00091001, alias: "SIENET General Purpose IMGF", vr: VR::SQ },
        ],
    },
    P {
        creator: "Image (ID, Version, Size, Dump, GUID)",
        entries: &[
            E { tag: 0x80031000, alias: "ID", vr: VR::LO },
            E { tag: 0x80031010, alias: "Version", vr: VR::LO },
            E { tag: 0x80031020, alias: "Size", vr: VR::UL },
            E { tag: 0x80031030, alias: "Dump", vr: VR::OB },
            E { tag: 0x80031040, alias: "GUID", vr: VR::LO },
        ],
    },
    P {
        creator: "Imaging Dynamics Company Ltd.",
        entries: &[
            E { tag: 0x43211005, alias: "Custom Tag", vr: VR::UN },
            E { tag: 0x43211010, alias: "Custom Tag", vr: VR::UN },
            E { tag: 0x43211041, alias: "Unknown", vr: VR::CS },
            E { tag: 0x43211042, alias: "Unknown", vr: VR::US },
            E { tag: 0x43211050, alias: "Custom Tag", vr: VR::DS },
            E { tag: 0x43211051, alias: "Custom Tag", vr: VR::DS },
            E { tag: 0x43211064, alias: "Custom Tag", vr: VR::LO },
        ],
    },
    P {
        creator: "KINETDX",
        entries: &[
            E { tag: 0x002110A5, alias: "Unknown", vr: VR::US },
            E { tag: 0x002110A8, alias: "Unknown", vr: VR::LO },
            E { tag: 0x002110AA, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "KINETDX_GRAPHICS",
        entries: &[
            E { tag: 0x002110A4, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "KRETZ_US",
        entries: &[
            E { tag: 0x7FE11001, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "Kodak Image Information",
        entries: &[
            E { tag: 0x00291015, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291016, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291017, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291018, alias: "Unknown", vr: VR::UT },
            E { tag: 0x00291019, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0029101A, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "LODOX_STATSCAN",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191003, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191004, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191005, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191006, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191007, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "LORAD Selenia",
        entries: &[
            E { tag: 0x00191006, alias: "Paddle ID", vr: VR::OB },
            E { tag: 0x00191007, alias: "Paddle Position", vr: VR::OB },
            E { tag: 0x00191008, alias: "Collimation Size", vr: VR::OB },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191027, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191028, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191029, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191030, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191032, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191033, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191034, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191035, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191041, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191051, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191052, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191053, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00191071, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "MAROTECH Inc.",
        entries: &[
            E { tag: 0x00371001, alias: "Some kind of version number", vr: VR::LO },
            E { tag: 0x00371021, alias: "Icon Row", vr: VR::US },
            E { tag: 0x00371022, alias: "Icon Col", vr: VR::US },
            E { tag: 0x00371023, alias: "JPEG Compressed Icon", vr: VR::OB },
        ],
    },
    P {
        creator: "MDS NORDION OTP ANATOMY MODELLING",
        entries: &[
            E { tag: 0x30051000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x30051002, alias: "Unknown", vr: VR::CS },
            E { tag: 0x30051004, alias: "Unknown", vr: VR::DS },
            E { tag: 0x30051006, alias: "Unknown", vr: VR::DS },
            E { tag: 0x30051008, alias: "Unknown", vr: VR::DS },
            E { tag: 0x3005100A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x3005100C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x3005100E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x30051010, alias: "Unknown", vr: VR::DS },
            E { tag: 0x30051012, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "MEDIFACE",
        entries: &[
            E { tag: 0x00211001, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211010, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211011, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211012, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211021, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211022, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211030, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211031, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211040, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211041, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211042, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211050, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211051, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211052, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211060, alias: "Unknown", vr: VR::UN },
            E { tag: 0x002110F1, alias: "Unknown", vr: VR::UN },
            E { tag: 0x002110F2, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291010, alias: "Window Center", vr: VR::DS },
            E { tag: 0x00291011, alias: "Window Width", vr: VR::DS },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291022, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "MERGE TECHNOLOGIES, INC.",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "MITRA LINKED ATTRIBUTES 1.0",
        entries: &[
            E { tag: 0x00311020, alias: "Global Patient ID", vr: VR::LO },
        ],
    },
    P {
        creator: "MITRA OBJECT ATTRIBUTES 1.0",
        entries: &[
            E { tag: 0x00331002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00331004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00331006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00331008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0033100A, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "MITRA OBJECT DOCUMENT 1.0",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "MITRA OBJECT UTF8 ATTRIBUTES 1.0",
        entries: &[
            E { tag: 0x00331002, alias: "Patient's Name", vr: VR::PN },
            E { tag: 0x00331004, alias: "Study Description", vr: VR::CS },
            E { tag: 0x00331006, alias: "Referring Physician's Name", vr: VR::PN },
            E { tag: 0x00331008, alias: "Requesting Physician's Name", vr: VR::PN },
            E { tag: 0x0033100A, alias: "Performing Physician's Name", vr: VR::PN },
            E { tag: 0x0033100C, alias: "Reason for Study", vr: VR::LO },
            E { tag: 0x0033100E, alias: "Study Comments", vr: VR::LT },
            E { tag: 0x00331013, alias: "Patient's Name Single Byte", vr: VR::PN },
            E { tag: 0x00331014, alias: "Patient's Name Ideographic", vr: VR::PN },
            E { tag: 0x00331015, alias: "Patient's Name Phonetic", vr: VR::PN },
            E { tag: 0x00331016, alias: "Referring Physician's Name Single Byte", vr: VR::PN },
            E { tag: 0x00331017, alias: "Referring Physician's Name Ideographic", vr: VR::PN },
            E { tag: 0x00331018, alias: "Referring Physician's Name Phonetic", vr: VR::PN },
            E { tag: 0x00331019, alias: "Requesting Physician's Name Single Byte", vr: VR::PN },
            E { tag: 0x0033101A, alias: "Requesting Physician's Name Ideographic", vr: VR::PN },
            E { tag: 0x0033101B, alias: "Requesting Physician's Name Phonetic", vr: VR::PN },
            E { tag: 0x0033101C, alias: "Performing Physician's Name Single Byte", vr: VR::PN },
            E { tag: 0x0033101D, alias: "Performing Physician's Name Ideographic", vr: VR::PN },
            E { tag: 0x0033101E, alias: "Performing Physician's Name Phonetic", vr: VR::PN },
        ],
    },
    P {
        creator: "MITRA PRESENTATION 1.0",
        entries: &[
            E { tag: 0x00291000, alias: "Mitra Rotation", vr: VR::CS },
            E { tag: 0x00291001, alias: "Mitra Window Width", vr: VR::LO },
            E { tag: 0x00291002, alias: "Mitra Window Centre", vr: VR::LO },
            E { tag: 0x00291003, alias: "Mitra Invert", vr: VR::IS },
            E { tag: 0x00291004, alias: "Mitra Has Tabstop", vr: VR::IS },
            E { tag: 0x00291005, alias: "Smooth Rotation", vr: VR::CS },
            E { tag: 0x00291010, alias: "Breast Box x0", vr: VR::CS },
            E { tag: 0x00291011, alias: "Breast Box y0", vr: VR::CS },
            E { tag: 0x00291012, alias: "Breast Box x1", vr: VR::CS },
            E { tag: 0x00291013, alias: "Breast Box y1", vr: VR::CS },
        ],
    },
    P {
        creator: "MMCPrivate",
        entries: &[
            E { tag: 0x00091000, alias: "Private Creator", vr: VR::LO },
            E { tag: 0x00091001, alias: "Technologist", vr: VR::LO },
            E { tag: 0x00091002, alias: "ScheduledStudyDateTime", vr: VR::LO },
            E { tag: 0x00091003, alias: "StudyAppData", vr: VR::OB },
            E { tag: 0x00091004, alias: "ProtocolObjectID", vr: VR::UI },
            E { tag: 0x00091005, alias: "Name", vr: VR::LO },
            E { tag: 0x00091006, alias: "Frequency", vr: VR::IS },
            E { tag: 0x00091007, alias: "UpdateFlag", vr: VR::SH },
            E { tag: 0x00091008, alias: "Directory", vr: VR::SH },
            E { tag: 0x00091009, alias: "Comments", vr: VR::LO },
            E { tag: 0x0009100A, alias: "Region", vr: VR::LO },
            E { tag: 0x0009100B, alias: "Laterality", vr: VR::SH },
            E { tag: 0x0009100C, alias: "TotalScanTime", vr: VR::TM },
            E { tag: 0x0009100D, alias: "ContrastMedium", vr: VR::LO },
            E { tag: 0x0009100E, alias: "CreateDateTime", vr: VR::LO },
            E { tag: 0x0009100F, alias: "Creator", vr: VR::LO },
            E { tag: 0x00091010, alias: "SiteName", vr: VR::LO },
            E { tag: 0x00091011, alias: "ReferringPhysician", vr: VR::LO },
            E { tag: 0x00091012, alias: "Radiologist", vr: VR::LO },
            E { tag: 0x00091013, alias: "Technologist", vr: VR::LO },
            E { tag: 0x00091014, alias: "ProtocolUid", vr: VR::UI },
            E { tag: 0x00091015, alias: "IsInLibrary", vr: VR::SH },
            E { tag: 0x00091016, alias: "Gating", vr: VR::LO },
            E { tag: 0x00091017, alias: "Note", vr: VR::ST },
            E { tag: 0x00091018, alias: "NumberOfTasks", vr: VR::IS },
            E { tag: 0x00091019, alias: "IsFlagRaised", vr: VR::SH },
            E { tag: 0x0009101A, alias: "IsArchived", vr: VR::SH },
            E { tag: 0x0009101B, alias: "IsDefault", vr: VR::SH },
            E { tag: 0x0009101C, alias: "ProtocolAppData", vr: VR::OB },
            E { tag: 0x0009101D, alias: "IsAllowCascadeSave", vr: VR::SH },
            E { tag: 0x0009101E, alias: "IsAllowCascadeProtect", vr: VR::SH },
            E { tag: 0x0009101F, alias: "TaskInfo", vr: VR::SQ },
            E { tag: 0x00091020, alias: "TaskInfoObjectID", vr: VR::UI },
            E { tag: 0x00091021, alias: "Name", vr: VR::LO },
            E { tag: 0x00091022, alias: "TaskStatus", vr: VR::SH },
            E { tag: 0x00091023, alias: "TaskPriority", vr: VR::SH },
            E { tag: 0x00091024, alias: "Leaf", vr: VR::SH },
            E { tag: 0x00091025, alias: "TaskID", vr: VR::LO },
            E { tag: 0x00091026, alias: "Frequency", vr: VR::IS },
            E { tag: 0x00091027, alias: "UpdateFlag", vr: VR::SH },
            E { tag: 0x00091028, alias: "Directory", vr: VR::SH },
            E { tag: 0x00091029, alias: "Comments", vr: VR::LO },
            E { tag: 0x0009102A, alias: "Category", vr: VR::SH },
            E { tag: 0x0009102B, alias: "Region", vr: VR::LO },
            E { tag: 0x0009102C, alias: "Laterality", vr: VR::SH },
            E { tag: 0x0009102D, alias: "ScanTime", vr: VR::TM },
            E { tag: 0x0009102E, alias: "ContrastMedium", vr: VR::LO },
            E { tag: 0x0009102F, alias: "CreateDateTime", vr: VR::LO },
            E { tag: 0x00091030, alias: "Creator", vr: VR::LO },
            E { tag: 0x00091031, alias: "SiteName", vr: VR::LO },
            E { tag: 0x00091032, alias: "ReferringPhysician", vr: VR::LO },
            E { tag: 0x00091033, alias: "Radiologist", vr: VR::LO },
            E { tag: 0x00091034, alias: "Technologist", vr: VR::LO },
            E { tag: 0x00091035, alias: "TaskUid", vr: VR::UI },
            E { tag: 0x00091036, alias: "TaskInfoUid", vr: VR::UI },
            E { tag: 0x00091037, alias: "IsInLibrary", vr: VR::SH },
            E { tag: 0x00091038, alias: "TaskOrder", vr: VR::IS },
            E { tag: 0x00091039, alias: "Gating", vr: VR::LO },
            E { tag: 0x0009103A, alias: "Plane", vr: VR::SH },
            E { tag: 0x0009103B, alias: "SequenceType", vr: VR::LO },
            E { tag: 0x0009103C, alias: "IsExecutive", vr: VR::SH },
            E { tag: 0x0009103D, alias: "Note", vr: VR::ST },
            E { tag: 0x0009103E, alias: "AutoStart", vr: VR::SH },
            E { tag: 0x0009103F, alias: "AutoSave", vr: VR::SH },
            E { tag: 0x00091040, alias: "AutoArchive", vr: VR::SH },
            E { tag: 0x00091041, alias: "QueueGroupID", vr: VR::IS },
            E { tag: 0x00091042, alias: "IsFlagRaised", vr: VR::SH },
            E { tag: 0x00091043, alias: "IsArchived", vr: VR::SH },
            E { tag: 0x00091044, alias: "IsDefault", vr: VR::SH },
            E { tag: 0x00091045, alias: "TaskInfoAppData", vr: VR::OB },
            E { tag: 0x00091046, alias: "IsAllowCascadeSave", vr: VR::SH },
            E { tag: 0x00091047, alias: "IsAllowCascadeProtect", vr: VR::SH },
            E { tag: 0x00091048, alias: "ProtocolName", vr: VR::LO },
            E { tag: 0x0009104E, alias: "Cms_BodyPartExamined", vr: VR::LO },
            E { tag: 0x0009104F, alias: "IsProtected", vr: VR::LO },
            E { tag: 0x00091050, alias: "?StudyReservationInfo", vr: VR::OB },
            E { tag: 0x00091051, alias: "Cmi_contrastBolusAgent", vr: VR::LO },
            E { tag: 0x00091052, alias: "Cms_institutionName", vr: VR::LO },
            E { tag: 0x00091053, alias: "Cms_institutionalDepartmentName", vr: VR::LO },
            E { tag: 0x00091054, alias: "Cms_seriesDescription", vr: VR::LO },
            E { tag: 0x00091055, alias: "Cms_operatorsName", vr: VR::LO },
            E { tag: 0x00091056, alias: "Cms_PerformingPhysiciansName", vr: VR::LO },
            E { tag: 0x00091057, alias: "Cms_institutionAddress", vr: VR::ST },
            E { tag: 0x00091058, alias: "Cmi_imageComments", vr: VR::LO },
            E { tag: 0x00091059, alias: "Cmi_instanceCreationDateTime", vr: VR::LO },
            E { tag: 0x0009105A, alias: "MppsStepStatus", vr: VR::LO },
            E { tag: 0x0009105B, alias: "FilmedCount", vr: VR::IS },
            E { tag: 0x0009105C, alias: "IsAllowCascadeSave", vr: VR::LO },
            E { tag: 0x0009105D, alias: "IsAllowCascadeProtect", vr: VR::LO },
            E { tag: 0x0009105E, alias: "IsDeleted", vr: VR::LO },
            E { tag: 0x0009105F, alias: "ProtocolObjectID", vr: VR::UI },
            E { tag: 0x00091060, alias: "TaskInfoAppData", vr: VR::OB },
            E { tag: 0x00091072, alias: "ProtocolTaskInfoObjectID", vr: VR::UI },
            E { tag: 0x00091073, alias: "ProtocolTaskOrder", vr: VR::IS },
            E { tag: 0x00091074, alias: "ProtocolTaskUid", vr: VR::UI },
            E { tag: 0x00091075, alias: "ProtocolTaskAppData", vr: VR::OB },
            E { tag: 0x00091076, alias: "ProtocolTaskIsAllowCascadeSave", vr: VR::SH },
            E { tag: 0x00091077, alias: "ProtocolTaskIsAllowCascadeProtect", vr: VR::SH },
            E { tag: 0x00111000, alias: "Private Creator", vr: VR::LO },
            E { tag: 0x00111001, alias: "IsRapidRegistration", vr: VR::LO },
            E { tag: 0x00111002, alias: "IsProtected", vr: VR::LO },
            E { tag: 0x00111003, alias: "FilmedCount", vr: VR::IS },
            E { tag: 0x00111004, alias: "ApplicationData", vr: VR::OB },
            E { tag: 0x00111005, alias: "IsAllowCascadeSave", vr: VR::LO },
            E { tag: 0x00111006, alias: "IsAllowCascadeProtect", vr: VR::LO },
            E { tag: 0x00111007, alias: "IsDeleted", vr: VR::LO },
            E { tag: 0x00191000, alias: "Private Creator", vr: VR::LO },
            E { tag: 0x00191001, alias: "ProcType", vr: VR::LO },
            E { tag: 0x00191002, alias: "Plane", vr: VR::LO },
            E { tag: 0x00191003, alias: "IsSnapShotSeries", vr: VR::SH },
            E { tag: 0x00191004, alias: "MaxFsColor", vr: VR::DS },
            E { tag: 0x00191005, alias: "SeriesCategoryType", vr: VR::LO },
            E { tag: 0x00191007, alias: "ImageContrastBolusAgent", vr: VR::LO },
            E { tag: 0x00191008, alias: "ImageSliceThickness", vr: VR::DS },
            E { tag: 0x00191009, alias: "ImageReconstructionDiameter", vr: VR::DS },
            E { tag: 0x0019100A, alias: "ImageEchoTime", vr: VR::LO },
            E { tag: 0x0019100B, alias: "ImageRepetitionTime", vr: VR::DS },
            E { tag: 0x0019100C, alias: "SequenceType", vr: VR::LO },
            E { tag: 0x0019100D, alias: "TaskUid", vr: VR::LO },
            E { tag: 0x0019100E, alias: "SeriesAppData", vr: VR::OB },
            E { tag: 0x0019100F, alias: "MultiSliceNumber", vr: VR::IS },
            E { tag: 0x00191010, alias: "ImageScanTime", vr: VR::LO },
            E { tag: 0x00191011, alias: "IsProtected", vr: VR::LO },
            E { tag: 0x00191012, alias: "ImageIncrement", vr: VR::IS },
            E { tag: 0x00191013, alias: "MppsStepStatus", vr: VR::LO },
            E { tag: 0x00191014, alias: "StorageCommittedCount", vr: VR::IS },
            E { tag: 0x00191015, alias: "ArchivedCount", vr: VR::IS },
            E { tag: 0x00191016, alias: "TransferredCount", vr: VR::IS },
            E { tag: 0x00191017, alias: "IsAllowCascadeSave", vr: VR::LO },
            E { tag: 0x00191018, alias: "IsAllowCascadeProtect", vr: VR::LO },
            E { tag: 0x00191019, alias: "IsDeleted", vr: VR::LO },
            E { tag: 0x0019101A, alias: "CharacterizedImageInstanceUid", vr: VR::UI },
            E { tag: 0x0019101B, alias: "CharacterizedImageCount", vr: VR::IS },
            E { tag: 0x0019101C, alias: "InternalWindowWidth", vr: VR::LO },
            E { tag: 0x0019101D, alias: "InternalWindowLevel", vr: VR::LO },
            E { tag: 0x0019101E, alias: "?FlipAngle?", vr: VR::LO },
            E { tag: 0x0019101F, alias: "?InversionTime?", vr: VR::LO },
            E { tag: 0x00191020, alias: "MultiFrameSopInstanceUid", vr: VR::UI },
            E { tag: 0x00191021, alias: "?ScanInterface.GroupParamArchive", vr: VR::OB },
            E { tag: 0x00291000, alias: "Private Creator", vr: VR::LO },
            E { tag: 0x00291001, alias: "SliceNumber", vr: VR::IS },
            E { tag: 0x00291002, alias: "PhaseNumber", vr: VR::IS },
            E { tag: 0x00291003, alias: "ProcType", vr: VR::LO },
            E { tag: 0x00291004, alias: "StopwatchTime", vr: VR::LO },
            E { tag: 0x00291005, alias: "Plane", vr: VR::LO },
            E { tag: 0x00291006, alias: "ScanTime", vr: VR::LO },
            E { tag: 0x00291008, alias: "DualSliceFlag", vr: VR::LO },
            E { tag: 0x00291009, alias: "SspRatio", vr: VR::LO },
            E { tag: 0x0029100A, alias: "GatingSignalSource", vr: VR::LO },
            E { tag: 0x0029100B, alias: "Rephase", vr: VR::LO },
            E { tag: 0x0029100C, alias: "HalfEcho", vr: VR::LO },
            E { tag: 0x0029100D, alias: "RectFOVRatio", vr: VR::LO },
            E { tag: 0x0029100E, alias: "HalfScan", vr: VR::LO },
            E { tag: 0x0029100F, alias: "NumShots", vr: VR::LO },
            E { tag: 0x00291010, alias: "ContrastAgent", vr: VR::LO },
            E { tag: 0x00291011, alias: "EchoAllocation", vr: VR::LO },
            E { tag: 0x00291012, alias: "NumEchoShift", vr: VR::LO },
            E { tag: 0x00291013, alias: "FatSat", vr: VR::LO },
            E { tag: 0x00291014, alias: "MTC", vr: VR::LO },
            E { tag: 0x00291015, alias: "NumPreSat", vr: VR::LO },
            E { tag: 0x00291016, alias: "TargetVelocity", vr: VR::LO },
            E { tag: 0x00291017, alias: "VENCAxis", vr: VR::LO },
            E { tag: 0x00291018, alias: "NumVENCDirection", vr: VR::LO },
            E { tag: 0x0029101C, alias: "IsScalableWindowLevel", vr: VR::LO },
            E { tag: 0x0029101D, alias: "ThreeDSettingLineAngle", vr: VR::LO },
            E { tag: 0x0029101E, alias: "MPGTotalAxis", vr: VR::LO },
            E { tag: 0x0029101F, alias: "MPGAxisNumber", vr: VR::LO },
            E { tag: 0x00291020, alias: "MultiEchoNumber", vr: VR::IS },
            E { tag: 0x00291021, alias: "NaviAverageGateWidth", vr: VR::DS },
            E { tag: 0x00291022, alias: "ShimCompensateValue", vr: VR::ST },
            E { tag: 0x00291023, alias: "GCOffset", vr: VR::LO },
            E { tag: 0x00291024, alias: "NaviMaxGateWidth", vr: VR::DS },
            E { tag: 0x00291025, alias: "NaviMinGateWidth", vr: VR::DS },
            E { tag: 0x00291026, alias: "NaviMaxGatePosition", vr: VR::DS },
            E { tag: 0x00291027, alias: "NaviMinGatePosition", vr: VR::DS },
            E { tag: 0x00291028, alias: "TimeDuration", vr: VR::DS },
            E { tag: 0x00291029, alias: "TablePosition", vr: VR::DS },
            E { tag: 0x0029102A, alias: "NaviInitialGateWidth", vr: VR::DS },
            E { tag: 0x0029102B, alias: "NaviFinalGateWidth", vr: VR::DS },
            E { tag: 0x0029102C, alias: "NaviInitialGatePosition", vr: VR::DS },
            E { tag: 0x0029102D, alias: "NaviFinalGatePosition", vr: VR::DS },
            E { tag: 0x0029102E, alias: "NaviAverageGatePosition", vr: VR::DS },
            E { tag: 0x0029102F, alias: "ImageAppData", vr: VR::OB },
            E { tag: 0x00291030, alias: "DiffusionBValue", vr: VR::FD },
            E { tag: 0x00291031, alias: "SharedFunctionalGroupsSequence", vr: VR::SQ },
            E { tag: 0x00291032, alias: "PerFrameFunctionalGroupsSequence", vr: VR::SQ },
            E { tag: 0x00291033, alias: "LossyImageCompressionRatio", vr: VR::DS },
            E { tag: 0x00291034, alias: "InstanceCreatorUID", vr: VR::UI },
            E { tag: 0x00291035, alias: "RelatedGeneralSOPClassUID", vr: VR::UI },
            E { tag: 0x00291036, alias: "OriginalSpecializedSOPClassUID", vr: VR::UI },
            E { tag: 0x00291037, alias: "TimezoneOffsetFromUTC", vr: VR::SH },
            E { tag: 0x00291038, alias: "SOPInstanceStatus", vr: VR::CS },
            E { tag: 0x00291039, alias: "SOPAuthorizationDateandTime", vr: VR::DT },
            E { tag: 0x0029103A, alias: "SOPAuthorizationComment", vr: VR::LT },
            E { tag: 0x0029103B, alias: "AuthorizationEquipmentCertificationNumber", vr: VR::LO },
            E { tag: 0x0029103C, alias: "ConcatenationFrameOffsetNumber", vr: VR::UL },
            E { tag: 0x0029103D, alias: "RepresentativeFrameNumber", vr: VR::US },
            E { tag: 0x0029103E, alias: "ConcatenationUID", vr: VR::UI },
            E { tag: 0x0029103F, alias: "InConcatenationNumber", vr: VR::US },
            E { tag: 0x00291040, alias: "CardiacSynchronizationTechnique", vr: VR::CS },
            E { tag: 0x00291041, alias: "CardiacSignalSource", vr: VR::CS },
            E { tag: 0x00291042, alias: "CardiacRRIntervalSpecified", vr: VR::FD },
            E { tag: 0x00291043, alias: "CardiacBeatRejectionTechnique", vr: VR::CS },
            E { tag: 0x00291044, alias: "LowRRValue", vr: VR::IS },
            E { tag: 0x00291045, alias: "HighRRValue", vr: VR::IS },
            E { tag: 0x00291046, alias: "IntervalsAcquired", vr: VR::IS },
            E { tag: 0x00291047, alias: "IntervalsRejected", vr: VR::IS },
            E { tag: 0x00291048, alias: "RespiratoryMotionCompensationTechnique", vr: VR::CS },
            E { tag: 0x00291049, alias: "RespiratorySignalSource", vr: VR::CS },
            E { tag: 0x0029104A, alias: "BulkMotionCompensationTechnique", vr: VR::CS },
            E { tag: 0x0029104B, alias: "BulkMotionSignalSource", vr: VR::CS },
            E { tag: 0x0029104C, alias: "PixelPresentation", vr: VR::CS },
            E { tag: 0x0029104D, alias: "VolumetricProperties", vr: VR::CS },
            E { tag: 0x0029104E, alias: "VolumeBasedCalculationTechnique", vr: VR::CS },
            E { tag: 0x0029104F, alias: "AcquisitionContextDescription", vr: VR::ST },
            E { tag: 0x00291050, alias: "ModalityLUTSequence", vr: VR::SQ },
            E { tag: 0x00291051, alias: "LUTDescriptor", vr: VR::LO },
            E { tag: 0x00291052, alias: "LUTExplanation", vr: VR::LO },
            E { tag: 0x00291053, alias: "LUTData", vr: VR::LO },
            E { tag: 0x00291054, alias: "PresentationLUTShape", vr: VR::CS },
            E { tag: 0x00291055, alias: "FrameAnatomySequence", vr: VR::SQ },
            E { tag: 0x00291056, alias: "FrameLaterality", vr: VR::CS },
            E { tag: 0x00291057, alias: "AnatomicRegionSequence", vr: VR::SQ },
            E { tag: 0x00291058, alias: "AnatomicRegionCodeValue", vr: VR::SH },
            E { tag: 0x00291059, alias: "AnatomicRegionCodingSchemeDesignator", vr: VR::SH },
            E { tag: 0x0029105A, alias: "AnatomicRegionCodingSchemeVersion", vr: VR::SH },
            E { tag: 0x0029105B, alias: "AnatomicRegionCodeMeaning", vr: VR::LO },
            E { tag: 0x0029105C, alias: "PixelValueTransformationSequence", vr: VR::SQ },
            E { tag: 0x0029105D, alias: "RescaleType", vr: VR::LO },
            E { tag: 0x0029105E, alias: "CardiacTriggerSequence", vr: VR::SQ },
            E { tag: 0x0029105F, alias: "TriggerDelayTime", vr: VR::FD },
            E { tag: 0x00291060, alias: "FrameVOILUTSequence", vr: VR::SQ },
            E { tag: 0x00291061, alias: "WindowCenterAndWidthExplanation", vr: VR::LO },
            E { tag: 0x00291062, alias: "AcquisitionContrast", vr: VR::CS },
            E { tag: 0x00291063, alias: "MRModifierSequence", vr: VR::SQ },
            E { tag: 0x00291064, alias: "ParallelAcquisitionTechnic", vr: VR::CS },
            E { tag: 0x00291065, alias: "ParallelReductionFactorSecIn", vr: VR::FD },
            E { tag: 0x00291066, alias: "InversionRecovery", vr: VR::CS },
            E { tag: 0x00291067, alias: "FlowCompensation", vr: VR::CS },
            E { tag: 0x00291068, alias: "FlowCompensationDirection", vr: VR::CS },
            E { tag: 0x00291069, alias: "SpatialPreSaturation", vr: VR::CS },
            E { tag: 0x0029106A, alias: "PartialFourier", vr: VR::CS },
            E { tag: 0x0029106B, alias: "PartialFourierDirection", vr: VR::CS },
            E { tag: 0x0029106C, alias: "ResonantNucleus", vr: VR::CS },
            E { tag: 0x0029106D, alias: "KSpaceFiltering", vr: VR::CS },
            E { tag: 0x0029106E, alias: "ApplicableSafetyStandardAgency", vr: VR::CS },
            E { tag: 0x0029106F, alias: "ApplicableSafetyStandardDescription", vr: VR::LO },
            E { tag: 0x00291070, alias: "MRReceiveCoilSequence", vr: VR::SQ },
            E { tag: 0x00291071, alias: "ReceiveCoilManufacturerName", vr: VR::LO },
            E { tag: 0x00291072, alias: "ReceiveCoilType", vr: VR::CS },
            E { tag: 0x00291073, alias: "QuadratureReceiveCoil", vr: VR::CS },
            E { tag: 0x00291074, alias: "MultiCoilConfiguration", vr: VR::LO },
            E { tag: 0x00291075, alias: "ComplexImageComponent", vr: VR::CS },
            E { tag: 0x00291076, alias: "PulseSequenceName", vr: VR::SH },
            E { tag: 0x00291077, alias: "EchoPulseSequence", vr: VR::CS },
            E { tag: 0x00291078, alias: "MultipleSpinEcho", vr: VR::CS },
            E { tag: 0x00291079, alias: "MultiPlanarExcitation", vr: VR::CS },
            E { tag: 0x0029107A, alias: "PhaseContrast", vr: VR::CS },
            E { tag: 0x0029107B, alias: "TimeOfFlightContrast", vr: VR::CS },
            E { tag: 0x0029107C, alias: "SteadyStatePulseSequence", vr: VR::CS },
            E { tag: 0x0029107D, alias: "EchoPlanarPulseSequence", vr: VR::CS },
            E { tag: 0x0029107E, alias: "SpectrallySelectedSuppression", vr: VR::CS },
            E { tag: 0x0029107F, alias: "OversamplingPhase", vr: VR::CS },
            E { tag: 0x00291080, alias: "SegmentedKSpaceTraversal", vr: VR::CS },
            E { tag: 0x00291081, alias: "CoverageOfKSpace", vr: VR::CS },
            E { tag: 0x00291082, alias: "MRTimingAndRelatedParametersSequence", vr: VR::SQ },
            E { tag: 0x00291083, alias: "RFEchoTrainLength", vr: VR::US },
            E { tag: 0x00291084, alias: "GradientEchoTrainLength", vr: VR::US },
            E { tag: 0x00291085, alias: "GradientOutputType", vr: VR::CS },
            E { tag: 0x00291086, alias: "GradientOutput", vr: VR::FD },
            E { tag: 0x00291087, alias: "MRFOVGeometrySequence", vr: VR::SQ },
            E { tag: 0x00291088, alias: "MRAcquisitionFrequencyEncodingSteps", vr: VR::US },
            E { tag: 0x00291089, alias: "MRAcquisitionPhaseEncodingStepsInPlane", vr: VR::US },
            E { tag: 0x0029108A, alias: "MRAcquisitionPhaseEncodingStepsOutOfPlane", vr: VR::US },
            E { tag: 0x0029108B, alias: "MRTransmitCoilSequence", vr: VR::SQ },
            E { tag: 0x0029108C, alias: "TransmitCoilName", vr: VR::SH },
            E { tag: 0x0029108D, alias: "TransmitCoilManufacturerName", vr: VR::LO },
            E { tag: 0x0029108E, alias: "TransmitCoilType", vr: VR::CS },
            E { tag: 0x0029108F, alias: "MREchoSequence", vr: VR::SQ },
            E { tag: 0x00291090, alias: "EffectiveEchoTime", vr: VR::FD },
            E { tag: 0x00291091, alias: "MRMetaboliteMapSequence", vr: VR::SQ },
            E { tag: 0x00291092, alias: "MetaboliteMapDescription", vr: VR::ST },
            E { tag: 0x00291093, alias: "MetaboliteMapCodeSequence", vr: VR::SQ },
            E { tag: 0x00291094, alias: "MetaboliteMapCodeValue", vr: VR::SH },
            E { tag: 0x00291095, alias: "MetaboliteMapCodingSchemeDesignator", vr: VR::SH },
            E { tag: 0x00291096, alias: "MetaboliteMapCodingSchemeVersion", vr: VR::SH },
            E { tag: 0x00291097, alias: "MetaboliteMapCodeMeaning", vr: VR::LO },
            E { tag: 0x00291098, alias: "MRImagingModifierSequence", vr: VR::SQ },
            E { tag: 0x00291099, alias: "MagnetizationTransfer", vr: VR::CS },
            E { tag: 0x0029109A, alias: "BloodSignalNulling", vr: VR::CS },
            E { tag: 0x0029109B, alias: "Tagging", vr: VR::CS },
            E { tag: 0x0029109C, alias: "TagSpacingFirstDimension", vr: VR::FD },
            E { tag: 0x0029109D, alias: "TagSpacingSecondDimension", vr: VR::FD },
            E { tag: 0x0029109E, alias: "TagAngleFirstAxis", vr: VR::FD },
            E { tag: 0x0029109F, alias: "TagAngleSecondAxis", vr: VR::SS },
            E { tag: 0x002910A0, alias: "TagThickness", vr: VR::FD },
            E { tag: 0x002910A1, alias: "TaggingDelay", vr: VR::FD },
            E { tag: 0x002910A2, alias: "TransmitterFrequency", vr: VR::FD },
            E { tag: 0x002910A3, alias: "PixelBandwidth", vr: VR::DS },
            E { tag: 0x002910A4, alias: "MRVelocityEncodingSequence", vr: VR::SQ },
            E { tag: 0x002910A5, alias: "VelocityEncodingDirection", vr: VR::FD },
            E { tag: 0x002910A6, alias: "VelocityEncodingMinimumValue", vr: VR::FD },
            E { tag: 0x002910A7, alias: "VelocityEncodingMaximumValue", vr: VR::FD },
            E { tag: 0x002910A8, alias: "MRImageFrameTypeSequence", vr: VR::SQ },
            E { tag: 0x002910A9, alias: "FrameType", vr: VR::CS },
            E { tag: 0x002910AA, alias: "PixelPresentation", vr: VR::CS },
            E { tag: 0x002910AB, alias: "VolumetricProperties", vr: VR::CS },
            E { tag: 0x002910AC, alias: "VolumeBasedCalculationTechnique", vr: VR::CS },
            E { tag: 0x002910AD, alias: "FilmedCount", vr: VR::IS },
            E { tag: 0x002910AE, alias: "IsTransferred", vr: VR::LO },
            E { tag: 0x002910AF, alias: "IsArchived", vr: VR::LO },
            E { tag: 0x002910B0, alias: "MppsStepStatus", vr: VR::LO },
            E { tag: 0x002910B1, alias: "CommitmentStatus", vr: VR::LO },
            E { tag: 0x002910B2, alias: "IsStorageCommitted", vr: VR::LO },
            E { tag: 0x002910B3, alias: "IsDicom", vr: VR::LO },
            E { tag: 0x002910B4, alias: "IsAllowCascadeSave", vr: VR::LO },
            E { tag: 0x002910B5, alias: "IsAllowCascadeProtect", vr: VR::LO },
            E { tag: 0x002910B6, alias: "IsDeleted", vr: VR::LO },
            E { tag: 0x002910B7, alias: "ApplicationData", vr: VR::OB },
            E { tag: 0x002910B8, alias: "IsAllowCascadeSave", vr: VR::LO },
            E { tag: 0x002910B9, alias: "IsAllowCascadeProtect", vr: VR::LO },
            E { tag: 0x002910BA, alias: "IsDeleted", vr: VR::LO },
            E { tag: 0x002910BB, alias: "VOI1", vr: VR::IS },
            E { tag: 0x002910BC, alias: "VOI2", vr: VR::IS },
            E { tag: 0x002910BD, alias: "BackgroundImageInstanceUID", vr: VR::UI },
            E { tag: 0x002910BE, alias: "IsStoredToPortableMedia", vr: VR::LO },
            E { tag: 0x002910BF, alias: "Voi1", vr: VR::DS },
            E { tag: 0x002910C0, alias: "SelectiveIRColumn", vr: VR::FD },
            E { tag: 0x002910C1, alias: "Voi2", vr: VR::DS },
            E { tag: 0x002910C2, alias: "MixingTime", vr: VR::DS },
            E { tag: 0x002910C3, alias: "SelectiveIRPosition", vr: VR::DS },
            E { tag: 0x002910C4, alias: "SelectiveIRRow", vr: VR::DS },
            E { tag: 0x002910C5, alias: "SelectiveIRColumn", vr: VR::DS },
            E { tag: 0x002910C6, alias: "SelectiveIROrientation", vr: VR::DS },
            E { tag: 0x002910C7, alias: "SelectiveIRThickness", vr: VR::LO },
            E { tag: 0x002910C8, alias: "RephaseOrderSlice", vr: VR::SH },
            E { tag: 0x002910C9, alias: "RephaseOrderPhase", vr: VR::SH },
            E { tag: 0x002910CA, alias: "RephaseOrderFreq", vr: VR::SH },
            E { tag: 0x002910CB, alias: "MetaboliteMapDescription", vr: VR::ST },
            E { tag: 0x002910CC, alias: "volumeLocalizationSeq", vr: VR::SQ },
            E { tag: 0x002910CD, alias: "SlabThickness", vr: VR::FD },
            E { tag: 0x002910CE, alias: "SlabOrientation", vr: VR::FD },
            E { tag: 0x002910CF, alias: "MidSlabPosition", vr: VR::FD },
            E { tag: 0x002910D0, alias: "AcqModeSliceDir", vr: VR::LO },
            E { tag: 0x002910D1, alias: "IRThicknessRatio", vr: VR::LO },
            E { tag: 0x002910D2, alias: "BBIRThicknessRatio", vr: VR::LO },
            E { tag: 0x002910D3, alias: "DeltaAngle", vr: VR::LO },
            E { tag: 0x002910D4, alias: "MultiFrameFrameNumber", vr: VR::IS },
            E { tag: 0x002910D5, alias: "EnhancedSopInstanceUid", vr: VR::UI },
            E { tag: 0x002910D6, alias: "PolarityOfPhaseEncoding", vr: VR::LO },
            E { tag: 0x002910D7, alias: "PresentationStates", vr: VR::OB },
            E { tag: 0x002910D9, alias: "Magnetic Field Direction", vr: VR::DS },
            E { tag: 0x00411001, alias: "RawDataAppData", vr: VR::OB },
            E { tag: 0x00411002, alias: "RawDataIndex", vr: VR::SQ },
            E { tag: 0x00411003, alias: "ChannelNumber", vr: VR::LO },
            E { tag: 0x00411004, alias: "AxisDirection", vr: VR::LO },
            E { tag: 0x00411005, alias: "SlabNumbe", vr: VR::LO },
            E { tag: 0x00411006, alias: "CardiacPhaseNumbe", vr: VR::LO },
            E { tag: 0x00411007, alias: "EchoNumber", vr: VR::LO },
            E { tag: 0x00411008, alias: "SliceEncodeNumber", vr: VR::LO },
            E { tag: 0x00411009, alias: "NsaNumber", vr: VR::LO },
            E { tag: 0x0041100A, alias: "RawData", vr: VR::OB },
            E { tag: 0x0041100B, alias: "RawDataMRInfo", vr: VR::SS },
            E { tag: 0x0041100C, alias: "NumberOfVoxels", vr: VR::IS },
            E { tag: 0x0041100D, alias: "MixingTime", vr: VR::DS },
            E { tag: 0x0041100E, alias: "ADDiff", vr: VR::DS },
            E { tag: 0x0041100F, alias: "ScanTime", vr: VR::LO },
            E { tag: 0x00411010, alias: "NumPreSat", vr: VR::LO },
            E { tag: 0x00411011, alias: "IsStoredToPortableMedia", vr: VR::LO },
            E { tag: 0x00411012, alias: "Voi1", vr: VR::DS },
            E { tag: 0x00411013, alias: "Voi2", vr: VR::DS },
            E { tag: 0x00411014, alias: "VoxelSize", vr: VR::DS },
            E { tag: 0x00411015, alias: "FreqPoint", vr: VR::IS },
            E { tag: 0x00411016, alias: "LowOrderShim", vr: VR::SH },
            E { tag: 0x00411017, alias: "EccLevel", vr: VR::SH },
            E { tag: 0x00411018, alias: "FwhmHz", vr: VR::FL },
            E { tag: 0x00411019, alias: "FwhmPpm", vr: VR::FL },
            E { tag: 0x0041101A, alias: "WaterSupRate", vr: VR::FL },
            E { tag: 0x00711001, alias: "ForegroundTransparency", vr: VR::FL },
            E { tag: 0x00711002, alias: "IsDisplayBackgroundImage", vr: VR::LO },
            E { tag: 0x00711003, alias: "ForegroundHorizontalShift", vr: VR::FL },
            E { tag: 0x00711004, alias: "ForegroundVerticalShift", vr: VR::FL },
            E { tag: 0x00711005, alias: "ForegroundRotationAngle", vr: VR::FL },
            E { tag: 0x00711006, alias: "ForegroundMagnification", vr: VR::FL },
            E { tag: 0x00711007, alias: "ApplicationData", vr: VR::OB },
        ],
    },
    P {
        creator: "Mayo/IBM Archive Project",
        entries: &[
            E { tag: 0x00211001, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211010, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211011, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211012, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211013, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211014, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211015, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211016, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211017, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211018, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211019, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101A, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101B, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101D, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0021101F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211020, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211040, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211041, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211050, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211060, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00211065, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "MeVis BreastCare",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00711001, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "MeVis eD: Geometry Information",
        entries: &[
            E { tag: 0x00091011, alias: "GeoScannerOrigin", vr: VR::UN },
            E { tag: 0x00091021, alias: "SlicesID", vr: VR::UN },
        ],
    },
    P {
        creator: "MeVis eD: Slice Information",
        entries: &[
            E { tag: 0x00091010, alias: "SlicesSOPInstanceUID", vr: VR::UN },
        ],
    },
    P {
        creator: "MeVis eD: Timepoint Information",
        entries: &[
            E { tag: 0x00091010, alias: "TPDateTime", vr: VR::UN },
            E { tag: 0x00091011, alias: "TPDateTimeType", vr: VR::UN },
            E { tag: 0x00091012, alias: "TPSeriesDescription", vr: VR::UN },
            E { tag: 0x00091013, alias: "TPGradientDirections", vr: VR::UN },
            E { tag: 0x00091021, alias: "GeometryID", vr: VR::UN },
        ],
    },
    P {
        creator: "MeVis eatDicom",
        entries: &[
            E { tag: 0x00091010, alias: "EatDicomVersion", vr: VR::UN },
            E { tag: 0x00091011, alias: "EatDicomOptions", vr: VR::UN },
            E { tag: 0x00091021, alias: "TimepointsID", vr: VR::UN },
        ],
    },
    P {
        creator: "MedIns HP Extensions",
        entries: &[
            E { tag: 0x54731003, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Mitra Markup 1.0",
        entries: &[
            E { tag: 0x00291000, alias: "Markup1", vr: VR::OB },
            E { tag: 0x00291001, alias: "Markup2", vr: VR::OB },
            E { tag: 0x00291002, alias: "Markup3", vr: VR::OB },
            E { tag: 0x00291003, alias: "Markup4", vr: VR::OB },
            E { tag: 0x00291004, alias: "Markup5", vr: VR::OB },
            E { tag: 0x00291005, alias: "Markup6", vr: VR::OB },
            E { tag: 0x00291006, alias: "Markup7", vr: VR::OB },
            E { tag: 0x00291007, alias: "Markup8", vr: VR::OB },
            E { tag: 0x00291008, alias: "Markup9", vr: VR::OB },
            E { tag: 0x00291009, alias: "Markup10", vr: VR::OB },
            E { tag: 0x00291010, alias: "Markup11", vr: VR::OB },
            E { tag: 0x00291011, alias: "Markup12", vr: VR::OB },
            E { tag: 0x00291012, alias: "Markup13", vr: VR::OB },
            E { tag: 0x00291013, alias: "Markup14", vr: VR::OB },
            E { tag: 0x00291014, alias: "Markup15", vr: VR::OB },
        ],
    },
    P {
        creator: "Mitra Object Document 1.0",
        entries: &[
            E { tag: 0x00291000, alias: "IMPAX object document", vr: VR::OB },
            E { tag: 0x00291010, alias: "IMPAX markup XML stored", vr: VR::OB },
        ],
    },
    P {
        creator: "Mitra Presentation 1.0",
        entries: &[
            E { tag: 0x00291000, alias: "Mitra Rotation", vr: VR::CS },
            E { tag: 0x00291001, alias: "Mitra Window Width", vr: VR::LO },
            E { tag: 0x00291002, alias: "Mitra Window Centre", vr: VR::LO },
            E { tag: 0x00291003, alias: "Mitra Invert", vr: VR::IS },
            E { tag: 0x00291004, alias: "Mitra Has Tabstop", vr: VR::IS },
            E { tag: 0x00291005, alias: "Smooth Rotation", vr: VR::CS },
        ],
    },
    P {
        creator: "Mortara_Inc",
        entries: &[
            E { tag: 0x14551000, alias: "ELI Interpretation Vector", vr: VR::OW },
            E { tag: 0x14551001, alias: "Custom ID", vr: VR::UN },
            E { tag: 0x14551002, alias: "Race", vr: VR::UT },
            E { tag: 0x14551003, alias: "Social Security Number", vr: VR::UT },
            E { tag: 0x14551004, alias: "Attending Physician", vr: VR::UT },
            E { tag: 0x14551005, alias: "Procedural Diagnosis", vr: VR::UT },
            E { tag: 0x14551006, alias: "Note1", vr: VR::UT },
            E { tag: 0x14551007, alias: "Note2", vr: VR::UT },
            E { tag: 0x14551008, alias: "Order Request Number", vr: VR::LO },
            E { tag: 0x14551010, alias: "Manufacturer Name", vr: VR::LO },
        ],
    },
    P {
        creator: "NUD_PRIVATE",
        entries: &[
            E { tag: 0x77771002, alias: "Interfile", vr: VR::UT },
            E { tag: 0x77771005, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "ObjectModel (ID, Version, Place, PlaceDescription)",
        entries: &[
            E { tag: 0x81011000, alias: "ID", vr: VR::LO },
            E { tag: 0x81011010, alias: "Version", vr: VR::LO },
        ],
    },
    P {
        creator: "PAPYRUS",
        entries: &[
            E { tag: 0x00091000, alias: "Original File Name", vr: VR::LT },
            E { tag: 0x00091010, alias: "Original File Location", vr: VR::LT },
            E { tag: 0x00091018, alias: "Data Set Identifier", vr: VR::LT },
            E { tag: 0x00411000, alias: "Papyrus Comments", vr: VR::LT },
            E { tag: 0x00411010, alias: "Folder Type", vr: VR::US },
            E { tag: 0x00411011, alias: "Patient Folder Data Set ID", vr: VR::LT },
            E { tag: 0x00411020, alias: "Folder Name", vr: VR::LT },
            E { tag: 0x00411030, alias: "Creation Date", vr: VR::DA },
            E { tag: 0x00411032, alias: "Creation Time", vr: VR::TM },
            E { tag: 0x00411034, alias: "Modified Date", vr: VR::DA },
            E { tag: 0x00411036, alias: "Modified Time", vr: VR::TM },
            E { tag: 0x00411040, alias: "Owner Name", vr: VR::LT },
            E { tag: 0x00411050, alias: "Folder Status", vr: VR::LT },
            E { tag: 0x00411060, alias: "Number of Images", vr: VR::UL },
            E { tag: 0x00411062, alias: "Number of Other", vr: VR::UL },
            E { tag: 0x004110A0, alias: "External Folder Element DSID", vr: VR::LT },
            E { tag: 0x004110A1, alias: "External Folder Element Data Set Type", vr: VR::US },
            E { tag: 0x004110A2, alias: "External Folder Element File Location", vr: VR::LT },
            E { tag: 0x004110A3, alias: "External Folder Element Length", vr: VR::UL },
            E { tag: 0x004110B0, alias: "Internal Folder Element DSID", vr: VR::LT },
            E { tag: 0x004110B1, alias: "Internal Folder Element Data Set Type", vr: VR::US },
            E { tag: 0x004110B2, alias: "Internal Offset To Data Set", vr: VR::UL },
            E { tag: 0x004110B3, alias: "Internal Offset To Image", vr: VR::UL },
        ],
    },
    P {
        creator: "PAPYRUS 3.0",
        entries: &[
            E { tag: 0x00411000, alias: "Comments", vr: VR::LT },
            E { tag: 0x00411010, alias: "Pointer Sequence", vr: VR::SQ },
            E { tag: 0x00411011, alias: "Image Pointer", vr: VR::UL },
            E { tag: 0x00411012, alias: "Pixel Offset", vr: VR::UL },
            E { tag: 0x00411013, alias: "Image Identifier Sequence", vr: VR::SQ },
            E { tag: 0x00411014, alias: "External PAPYRUS-File Reference Sequence", vr: VR::SQ },
            E { tag: 0x00411015, alias: "Number of images", vr: VR::US },
            E { tag: 0x00411021, alias: "Referenced SOP Class UID", vr: VR::UI },
            E { tag: 0x00411022, alias: "Referenced SOP Instance UID", vr: VR::UI },
            E { tag: 0x00411031, alias: "Referenced File Name", vr: VR::LO },
            E { tag: 0x00411032, alias: "Referenced File Path", vr: VR::LO },
            E { tag: 0x00411041, alias: "Referenced Image SOP Class UID", vr: VR::UI },
            E { tag: 0x00411042, alias: "Referenced Image SOP Instance UID", vr: VR::UI },
            E { tag: 0x00411050, alias: "Image Sequence", vr: VR::SQ },
            E { tag: 0x60101000, alias: "Overlay ID", vr: VR::IS },
            E { tag: 0x60101001, alias: "Linked Overlays", vr: VR::LT },
            E { tag: 0x60101010, alias: "Overlay Rows", vr: VR::US },
            E { tag: 0x60101011, alias: "Overlay Columns", vr: VR::US },
            E { tag: 0x60101040, alias: "Overlay Type", vr: VR::LO },
            E { tag: 0x60101050, alias: "Overlay Origin", vr: VR::US },
            E { tag: 0x60101060, alias: "Editable", vr: VR::LO },
            E { tag: 0x60101070, alias: "Overlay Font", vr: VR::LO },
            E { tag: 0x60101072, alias: "Overlay Style", vr: VR::LO },
            E { tag: 0x60101074, alias: "Overlay Font Size", vr: VR::US },
            E { tag: 0x60101076, alias: "Overlay Color", vr: VR::LO },
            E { tag: 0x60101078, alias: "Shadow Size", vr: VR::US },
            E { tag: 0x60101080, alias: "Fill Pattern", vr: VR::LO },
            E { tag: 0x60101082, alias: "Overlay Pen Size", vr: VR::US },
            E { tag: 0x601010A0, alias: "Label", vr: VR::LO },
            E { tag: 0x601010A2, alias: "Post It Text", vr: VR::LT },
            E { tag: 0x601010A4, alias: "Anchor Point", vr: VR::US },
            E { tag: 0x601010B0, alias: "ROI Type", vr: VR::LO },
            E { tag: 0x601010B2, alias: "Attached Annotation", vr: VR::LT },
            E { tag: 0x601010BA, alias: "Contour Points", vr: VR::US },
            E { tag: 0x601010BC, alias: "Mask Data", vr: VR::US },
            E { tag: 0x601010C0, alias: "UIN Overlay Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "PHILIPS MR",
        entries: &[
            E { tag: 0x00091010, alias: "SPI Release", vr: VR::LO },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "PHILIPS MR R5.5/PART",
        entries: &[
            E { tag: 0x00191000, alias: "Field of View", vr: VR::DS },
        ],
    },
    P {
        creator: "PHILIPS MR R5.6/PART",
        entries: &[
            E { tag: 0x00191000, alias: "Field of View", vr: VR::DS },
        ],
    },
    P {
        creator: "PHILIPS MR SPECTRO;1",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191005, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191006, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191007, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191009, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191010, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191012, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191013, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191014, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191015, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191016, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191017, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191018, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191020, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191021, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191022, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191023, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191024, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191025, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191027, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191028, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191029, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191041, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191042, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191043, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191045, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191046, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191047, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191048, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191049, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191060, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191061, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191071, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191072, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191073, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191074, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191076, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191077, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191078, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191079, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191080, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "PHILIPS MR/LAST",
        entries: &[
            E { tag: 0x00191009, alias: "Main Magnetic Field", vr: VR::DS },
            E { tag: 0x0019100E, alias: "Flow Compensation", vr: VR::IS },
            E { tag: 0x001910B1, alias: "Minimum RR Interval", vr: VR::IS },
            E { tag: 0x001910B2, alias: "Maximum RR Interval", vr: VR::IS },
            E { tag: 0x001910B3, alias: "Number of Rejections", vr: VR::IS },
            E { tag: 0x001910B4, alias: "Number of RR Intervals", vr: VR::IS },
            E { tag: 0x001910B5, alias: "Arrhythmia Rejection", vr: VR::IS },
            E { tag: 0x001910B7, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910C0, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910C6, alias: "Cycled Multiple Slice", vr: VR::IS },
            E { tag: 0x001910CE, alias: "REST", vr: VR::IS },
            E { tag: 0x001910D5, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910D6, alias: "Fourier Interpolation", vr: VR::IS },
            E { tag: 0x001910D9, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910E0, alias: "Prepulse", vr: VR::IS },
            E { tag: 0x001910E1, alias: "Prepulse Delay", vr: VR::DS },
            E { tag: 0x001910E2, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910E3, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910E4, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910E5, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910F0, alias: "WS Protocol String 1", vr: VR::LT },
            E { tag: 0x001910F1, alias: "WS Protocol String 2", vr: VR::LT },
            E { tag: 0x001910F2, alias: "WS Protocol String 3", vr: VR::LT },
            E { tag: 0x001910F3, alias: "WS Protocol String 4", vr: VR::LT },
            E { tag: 0x00211000, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211010, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211020, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211021, alias: "Slice Gap", vr: VR::DS },
            E { tag: 0x00211022, alias: "Stack Radial Angle", vr: VR::DS },
            E { tag: 0x00271000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00271011, alias: "Unknown", vr: VR::US },
            E { tag: 0x00271012, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00271013, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00271014, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00271015, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00271016, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291010, alias: "FP Min", vr: VR::DS },
            E { tag: 0x00291020, alias: "FP Max", vr: VR::DS },
            E { tag: 0x00291030, alias: "Scaled Minimum", vr: VR::DS },
            E { tag: 0x00291040, alias: "Scaled Maximum", vr: VR::DS },
            E { tag: 0x00291050, alias: "Window Minimum", vr: VR::DS },
            E { tag: 0x00291060, alias: "Window Maximum", vr: VR::DS },
            E { tag: 0x00291061, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291062, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291071, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291072, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291080, alias: "View Center", vr: VR::IS },
            E { tag: 0x00291081, alias: "View Size", vr: VR::IS },
            E { tag: 0x00291082, alias: "View Zoom", vr: VR::IS },
            E { tag: 0x00291083, alias: "View Transform", vr: VR::IS },
            E { tag: 0x00411007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00411009, alias: "Unknown", vr: VR::DS },
            E { tag: 0x60011000, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "PHILIPS MR/PART",
        entries: &[
            E { tag: 0x00191000, alias: "Field of View", vr: VR::DS },
            E { tag: 0x00191001, alias: "Stack Type", vr: VR::IS },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191003, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191005, alias: "CC Angulation", vr: VR::DS },
            E { tag: 0x00191006, alias: "AP Angulation", vr: VR::DS },
            E { tag: 0x00191007, alias: "LR Angulation", vr: VR::DS },
            E { tag: 0x00191008, alias: "Patient Orientation 1", vr: VR::IS },
            E { tag: 0x00191009, alias: "Patient Orientation", vr: VR::IS },
            E { tag: 0x0019100A, alias: "Slice Orientation", vr: VR::IS },
            E { tag: 0x0019100B, alias: "LR Offcenter", vr: VR::DS },
            E { tag: 0x0019100C, alias: "CC Offcenter", vr: VR::DS },
            E { tag: 0x0019100D, alias: "AP Offcenter", vr: VR::DS },
            E { tag: 0x0019100E, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019100F, alias: "Number of Slices", vr: VR::IS },
            E { tag: 0x00191010, alias: "Slice Factor", vr: VR::DS },
            E { tag: 0x00191011, alias: "Echo Times", vr: VR::DS },
            E { tag: 0x00191014, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00191015, alias: "Dynamic Study", vr: VR::IS },
            E { tag: 0x00191018, alias: "Heartbeat Interval", vr: VR::DS },
            E { tag: 0x00191019, alias: "Repetition Time FFE", vr: VR::DS },
            E { tag: 0x0019101A, alias: "FFE Flip Angle", vr: VR::DS },
            E { tag: 0x0019101B, alias: "Number of Scans", vr: VR::IS },
            E { tag: 0x0019101C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019101D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019101E, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191021, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191022, alias: "Dynamic Scan Time Begin", vr: VR::DS },
            E { tag: 0x00191023, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191024, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191025, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191027, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191028, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00191029, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191030, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191045, alias: "Reconstruction Resolution", vr: VR::IS },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191051, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191052, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191053, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191054, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191055, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191056, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191057, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191058, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191059, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191060, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191061, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191062, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191063, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191064, alias: "Repetition Time SE", vr: VR::DS },
            E { tag: 0x00191065, alias: "Repetition Time IR", vr: VR::DS },
            E { tag: 0x00191066, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191067, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191069, alias: "Number of Phases", vr: VR::IS },
            E { tag: 0x0019106A, alias: "Cardiac Frequency", vr: VR::IS },
            E { tag: 0x0019106B, alias: "Inversion Delay", vr: VR::DS },
            E { tag: 0x0019106C, alias: "Gate Delay", vr: VR::DS },
            E { tag: 0x0019106D, alias: "Gate Width", vr: VR::DS },
            E { tag: 0x0019106E, alias: "Trigger Delay Time", vr: VR::DS },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191080, alias: "Number of Chemical Shifts", vr: VR::IS },
            E { tag: 0x00191081, alias: "Chemical Shift", vr: VR::DS },
            E { tag: 0x00191084, alias: "Number of Rows", vr: VR::IS },
            E { tag: 0x00191085, alias: "Number of Samples", vr: VR::IS },
            E { tag: 0x0019108A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019108B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019108C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019108D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019108E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0019108F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00191094, alias: "Magnetization Transfer Contrast", vr: VR::LO },
            E { tag: 0x00191095, alias: "Spectral Presaturation With Inversion Recovery", vr: VR::LO },
            E { tag: 0x00191096, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191097, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191099, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001910A0, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A3, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A4, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910B4, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910B5, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910B6, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910C0, alias: "Trigger Delay Times", vr: VR::DS },
            E { tag: 0x001910C8, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910C9, alias: "Foldover Direction Transverse", vr: VR::IS },
            E { tag: 0x001910CA, alias: "Foldover Direction Sagittal", vr: VR::IS },
            E { tag: 0x001910CB, alias: "Foldover Direction Coronal", vr: VR::IS },
            E { tag: 0x001910CC, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910CD, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910CE, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910CF, alias: "Number of Echoes", vr: VR::IS },
            E { tag: 0x001910D0, alias: "Scan Resolution", vr: VR::IS },
            E { tag: 0x001910D1, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910D2, alias: "Water Fat Shift", vr: VR::LO },
            E { tag: 0x001910D3, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910D4, alias: "Artifact Reduction", vr: VR::IS },
            E { tag: 0x001910D5, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910D6, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910D7, alias: "Scan Percentage", vr: VR::DS },
            E { tag: 0x001910D8, alias: "Halfscan", vr: VR::IS },
            E { tag: 0x001910D9, alias: "EPI Factor", vr: VR::IS },
            E { tag: 0x001910DA, alias: "Turbo Factor", vr: VR::IS },
            E { tag: 0x001910DB, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910E0, alias: "Percentage of Scan Completed", vr: VR::IS },
            E { tag: 0x001910E1, alias: "Prepulse Delay", vr: VR::DS },
            E { tag: 0x001910E3, alias: "Phase Contrast Velocity", vr: VR::DS },
            E { tag: 0x001910F0, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910F6, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910F7, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910F8, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910F9, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910FA, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910FB, alias: "Unknown", vr: VR::CS },
            E { tag: 0x001910FC, alias: "Resonance Frequency", vr: VR::IS },
            E { tag: 0x00211000, alias: "Scan Date", vr: VR::IS },
            E { tag: 0x00211006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211007, alias: "Unknown", vr: VR::DT },
            E { tag: 0x00211008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211009, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0021100A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0021100F, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211010, alias: "Image Type", vr: VR::IS },
            E { tag: 0x00211013, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00211014, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211015, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211020, alias: "Slice Number", vr: VR::IS },
            E { tag: 0x00211021, alias: "Slice Gap", vr: VR::IS },
            E { tag: 0x00211030, alias: "Echo Number", vr: VR::IS },
            E { tag: 0x00211031, alias: "Patient Reference ID", vr: VR::DS },
            E { tag: 0x00211035, alias: "Chemical Shift Number", vr: VR::IS },
            E { tag: 0x00211040, alias: "Phase Number", vr: VR::IS },
            E { tag: 0x00211050, alias: "Dynamic Scan Number", vr: VR::IS },
            E { tag: 0x00211060, alias: "Number of Rows In Object", vr: VR::IS },
            E { tag: 0x00211061, alias: "Row Number", vr: VR::IS },
            E { tag: 0x00211062, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291000, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291011, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291031, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291050, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291051, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291052, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291053, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002910C3, alias: "Scan Resolution", vr: VR::IS },
            E { tag: 0x002910C4, alias: "Field of View", vr: VR::IS },
            E { tag: 0x002910D5, alias: "Slice Thickness", vr: VR::LT },
        ],
    },
    P {
        creator: "PHILIPS MR/PART 12",
        entries: &[
            E { tag: 0x00091010, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "PHILIPS MR/PART 6",
        entries: &[
            E { tag: 0x00191010, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "PHILIPS MR/PART 7",
        entries: &[
            E { tag: 0x00191000, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "PHILIPS NM -Private",
        entries: &[
            E { tag: 0x70511000, alias: "Current Segment", vr: VR::US },
            E { tag: 0x70511001, alias: "Number of Segments", vr: VR::US },
            E { tag: 0x70511002, alias: "Segment Start Position", vr: VR::FL },
            E { tag: 0x70511003, alias: "Segment Stop Position", vr: VR::FL },
            E { tag: 0x70511004, alias: "Rel. COR offset - X dir.", vr: VR::FL },
            E { tag: 0x70511005, alias: "Rel. COR offset - Z dir.", vr: VR::FL },
            E { tag: 0x70511006, alias: "Current Rotation Number", vr: VR::US },
            E { tag: 0x70511007, alias: "Number of Rotations", vr: VR::US },
            E { tag: 0x70511010, alias: "Alignment Translations", vr: VR::DS },
            E { tag: 0x70511011, alias: "Alignment Rotations", vr: VR::DS },
            E { tag: 0x70511012, alias: "Alignment Timestamp", vr: VR::DS },
        ],
    },
    P {
        creator: "PHILIPS UNDOCUMENTED",
        entries: &[
            E { tag: 0x55031012, alias: "Slice Thickness", vr: VR::SL },
            E { tag: 0x55031024, alias: "Plane", vr: VR::UT },
            E { tag: 0x55031030, alias: "Number of Slices", vr: VR::SS },
            E { tag: 0x550310A5, alias: "Prepulse", vr: VR::UT },
            E { tag: 0x550310AE, alias: "Contrast Agent Flag", vr: VR::SS },
            E { tag: 0x550310BB, alias: "Coil ID", vr: VR::SL },
            E { tag: 0x550310C2, alias: "Anatomy", vr: VR::UT },
            E { tag: 0x550310C3, alias: "Indication", vr: VR::UT },
        ],
    },
    P {
        creator: "PHILIPS UNKNOWN",
        entries: &[
            E { tag: 0x20011001, alias: "MRSeriesNrOfEchoes", vr: VR::UN },
            E { tag: 0x20011002, alias: "MRSeriesScanningTechniqueDescription", vr: VR::UN },
            E { tag: 0x20011005, alias: "StackSequence", vr: VR::UN },
            E { tag: 0x20011006, alias: "Examination Source", vr: VR::UN },
            E { tag: 0x20011007, alias: "MRSeriesAcquisitionNumber", vr: VR::UN },
            E { tag: 0x20011008, alias: "MRSeriesNrOfDynamicScans", vr: VR::UN },
            E { tag: 0x20051002, alias: "MRSeriesNrOfChemicalShifts", vr: VR::UN },
            E { tag: 0x20051003, alias: "MRSeriesRepetitionTime", vr: VR::UN },
            E { tag: 0x20051036, alias: "MRVolumeSelection", vr: VR::UN },
        ],
    },
    P {
        creator: "PHILIPS-MR-1",
        entries: &[
            E { tag: 0x00191011, alias: "Chemical Shift Number", vr: VR::IS },
            E { tag: 0x00191012, alias: "Phase Number (Philips)", vr: VR::IS },
            E { tag: 0x00211001, alias: "Reconstruction Number", vr: VR::IS },
            E { tag: 0x00211002, alias: "Slice Number", vr: VR::IS },
        ],
    },
    P {
        creator: "PMOD_1",
        entries: &[
            E { tag: 0x00551001, alias: "Frame Start Times Vector", vr: VR::FD },
            E { tag: 0x00551002, alias: "Frame Positions Vector", vr: VR::FD },
            E { tag: 0x00551003, alias: "Frame Orientations Vector", vr: VR::FD },
            E { tag: 0x00551004, alias: "Frame Durations (ms) Vector", vr: VR::FD },
            E { tag: 0x00551005, alias: "Frame Rescale Slope Vector", vr: VR::FD },
        ],
    },
    P {
        creator: "PMOD_GENPET",
        entries: &[
            E { tag: 0x7FE11001, alias: "Slices Names", vr: VR::UT },
            E { tag: 0x7FE11002, alias: "Gene Codes", vr: VR::UT },
            E { tag: 0x7FE11003, alias: "Gene Labels", vr: VR::UT },
        ],
    },
    P {
        creator: "PMS-THORA-5.1",
        entries: &[
            E { tag: 0x00891020, alias: "Stamp Image Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "PMTF INFORMATION DATA",
        entries: &[
            E { tag: 0x00290089, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291031, alias: "PMTF Information 1", vr: VR::LO },
            E { tag: 0x00291032, alias: "PMTF Information 2", vr: VR::UL },
            E { tag: 0x00291033, alias: "PMTF Information 3", vr: VR::UL },
            E { tag: 0x00291034, alias: "PMTF Information 4", vr: VR::CS },
            E { tag: 0x00291089, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::OB },
            E { tag: 0x70151073, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "PMTF INFORMATION DATA^10",
        entries: &[
            E { tag: 0x00090002, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00110003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110004, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00110005, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110006, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110007, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190015, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190017, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190020, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190021, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190022, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190023, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190024, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190025, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190027, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019002F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::OF },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190033, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190039, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190045, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190047, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190049, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190050, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190051, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190054, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190056, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190064, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190065, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190069, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019006B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190071, alias: "?Sequence Options?", vr: VR::LO },
            E { tag: 0x00190073, alias: "?Sequence Options?", vr: VR::LO },
            E { tag: 0x00190075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190085, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190088, alias: "?RxGain?", vr: VR::LO },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019008D, alias: "Unknown", vr: VR::TM },
            E { tag: 0x0019008E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190096, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A8, alias: "?Repetition Time (/100)", vr: VR::FL },
            E { tag: 0x001900A9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Acquisition Duration", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CA, alias: "Sequence Name", vr: VR::LO },
            E { tag: 0x001900CB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CC, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CF, alias: "?acquired series (vs postproc)?", vr: VR::SL },
            E { tag: 0x001900D0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D9, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900DA, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900DC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EA, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900EB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::FD },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210002, alias: "?num?", vr: VR::SL },
            E { tag: 0x00210003, alias: "?num?", vr: VR::SL },
            E { tag: 0x00210004, alias: "?num?", vr: VR::US },
            E { tag: 0x00210008, alias: "?vect1?", vr: VR::DS },
            E { tag: 0x00210009, alias: "?vect2?", vr: VR::DS },
            E { tag: 0x0021000A, alias: "?vect3?", vr: VR::DS },
            E { tag: 0x0021000B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0021000F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210012, alias: "?vect4?", vr: VR::DS },
            E { tag: 0x00210014, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210015, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210018, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021001A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021001D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0021001F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00290001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290005, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00290006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029001C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290021, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0029004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290054, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290067, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00290068, alias: "Unknown", vr: VR::SS },
            E { tag: 0x0029006E, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE2010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE3010000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010001, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010003, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010004, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010005, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010007, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010008, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010012, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010013, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010020, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010021, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010022, alias: "Unknown", vr: VR::LO },
            E { tag: 0xE4010023, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010024, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xEE010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xEE010001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xEE010002, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "PMTF INFORMATION DATA^11",
        entries: &[
            E { tag: 0x00190001, alias: "?character set?", vr: VR::OB },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::OB },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019001A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190024, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190025, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190037, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190039, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003A, alias: "Repetition Time (/100)", vr: VR::FL },
            E { tag: 0x0019003C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190041, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "?QD Whole Body?", vr: VR::LT },
            E { tag: 0x00190049, alias: "?Atlas Head?", vr: VR::LT },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004D, alias: "?postproc series?", vr: VR::SL },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019005C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190065, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190067, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190068, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019006E, alias: "Sequence Name", vr: VR::SH },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190070, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190071, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190072, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190080, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190084, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190090, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190091, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190094, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B3, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B9, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F1, alias: "Diffusion b-value x Toshiba Orientation", vr: VR::SL },
            E { tag: 0x001900F2, alias: "Specific Absorption Rate Values", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F7, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "PMTF INFORMATION DATA^12",
        entries: &[
            E { tag: 0x00190000, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "?Sequence Variant?", vr: VR::SH },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190032, alias: "Sequence Name", vr: VR::SH },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190079, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190080, alias: "Acquisition Duration", vr: VR::FD },
            E { tag: 0x00190085, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190088, alias: "Series Number Global", vr: VR::US },
            E { tag: 0x00190089, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Series Number Global", vr: VR::US },
            E { tag: 0x0019008B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008F, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190093, alias: "?num?", vr: VR::SL },
            E { tag: 0x00190094, alias: "Referenced SOP Instance UID", vr: VR::UI },
            E { tag: 0x00190095, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190096, alias: "?PASTA?", vr: VR::LO },
            E { tag: 0x00190098, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190099, alias: "Sequence Name", vr: VR::LO },
            E { tag: 0x0019009A, alias: "Series Number Local", vr: VR::DS },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019009C, alias: "?num?", vr: VR::DS },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A9, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B3, alias: "Acquisition Duration", vr: VR::FD },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900C2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900CC, alias: "Unknown", vr: VR::SS },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SS },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D7, alias: "?vect2?", vr: VR::US },
            E { tag: 0x001900D8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900DE, alias: "?num?", vr: VR::US },
            E { tag: 0x001900DF, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E1, alias: "?vect3?", vr: VR::FL },
            E { tag: 0x001900E2, alias: "?vect3?", vr: VR::FL },
            E { tag: 0x001900E3, alias: "?time?", vr: VR::TM },
            E { tag: 0x001900E4, alias: "?vect2?", vr: VR::US },
            E { tag: 0x001900EB, alias: "?num?", vr: VR::US },
            E { tag: 0x001900EC, alias: "?num?", vr: VR::US },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "PMTF INFORMATION DATA^13",
        entries: &[
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4030001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030002, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030003, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "POLYTRON-SMS 2.5",
        entries: &[
            E { tag: 0x00091002, alias: "Private Data 1", vr: VR::UN },
            E { tag: 0x00091003, alias: "Private Data 2", vr: VR::UN },
            E { tag: 0x00091004, alias: "Private Data 3", vr: VR::UN },
            E { tag: 0x00091006, alias: "Private Data 3", vr: VR::UN },
            E { tag: 0x00091010, alias: "Private Data 4", vr: VR::UN },
            E { tag: 0x00091011, alias: "Private Data 5", vr: VR::UN },
            E { tag: 0x00091012, alias: "Private Data 6", vr: VR::UN },
            E { tag: 0x00891010, alias: "Private Data", vr: VR::UN },
        ],
    },
    P {
        creator: "PRIVATE ORIGINAL ATTRIBUTES",
        entries: &[
            E { tag: 0x00211001, alias: "Private Original Attribute Sequence", vr: VR::SQ },
            E { tag: 0x00211005, alias: "Replacement Date and Time", vr: VR::DT },
            E { tag: 0x00211010, alias: "Replacement Originator Name", vr: VR::LO },
            E { tag: 0x00211012, alias: "Replacement Reason", vr: VR::LO },
            E { tag: 0x00211070, alias: "Original Operators' Name", vr: VR::PN },
        ],
    },
    P {
        creator: "Philips EV Imaging DD 017",
        entries: &[
            E { tag: 0x200710C1, alias: "? SOP Instance UID ?", vr: VR::UI },
            E { tag: 0x200710C2, alias: "Unknown", vr: VR::DA },
            E { tag: 0x200710C3, alias: "Unknown", vr: VR::TM },
            E { tag: 0x200710C4, alias: "Unknown", vr: VR::UL },
        ],
    },
    P {
        creator: "Philips EV Imaging DD 019",
        entries: &[
            E { tag: 0x20071000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20071001, alias: "Unknown", vr: VR::UT },
        ],
    },
    P {
        creator: "Philips EV Imaging DD 021",
        entries: &[
            E { tag: 0x20071004, alias: "Vol Seg Edit Connectivity", vr: VR::SL },
        ],
    },
    P {
        creator: "Philips EV Imaging DD 022",
        entries: &[
            E { tag: 0x20071000, alias: "", vr: VR::ST },
            E { tag: 0x20071012, alias: "Volume Sequence Capture", vr: VR::UN },
            E { tag: 0x20071017, alias: "Original SOP Instance", vr: VR::UN },
            E { tag: 0x20071033, alias: "", vr: VR::LO },
            E { tag: 0x20071034, alias: "", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips EnVisor",
        entries: &[
            E { tag: 0x77771001, alias: "Date 3D/PanView internal 'dataset' file was acquired", vr: VR::DA },
            E { tag: 0x77771002, alias: "Type of 3D or PanView", vr: VR::CS },
            E { tag: 0x77771003, alias: "Internal 3D/PanView software version number", vr: VR::LO },
            E { tag: 0x77771014, alias: "Private string", vr: VR::LT },
        ],
    },
    P {
        creator: "Philips Imaging DD 001",
        entries: &[
            E { tag: 0x20011001, alias: "Chemical Shift", vr: VR::FL },
            E { tag: 0x20011002, alias: "Chemical Shift Number MR", vr: VR::IS },
            E { tag: 0x20011003, alias: "Diffusion B-Factor", vr: VR::FL },
            E { tag: 0x20011004, alias: "Diffusion Direction", vr: VR::CS },
            E { tag: 0x20011005, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011006, alias: "Image Enhanced", vr: VR::CS },
            E { tag: 0x20011007, alias: "Image Type ED ES", vr: VR::CS },
            E { tag: 0x20011008, alias: "Phase Number", vr: VR::IS },
            E { tag: 0x20011009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2001100A, alias: "Slice Number MR", vr: VR::IS },
            E { tag: 0x2001100B, alias: "Slice Orientation", vr: VR::CS },
            E { tag: 0x2001100C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2001100E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2001100F, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011010, alias: "Cardiac Sync", vr: VR::CS },
            E { tag: 0x20011011, alias: "Diffusion Echo Time", vr: VR::FL },
            E { tag: 0x20011012, alias: "Dynamic Series", vr: VR::CS },
            E { tag: 0x20011013, alias: "EPI Factor", vr: VR::SL },
            E { tag: 0x20011014, alias: "Number of Echoes", vr: VR::SL },
            E { tag: 0x20011015, alias: "Number of Locations", vr: VR::SS },
            E { tag: 0x20011016, alias: "Number of PC Directions", vr: VR::SS },
            E { tag: 0x20011017, alias: "Number of Phases MR", vr: VR::SL },
            E { tag: 0x20011018, alias: "Number of Slices MR", vr: VR::SL },
            E { tag: 0x20011019, alias: "Partial Matrix Scanned", vr: VR::CS },
            E { tag: 0x2001101A, alias: "PC Velocity", vr: VR::FL },
            E { tag: 0x2001101B, alias: "Prepulse Delay", vr: VR::FL },
            E { tag: 0x2001101C, alias: "Prepulse Type", vr: VR::CS },
            E { tag: 0x2001101D, alias: "Reconstruction Number MR", vr: VR::IS },
            E { tag: 0x2001101E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2001101F, alias: "Respiration Sync", vr: VR::CS },
            E { tag: 0x20011020, alias: "Scanning Technique Description MR", vr: VR::LO },
            E { tag: 0x20011021, alias: "SPIR", vr: VR::CS },
            E { tag: 0x20011022, alias: "Water Fat Shift", vr: VR::FL },
            E { tag: 0x20011023, alias: "Flip Angle Philips", vr: VR::DS },
            E { tag: 0x20011024, alias: "Interactive", vr: VR::CS },
            E { tag: 0x20011025, alias: "Echo Time Display MR", vr: VR::SH },
            E { tag: 0x20011026, alias: "Presentation State Subtraction Active", vr: VR::CS },
            E { tag: 0x20011028, alias: "Edge Enhancement Gain Factor Sub", vr: VR::FL },
            E { tag: 0x20011029, alias: "Edge Enhancement Gain Factor Non Sub", vr: VR::FL },
            E { tag: 0x2001102A, alias: "Edge Enhancement Taste Adapt Sub", vr: VR::CS },
            E { tag: 0x2001102B, alias: "Edge Enhancement Taste Non Sub", vr: VR::CS },
            E { tag: 0x2001102C, alias: "Harmonization Factor", vr: VR::FL },
            E { tag: 0x2001102D, alias: "Number of Stack Slices", vr: VR::SS },
            E { tag: 0x2001102F, alias: "Harmonization Gain", vr: VR::FL },
            E { tag: 0x20011030, alias: "Log Subtraction Gain Step", vr: VR::UL },
            E { tag: 0x20011031, alias: "Mixing NR of Mask Image Numbers", vr: VR::US },
            E { tag: 0x20011032, alias: "Stack Radial Angle", vr: VR::FL },
            E { tag: 0x20011033, alias: "Stack Radial Axis", vr: VR::CS },
            E { tag: 0x20011034, alias: "Mixing Mask Operation", vr: VR::CS },
            E { tag: 0x20011035, alias: "Stack Slice Number", vr: VR::SS },
            E { tag: 0x20011036, alias: "Stack Type", vr: VR::CS },
            E { tag: 0x20011037, alias: "Mixing Operation Type", vr: VR::CS },
            E { tag: 0x20011039, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2001103A, alias: "Pixel Shift", vr: VR::CS },
            E { tag: 0x2001103D, alias: "Unknown", vr: VR::UL },
            E { tag: 0x2001103F, alias: "Interpolation Method", vr: VR::CS },
            E { tag: 0x20011042, alias: "Subtraction Land Marking Active", vr: VR::CS },
            E { tag: 0x20011046, alias: "Graphic Line Style", vr: VR::CS },
            E { tag: 0x20011047, alias: "Graphic Line Width", vr: VR::FL },
            E { tag: 0x20011048, alias: "Graphic Annotation ID", vr: VR::SS },
            E { tag: 0x2001104B, alias: "Poly Line Interpolation Method", vr: VR::CS },
            E { tag: 0x2001104C, alias: "Poly Line Begin Point Style", vr: VR::CS },
            E { tag: 0x2001104D, alias: "Poly Line End Point Style", vr: VR::CS },
            E { tag: 0x2001104E, alias: "Window Smoothing Taste", vr: VR::CS },
            E { tag: 0x2001104F, alias: "Harmonization Offset", vr: VR::FD },
            E { tag: 0x20011050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20011052, alias: "Unknown", vr: VR::UI },
            E { tag: 0x20011053, alias: "Window Invert", vr: VR::CS },
            E { tag: 0x20011055, alias: "Graphic Line Color", vr: VR::CS },
            E { tag: 0x20011056, alias: "Graphic Type (Private)", vr: VR::CS },
            E { tag: 0x2001105A, alias: "Graphic Annotation Model", vr: VR::ST },
            E { tag: 0x2001105D, alias: "Measurement Text Units", vr: VR::ST },
            E { tag: 0x2001105E, alias: "Measurement Text Type", vr: VR::ST },
            E { tag: 0x2001105F, alias: "Stack Sequence", vr: VR::SQ },
            E { tag: 0x20011060, alias: "Number of Stacks", vr: VR::SL },
            E { tag: 0x20011061, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20011062, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20011063, alias: "Examination Source", vr: VR::CS },
            E { tag: 0x20011064, alias: "Text Type", vr: VR::SH },
            E { tag: 0x20011065, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20011067, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20011068, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20011069, alias: "Display Shutter Sequence", vr: VR::SQ },
            E { tag: 0x2001106A, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2001106B, alias: "Has Edge Enhancement Sequence", vr: VR::SQ },
            E { tag: 0x2001106D, alias: "Text Font", vr: VR::LO },
            E { tag: 0x2001106E, alias: "Series Type", vr: VR::SH },
            E { tag: 0x2001106F, alias: "Mixing Sequence", vr: VR::SQ },
            E { tag: 0x20011071, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20011073, alias: "Referenced Mask Image Sequence", vr: VR::SQ },
            E { tag: 0x20011074, alias: "Window Center Sub", vr: VR::DS },
            E { tag: 0x20011075, alias: "Window Width Sub", vr: VR::DS },
            E { tag: 0x20011076, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20011077, alias: "GL TrafoType", vr: VR::CS },
            E { tag: 0x20011079, alias: "Harmonisation Sequence", vr: VR::SQ },
            E { tag: 0x2001107A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2001107B, alias: "Acquisition Number", vr: VR::IS },
            E { tag: 0x2001107E, alias: "Edge Enhancement Gain Taste", vr: VR::US },
            E { tag: 0x2001107F, alias: "Edge Enhancement Gain Taste Sub", vr: VR::US },
            E { tag: 0x20011080, alias: "Text Anchor Point Alignment", vr: VR::LO },
            E { tag: 0x20011081, alias: "Number of Dynamic Scans", vr: VR::IS },
            E { tag: 0x20011082, alias: "Echo Train Length", vr: VR::IS },
            E { tag: 0x20011083, alias: "Imaging Frequency", vr: VR::DS },
            E { tag: 0x20011084, alias: "Inversion Time", vr: VR::DS },
            E { tag: 0x20011085, alias: "Magnetic Field Strength", vr: VR::DS },
            E { tag: 0x20011086, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20011087, alias: "Imaged Nucleus", vr: VR::SH },
            E { tag: 0x20011088, alias: "Number of Averages", vr: VR::DS },
            E { tag: 0x20011089, alias: "Phase FOV Percent", vr: VR::DS },
            E { tag: 0x2001108A, alias: "Sampling Percent", vr: VR::DS },
            E { tag: 0x2001108B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x20011093, alias: "Text Style", vr: VR::LO },
            E { tag: 0x20011094, alias: "Processing Order Specialization", vr: VR::LO },
            E { tag: 0x20011097, alias: "Nucleus", vr: VR::SH },
            E { tag: 0x2001109A, alias: "Graphic Number Sequence", vr: VR::SQ },
            E { tag: 0x2001109B, alias: "Graphic Number", vr: VR::UL },
            E { tag: 0x2001109C, alias: "Unknown", vr: VR::LO },
            E { tag: 0x2001109D, alias: "PIIM Subtraction Type", vr: VR::LO },
            E { tag: 0x2001109F, alias: "Pixel Processing Kernel Size", vr: VR::US },
            E { tag: 0x200110A1, alias: "Is Raw Image", vr: VR::CS },
            E { tag: 0x200110A2, alias: "LOG Subtraction Curve Taste", vr: VR::US },
            E { tag: 0x200110A3, alias: "Text Color Foreground", vr: VR::UL },
            E { tag: 0x200110A4, alias: "Text Color Background", vr: VR::UL },
            E { tag: 0x200110A5, alias: "Text Color Shadow", vr: VR::UL },
            E { tag: 0x200110C1, alias: "Nested Object Type Name", vr: VR::LO },
            E { tag: 0x200110CC, alias: "Unknown", vr: VR::ST },
            E { tag: 0x200110DA, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200110F1, alias: "Prospective Motion Correction", vr: VR::FL },
            E { tag: 0x200110F2, alias: "Retrospective Motion Correction", vr: VR::FL },
            E { tag: 0x20051014, alias: "Diffusion", vr: VR::CS },
            E { tag: 0x20051020, alias: "Number of Chemical Shifts", vr: VR::SL },
            E { tag: 0x20051030, alias: "Repetition Time", vr: VR::FL },
            E { tag: 0x20051033, alias: "Scan Duration", vr: VR::FL },
            E { tag: 0x20051035, alias: "Data Type", vr: VR::CS },
            E { tag: 0x2005108B, alias: "Transmitting Coil", vr: VR::SH },
            E { tag: 0x2005109F, alias: "Spectral Selective Excitation Pulse", vr: VR::CS },
            E { tag: 0x200510A1, alias: "Syncra Scan Type", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips Imaging DD 002",
        entries: &[
            E { tag: 0x20011001, alias: "Unknown", vr: VR::US },
            E { tag: 0x20011002, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011013, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011014, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011015, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011016, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011017, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011018, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20011019, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101C, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101D, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101E, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001101F, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011020, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011021, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011022, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011023, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011024, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011025, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011026, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011027, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011028, alias: "Unknown", vr: VR::US },
            E { tag: 0x20011029, alias: "Unknown", vr: VR::US },
            E { tag: 0x2001102A, alias: "Unknown", vr: VR::US },
            E { tag: 0x2001102B, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2001102C, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001102D, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2001102E, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2001102F, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011030, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011031, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011032, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011033, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011034, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011035, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011036, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011037, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20011039, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2001103A, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2001103B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2001103C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2001103D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2001103E, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2001103F, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20011040, alias: "Unknown", vr: VR::SS },
        ],
    },
    P {
        creator: "Philips Imaging DD 065",
        entries: &[
            E { tag: 0x40071000, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips Imaging DD 067",
        entries: &[
            E { tag: 0x40011000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x40011001, alias: "Unknown", vr: VR::CS },
            E { tag: 0x40011008, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips Imaging DD 070",
        entries: &[
            E { tag: 0x40011010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x40011011, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x40011012, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x40011016, alias: "Unknown", vr: VR::ST },
            E { tag: 0x40011017, alias: "Unknown", vr: VR::ST },
            E { tag: 0x40011018, alias: "Unknown", vr: VR::ST },
            E { tag: 0x4001101C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x4001101D, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "Philips Imaging DD 073",
        entries: &[
            E { tag: 0x40071048, alias: "Unknown", vr: VR::FL },
        ],
    },
    P {
        creator: "Philips Imaging DD 129",
        entries: &[
            E { tag: 0x20011000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20011001, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 001",
        entries: &[
            E { tag: 0x20051000, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051001, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051002, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051004, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051005, alias: "Synergy Reconstruction Type", vr: VR::CS },
            E { tag: 0x20051008, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005100F, alias: "Window Center", vr: VR::DS },
            E { tag: 0x20051010, alias: "Window Width", vr: VR::DS },
            E { tag: 0x20051011, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051012, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051013, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051014, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051015, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051016, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051017, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051018, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051019, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005101A, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2005101B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005101C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005101D, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2005101E, alias: "Unknown", vr: VR::SH },
            E { tag: 0x2005101F, alias: "Unknown", vr: VR::SH },
            E { tag: 0x20051020, alias: "Number of Chemical Shift", vr: VR::SL },
            E { tag: 0x20051021, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051022, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051023, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051025, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051026, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051027, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051028, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051029, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005102B, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2005102C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005102E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051030, alias: "Repetition Time", vr: VR::FL },
            E { tag: 0x20051031, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051032, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051033, alias: "Acquisition Duration", vr: VR::FL },
            E { tag: 0x20051034, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051035, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051036, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051037, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051038, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051039, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005103A, alias: "Unknown", vr: VR::SH },
            E { tag: 0x2005103B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005103C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005103D, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2005103E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20051048, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051049, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005104A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005104B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005104C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005104D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005104E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005104F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051050, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051051, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051052, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051053, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051054, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051055, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051056, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051058, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051059, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005105A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005105B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005105C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005105D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005105E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005105F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051060, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051061, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051063, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051064, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051065, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051066, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051067, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051068, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051069, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005106A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005106B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005106C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005106D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005106E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005106F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051071, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051072, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051073, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051074, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051075, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051076, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051078, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051079, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005107A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005107B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005107E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051080, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051081, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051083, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051084, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051085, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051086, alias: "Unknown", vr: VR::SS },
            E { tag: 0x2005109E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2005109F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200510A0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x200510A1, alias: "Syncra Scan Type", vr: VR::CS },
            E { tag: 0x200510A2, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200510A3, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200510A4, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200510A5, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200510A6, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200510A7, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200510A8, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200510A9, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200510B0, alias: "Diffusion Direction RL", vr: VR::FL },
            E { tag: 0x200510B1, alias: "Diffusion Direction AP", vr: VR::FL },
            E { tag: 0x200510B2, alias: "Diffusion Direction FH", vr: VR::FL },
            E { tag: 0x200510C0, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 002",
        entries: &[
            E { tag: 0x20051032, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051037, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051038, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051039, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051040, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051043, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20051044, alias: "Unknown", vr: VR::OW },
            E { tag: 0x20051047, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051099, alias: "Unknown", vr: VR::UL },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 003",
        entries: &[
            E { tag: 0x20051000, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051013, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20051045, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051049, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051051, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051052, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051053, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051056, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051057, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051081, alias: "Unknown", vr: VR::UI },
            E { tag: 0x20051082, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 004",
        entries: &[
            E { tag: 0x20051000, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051001, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051002, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051003, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051004, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051005, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051006, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051008, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051010, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051012, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051013, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051014, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051015, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051016, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051017, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051025, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051026, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051027, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051028, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051029, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051030, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051031, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051033, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051034, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051035, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051036, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051037, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051038, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051039, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051040, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051041, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051042, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051043, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051044, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051045, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051046, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051047, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051048, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051049, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051050, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051051, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051052, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051053, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051054, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051055, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051056, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051057, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051058, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051059, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051060, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051061, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051062, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051063, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051064, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051070, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051071, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051072, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051073, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051074, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051075, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051076, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051077, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051078, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051079, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051081, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051082, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051089, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051090, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051091, alias: "Unknown", vr: VR::PN },
            E { tag: 0x20051092, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051093, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051095, alias: "Unknown", vr: VR::UI },
            E { tag: 0x20051096, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051097, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20051098, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051099, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 005",
        entries: &[
            E { tag: 0x20051000, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051002, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051003, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051006, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051007, alias: "Unknown", vr: VR::SS },
            E { tag: 0x20051009, alias: "Unknown", vr: VR::DS },
            E { tag: 0x2005100A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x2005100B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x2005100E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2005100F, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x20051010, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051011, alias: "Unknown", vr: VR::UI },
            E { tag: 0x20051012, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051013, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051014, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20051015, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20051016, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051018, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051019, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005101A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005101B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005101C, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005101D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x2005101E, alias: "Unknown", vr: VR::UL },
            E { tag: 0x2005101F, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051020, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051021, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051022, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051023, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051024, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051025, alias: "Unknown", vr: VR::UL },
            E { tag: 0x20051026, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051027, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x2005102A, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102C, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005102E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005102F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051030, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051031, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051032, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051035, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051036, alias: "Unknown", vr: VR::UI },
            E { tag: 0x20051037, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051038, alias: "Unknown", vr: VR::LT },
            E { tag: 0x20051039, alias: "Unknown", vr: VR::LT },
            E { tag: 0x2005103A, alias: "Unknown", vr: VR::LT },
            E { tag: 0x2005103B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005103C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005103D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005103E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005103F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051040, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051041, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051042, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051043, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051044, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20051045, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051046, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051047, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051048, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20051049, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005104A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005104B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005104C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x2005104D, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005104E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x2005104F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20051050, alias: "Unknown", vr: VR::SS },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 006",
        entries: &[
            E { tag: 0x20050053, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050054, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050055, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050056, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050057, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050058, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050059, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050060, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050061, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050062, alias: "Unknown", vr: VR::LT },
            E { tag: 0x20050063, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050064, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050065, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050066, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050067, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050068, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050071, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050072, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050073, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050074, alias: "Unknown", vr: VR::DS },
            E { tag: 0x20050075, alias: "Unknown", vr: VR::DS },
            E { tag: 0x20050076, alias: "Unknown", vr: VR::LT },
            E { tag: 0x20050078, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050079, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050081, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050082, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050083, alias: "Unknown", vr: VR::LT },
            E { tag: 0x20050085, alias: "Unknown", vr: VR::DS },
            E { tag: 0x20050086, alias: "Unknown", vr: VR::LT },
            E { tag: 0x20050087, alias: "Unknown", vr: VR::DS },
            E { tag: 0x20050095, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050096, alias: "Unknown", vr: VR::IS },
            E { tag: 0x20050097, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050099, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "Philips MR Imaging DD 007",
        entries: &[
            E { tag: 0x20050000, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20050001, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050002, alias: "Unknown", vr: VR::FL },
            E { tag: 0x20050003, alias: "Unknown", vr: VR::FL },
        ],
    },
    P {
        creator: "Philips NM Private Group",
        entries: &[
            E { tag: 0x70431000, alias: "Unknown", vr: VR::SH },
        ],
    },
    P {
        creator: "Philips PET Private Group",
        entries: &[
            E { tag: 0x05111000, alias: "Unknown", vr: VR::US },
            E { tag: 0x05111001, alias: "Unknown", vr: VR::US },
            E { tag: 0x05111002, alias: "Unknown", vr: VR::OB },
            E { tag: 0x05111003, alias: "Unknown", vr: VR::OB },
            E { tag: 0x05111032, alias: "Unknown", vr: VR::DS },
            E { tag: 0x05111050, alias: "Unknown", vr: VR::DS },
            E { tag: 0x70531000, alias: "SUV Scale Factor", vr: VR::DS },
            E { tag: 0x70531001, alias: "Private", vr: VR::OB },
            E { tag: 0x70531002, alias: "Private", vr: VR::OB },
            E { tag: 0x70531003, alias: "Original image file name", vr: VR::ST },
            E { tag: 0x70531004, alias: "File Data", vr: VR::OB },
            E { tag: 0x70531005, alias: "Worklist Info File Name", vr: VR::LO },
            E { tag: 0x70531006, alias: "Unknown", vr: VR::OB },
            E { tag: 0x70531007, alias: "Acquisition File Sequence", vr: VR::SQ },
            E { tag: 0x70531008, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x70531009, alias: "Activity Concentration Scale Factor", vr: VR::DS },
            E { tag: 0x7053100B, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053100C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053100D, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053100E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053100F, alias: "Segment Size", vr: VR::UL },
            E { tag: 0x70531010, alias: "Segment Number", vr: VR::US },
            E { tag: 0x70531011, alias: "Number of Segments", vr: VR::US },
            E { tag: 0x70531012, alias: "File Data Sequence", vr: VR::SQ },
            E { tag: 0x70531013, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531014, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531015, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531016, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531017, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531018, alias: "Unknown", vr: VR::SS },
            E { tag: 0x70531050, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531051, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531052, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531053, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531054, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531055, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531056, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531057, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531058, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531059, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105A, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105B, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105D, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053105F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531060, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531061, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531062, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531063, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531064, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531065, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531066, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531067, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531068, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531069, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106A, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106B, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106D, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7053106F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531070, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531071, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531072, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531073, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531074, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531075, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531076, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531077, alias: "Unknown", vr: VR::UN },
            E { tag: 0x70531078, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310C2, alias: "Unknown", vr: VR::UI },
            E { tag: 0x705310D0, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D1, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D2, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D3, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D5, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D6, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D7, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D8, alias: "Unknown", vr: VR::UN },
            E { tag: 0x705310D9, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "Philips RAD Imaging DD 001",
        entries: &[
            E { tag: 0x200B1000, alias: "Unknown", vr: VR::PN },
            E { tag: 0x200B1001, alias: "Unknown", vr: VR::US },
            E { tag: 0x200B1002, alias: "Unknown", vr: VR::US },
            E { tag: 0x200B1005, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B1011, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200B1027, alias: "Unknown", vr: VR::DT },
            E { tag: 0x200B1028, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200B1029, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200B102A, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200B102B, alias: "Unknown", vr: VR::DA },
            E { tag: 0x200B102C, alias: "Unknown", vr: VR::TM },
            E { tag: 0x200B102D, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200B103B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200B1040, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B1041, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B1042, alias: "Unknown", vr: VR::UI },
            E { tag: 0x200B1043, alias: "Unknown", vr: VR::UI },
            E { tag: 0x200B1047, alias: "Unknown", vr: VR::DA },
            E { tag: 0x200B1048, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B104C, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B104D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B104F, alias: "Unknown", vr: VR::DT },
            E { tag: 0x200B1052, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Philips RAD Imaging DD 097",
        entries: &[
            E { tag: 0x200B1000, alias: "Unknown", vr: VR::ST },
            E { tag: 0x200B1001, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200B1002, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1050, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1051, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1052, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1053, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1054, alias: "Unknown", vr: VR::ST },
            E { tag: 0x200B1060, alias: "Unknown", vr: VR::LT },
            E { tag: 0x200B1063, alias: "Unknown", vr: VR::LT },
            E { tag: 0x200B1065, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B106E, alias: "Unknown", vr: VR::US },
            E { tag: 0x200B1072, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200B1073, alias: "Unknown", vr: VR::SS },
            E { tag: 0x200B1074, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B1075, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200B1076, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B1078, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B1079, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B107A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B107B, alias: "Unknown", vr: VR::US },
            E { tag: 0x200B107C, alias: "Unknown", vr: VR::US },
            E { tag: 0x200B107D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B107E, alias: "Unknown", vr: VR::UI },
            E { tag: 0x200B1081, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200B1082, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200B1085, alias: "Unknown", vr: VR::IS },
            E { tag: 0x200B1086, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200B1088, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200B1089, alias: "Unknown", vr: VR::LT },
            E { tag: 0x200B1090, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200B1096, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B1099, alias: "Unknown", vr: VR::SH },
            E { tag: 0x200B109A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200B109B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200B10A0, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "Philips US Imaging 60",
        entries: &[
            E { tag: 0x00311030, alias: "Private data", vr: VR::UL },
            E { tag: 0x00311031, alias: "Private data", vr: VR::UL },
        ],
    },
    P {
        creator: "Philips US Imaging DD 017",
        entries: &[
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 023",
        entries: &[
            E { tag: 0x200D1037, alias: "Unknown", vr: VR::DA },
            E { tag: 0x200D1038, alias: "Unknown", vr: VR::TM },
        ],
    },
    P {
        creator: "Philips US Imaging DD 033",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100B, alias: "Raw Data US", vr: VR::OB },
            E { tag: 0x200D1014, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "Philips US Imaging DD 034",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100C, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100D, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100E, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100F, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1021, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1022, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1023, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1024, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 035",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100C, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100D, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 036",
        entries: &[
            E { tag: 0x200D1001, alias: "Cols/Rows/Frames", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Cols/Rows/Frames", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Voxel Spacing", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 038",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 039",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1008, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100C, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D100D, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 040",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1020, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 041",
        entries: &[
            E { tag: 0x200D101E, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 042",
        entries: &[
            E { tag: 0x200D1050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1053, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1054, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1055, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1056, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1057, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1058, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1059, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105C, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105D, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105E, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D105F, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1060, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1071, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1072, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1073, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1074, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1076, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1077, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Philips US Imaging DD 043",
        entries: &[
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::SH },
        ],
    },
    P {
        creator: "Philips US Imaging DD 109",
        entries: &[
            E { tag: 0x200D1000, alias: "Unknown", vr: VR::US },
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::ST },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D1008, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::OB },
            E { tag: 0x200D100A, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100B, alias: "Unknown", vr: VR::OB },
            E { tag: 0x200D100C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x200D100E, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D100F, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1010, alias: "Unknown", vr: VR::SL },
            E { tag: 0x200D1011, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1012, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "Philips US Imaging DD 113",
        entries: &[
            E { tag: 0x200D1001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x200D1002, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1003, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1004, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1005, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1006, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1007, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D1009, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100B, alias: "Unknown", vr: VR::CS },
            E { tag: 0x200D100C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100D, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100E, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D100F, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200D1010, alias: "Unknown", vr: VR::DS },
            E { tag: 0x200D1011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x200D1012, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1013, alias: "Unknown", vr: VR::FL },
            E { tag: 0x200D1014, alias: "Unknown", vr: VR::US },
            E { tag: 0x200D1015, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D1017, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D1018, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D1019, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D101A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D101B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D101C, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D101D, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D101E, alias: "Unknown", vr: VR::UL },
            E { tag: 0x200D101F, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D1020, alias: "Unknown", vr: VR::FD },
            E { tag: 0x200D1021, alias: "Unknown", vr: VR::FD },
        ],
    },
    P {
        creator: "Philips US Private 3D",
        entries: &[
            E { tag: 0x200D1002, alias: "Threedpr Echo Vision Setting", vr: VR::IS },
            E { tag: 0x200D1003, alias: "Threedpr Color Vision Setting", vr: VR::IS },
            E { tag: 0x200D1004, alias: "Threedpr Transparency", vr: VR::IS },
            E { tag: 0x200D1006, alias: "Threedpr Vision Vol Brightness", vr: VR::FD },
            E { tag: 0x200D1008, alias: "Threedpr Low Threshold", vr: VR::IS },
            E { tag: 0x200D1009, alias: "Threedpr Image Layout", vr: VR::IS },
            E { tag: 0x200D100A, alias: "Threedpr Xres Filtering Enabled", vr: VR::IS },
            E { tag: 0x200D100B, alias: "Threedpr Echo Twod Chroma Map Enabled", vr: VR::IS },
            E { tag: 0x200D100C, alias: "Threedpr Box Outlined Enabled", vr: VR::IS },
            E { tag: 0x200D100D, alias: "Threedpr Smoothing Enabled", vr: VR::IS },
            E { tag: 0x200D100E, alias: "Threedpr Reference Graphic Enabled", vr: VR::IS },
            E { tag: 0x200D100F, alias: "Threedpr Map Inverted", vr: VR::IS },
            E { tag: 0x200D1010, alias: "Threedpr Color Stream Display Enabled", vr: VR::IS },
            E { tag: 0x200D1011, alias: "Threedpr Echo Stream Display Enabled", vr: VR::IS },
            E { tag: 0x200D1012, alias: "Vdb Param Color 3d Estimate Scale(2)", vr: VR::LO },
            E { tag: 0x200D1013, alias: "Vdb Param Color 3d Estimate Apex Position (1)", vr: VR::LO },
            E { tag: 0x200D1014, alias: "Vdb Param Color 3d Estimate Color Offset", vr: VR::LO },
            E { tag: 0x200D1015, alias: "Vdb Param Echo 3d Estimate Dimension(2)", vr: VR::LO },
            E { tag: 0x200D1016, alias: "Private Native Threed Data Sequence(1)", vr: VR::SQ },
            E { tag: 0x200D1017, alias: "Threedpr Basic Vol Brightness", vr: VR::IS },
            E { tag: 0x200D1018, alias: "Vdb Param Echo 3d Estimate Apex Position (2)", vr: VR::LO },
            E { tag: 0x200D101A, alias: "Threedpr Echo Smoothing", vr: VR::IS },
            E { tag: 0x200D101B, alias: "Threedpr Color Smoothing", vr: VR::IS },
            E { tag: 0x200D101E, alias: "Threedpr Num Completed Trim Planes", vr: VR::IS },
            E { tag: 0x200D101F, alias: "Threedpr Depth", vr: VR::IS },
            E { tag: 0x200D1020, alias: "Private Native Data Stream Array", vr: VR::SQ },
            E { tag: 0x200D1021, alias: "Threedpr Wall Filter", vr: VR::IS },
            E { tag: 0x200D1023, alias: "Threedpr Baseline", vr: VR::FD },
            E { tag: 0x200D1024, alias: "Threedpr Gain", vr: VR::FD },
            E { tag: 0x200D1025, alias: "Threedpr Compress", vr: VR::FD },
            E { tag: 0x200D1026, alias: "Threedpr Light Brightness", vr: VR::FD },
            E { tag: 0x200D1027, alias: "Threedpr Elevation Scale Factor", vr: VR::FD },
            E { tag: 0x200D1028, alias: "Threedpr Zoom Factor", vr: VR::FD },
            E { tag: 0x200D1029, alias: "Threedpr Color Write Priority", vr: VR::FD },
            E { tag: 0x200D102A, alias: "Threedpr Rotation Angle X", vr: VR::FD },
            E { tag: 0x200D102B, alias: "Threedpr Rotation Angle Y", vr: VR::FD },
            E { tag: 0x200D102C, alias: "Threedpr Data Voi Center", vr: VR::FD },
            E { tag: 0x200D102D, alias: "Threedpr View Translation", vr: VR::FD },
            E { tag: 0x200D102E, alias: "Threedpr Data Voi Min Point", vr: VR::FD },
            E { tag: 0x200D102F, alias: "Threedpr Data Voi Max Point", vr: VR::FD },
            E { tag: 0x200D1030, alias: "Threedpr Box Crop Min Point", vr: VR::FD },
            E { tag: 0x200D1031, alias: "Threedpr Box Crop Max Point", vr: VR::FD },
            E { tag: 0x200D1032, alias: "Threedpr Mpr Rotation Matrix", vr: VR::FD },
            E { tag: 0x200D1033, alias: "Threedpr Vol Rotation Matrix", vr: VR::FD },
            E { tag: 0x200D1034, alias: "Threedpr Trim Plane Equation", vr: VR::FD },
            E { tag: 0x200D1035, alias: "Threedpr Subpage Data Version", vr: VR::IS },
            E { tag: 0x200D1036, alias: "Threedpr Is Arbitrary Crop", vr: VR::IS },
            E { tag: 0x200D1037, alias: "Threedpr Arbitrary Crop Dist From Center", vr: VR::IS },
            E { tag: 0x200D1038, alias: "Threedpr Arbitrary Crop Rotation Matrix", vr: VR::FD },
            E { tag: 0x200D1039, alias: "Threedpr Color Gain", vr: VR::FD },
            E { tag: 0x200D1040, alias: "Threedpr Color Wall Filter Index", vr: VR::IS },
            E { tag: 0x200D1041, alias: "Threedpr Cursor Enabled", vr: VR::IS },
            E { tag: 0x200D1042, alias: "Threedpr Bondbox Graphic Enabled", vr: VR::IS },
        ],
    },
    P {
        creator: "Philips X-ray Imaging DD 001",
        entries: &[
            E { tag: 0x20031000, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20031001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20031002, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20031003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20031004, alias: "Private enhanced Table Sequence", vr: VR::SQ },
            E { tag: 0x20031006, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x20031011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031012, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031013, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031014, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20031015, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20031016, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031017, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031018, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031019, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031022, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031024, alias: "Unknown", vr: VR::FD },
            E { tag: 0x20031025, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x20031027, alias: "Unknown", vr: VR::SH },
            E { tag: 0x20031028, alias: "Unknown", vr: VR::SH },
            E { tag: 0x20031029, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2003102A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x2003102B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x2003102C, alias: "Unknown", vr: VR::SH },
            E { tag: 0x2003102D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x2003102E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x2003102F, alias: "IsResolvePatientMixApplied", vr: VR::CS },
            E { tag: 0x20031030, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20031031, alias: "Unknown", vr: VR::CS },
            E { tag: 0x20031032, alias: "Unknown", vr: VR::UI },
        ],
    },
    P {
        creator: "Picker MR Private Group",
        entries: &[
            E { tag: 0x71011000, alias: "Unknown", vr: VR::OB },
            E { tag: 0x71011001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x71011002, alias: "Unknown", vr: VR::OB },
            E { tag: 0x71011003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x71011004, alias: "Unknown", vr: VR::SH },
            E { tag: 0x71011005, alias: "Unknown", vr: VR::SH },
            E { tag: 0x71011006, alias: "Unknown", vr: VR::SH },
            E { tag: 0x71011010, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "Picker NM Private Group",
        entries: &[
            E { tag: 0x70011001, alias: "Private", vr: VR::UI },
            E { tag: 0x70011002, alias: "Private", vr: VR::OB },
            E { tag: 0x70011003, alias: "Private", vr: VR::OB },
            E { tag: 0x70011004, alias: "Private", vr: VR::OB },
            E { tag: 0x70011005, alias: "Private", vr: VR::OB },
            E { tag: 0x70011006, alias: "Private", vr: VR::OB },
            E { tag: 0x70011007, alias: "Private", vr: VR::OB },
            E { tag: 0x70011008, alias: "Private", vr: VR::OB },
            E { tag: 0x70011009, alias: "Private", vr: VR::OB },
            E { tag: 0x70011010, alias: "Private", vr: VR::SQ },
            E { tag: 0x70011011, alias: "Private", vr: VR::LO },
            E { tag: 0x70011012, alias: "Private", vr: VR::OB },
            E { tag: 0x70011013, alias: "Private", vr: VR::US },
            E { tag: 0x70011014, alias: "Private", vr: VR::OB },
            E { tag: 0x70011015, alias: "Private", vr: VR::OB },
            E { tag: 0x70011016, alias: "Private", vr: VR::OB },
            E { tag: 0x70011017, alias: "Contains a text string that includes some of the same information that appears on the Odyssey Clinical Index (image directory) for this Image.", vr: VR::LO },
            E { tag: 0x70431000, alias: "Private.", vr: VR::SH },
        ],
    },
    P {
        creator: "QCA",
        entries: &[
            E { tag: 0x00091000, alias: "Analysis Type", vr: VR::CS },
            E { tag: 0x00091004, alias: "Segment Name", vr: VR::LO },
            E { tag: 0x00091012, alias: "Pre Catheter size", vr: VR::DS },
            E { tag: 0x00091013, alias: "Pre Reference Diameter", vr: VR::DS },
            E { tag: 0x00091014, alias: "Pre Minimum Lumen Diameter", vr: VR::DS },
            E { tag: 0x00091015, alias: "Pre Average Diameter", vr: VR::DS },
            E { tag: 0x00091016, alias: "Pre Stenosis Length", vr: VR::DS },
            E { tag: 0x00091017, alias: "Pre Stenosis %", vr: VR::DS },
            E { tag: 0x00091018, alias: "Pre Geometric Area Reduction %", vr: VR::DS },
            E { tag: 0x00091022, alias: "Post Catheter Size", vr: VR::DS },
            E { tag: 0x00091023, alias: "Post Reference Diameter", vr: VR::DS },
            E { tag: 0x00091024, alias: "Post Minimum Lumen Diameter", vr: VR::DS },
            E { tag: 0x00091025, alias: "Post Average Diameter", vr: VR::DS },
            E { tag: 0x00091026, alias: "Post Stenosis Length", vr: VR::DS },
            E { tag: 0x00091027, alias: "Post Stenosis %", vr: VR::DS },
            E { tag: 0x00091028, alias: "Post Geometric Area Reduction %", vr: VR::DS },
        ],
    },
    P {
        creator: "QCA_RESULTS",
        entries: &[
            E { tag: 0x00091000, alias: "Analysis Views", vr: VR::CS },
            E { tag: 0x00091010, alias: "Segment", vr: VR::LO },
            E { tag: 0x00091011, alias: "Pre Catheter Name", vr: VR::LO },
            E { tag: 0x00091012, alias: "Pre Catheter Size", vr: VR::DS },
            E { tag: 0x00091013, alias: "Pre Reference Diameter", vr: VR::DS },
            E { tag: 0x00091014, alias: "Pre Minimum Lumen Diameter", vr: VR::DS },
            E { tag: 0x00091015, alias: "Pre Average Diameter", vr: VR::DS },
            E { tag: 0x00091016, alias: "Pre Stenosis Length", vr: VR::DS },
            E { tag: 0x00091017, alias: "Pre Stenosis %", vr: VR::IS },
            E { tag: 0x00091018, alias: "Pre Geometric Area Reduction %", vr: VR::IS },
            E { tag: 0x00091021, alias: "Post Catheter Name", vr: VR::LO },
            E { tag: 0x00091022, alias: "Post Catheter Size", vr: VR::DS },
            E { tag: 0x00091023, alias: "Post Reference Diameter", vr: VR::DS },
            E { tag: 0x00091024, alias: "Post Minimum Lumen Diameter", vr: VR::DS },
            E { tag: 0x00091025, alias: "Post Average Diameter", vr: VR::DS },
            E { tag: 0x00091026, alias: "Post Stenosis Length", vr: VR::DS },
            E { tag: 0x00091027, alias: "Post Stenosis %", vr: VR::IS },
            E { tag: 0x00091028, alias: "Post Geometric Area Reduction %", vr: VR::IS },
            E { tag: 0x00091040, alias: "Calibration Frame", vr: VR::IS },
            E { tag: 0x00091041, alias: "End Diastolic Frame", vr: VR::IS },
            E { tag: 0x00091042, alias: "End Systolic Frame", vr: VR::IS },
            E { tag: 0x00091043, alias: "End Diastolic Volume", vr: VR::DS },
            E { tag: 0x00091044, alias: "End Systolic Volume", vr: VR::DS },
            E { tag: 0x00091045, alias: "Stroke Volume", vr: VR::DS },
            E { tag: 0x00091047, alias: "Ejection Fraction", vr: VR::DS },
            E { tag: 0x00091048, alias: "Body Surface Area", vr: VR::DS },
            E { tag: 0x00091049, alias: "Artery Territory Region", vr: VR::SH },
            E { tag: 0x00091050, alias: "Number of Diseased Vessels", vr: VR::IS },
            E { tag: 0x00091051, alias: "Hypokinesis in Region", vr: VR::DS },
            E { tag: 0x00091052, alias: "Hyperkinesis in Opposite Region", vr: VR::DS },
            E { tag: 0x00091053, alias: "Percent Total LV Hypokinesis", vr: VR::IS },
            E { tag: 0x00091055, alias: "Calibration Factor", vr: VR::DS },
        ],
    },
    P {
        creator: "QUASAR_INTERNAL_USE",
        entries: &[
            E { tag: 0x00091001, alias: "Rate Vector", vr: VR::UL },
            E { tag: 0x00091002, alias: "Count Vector", vr: VR::UL },
            E { tag: 0x00091003, alias: "Time Vector", vr: VR::UL },
            E { tag: 0x00091007, alias: "Angle Vector", vr: VR::US },
            E { tag: 0x00091008, alias: "Camera Shape", vr: VR::US },
            E { tag: 0x00091010, alias: "WholeBody Spots", vr: VR::US },
            E { tag: 0x00091011, alias: "Worklist Flag", vr: VR::US },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091013, alias: "Sequence Type", vr: VR::ST },
            E { tag: 0x00091014, alias: "Sequence Name", vr: VR::ST },
            E { tag: 0x00091015, alias: "Avr RR Time Vector", vr: VR::UL },
            E { tag: 0x00091016, alias: "Low Limit Vector", vr: VR::UL },
            E { tag: 0x00091017, alias: "High Limit Vector", vr: VR::UL },
            E { tag: 0x00091018, alias: "Begin Index Vector", vr: VR::UL },
            E { tag: 0x00091019, alias: "End Index Vector", vr: VR::UL },
            E { tag: 0x0009101A, alias: "Raw Time Vector", vr: VR::UL },
            E { tag: 0x0009101B, alias: "Image Type String", vr: VR::LO },
            E { tag: 0x0009101D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0009101E, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00091022, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00091023, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091039, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00091040, alias: "Unknown", vr: VR::DA },
            E { tag: 0x00091041, alias: "Unknown", vr: VR::TM },
            E { tag: 0x00091042, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091044, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00371010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0037101B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00371030, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00371040, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00371050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00371060, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00371070, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "QVA",
        entries: &[
            E { tag: 0x00091040, alias: "Calibration Frame", vr: VR::IS },
            E { tag: 0x00091041, alias: "End Diastolic Frame", vr: VR::IS },
            E { tag: 0x00091042, alias: "End Systolic Frame", vr: VR::IS },
            E { tag: 0x00091043, alias: "End Diastolic Volume", vr: VR::DS },
            E { tag: 0x00091044, alias: "End Systolic Volume", vr: VR::DS },
            E { tag: 0x00091045, alias: "Stroke Volume", vr: VR::DS },
            E { tag: 0x00091046, alias: "Cardiac Output", vr: VR::DS },
            E { tag: 0x00091047, alias: "Ejection Fraction", vr: VR::DS },
            E { tag: 0x00091048, alias: "Body Surface Area", vr: VR::DS },
            E { tag: 0x00091049, alias: "Artery Territory Region", vr: VR::SH },
            E { tag: 0x00091050, alias: "Number of Diseased Vessels", vr: VR::IS },
            E { tag: 0x00091051, alias: "Hypokinesis in Region", vr: VR::DS },
            E { tag: 0x00091052, alias: "Hyperkinesis in Region", vr: VR::DS },
            E { tag: 0x00091053, alias: "percent of chords with hyperkinesis <- 2 SD", vr: VR::IS },
            E { tag: 0x00091054, alias: "percent of chords with alkinesis/dyskinesis", vr: VR::IS },
            E { tag: 0x00091055, alias: "Calibration Factor", vr: VR::DS },
        ],
    },
    P {
        creator: "REPORT_FROM_APP",
        entries: &[
            E { tag: 0x00391095, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "RadWorksMarconi",
        entries: &[
            E { tag: 0x00291024, alias: "<internal", vr: VR::US },
        ],
    },
    P {
        creator: "RadWorksTBR",
        entries: &[
            E { tag: 0x31111002, alias: "AMI Compression Type (RET)", vr: VR::CS },
            E { tag: 0x311110FF, alias: "<internal", vr: VR::SQ },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Change Item Creator ID",
        entries: &[
            E { tag: 0x00211001, alias: "Change List Item UID", vr: VR::UI },
            E { tag: 0x00211002, alias: "Operation", vr: VR::US },
            E { tag: 0x00211003, alias: "Date", vr: VR::DT },
            E { tag: 0x00211004, alias: "User Name", vr: VR::PN },
            E { tag: 0x00211005, alias: "Additional Change Data", vr: VR::OB },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Change List Creator ID",
        entries: &[
            E { tag: 0x00211001, alias: "Reference Image Instance UID", vr: VR::UI },
            E { tag: 0x00211002, alias: "Sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Image Security Creator ID",
        entries: &[
            E { tag: 0x00291001, alias: "Image Checksum", vr: VR::UL },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Note Item Creator ID",
        entries: &[
            E { tag: 0x00211001, alias: "Text", vr: VR::ST },
            E { tag: 0x00211002, alias: "Date", vr: VR::DT },
            E { tag: 0x00211003, alias: "User Name", vr: VR::PN },
            E { tag: 0x00211004, alias: "Coordinates (X, Y)", vr: VR::US },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Note List Creator ID",
        entries: &[
            E { tag: 0x00211001, alias: "Reference Image Instance UID", vr: VR::UI },
            E { tag: 0x00211002, alias: "Item", vr: VR::SQ },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Viewset Creator ID",
        entries: &[
            E { tag: 0x00091001, alias: "Instance UID", vr: VR::UI },
            E { tag: 0x00091002, alias: "Name", vr: VR::LO },
            E { tag: 0x00091003, alias: "Sequence", vr: VR::SQ },
            E { tag: 0x00091004, alias: "Mode", vr: VR::UL },
            E { tag: 0x00091005, alias: "Type", vr: VR::UL },
        ],
    },
    P {
        creator: "SCHICK TECHNOLOGIES - Viewset Item Creator ID",
        entries: &[
            E { tag: 0x00091001, alias: "Image Instance UID", vr: VR::UI },
            E { tag: 0x00091002, alias: "Number", vr: VR::US },
            E { tag: 0x00091003, alias: "Order", vr: VR::US },
            E { tag: 0x00091004, alias: "Type", vr: VR::US },
            E { tag: 0x00091005, alias: "Orientation", vr: VR::US },
            E { tag: 0x00091006, alias: "Rect (Left, Top, Width, Height)", vr: VR::US },
            E { tag: 0x00091007, alias: "Caption Height", vr: VR::US },
            E { tag: 0x00091008, alias: "Source", vr: VR::US },
            E { tag: 0x00091009, alias: "Default Width", vr: VR::US },
            E { tag: 0x0009100A, alias: "Default Height", vr: VR::US },
        ],
    },
    P {
        creator: "SECTRA_Ident_01",
        entries: &[
            E { tag: 0x00091001, alias: "Request number", vr: VR::LO },
            E { tag: 0x00091002, alias: "Examination number", vr: VR::LO },
            E { tag: 0x00091004, alias: "Series ID", vr: VR::LO },
            E { tag: 0x00091005, alias: "Series Order", vr: VR::LO },
            E { tag: 0x00091006, alias: "File Name", vr: VR::LO },
            E { tag: 0x00091007, alias: "Image Data ID", vr: VR::LO },
        ],
    },
    P {
        creator: "SECTRA_ImageInfo_01",
        entries: &[
            E { tag: 0x00291001, alias: "Image info", vr: VR::OB },
            E { tag: 0x00291002, alias: "Marking", vr: VR::CS },
            E { tag: 0x00291003, alias: "No Decompression", vr: VR::LO },
            E { tag: 0x00291004, alias: "Image Info new", vr: VR::OB },
        ],
    },
    P {
        creator: "SECTRA_OverlayInfo_01",
        entries: &[
            E { tag: 0x60011001, alias: "Sectra Overlay", vr: VR::LO },
        ],
    },
    P {
        creator: "SEGAMI MIML",
        entries: &[
            E { tag: 0x00311098, alias: "?Binary XML with PHI?", vr: VR::OW },
        ],
    },
    P {
        creator: "SEGAMI__MEMO",
        entries: &[
            E { tag: 0x00351097, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00351098, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "SET WINDOW",
        entries: &[
            E { tag: 0x00191000, alias: "Set Window Image Filter", vr: VR::UN },
            E { tag: 0x00191001, alias: "Set Window Magnification Power", vr: VR::UN },
        ],
    },
    P {
        creator: "SHS MagicView 300",
        entries: &[
            E { tag: 0x00291001, alias: "Zoom Factor", vr: VR::FD },
            E { tag: 0x00291002, alias: "Pan Factor X", vr: VR::FD },
            E { tag: 0x00291003, alias: "Pan Factor Y", vr: VR::FD },
            E { tag: 0x00291004, alias: "NmPalette", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS CM VA0  ACQU",
        entries: &[
            E { tag: 0x00191010, alias: "Parameter File Name", vr: VR::LO },
            E { tag: 0x00191011, alias: "Sequence File Name", vr: VR::LO },
            E { tag: 0x00191012, alias: "Sequence File Owner", vr: VR::LO },
            E { tag: 0x00191013, alias: "Sequence Description", vr: VR::LO },
            E { tag: 0x00191014, alias: "EPI File Name", vr: VR::LO },
            E { tag: 0x00191020, alias: "Mode File Name", vr: VR::LO },
            E { tag: 0x00191030, alias: "Convolution Kernel Comment 1", vr: VR::LO },
            E { tag: 0x00191031, alias: "Convolution Kernel Comment 2", vr: VR::LO },
            E { tag: 0x00191032, alias: "Convolution Kernel Comment 3", vr: VR::LO },
            E { tag: 0x00191033, alias: "Convolution Kernel Comment 4", vr: VR::LO },
            E { tag: 0x00191034, alias: "Convolution Kernel Comment 5", vr: VR::LO },
            E { tag: 0x00191035, alias: "Convolution Kernel Comment 6", vr: VR::LO },
            E { tag: 0x00191036, alias: "Convolution Kernel Comment 7", vr: VR::LO },
            E { tag: 0x00191037, alias: "Convolution Kernel Comment 8", vr: VR::LO },
            E { tag: 0x00191038, alias: "Convolution Kernel Comment 9", vr: VR::LO },
            E { tag: 0x00191039, alias: "Convolution Kernel Comment 10", vr: VR::LO },
            E { tag: 0x0019103A, alias: "Reading Code RZ1 Comment 1", vr: VR::LO },
            E { tag: 0x0019103B, alias: "Reading Code RZ1 Comment 2", vr: VR::LO },
            E { tag: 0x0019103C, alias: "Reading Code RZ2 Comment 1", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS CM VA0  CMS",
        entries: &[
            E { tag: 0x00091000, alias: "Number of Measurements", vr: VR::IS },
            E { tag: 0x00091010, alias: "Storage Mode", vr: VR::CS },
            E { tag: 0x00091012, alias: "Evaluation Mask Image", vr: VR::UL },
            E { tag: 0x00091026, alias: "Table Zero Date", vr: VR::DA },
            E { tag: 0x00091027, alias: "Table Zero Time", vr: VR::TM },
            E { tag: 0x0011100A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00111010, alias: "Registration Date", vr: VR::DA },
            E { tag: 0x00111011, alias: "Registration Time", vr: VR::TM },
            E { tag: 0x00111022, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00111023, alias: "Used Patient Weight", vr: VR::DS },
            E { tag: 0x00111040, alias: "Organ Code", vr: VR::IS },
            E { tag: 0x00131000, alias: "Modifying Physician", vr: VR::PN },
            E { tag: 0x00131010, alias: "Modification Date", vr: VR::DA },
            E { tag: 0x00131012, alias: "Modification Time", vr: VR::TM },
            E { tag: 0x00131020, alias: "Patient Name", vr: VR::PN },
            E { tag: 0x00131022, alias: "Patient Id", vr: VR::LO },
            E { tag: 0x00131030, alias: "Patient Birthdate", vr: VR::DA },
            E { tag: 0x00131031, alias: "Patient Weight", vr: VR::DS },
            E { tag: 0x00131032, alias: "Patient Maiden Name", vr: VR::PN },
            E { tag: 0x00131033, alias: "Referring Physician", vr: VR::PN },
            E { tag: 0x00131034, alias: "Admitting Diagnosis", vr: VR::LO },
            E { tag: 0x00131035, alias: "Patient Sex", vr: VR::CS },
            E { tag: 0x00131040, alias: "Procedure Description", vr: VR::LO },
            E { tag: 0x00131042, alias: "Rest Direction", vr: VR::CS },
            E { tag: 0x00131044, alias: "Patient Position", vr: VR::CS },
            E { tag: 0x00131046, alias: "View Direction", vr: VR::SH },
            E { tag: 0x00131050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131053, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131054, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131055, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00131056, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191010, alias: "Net Frequency", vr: VR::IS },
            E { tag: 0x00191020, alias: "Measurement Mode", vr: VR::CS },
            E { tag: 0x00191030, alias: "Calculation Mode", vr: VR::CS },
            E { tag: 0x00191050, alias: "Noise Level", vr: VR::IS },
            E { tag: 0x00191060, alias: "Number of Data Bytes", vr: VR::IS },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191080, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211020, alias: "Field of View", vr: VR::DS },
            E { tag: 0x00211022, alias: "Image Magnification Factor", vr: VR::DS },
            E { tag: 0x00211024, alias: "Image Scroll Offset", vr: VR::DS },
            E { tag: 0x00211026, alias: "Image Pixel Offset", vr: VR::IS },
            E { tag: 0x00211030, alias: "View Direction", vr: VR::CS },
            E { tag: 0x00211032, alias: "Rest Direction", vr: VR::CS },
            E { tag: 0x00211060, alias: "Image Position", vr: VR::DS },
            E { tag: 0x00211061, alias: "Image Normal", vr: VR::DS },
            E { tag: 0x00211063, alias: "Image Distance", vr: VR::DS },
            E { tag: 0x00211065, alias: "Image Positioning History Mask", vr: VR::US },
            E { tag: 0x0021106A, alias: "Image Row", vr: VR::DS },
            E { tag: 0x0021106B, alias: "Image Column", vr: VR::DS },
            E { tag: 0x00211070, alias: "Patient Orientation Set 1", vr: VR::CS },
            E { tag: 0x00211071, alias: "Patient Orientation Set 2", vr: VR::CS },
            E { tag: 0x00211080, alias: "Study Name", vr: VR::LO },
            E { tag: 0x00211082, alias: "Study Type", vr: VR::SH },
            E { tag: 0x00291010, alias: "Window Style", vr: VR::CS },
            E { tag: 0x00291011, alias: "Window Look", vr: VR::CS },
            E { tag: 0x00291012, alias: "Window Table Index", vr: VR::UL },
            E { tag: 0x00291013, alias: "Window Scale", vr: VR::CS },
            E { tag: 0x00291020, alias: "Pixel Quality Code", vr: VR::CS },
            E { tag: 0x00291022, alias: "Pixel Quality Value", vr: VR::IS },
            E { tag: 0x00291050, alias: "Archive Code", vr: VR::CS },
            E { tag: 0x00291051, alias: "Exposure Code", vr: VR::CS },
            E { tag: 0x00291052, alias: "Sort Code", vr: VR::IS },
            E { tag: 0x00291053, alias: "Remote Copy Code", vr: VR::CS },
            E { tag: 0x00291060, alias: "Splash", vr: VR::IS },
            E { tag: 0x00511010, alias: "Image Text", vr: VR::LO },
            E { tag: 0x60211000, alias: "Image Graphics Format Code", vr: VR::CS },
            E { tag: 0x60211010, alias: "Image Graphics", vr: VR::OB },
            E { tag: 0x7FE11000, alias: "Binary Data", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS CM VA0  COAD",
        entries: &[
            E { tag: 0x0019103D, alias: "Reading Code RZ2 Comment 2", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS CM VA0  LAB",
        entries: &[
            E { tag: 0x00091010, alias: "Generator Identification Label", vr: VR::LO },
            E { tag: 0x00091011, alias: "Gantry Identification Label", vr: VR::LO },
            E { tag: 0x00091012, alias: "XRay Tube Identification Label", vr: VR::LO },
            E { tag: 0x00091013, alias: "Detector Identification Label", vr: VR::LO },
            E { tag: 0x00091014, alias: "DAS Identification Label", vr: VR::LO },
            E { tag: 0x00091015, alias: "SMI Identification Label", vr: VR::LO },
            E { tag: 0x00091016, alias: "CPU Identification Label", vr: VR::LO },
            E { tag: 0x00091020, alias: "Header Version", vr: VR::SH },
        ],
    },
    P {
        creator: "SIEMENS CM VA0  OST",
        entries: &[
            E { tag: 0x60231000, alias: "Osteo Contour Comment", vr: VR::LO },
            E { tag: 0x60231010, alias: "Osteo Contour Buffer", vr: VR::SS },
        ],
    },
    P {
        creator: "SIEMENS CSA ENVELOPE",
        entries: &[
            E { tag: 0x00291010, alias: "syngo Report Data", vr: VR::OB },
            E { tag: 0x00291011, alias: "syngo Report Presentation", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS CSA HEADER",
        entries: &[
            E { tag: 0x00291008, alias: "CSA Image Header Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "CSA Image Header Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "CSA Image Header Info", vr: VR::OB },
            E { tag: 0x00291018, alias: "CSA Series Header Type", vr: VR::CS },
            E { tag: 0x00291019, alias: "CSA Series Header Version", vr: VR::LO },
            E { tag: 0x00291020, alias: "CSA Series Header Info", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS CSA NON-IMAGE",
        entries: &[
            E { tag: 0x00291008, alias: "CSA Data Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "CSA Data Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "CSA Data Info", vr: VR::OB },
            E { tag: 0x7FE11010, alias: "CSA Data", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS CSA REPORT",
        entries: &[
            E { tag: 0x00291008, alias: "syngo Report Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "syngo Report", vr: VR::LO },
            E { tag: 0x00291015, alias: "SR Variant", vr: VR::US },
            E { tag: 0x00291017, alias: "SC SOP Instance UID", vr: VR::UI },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  COAD",
        entries: &[
            E { tag: 0x00191010, alias: "Distance Source to SourceSide Collimator", vr: VR::IS },
            E { tag: 0x00191011, alias: "Distance Source to DetectorSide Collimator", vr: VR::IS },
            E { tag: 0x00191020, alias: "Number of Possible Channels", vr: VR::IS },
            E { tag: 0x00191021, alias: "Mean Channel Number", vr: VR::IS },
            E { tag: 0x00191022, alias: "Detector Spacing", vr: VR::DS },
            E { tag: 0x00191023, alias: "Detector Center", vr: VR::DS },
            E { tag: 0x00191024, alias: "Reading Integration Time", vr: VR::DS },
            E { tag: 0x00191050, alias: "Detector Alignment", vr: VR::DS },
            E { tag: 0x00191052, alias: "Table Feed", vr: VR::DS },
            E { tag: 0x00191054, alias: "Rotation Time", vr: VR::DS },
            E { tag: 0x00191060, alias: "Focus Alignment", vr: VR::DS },
            E { tag: 0x00191065, alias: "Focal Spot Deflection Amplitude", vr: VR::UL },
            E { tag: 0x00191066, alias: "Focal Spot Deflection Phase", vr: VR::UL },
            E { tag: 0x00191067, alias: "Focal Spot Deflection Offset", vr: VR::UL },
            E { tag: 0x00191070, alias: "Water Scaling Factor", vr: VR::DS },
            E { tag: 0x00191071, alias: "Interpolation Factor", vr: VR::DS },
            E { tag: 0x00191080, alias: "Patient Region", vr: VR::CS },
            E { tag: 0x00191082, alias: "Patient Phase of Life", vr: VR::CS },
            E { tag: 0x00191090, alias: "Osteo Offset", vr: VR::DS },
            E { tag: 0x00191092, alias: "Osteo Regression Line Slope", vr: VR::DS },
            E { tag: 0x00191093, alias: "Osteo Regression Line Intercept", vr: VR::DS },
            E { tag: 0x00191094, alias: "Osteo Standardization Code", vr: VR::IS },
            E { tag: 0x00191096, alias: "Osteo Phantom Number", vr: VR::IS },
            E { tag: 0x001910A0, alias: "Spiral Location Begin", vr: VR::DS },
            E { tag: 0x001910A1, alias: "Spiral Location End", vr: VR::DS },
            E { tag: 0x001910A2, alias: "Spiral z Profile", vr: VR::CS },
            E { tag: 0x001910A3, alias: "Spiral Reconstruction Direction", vr: VR::CS },
            E { tag: 0x001910A4, alias: "Spiral Reconstruction Increment", vr: VR::DS },
            E { tag: 0x001910A5, alias: "Scan Tilt Plane Distance", vr: VR::DS },
            E { tag: 0x001910A6, alias: "Mode Processing Step Mask", vr: VR::UL },
            E { tag: 0x001910A7, alias: "Mode Processing Control Mask", vr: VR::UL },
            E { tag: 0x001910A8, alias: "Mode Master Mask", vr: VR::UL },
            E { tag: 0x001910A9, alias: "Mode Store Step", vr: VR::IS },
            E { tag: 0x001910AA, alias: "Focus Size Type", vr: VR::CS },
            E { tag: 0x001910AB, alias: "Mode Adaptive Filter Kind", vr: VR::IS },
            E { tag: 0x001910AC, alias: "Mode MBH Kind", vr: VR::IS },
            E { tag: 0x001910AD, alias: "Mode Balancing Kind", vr: VR::IS },
            E { tag: 0x001910AE, alias: "Number of Readings per Rotation", vr: VR::IS },
            E { tag: 0x001910AF, alias: "Correction Table Reference Time", vr: VR::DS },
            E { tag: 0x001910B0, alias: "Feed per Rotation", vr: VR::DS },
            E { tag: 0x001910B1, alias: "Unknown", vr: VR::UN },
            E { tag: 0x001910BD, alias: "Pulmo Trigger Level", vr: VR::IS },
            E { tag: 0x001910BE, alias: "Expiratoric Reserve", vr: VR::DS },
            E { tag: 0x001910BF, alias: "Vital Capacity", vr: VR::DS },
            E { tag: 0x001910C0, alias: "Pulmo Water", vr: VR::DS },
            E { tag: 0x001910C1, alias: "Pulmo Air", vr: VR::DS },
            E { tag: 0x001910C2, alias: "Pulmo Date", vr: VR::DA },
            E { tag: 0x001910C3, alias: "Pulmo Time", vr: VR::TM },
            E { tag: 0x001910C4, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001910C5, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  GEN",
        entries: &[
            E { tag: 0x00191010, alias: "Source Side Collimator Aperture", vr: VR::DS },
            E { tag: 0x00191011, alias: "Detector Side Collimator Aperture", vr: VR::DS },
            E { tag: 0x00191020, alias: "Exposure Time current", vr: VR::IS },
            E { tag: 0x00191021, alias: "Exposure current", vr: VR::IS },
            E { tag: 0x00191025, alias: "Generator Power current", vr: VR::DS },
            E { tag: 0x00191026, alias: "Generator Voltage current", vr: VR::DS },
            E { tag: 0x00191040, alias: "Master Control Mask", vr: VR::UL },
            E { tag: 0x00191042, alias: "Processing Mask", vr: VR::US },
            E { tag: 0x00191044, alias: "IRS Data Number", vr: VR::UL },
            E { tag: 0x00191045, alias: "IRS Data Length", vr: VR::UL },
            E { tag: 0x00191062, alias: "Number of Virtual Channels", vr: VR::IS },
            E { tag: 0x00191070, alias: "Number of Reading", vr: VR::IS },
            E { tag: 0x00191071, alias: "Reading Code", vr: VR::CS },
            E { tag: 0x00191074, alias: "Number of Projections", vr: VR::IS },
            E { tag: 0x00191075, alias: "Number of Bytes", vr: VR::IS },
            E { tag: 0x00191080, alias: "Reconstruction Algorithm Set", vr: VR::IS },
            E { tag: 0x00191081, alias: "Reconstruction Algorithm Index", vr: VR::IS },
            E { tag: 0x00191082, alias: "Regeneration Software Version", vr: VR::LO },
            E { tag: 0x00191084, alias: "Multiscan Reconstruction Increment", vr: VR::IS },
            E { tag: 0x00191085, alias: "Multiscan Reconstruction Width", vr: VR::IS },
            E { tag: 0x00191088, alias: "Range Number", vr: VR::IS },
            E { tag: 0x00211010, alias: "Rotation Angle", vr: VR::IS },
            E { tag: 0x00211011, alias: "Start Angle", vr: VR::IS },
            E { tag: 0x00211020, alias: "Scan", vr: VR::IS },
            E { tag: 0x00211030, alias: "Tube Position", vr: VR::IS },
            E { tag: 0x00211032, alias: "Length of Topogram", vr: VR::DS },
            E { tag: 0x00211034, alias: "Correction Factor", vr: VR::DS },
            E { tag: 0x00211036, alias: "Maximum Table Position", vr: VR::DS },
            E { tag: 0x00211040, alias: "Table Move Direction Code", vr: VR::IS },
            E { tag: 0x00211045, alias: "VOI Start Row", vr: VR::IS },
            E { tag: 0x00211046, alias: "VOI Stop Row", vr: VR::IS },
            E { tag: 0x00211047, alias: "VOI Start Column", vr: VR::IS },
            E { tag: 0x00211048, alias: "VOI Stop Column", vr: VR::IS },
            E { tag: 0x00211049, alias: "VOI Start Slice", vr: VR::IS },
            E { tag: 0x0021104A, alias: "VOI Stop Slice", vr: VR::IS },
            E { tag: 0x00211050, alias: "Vector Start Row", vr: VR::IS },
            E { tag: 0x00211051, alias: "Vector Row Step", vr: VR::IS },
            E { tag: 0x00211052, alias: "Vector Start Column", vr: VR::IS },
            E { tag: 0x00211053, alias: "Vector Column Step", vr: VR::IS },
            E { tag: 0x00211060, alias: "Range Type Code", vr: VR::IS },
            E { tag: 0x00211062, alias: "Reference Type Code", vr: VR::IS },
            E { tag: 0x00211070, alias: "Object Orientation", vr: VR::DS },
            E { tag: 0x00211072, alias: "Light Orientation", vr: VR::DS },
            E { tag: 0x00211075, alias: "Light Brightness", vr: VR::DS },
            E { tag: 0x00211076, alias: "Light Contrast", vr: VR::DS },
            E { tag: 0x0021107A, alias: "Overlay Threshold", vr: VR::IS },
            E { tag: 0x0021107B, alias: "Surface Threshold", vr: VR::IS },
            E { tag: 0x0021107C, alias: "Grey Scale Threshold", vr: VR::IS },
            E { tag: 0x002110A0, alias: "Field of Interest", vr: VR::IS },
            E { tag: 0x002110A1, alias: "Image Filter Label", vr: VR::CS },
            E { tag: 0x002110A2, alias: "Exemplary Element 1", vr: VR::CS },
            E { tag: 0x002110A3, alias: "Exemplary Element 2", vr: VR::IS },
            E { tag: 0x002110A4, alias: "Exemplary Element 3", vr: VR::LO },
            E { tag: 0x002110A5, alias: "Reference Data Set 1", vr: VR::IS },
            E { tag: 0x002110A6, alias: "Reference Data Set 2", vr: VR::IS },
            E { tag: 0x002110A7, alias: "Window Program Name", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  IDE",
        entries: &[
            E { tag: 0x00091010, alias: "SOMATOM Software Name", vr: VR::LO },
            E { tag: 0x00091030, alias: "Data Set Subtype 1", vr: VR::CS },
            E { tag: 0x00091031, alias: "Data Set Subtype 2", vr: VR::SH },
            E { tag: 0x00091032, alias: "Data Set Subtype 3", vr: VR::SH },
            E { tag: 0x00091033, alias: "Data Set Subtype 4", vr: VR::SH },
            E { tag: 0x00091034, alias: "Data Set Subtype 5", vr: VR::SH },
            E { tag: 0x00091040, alias: "Directory Text", vr: VR::LO },
            E { tag: 0x00091042, alias: "Exposure Time Text", vr: VR::LO },
            E { tag: 0x00091050, alias: "Scan Date", vr: VR::DA },
            E { tag: 0x00091051, alias: "Scan Time", vr: VR::TM },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  ORI",
        entries: &[
            E { tag: 0x00091020, alias: "Original Data Set Owner", vr: VR::LO },
            E { tag: 0x00091030, alias: "Original Modality", vr: VR::SH },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  ORMR",
        entries: &[
            E { tag: 0x00211080, alias: "Repetition Time", vr: VR::DS },
            E { tag: 0x00211081, alias: "Echo Time", vr: VR::DS },
            E { tag: 0x00211082, alias: "Inversion Time", vr: VR::DS },
            E { tag: 0x00211083, alias: "Number of Averages", vr: VR::IS },
            E { tag: 0x00211084, alias: "Imaging Frequency", vr: VR::DS },
            E { tag: 0x00211085, alias: "Imaged Nucleus", vr: VR::LO },
            E { tag: 0x00211086, alias: "Echo Number", vr: VR::IS },
            E { tag: 0x00211090, alias: "Flip Angle", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  OST",
        entries: &[
            E { tag: 0x60211000, alias: "Osteo Contour Comment", vr: VR::LO },
            E { tag: 0x60211010, alias: "Osteo Contour Buffer", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS CT VA0  RAW",
        entries: &[
            E { tag: 0x00211010, alias: "Creation Mask service", vr: VR::UL },
            E { tag: 0x00211020, alias: "Evaluation Mask service", vr: VR::UL },
            E { tag: 0x00211030, alias: "Extended Processing Mask", vr: VR::US },
            E { tag: 0x00211040, alias: "Manipulation Mask", vr: VR::UL },
            E { tag: 0x00211041, alias: "Composition Mask", vr: VR::UL },
            E { tag: 0x00211042, alias: "Alteration Mask", vr: VR::UL },
            E { tag: 0x00211043, alias: "Generation Mask", vr: VR::UL },
            E { tag: 0x00211044, alias: "Function Mask", vr: VR::UL },
            E { tag: 0x00211050, alias: "Mode Mirroring Code", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS DFR.01",
        entries: &[
            E { tag: 0x00211010, alias: "Graphic Evaluation Mask", vr: VR::US },
            E { tag: 0x00211011, alias: "Type 1st Graphic Element", vr: VR::US },
            E { tag: 0x00211012, alias: "Description 1st Graphic Element", vr: VR::LO },
            E { tag: 0x00211013, alias: "Type 2nd Graphic Element", vr: VR::US },
            E { tag: 0x00211014, alias: "Description 2nd Graphic Element", vr: VR::LO },
            E { tag: 0x00211015, alias: "Type 3rd Graphic Element", vr: VR::US },
            E { tag: 0x00211016, alias: "Description 3rd Graphic Element", vr: VR::LO },
            E { tag: 0x00211017, alias: "Type 4th Graphic Element", vr: VR::US },
            E { tag: 0x00211018, alias: "Description 4th Graphic Element", vr: VR::LO },
            E { tag: 0x00211019, alias: "Type 5th Graphic Element", vr: VR::US },
            E { tag: 0x00211020, alias: "Description 5th Graphic Element", vr: VR::LO },
            E { tag: 0x00211021, alias: "Type 6th Graphic Element", vr: VR::US },
            E { tag: 0x00211022, alias: "Description 6th Graphic Element", vr: VR::LO },
            E { tag: 0x00211023, alias: "Type 7th Graphic Element", vr: VR::US },
            E { tag: 0x00211024, alias: "Description 7th Graphic Element", vr: VR::LO },
            E { tag: 0x00211025, alias: "Type 8th Graphic Element", vr: VR::US },
            E { tag: 0x00211026, alias: "Description 8th Graphic Element", vr: VR::LO },
            E { tag: 0x00211027, alias: "Type 9th Graphic Element", vr: VR::US },
            E { tag: 0x00211028, alias: "Description 9th Graphic Element", vr: VR::LO },
            E { tag: 0x00211029, alias: "Type 10th Graphic Element", vr: VR::US },
            E { tag: 0x00211030, alias: "Description 10th Graphic Element", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS DFR.01 MANIPULATED",
        entries: &[
            E { tag: 0x00171011, alias: "Window Width Native Current", vr: VR::US },
            E { tag: 0x00171012, alias: "Window Center Native", vr: VR::US },
            E { tag: 0x00171014, alias: "Window Width Subtraction Current", vr: VR::US },
            E { tag: 0x00171015, alias: "Window Center Subtraction Current", vr: VR::US },
            E { tag: 0x00171025, alias: "Edge Enhancement Native Current", vr: VR::DS },
            E { tag: 0x00171027, alias: "Edge Enhancement Subtraction Current", vr: VR::DS },
            E { tag: 0x00171029, alias: "Edge Enhancement %", vr: VR::IS },
            E { tag: 0x00171030, alias: "Harmonization %", vr: VR::IS },
            E { tag: 0x00171031, alias: "DSA Representation Current", vr: VR::US },
            E { tag: 0x00171032, alias: "Number of Mask Current", vr: VR::US },
            E { tag: 0x00171033, alias: "RightLeft Mark Current", vr: VR::US },
            E { tag: 0x00171035, alias: "Number of Series Representative", vr: VR::US },
            E { tag: 0x00171037, alias: "Rectangle Shutter Horizontal Current", vr: VR::US },
            E { tag: 0x00171038, alias: "Rectangle Shutter Vertical Aktuell", vr: VR::US },
            E { tag: 0x00171071, alias: "Landmark Current", vr: VR::LO },
            E { tag: 0x00171072, alias: "Zoom Factor", vr: VR::US },
            E { tag: 0x00171073, alias: "Roam Horizontal", vr: VR::US },
            E { tag: 0x00171074, alias: "Roam Vertical", vr: VR::US },
            E { tag: 0x00171077, alias: "Pixel Shift Horizontal", vr: VR::IS },
            E { tag: 0x00171078, alias: "Pixel Shift Vertical", vr: VR::IS },
            E { tag: 0x00171079, alias: "Image Flip", vr: VR::US },
            E { tag: 0x0017107A, alias: "Invers Current", vr: VR::US },
            E { tag: 0x00171080, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171083, alias: "Documentation Flag", vr: VR::SH },
            E { tag: 0x00171084, alias: "Right Marker", vr: VR::LO },
            E { tag: 0x00171091, alias: "Summation Count Filling", vr: VR::US },
            E { tag: 0x00171092, alias: "Summation Count Mask", vr: VR::US },
            E { tag: 0x00171093, alias: "PeakOp Start", vr: VR::US },
            E { tag: 0x00171094, alias: "PeakOp End", vr: VR::US },
            E { tag: 0x001710A0, alias: "Cine Rate", vr: VR::IS },
            E { tag: 0x001710A1, alias: "Image Name", vr: VR::LO },
            E { tag: 0x001710A2, alias: "Image Name Extension 1", vr: VR::LO },
            E { tag: 0x001710A3, alias: "Image Name Extension 2", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS DFR.01 ORIGINAL",
        entries: &[
            E { tag: 0x00171010, alias: "Image Type", vr: VR::CS },
            E { tag: 0x00171011, alias: "Window Width Native Original", vr: VR::US },
            E { tag: 0x00171012, alias: "Window Center Native", vr: VR::US },
            E { tag: 0x00171014, alias: "Window Width Subtraction Original", vr: VR::US },
            E { tag: 0x00171015, alias: "Window Center Subtraction Original", vr: VR::US },
            E { tag: 0x00171016, alias: "LUT Number for Autowindow", vr: VR::US },
            E { tag: 0x00171018, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171021, alias: "Edge Enhancement Kernelsize Original", vr: VR::US },
            E { tag: 0x00171022, alias: "Edge Enhancement Mode Original", vr: VR::US },
            E { tag: 0x00171023, alias: "Edge Enhancement LUTs Original", vr: VR::US },
            E { tag: 0x00171024, alias: "Filter Type Native Original", vr: VR::US },
            E { tag: 0x00171025, alias: "Edge Enhancement Native Original", vr: VR::DS },
            E { tag: 0x00171026, alias: "Filter Type Subtraction Original", vr: VR::US },
            E { tag: 0x00171027, alias: "Edge Enhancement Subtraction Original", vr: VR::DS },
            E { tag: 0x0017102A, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171030, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00171031, alias: "DSA Representation Original", vr: VR::US },
            E { tag: 0x00171032, alias: "Number of Maske Original", vr: VR::US },
            E { tag: 0x00171033, alias: "RightLeft Mark Original", vr: VR::US },
            E { tag: 0x00171034, alias: "Slice Position", vr: VR::DS },
            E { tag: 0x00171037, alias: "Rectangle Shutter Horizontal Original", vr: VR::US },
            E { tag: 0x00171038, alias: "Rectangle Vertical Original", vr: VR::US },
            E { tag: 0x00171041, alias: "Dose", vr: VR::DS },
            E { tag: 0x00171042, alias: "Dose Area Product", vr: VR::DS },
            E { tag: 0x00171043, alias: "Focus", vr: VR::US },
            E { tag: 0x00171044, alias: "SDM", vr: VR::US },
            E { tag: 0x00171045, alias: "Image Intensifer Format", vr: VR::DS },
            E { tag: 0x00171046, alias: "Density Correction", vr: VR::DS },
            E { tag: 0x00171047, alias: "Exposure Curves", vr: VR::LO },
            E { tag: 0x00171048, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00171049, alias: "Organ Program Name", vr: VR::LO },
            E { tag: 0x0017104A, alias: "Count of Images in Series", vr: VR::IS },
            E { tag: 0x00171051, alias: "Image Intensifier Zoom", vr: VR::US },
            E { tag: 0x00171052, alias: "ROSA Receiver ErrCount", vr: VR::UL },
            E { tag: 0x00171061, alias: "Table Position Peri", vr: VR::US },
            E { tag: 0x00171062, alias: "Series Numbers of Petri", vr: VR::US },
            E { tag: 0x00171071, alias: "Landmark Original", vr: VR::LO },
            E { tag: 0x00171072, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171073, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171074, alias: "Unknown", vr: VR::US },
            E { tag: 0x00171079, alias: "Unknown", vr: VR::US },
            E { tag: 0x0017107A, alias: "Invers Original", vr: VR::US },
            E { tag: 0x0017107B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001710A0, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001710B1, alias: "L2 Image", vr: VR::CS },
            E { tag: 0x001710B2, alias: "L2 Acquisition Monitor", vr: VR::IS },
            E { tag: 0x001710C1, alias: "Folder UID", vr: VR::LO },
            E { tag: 0x001710C2, alias: "Series UID", vr: VR::LO },
            E { tag: 0x001710C3, alias: "Subseries UID", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS DFR.01 SERVICE",
        entries: &[
            E { tag: 0x00211010, alias: "1 Plot", vr: VR::US },
            E { tag: 0x00211011, alias: "Width 1 Plot", vr: VR::US },
            E { tag: 0x00211012, alias: "2 Plot", vr: VR::US },
            E { tag: 0x00211013, alias: "Width 2 Plot", vr: VR::US },
            E { tag: 0x00211020, alias: "xCoordinate Region of Interest", vr: VR::US },
            E { tag: 0x00211021, alias: "yCoordinate Region of Interest", vr: VR::US },
            E { tag: 0x00211022, alias: "Width Region of Interest", vr: VR::US },
            E { tag: 0x00211023, alias: "Height Region of Interest", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS DICOM",
        entries: &[
            E { tag: 0x00091010, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS DLR.01",
        entries: &[
            E { tag: 0x00191010, alias: "Measurement Mode", vr: VR::LO },
            E { tag: 0x00191011, alias: "Image Type", vr: VR::LO },
            E { tag: 0x00191015, alias: "Software Version", vr: VR::LO },
            E { tag: 0x00191020, alias: "MPM Code", vr: VR::LO },
            E { tag: 0x00191021, alias: "Latitude", vr: VR::LO },
            E { tag: 0x00191022, alias: "Sensitivity", vr: VR::LO },
            E { tag: 0x00191023, alias: "EDR", vr: VR::LO },
            E { tag: 0x00191024, alias: "l fix", vr: VR::LO },
            E { tag: 0x00191025, alias: "s fix", vr: VR::LO },
            E { tag: 0x00191026, alias: "Preset Mode", vr: VR::LO },
            E { tag: 0x00191027, alias: "Region", vr: VR::LO },
            E { tag: 0x00191028, alias: "Subregion", vr: VR::LO },
            E { tag: 0x00191030, alias: "Orientation", vr: VR::LO },
            E { tag: 0x00191031, alias: "Mark on Film", vr: VR::LO },
            E { tag: 0x00191032, alias: "Rotation on DRC", vr: VR::LO },
            E { tag: 0x00191040, alias: "Reader Type", vr: VR::LO },
            E { tag: 0x00191041, alias: "Sub Modality", vr: VR::LO },
            E { tag: 0x00191042, alias: "Reader Serial Number", vr: VR::LO },
            E { tag: 0x00191050, alias: "Cassette Scale", vr: VR::LO },
            E { tag: 0x00191051, alias: "Cassette Matrix", vr: VR::LO },
            E { tag: 0x00191052, alias: "Cassette Submatrix", vr: VR::LO },
            E { tag: 0x00191053, alias: "Barcode", vr: VR::LO },
            E { tag: 0x00191060, alias: "GT Contrast Type", vr: VR::LO },
            E { tag: 0x00191061, alias: "GA Rotation Amount", vr: VR::LO },
            E { tag: 0x00191062, alias: "GC Rotation Center", vr: VR::LO },
            E { tag: 0x00191063, alias: "GS Density Shift", vr: VR::LO },
            E { tag: 0x00191064, alias: "RN Frequency Rank", vr: VR::US },
            E { tag: 0x00191065, alias: "RE Frequency Enhancement", vr: VR::LO },
            E { tag: 0x00191066, alias: "RT Frequency Type", vr: VR::LO },
            E { tag: 0x00191067, alias: "Kernel Length", vr: VR::LO },
            E { tag: 0x00191068, alias: "Kernel Mode", vr: VR::UL },
            E { tag: 0x00191069, alias: "Convulsion Mode", vr: VR::UL },
            E { tag: 0x00191070, alias: "PLA Source", vr: VR::LO },
            E { tag: 0x00191071, alias: "PLA Destination", vr: VR::LO },
            E { tag: 0x00191075, alias: "UID Original Image", vr: VR::LO },
            E { tag: 0x00191076, alias: "Filterset", vr: VR::LO },
            E { tag: 0x00191080, alias: "Reader Header", vr: VR::LO },
            E { tag: 0x00191090, alias: "PLA of Secondary Destination", vr: VR::LO },
            E { tag: 0x001910A0, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00411010, alias: "Number of Hardcopies", vr: VR::US },
            E { tag: 0x00411020, alias: "Film Format", vr: VR::CS },
            E { tag: 0x00411030, alias: "Film Size", vr: VR::LO },
            E { tag: 0x00411031, alias: "Full Film Format", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS ISI",
        entries: &[
            E { tag: 0x00031008, alias: "ISI Command Field", vr: VR::US },
            E { tag: 0x00031011, alias: "Attach ID Application Code", vr: VR::US },
            E { tag: 0x00031012, alias: "Attach ID Message Count", vr: VR::UL },
            E { tag: 0x00031013, alias: "Attach ID Date", vr: VR::DA },
            E { tag: 0x00031014, alias: "Attach ID Time", vr: VR::TM },
            E { tag: 0x00031020, alias: "Message Type", vr: VR::US },
            E { tag: 0x00031030, alias: "Max Waiting Date", vr: VR::DA },
            E { tag: 0x00031031, alias: "Max Waiting Time", vr: VR::TM },
            E { tag: 0x00091001, alias: "RIS Patient Info IMGEF", vr: VR::UN },
            E { tag: 0x00091014, alias: "RIS Patient Info IMGEF", vr: VR::SQ },
            E { tag: 0x00111003, alias: "Patient UID", vr: VR::LO },
            E { tag: 0x00111004, alias: "Patient ID", vr: VR::LO },
            E { tag: 0x0011100A, alias: "Case ID", vr: VR::LO },
            E { tag: 0x00111022, alias: "Request ID", vr: VR::LO },
            E { tag: 0x00111023, alias: "Examination UID", vr: VR::LO },
            E { tag: 0x00111041, alias: "Medical Alerts", vr: VR::LO },
            E { tag: 0x00111042, alias: "Contrast Allergy", vr: VR::LO },
            E { tag: 0x001110A1, alias: "Patient Registration Date", vr: VR::DA },
            E { tag: 0x001110A2, alias: "Patient Registration Time", vr: VR::TM },
            E { tag: 0x001110B0, alias: "Patient Last Name", vr: VR::LO },
            E { tag: 0x001110B2, alias: "Patient First Name", vr: VR::LO },
            E { tag: 0x001110B4, alias: "Patient Hospital Status", vr: VR::CS },
            E { tag: 0x001110BC, alias: "Current Location Time", vr: VR::LO },
            E { tag: 0x001110C0, alias: "Patient Insurance Status", vr: VR::CS },
            E { tag: 0x001110D0, alias: "Patient Billing Type", vr: VR::LO },
            E { tag: 0x001110D2, alias: "Patient Billing Address", vr: VR::LO },
            E { tag: 0x00311012, alias: "Examination Reason", vr: VR::LO },
            E { tag: 0x00311030, alias: "Requested Date", vr: VR::DA },
            E { tag: 0x00311032, alias: "Worklist Request Start Time", vr: VR::TM },
            E { tag: 0x00311033, alias: "Worklist Request End Time", vr: VR::TM },
            E { tag: 0x0031104A, alias: "Requested Time", vr: VR::TM },
            E { tag: 0x00311080, alias: "Requested Location", vr: VR::LO },
            E { tag: 0x00551046, alias: "Current Ward", vr: VR::PN },
            E { tag: 0x01931002, alias: "RIS Key", vr: VR::LO },
            E { tag: 0x03071001, alias: "RIS Worklist IMGEF", vr: VR::SQ },
            E { tag: 0x03091001, alias: "RIS Report IMGEF", vr: VR::SQ },
            E { tag: 0x40091001, alias: "Report ID", vr: VR::LO },
            E { tag: 0x40091020, alias: "Report Status", vr: VR::CS },
            E { tag: 0x40091030, alias: "Report Creation Date", vr: VR::DA },
            E { tag: 0x40091070, alias: "Report Approving Physician", vr: VR::PN },
            E { tag: 0x400910E0, alias: "Report Text", vr: VR::LO },
            E { tag: 0x400910E1, alias: "Report Author", vr: VR::LO },
            E { tag: 0x400910E2, alias: "Report Transcriber", vr: VR::PN },
            E { tag: 0x400910E3, alias: "Reporting Radiologist", vr: VR::PN },
        ],
    },
    P {
        creator: "SIEMENS MED",
        entries: &[
            E { tag: 0x00091010, alias: "Recognition Code", vr: VR::LO },
            E { tag: 0x00091030, alias: "Byte Offset of Original Header", vr: VR::US },
            E { tag: 0x00091031, alias: "Length of Original Header", vr: VR::US },
            E { tag: 0x00091040, alias: "Byte Offset of Pixel Matrix", vr: VR::US },
            E { tag: 0x00091041, alias: "Length of Pixel Matrix", vr: VR::UL },
            E { tag: 0x00091050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091080, alias: "", vr: VR::ST },
            E { tag: 0x000910F5, alias: "PDM EFID Placeholder", vr: VR::LO },
            E { tag: 0x000910F6, alias: "PDM Data Object Type Extension", vr: VR::LO },
            E { tag: 0x00111010, alias: "", vr: VR::ST },
            E { tag: 0x00171000, alias: "", vr: VR::ST },
            E { tag: 0x00171020, alias: "", vr: VR::ST },
            E { tag: 0x00171060, alias: "", vr: VR::LO },
            E { tag: 0x00171080, alias: "", vr: VR::ST },
            E { tag: 0x00191001, alias: "", vr: VR::SS },
            E { tag: 0x00191002, alias: "", vr: VR::SS },
            E { tag: 0x00191003, alias: "", vr: VR::SS },
            E { tag: 0x00191004, alias: "", vr: VR::ST },
            E { tag: 0x00191005, alias: "", vr: VR::ST },
            E { tag: 0x00191006, alias: "", vr: VR::SL },
            E { tag: 0x00191007, alias: "", vr: VR::SL },
            E { tag: 0x00191008, alias: "", vr: VR::SS },
            E { tag: 0x00191009, alias: "", vr: VR::SS },
            E { tag: 0x0019100A, alias: "", vr: VR::ST },
            E { tag: 0x0019100B, alias: "", vr: VR::ST },
            E { tag: 0x0019100C, alias: "", vr: VR::SS },
            E { tag: 0x0019100D, alias: "", vr: VR::ST },
            E { tag: 0x0019100E, alias: "", vr: VR::ST },
            E { tag: 0x0019100F, alias: "", vr: VR::SL },
            E { tag: 0x00191011, alias: "", vr: VR::ST },
            E { tag: 0x00191012, alias: "", vr: VR::ST },
            E { tag: 0x00191013, alias: "", vr: VR::SS },
            E { tag: 0x00191014, alias: "", vr: VR::SS },
            E { tag: 0x00191015, alias: "", vr: VR::SS },
            E { tag: 0x00191016, alias: "", vr: VR::SS },
            E { tag: 0x00191017, alias: "", vr: VR::SS },
            E { tag: 0x00191018, alias: "", vr: VR::SS },
            E { tag: 0x00191019, alias: "", vr: VR::SS },
            E { tag: 0x0019101A, alias: "", vr: VR::SS },
            E { tag: 0x0019101B, alias: "", vr: VR::SS },
            E { tag: 0x0019101C, alias: "", vr: VR::SS },
            E { tag: 0x0019101D, alias: "", vr: VR::SS },
            E { tag: 0x00191020, alias: "", vr: VR::SL },
            E { tag: 0x00191021, alias: "", vr: VR::SL },
            E { tag: 0x00191022, alias: "", vr: VR::SL },
            E { tag: 0x00191023, alias: "", vr: VR::SS },
            E { tag: 0x00191024, alias: "", vr: VR::SL },
            E { tag: 0x00191025, alias: "", vr: VR::SL },
            E { tag: 0x00191026, alias: "", vr: VR::SL },
            E { tag: 0x00191027, alias: "", vr: VR::SL },
            E { tag: 0x00191030, alias: "", vr: VR::SL },
            E { tag: 0x00191031, alias: "", vr: VR::SL },
            E { tag: 0x00191032, alias: "", vr: VR::SL },
            E { tag: 0x00191041, alias: "", vr: VR::SS },
            E { tag: 0x00191042, alias: "", vr: VR::SS },
            E { tag: 0x00191043, alias: "", vr: VR::SS },
            E { tag: 0x00191044, alias: "", vr: VR::SS },
            E { tag: 0x00191045, alias: "", vr: VR::SL },
            E { tag: 0x00191047, alias: "", vr: VR::SL },
            E { tag: 0x00191048, alias: "", vr: VR::ST },
            E { tag: 0x00191049, alias: "", vr: VR::SL },
            E { tag: 0x0019104A, alias: "", vr: VR::SS },
            E { tag: 0x0019104B, alias: "", vr: VR::SS },
            E { tag: 0x0019104C, alias: "", vr: VR::SL },
            E { tag: 0x0019104E, alias: "", vr: VR::SS },
            E { tag: 0x0019104F, alias: "", vr: VR::SS },
            E { tag: 0x00191050, alias: "", vr: VR::SS },
            E { tag: 0x00191051, alias: "", vr: VR::SS },
            E { tag: 0x00191052, alias: "", vr: VR::SS },
            E { tag: 0x00191053, alias: "", vr: VR::SS },
            E { tag: 0x00191054, alias: "", vr: VR::ST },
            E { tag: 0x00191055, alias: "", vr: VR::SS },
            E { tag: 0x00191056, alias: "", vr: VR::SS },
            E { tag: 0x00191057, alias: "", vr: VR::SS },
            E { tag: 0x00191058, alias: "", vr: VR::SS },
            E { tag: 0x00191059, alias: "", vr: VR::SS },
            E { tag: 0x0019105A, alias: "", vr: VR::ST },
            E { tag: 0x0019105B, alias: "", vr: VR::SS },
            E { tag: 0x0019105C, alias: "", vr: VR::SS },
            E { tag: 0x0019105D, alias: "", vr: VR::SS },
            E { tag: 0x0019105E, alias: "", vr: VR::SS },
            E { tag: 0x0019105F, alias: "", vr: VR::SS },
            E { tag: 0x00191060, alias: "", vr: VR::SS },
            E { tag: 0x00191061, alias: "", vr: VR::SS },
            E { tag: 0x00191062, alias: "", vr: VR::SS },
            E { tag: 0x00191063, alias: "", vr: VR::SS },
            E { tag: 0x00191064, alias: "", vr: VR::SS },
            E { tag: 0x00191065, alias: "", vr: VR::SS },
            E { tag: 0x00191066, alias: "", vr: VR::SS },
            E { tag: 0x00191067, alias: "", vr: VR::SS },
            E { tag: 0x00191068, alias: "", vr: VR::SL },
            E { tag: 0x00191069, alias: "", vr: VR::SS },
            E { tag: 0x0019106A, alias: "", vr: VR::SS },
            E { tag: 0x0019106B, alias: "", vr: VR::SL },
            E { tag: 0x0019106C, alias: "", vr: VR::SL },
            E { tag: 0x0019106D, alias: "", vr: VR::SS },
            E { tag: 0x0019106E, alias: "", vr: VR::SS },
            E { tag: 0x0019106F, alias: "", vr: VR::SS },
            E { tag: 0x00191070, alias: "", vr: VR::ST },
            E { tag: 0x00191071, alias: "", vr: VR::ST },
            E { tag: 0x00191072, alias: "", vr: VR::SL },
            E { tag: 0x00191073, alias: "", vr: VR::SL },
            E { tag: 0x00191074, alias: "", vr: VR::SS },
            E { tag: 0x00191075, alias: "", vr: VR::SS },
            E { tag: 0x00191076, alias: "", vr: VR::SS },
            E { tag: 0x00191077, alias: "", vr: VR::SS },
            E { tag: 0x00191078, alias: "", vr: VR::SS },
            E { tag: 0x0019107F, alias: "", vr: VR::SS },
            E { tag: 0x00191080, alias: "", vr: VR::SS },
            E { tag: 0x00191081, alias: "", vr: VR::SS },
            E { tag: 0x00191082, alias: "", vr: VR::SL },
            E { tag: 0x00191083, alias: "", vr: VR::SS },
            E { tag: 0x00191084, alias: "", vr: VR::SS },
            E { tag: 0x00191085, alias: "", vr: VR::SS },
            E { tag: 0x00191086, alias: "", vr: VR::SS },
            E { tag: 0x00191087, alias: "", vr: VR::SS },
            E { tag: 0x00191088, alias: "", vr: VR::SL },
            E { tag: 0x00191089, alias: "", vr: VR::SS },
            E { tag: 0x0019108A, alias: "", vr: VR::SL },
            E { tag: 0x0019108D, alias: "", vr: VR::SL },
            E { tag: 0x0019108E, alias: "", vr: VR::SL },
            E { tag: 0x00191091, alias: "", vr: VR::SL },
            E { tag: 0x00191092, alias: "", vr: VR::SL },
            E { tag: 0x00191093, alias: "", vr: VR::SL },
            E { tag: 0x00191094, alias: "", vr: VR::SS },
            E { tag: 0x00191095, alias: "", vr: VR::SS },
            E { tag: 0x00191096, alias: "", vr: VR::SL },
            E { tag: 0x00191097, alias: "", vr: VR::SS },
            E { tag: 0x001910A0, alias: "", vr: VR::SS },
            E { tag: 0x001910A1, alias: "", vr: VR::SS },
            E { tag: 0x001910A2, alias: "", vr: VR::SL },
            E { tag: 0x001910A3, alias: "", vr: VR::SL },
            E { tag: 0x001910A4, alias: "", vr: VR::SL },
            E { tag: 0x001910C0, alias: "", vr: VR::SL },
            E { tag: 0x001910C1, alias: "", vr: VR::SS },
            E { tag: 0x001910C2, alias: "", vr: VR::SS },
            E { tag: 0x001910F0, alias: "", vr: VR::SL },
            E { tag: 0x001F1000, alias: "", vr: VR::LO },
            E { tag: 0x001F1002, alias: "", vr: VR::LO },
            E { tag: 0x001F1050, alias: "", vr: VR::LO },
            E { tag: 0x001F1070, alias: "", vr: VR::LO },
            E { tag: 0x00211010, alias: "Zoom", vr: VR::DS },
            E { tag: 0x00211011, alias: "Target", vr: VR::DS },
            E { tag: 0x00211012, alias: "Tube Angle During Topogram", vr: VR::IS },
            E { tag: 0x00211020, alias: "ROI Mask", vr: VR::US },
            E { tag: 0x00311001, alias: "", vr: VR::ST },
            E { tag: 0x00311003, alias: "", vr: VR::SL },
            E { tag: 0x00311004, alias: "", vr: VR::ST },
            E { tag: 0x00311005, alias: "", vr: VR::SS },
            E { tag: 0x00311006, alias: "", vr: VR::SL },
            E { tag: 0x00311007, alias: "", vr: VR::SS },
            E { tag: 0x00311008, alias: "", vr: VR::ST },
            E { tag: 0x00311009, alias: "", vr: VR::SS },
            E { tag: 0x0031100A, alias: "", vr: VR::SS },
            E { tag: 0x0031100B, alias: "", vr: VR::SS },
            E { tag: 0x0031100C, alias: "", vr: VR::SS },
            E { tag: 0x0031100D, alias: "", vr: VR::SS },
            E { tag: 0x0031100E, alias: "", vr: VR::SS },
            E { tag: 0x0031100F, alias: "", vr: VR::SL },
            E { tag: 0x00311010, alias: "", vr: VR::SL },
            E { tag: 0x00311011, alias: "", vr: VR::SS },
            E { tag: 0x00411012, alias: "", vr: VR::SL },
            E { tag: 0x00411013, alias: "", vr: VR::SL },
            E { tag: 0x00411021, alias: "", vr: VR::ST },
            E { tag: 0x00411030, alias: "", vr: VR::ST },
            E { tag: 0x00411032, alias: "", vr: VR::ST },
            E { tag: 0x00411034, alias: "", vr: VR::ST },
            E { tag: 0x00411036, alias: "", vr: VR::ST },
            E { tag: 0x00411040, alias: "", vr: VR::ST },
            E { tag: 0x00411050, alias: "", vr: VR::SS },
            E { tag: 0x00411052, alias: "", vr: VR::SS },
            E { tag: 0x00411060, alias: "", vr: VR::ST },
            E { tag: 0x00411070, alias: "", vr: VR::SL },
            E { tag: 0x00411071, alias: "", vr: VR::SL },
            E { tag: 0x00511010, alias: "", vr: VR::SS },
            E { tag: 0x00511011, alias: "", vr: VR::SS },
            E { tag: 0x00511012, alias: "", vr: VR::SS },
            E { tag: 0x00511013, alias: "", vr: VR::SS },
            E { tag: 0x00511014, alias: "", vr: VR::SS },
            E { tag: 0x00511015, alias: "", vr: VR::SL },
            E { tag: 0x00511018, alias: "", vr: VR::SL },
            E { tag: 0x00531001, alias: "", vr: VR::SS },
            E { tag: 0x00531002, alias: "", vr: VR::SS },
            E { tag: 0x00531003, alias: "", vr: VR::SS },
            E { tag: 0x00531010, alias: "", vr: VR::SS },
            E { tag: 0x00531011, alias: "", vr: VR::SL },
            E { tag: 0x00531012, alias: "", vr: VR::SS },
            E { tag: 0x00531013, alias: "", vr: VR::SL },
            E { tag: 0x00531014, alias: "", vr: VR::SS },
            E { tag: 0x00531015, alias: "", vr: VR::SL },
            E { tag: 0x00531016, alias: "", vr: VR::SS },
            E { tag: 0x00531017, alias: "", vr: VR::SS },
            E { tag: 0x00531018, alias: "", vr: VR::SL },
            E { tag: 0x00531019, alias: "", vr: VR::SS },
            E { tag: 0x0053101A, alias: "", vr: VR::SS },
            E { tag: 0x0053101B, alias: "", vr: VR::SL },
            E { tag: 0x0053101C, alias: "", vr: VR::SS },
            E { tag: 0x0053101D, alias: "", vr: VR::SL },
            E { tag: 0x00531030, alias: "", vr: VR::SS },
            E { tag: 0x00531031, alias: "", vr: VR::SS },
            E { tag: 0x00531032, alias: "", vr: VR::SS },
            E { tag: 0x00531033, alias: "", vr: VR::SL },
            E { tag: 0x00531040, alias: "", vr: VR::SS },
            E { tag: 0x00531041, alias: "", vr: VR::SL },
            E { tag: 0x00531042, alias: "", vr: VR::SS },
            E { tag: 0x00531043, alias: "", vr: VR::SL },
            E { tag: 0x00531044, alias: "", vr: VR::SS },
            E { tag: 0x00551001, alias: "", vr: VR::SS },
            E { tag: 0x00551002, alias: "", vr: VR::SS },
            E { tag: 0x00551003, alias: "", vr: VR::SL },
            E { tag: 0x00551020, alias: "", vr: VR::SS },
            E { tag: 0x00551021, alias: "", vr: VR::ST },
            E { tag: 0x00551022, alias: "", vr: VR::SS },
            E { tag: 0x00551023, alias: "", vr: VR::ST },
            E { tag: 0x00551024, alias: "", vr: VR::SS },
            E { tag: 0x00551025, alias: "", vr: VR::ST },
            E { tag: 0x00551029, alias: "", vr: VR::SL },
            E { tag: 0x0055102A, alias: "", vr: VR::SL },
            E { tag: 0x00551030, alias: "", vr: VR::SS },
            E { tag: 0x00551031, alias: "", vr: VR::ST },
            E { tag: 0x00551032, alias: "", vr: VR::SS },
            E { tag: 0x00551033, alias: "", vr: VR::ST },
            E { tag: 0x00551034, alias: "", vr: VR::SS },
            E { tag: 0x00551035, alias: "", vr: VR::ST },
            E { tag: 0x00551038, alias: "", vr: VR::SL },
            E { tag: 0x00551039, alias: "", vr: VR::SL },
            E { tag: 0x0055103A, alias: "", vr: VR::SL },
            E { tag: 0x00551040, alias: "", vr: VR::SS },
            E { tag: 0x00551048, alias: "", vr: VR::SL },
            E { tag: 0x00551050, alias: "", vr: VR::SS },
            E { tag: 0x00551051, alias: "", vr: VR::SS },
            E { tag: 0x00551052, alias: "", vr: VR::SS },
            E { tag: 0x00551053, alias: "", vr: VR::ST },
            E { tag: 0x00551054, alias: "", vr: VR::SL },
            E { tag: 0x00551055, alias: "", vr: VR::SL },
            E { tag: 0x00551056, alias: "", vr: VR::SS },
            E { tag: 0x00551057, alias: "", vr: VR::SS },
            E { tag: 0x00551058, alias: "", vr: VR::SL },
            E { tag: 0x00551059, alias: "", vr: VR::ST },
            E { tag: 0x0055105A, alias: "", vr: VR::SL },
            E { tag: 0x0055105B, alias: "", vr: VR::SS },
            E { tag: 0x0055105C, alias: "", vr: VR::ST },
            E { tag: 0x0055105D, alias: "", vr: VR::SS },
            E { tag: 0x0055105E, alias: "", vr: VR::SS },
            E { tag: 0x0055105F, alias: "", vr: VR::SL },
            E { tag: 0x00551060, alias: "", vr: VR::SS },
            E { tag: 0x00551061, alias: "", vr: VR::SS },
            E { tag: 0x00551064, alias: "", vr: VR::SS },
            E { tag: 0x00551065, alias: "", vr: VR::SS },
            E { tag: 0x00551066, alias: "", vr: VR::SS },
            E { tag: 0x00551067, alias: "", vr: VR::SS },
            E { tag: 0x00551068, alias: "", vr: VR::SS },
            E { tag: 0x00551069, alias: "", vr: VR::SS },
            E { tag: 0x0055106A, alias: "", vr: VR::SS },
            E { tag: 0x0055106B, alias: "", vr: VR::SS },
            E { tag: 0x0055106C, alias: "", vr: VR::SS },
            E { tag: 0x0055106D, alias: "", vr: VR::SS },
            E { tag: 0x00551070, alias: "", vr: VR::SL },
            E { tag: 0x00551071, alias: "", vr: VR::SL },
            E { tag: 0x00551072, alias: "", vr: VR::SL },
            E { tag: 0x00551073, alias: "", vr: VR::SL },
            E { tag: 0x00551074, alias: "", vr: VR::SL },
            E { tag: 0x00551075, alias: "", vr: VR::SL },
            E { tag: 0x00551076, alias: "", vr: VR::SL },
            E { tag: 0x00551077, alias: "", vr: VR::SL },
            E { tag: 0x00551078, alias: "", vr: VR::SL },
            E { tag: 0x00551080, alias: "", vr: VR::SS },
            E { tag: 0x00551081, alias: "", vr: VR::SS },
            E { tag: 0x00551082, alias: "", vr: VR::SS },
            E { tag: 0x00551083, alias: "", vr: VR::SS },
            E { tag: 0x00551084, alias: "", vr: VR::SS },
            E { tag: 0x00551085, alias: "", vr: VR::SS },
            E { tag: 0x00551086, alias: "", vr: VR::SL },
            E { tag: 0x00551087, alias: "", vr: VR::SL },
            E { tag: 0x00551088, alias: "", vr: VR::SL },
            E { tag: 0x00551089, alias: "", vr: VR::ST },
            E { tag: 0x0055108A, alias: "", vr: VR::SS },
            E { tag: 0x0055108B, alias: "", vr: VR::SL },
            E { tag: 0x0055108C, alias: "", vr: VR::SS },
            E { tag: 0x0055108D, alias: "", vr: VR::SL },
            E { tag: 0x0055108E, alias: "", vr: VR::SL },
            E { tag: 0x0055108F, alias: "", vr: VR::SL },
            E { tag: 0x00551090, alias: "", vr: VR::SL },
            E { tag: 0x00551091, alias: "", vr: VR::SS },
            E { tag: 0x00551092, alias: "", vr: VR::ST },
            E { tag: 0x00551093, alias: "", vr: VR::SL },
            E { tag: 0x00551094, alias: "", vr: VR::SL },
            E { tag: 0x00551095, alias: "", vr: VR::SL },
            E { tag: 0x00551096, alias: "", vr: VR::SL },
            E { tag: 0x00551097, alias: "", vr: VR::SL },
            E { tag: 0x00551098, alias: "", vr: VR::SS },
            E { tag: 0x00551099, alias: "", vr: VR::SS },
            E { tag: 0x0055109A, alias: "", vr: VR::SS },
            E { tag: 0x0055109B, alias: "", vr: VR::SS },
            E { tag: 0x0055109C, alias: "", vr: VR::SS },
            E { tag: 0x0055109D, alias: "", vr: VR::SL },
            E { tag: 0x0055109E, alias: "", vr: VR::SL },
            E { tag: 0x0055109F, alias: "", vr: VR::SL },
            E { tag: 0x005510A0, alias: "", vr: VR::SL },
            E { tag: 0x005510A1, alias: "", vr: VR::SL },
            E { tag: 0x005510A2, alias: "", vr: VR::SL },
            E { tag: 0x005510A3, alias: "", vr: VR::SL },
            E { tag: 0x005510A4, alias: "", vr: VR::SS },
            E { tag: 0x005510A5, alias: "", vr: VR::SL },
            E { tag: 0x005510A6, alias: "", vr: VR::SL },
            E { tag: 0x005510A7, alias: "", vr: VR::SS },
            E { tag: 0x005510A8, alias: "", vr: VR::SS },
            E { tag: 0x005510A9, alias: "", vr: VR::SS },
            E { tag: 0x005510B0, alias: "", vr: VR::SS },
            E { tag: 0x005510B1, alias: "", vr: VR::SS },
            E { tag: 0x00571010, alias: "", vr: VR::SS },
            E { tag: 0x00571020, alias: "", vr: VR::SL },
            E { tag: 0x50011001, alias: "", vr: VR::SS },
            E { tag: 0x50011002, alias: "", vr: VR::SL },
            E { tag: 0x50011003, alias: "", vr: VR::SL },
            E { tag: 0x50011004, alias: "", vr: VR::SL },
            E { tag: 0x50011005, alias: "", vr: VR::SL },
            E { tag: 0x50011006, alias: "", vr: VR::SL },
            E { tag: 0x50011007, alias: "", vr: VR::ST },
            E { tag: 0x50011008, alias: "", vr: VR::ST },
            E { tag: 0x50011009, alias: "", vr: VR::ST },
            E { tag: 0x5001100A, alias: "", vr: VR::ST },
            E { tag: 0x5001100B, alias: "", vr: VR::SL },
            E { tag: 0x70011010, alias: "", vr: VR::SS },
            E { tag: 0x70031010, alias: "", vr: VR::SL },
            E { tag: 0x70051010, alias: "", vr: VR::SS },
            E { tag: 0x7FE11010, alias: "", vr: VR::SS },
            E { tag: 0x7FE31014, alias: "", vr: VR::SL },
            E { tag: 0x7FE31015, alias: "", vr: VR::SL },
            E { tag: 0x7FE31016, alias: "", vr: VR::SL },
            E { tag: 0x7FE31017, alias: "", vr: VR::SL },
            E { tag: 0x7FE31018, alias: "", vr: VR::SL },
            E { tag: 0x7FE31019, alias: "", vr: VR::ST },
            E { tag: 0x7FE3101A, alias: "", vr: VR::SL },
            E { tag: 0x7FE3101B, alias: "", vr: VR::SS },
            E { tag: 0x7FE3101C, alias: "", vr: VR::SS },
            E { tag: 0x7FE3101E, alias: "", vr: VR::SL },
            E { tag: 0x7FE3101F, alias: "", vr: VR::SL },
            E { tag: 0x7FE31020, alias: "", vr: VR::SL },
            E { tag: 0x7FE31021, alias: "", vr: VR::SL },
            E { tag: 0x7FE31022, alias: "", vr: VR::SL },
            E { tag: 0x7FE31023, alias: "", vr: VR::SL },
            E { tag: 0x7FE31024, alias: "", vr: VR::ST },
            E { tag: 0x7FE31025, alias: "", vr: VR::SS },
        ],
    },
    P {
        creator: "SIEMENS MED DISPLAY",
        entries: &[
            E { tag: 0x00291004, alias: "Photometric Interpretation", vr: VR::CS },
            E { tag: 0x00291010, alias: "Rows", vr: VR::US },
            E { tag: 0x00291011, alias: "Columns", vr: VR::US },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291050, alias: "Origin", vr: VR::US },
            E { tag: 0x00291080, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291099, alias: "Shutter Type", vr: VR::LO },
            E { tag: 0x002910A0, alias: "Rows of Rectangular Shutter", vr: VR::US },
            E { tag: 0x002910A1, alias: "Columns of Rectangular Shutter", vr: VR::US },
            E { tag: 0x002910A2, alias: "Origin of Rectangular Shutter", vr: VR::US },
            E { tag: 0x002910B0, alias: "Radius of Circular Shutter", vr: VR::US },
            E { tag: 0x002910B2, alias: "Origin of Circular Shutter", vr: VR::US },
            E { tag: 0x002910C1, alias: "Contour of Irregular Shutter", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS MED DISPLAY 0000",
        entries: &[
            E { tag: 0x00291099, alias: "Unknown", vr: VR::LO },
            E { tag: 0x002910C1, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS MED ECAT FILE INFO",
        entries: &[
            E { tag: 0x00211000, alias: "ECAT_Main_Header", vr: VR::OB },
            E { tag: 0x00211001, alias: "ECAT_Image_Subheader", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS MED HG",
        entries: &[
            E { tag: 0x00291010, alias: "List of Group Numbers", vr: VR::US },
            E { tag: 0x00291015, alias: "List of Shadow Owner Codes", vr: VR::LO },
            E { tag: 0x00291020, alias: "List of Element Numbers", vr: VR::US },
            E { tag: 0x00291030, alias: "List of Total Display Length", vr: VR::US },
            E { tag: 0x00291040, alias: "List of Display Prefix", vr: VR::LO },
            E { tag: 0x00291050, alias: "List of Display Postfix", vr: VR::LO },
            E { tag: 0x00291060, alias: "List of Text Position", vr: VR::US },
            E { tag: 0x00291070, alias: "List of Text Concatenation", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS MED MAMMO",
        entries: &[
            E { tag: 0x0029105A, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED MG",
        entries: &[
            E { tag: 0x00291010, alias: "List of Group Numbers", vr: VR::US },
            E { tag: 0x00291015, alias: "List of Shadow Owner Codes", vr: VR::LO },
            E { tag: 0x00291020, alias: "List of Element Numbers", vr: VR::US },
            E { tag: 0x00291030, alias: "List of Total Display Length", vr: VR::US },
            E { tag: 0x00291040, alias: "List of Display Prefix", vr: VR::LO },
            E { tag: 0x00291050, alias: "List of Display Postfix", vr: VR::LO },
            E { tag: 0x00291060, alias: "List of Text Position", vr: VR::US },
            E { tag: 0x00291070, alias: "List of Text Concatenation", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS MED NM",
        entries: &[
            E { tag: 0x00091080, alias: "Main Acquisition View", vr: VR::LO },
            E { tag: 0x00111010, alias: "Organ System", vr: VR::CS },
            E { tag: 0x00171000, alias: "Unknown 0017 0000", vr: VR::ST },
            E { tag: 0x00171020, alias: "Scanning Sequence", vr: VR::CS },
            E { tag: 0x00171060, alias: "KVP", vr: VR::DS },
            E { tag: 0x00171070, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00171080, alias: "Unknown 0017 0080", vr: VR::ST },
            E { tag: 0x00191001, alias: "Number of detectors", vr: VR::SS },
            E { tag: 0x00191002, alias: "Number of scalers", vr: VR::SS },
            E { tag: 0x00191003, alias: "Scaler Index", vr: VR::SS },
            E { tag: 0x00191004, alias: "Radionuclide Units", vr: VR::DS },
            E { tag: 0x00191005, alias: "Radiopharmaceutical", vr: VR::DS },
            E { tag: 0x00191006, alias: "Display Magnification", vr: VR::FL },
            E { tag: 0x00191007, alias: "Scaler Counts", vr: VR::SL },
            E { tag: 0x00191008, alias: "Processed Flag 0 Raw 1 Process", vr: VR::SS },
            E { tag: 0x00191009, alias: "MicroDELTA Parameter Blocks", vr: VR::SS },
            E { tag: 0x0019100A, alias: "MicroDELTA File Extension", vr: VR::LO },
            E { tag: 0x0019100B, alias: "MicroDELTA Write ID", vr: VR::LO },
            E { tag: 0x0019100C, alias: "MicroDELTA Write Version", vr: VR::SS },
            E { tag: 0x0019100D, alias: "MicroDELTA Mode", vr: VR::CS },
            E { tag: 0x0019100E, alias: "MicroDELTA ESFR", vr: VR::LO },
            E { tag: 0x0019100F, alias: "Siemens ICON Data Type", vr: VR::SL },
            E { tag: 0x00191010, alias: "MicroDELTA Expansion Blocks", vr: VR::OW },
            E { tag: 0x00191011, alias: "MicroDELTA Version", vr: VR::CS },
            E { tag: 0x00191012, alias: "MicroDELTA Trend", vr: VR::LO },
            E { tag: 0x00191013, alias: "MicroDELTA Number of Overlays", vr: VR::SS },
            E { tag: 0x00191014, alias: "MicroDELTA Overlay Block", vr: VR::SS },
            E { tag: 0x00191015, alias: "MicroDELTA Color Table Extra", vr: VR::SS },
            E { tag: 0x00191016, alias: "Number of Views", vr: VR::SS },
            E { tag: 0x00191017, alias: "MicroDELTA Thyroid Markers", vr: VR::SS },
            E { tag: 0x00191018, alias: "MicroDELTA Color Table Used", vr: VR::SS },
            E { tag: 0x00191019, alias: "Save Screen Flag 1 SaveScreen", vr: VR::SS },
            E { tag: 0x0019101A, alias: "MicroDELTA Overlay Map", vr: VR::SS },
            E { tag: 0x0019101B, alias: "SGI Converted LookUp Table", vr: VR::SS },
            E { tag: 0x0019101C, alias: "MicroDELTA Mystery1", vr: VR::SS },
            E { tag: 0x0019101D, alias: "MicroDELTA Mystery2", vr: VR::SS },
            E { tag: 0x00191020, alias: "Time of Lasix Administration", vr: VR::FL },
            E { tag: 0x00191021, alias: "Lasix Dose mg", vr: VR::FL },
            E { tag: 0x00191022, alias: "Bladder Esidual Volume cc", vr: VR::FL },
            E { tag: 0x00191023, alias: "Bladder Image Extension", vr: VR::SS },
            E { tag: 0x00191024, alias: "Serum Creatinine mg/dl", vr: VR::FL },
            E { tag: 0x00191025, alias: "TER", vr: VR::FL },
            E { tag: 0x00191026, alias: "TER Lower Limit", vr: VR::FL },
            E { tag: 0x00191027, alias: "Body Surface Area m2", vr: VR::FL },
            E { tag: 0x00191030, alias: "CTQ Normal Margin", vr: VR::FL },
            E { tag: 0x00191031, alias: "CTQ Abnormal Sagittal Margin", vr: VR::FL },
            E { tag: 0x00191032, alias: "CTQ Abnormal Oblique Margin", vr: VR::FL },
            E { tag: 0x00191041, alias: "Accepted Beats", vr: VR::SS },
            E { tag: 0x00191042, alias: "Rejected Beats", vr: VR::SS },
            E { tag: 0x00191043, alias: "Percent of R-R Interval Framed", vr: VR::SS },
            E { tag: 0x00191044, alias: "Beats Rejected Post PVC", vr: VR::SS },
            E { tag: 0x00191045, alias: "Trigger Source", vr: VR::FL },
            E { tag: 0x00191047, alias: "Beat Rejection Parameter", vr: VR::FL },
            E { tag: 0x00191048, alias: "Rejection Scheme", vr: VR::SH },
            E { tag: 0x00191049, alias: "PVC Threshold msec", vr: VR::FL },
            E { tag: 0x0019104A, alias: "Framing Mode 1 Forward 2 F/B", vr: VR::SS },
            E { tag: 0x0019104B, alias: "Autobeat Window 0 No 1 Yes", vr: VR::SS },
            E { tag: 0x0019104C, alias: "MicroDELTA Acquisition Time per Window msec", vr: VR::FL },
            E { tag: 0x0019104E, alias: "CTQ Unknown Entry Count", vr: VR::SS },
            E { tag: 0x0019104F, alias: "CTQ Matrix Size", vr: VR::SS },
            E { tag: 0x00191050, alias: "CTQ has been used", vr: VR::SS },
            E { tag: 0x00191051, alias: "CTQ Male Entry Count", vr: VR::SS },
            E { tag: 0x00191052, alias: "CTQ Female Entry Count", vr: VR::SS },
            E { tag: 0x00191053, alias: "CTQ Stress/Rest Time Delay", vr: VR::SS },
            E { tag: 0x00191054, alias: "CTQ Time Delay Units", vr: VR::CS },
            E { tag: 0x00191055, alias: "CTQ Min Oblique Stack Count", vr: VR::SS },
            E { tag: 0x00191056, alias: "CTQ Max Oblique Stack Count", vr: VR::SS },
            E { tag: 0x00191057, alias: "CTQ Min Sagittal Stack Count", vr: VR::SS },
            E { tag: 0x00191058, alias: "CTQ Max Sagittal Stack Count", vr: VR::SS },
            E { tag: 0x00191059, alias: "CTQ Min Radius Percent", vr: VR::SS },
            E { tag: 0x0019105A, alias: "CTQ Find Max or Average Flag", vr: VR::UL },
            E { tag: 0x0019105B, alias: "CTQ Basal Slice Index Offset", vr: VR::SS },
            E { tag: 0x0019105C, alias: "CTQ Starting Line Index", vr: VR::SS },
            E { tag: 0x0019105D, alias: "CTQ Line Counts", vr: VR::SS },
            E { tag: 0x0019105E, alias: "CTQ Radii Counts", vr: VR::SS },
            E { tag: 0x0019105F, alias: "CTQ Radial Angle Increment", vr: VR::SS },
            E { tag: 0x00191060, alias: "Whole Body Processed Flag 0:Raw 1:Img1-Img2 2:Im", vr: VR::SS },
            E { tag: 0x00191061, alias: "Whole Body View", vr: VR::SS },
            E { tag: 0x00191062, alias: "WB Display Field Length in cm", vr: VR::SS },
            E { tag: 0x00191063, alias: "WB Scan Length in cm", vr: VR::SS },
            E { tag: 0x00191064, alias: "WB Scan Speed cm/min", vr: VR::SS },
            E { tag: 0x00191065, alias: "WB Scan Speed Mode 0 Manual 1 Auto", vr: VR::SS },
            E { tag: 0x00191066, alias: "WB Scan Start 0 Head 1 Feet", vr: VR::SS },
            E { tag: 0x00191067, alias: "WB Scan Direction 0 Lt to Rt 1 Rt to Lt", vr: VR::SS },
            E { tag: 0x00191068, alias: "WB Aperature Width in cm", vr: VR::FL },
            E { tag: 0x00191069, alias: "WB Number of Passes", vr: VR::SS },
            E { tag: 0x0019106A, alias: "WB Scan Time msec", vr: VR::SS },
            E { tag: 0x0019106B, alias: "WB Anterior kCounts View 1", vr: VR::FL },
            E { tag: 0x0019106C, alias: "WB Anterior kCounts View 2", vr: VR::FL },
            E { tag: 0x0019106D, alias: "WB Count Density", vr: VR::FL },
            E { tag: 0x0019106E, alias: "WB Reference View Flag 1 Ref", vr: VR::SS },
            E { tag: 0x0019106F, alias: "WB Coun Density Speed cm/min", vr: VR::FL },
            E { tag: 0x00191070, alias: "WB Scan Type", vr: VR::CS },
            E { tag: 0x00191071, alias: "WB IRIS Corr. File Name", vr: VR::LO },
            E { tag: 0x00191072, alias: "WB Posterior kCounts View 1", vr: VR::FL },
            E { tag: 0x00191073, alias: "WB Posterior kCounts View 2", vr: VR::FL },
            E { tag: 0x00191074, alias: "WB Ant View Status No Img 1 Complete 2 Aborted", vr: VR::SS },
            E { tag: 0x00191075, alias: "WB Post View Status No Img 1 Complete 2 Aborted", vr: VR::SS },
            E { tag: 0x00191076, alias: "WB Post Scan Time msec", vr: VR::SS },
            E { tag: 0x00191077, alias: "WB Ant Pixel Overflow 1 Yes", vr: VR::SS },
            E { tag: 0x00191078, alias: "WB Post Pixel Overflow 1 Yes", vr: VR::SS },
            E { tag: 0x0019107F, alias: "Segment Type 0 Segments 2 Time/Segment", vr: VR::SS },
            E { tag: 0x00191080, alias: "Number of Segments", vr: VR::SS },
            E { tag: 0x00191081, alias: "PVC Rejection Flag 1 Reject", vr: VR::SS },
            E { tag: 0x00191082, alias: "Total Counts Framed", vr: VR::SL },
            E { tag: 0x00191083, alias: "Total Beats Framed", vr: VR::SS },
            E { tag: 0x00191084, alias: "Present Time of Day", vr: VR::SS },
            E { tag: 0x00191085, alias: "Framed Counts", vr: VR::SS },
            E { tag: 0x00191086, alias: "Time per Segment msec", vr: VR::SS },
            E { tag: 0x00191087, alias: "BBA Miscellaneous Info", vr: VR::LT },
            E { tag: 0x00191088, alias: "Tolerance %", vr: VR::SL },
            E { tag: 0x00191089, alias: "SWG", vr: VR::SS },
            E { tag: 0x0019108A, alias: "R-R Interval msec", vr: VR::SL },
            E { tag: 0x0019108D, alias: "Total Counts in Acquisition", vr: VR::SL },
            E { tag: 0x0019108E, alias: "Rejection Type", vr: VR::SL },
            E { tag: 0x00191091, alias: "Phase Total Counts", vr: VR::FL },
            E { tag: 0x00191092, alias: "Phase Pause msec", vr: VR::SL },
            E { tag: 0x00191093, alias: "Phase Start msec", vr: VR::SL },
            E { tag: 0x00191094, alias: "Window Center Method 0 Avg 1 Peak", vr: VR::SS },
            E { tag: 0x00191095, alias: "Number of Beats to Average", vr: VR::SS },
            E { tag: 0x00191096, alias: "Histogram Bin Interval msec", vr: VR::FL },
            E { tag: 0x00191097, alias: "Counts in Peak Bin", vr: VR::SS },
            E { tag: 0x001910A0, alias: "VPDs Detected", vr: VR::SS },
            E { tag: 0x001910A1, alias: "Number of Phases", vr: VR::SS },
            E { tag: 0x001910A2, alias: "Requested Frame Duration msec", vr: VR::SL },
            E { tag: 0x001910A3, alias: "Frames per Phase", vr: VR::SL },
            E { tag: 0x001910A4, alias: "Frame Pause msec", vr: VR::SL },
            E { tag: 0x001910A5, alias: "Number of repeats per phase", vr: VR::SS },
            E { tag: 0x001910A6, alias: "Cycles per repeat", vr: VR::SS },
            E { tag: 0x001910A7, alias: "Repeat start time", vr: VR::SL },
            E { tag: 0x001910A8, alias: "Repeat stop time", vr: VR::SL },
            E { tag: 0x001910A9, alias: "Effective repeat time", vr: VR::SL },
            E { tag: 0x001910AA, alias: "Acquired cycles per repeat", vr: VR::SS },
            E { tag: 0x001910C0, alias: "Head Tilt degrees", vr: VR::FL },
            E { tag: 0x001910C1, alias: "Uniformity Correction Flag", vr: VR::SS },
            E { tag: 0x001910C2, alias: "Overflow Flag", vr: VR::SS },
            E { tag: 0x001910C3, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910F0, alias: "Pixel Dimensions", vr: VR::FL },
            E { tag: 0x001F1050, alias: "Location RET", vr: VR::DS },
            E { tag: 0x00211000, alias: "ECAT File Menu Header", vr: VR::OB },
            E { tag: 0x00211001, alias: "ECAT File Subheader", vr: VR::OB },
            E { tag: 0x00211010, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00311001, alias: "Camera Name", vr: VR::LO },
            E { tag: 0x00311002, alias: "Unknown Element 0031 02", vr: VR::SS },
            E { tag: 0x00311003, alias: "External Zoom", vr: VR::FL },
            E { tag: 0x00311004, alias: "CCD Energy Level", vr: VR::FL },
            E { tag: 0x00311005, alias: "CCD Window Width", vr: VR::SS },
            E { tag: 0x00311006, alias: "CCD Window Offset", vr: VR::FL },
            E { tag: 0x00311007, alias: "CCD Peaking Mode", vr: VR::SS },
            E { tag: 0x00311008, alias: "CCD Autopeak Function", vr: VR::LO },
            E { tag: 0x00311009, alias: "CCD Count Rate Mode", vr: VR::SS },
            E { tag: 0x0031100A, alias: "Analog Output Device", vr: VR::SS },
            E { tag: 0x0031100B, alias: "Communication Type", vr: VR::SS },
            E { tag: 0x0031100C, alias: "Control Unit 1 ICON 2 DOT 3 ScintiView", vr: VR::SS },
            E { tag: 0x0031100D, alias: "WAM Mode", vr: VR::SS },
            E { tag: 0x0031100E, alias: "Event Control Mode", vr: VR::SS },
            E { tag: 0x0031100F, alias: "Zoom X Offset 0.1 mm", vr: VR::FL },
            E { tag: 0x00311010, alias: "Zoom Y Offset 0.1 mm", vr: VR::FL },
            E { tag: 0x00311011, alias: "Gantry Zero Position", vr: VR::SS },
            E { tag: 0x00311012, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00311013, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00311014, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00311015, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00311016, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00311017, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00311020, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00311021, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00331000, alias: "Flood correction Matrix Det 1 up to SR 2.0", vr: VR::FL },
            E { tag: 0x00331001, alias: "Flood correction Matrix Det 2 up to SR 2.0", vr: VR::FL },
            E { tag: 0x00331010, alias: "COR Data for Detector 1", vr: VR::FL },
            E { tag: 0x00331011, alias: "COR Data for Detector 2", vr: VR::FL },
            E { tag: 0x00331014, alias: "MHR ( Y-Shift) data for detector 1", vr: VR::FL },
            E { tag: 0x00331015, alias: "MHR ( Y-Shift) data for detector 2", vr: VR::FL },
            E { tag: 0x00331018, alias: "NCO Data for detector 1", vr: VR::FL },
            E { tag: 0x00331019, alias: "NCO Data for detector 2", vr: VR::FL },
            E { tag: 0x00331020, alias: "Bed correction angle", vr: VR::FL },
            E { tag: 0x00331021, alias: "Gantry correction angle", vr: VR::FL },
            E { tag: 0x00331022, alias: "Bed U/D correction data", vr: VR::SS },
            E { tag: 0x00331023, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00331024, alias: "Backprojection angle head 1", vr: VR::FL },
            E { tag: 0x00331025, alias: "Backprojection angle head 2", vr: VR::FL },
            E { tag: 0x00331028, alias: "Number of point sources used for NCO and MHR", vr: VR::SL },
            E { tag: 0x00331029, alias: "Crystal thickness", vr: VR::FL },
            E { tag: 0x00331030, alias: "Preset name used for acquisition", vr: VR::LO },
            E { tag: 0x00331031, alias: "Camera config angle", vr: VR::FL },
            E { tag: 0x00331032, alias: "Crystal type Startburst or not", vr: VR::LO },
            E { tag: 0x00331033, alias: "Gantry step for COIN acquisitions", vr: VR::SL },
            E { tag: 0x00331034, alias: "Bed step for wholebody or Coin acquisitions", vr: VR::FL },
            E { tag: 0x00331035, alias: "Weight factor table for coincidence acquisitions", vr: VR::FL },
            E { tag: 0x00331036, alias: "Transaxial acceptance width for coincidence", vr: VR::FL },
            E { tag: 0x00331037, alias: "Starburst flags", vr: VR::SL },
            E { tag: 0x00331038, alias: "Pixel scale factor", vr: VR::FL },
            E { tag: 0x00351000, alias: "Specialized tomo type", vr: VR::LO },
            E { tag: 0x00351001, alias: "Energy window type", vr: VR::LO },
            E { tag: 0x00351002, alias: "Start and end row illuminated by wind position", vr: VR::SS },
            E { tag: 0x00351003, alias: "Blank scan image for profile", vr: VR::LO },
            E { tag: 0x00351004, alias: "Repeat number of the original dynamic SPECT", vr: VR::SS },
            E { tag: 0x00351005, alias: "Phase number of the original dynamic SPECT", vr: VR::SS },
            E { tag: 0x00371000, alias: "Flood correction matrix Detector 1", vr: VR::OW },
            E { tag: 0x00371080, alias: "Flood correction matrix Detector 2", vr: VR::OW },
            E { tag: 0x00391000, alias: "Toshiba CBF activity results", vr: VR::LT },
            E { tag: 0x00411001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00411002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00411005, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00411006, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00411007, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00411008, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00411011, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00411021, alias: "Text Relating to Data Object", vr: VR::LO },
            E { tag: 0x00411030, alias: "Folder Creation Date", vr: VR::DA },
            E { tag: 0x00411032, alias: "Folder Creation Time", vr: VR::TM },
            E { tag: 0x00411034, alias: "Folder Modification Date", vr: VR::DA },
            E { tag: 0x00411036, alias: "Folder Modification Time", vr: VR::TM },
            E { tag: 0x00411040, alias: "Folder Owner Name", vr: VR::LO },
            E { tag: 0x00411050, alias: "Folder Status", vr: VR::SS },
            E { tag: 0x00411070, alias: "Number of Images in Folder", vr: VR::SL },
            E { tag: 0x00411071, alias: "Number of Non-Image Objects", vr: VR::SL },
            E { tag: 0x00431001, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00431003, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00431004, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00511010, alias: "Number of Windows", vr: VR::SS },
            E { tag: 0x00511011, alias: "Average Heart Rate BPM", vr: VR::SS },
            E { tag: 0x00511012, alias: "Accepted Beats in Window", vr: VR::SS },
            E { tag: 0x00511013, alias: "Average R-R Interval msec", vr: VR::SS },
            E { tag: 0x00511014, alias: "Window Width %", vr: VR::SS },
            E { tag: 0x00511015, alias: "Hottest Pixel in Window", vr: VR::SL },
            E { tag: 0x00511018, alias: "Total Counts in Window", vr: VR::SL },
            E { tag: 0x00531001, alias: "Terminate on Pixel Overflow Flag 1 Terminate", vr: VR::SS },
            E { tag: 0x00531002, alias: "Terminate on List Buffer Overflow Flag", vr: VR::SS },
            E { tag: 0x00531003, alias: "First Frame by Counts", vr: VR::SS },
            E { tag: 0x00531010, alias: "Elapsed Time Start Condition 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x00531011, alias: "Elapsed Time Start msec", vr: VR::SL },
            E { tag: 0x00531012, alias: "Count Rate Start Condition 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x00531013, alias: "Count Rate Start", vr: VR::SL },
            E { tag: 0x00531014, alias: "Start Heart Beat Count Condition 0 None 1 Any 2", vr: VR::SS },
            E { tag: 0x00531015, alias: "Start Heart Beat Count", vr: VR::SL },
            E { tag: 0x00531016, alias: "Start Heart Rate Threshold Condition", vr: VR::SS },
            E { tag: 0x00531017, alias: "Start Heart Rate Increasing Flag", vr: VR::SS },
            E { tag: 0x00531018, alias: "Start Heart Rate", vr: VR::SL },
            E { tag: 0x00531019, alias: "Manual Stop Flag 0 No 1 Yes", vr: VR::SL },
            E { tag: 0x0053101A, alias: "Elapsed Time Stop Condition 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x0053101B, alias: "Elapsed Time Stop", vr: VR::SL },
            E { tag: 0x0053101C, alias: "Stop Heart Beat Count Condition 0 None 1 Any 2 A", vr: VR::SS },
            E { tag: 0x0053101D, alias: "Stop Heart Beat Count", vr: VR::SL },
            E { tag: 0x00531030, alias: "Info Density Stop Condition", vr: VR::SS },
            E { tag: 0x00531031, alias: "Info Density Stop Scaler", vr: VR::SS },
            E { tag: 0x00531032, alias: "Info Density Stop Region", vr: VR::SS },
            E { tag: 0x00531033, alias: "Info Density Stop Count", vr: VR::SL },
            E { tag: 0x00531040, alias: "Stop Condition Status 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x00531041, alias: "Stop Count", vr: VR::SL },
            E { tag: 0x00531042, alias: "Stop Count Condition Status 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x00531043, alias: "Stop Time", vr: VR::SL },
            E { tag: 0x00531044, alias: "Stop Time Condition Status 0 None 1 Any 2 All", vr: VR::SS },
            E { tag: 0x00551002, alias: "Unknown 55-02", vr: VR::SS },
            E { tag: 0x00551004, alias: "Prompt window width", vr: VR::SS },
            E { tag: 0x00551005, alias: "Random window width", vr: VR::SS },
            E { tag: 0x00551020, alias: "Flood Corr. Flag 1 Corr", vr: VR::SS },
            E { tag: 0x00551021, alias: "Flood Acquisition Date", vr: VR::DS },
            E { tag: 0x00551030, alias: "COR Corr. Flag 1 Corr", vr: VR::SS },
            E { tag: 0x00551031, alias: "COR Acquisition Date", vr: VR::DA },
            E { tag: 0x00551038, alias: "COR Data", vr: VR::OW },
            E { tag: 0x00551040, alias: "Noncircular Orbit Flag 1 NCO", vr: VR::SS },
            E { tag: 0x00551048, alias: "NCO Values", vr: VR::FL },
            E { tag: 0x0055104C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0055104D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00551050, alias: "Prone Flag 0 Supine 1 Prone", vr: VR::SS },
            E { tag: 0x00551051, alias: "Orbit 0 Circular 1 Elliptical", vr: VR::SS },
            E { tag: 0x00551052, alias: "Total Rotation in Degrees", vr: VR::SS },
            E { tag: 0x00551053, alias: "Zero Location (T=Top B=Bottom)", vr: VR::CS },
            E { tag: 0x00551054, alias: "Angular Step", vr: VR::FL },
            E { tag: 0x00551055, alias: "Start Angle", vr: VR::FL },
            E { tag: 0x00551056, alias: "Decay Corr. Flag 1 Corr", vr: VR::SS },
            E { tag: 0x00551057, alias: "Scatter Corr. Flag 1 Corr", vr: VR::SS },
            E { tag: 0x00551058, alias: "Radial Map Info in 0.1 mm Inc.", vr: VR::FL },
            E { tag: 0x00551059, alias: "Direction H HeadIn F FeetIn", vr: VR::CS },
            E { tag: 0x0055105A, alias: "Table Traverse cm", vr: VR::FL },
            E { tag: 0x0055105B, alias: "Multiplexor Flag 1 Multiplexed", vr: VR::SS },
            E { tag: 0x0055105C, alias: "Pixel Acquisition Date", vr: VR::DS },
            E { tag: 0x0055105D, alias: "Angle Type 1 360o 2 128 Steps", vr: VR::SS },
            E { tag: 0x0055105E, alias: "Orbit Definition", vr: VR::SL },
            E { tag: 0x0055105F, alias: "Arm Length in cm", vr: VR::FL },
            E { tag: 0x00551060, alias: "Auto Window Beats", vr: VR::SS },
            E { tag: 0x00551061, alias: "Non-gated File Extension", vr: VR::SS },
            E { tag: 0x00551064, alias: "TIC Error Values", vr: VR::SS },
            E { tag: 0x00551065, alias: "X1", vr: VR::SS },
            E { tag: 0x00551066, alias: "Y1", vr: VR::SS },
            E { tag: 0x00551067, alias: "X2", vr: VR::SS },
            E { tag: 0x00551068, alias: "Y2", vr: VR::SS },
            E { tag: 0x00551069, alias: "X3", vr: VR::SS },
            E { tag: 0x0055106A, alias: "Y3", vr: VR::SS },
            E { tag: 0x0055106B, alias: "X4", vr: VR::SS },
            E { tag: 0x0055106C, alias: "Y4", vr: VR::SS },
            E { tag: 0x0055106D, alias: "Pixel Size Flag 1 Valid Size", vr: VR::SS },
            E { tag: 0x00551070, alias: "X Focal Length", vr: VR::FL },
            E { tag: 0x00551071, alias: "Y Focal Length", vr: VR::FL },
            E { tag: 0x00551072, alias: "X Gains", vr: VR::FL },
            E { tag: 0x00551073, alias: "Y Gains", vr: VR::FL },
            E { tag: 0x00551074, alias: "X Offset", vr: VR::FL },
            E { tag: 0x00551075, alias: "Y Offset", vr: VR::FL },
            E { tag: 0x00551076, alias: "X Principal Ray", vr: VR::FL },
            E { tag: 0x00551077, alias: "Y Principal Ray", vr: VR::FL },
            E { tag: 0x00551078, alias: "Total Flood Counts", vr: VR::FL },
            E { tag: 0x0055107E, alias: "Collimator thickness", vr: VR::FL },
            E { tag: 0x0055107F, alias: "Collimator angular resolution", vr: VR::FL },
            E { tag: 0x00551080, alias: "File Altered Flag 1 Altered", vr: VR::SS },
            E { tag: 0x00551081, alias: "Raw Data Extension", vr: VR::SS },
            E { tag: 0x00551082, alias: "Sagittal Extension", vr: VR::SS },
            E { tag: 0x00551083, alias: "Coronal Extension", vr: VR::SS },
            E { tag: 0x00551084, alias: "Transverse Extension", vr: VR::SS },
            E { tag: 0x00551085, alias: "Volume Rendered Extension", vr: VR::SS },
            E { tag: 0x00551086, alias: "Plane of Transverse Bottom Slice", vr: VR::SL },
            E { tag: 0x00551087, alias: "Plane of Transverse Top Slice", vr: VR::SL },
            E { tag: 0x00551088, alias: "Slice Step in Pixels", vr: VR::SL },
            E { tag: 0x0055108A, alias: "Y-Axis Smooth Slice Weights", vr: VR::SL },
            E { tag: 0x0055108B, alias: "Number of Transverse Slices", vr: VR::SL },
            E { tag: 0x0055108C, alias: "Attenuation Flag 1 Chang 2 Sorenson", vr: VR::SS },
            E { tag: 0x0055108D, alias: "Sagittal Right Limit", vr: VR::SS },
            E { tag: 0x0055108E, alias: "Sagittal Left Limit", vr: VR::SS },
            E { tag: 0x0055108F, alias: "Coronal Posterior Limit", vr: VR::SS },
            E { tag: 0x00551090, alias: "Coronal Anterior Limit", vr: VR::SS },
            E { tag: 0x00551091, alias: "Isotope Used in Reconstruction", vr: VR::SS },
            E { tag: 0x00551093, alias: "Filter Cutoff Frequency", vr: VR::FL },
            E { tag: 0x00551094, alias: "Filter Alpha Coefficient", vr: VR::FL },
            E { tag: 0x00551095, alias: "Filter Order", vr: VR::FL },
            E { tag: 0x00551096, alias: "Attenuation Coefficient", vr: VR::FL },
            E { tag: 0x00551097, alias: "Percent Ray", vr: VR::SS },
            E { tag: 0x00551098, alias: "Astigmatic Collimator 1 Neuro 2 Cardiac", vr: VR::SS },
            E { tag: 0x00551099, alias: "XP0", vr: VR::SS },
            E { tag: 0x0055109A, alias: "ZP0", vr: VR::SS },
            E { tag: 0x0055109B, alias: "XP1", vr: VR::SS },
            E { tag: 0x0055109C, alias: "ZP1", vr: VR::SS },
            E { tag: 0x0055109D, alias: "Rotation Matrix", vr: VR::FL },
            E { tag: 0x0055109E, alias: "X0", vr: VR::FL },
            E { tag: 0x0055109F, alias: "X1", vr: VR::FL },
            E { tag: 0x005510A0, alias: "Y0", vr: VR::FL },
            E { tag: 0x005510A1, alias: "Y1", vr: VR::FL },
            E { tag: 0x005510A2, alias: "Z0", vr: VR::FL },
            E { tag: 0x005510A3, alias: "Z1", vr: VR::FL },
            E { tag: 0x005510A4, alias: "Transverse Scale", vr: VR::SS },
            E { tag: 0x005510A5, alias: "Pixel Size X Direction mm", vr: VR::FL },
            E { tag: 0x005510A6, alias: "Pixel Size Y Direction mm", vr: VR::FL },
            E { tag: 0x005510A7, alias: "Regions File Extension", vr: VR::SS },
            E { tag: 0x005510A8, alias: "Gated SPECT Bins Reconstructed", vr: VR::SS },
            E { tag: 0x005510A9, alias: "Vertical Oblique Image", vr: VR::SS },
            E { tag: 0x005510B0, alias: "Oblique Region First Slice", vr: VR::SS },
            E { tag: 0x005510B1, alias: "Oblique Region Last Slice", vr: VR::SS },
            E { tag: 0x005510C0, alias: "Unknown", vr: VR::SS },
            E { tag: 0x005510C2, alias: "Unknown", vr: VR::SS },
            E { tag: 0x005510C3, alias: "Unknown", vr: VR::SS },
            E { tag: 0x005510C4, alias: "Unknown", vr: VR::SS },
            E { tag: 0x005510D0, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00571001, alias: "e.soft DICOM original image type", vr: VR::LO },
            E { tag: 0x00571002, alias: "Dose calibration factor", vr: VR::FL },
            E { tag: 0x00571003, alias: "Units", vr: VR::LO },
            E { tag: 0x00571004, alias: "Decay correction", vr: VR::LO },
            E { tag: 0x00571005, alias: "Radio nuclide half life", vr: VR::SL },
            E { tag: 0x00571006, alias: "Rescale intercept", vr: VR::FL },
            E { tag: 0x00571007, alias: "Rescale Slope", vr: VR::FL },
            E { tag: 0x00571008, alias: "Frame reference time", vr: VR::FL },
            E { tag: 0x00571009, alias: "Number of Radiopharmaceutical information seq", vr: VR::SL },
            E { tag: 0x0057100A, alias: "Decay factor", vr: VR::FL },
            E { tag: 0x0057100B, alias: "Counts source", vr: VR::LO },
            E { tag: 0x0057100C, alias: "Radio Nuclide positron fraction", vr: VR::SL },
            E { tag: 0x00571010, alias: "Low Count", vr: VR::SS },
            E { tag: 0x00571020, alias: "Vertical Oblique Image", vr: VR::SS },
            E { tag: 0x50011002, alias: "Number of Points", vr: VR::SL },
            E { tag: 0x50011003, alias: "Minimal Value on X-Axis", vr: VR::FL },
            E { tag: 0x50011004, alias: "Maximal Value on X-Axis", vr: VR::FL },
            E { tag: 0x50011005, alias: "Minimal Value on Y-Axis", vr: VR::FL },
            E { tag: 0x50011006, alias: "Maximal Value on Y-Axis", vr: VR::FL },
            E { tag: 0x50011007, alias: "Curve Name", vr: VR::LO },
            E { tag: 0x50011008, alias: "Curve Function Label", vr: VR::LO },
            E { tag: 0x50011009, alias: "X Axis Label", vr: VR::SH },
            E { tag: 0x5001100A, alias: "Y Axis Label", vr: VR::SH },
            E { tag: 0x5001100B, alias: "Curve Data", vr: VR::OW },
            E { tag: 0x7FE11010, alias: "Compressed Pixel Data", vr: VR::OW },
            E { tag: 0x7FE31000, alias: "Length of Group 0x7FE3", vr: VR::UL },
            E { tag: 0x7FE31014, alias: "Lowest Pixel Value in THIS Frame", vr: VR::SL },
            E { tag: 0x7FE31015, alias: "Highest Pixel Value in THIS Frame", vr: VR::SL },
            E { tag: 0x7FE31016, alias: "Total Counts in THIS Frame", vr: VR::SL },
            E { tag: 0x7FE31017, alias: "Duration of THIS Frame msec", vr: VR::SL },
            E { tag: 0x7FE31018, alias: "Frame Angle", vr: VR::FL },
            E { tag: 0x7FE31019, alias: "Frame Name", vr: VR::LO },
            E { tag: 0x7FE3101A, alias: "View Number", vr: VR::FL },
            E { tag: 0x7FE3101B, alias: "Pixel Overflow Flag 1 Pixel Overflow", vr: VR::SS },
            E { tag: 0x7FE3101C, alias: "Buffer Overflow Flag 1 Buffer Overflow", vr: VR::SS },
            E { tag: 0x7FE3101E, alias: "Start Time of THIS Frame From Start of Study", vr: VR::SL },
            E { tag: 0x7FE3101F, alias: "ROI Maximum Pixel Value", vr: VR::SL },
            E { tag: 0x7FE31020, alias: "ROI Average Pixel Value", vr: VR::SL },
            E { tag: 0x7FE31021, alias: "ROI Total Counts", vr: VR::SL },
            E { tag: 0x7FE31022, alias: "ROI Total Pixels", vr: VR::SL },
            E { tag: 0x7FE31023, alias: "ROI Center Coordinates", vr: VR::SL },
            E { tag: 0x7FE31024, alias: "ROI Name", vr: VR::LO },
            E { tag: 0x7FE31025, alias: "ROI Data", vr: VR::SS },
            E { tag: 0x7FE31026, alias: "Unknown", vr: VR::OW },
            E { tag: 0x7FE31027, alias: "Unknown", vr: VR::OW },
            E { tag: 0x7FE31028, alias: "Unknown", vr: VR::OW },
            E { tag: 0x7FE31029, alias: "Unknown", vr: VR::OW },
        ],
    },
    P {
        creator: "SIEMENS MED OCS ACKNOWLEDGE FLAG",
        entries: &[
            E { tag: 0x00391081, alias: "Acknowledge Flag", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS AE TITLE",
        entries: &[
            E { tag: 0x00391079, alias: "AE Title", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CALIBRATION DATE",
        entries: &[
            E { tag: 0x00391081, alias: "Calibration Date", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT ARC INCREMENT",
        entries: &[
            E { tag: 0x00391084, alias: "Arc Increment", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT ENDING GANTRY ANGLE",
        entries: &[
            E { tag: 0x00391083, alias: "Ending Projection Angle", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT MISSING PROJECTION FRAMES",
        entries: &[
            E { tag: 0x00391091, alias: "Missing Projection Frames", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT PATIENT_ORIENTATION",
        entries: &[
            E { tag: 0x00391093, alias: "Patient Orientation", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT PROJECTION GANTRY POSITIONS",
        entries: &[
            E { tag: 0x00391086, alias: "Gantry Positions", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT RECON FILTER TYPE",
        entries: &[
            E { tag: 0x00391090, alias: "Reconstruction Filter Type", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT RECON NUM OF SLICES",
        entries: &[
            E { tag: 0x00391089, alias: "Number of Reconstruction Slices", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT RECON SLICE HEIGHT",
        entries: &[
            E { tag: 0x00391088, alias: "Reconstruction Slice Height", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT RECON SLICE WIDTH",
        entries: &[
            E { tag: 0x00391087, alias: "Reconstruction Slice Width", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT ROTATION DIRECTION",
        entries: &[
            E { tag: 0x00391085, alias: "Gantry Rotation Direction", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT STARTING GANTRY ANGLE",
        entries: &[
            E { tag: 0x00391082, alias: "Starting Projection Angle", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS CBT TOTAL_MU",
        entries: &[
            E { tag: 0x00391096, alias: "Total MU delivered during Cone Beam", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS COMMON EDGE",
        entries: &[
            E { tag: 0x00391083, alias: "Common Edge", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS FIELD ID",
        entries: &[
            E { tag: 0x00391080, alias: "Field ID", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS FIELD NAME",
        entries: &[
            E { tag: 0x00391079, alias: "Field Name", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS FP Alignment Calibration Date / Time",
        entries: &[
            E { tag: 0x00391095, alias: "Flat Panel Alignment last calibration Date and Time", vr: VR::DT },
        ],
    },
    P {
        creator: "SIEMENS MED OCS I0 MU SCALE FACTOR",
        entries: &[
            E { tag: 0x00391097, alias: "I0-MU Scale Factor", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS ICC MATRIX",
        entries: &[
            E { tag: 0x00391082, alias: "Image Center in Image Coordinates", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS NUMBER OF FRAMES",
        entries: &[
            E { tag: 0x00391076, alias: "Number of Frames", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS NUMBER OF SUB FRAMES",
        entries: &[
            E { tag: 0x00391077, alias: "Number of Sub-Frames", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS PI ALIGNMENT IMAGE",
        entries: &[
            E { tag: 0x00391094, alias: "Alignment Offset Valid Flag", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS PI VALID PIXEL MAP",
        entries: &[
            E { tag: 0x00391092, alias: "Valid Pixel Map Flag", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS REFERENCE IMAGE",
        entries: &[
            E { tag: 0x00391076, alias: "Reference Image", vr: VR::UI },
        ],
    },
    P {
        creator: "SIEMENS MED OCS REFERENCE LOCKED",
        entries: &[
            E { tag: 0x00391088, alias: "Locked Flag", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS SC IMG PIX SPC",
        entries: &[
            E { tag: 0x00391087, alias: "Non-RT Image Pixel Spacing", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS SC IMG SAD",
        entries: &[
            E { tag: 0x00391086, alias: "Non-RT Image SAD", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS SC IMG SID",
        entries: &[
            E { tag: 0x00391085, alias: "Non-RT Image SID", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS SCALED IMG PIX SPC",
        entries: &[
            E { tag: 0x00391084, alias: "Scaled Image Pixel Spacing", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS SITE NAME",
        entries: &[
            E { tag: 0x00391078, alias: "Site Name", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS TABLE LATERAL POSITION",
        entries: &[
            E { tag: 0x00391098, alias: "Lateral Table Position", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS TABLE LONGITUDINAL POSITION",
        entries: &[
            E { tag: 0x00391099, alias: "Longitudinal Table Position", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS TABLE OFFSET",
        entries: &[
            E { tag: 0x00391078, alias: "Offset", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS TABLE VERTICAL POSITION",
        entries: &[
            E { tag: 0x00391075, alias: "vertical Table Position", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED OCS TRANS MATRIX",
        entries: &[
            E { tag: 0x00391080, alias: "Transformation Matrix", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MED SMS USG ANTARES",
        entries: &[
            E { tag: 0x00191000, alias: "Private Creator Version", vr: VR::SH },
            E { tag: 0x00191003, alias: "Frame Rate", vr: VR::UN },
            E { tag: 0x0019100C, alias: "Burned in Graphics", vr: VR::US },
            E { tag: 0x0019100D, alias: "SieClear Index", vr: VR::UN },
            E { tag: 0x00191020, alias: "B-Mode Submode", vr: VR::UN },
            E { tag: 0x00191021, alias: "B-Mode Dynamic Range", vr: VR::UN },
            E { tag: 0x00191022, alias: "B-Mode Overall Gain", vr: VR::UN },
            E { tag: 0x00191023, alias: "B-Mode Res/Speed Index", vr: VR::UN },
            E { tag: 0x00191024, alias: "B-Mode Edge Enhance Index", vr: VR::UN },
            E { tag: 0x00191025, alias: "B-Mode Persistence Index", vr: VR::UN },
            E { tag: 0x00191026, alias: "B-Mode Map Index", vr: VR::UN },
            E { tag: 0x00191027, alias: "???", vr: VR::US },
            E { tag: 0x00191028, alias: "???", vr: VR::US },
            E { tag: 0x00191029, alias: "???", vr: VR::US },
            E { tag: 0x0019102A, alias: "B-Mode Tint Type", vr: VR::US },
            E { tag: 0x0019102D, alias: "B-Mode Tint Index", vr: VR::UN },
            E { tag: 0x0019102E, alias: "ClarifyVE Index", vr: VR::UN },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019103A, alias: "Image Flag", vr: VR::UN },
            E { tag: 0x0019103B, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191040, alias: "Color Flow state", vr: VR::UN },
            E { tag: 0x00191041, alias: "Color Flow Wall Filter Index", vr: VR::UN },
            E { tag: 0x00191042, alias: "Color Flow Submode", vr: VR::UN },
            E { tag: 0x00191043, alias: "Color Flow Overall Gain", vr: VR::UN },
            E { tag: 0x00191044, alias: "Color Flow Res/Speed Index", vr: VR::UN },
            E { tag: 0x00191046, alias: "Color Flow Smooth Index", vr: VR::UN },
            E { tag: 0x00191047, alias: "Color Flow Persistence Index", vr: VR::UN },
            E { tag: 0x00191048, alias: "Color Flow Map Index", vr: VR::UN },
            E { tag: 0x00191049, alias: "Color Flow Priority Index", vr: VR::UN },
            E { tag: 0x00191054, alias: "Color Flow Maximum Velocity", vr: VR::UN },
            E { tag: 0x00191060, alias: "Doppler Dynamic Range", vr: VR::UN },
            E { tag: 0x00191061, alias: "Doppler Overall Gain", vr: VR::UN },
            E { tag: 0x00191062, alias: "Doppler Wall Filter", vr: VR::UN },
            E { tag: 0x00191063, alias: "Doppler Gate Size", vr: VR::UN },
            E { tag: 0x00191065, alias: "Doppler Map Index", vr: VR::UN },
            E { tag: 0x00191066, alias: "Doppler Submode", vr: VR::UN },
            E { tag: 0x00191067, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191069, alias: "Doppler Time/Freq Res Index", vr: VR::UN },
            E { tag: 0x0019106A, alias: "Doppler Trace Inverted", vr: VR::US },
            E { tag: 0x0019106C, alias: "Doppler Tint Type", vr: VR::US },
            E { tag: 0x00191072, alias: "Doppler Tint Index", vr: VR::UN },
            E { tag: 0x00191080, alias: "M-Mode Dynamic Range", vr: VR::UN },
            E { tag: 0x00191081, alias: "M-Mode Overall Gain", vr: VR::UN },
            E { tag: 0x00191082, alias: "M-Mode Edge Enhance Index", vr: VR::UN },
            E { tag: 0x00191083, alias: "M-Mode Map Index", vr: VR::UN },
            E { tag: 0x00191086, alias: "M-Mode Tint Type", vr: VR::UN },
            E { tag: 0x00191087, alias: "M-Mode Submode", vr: VR::UN },
            E { tag: 0x00191088, alias: "M-Mode Tint Index", vr: VR::UN },
            E { tag: 0x001910A0, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "SIEMENS MED SMS USG ANTARES 3D VOLUME",
        entries: &[
            E { tag: 0x00391000, alias: "Release Version", vr: VR::UN },
            E { tag: 0x00391003, alias: "VolumeAcquisitionDuration", vr: VR::UN },
            E { tag: 0x00391004, alias: "VolumeRawDataType", vr: VR::UN },
            E { tag: 0x00391005, alias: "ScanType", vr: VR::UN },
            E { tag: 0x00391006, alias: "ZlateralMin", vr: VR::UN },
            E { tag: 0x00391007, alias: "ZlateralSpan", vr: VR::UN },
            E { tag: 0x00391008, alias: "ZRadiusOfCurvature", vr: VR::UN },
            E { tag: 0x00391009, alias: "WobbleCorrection", vr: VR::UN },
            E { tag: 0x00391010, alias: "ScaleAlongWidth", vr: VR::UN },
            E { tag: 0x00391011, alias: "ScaleAlongHeight", vr: VR::UN },
            E { tag: 0x00391012, alias: "ScaleAlongDepth", vr: VR::UN },
            E { tag: 0x00391013, alias: "BufferSize", vr: VR::UN },
            E { tag: 0x00391014, alias: "AcquisitionRate", vr: VR::UN },
            E { tag: 0x00391015, alias: "DepthMinCm", vr: VR::UN },
            E { tag: 0x00391016, alias: "IsLeftRightFlippedEn", vr: VR::UN },
            E { tag: 0x00391017, alias: "IsUpDownFlippedEn", vr: VR::UN },
            E { tag: 0x00391018, alias: "IsVolumeGeomAccurate", vr: VR::UN },
            E { tag: 0x00391019, alias: "BByteMaskOffset", vr: VR::UN },
            E { tag: 0x00391020, alias: "BByteMaskSize", vr: VR::UN },
            E { tag: 0x00391021, alias: "DepthMaxCm", vr: VR::UN },
            E { tag: 0x00391022, alias: "AcqPlaneRotationDeg", vr: VR::UN },
            E { tag: 0x00391023, alias: "BeamAxialSpan", vr: VR::UN },
            E { tag: 0x00391024, alias: "BeamLateralMin", vr: VR::UN },
            E { tag: 0x00391025, alias: "BeamLateralSpan", vr: VR::UN },
            E { tag: 0x00391026, alias: "BeamAxialMin", vr: VR::UN },
            E { tag: 0x00391027, alias: "NumDisplaySamples", vr: VR::UN },
            E { tag: 0x00391028, alias: "DVolumeWidth", vr: VR::UN },
            E { tag: 0x00391029, alias: "DVolumeDepth", vr: VR::UN },
            E { tag: 0x00391030, alias: "DVolumeHeight", vr: VR::UN },
            E { tag: 0x00391031, alias: "DVolumePosX", vr: VR::UN },
            E { tag: 0x00391032, alias: "DVolumePosY", vr: VR::UN },
            E { tag: 0x00391033, alias: "DVolumePosZ", vr: VR::UN },
            E { tag: 0x00391034, alias: "DBeamAxialMin", vr: VR::UN },
            E { tag: 0x00391035, alias: "DBeamAxialSpan", vr: VR::UN },
            E { tag: 0x00391036, alias: "DBeamLateralMin", vr: VR::UN },
            E { tag: 0x00391037, alias: "DBeamLateralSpan", vr: VR::UN },
            E { tag: 0x00391038, alias: "NumOfVolumesInSequence", vr: VR::UN },
            E { tag: 0x00391039, alias: "DByteMaskOffset", vr: VR::UN },
            E { tag: 0x00391040, alias: "DByteMaskSize", vr: VR::UN },
            E { tag: 0x00391050, alias: "PrivateCreatorVersionOfBookmark", vr: VR::UN },
            E { tag: 0x00391051, alias: "BCutPlaneEnable", vr: VR::UN },
            E { tag: 0x00391052, alias: "BMprColorMapIndex", vr: VR::UN },
            E { tag: 0x00391053, alias: "BMprDynamicRangeDb", vr: VR::UN },
            E { tag: 0x00391054, alias: "BMprGrayMapIndex", vr: VR::UN },
            E { tag: 0x00391055, alias: "BVolumeRenderMode", vr: VR::UN },
            E { tag: 0x00391056, alias: "BVrBrightness", vr: VR::UN },
            E { tag: 0x00391057, alias: "BVrContrast", vr: VR::UN },
            E { tag: 0x00391058, alias: "BVrColorMapIndex", vr: VR::UN },
            E { tag: 0x00391059, alias: "BVrDynamicRangeDb", vr: VR::UN },
            E { tag: 0x0039105A, alias: "BVrGrayMapIndex", vr: VR::UN },
            E { tag: 0x0039105B, alias: "BVrOpacity", vr: VR::UN },
            E { tag: 0x0039105C, alias: "BVrThresholdHigh", vr: VR::UN },
            E { tag: 0x0039105D, alias: "BVrThresholdLow", vr: VR::UN },
            E { tag: 0x0039105E, alias: "BPreProcessFilterMix", vr: VR::UN },
            E { tag: 0x0039105F, alias: "CCutPlaneEnable", vr: VR::UN },
            E { tag: 0x00391060, alias: "CFrontClipMode", vr: VR::UN },
            E { tag: 0x00391061, alias: "CMprColorMapIndex", vr: VR::UN },
            E { tag: 0x00391062, alias: "CMprColorFlowPriorityIndex", vr: VR::UN },
            E { tag: 0x00391063, alias: "CVolumeRenderMode", vr: VR::UN },
            E { tag: 0x00391064, alias: "CVrColorMapIndex", vr: VR::UN },
            E { tag: 0x00391065, alias: "CVrColorFlowPriorityIndex", vr: VR::UN },
            E { tag: 0x00391066, alias: "CVrOpacity", vr: VR::UN },
            E { tag: 0x00391067, alias: "CVrThresholdHigh", vr: VR::UN },
            E { tag: 0x00391068, alias: "CVrThresholdLow", vr: VR::UN },
            E { tag: 0x00391069, alias: "VoiMode", vr: VR::UN },
            E { tag: 0x0039106A, alias: "VoiRotationOffsetDeg", vr: VR::UN },
            E { tag: 0x0039106B, alias: "VoiSizeRatioX", vr: VR::UN },
            E { tag: 0x0039106C, alias: "VoiSizeRatioY", vr: VR::UN },
            E { tag: 0x0039106D, alias: "VoiSizeRatioZ", vr: VR::UN },
            E { tag: 0x0039106E, alias: "VoiSyncPlane", vr: VR::UN },
            E { tag: 0x0039106F, alias: "VoiViewMode", vr: VR::UN },
            E { tag: 0x00391070, alias: "VrOrientationA", vr: VR::UN },
            E { tag: 0x00391071, alias: "MprOrientationA", vr: VR::UN },
            E { tag: 0x00391072, alias: "VrOffsetVector", vr: VR::UN },
            E { tag: 0x00391073, alias: "BlendingRatio", vr: VR::UN },
            E { tag: 0x00391074, alias: "FusionBlendMode", vr: VR::UN },
            E { tag: 0x00391075, alias: "QualityFactor", vr: VR::UN },
            E { tag: 0x00391076, alias: "RendererType", vr: VR::UN },
            E { tag: 0x00391077, alias: "SliceMode", vr: VR::UN },
            E { tag: 0x00391078, alias: "ActiveQuad", vr: VR::UN },
            E { tag: 0x00391079, alias: "ScreenMode", vr: VR::UN },
            E { tag: 0x0039107A, alias: "CutPlaneSide", vr: VR::UN },
            E { tag: 0x0039107B, alias: "WireframeMode", vr: VR::UN },
            E { tag: 0x0039107C, alias: "CrossmarkMode", vr: VR::UN },
            E { tag: 0x0039107D, alias: "MprDisplayType", vr: VR::UN },
            E { tag: 0x0039107E, alias: "VolumeDisplayType", vr: VR::UN },
            E { tag: 0x0039107F, alias: "LastReset", vr: VR::UN },
            E { tag: 0x00391080, alias: "LastNonFullScreenMode", vr: VR::UN },
            E { tag: 0x00391081, alias: "MprToolIndex", vr: VR::UN },
            E { tag: 0x00391082, alias: "VoiToolIndex", vr: VR::UN },
            E { tag: 0x00391083, alias: "ToolLoopMode", vr: VR::UN },
            E { tag: 0x00391084, alias: "VolumeArbMode", vr: VR::UN },
            E { tag: 0x00391085, alias: "MprZoomEn", vr: VR::UN },
            E { tag: 0x00391086, alias: "IsVolumeZoomEn", vr: VR::UN },
            E { tag: 0x00391087, alias: "ZoomLevelMpr", vr: VR::UN },
            E { tag: 0x00391088, alias: "ZoomLevelVolume", vr: VR::UN },
            E { tag: 0x00391089, alias: "IsAutoRotateEn", vr: VR::UN },
            E { tag: 0x0039108A, alias: "AutoRotateAxis", vr: VR::UN },
            E { tag: 0x0039108B, alias: "AutoRotateRangeIndex", vr: VR::UN },
            E { tag: 0x0039108C, alias: "AutoRotateSpeedIndex", vr: VR::UN },
            E { tag: 0x0039108D, alias: "CVrBrightness", vr: VR::UN },
            E { tag: 0x0039108E, alias: "CFlowStateIndex", vr: VR::UN },
            E { tag: 0x0039108F, alias: "BSubmodeIndex", vr: VR::UN },
            E { tag: 0x00391090, alias: "CSubmodeIndex", vr: VR::UN },
            E { tag: 0x00391091, alias: "DICOMAttrNameCutPlane", vr: VR::UN },
            E { tag: 0x00391092, alias: "BookmarkChunkId", vr: VR::UN },
            E { tag: 0x00391093, alias: "SequenceMinChunkId", vr: VR::UN },
            E { tag: 0x00391094, alias: "SequenceMaxChunkId", vr: VR::UN },
            E { tag: 0x00391095, alias: "VolumeRateHz", vr: VR::UN },
            E { tag: 0x0039109A, alias: "VoiPositionOffsetX", vr: VR::UN },
            E { tag: 0x0039109B, alias: "VoiPositionOffsetY", vr: VR::UN },
            E { tag: 0x0039109C, alias: "VoiPositionOffsetZ", vr: VR::UN },
            E { tag: 0x0039109D, alias: "VrToolIndex", vr: VR::UN },
            E { tag: 0x0039109E, alias: "ShadingPercent", vr: VR::UN },
            E { tag: 0x0039109F, alias: "VolumeType", vr: VR::UN },
            E { tag: 0x003910A0, alias: "DICOMAttrNameVrQuadDisplayType", vr: VR::UN },
            E { tag: 0x003910A1, alias: "DICOMAttrNameMprCenterLocation", vr: VR::UN },
            E { tag: 0x003910E0, alias: "DICOMAttrNameSliceRangeType", vr: VR::UN },
            E { tag: 0x003910E1, alias: "DICOMAttrNameSliceMPRPlane", vr: VR::UN },
            E { tag: 0x003910E2, alias: "DICOMAttrNameSliceLayout", vr: VR::UN },
            E { tag: 0x003910E3, alias: "DICOMAttrNameSliceSpacing", vr: VR::UN },
            E { tag: 0x003910E4, alias: "DICOMAttrNameThinVrMode", vr: VR::UN },
            E { tag: 0x003910E5, alias: "DICOMAttrNameThinVrThickness", vr: VR::UN },
            E { tag: 0x003910E6, alias: "DICOMAttrNameVoiPivotX", vr: VR::UN },
            E { tag: 0x003910E7, alias: "DICOMAttrNameVoiPivotY", vr: VR::UN },
            E { tag: 0x003910E8, alias: "DICOMAttrNameVoiPivotZ", vr: VR::UN },
            E { tag: 0x003910E9, alias: "DICOMAttrNameCTopVoiQuad", vr: VR::UN },
        ],
    },
    P {
        creator: "SIEMENS MED SMS USG S2000",
        entries: &[
            E { tag: 0x00191000, alias: "Private Creator Version", vr: VR::UN },
            E { tag: 0x00191003, alias: "Frame Rate", vr: VR::FD },
            E { tag: 0x0019100C, alias: "Burned in Graphics", vr: VR::US },
            E { tag: 0x0019100D, alias: "SieClear Index", vr: VR::SH },
            E { tag: 0x00191020, alias: "B-Mode Submode", vr: VR::SH },
            E { tag: 0x00191021, alias: "B-Mode Dynamic Range", vr: VR::FD },
            E { tag: 0x00191022, alias: "B-Mode Overall Gain", vr: VR::FD },
            E { tag: 0x00191023, alias: "B-Mode Res/Speed Index", vr: VR::US },
            E { tag: 0x00191024, alias: "B-Mode Edge Enhance Index", vr: VR::US },
            E { tag: 0x00191025, alias: "B-Mode Persistence Index", vr: VR::US },
            E { tag: 0x00191026, alias: "B-Mode Map Index", vr: VR::US },
            E { tag: 0x00191027, alias: "??", vr: VR::US },
            E { tag: 0x00191028, alias: "??", vr: VR::US },
            E { tag: 0x00191029, alias: "??", vr: VR::US },
            E { tag: 0x0019102A, alias: "B-Mode Tint Type", vr: VR::US },
            E { tag: 0x0019102D, alias: "B-Mode Tint Index", vr: VR::US },
            E { tag: 0x0019102E, alias: "ClarifyVE Index", vr: VR::SH },
            E { tag: 0x0019103A, alias: "Image Flag", vr: VR::UN },
            E { tag: 0x0019103B, alias: "IPDA/MDA Data", vr: VR::LT },
            E { tag: 0x0019103C, alias: "Thumbnail Data", vr: VR::UT },
            E { tag: 0x00191040, alias: "Color Flow state", vr: VR::UN },
            E { tag: 0x00191041, alias: "Color Flow Wall Filter Index", vr: VR::UN },
            E { tag: 0x00191042, alias: "Color Flow Submode", vr: VR::UN },
            E { tag: 0x00191043, alias: "Color Flow Overall Gain", vr: VR::UN },
            E { tag: 0x00191044, alias: "Color Flow Res/Speed Index", vr: VR::UN },
            E { tag: 0x00191046, alias: "Color Flow Smooth Index", vr: VR::UN },
            E { tag: 0x00191047, alias: "Color Flow Persistence Index", vr: VR::UN },
            E { tag: 0x00191048, alias: "Color Flow Map Index", vr: VR::UN },
            E { tag: 0x00191049, alias: "Color Flow Priority Index", vr: VR::UN },
            E { tag: 0x00191054, alias: "Color Flow Maximum Velocity", vr: VR::UN },
            E { tag: 0x00191060, alias: "Doppler Dynamic Range", vr: VR::UN },
            E { tag: 0x00191061, alias: "Doppler Overall Gain", vr: VR::UN },
            E { tag: 0x00191062, alias: "S2000 Private Doppler Wall Filter", vr: VR::UN },
            E { tag: 0x00191063, alias: "Attributes Doppler Gate Size", vr: VR::UN },
            E { tag: 0x00191065, alias: "Doppler Map Index", vr: VR::UN },
            E { tag: 0x00191066, alias: "Doppler Submode", vr: VR::UN },
            E { tag: 0x00191069, alias: "Doppler Time/Freq Res Index", vr: VR::UN },
            E { tag: 0x0019106A, alias: "Doppler Trace Inverted", vr: VR::UN },
            E { tag: 0x0019106C, alias: "Doppler Tint Type", vr: VR::UN },
            E { tag: 0x00191072, alias: "Doppler Tint Index", vr: VR::UN },
            E { tag: 0x00191080, alias: "M-Mode Dynamic Range", vr: VR::UN },
            E { tag: 0x00191081, alias: "M-Mode Overall Gain", vr: VR::UN },
            E { tag: 0x00191082, alias: "M-Mode Edge Enhance Index", vr: VR::UN },
            E { tag: 0x00191083, alias: "M-Mode Map Index", vr: VR::UN },
            E { tag: 0x00191086, alias: "M-Mode Tint Type", vr: VR::UN },
            E { tag: 0x00191087, alias: "M-Mode Submode", vr: VR::UN },
            E { tag: 0x00191088, alias: "M-Mode Tint Index", vr: VR::UN },
        ],
    },
    P {
        creator: "SIEMENS MED SP DXMG WH AWS 1",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00411002, alias: "Reason for the Requested Procedure", vr: VR::SH },
            E { tag: 0x00551001, alias: "Projection View Display String", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS MEDCOM HEADER",
        entries: &[
            E { tag: 0x00291008, alias: "MedCom Header Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "MedCom Header Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "MedCom Header Info", vr: VR::OB },
            E { tag: 0x00291020, alias: "MedCom History Information", vr: VR::OB },
            E { tag: 0x00291031, alias: "PMTF Information 1", vr: VR::LO },
            E { tag: 0x00291032, alias: "PMTF Information 2", vr: VR::UL },
            E { tag: 0x00291033, alias: "PMTF Information 3", vr: VR::UL },
            E { tag: 0x00291034, alias: "PMTF Information 4", vr: VR::CS },
            E { tag: 0x00291035, alias: "PMTF Information 5", vr: VR::UL },
            E { tag: 0x00291040, alias: "Application Header Sequence", vr: VR::SQ },
            E { tag: 0x00291041, alias: "Application Header Type", vr: VR::CS },
            E { tag: 0x00291042, alias: "Application Header ID", vr: VR::LO },
            E { tag: 0x00291043, alias: "Application Header Version", vr: VR::LO },
            E { tag: 0x00291044, alias: "Application Header Info", vr: VR::OB },
            E { tag: 0x00291050, alias: "Workflow Control Flags", vr: VR::LO },
            E { tag: 0x00291051, alias: "Arch. Management Flag Keep Online", vr: VR::CS },
            E { tag: 0x00291052, alias: "Arch. Mgmnt Flag Do Not Archive", vr: VR::CS },
            E { tag: 0x00291053, alias: "Image Location Status", vr: VR::CS },
            E { tag: 0x00291054, alias: "Estimated Retrieve Time", vr: VR::DS },
            E { tag: 0x00291055, alias: "Data Size of Retrieved Images", vr: VR::DS },
            E { tag: 0x00291070, alias: "Siemens Link Sequence", vr: VR::SQ },
            E { tag: 0x00291071, alias: "Referenced Tag", vr: VR::AT },
            E { tag: 0x00291072, alias: "Referenced Tag Type", vr: VR::CS },
            E { tag: 0x00291073, alias: "Referenced Value Length", vr: VR::UL },
            E { tag: 0x00291074, alias: "Referenced Object Device Type", vr: VR::CS },
            E { tag: 0x00291075, alias: "Referenced Object Device Location", vr: VR::OB },
            E { tag: 0x00291076, alias: "Referenced Object ID", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS MEDCOM HEADER2",
        entries: &[
            E { tag: 0x00291060, alias: "Series Workflow Status", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS MEDCOM OOG",
        entries: &[
            E { tag: 0x00291008, alias: "MedCom OOG Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "MedCom OOG Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "MedCom OOG Info", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS MR EXTRACTED CSA HEADER",
        entries: &[
            E { tag: 0x00250001, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00250002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00250003, alias: "Unknown", vr: VR::AT },
            E { tag: 0x00250004, alias: "Unknown", vr: VR::SS },
        ],
    },
    P {
        creator: "SIEMENS MR FMRI",
        entries: &[
            E { tag: 0x00210010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210011, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210012, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210013, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210014, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210015, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210016, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210017, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210018, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00210019, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210031, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210040, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210042, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210043, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210044, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210045, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210046, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210047, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00210048, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00210049, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0021004A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0021004B, alias: "Unknown", vr: VR::FL },
        ],
    },
    P {
        creator: "SIEMENS MR HEADER",
        entries: &[
            E { tag: 0x00191008, alias: "CSA Image Header Type", vr: VR::CS },
            E { tag: 0x00191009, alias: "CSA Image Header Version ??", vr: VR::LO },
            E { tag: 0x0019100A, alias: "NumberOfImagesInMosaic", vr: VR::US },
            E { tag: 0x0019100B, alias: "SliceMeasurementDuration", vr: VR::DS },
            E { tag: 0x0019100C, alias: "B_value", vr: VR::IS },
            E { tag: 0x0019100D, alias: "DiffusionDirectionality", vr: VR::CS },
            E { tag: 0x0019100E, alias: "DiffusionGradientDirection", vr: VR::FD },
            E { tag: 0x0019100F, alias: "GradientMode", vr: VR::SH },
            E { tag: 0x00191011, alias: "FlowCompensation", vr: VR::SH },
            E { tag: 0x00191012, alias: "TablePositionOrigin", vr: VR::SL },
            E { tag: 0x00191013, alias: "ImaAbsTablePosition", vr: VR::SL },
            E { tag: 0x00191014, alias: "ImaRelTablePosition", vr: VR::IS },
            E { tag: 0x00191015, alias: "SlicePosition_PCS", vr: VR::FD },
            E { tag: 0x00191016, alias: "TimeAfterStart", vr: VR::DS },
            E { tag: 0x00191017, alias: "SliceResolution", vr: VR::DS },
            E { tag: 0x00191018, alias: "RealDwellTime", vr: VR::IS },
            E { tag: 0x00191023, alias: "FMRIStimulInfo", vr: VR::IS },
            E { tag: 0x00191025, alias: "RBMoCoTrans", vr: VR::FD },
            E { tag: 0x00191026, alias: "RBMoCoRot", vr: VR::FD },
            E { tag: 0x00191027, alias: "B_matrix", vr: VR::FD },
            E { tag: 0x00191028, alias: "BandwidthPerPixelPhaseEncode", vr: VR::FD },
            E { tag: 0x00191029, alias: "MosaicRefAcqTimes", vr: VR::FD },
            E { tag: 0x00511008, alias: "CSA Image Header Type", vr: VR::CS },
            E { tag: 0x00511009, alias: "CSA Image Header Version ??", vr: VR::LO },
            E { tag: 0x0051100A, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0051100B, alias: "AcquisitionMatrixText", vr: VR::SH },
            E { tag: 0x0051100C, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0051100D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0051100E, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0051100F, alias: "CoilString", vr: VR::LO },
            E { tag: 0x00511011, alias: "PATModeText", vr: VR::LO },
            E { tag: 0x00511012, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00511013, alias: "PositivePCSDirections", vr: VR::SH },
            E { tag: 0x00511015, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00511016, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00511017, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00511018, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00511019, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS MR SDI 02",
        entries: &[
            E { tag: 0x00210023, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00210046, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00210067, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00210068, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00210089, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "SIEMENS MR SDS 01",
        entries: &[
            E { tag: 0x0021003F, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "SIEMENS MR VA0  COAD",
        entries: &[
            E { tag: 0x00191012, alias: "Magnetic Field Strength", vr: VR::DS },
            E { tag: 0x00191014, alias: "ADC Voltage", vr: VR::DS },
            E { tag: 0x00191016, alias: "ADC Offset", vr: VR::DS },
            E { tag: 0x00191020, alias: "Transmitter Amplitude", vr: VR::DS },
            E { tag: 0x00191021, alias: "Number of Transmitter Amplitudes", vr: VR::IS },
            E { tag: 0x00191022, alias: "Transmitter Attenuator", vr: VR::DS },
            E { tag: 0x00191024, alias: "Transmitter Calibration", vr: VR::DS },
            E { tag: 0x00191026, alias: "Transmitter Reference", vr: VR::DS },
            E { tag: 0x00191050, alias: "Receiver Total Gain", vr: VR::DS },
            E { tag: 0x00191051, alias: "Receiver Amplifier Gain", vr: VR::DS },
            E { tag: 0x00191052, alias: "Receiver Preamplifier Gain", vr: VR::DS },
            E { tag: 0x00191054, alias: "Receiver Cable Attenuation", vr: VR::DS },
            E { tag: 0x00191055, alias: "Receiver Reference Gain", vr: VR::DS },
            E { tag: 0x00191056, alias: "Receiver Filter Frequency", vr: VR::IS },
            E { tag: 0x00191060, alias: "Reconstruction Scale Factor", vr: VR::DS },
            E { tag: 0x00191062, alias: "Reference Scale Factor", vr: VR::DS },
            E { tag: 0x00191070, alias: "Phase Gradient Amplitude", vr: VR::DS },
            E { tag: 0x00191071, alias: "Readout Gradient Amplitude", vr: VR::DS },
            E { tag: 0x00191072, alias: "Selection Gradient Amplitude", vr: VR::DS },
            E { tag: 0x00191080, alias: "Gradient Delay Time", vr: VR::DS },
            E { tag: 0x00191082, alias: "Total Gradient Delay Time", vr: VR::DS },
            E { tag: 0x00191090, alias: "Sensitivity Correction Label", vr: VR::LO },
            E { tag: 0x00191091, alias: "Saturation Phase Encoding Vector Coronal Component", vr: VR::DS },
            E { tag: 0x00191092, alias: "Saturation Readout Vector Coronal Component", vr: VR::DS },
            E { tag: 0x001910A0, alias: "RF Watchdog Mask", vr: VR::IS },
            E { tag: 0x001910A1, alias: "EPI Reconstruction Slope", vr: VR::DS },
            E { tag: 0x001910A2, alias: "RF Power Error Indicator", vr: VR::DS },
            E { tag: 0x001910A5, alias: "Specific Absorption Rate SAR", vr: VR::DS },
            E { tag: 0x001910A6, alias: "Specific Energy Dose SED", vr: VR::DS },
            E { tag: 0x001910B0, alias: "Adjustment Status Mask", vr: VR::UL },
            E { tag: 0x001910C1, alias: "EPI Capacity", vr: VR::DS },
            E { tag: 0x001910C2, alias: "EPI Inductance", vr: VR::DS },
            E { tag: 0x001910C3, alias: "EPI Switch Configuration Code", vr: VR::IS },
            E { tag: 0x001910C4, alias: "EPI Switch Hardware Code", vr: VR::IS },
            E { tag: 0x001910C5, alias: "EPI Switch Delay Time", vr: VR::DS },
            E { tag: 0x001910D1, alias: "Flow Sensitivity", vr: VR::DS },
            E { tag: 0x001910D2, alias: "Calculation Submode", vr: VR::CS },
            E { tag: 0x001910D3, alias: "Field of View Ratio", vr: VR::DS },
            E { tag: 0x001910D4, alias: "Base Raw Matrix Size", vr: VR::IS },
            E { tag: 0x001910D5, alias: "2D Phase Oversampling Lines", vr: VR::IS },
            E { tag: 0x001910D6, alias: "3D Phase Oversampling Part", vr: VR::IS },
            E { tag: 0x001910D7, alias: "Echo Line Position", vr: VR::IS },
            E { tag: 0x001910D8, alias: "Echo Column Position", vr: VR::IS },
            E { tag: 0x001910D9, alias: "Lines per Segment", vr: VR::IS },
            E { tag: 0x001910DA, alias: "Phase Coding Direction", vr: VR::CS },
        ],
    },
    P {
        creator: "SIEMENS MR VA0  GEN",
        entries: &[
            E { tag: 0x00191010, alias: "Total Measurement Time nominal", vr: VR::DS },
            E { tag: 0x00191011, alias: "Total Measurement Time current", vr: VR::DS },
            E { tag: 0x00191012, alias: "Start Delay Time", vr: VR::DS },
            E { tag: 0x00191013, alias: "Dwell Time", vr: VR::DS },
            E { tag: 0x00191014, alias: "Number of Phases", vr: VR::IS },
            E { tag: 0x00191016, alias: "Sequence Control Mask", vr: VR::UL },
            E { tag: 0x00191018, alias: "Measurement Status Mask", vr: VR::UL },
            E { tag: 0x00191020, alias: "Number of Fourier Lines nominal", vr: VR::IS },
            E { tag: 0x00191021, alias: "Number of Fourier Lines current", vr: VR::IS },
            E { tag: 0x00191026, alias: "Number of Fourier Lines after Zero", vr: VR::IS },
            E { tag: 0x00191028, alias: "First Measured Fourier Line", vr: VR::IS },
            E { tag: 0x00191030, alias: "Acquisition Columns", vr: VR::IS },
            E { tag: 0x00191031, alias: "Reconstruction Columns", vr: VR::IS },
            E { tag: 0x00191040, alias: "AC Element Number", vr: VR::IS },
            E { tag: 0x00191041, alias: "AC Element Select Mask", vr: VR::UL },
            E { tag: 0x00191042, alias: "AC Element Data Mask", vr: VR::UL },
            E { tag: 0x00191043, alias: "AC Element To ADC Connect", vr: VR::IS },
            E { tag: 0x00191044, alias: "AC Element Noise Level", vr: VR::IS },
            E { tag: 0x00191045, alias: "AC ADC Pair Number", vr: VR::IS },
            E { tag: 0x00191046, alias: "AC Combination Mask", vr: VR::UL },
            E { tag: 0x00191050, alias: "Number of Averages current", vr: VR::IS },
            E { tag: 0x00191060, alias: "Flip Angle", vr: VR::DS },
            E { tag: 0x00191070, alias: "Number of Prescans", vr: VR::IS },
            E { tag: 0x00191081, alias: "Filter Type for Raw Data", vr: VR::CS },
            E { tag: 0x00191082, alias: "Filter Parameter for Raw Data", vr: VR::DS },
            E { tag: 0x00191083, alias: "Filter Type for Image Data", vr: VR::CS },
            E { tag: 0x00191084, alias: "Filter Parameter for Image Data", vr: VR::DS },
            E { tag: 0x00191085, alias: "Filter Type for Phase Correction", vr: VR::CS },
            E { tag: 0x00191086, alias: "Filter Parameter for Phase Correction", vr: VR::DS },
            E { tag: 0x00191087, alias: "Normalization Filter Type for Image Data", vr: VR::CS },
            E { tag: 0x00191088, alias: "Normalization Filter Parameter for Image Data", vr: VR::DS },
            E { tag: 0x00191090, alias: "Number of Saturation Regions", vr: VR::IS },
            E { tag: 0x00191091, alias: "Saturation Phase Encoding Vector Sagittal Compo...", vr: VR::DS },
            E { tag: 0x00191092, alias: "Saturation Readout Vector Sagittal Component", vr: VR::DS },
            E { tag: 0x00191093, alias: "EPI Stimulation Monitor Mode", vr: VR::IS },
            E { tag: 0x00191094, alias: "Image Rotation Angle", vr: VR::DS },
            E { tag: 0x00191096, alias: "Coil ID Mask", vr: VR::UL },
            E { tag: 0x00191097, alias: "Coil Class Mask", vr: VR::UL },
            E { tag: 0x00191098, alias: "Coil Position", vr: VR::DS },
            E { tag: 0x001910A0, alias: "EPI Reconstruction Phase", vr: VR::DS },
            E { tag: 0x001910A1, alias: "EPI Reconstruction Slope", vr: VR::DS },
            E { tag: 0x00211020, alias: "Phase Corrections Rows sequence", vr: VR::IS },
            E { tag: 0x00211021, alias: "Phase Corrections Columns sequence", vr: VR::IS },
            E { tag: 0x00211022, alias: "Phase Corrections Rows reconstruction", vr: VR::IS },
            E { tag: 0x00211024, alias: "Phase Corrections Columns reconstruction", vr: VR::IS },
            E { tag: 0x00211030, alias: "Number of 3D Raw Partitions nominal", vr: VR::IS },
            E { tag: 0x00211031, alias: "Number of 3D Raw Partitions current", vr: VR::IS },
            E { tag: 0x00211034, alias: "Number of 3D Image Partitions", vr: VR::IS },
            E { tag: 0x00211036, alias: "Actual 3D Image Partition Number", vr: VR::IS },
            E { tag: 0x00211039, alias: "Slab Thickness", vr: VR::DS },
            E { tag: 0x00211040, alias: "Number of Slices nominal", vr: VR::IS },
            E { tag: 0x00211041, alias: "Number of Slices current", vr: VR::IS },
            E { tag: 0x00211042, alias: "Current Slice Number", vr: VR::IS },
            E { tag: 0x00211043, alias: "Current Group Number", vr: VR::IS },
            E { tag: 0x00211044, alias: "Current Slice Distance Factor", vr: VR::DS },
            E { tag: 0x00211045, alias: "MIP Start Row", vr: VR::IS },
            E { tag: 0x00211046, alias: "MIP Stop Row", vr: VR::IS },
            E { tag: 0x00211047, alias: "MIP Start Column", vr: VR::IS },
            E { tag: 0x00211048, alias: "MIP Stop Column", vr: VR::IS },
            E { tag: 0x00211049, alias: "MIP Start Slice", vr: VR::IS },
            E { tag: 0x0021104A, alias: "MIP Stop Slice", vr: VR::IS },
            E { tag: 0x0021104F, alias: "Order of Slices", vr: VR::CS },
            E { tag: 0x00211050, alias: "Signal Mask", vr: VR::UL },
            E { tag: 0x00211052, alias: "Delay after Trigger", vr: VR::IS },
            E { tag: 0x00211053, alias: "RRInterval", vr: VR::IS },
            E { tag: 0x00211054, alias: "Number of Trigger Pulses", vr: VR::DS },
            E { tag: 0x00211056, alias: "Repetition Time effective", vr: VR::DS },
            E { tag: 0x00211057, alias: "Gate Phase", vr: VR::CS },
            E { tag: 0x00211058, alias: "Gate Threshold", vr: VR::DS },
            E { tag: 0x00211059, alias: "Gated Ratio", vr: VR::DS },
            E { tag: 0x00211060, alias: "Number of Interpolated Images", vr: VR::IS },
            E { tag: 0x00211070, alias: "Number of Echoes", vr: VR::IS },
            E { tag: 0x00211072, alias: "Second Echo Time", vr: VR::DS },
            E { tag: 0x00211073, alias: "Second Repetition Time", vr: VR::DS },
            E { tag: 0x00211080, alias: "Cardiac Code", vr: VR::IS },
            E { tag: 0x00211091, alias: "Saturation Phase Encoding Vector Transversal Co...", vr: VR::DS },
            E { tag: 0x00211092, alias: "Saturation Readout Vector Transversal Component", vr: VR::DS },
            E { tag: 0x00211093, alias: "EPI Change Value of Magnitude", vr: VR::DS },
            E { tag: 0x00211094, alias: "EPI Change Value of X Component", vr: VR::DS },
            E { tag: 0x00211095, alias: "EPI Change Value of Y Component", vr: VR::DS },
            E { tag: 0x00211096, alias: "EPI Change Value of Z Component", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS MR VA0  RAW",
        entries: &[
            E { tag: 0x00211000, alias: "Sequence Type", vr: VR::CS },
            E { tag: 0x00211001, alias: "Vector Size original", vr: VR::IS },
            E { tag: 0x00211002, alias: "Vector Size extended", vr: VR::IS },
            E { tag: 0x00211003, alias: "Acquired Spectral Range", vr: VR::DS },
            E { tag: 0x00211004, alias: "VOI Position", vr: VR::DS },
            E { tag: 0x00211005, alias: "VOI Size", vr: VR::DS },
            E { tag: 0x00211006, alias: "CSI Matrix Size original", vr: VR::IS },
            E { tag: 0x00211007, alias: "CSI Matrix Size extended", vr: VR::IS },
            E { tag: 0x00211008, alias: "Spatial Grid Shift", vr: VR::DS },
            E { tag: 0x00211009, alias: "Signal Limits Minimum", vr: VR::DS },
            E { tag: 0x00211010, alias: "Signal Limits Maximum", vr: VR::DS },
            E { tag: 0x00211011, alias: "Spec Info Mask", vr: VR::DS },
            E { tag: 0x00211012, alias: "EPI Time Rate of Change of Magnitude", vr: VR::DS },
            E { tag: 0x00211013, alias: "EPI Time Rate of Change of X Component", vr: VR::DS },
            E { tag: 0x00211014, alias: "EPI Time Rate of Change of Y Component", vr: VR::DS },
            E { tag: 0x00211015, alias: "EPI Time Rate of Change of Z Component", vr: VR::DS },
            E { tag: 0x00211016, alias: "EPI Time Rate of Change Legal Limit 1", vr: VR::DS },
            E { tag: 0x00211017, alias: "EPI Operation Mode Flag", vr: VR::IS },
            E { tag: 0x00211018, alias: "EPI Field Calculation Safety Factor", vr: VR::DS },
            E { tag: 0x00211019, alias: "EPI Legal Limit 1 of Change Value", vr: VR::DS },
            E { tag: 0x00211020, alias: "EPI Legal Limit 2 of Change Value", vr: VR::DS },
            E { tag: 0x00211021, alias: "EPI Rise Time", vr: VR::DS },
            E { tag: 0x00211030, alias: "AC ADC Offset", vr: VR::DS },
            E { tag: 0x00211031, alias: "AC Preamplifier Gain", vr: VR::DS },
            E { tag: 0x00211050, alias: "Saturation Type", vr: VR::CS },
            E { tag: 0x00211051, alias: "Saturation Normal Vector", vr: VR::DS },
            E { tag: 0x00211052, alias: "Saturation Position Vector", vr: VR::DS },
            E { tag: 0x00211053, alias: "Saturation Thickness", vr: VR::DS },
            E { tag: 0x00211054, alias: "Saturation Width", vr: VR::DS },
            E { tag: 0x00211055, alias: "Saturation Distance", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS NUMARIS II",
        entries: &[
            E { tag: 0x7FE31000, alias: "Image Graphics Format Code", vr: VR::LO },
            E { tag: 0x7FE31010, alias: "Image Graphics", vr: VR::OB },
            E { tag: 0x7FE31020, alias: "Image Graphics Dummy", vr: VR::OB },
        ],
    },
    P {
        creator: "SIEMENS RA GEN",
        entries: &[
            E { tag: 0x00111020, alias: "Total Fluoro TimePatient", vr: VR::UL },
            E { tag: 0x00111025, alias: "Dose Area ProductPatient", vr: VR::UL },
            E { tag: 0x00111030, alias: "Patient DOB", vr: VR::LO },
            E { tag: 0x00111035, alias: "Patient Initial Puck Counter", vr: VR::LO },
            E { tag: 0x00111040, alias: "Patient Initial Position", vr: VR::CS },
            E { tag: 0x00191015, alias: "Acquired Plane", vr: VR::CS },
            E { tag: 0x00191020, alias: "Scene Flag", vr: VR::UL },
            E { tag: 0x00191022, alias: "ReferencePhotofile Flag", vr: VR::UL },
            E { tag: 0x00191024, alias: "Scene Name", vr: VR::LO },
            E { tag: 0x00191026, alias: "Acquisition Index", vr: VR::US },
            E { tag: 0x00191028, alias: "Mixed Pulse Mode", vr: VR::US },
            E { tag: 0x0019102A, alias: "Number of Positions", vr: VR::US },
            E { tag: 0x0019102C, alias: "Number of Phases", vr: VR::US },
            E { tag: 0x0019102E, alias: "Framerate for 8 Positions", vr: VR::US },
            E { tag: 0x00191030, alias: "FramesPhase", vr: VR::US },
            E { tag: 0x00191032, alias: "Stepping Direction", vr: VR::US },
            E { tag: 0x00191034, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191036, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191038, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019103A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019103C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019103E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191044, alias: "Image Transfer Delay", vr: VR::US },
            E { tag: 0x00191046, alias: "InversNot Invers", vr: VR::UL },
            E { tag: 0x00191048, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019104A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019104C, alias: "Blanking Circle Diameter", vr: VR::US },
            E { tag: 0x00191050, alias: "Stand Data Valid", vr: VR::US },
            E { tag: 0x00191052, alias: "Table Tilt", vr: VR::US },
            E { tag: 0x00191054, alias: "Table Axis Rotation", vr: VR::US },
            E { tag: 0x00191056, alias: "Table Longitudinal Position", vr: VR::US },
            E { tag: 0x00191058, alias: "Table Side Offset", vr: VR::US },
            E { tag: 0x0019105A, alias: "Table Isocenter Height", vr: VR::US },
            E { tag: 0x0019105C, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0019105E, alias: "Collimation Data Valid", vr: VR::UL },
            E { tag: 0x00191060, alias: "PERI Sequence Number", vr: VR::UL },
            E { tag: 0x00191062, alias: "PERI Total Scenes", vr: VR::UL },
            E { tag: 0x00191064, alias: "PERI Overlap Top", vr: VR::US },
            E { tag: 0x00191066, alias: "PERI Overlap Bottom", vr: VR::UL },
            E { tag: 0x00191068, alias: "Raw Image Number", vr: VR::UL },
            E { tag: 0x0019106A, alias: "XRay Data Valid", vr: VR::UL },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191072, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191074, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191076, alias: "Filling Average Factor", vr: VR::UL },
            E { tag: 0x00191078, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019107A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019107C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019107E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191080, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191082, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191084, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191086, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191088, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019108A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019108C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019108E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191092, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191094, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191096, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191098, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019109A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019109C, alias: "II Level CalibrationFactor", vr: VR::UL },
            E { tag: 0x0019109E, alias: "Nativ Review Flag", vr: VR::UL },
            E { tag: 0x00211015, alias: "Images in Study", vr: VR::US },
            E { tag: 0x00211020, alias: "Scenes in Study", vr: VR::US },
            E { tag: 0x00211025, alias: "Images in Photofile", vr: VR::US },
            E { tag: 0x00211030, alias: "Images in All Scenes", vr: VR::US },
            E { tag: 0x00211040, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS RA PLANE A",
        entries: &[
            E { tag: 0x00191020, alias: "System Calibration Factor Plane A", vr: VR::UL },
            E { tag: 0x00191022, alias: "XRay Parameter Set Number", vr: VR::US },
            E { tag: 0x00191024, alias: "XRay System", vr: VR::US },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191028, alias: "Acquired Display Mode", vr: VR::US },
            E { tag: 0x0019102A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019102C, alias: "Acquisition Framerate", vr: VR::US },
            E { tag: 0x0019102E, alias: "Maximum Frame Limit", vr: VR::US },
            E { tag: 0x00191030, alias: "Native Filter Type", vr: VR::US },
            E { tag: 0x00191032, alias: "Subtracted Filter Type", vr: VR::US },
            E { tag: 0x00191034, alias: "Filter Factor Native", vr: VR::US },
            E { tag: 0x00191036, alias: "Anatomic Blackground Factor", vr: VR::US },
            E { tag: 0x00191038, alias: "Window Upper Limit Native", vr: VR::US },
            E { tag: 0x0019103A, alias: "Window Lower Limit Native", vr: VR::US },
            E { tag: 0x0019103C, alias: "Window Brightness Phase 1 Sub", vr: VR::US },
            E { tag: 0x0019103E, alias: "Window Brightness Phase 2 Sub", vr: VR::US },
            E { tag: 0x00191040, alias: "Window Contrast Phase 1 Sub", vr: VR::US },
            E { tag: 0x00191042, alias: "Window Contrast Phase 2 Sub", vr: VR::US },
            E { tag: 0x00191044, alias: "Filter Factor Subtraction", vr: VR::US },
            E { tag: 0x00191046, alias: "Peak Opacified", vr: VR::US },
            E { tag: 0x00191048, alias: "Mask Frame", vr: VR::UL },
            E { tag: 0x0019104A, alias: "BIH Frame", vr: VR::UL },
            E { tag: 0x0019104C, alias: "Center Beam Angulation CaudCran", vr: VR::US },
            E { tag: 0x0019104E, alias: "Center Beam Angulation LR Anterior", vr: VR::US },
            E { tag: 0x00191050, alias: "Longitudinal Position", vr: VR::US },
            E { tag: 0x00191052, alias: "Side Offset", vr: VR::US },
            E { tag: 0x00191054, alias: "Isocenter Height", vr: VR::US },
            E { tag: 0x00191056, alias: "Image Twist", vr: VR::US },
            E { tag: 0x00191058, alias: "SourceImage Distance", vr: VR::US },
            E { tag: 0x0019105A, alias: "Mechanical Magnet Factor", vr: VR::US },
            E { tag: 0x0019105C, alias: "Calibration Flag", vr: VR::UL },
            E { tag: 0x0019105E, alias: "Calibration Angle CranCaud", vr: VR::UL },
            E { tag: 0x00191060, alias: "Calibration Angle RAOLAO", vr: VR::UL },
            E { tag: 0x00191062, alias: "Calibration Table to Floor Distance", vr: VR::UL },
            E { tag: 0x00191064, alias: "Calibration Isocenter to Floor Distance", vr: VR::UL },
            E { tag: 0x00191066, alias: "Calibration Isocenter to Source Distance", vr: VR::UL },
            E { tag: 0x00191068, alias: "Calibration Source to II Distance", vr: VR::UL },
            E { tag: 0x0019106A, alias: "Calibration II Zoom", vr: VR::UL },
            E { tag: 0x0019106C, alias: "Calibration II Field", vr: VR::UL },
            E { tag: 0x0019106E, alias: "Calibration Factor", vr: VR::UL },
            E { tag: 0x00191070, alias: "Calibration Object to Image Distance", vr: VR::UL },
            E { tag: 0x00191072, alias: "Calibration System Factor", vr: VR::UL },
            E { tag: 0x00191074, alias: "Calibration System Correction", vr: VR::UL },
            E { tag: 0x00191076, alias: "Calibration System II Formats", vr: VR::UL },
            E { tag: 0x00191078, alias: "Calibration Gantry Data Valid", vr: VR::UL },
            E { tag: 0x0019107A, alias: "Collimator Square Breadth", vr: VR::US },
            E { tag: 0x0019107C, alias: "Collimator Square Height", vr: VR::US },
            E { tag: 0x0019107E, alias: "Collimator Square Diameter", vr: VR::US },
            E { tag: 0x00191080, alias: "Collimator Finger Turn Angle", vr: VR::US },
            E { tag: 0x00191082, alias: "Collimator Finger Position", vr: VR::US },
            E { tag: 0x00191084, alias: "Collimator Diaphragm Turn Angle", vr: VR::US },
            E { tag: 0x00191086, alias: "Collimator Diagphragm Position 1", vr: VR::US },
            E { tag: 0x00191088, alias: "Collimator Diaphragm Position 2", vr: VR::US },
            E { tag: 0x0019108A, alias: "Collimator Diaphragm Mode", vr: VR::US },
            E { tag: 0x0019108C, alias: "Collimator Beam Limit Breadth", vr: VR::US },
            E { tag: 0x0019108E, alias: "Collimator Beam Limit Height", vr: VR::US },
            E { tag: 0x00191090, alias: "Collimator Beam Limit Diameter", vr: VR::US },
            E { tag: 0x00191092, alias: "XRay Control Mode", vr: VR::US },
            E { tag: 0x00191094, alias: "XRay System", vr: VR::US },
            E { tag: 0x00191096, alias: "Focal Spot", vr: VR::US },
            E { tag: 0x00191098, alias: "Exposure Control", vr: VR::US },
            E { tag: 0x0019109A, alias: "XRay Voltage", vr: VR::US },
            E { tag: 0x0019109C, alias: "XRay Current", vr: VR::US },
            E { tag: 0x0019109E, alias: "XRay Current Time Product", vr: VR::UL },
            E { tag: 0x001910A0, alias: "Pulse Time", vr: VR::US },
            E { tag: 0x001910A2, alias: "Scene Time Fluoro Clock", vr: VR::US },
            E { tag: 0x001910A4, alias: "Maximum PulseRate", vr: VR::US },
            E { tag: 0x001910A6, alias: "Pulses per Scene", vr: VR::US },
            E { tag: 0x001910A8, alias: "Dose Area Product of Scene", vr: VR::UL },
            E { tag: 0x001910AA, alias: "Dose", vr: VR::US },
            E { tag: 0x001910AC, alias: "Dose Rate", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS RA PLANE B",
        entries: &[
            E { tag: 0x00191020, alias: "System Calibration Factor Plane B", vr: VR::UL },
            E { tag: 0x00191022, alias: "XRay Parameter Set Number", vr: VR::US },
            E { tag: 0x00191024, alias: "XRay System", vr: VR::US },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191028, alias: "Acquired Display Mode", vr: VR::US },
            E { tag: 0x0019102A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019102C, alias: "Acquisition Framerate", vr: VR::US },
            E { tag: 0x0019102E, alias: "Maximum Frame Limit", vr: VR::US },
            E { tag: 0x00191030, alias: "Native Filter Type", vr: VR::US },
            E { tag: 0x00191032, alias: "Subtracted Filter Type", vr: VR::US },
            E { tag: 0x00191034, alias: "Filter Factor Native", vr: VR::US },
            E { tag: 0x00191036, alias: "Anatomic Blackground Factor", vr: VR::US },
            E { tag: 0x00191038, alias: "Window Upper Limit Native", vr: VR::US },
            E { tag: 0x0019103A, alias: "Window Lower Limit Native", vr: VR::US },
            E { tag: 0x0019103C, alias: "Window Brightness Phase 1 Sub", vr: VR::US },
            E { tag: 0x0019103E, alias: "Window Brightness Phase 2 Sub", vr: VR::US },
            E { tag: 0x00191040, alias: "Window Contrast Phase 1 Sub", vr: VR::US },
            E { tag: 0x00191042, alias: "Window Contrast Phase 2 Sub", vr: VR::US },
            E { tag: 0x00191044, alias: "Filter Factor Subtraction", vr: VR::US },
            E { tag: 0x00191046, alias: "Peak Opacified", vr: VR::US },
            E { tag: 0x00191048, alias: "Mask Frame", vr: VR::UL },
            E { tag: 0x0019104A, alias: "BIH Frame", vr: VR::UL },
            E { tag: 0x0019104C, alias: "Center Beam Angulation CaudCran", vr: VR::US },
            E { tag: 0x0019104E, alias: "Center Beam Angulation LR Anterior", vr: VR::US },
            E { tag: 0x00191050, alias: "Longitudinal Position", vr: VR::US },
            E { tag: 0x00191052, alias: "Side Offset", vr: VR::US },
            E { tag: 0x00191054, alias: "Isocenter Height", vr: VR::US },
            E { tag: 0x00191056, alias: "Image Twist", vr: VR::US },
            E { tag: 0x00191058, alias: "SourceImage Distance", vr: VR::US },
            E { tag: 0x0019105A, alias: "Mechanical Magnet Factor", vr: VR::US },
            E { tag: 0x0019105C, alias: "Calibration Flag", vr: VR::UL },
            E { tag: 0x0019105E, alias: "Calibration Angle CranCaud", vr: VR::UL },
            E { tag: 0x00191060, alias: "Calibration Angle RAOLAO", vr: VR::UL },
            E { tag: 0x00191062, alias: "Calibration Table to Floor Distance", vr: VR::UL },
            E { tag: 0x00191064, alias: "Calibration Isocenter to Floor Distance", vr: VR::UL },
            E { tag: 0x00191066, alias: "Calibration Isocenter to Source Distance", vr: VR::UL },
            E { tag: 0x00191068, alias: "Calibration Source to II Distance", vr: VR::UL },
            E { tag: 0x0019106A, alias: "Calibration II Zoom", vr: VR::UL },
            E { tag: 0x0019106C, alias: "Calibration II Field", vr: VR::UL },
            E { tag: 0x0019106E, alias: "Calibration Factor", vr: VR::UL },
            E { tag: 0x00191070, alias: "Calibration Object to Image Distance", vr: VR::UL },
            E { tag: 0x00191072, alias: "Calibration System Factor", vr: VR::UL },
            E { tag: 0x00191074, alias: "Calibration System Correction", vr: VR::UL },
            E { tag: 0x00191076, alias: "Calibration System II Formats", vr: VR::UL },
            E { tag: 0x00191078, alias: "Calibration Gantry Data Valid", vr: VR::UL },
            E { tag: 0x0019107A, alias: "Collimator Square Breadth", vr: VR::US },
            E { tag: 0x0019107C, alias: "Collimator Square Height", vr: VR::US },
            E { tag: 0x0019107E, alias: "Collimator Square Diameter", vr: VR::US },
            E { tag: 0x00191080, alias: "Collimator Finger Turn Angle", vr: VR::US },
            E { tag: 0x00191082, alias: "Collimator Finger Position", vr: VR::US },
            E { tag: 0x00191084, alias: "Collimator Diaphragm Turn Angle", vr: VR::US },
            E { tag: 0x00191086, alias: "Collimator Diagphragm Position 1", vr: VR::US },
            E { tag: 0x00191088, alias: "Collimator Diaphragm Position 2", vr: VR::US },
            E { tag: 0x0019108A, alias: "Collimator Diaphragm Mode", vr: VR::US },
            E { tag: 0x0019108C, alias: "Collimator Beam Limit Breadth", vr: VR::US },
            E { tag: 0x0019108E, alias: "Collimator Beam Limit Height", vr: VR::US },
            E { tag: 0x00191090, alias: "Collimator Beam Limit Diameter", vr: VR::US },
            E { tag: 0x00191092, alias: "XRay Control Mode", vr: VR::US },
            E { tag: 0x00191094, alias: "XRay System", vr: VR::US },
            E { tag: 0x00191096, alias: "Focal Spot", vr: VR::US },
            E { tag: 0x00191098, alias: "Exposure Control", vr: VR::US },
            E { tag: 0x0019109A, alias: "XRay Voltage", vr: VR::US },
            E { tag: 0x0019109C, alias: "XRay Current", vr: VR::US },
            E { tag: 0x0019109E, alias: "XRay Current Time Product", vr: VR::UL },
            E { tag: 0x001910A0, alias: "Pulse Time", vr: VR::US },
            E { tag: 0x001910A2, alias: "Scene Time Fluoro Clock", vr: VR::US },
            E { tag: 0x001910A4, alias: "Maximum PulseRate", vr: VR::US },
            E { tag: 0x001910A6, alias: "Pulses per Scene", vr: VR::US },
            E { tag: 0x001910A8, alias: "Dose Area Product of Scene", vr: VR::UL },
            E { tag: 0x001910AA, alias: "Dose", vr: VR::US },
            E { tag: 0x001910AC, alias: "Dose Rate", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS RIS",
        entries: &[
            E { tag: 0x00111010, alias: "Patient UID", vr: VR::LO },
            E { tag: 0x00111011, alias: "Patient ID", vr: VR::LO },
            E { tag: 0x00111020, alias: "Patient Registration Date", vr: VR::DA },
            E { tag: 0x00111021, alias: "Patient Registration Time", vr: VR::TM },
            E { tag: 0x00111030, alias: "Patient NameRIS", vr: VR::PN },
            E { tag: 0x00111031, alias: "Patient Prename", vr: VR::PN },
            E { tag: 0x00111040, alias: "Patient Hospital Status", vr: VR::LO },
            E { tag: 0x00111041, alias: "Medical Alerts", vr: VR::LO },
            E { tag: 0x00111042, alias: "Contrast Allergy", vr: VR::LO },
            E { tag: 0x00311010, alias: "Request UID", vr: VR::LO },
            E { tag: 0x00311045, alias: "Requesting Physician", vr: VR::LO },
            E { tag: 0x00311050, alias: "Requested Physician", vr: VR::LO },
            E { tag: 0x00331010, alias: "Patient Study UID", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS SIENET",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS SMS-AX  ACQ 1.0",
        entries: &[
            E { tag: 0x00211000, alias: "Acquisition Type", vr: VR::US },
            E { tag: 0x00211001, alias: "Acquisition Mode", vr: VR::US },
            E { tag: 0x00211002, alias: "Footswitch Index", vr: VR::US },
            E { tag: 0x00211003, alias: "Acquisition Room", vr: VR::US },
            E { tag: 0x00211004, alias: "Current Time Product", vr: VR::SL },
            E { tag: 0x00211005, alias: "Dose", vr: VR::SL },
            E { tag: 0x00211006, alias: "Skin Dose Percent", vr: VR::SL },
            E { tag: 0x00211007, alias: "Skin Dose Accumulation", vr: VR::SL },
            E { tag: 0x00211008, alias: "Skin Dose Rate", vr: VR::SL },
            E { tag: 0x00211009, alias: "Impac Filename", vr: VR::UL },
            E { tag: 0x0021100A, alias: "Copper Filter", vr: VR::UL },
            E { tag: 0x0021100B, alias: "Measuring Field", vr: VR::US },
            E { tag: 0x0021100C, alias: "Post Blanking Circle", vr: VR::SS },
            E { tag: 0x0021100D, alias: "Dyna Angles", vr: VR::SS },
            E { tag: 0x0021100E, alias: "Total Steps", vr: VR::SS },
            E { tag: 0x0021100F, alias: "Dyna X-Ray Info", vr: VR::SL },
            E { tag: 0x00211010, alias: "Modality LUT Input Gamma", vr: VR::US },
            E { tag: 0x00211011, alias: "Modality LUT Output Gamma", vr: VR::US },
            E { tag: 0x00211012, alias: "SH_STPAR", vr: VR::OB },
            E { tag: 0x00211013, alias: "Acquisition Zoom", vr: VR::US },
            E { tag: 0x00211014, alias: "Dyna Angulation Step Width", vr: VR::SS },
            E { tag: 0x00211015, alias: "Harmonization", vr: VR::US },
            E { tag: 0x00211016, alias: "DR Single Flag", vr: VR::US },
            E { tag: 0x00211017, alias: "Source to Isocenter", vr: VR::SL },
            E { tag: 0x00211018, alias: "Pressure Data", vr: VR::US },
            E { tag: 0x00211019, alias: "ECG Index Array", vr: VR::SL },
            E { tag: 0x0021101A, alias: "FD Flag", vr: VR::US },
            E { tag: 0x0021101B, alias: "SH_ZOOM", vr: VR::OB },
            E { tag: 0x0021101C, alias: "SH_COLPAR", vr: VR::OB },
            E { tag: 0x0021101D, alias: "K-Factor", vr: VR::US },
            E { tag: 0x0021101E, alias: "EVE", vr: VR::US },
            E { tag: 0x0021101F, alias: "Total Scene Time", vr: VR::SL },
            E { tag: 0x00211020, alias: "Restore Flag", vr: VR::US },
            E { tag: 0x00211021, alias: "Stand Movement Flag", vr: VR::US },
            E { tag: 0x00211022, alias: "FD Rows", vr: VR::US },
            E { tag: 0x00211023, alias: "FD Columns", vr: VR::US },
            E { tag: 0x00211024, alias: "Table Movement Flag", vr: VR::US },
            E { tag: 0x00211025, alias: "Original Organ Program Name", vr: VR::LO },
            E { tag: 0x00211026, alias: "Crispy XPI Filter", vr: VR::DS },
            E { tag: 0x00211028, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00211030, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00211040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211057, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00211058, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211071, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS SMS-AX  ORIGINAL IMAGE INFO 1.0",
        entries: &[
            E { tag: 0x00251000, alias: "View Native", vr: VR::US },
            E { tag: 0x00251001, alias: "Original Series Number", vr: VR::US },
            E { tag: 0x00251002, alias: "Original Image Number", vr: VR::US },
            E { tag: 0x00251003, alias: "Win Center", vr: VR::US },
            E { tag: 0x00251004, alias: "Win Width", vr: VR::US },
            E { tag: 0x00251005, alias: "Win Brightness", vr: VR::US },
            E { tag: 0x00251006, alias: "Win Contrast", vr: VR::US },
            E { tag: 0x00251007, alias: "Original Frame Number", vr: VR::US },
            E { tag: 0x00251008, alias: "Original Mask Frame Number", vr: VR::US },
            E { tag: 0x00251009, alias: "Opac", vr: VR::US },
            E { tag: 0x0025100A, alias: "Original Number of Frames", vr: VR::US },
            E { tag: 0x0025100B, alias: "Original Scene Duration", vr: VR::DS },
            E { tag: 0x0025100C, alias: "Identifier LOID", vr: VR::LO },
            E { tag: 0x0025100D, alias: "Original Scene VFR Info", vr: VR::SS },
            E { tag: 0x0025100E, alias: "Original Frame ECG Position", vr: VR::SS },
            E { tag: 0x0025100F, alias: "Original ECG 1st Frame Offset (retired)", vr: VR::SS },
            E { tag: 0x00251010, alias: "Zoom Flag", vr: VR::SS },
            E { tag: 0x00251011, alias: "Flex", vr: VR::US },
            E { tag: 0x00251012, alias: "Number of Mask Frames", vr: VR::US },
            E { tag: 0x00251013, alias: "Number of Fill Frames", vr: VR::US },
            E { tag: 0x00251014, alias: "Series Number", vr: VR::IS },
            E { tag: 0x00251015, alias: "Image Number", vr: VR::IS },
            E { tag: 0x00251016, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS SMS-AX  QUANT 1.0",
        entries: &[
            E { tag: 0x00231000, alias: "Horizontal Calibration Pixel Size", vr: VR::DS },
            E { tag: 0x00231001, alias: "Vertical Calibration Pixel Size", vr: VR::DS },
            E { tag: 0x00231002, alias: "Calibration Object", vr: VR::LO },
            E { tag: 0x00231003, alias: "Calibration Object Size", vr: VR::DS },
            E { tag: 0x00231004, alias: "Calibration Method", vr: VR::LO },
            E { tag: 0x00231005, alias: "Filename", vr: VR::ST },
            E { tag: 0x00231006, alias: "Frame Number", vr: VR::IS },
            E { tag: 0x00231007, alias: "Calibration Factor Multiplicity", vr: VR::IS },
            E { tag: 0x00231008, alias: "Calibration TOD Value", vr: VR::IS },
        ],
    },
    P {
        creator: "SIEMENS SMS-AX  VIEW 1.0",
        entries: &[
            E { tag: 0x00191000, alias: "Review Mode", vr: VR::US },
            E { tag: 0x00191001, alias: "Anatomical Background Percent", vr: VR::US },
            E { tag: 0x00191002, alias: "Number of Phases", vr: VR::US },
            E { tag: 0x00191003, alias: "Apply Anatomical Background", vr: VR::US },
            E { tag: 0x00191004, alias: "Pixel Shift Array", vr: VR::SS },
            E { tag: 0x00191005, alias: "Brightness", vr: VR::US },
            E { tag: 0x00191006, alias: "Contrast", vr: VR::US },
            E { tag: 0x00191007, alias: "Enabled", vr: VR::US },
            E { tag: 0x00191008, alias: "Native Edge Enh. Percent Gain", vr: VR::US },
            E { tag: 0x00191009, alias: "Native Edge Enh. LUT Index", vr: VR::SS },
            E { tag: 0x0019100A, alias: "Native Edge Enh. Kernel Size", vr: VR::SS },
            E { tag: 0x0019100B, alias: "Subtr. Edge Enh. Percent Gain", vr: VR::US },
            E { tag: 0x0019100C, alias: "Subtr. Edge Enh. LUT Index", vr: VR::SS },
            E { tag: 0x0019100D, alias: "Subtr. Edge Enh. Kernel Size", vr: VR::SS },
            E { tag: 0x0019100E, alias: "Fade Percent", vr: VR::US },
            E { tag: 0x0019100F, alias: "Flipped Before Laterality Applied", vr: VR::US },
            E { tag: 0x00191010, alias: "Apply Fade", vr: VR::US },
            E { tag: 0x00191011, alias: "RefImages Taken", vr: VR::US },
            E { tag: 0x00191012, alias: "Zoom", vr: VR::US },
            E { tag: 0x00191013, alias: "Pan X", vr: VR::SS },
            E { tag: 0x00191014, alias: "Pan Y", vr: VR::SS },
            E { tag: 0x00191015, alias: "Native Edge Enh. Adv Perc. Gain", vr: VR::SS },
            E { tag: 0x00191016, alias: "Subtr. Edge Enh. Adv Perc. Gain", vr: VR::SS },
            E { tag: 0x00191017, alias: "Invert Flag", vr: VR::US },
            E { tag: 0x0019101A, alias: "Quant 1K Overlay", vr: VR::OB },
            E { tag: 0x0019101B, alias: "Original Resolution", vr: VR::US },
            E { tag: 0x0019101C, alias: "Auto Window Center", vr: VR::DS },
            E { tag: 0x0019101D, alias: "Auto Window Width", vr: VR::DS },
        ],
    },
    P {
        creator: "SIEMENS Selma",
        entries: &[
            E { tag: 0x00191006, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191007, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191026, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191029, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191030, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191033, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191034, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191035, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "SIEMENS WH SR 1.0",
        entries: &[
            E { tag: 0x00711001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00711002, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "SIEMENS_FLCOMPACT_VA01A_PROC",
        entries: &[
            E { tag: 0x0017100A, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017100B, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017100C, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017100D, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x0017100E, alias: "Internal Data", vr: VR::LO },
            E { tag: 0x0017100F, alias: "Internal Data", vr: VR::LO },
            E { tag: 0x00171014, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171016, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171017, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171018, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171019, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017101A, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017101B, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017101C, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017101E, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017101F, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171020, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171021, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171022, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171023, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171024, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171025, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171026, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171027, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171028, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x00171029, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x00171048, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171049, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017104A, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017104B, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017104C, alias: "Internal Data", vr: VR::LO },
            E { tag: 0x0017104D, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017104E, alias: "Internal Data", vr: VR::LO },
            E { tag: 0x0017104F, alias: "Internal Data", vr: VR::LO },
            E { tag: 0x00171050, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171051, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x00171052, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171053, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171054, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x00171055, alias: "Internal Data", vr: VR::SS },
            E { tag: 0x0017105A, alias: "Internal Data", vr: VR::OW },
            E { tag: 0x0017105B, alias: "Internal Data", vr: VR::OW },
            E { tag: 0x0017105C, alias: "Internal Data", vr: VR::OW },
            E { tag: 0x00171064, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171066, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171067, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171068, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171085, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171086, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171087, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171088, alias: "Internal Data", vr: VR::US },
            E { tag: 0x00171089, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017108A, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017108B, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017108C, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x0017108D, alias: "Internal Data", vr: VR::FL },
            E { tag: 0x0017108E, alias: "Internal Data", vr: VR::US },
            E { tag: 0x0017108F, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A0, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A1, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A2, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A3, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A4, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A5, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710A6, alias: "Internal Data", vr: VR::US },
            E { tag: 0x001710B0, alias: "Internal Data", vr: VR::DS },
            E { tag: 0x001710C0, alias: "Internal Data", vr: VR::LO },
        ],
    },
    P {
        creator: "SIENET",
        entries: &[
            E { tag: 0x00091001, alias: "SIENET Command Field", vr: VR::US },
            E { tag: 0x00091014, alias: "Receiver PLA", vr: VR::LO },
            E { tag: 0x00091016, alias: "Transfer Priority", vr: VR::US },
            E { tag: 0x00091029, alias: "Actual User", vr: VR::LO },
            E { tag: 0x00091070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00091071, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00091072, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091073, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091074, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00911020, alias: "Patient Name", vr: VR::PN },
            E { tag: 0x00951001, alias: "Examination Folder ID", vr: VR::LO },
            E { tag: 0x00951004, alias: "Folder Reported Status", vr: VR::SL },
            E { tag: 0x00951005, alias: "Folder Reporting Radiologist", vr: VR::LO },
            E { tag: 0x00951007, alias: "SIENET ISA PLA", vr: VR::LO },
            E { tag: 0x0095100C, alias: "Folder Rebuild Status", vr: VR::SL },
            E { tag: 0x00951020, alias: "Unknown", vr: VR::UN },
            E { tag: 0x009510FB, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00991002, alias: "Key Images", vr: VR::UL },
            E { tag: 0x00991005, alias: "Image Number", vr: VR::SL },
        ],
    },
    P {
        creator: "SPI",
        entries: &[
            E { tag: 0x00091010, alias: "Comments", vr: VR::LO },
            E { tag: 0x00091015, alias: "Unique Identifier", vr: VR::LO },
            E { tag: 0x00091040, alias: "Data Object Type", vr: VR::US },
            E { tag: 0x00091041, alias: "Data Object Subtype", vr: VR::SH },
            E { tag: 0x00111010, alias: "Organ", vr: VR::LO },
            E { tag: 0x00111015, alias: "Allergy Indication", vr: VR::LO },
            E { tag: 0x00111020, alias: "Pregnancy", vr: VR::CS },
            E { tag: 0x00291060, alias: "Compression Algorithm", vr: VR::LO },
        ],
    },
    P {
        creator: "SPI RELEASE 1",
        entries: &[
            E { tag: 0x00091010, alias: "Comments", vr: VR::LO },
            E { tag: 0x00091015, alias: "Unique Identifier", vr: VR::LO },
            E { tag: 0x00091040, alias: "Data Object Type", vr: VR::US },
            E { tag: 0x00091041, alias: "Data Object Subtype", vr: VR::SH },
            E { tag: 0x00111010, alias: "Organ", vr: VR::LO },
            E { tag: 0x00111015, alias: "Allergy Indication", vr: VR::LO },
            E { tag: 0x00111020, alias: "Pregnancy", vr: VR::CS },
            E { tag: 0x00291060, alias: "Compression Algorithm", vr: VR::LO },
        ],
    },
    P {
        creator: "SPI Release 1",
        entries: &[
            E { tag: 0x00091008, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00091010, alias: "Comments", vr: VR::LO },
            E { tag: 0x00091015, alias: "UID", vr: VR::LO },
            E { tag: 0x00091040, alias: "Data Object Type", vr: VR::US },
            E { tag: 0x00091041, alias: "Data Object Subtype", vr: VR::SH },
            E { tag: 0x00111010, alias: "Organ", vr: VR::LO },
            E { tag: 0x00111015, alias: "Allergy Indication", vr: VR::LO },
            E { tag: 0x00111020, alias: "Pregnancy", vr: VR::LO },
            E { tag: 0x00291060, alias: "Compression Algorithm", vr: VR::LO },
        ],
    },
    P {
        creator: "SPI-P Release 1",
        entries: &[
            E { tag: 0x00091000, alias: "Data Object Recognition Code", vr: VR::LT },
            E { tag: 0x00091004, alias: "Image Data Consistency", vr: VR::LO },
            E { tag: 0x00091008, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091012, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091015, alias: "Unique Identifier", vr: VR::LO },
            E { tag: 0x00091016, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091018, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091021, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00091022, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091031, alias: "PACS Unique Identifier", vr: VR::LO },
            E { tag: 0x00091034, alias: "Cluster Unique Identifier", vr: VR::LO },
            E { tag: 0x00091038, alias: "System Unique Identifier", vr: VR::LO },
            E { tag: 0x00091039, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091040, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091051, alias: "Study Unique Identifier", vr: VR::LO },
            E { tag: 0x00091061, alias: "Series Unique Identifier", vr: VR::LO },
            E { tag: 0x00091091, alias: "Unknown", vr: VR::LT },
            E { tag: 0x000910A0, alias: "Unknown", vr: VR::UN },
            E { tag: 0x000910F2, alias: "Unknown", vr: VR::LT },
            E { tag: 0x000910F3, alias: "Unknown", vr: VR::UN },
            E { tag: 0x000910F4, alias: "Unknown", vr: VR::LT },
            E { tag: 0x000910F5, alias: "Unknown", vr: VR::UN },
            E { tag: 0x000910F7, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00111010, alias: "Patient Entry ID", vr: VR::LT },
            E { tag: 0x00111020, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111021, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111022, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111030, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111031, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00111032, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191001, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191010, alias: "Mains Frequency", vr: VR::US },
            E { tag: 0x00191025, alias: "Original Pixel Data Quality", vr: VR::LO },
            E { tag: 0x00191030, alias: "ECG Triggering", vr: VR::US },
            E { tag: 0x00191031, alias: "ECG 1 Offset", vr: VR::UN },
            E { tag: 0x00191032, alias: "ECG 2 Offset 1", vr: VR::UN },
            E { tag: 0x00191033, alias: "ECG 2 Offset 2", vr: VR::UN },
            E { tag: 0x00191050, alias: "Video Scan Mode", vr: VR::US },
            E { tag: 0x00191051, alias: "Video LineRate", vr: VR::US },
            E { tag: 0x00191060, alias: "Xray Technique", vr: VR::US },
            E { tag: 0x00191061, alias: "Image Identifier Format", vr: VR::DS },
            E { tag: 0x00191062, alias: "Iris Diaphragm", vr: VR::US },
            E { tag: 0x00191063, alias: "Filter", vr: VR::CS },
            E { tag: 0x00191064, alias: "Cine Parallel", vr: VR::CS },
            E { tag: 0x00191065, alias: "Cine Master", vr: VR::CS },
            E { tag: 0x00191070, alias: "Exposure Channel", vr: VR::US },
            E { tag: 0x00191071, alias: "Exposure Channel First Image", vr: VR::UN },
            E { tag: 0x00191072, alias: "Processing Channel", vr: VR::US },
            E { tag: 0x00191080, alias: "Acquisition Delay", vr: VR::DS },
            E { tag: 0x00191081, alias: "Relative Image Time", vr: VR::UN },
            E { tag: 0x00191090, alias: "Video White Compression", vr: VR::CS },
            E { tag: 0x001910A0, alias: "Angulation", vr: VR::US },
            E { tag: 0x001910A1, alias: "Rotation", vr: VR::US },
            E { tag: 0x00211012, alias: "Series Unique Identifier", vr: VR::LT },
            E { tag: 0x00211014, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291000, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291020, alias: "Pixel Aspect Ratio", vr: VR::DS },
            E { tag: 0x00291025, alias: "Processed Pixel Data Quality", vr: VR::LO },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291038, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291060, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291061, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291067, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291070, alias: "Window ID", vr: VR::LT },
            E { tag: 0x00291071, alias: "Video Invert Subtracted", vr: VR::CS },
            E { tag: 0x00291072, alias: "Video Invert Nonsubtracted", vr: VR::CS },
            E { tag: 0x00291077, alias: "Window Select Status", vr: VR::CS },
            E { tag: 0x00291078, alias: "ECG Display Printing ID", vr: VR::LT },
            E { tag: 0x00291079, alias: "ECG Display Printing", vr: VR::CS },
            E { tag: 0x0029107E, alias: "ECG Display Printing Enable Status", vr: VR::CS },
            E { tag: 0x0029107F, alias: "ECG Display Printing Select Status", vr: VR::CS },
            E { tag: 0x00291080, alias: "Physiological Display ID", vr: VR::LT },
            E { tag: 0x00291081, alias: "Preferred Physiological Channel Display", vr: VR::US },
            E { tag: 0x0029108E, alias: "Physiological Display Enable Status", vr: VR::CS },
            E { tag: 0x0029108F, alias: "Physiological Display Select Status", vr: VR::CS },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291091, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029109F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x002910A0, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002910A1, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910AF, alias: "Unknown", vr: VR::CS },
            E { tag: 0x002910B0, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002910B1, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910BF, alias: "Unknown", vr: VR::CS },
            E { tag: 0x002910C0, alias: "Functional Shutter ID", vr: VR::LT },
            E { tag: 0x002910C1, alias: "Field Of Shutter", vr: VR::US },
            E { tag: 0x002910C5, alias: "Field Of Shutter Rectangle", vr: VR::LT },
            E { tag: 0x002910CE, alias: "Shutter Enable Status", vr: VR::CS },
            E { tag: 0x002910CF, alias: "Shutter Select Status", vr: VR::CS },
            E { tag: 0x7FE11010, alias: "Pixel Data", vr: VR::UN },
        ],
    },
    P {
        creator: "SPI-P Release 1;1",
        entries: &[
            E { tag: 0x000910C0, alias: "Unknown", vr: VR::LT },
            E { tag: 0x000910C1, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191000, alias: "Physiological Data Type", vr: VR::UN },
            E { tag: 0x00191001, alias: "Physiological Data Channel And Kind", vr: VR::UN },
            E { tag: 0x00191002, alias: "Sample Bits Allocated", vr: VR::US },
            E { tag: 0x00191003, alias: "Sample Bits Stored", vr: VR::US },
            E { tag: 0x00191004, alias: "Sample High Bit", vr: VR::US },
            E { tag: 0x00191005, alias: "Sample Representation", vr: VR::US },
            E { tag: 0x00191006, alias: "Smallest Sample Value", vr: VR::UN },
            E { tag: 0x00191007, alias: "Largest Sample Value", vr: VR::UN },
            E { tag: 0x00191008, alias: "Number Of Samples", vr: VR::UN },
            E { tag: 0x00191009, alias: "Sample Data", vr: VR::UN },
            E { tag: 0x0019100A, alias: "Sample Rate", vr: VR::UN },
            E { tag: 0x00191010, alias: "Physiological Data Type 2", vr: VR::UN },
            E { tag: 0x00191011, alias: "Physiological Data Channel And Kind 2", vr: VR::UN },
            E { tag: 0x00191012, alias: "Sample Bits Allocated 2", vr: VR::US },
            E { tag: 0x00191013, alias: "Sample Bits Stored 2", vr: VR::US },
            E { tag: 0x00191014, alias: "Sample High Bit 2", vr: VR::US },
            E { tag: 0x00191015, alias: "Sample Representation 2", vr: VR::US },
            E { tag: 0x00191016, alias: "Smallest Sample Value 2", vr: VR::UN },
            E { tag: 0x00191017, alias: "Largest Sample Value 2", vr: VR::UN },
            E { tag: 0x00191018, alias: "Number Of Samples 2", vr: VR::UN },
            E { tag: 0x00191019, alias: "Sample Data 2", vr: VR::UN },
            E { tag: 0x0019101A, alias: "Sample Rate 2", vr: VR::UN },
            E { tag: 0x00211041, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211043, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211044, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211045, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211050, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211051, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211052, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211060, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211061, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211067, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0021106A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021106B, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021106C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021106D, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021106E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021106F, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291000, alias: "Zoom ID", vr: VR::LT },
            E { tag: 0x00291001, alias: "Zoom Rectangle", vr: VR::DS },
            E { tag: 0x00291003, alias: "Zoom Factor", vr: VR::DS },
            E { tag: 0x00291004, alias: "Zoom Function", vr: VR::US },
            E { tag: 0x0029100E, alias: "Zoom Enable Status", vr: VR::CS },
            E { tag: 0x0029100F, alias: "Zoom Select Status", vr: VR::CS },
            E { tag: 0x00291040, alias: "Magnifying Glass ID", vr: VR::LT },
            E { tag: 0x00291041, alias: "Magnifying Glass Rectangle", vr: VR::DS },
            E { tag: 0x00291043, alias: "Magnifying Glass Factor", vr: VR::DS },
            E { tag: 0x00291044, alias: "Magnifying Glass Function", vr: VR::US },
            E { tag: 0x0029104E, alias: "Magnifying Glass Enable Status", vr: VR::CS },
            E { tag: 0x0029104F, alias: "Magnifying Glass Select Status", vr: VR::CS },
        ],
    },
    P {
        creator: "SPI-P Release 1;2",
        entries: &[
            E { tag: 0x00291000, alias: "Subtraction Mask ID", vr: VR::LT },
            E { tag: 0x00291004, alias: "Masking Function", vr: VR::UN },
            E { tag: 0x0029100C, alias: "Proprietary Masking Parameters", vr: VR::UN },
            E { tag: 0x0029101E, alias: "Subtraction Mask Enable Status", vr: VR::CS },
            E { tag: 0x0029101F, alias: "Subtraction Mask Select Status", vr: VR::CS },
        ],
    },
    P {
        creator: "SPI-P Release 1;3",
        entries: &[
            E { tag: 0x00291000, alias: "Image Enhancement ID", vr: VR::LT },
            E { tag: 0x00291001, alias: "Image Enhancement", vr: VR::LT },
            E { tag: 0x00291002, alias: "Convolution ID", vr: VR::LT },
            E { tag: 0x00291003, alias: "Convolution Type", vr: VR::LT },
            E { tag: 0x00291004, alias: "Convolution Kernel Size ID", vr: VR::LT },
            E { tag: 0x00291005, alias: "Convolution Kernel Size", vr: VR::US },
            E { tag: 0x00291006, alias: "Convolution Kernel", vr: VR::US },
            E { tag: 0x0029100C, alias: "Enhancement Gain", vr: VR::DS },
            E { tag: 0x0029101E, alias: "Image Enhancement Enable Status", vr: VR::CS },
            E { tag: 0x0029101F, alias: "Image Enhancement Select Status", vr: VR::CS },
        ],
    },
    P {
        creator: "SPI-P Release 2;1",
        entries: &[
            E { tag: 0x00111018, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0023100D, alias: "Unknown", vr: VR::UI },
            E { tag: 0x0023100E, alias: "Unknown", vr: VR::UI },
        ],
    },
    P {
        creator: "SPI-P-CTBE Release 1",
        entries: &[
            E { tag: 0x00191000, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191003, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191004, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191005, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019100B, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019100C, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191014, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191018, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191019, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019101A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019101B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019101C, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019101D, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "SPI-P-CTBE-Private Release 1",
        entries: &[
            E { tag: 0x00211000, alias: "Unknown", vr: VR::DS },
        ],
    },
    P {
        creator: "SPI-P-GV-CT Release 1",
        entries: &[
            E { tag: 0x00091000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091020, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091030, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091040, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091060, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091080, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00091090, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191008, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191009, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019100A, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191020, alias: "Unknown", vr: VR::TM },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191060, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191061, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191063, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191064, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191065, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191070, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191080, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191081, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191090, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001910A0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001910A1, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910A2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910A3, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910B0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001910B1, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211020, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211030, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211040, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00211060, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211070, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211080, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00211090, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002110A0, alias: "Unknown", vr: VR::US },
            E { tag: 0x002110A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002110A2, alias: "Unknown", vr: VR::DS },
            E { tag: 0x002110A3, alias: "Unknown", vr: VR::LT },
            E { tag: 0x002110A4, alias: "Unknown", vr: VR::LT },
            E { tag: 0x002110B0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x002110C0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291031, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291033, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291080, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::LO },
            E { tag: 0x002910D0, alias: "Unknown", vr: VR::IS },
            E { tag: 0x002910D1, alias: "Unknown", vr: VR::IS },
        ],
    },
    P {
        creator: "SPI-P-PCR Release 2",
        entries: &[
            E { tag: 0x00191010, alias: "Reader Mode", vr: VR::US },
            E { tag: 0x00191020, alias: "MRM Code", vr: VR::ST },
            E { tag: 0x00191030, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191040, alias: "Latitude", vr: VR::DS },
            E { tag: 0x00191060, alias: "View Name", vr: VR::ST },
            E { tag: 0x00191080, alias: "Unknown", vr: VR::US },
            E { tag: 0x001910A1, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910A3, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A4, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A5, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A6, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A7, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A8, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910A9, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AA, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AB, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AC, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AD, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AE, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910AF, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910B0, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910B1, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910B2, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910B3, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910B4, alias: "Unknown", vr: VR::IS },
            E { tag: 0x001910B5, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910B6, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910B7, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910B8, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001910B9, alias: "Unknown", vr: VR::ST },
            E { tag: 0x001910BA, alias: "Unknown", vr: VR::ST },
        ],
    },
    P {
        creator: "SPI-P-Private-CWS Release 1",
        entries: &[
            E { tag: 0x00211000, alias: "Window Of Images ID", vr: VR::LT },
            E { tag: 0x00211001, alias: "Window Of Images Type", vr: VR::CS },
            E { tag: 0x00211002, alias: "WindowOfImagesScope", vr: VR::IS },
        ],
    },
    P {
        creator: "SPI-P-Private-DCI Release 1",
        entries: &[
            E { tag: 0x00191010, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191011, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191012, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191013, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191014, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191015, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191016, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191017, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "SPI-P-Private-DiDi Release 1",
        entries: &[
            E { tag: 0x00191000, alias: "Post Mode String", vr: VR::LT },
            E { tag: 0x00191001, alias: "Post Data", vr: VR::LT },
            E { tag: 0x00191010, alias: "Image Header", vr: VR::LT },
            E { tag: 0x00191020, alias: "SCPreName", vr: VR::LT },
            E { tag: 0x00191021, alias: "SCPostNo", vr: VR::IS },
            E { tag: 0x00191022, alias: "SCSOPInstanceUID", vr: VR::LT },
            E { tag: 0x00191023, alias: "SCCreationDate", vr: VR::LT },
            E { tag: 0x00191024, alias: "SCProcessingCategory", vr: VR::LT },
            E { tag: 0x00191025, alias: "SCPostMethod", vr: VR::LT },
            E { tag: 0x00191026, alias: "SCPostString", vr: VR::LT },
            E { tag: 0x00191027, alias: "SCPostString2", vr: VR::LT },
            E { tag: 0x00191028, alias: "SCMode", vr: VR::LT },
            E { tag: 0x00191029, alias: "SCStCommit", vr: VR::IS },
            E { tag: 0x0019102A, alias: "SCExported", vr: VR::IS },
            E { tag: 0x0019102B, alias: "SCPrinted", vr: VR::IS },
            E { tag: 0x0019102C, alias: "SCOperator", vr: VR::LT },
            E { tag: 0x00191040, alias: "SCPreName", vr: VR::LT },
            E { tag: 0x00191041, alias: "SCPreType", vr: VR::IS },
        ],
    },
    P {
        creator: "SPI-P-Private_CDS Release 1",
        entries: &[
            E { tag: 0x00211040, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291000, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1",
        entries: &[
            E { tag: 0x00191030, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00191031, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291007, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291008, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029100C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029100D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029100E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029100F, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291018, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0029101B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029101C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029101D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029101E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029104C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029104D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029104E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0029104F, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291050, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291051, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291066, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291067, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291068, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029106A, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0029106B, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291072, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291091, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00291093, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291094, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;1",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291005, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291022, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291023, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291024, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291025, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291026, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::ST },
            E { tag: 0x0029106B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910C1, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910CB, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910CC, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910CD, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910D4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910D5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910D6, alias: "Unknown", vr: VR::ST },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;2",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291002, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291005, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::LT },
            E { tag: 0x002910A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910A1, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910A2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910A3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910A5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910A6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910D9, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;3",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00291060, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910B0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910B1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910B2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910B4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910B9, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910C0, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910C1, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910C2, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910C3, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910C4, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910C5, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;4",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291002, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291006, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291007, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291008, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029100A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029100B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029100C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029100E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00291090, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0029109A, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029109C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x002910A0, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910A1, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910B1, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910B2, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910B3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910B5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x002910B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910BC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x002910BE, alias: "Unknown", vr: VR::US },
            E { tag: 0x002910D6, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910D7, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910D8, alias: "Unknown", vr: VR::UN },
            E { tag: 0x002910D9, alias: "Unknown", vr: VR::UN },
            E { tag: 0x002910DA, alias: "Unknown", vr: VR::UN },
            E { tag: 0x002910DC, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910DD, alias: "Unknown", vr: VR::FD },
            E { tag: 0x002910E0, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;5",
        entries: &[
            E { tag: 0x00291050, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291055, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "SPI-P-Private_ICS Release 1;6",
        entries: &[
            E { tag: 0x00291001, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291002, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291003, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291006, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291007, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291008, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029100F, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291014, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291015, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291017, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029101A, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0029101F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291023, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291024, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029102A, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029102B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029102C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029102D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029102E, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029102F, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291034, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291035, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291036, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291037, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291038, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291039, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029103D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291043, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291044, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291048, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029104F, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029105C, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0029105D, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029105E, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029105F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291063, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291065, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291066, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291067, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291068, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291069, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029106D, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "SPI-P-XSB-DCI Release 1",
        entries: &[
            E { tag: 0x00191010, alias: "Video Beam Boost", vr: VR::LT },
            E { tag: 0x00191011, alias: "Channel Generating Video Sync", vr: VR::US },
            E { tag: 0x00191012, alias: "Video Gain", vr: VR::US },
            E { tag: 0x00191013, alias: "Video Offset", vr: VR::US },
            E { tag: 0x00191020, alias: "RTD Data Compression Factor", vr: VR::DS },
        ],
    },
    P {
        creator: "SPI-P-XSB-VISUB Release 1",
        entries: &[
            E { tag: 0x00191000, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191010, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191011, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191012, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191020, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191040, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00191050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291000, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291001, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291002, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0029100F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291010, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291011, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291012, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0029101F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291020, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291022, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0029102F, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291031, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::CS },
            E { tag: 0x0029103F, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "STENTOR",
        entries: &[
            E { tag: 0x00731001, alias: "Unknown", vr: VR::ST },
            E { tag: 0x00731002, alias: "Private Creator", vr: VR::LT },
            E { tag: 0x00731003, alias: "Stentor Remote AETitle Element", vr: VR::UT },
            E { tag: 0x00731004, alias: "Stentor Local AETitle Element", vr: VR::UT },
            E { tag: 0x00731006, alias: "Stentor Transfer Syntax Value", vr: VR::UT },
        ],
    },
    P {
        creator: "SVISION",
        entries: &[
            E { tag: 0x00171000, alias: "Extended Body Part", vr: VR::LO },
            E { tag: 0x00171010, alias: "Extended View Position", vr: VR::LO },
            E { tag: 0x00171020, alias: "Scheduled Procedure Step", vr: VR::SQ },
            E { tag: 0x001710A0, alias: "Fixed Grid System", vr: VR::IS },
            E { tag: 0x001710F0, alias: "Images SOP Class", vr: VR::IS },
            E { tag: 0x00191000, alias: "AEC Field", vr: VR::IS },
            E { tag: 0x00191001, alias: "AEC Film Screen", vr: VR::IS },
            E { tag: 0x00191002, alias: "AEC Density", vr: VR::IS },
            E { tag: 0x00191010, alias: "Patient Thickness", vr: VR::IS },
            E { tag: 0x00191016, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191018, alias: "Beam Distance", vr: VR::IS },
            E { tag: 0x00191020, alias: "Workstation Number", vr: VR::IS },
            E { tag: 0x00191028, alias: "Tube Number", vr: VR::IS },
            E { tag: 0x00191030, alias: "Bucky Grid", vr: VR::IS },
            E { tag: 0x00191034, alias: "Focus", vr: VR::IS },
            E { tag: 0x00191038, alias: "Age Group", vr: VR::IS },
            E { tag: 0x00191040, alias: "Collimator Distance X", vr: VR::IS },
            E { tag: 0x00191041, alias: "Collimator Distance Y", vr: VR::IS },
            E { tag: 0x00191050, alias: "Bucky Height", vr: VR::IS },
            E { tag: 0x00191060, alias: "Bucky Angle", vr: VR::IS },
            E { tag: 0x00191068, alias: "C-Arm Angle", vr: VR::IS },
            E { tag: 0x00191069, alias: "Collimator Angle", vr: VR::IS },
            E { tag: 0x00191070, alias: "Filter Number", vr: VR::IS },
            E { tag: 0x00191074, alias: "Filter Material 1", vr: VR::LO },
            E { tag: 0x00191075, alias: "Filter Material 2", vr: VR::LO },
            E { tag: 0x00191078, alias: "Filter Thickness 1", vr: VR::DS },
            E { tag: 0x00191079, alias: "Filter Thickness 2", vr: VR::DS },
            E { tag: 0x00191080, alias: "Bucky Format", vr: VR::IS },
            E { tag: 0x00191081, alias: "Object Position", vr: VR::IS },
            E { tag: 0x00191090, alias: "Desk Command", vr: VR::LO },
            E { tag: 0x00191091, alias: "Central Beam X", vr: VR::IS },
            E { tag: 0x00191092, alias: "Central Beam Y", vr: VR::IS },
            E { tag: 0x00191093, alias: "Tube Turn Angle", vr: VR::IS },
            E { tag: 0x00191094, alias: "Stand drive level", vr: VR::IS },
            E { tag: 0x001910A0, alias: "Extended Exposure Time", vr: VR::DS },
            E { tag: 0x001910A1, alias: "Actual Exposure Time", vr: VR::DS },
            E { tag: 0x001910A8, alias: "Extended X-ray Tube Current", vr: VR::DS },
            E { tag: 0x001910B0, alias: "Dose Indicator", vr: VR::IS },
            E { tag: 0x001910B1, alias: "Shift Reference Value", vr: VR::IS },
            E { tag: 0x001910F0, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00211000, alias: "Noise Reduction", vr: VR::DS },
            E { tag: 0x00211001, alias: "Contrast Amplification", vr: VR::DS },
            E { tag: 0x00211002, alias: "Edge Contrast Boosting", vr: VR::DS },
            E { tag: 0x00211003, alias: "Latitude Reduction", vr: VR::DS },
            E { tag: 0x00211010, alias: "Find Range Algorithm", vr: VR::LO },
            E { tag: 0x00211011, alias: "Threshold C-Algorithm", vr: VR::DS },
            E { tag: 0x00211020, alias: "Sensometric Curve", vr: VR::LO },
            E { tag: 0x00211030, alias: "Lower Window Offset", vr: VR::DS },
            E { tag: 0x00211031, alias: "Upper Window Offset", vr: VR::DS },
            E { tag: 0x00211040, alias: "Min. Printable Density", vr: VR::DS },
            E { tag: 0x00211041, alias: "Max. Printable Density", vr: VR::DS },
            E { tag: 0x00211050, alias: "Min. Window Latitude", vr: VR::DS },
            E { tag: 0x00211051, alias: "Max. Window Latitude", vr: VR::DS },
            E { tag: 0x00211052, alias: "Rel. Window Alignment", vr: VR::DS },
            E { tag: 0x00211060, alias: "Decomposition Layer", vr: VR::DS },
            E { tag: 0x00211090, alias: "Brightness", vr: VR::DS },
            E { tag: 0x00211091, alias: "Contrast", vr: VR::DS },
            E { tag: 0x00211092, alias: "Shape Factor", vr: VR::DS },
            E { tag: 0x00231000, alias: "Image Laterality", vr: VR::LO },
            E { tag: 0x00231001, alias: "Letter Position", vr: VR::IS },
            E { tag: 0x00231002, alias: "Burned In Annotation", vr: VR::IS },
            E { tag: 0x00231003, alias: "Unknown", vr: VR::LO },
            E { tag: 0x002310F0, alias: "Image SOP Class", vr: VR::IS },
            E { tag: 0x00251000, alias: "Original Image", vr: VR::IS },
            E { tag: 0x00251001, alias: "Not Processed Image", vr: VR::IS },
            E { tag: 0x00251002, alias: "Cut Out Image", vr: VR::IS },
            E { tag: 0x00251003, alias: "Duplicated Image", vr: VR::IS },
            E { tag: 0x00251004, alias: "Stored Image", vr: VR::IS },
            E { tag: 0x00251005, alias: "Retrieved Image", vr: VR::IS },
            E { tag: 0x00251006, alias: "New Image", vr: VR::IS },
            E { tag: 0x00251007, alias: "Media Stored Image", vr: VR::IS },
            E { tag: 0x00251008, alias: "Image State", vr: VR::IS },
            E { tag: 0x00251009, alias: "Image Stitched Manually", vr: VR::IS },
            E { tag: 0x0025100A, alias: "Image Stitched Automatically", vr: VR::IS },
            E { tag: 0x00251020, alias: "Source Image File", vr: VR::LO },
            E { tag: 0x00251021, alias: "Source UID", vr: VR::LO },
            E { tag: 0x00271000, alias: "Number of Series", vr: VR::IS },
            E { tag: 0x00271001, alias: "Number of Studies", vr: VR::IS },
            E { tag: 0x00271010, alias: "Oldest Series", vr: VR::DT },
            E { tag: 0x00271011, alias: "Newest Series", vr: VR::DT },
            E { tag: 0x00271012, alias: "Oldest Study", vr: VR::DT },
            E { tag: 0x00271013, alias: "Newest Study", vr: VR::DT },
            E { tag: 0x00291000, alias: "Key Note Instance UID", vr: VR::IS },
            E { tag: 0x00291001, alias: "Storage State", vr: VR::IS },
            E { tag: 0x00291002, alias: "Referenced Image SOP Class", vr: VR::IS },
            E { tag: 0x00291003, alias: "Ref. Image Instance UID", vr: VR::LO },
            E { tag: 0x00291004, alias: "Rel. Presentation State Number", vr: VR::IS },
            E { tag: 0x00291005, alias: "Rel. Presentation State UID", vr: VR::LO },
        ],
    },
    P {
        creator: "Siemens Ultrasound Miscellaneous",
        entries: &[
            E { tag: 0x00191020, alias: "Unknown", vr: VR::SH },
        ],
    },
    P {
        creator: "Siemens: Thorax/Multix FD Image Stamp",
        entries: &[
            E { tag: 0x00231000, alias: "Unknown", vr: VR::US },
            E { tag: 0x00231001, alias: "Unknown", vr: VR::US },
            E { tag: 0x00231002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00231003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00231004, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "Siemens: Thorax/Multix FD Lab Settings",
        entries: &[
            E { tag: 0x00191000, alias: "Internal Value", vr: VR::LT },
            E { tag: 0x00191002, alias: "Total Area Dose Product", vr: VR::LO },
            E { tag: 0x00191004, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00191005, alias: "Cassette Image", vr: VR::US },
            E { tag: 0x00191006, alias: "Table Object Distance", vr: VR::FD },
            E { tag: 0x00191007, alias: "Table Detector Distance", vr: VR::FD },
            E { tag: 0x00191008, alias: "Ortho Step Distance", vr: VR::US },
            E { tag: 0x00211008, alias: "Auto Window Flag", vr: VR::US },
            E { tag: 0x00211009, alias: "Auto Window Center", vr: VR::SL },
            E { tag: 0x0021100A, alias: "Auto Window Width", vr: VR::SL },
            E { tag: 0x0021100B, alias: "Filter ID", vr: VR::SS },
            E { tag: 0x00211014, alias: "Anatomic Correct View", vr: VR::US },
            E { tag: 0x00211015, alias: "Auto Window Shift", vr: VR::SS },
            E { tag: 0x00211016, alias: "Auto Window Expansion", vr: VR::DS },
            E { tag: 0x00211017, alias: "System Type", vr: VR::LO },
            E { tag: 0x00211030, alias: "Anatomic Sort Number", vr: VR::SH },
            E { tag: 0x00211031, alias: "Acquisition Sort Number", vr: VR::SH },
        ],
    },
    P {
        creator: "Siemens: Thorax/Multix FD Post Processing",
        entries: &[
            E { tag: 0x00211000, alias: "Label Left", vr: VR::US },
            E { tag: 0x00211001, alias: "Label Left Orientation", vr: VR::SS },
            E { tag: 0x00211002, alias: "Label Left Percentage X", vr: VR::FL },
            E { tag: 0x00211003, alias: "Label Left Percentage X", vr: VR::FL },
            E { tag: 0x00211004, alias: "Label Right", vr: VR::US },
            E { tag: 0x00211005, alias: "Label Right Orientation", vr: VR::SS },
            E { tag: 0x00211006, alias: "Label Right Percentage X", vr: VR::FL },
            E { tag: 0x00211007, alias: "Label Right Percentage X", vr: VR::FL },
            E { tag: 0x00211008, alias: "Auto Window Flag", vr: VR::US },
            E { tag: 0x00211009, alias: "Auto Window Center", vr: VR::SL },
            E { tag: 0x0021100A, alias: "Auto Window Width", vr: VR::SL },
            E { tag: 0x0021100B, alias: "Filter ID", vr: VR::SS },
            E { tag: 0x0021100C, alias: "Degree of Edge Enhancement", vr: VR::FL },
            E { tag: 0x0021100D, alias: "Half Kernel Size", vr: VR::SS },
            E { tag: 0x0021100E, alias: "Dose Control Value", vr: VR::US },
            E { tag: 0x0021100F, alias: "Deleted Image Flag", vr: VR::US },
            E { tag: 0x00211010, alias: "Same Size Support", vr: VR::US },
            E { tag: 0x00211011, alias: "Label Left Init Position", vr: VR::SS },
            E { tag: 0x00211012, alias: "Label Right Init Position", vr: VR::SS },
            E { tag: 0x00211013, alias: "Patient Rotation", vr: VR::SS },
            E { tag: 0x00211014, alias: "Anatomic Correct View Flag", vr: VR::US },
            E { tag: 0x00211015, alias: "Auto Window Shift", vr: VR::SS },
            E { tag: 0x00211016, alias: "Auto Window Expansion", vr: VR::DS },
        ],
    },
    P {
        creator: "Siemens: Thorax/Multix FD Raw Image Settings",
        entries: &[
            E { tag: 0x00251000, alias: "Raw Image Amplification", vr: VR::SS },
            E { tag: 0x00251001, alias: "Gamma LUT", vr: VR::SS },
            E { tag: 0x00251002, alias: "Collimator Manual", vr: VR::US },
            E { tag: 0x00251003, alias: "Collimator Up Left X Percentage", vr: VR::FL },
            E { tag: 0x00251004, alias: "Collimator Up Right X Percentage", vr: VR::FL },
            E { tag: 0x00251005, alias: "Collimator Low Left X Percentage", vr: VR::FL },
            E { tag: 0x00251006, alias: "Collimator Low Right X Percentage", vr: VR::FL },
            E { tag: 0x00251007, alias: "Collimator Up Left Y Percentage", vr: VR::FL },
            E { tag: 0x00251008, alias: "Collimator Up Right Y Percentage", vr: VR::FL },
            E { tag: 0x00251009, alias: "Collimator Low Left Y Percentage", vr: VR::FL },
            E { tag: 0x0025100A, alias: "Collimator Low Right Y Percentage", vr: VR::FL },
            E { tag: 0x0025100B, alias: "Set Gray", vr: VR::US },
            E { tag: 0x0025100C, alias: "Harmonization Kernel", vr: VR::SS },
            E { tag: 0x0025100D, alias: "Harmonization Gain", vr: VR::FL },
            E { tag: 0x0025100E, alias: "Edge Enhancement Kernel", vr: VR::SS },
            E { tag: 0x0025100F, alias: "Edge Enhancement Gain", vr: VR::FL },
            E { tag: 0x00251010, alias: "Raw Image File Name", vr: VR::LT },
            E { tag: 0x00251011, alias: "Global Offset", vr: VR::SS },
            E { tag: 0x00251012, alias: "Image Columns", vr: VR::SS },
            E { tag: 0x00251013, alias: "Image Rows", vr: VR::SS },
            E { tag: 0x00251014, alias: "Image Alignment", vr: VR::SS },
            E { tag: 0x00251015, alias: "Active Image Columns", vr: VR::SS },
            E { tag: 0x00251016, alias: "Active Image Rows", vr: VR::SS },
            E { tag: 0x00251017, alias: "System Type", vr: VR::LO },
            E { tag: 0x00251018, alias: "Auto Gain", vr: VR::US },
            E { tag: 0x00251019, alias: "Ortho Sub Sampling", vr: VR::US },
            E { tag: 0x0025101A, alias: "Image Crop Upper Left", vr: VR::US },
            E { tag: 0x0025101B, alias: "Image Crop Upper Right", vr: VR::US },
            E { tag: 0x0025101C, alias: "Image Crop Lower Left", vr: VR::US },
            E { tag: 0x0025101D, alias: "Image Crop Lower Right", vr: VR::US },
            E { tag: 0x00251030, alias: "Manual Cropping", vr: VR::US },
            E { tag: 0x00251031, alias: "Gamma LUT Parameter 1", vr: VR::SS },
            E { tag: 0x00251032, alias: "Gamma LUT Parameter 2", vr: VR::DS },
            E { tag: 0x00251033, alias: "Gamma LUT Parameter 3", vr: VR::SS },
            E { tag: 0x00251034, alias: "Gamma LUT Parameter 4", vr: VR::SS },
        ],
    },
    P {
        creator: "Siemens: Thorax/Multix FD Version",
        entries: &[
            E { tag: 0x00171000, alias: "Build Date", vr: VR::LO },
            E { tag: 0x00171001, alias: "IOD Version", vr: VR::LO },
        ],
    },
    P {
        creator: "Silhouette Annot V1.0",
        entries: &[
            E { tag: 0x00291011, alias: "Annotation Name", vr: VR::IS },
            E { tag: 0x00291012, alias: "Annotation Font", vr: VR::LT },
            E { tag: 0x00291013, alias: "Annotation Text Foreground Color", vr: VR::LT },
            E { tag: 0x00291014, alias: "Annotation Text Background Color", vr: VR::LT },
            E { tag: 0x00291015, alias: "Annotation Text Backing Mode", vr: VR::UL },
            E { tag: 0x00291016, alias: "Annotation Text Justification", vr: VR::UL },
            E { tag: 0x00291017, alias: "Annotation Text Location", vr: VR::UL },
            E { tag: 0x00291018, alias: "Annotation Text String", vr: VR::LT },
            E { tag: 0x00291019, alias: "Annotation Text Attach Mode", vr: VR::UL },
            E { tag: 0x00291020, alias: "Annotation Text Cursor Mode", vr: VR::UL },
            E { tag: 0x00291021, alias: "Annotation Text Shadow Offset X", vr: VR::UL },
            E { tag: 0x00291022, alias: "Annotation Text Shadow Offset Y", vr: VR::UL },
            E { tag: 0x00291023, alias: "Annotation Line Color", vr: VR::LT },
            E { tag: 0x00291024, alias: "Annotation Line Thickness", vr: VR::UL },
            E { tag: 0x00291025, alias: "Annotation Line Type", vr: VR::UL },
            E { tag: 0x00291026, alias: "Annotation Line Style", vr: VR::UL },
            E { tag: 0x00291027, alias: "Annotation Line Dash Length", vr: VR::UL },
            E { tag: 0x00291028, alias: "Annotation Line Attach Mode", vr: VR::UL },
            E { tag: 0x00291029, alias: "Annotation Line Point Count", vr: VR::UL },
            E { tag: 0x00291030, alias: "Annotation Line Points", vr: VR::FD },
            E { tag: 0x00291031, alias: "Annotation Line Control Size", vr: VR::UL },
            E { tag: 0x00291032, alias: "Annotation Marker Color", vr: VR::LT },
            E { tag: 0x00291033, alias: "Annotation Marker Type", vr: VR::UL },
            E { tag: 0x00291034, alias: "Annotation Marker Size", vr: VR::UL },
            E { tag: 0x00291035, alias: "Annotation Marker Location", vr: VR::FD },
            E { tag: 0x00291036, alias: "Annotation Marker Attach Mode", vr: VR::UL },
            E { tag: 0x00291037, alias: "Annotation Geom Color", vr: VR::LT },
            E { tag: 0x00291038, alias: "Annotation Geom Thickness", vr: VR::UL },
            E { tag: 0x00291039, alias: "Annotation Geom Line Style", vr: VR::UL },
            E { tag: 0x00291040, alias: "Annotation Geom Dash Length", vr: VR::UL },
            E { tag: 0x00291041, alias: "Annotation Geom Fill Pattern", vr: VR::UL },
            E { tag: 0x00291042, alias: "Annotation Interactivity", vr: VR::UL },
            E { tag: 0x00291043, alias: "Annotation Arrow Length", vr: VR::FD },
            E { tag: 0x00291044, alias: "Annotation Arrow Angle", vr: VR::FD },
            E { tag: 0x00291045, alias: "Annotation Don't Save", vr: VR::UL },
        ],
    },
    P {
        creator: "Silhouette Graphics Export V1.0",
        entries: &[
            E { tag: 0x00291000, alias: "Unknown", vr: VR::UI },
        ],
    },
    P {
        creator: "Silhouette Line V1.0",
        entries: &[
            E { tag: 0x00291011, alias: "Line Name", vr: VR::IS },
            E { tag: 0x00291012, alias: "Line Name Font", vr: VR::LT },
            E { tag: 0x00291013, alias: "Line Name Display", vr: VR::UL },
            E { tag: 0x00291014, alias: "Line Normal Color", vr: VR::LT },
            E { tag: 0x00291015, alias: "Line Type", vr: VR::UL },
            E { tag: 0x00291016, alias: "Line Thickness", vr: VR::UL },
            E { tag: 0x00291017, alias: "Line Style", vr: VR::UL },
            E { tag: 0x00291018, alias: "Line Dash Length", vr: VR::UL },
            E { tag: 0x00291019, alias: "Line Interactivity", vr: VR::UL },
            E { tag: 0x00291020, alias: "Line Measurement Color", vr: VR::LT },
            E { tag: 0x00291021, alias: "Line Measurement Font", vr: VR::LT },
            E { tag: 0x00291022, alias: "Line Measurement Dash Length", vr: VR::UL },
            E { tag: 0x00291023, alias: "Line Point Space", vr: VR::UL },
            E { tag: 0x00291024, alias: "Line Points", vr: VR::FD },
            E { tag: 0x00291025, alias: "Line Control Point Size", vr: VR::UL },
            E { tag: 0x00291026, alias: "Line Control Point Space", vr: VR::UL },
            E { tag: 0x00291027, alias: "Line Control Points", vr: VR::FD },
            E { tag: 0x00291028, alias: "Line Label", vr: VR::LT },
            E { tag: 0x00291029, alias: "Line Don't Save", vr: VR::UL },
        ],
    },
    P {
        creator: "Silhouette ROI V1.0",
        entries: &[
            E { tag: 0x00291011, alias: "ROI Name", vr: VR::IS },
            E { tag: 0x00291012, alias: "ROI Name Font", vr: VR::LT },
            E { tag: 0x00291013, alias: "ROI Normal Color", vr: VR::LT },
            E { tag: 0x00291014, alias: "ROI Fill Pattern", vr: VR::UL },
            E { tag: 0x00291015, alias: "ROI Bp Seg", vr: VR::UL },
            E { tag: 0x00291016, alias: "ROI Bp Seg Pairs", vr: VR::UN },
            E { tag: 0x00291017, alias: "ROI Seed Space", vr: VR::UL },
            E { tag: 0x00291018, alias: "ROI Seeds", vr: VR::UN },
            E { tag: 0x00291019, alias: "ROI Line Thickness", vr: VR::UL },
            E { tag: 0x00291020, alias: "ROI Line Style", vr: VR::UL },
            E { tag: 0x00291021, alias: "ROI Line Dash Length", vr: VR::UL },
            E { tag: 0x00291022, alias: "ROI Interactivity", vr: VR::UL },
            E { tag: 0x00291023, alias: "ROI Name Position", vr: VR::UL },
            E { tag: 0x00291024, alias: "ROI Name Display", vr: VR::UL },
            E { tag: 0x00291025, alias: "ROI Label", vr: VR::LT },
            E { tag: 0x00291026, alias: "ROI Shape", vr: VR::UL },
            E { tag: 0x00291027, alias: "ROI Shape Tilt", vr: VR::FD },
            E { tag: 0x00291028, alias: "ROI Shape Points Count", vr: VR::UL },
            E { tag: 0x00291029, alias: "ROI Shape Points Space", vr: VR::UL },
            E { tag: 0x00291030, alias: "ROI Shape Points", vr: VR::FD },
            E { tag: 0x00291031, alias: "ROI Shape Control Points Count", vr: VR::UL },
            E { tag: 0x00291032, alias: "ROI Shape Control Points Space", vr: VR::UL },
            E { tag: 0x00291033, alias: "ROI Shape Control Points", vr: VR::FD },
            E { tag: 0x00291034, alias: "ROI Don't Save", vr: VR::UL },
        ],
    },
    P {
        creator: "Silhouette Sequence Ids V1.0",
        entries: &[
            E { tag: 0x00291041, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291042, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00291043, alias: "Unknown", vr: VR::SQ },
        ],
    },
    P {
        creator: "Silhouette V1.0",
        entries: &[
            E { tag: 0x00291013, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291014, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291017, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291018, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291019, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101A, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0029101B, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101D, alias: "Unknown", vr: VR::UN },
            E { tag: 0x0029101E, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291021, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291022, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291023, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291024, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291025, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291027, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291028, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291029, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291052, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291053, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291054, alias: "Unknown", vr: VR::UN },
            E { tag: 0x00291055, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291056, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00291057, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "Silhouette VRS 3.0",
        entries: &[
            E { tag: 0x00291013, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291014, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291017, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291018, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00291019, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0029101B, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101D, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0029101E, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00291027, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291028, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00291030, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291032, alias: "Unknown", vr: VR::US },
            E { tag: 0x00291034, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00291035, alias: "Unknown", vr: VR::CS },
            E { tag: 0x00291036, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "Sound Technologies",
        entries: &[
            E { tag: 0xF0011000, alias: "Patient Species", vr: VR::CS },
            E { tag: 0xF0011001, alias: "Patient Breed", vr: VR::CS },
            E { tag: 0xF0011002, alias: "Patient Category Size", vr: VR::CS },
            E { tag: 0xF0011003, alias: "Patient Sex Extended (including spay/neuter)", vr: VR::CS },
            E { tag: 0xF0011004, alias: "Image View", vr: VR::CS },
            E { tag: 0xF0011005, alias: "Anatomy Imaged", vr: VR::CS },
            E { tag: 0xF0011006, alias: "Image enhancements", vr: VR::CS },
            E { tag: 0xF0011007, alias: "Detector Settings", vr: VR::CS },
            E { tag: 0xF0011008, alias: "Application Version", vr: VR::CS },
            E { tag: 0xF0011009, alias: "Image Laterality Extended", vr: VR::LO },
            E { tag: 0xF001100A, alias: "Client Name", vr: VR::PN },
            E { tag: 0xF001100B, alias: "Reference Study Instance Uid", vr: VR::UI },
            E { tag: 0xF001100C, alias: "Reference Series Instance Uid", vr: VR::UI },
            E { tag: 0xF001100D, alias: "Exam Ref Id", vr: VR::LO },
            E { tag: 0xF001100E, alias: "Physician Of Record Address", vr: VR::ST },
            E { tag: 0xF001100F, alias: "Physician Of Record Phone Numbers", vr: VR::SH },
            E { tag: 0xF0011010, alias: "Reason For Study", vr: VR::LT },
            E { tag: 0xF0011011, alias: "Protocol", vr: VR::LO },
            E { tag: 0xF0011012, alias: "CaptureInputType", vr: VR::LO },
            E { tag: 0xF0011013, alias: "Exam Complaint", vr: VR::LT },
            E { tag: 0xF0011014, alias: "Exam Web Code", vr: VR::LO },
            E { tag: 0xF0011015, alias: "Exam Category", vr: VR::LO },
            E { tag: 0xF0011016, alias: "Exam Diagnosis", vr: VR::SH },
            E { tag: 0xF0011017, alias: "Exam Created By", vr: VR::PN },
            E { tag: 0xF0011018, alias: "Exam Created By Group", vr: VR::LO },
            E { tag: 0xF0011019, alias: "Exam Required By DateTime", vr: VR::DT },
            E { tag: 0xF001101A, alias: "Capture Type", vr: VR::LO },
            E { tag: 0xF001101B, alias: "Telemed Exam ID", vr: VR::IS },
            E { tag: 0xF001101C, alias: "Exam Created By Guid", vr: VR::LO },
            E { tag: 0xF001101D, alias: "Client Name Guid", vr: VR::LO },
        ],
    },
    P {
        creator: "TMDTK",
        entries: &[
            E { tag: 0x7EFF1021, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x7EFF102E, alias: "Unknown", vr: VR::LT },
        ],
    },
    P {
        creator: "TOSHIBA COMAPL HEADER",
        entries: &[
            E { tag: 0x00291008, alias: "COMAPL Header Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "COMAPL Header Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "", vr: VR::LO },
            E { tag: 0x00291020, alias: "COMAPL History Information", vr: VR::OB },
            E { tag: 0x00291031, alias: "", vr: VR::LO },
            E { tag: 0x00291032, alias: "", vr: VR::UL },
            E { tag: 0x00291033, alias: "", vr: VR::UL },
            E { tag: 0x00291034, alias: "", vr: VR::CS },
        ],
    },
    P {
        creator: "TOSHIBA COMAPL OOG",
        entries: &[
            E { tag: 0x00291008, alias: "COMAPL OOG Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "COMAPL OOG Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "", vr: VR::LO },
            E { tag: 0x00291031, alias: "", vr: VR::LO },
            E { tag: 0x00291032, alias: "", vr: VR::UL },
            E { tag: 0x00291033, alias: "", vr: VR::UL },
            E { tag: 0x00291034, alias: "", vr: VR::CS },
        ],
    },
    P {
        creator: "TOSHIBA ENCRYPTED SR DATA",
        entries: &[
            E { tag: 0x70151000, alias: "Toshiba US Private Data", vr: VR::OB },
        ],
    },
    P {
        creator: "TOSHIBA MDW HEADER",
        entries: &[
            E { tag: 0x00291008, alias: "Application Header Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "Application Header Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "Image Header Type", vr: VR::CS },
            E { tag: 0x00291011, alias: "Image Header Version", vr: VR::LO },
            E { tag: 0x00291012, alias: "Image Header Info", vr: VR::OB },
            E { tag: 0x00291018, alias: "Series Header Type", vr: VR::CS },
            E { tag: 0x00291019, alias: "Series Header Version", vr: VR::LO },
            E { tag: 0x00291020, alias: "Series Header Info", vr: VR::OB },
            E { tag: 0x00291031, alias: "PMTF Information 1", vr: VR::LO },
            E { tag: 0x00291032, alias: "PMTF Information 2", vr: VR::UL },
            E { tag: 0x00291033, alias: "PMTF Information 3", vr: VR::UL },
            E { tag: 0x00291034, alias: "PMTF Information 4", vr: VR::CS },
        ],
    },
    P {
        creator: "TOSHIBA MDW NON-IMAGE",
        entries: &[
            E { tag: 0x00291008, alias: "Application Header Type", vr: VR::CS },
            E { tag: 0x00291009, alias: "Application Header Version", vr: VR::LO },
            E { tag: 0x00291010, alias: "Application Header Data", vr: VR::OB },
            E { tag: 0x00291020, alias: "Application Header Data", vr: VR::OB },
            E { tag: 0x7FE11010, alias: "Toshiba US Private Data", vr: VR::OB },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_1.0",
        entries: &[
            E { tag: 0x00091001, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00091002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00091004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00111001, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00111002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191001, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211001, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00211003, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11001, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11002, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11003, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11010, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_CT3",
        entries: &[
            E { tag: 0x70051000, alias: "CT Private Data 1", vr: VR::OB },
            E { tag: 0x70051003, alias: "Cardiac R-R Mean Time", vr: VR::SH },
            E { tag: 0x70051004, alias: "Cardiac Reconstruction Gating Phase in Percent", vr: VR::SH },
            E { tag: 0x70051005, alias: "Cardiac Reconstruction Gating Phase in ms", vr: VR::SH },
            E { tag: 0x70051006, alias: "Cardiac Reconstruction Mode", vr: VR::SH },
            E { tag: 0x70051007, alias: "Reconstruction Center", vr: VR::DS },
            E { tag: 0x70051008, alias: "Detector Slice Thickness in mm", vr: VR::DS },
            E { tag: 0x70051009, alias: "Number of Detector rows to Reconstruct", vr: VR::LO },
            E { tag: 0x7005100A, alias: "Table Speed in mm/rot", vr: VR::DS },
            E { tag: 0x7005100B, alias: "Filter", vr: VR::SH },
            E { tag: 0x7005100C, alias: "Reconstruction Correction Type", vr: VR::SH },
            E { tag: 0x7005100D, alias: "Organ", vr: VR::CS },
            E { tag: 0x7005100E, alias: "File Type Remarks", vr: VR::SH },
            E { tag: 0x7005100F, alias: "Direction", vr: VR::SH },
            E { tag: 0x70051010, alias: "CT Private Data 2", vr: VR::OB },
            E { tag: 0x70051011, alias: "Series Comment", vr: VR::LT },
            E { tag: 0x70051012, alias: "Position", vr: VR::SH },
            E { tag: 0x70051013, alias: "Expert Plan No.", vr: VR::US },
            E { tag: 0x70051014, alias: "Reconstruction ROI No.", vr: VR::US },
            E { tag: 0x70051015, alias: "Special Helical ACQ No.", vr: VR::US },
            E { tag: 0x70051016, alias: "Volume UID", vr: VR::UI },
            E { tag: 0x70051017, alias: "Total Frame Count in the Volume", vr: VR::US },
            E { tag: 0x70051018, alias: "Frame No.", vr: VR::US },
            E { tag: 0x70051019, alias: "Frame Sort Key", vr: VR::UL },
            E { tag: 0x7005101A, alias: "Frame Sort Order", vr: VR::US },
            E { tag: 0x7005101B, alias: "Convolution Kernel", vr: VR::SH },
            E { tag: 0x7005101C, alias: "Contrast/Bolus Agent", vr: VR::LO },
            E { tag: 0x7005101D, alias: "Reconstruction Number", vr: VR::UL },
            E { tag: 0x7005101E, alias: "Raw Data Number", vr: VR::UL },
            E { tag: 0x7005101F, alias: "Volume Number", vr: VR::LO },
            E { tag: 0x70051020, alias: "Local Series Number", vr: VR::UL },
            E { tag: 0x70051021, alias: "Decrease in Artifact Filter", vr: VR::LO },
            E { tag: 0x70051022, alias: "Reconstruction Interval", vr: VR::DS },
            E { tag: 0x70051023, alias: "Pitch Factor", vr: VR::DS },
            E { tag: 0x70051024, alias: "The Acquisition Date of NRA", vr: VR::DA },
            E { tag: 0x70051025, alias: "Large Data File Attribute", vr: VR::UL },
            E { tag: 0x70051026, alias: "Large Data File Name", vr: VR::CS },
            E { tag: 0x70051028, alias: "Enhanced CT Private Sequence", vr: VR::SQ },
            E { tag: 0x70051029, alias: "Frame UID", vr: VR::UI },
            E { tag: 0x70051030, alias: "Main Modality in Study", vr: VR::CS },
            E { tag: 0x70051035, alias: "Scan Range", vr: VR::DS },
            E { tag: 0x70051036, alias: "CT Private Data 3", vr: VR::OB },
            E { tag: 0x70051037, alias: "Total Frames", vr: VR::IS },
            E { tag: 0x70051038, alias: "Start Frame", vr: VR::IS },
            E { tag: 0x70051039, alias: "End Frame", vr: VR::IS },
            E { tag: 0x70051040, alias: "DLP Dose Length Product", vr: VR::FD },
            E { tag: 0x70051041, alias: "Raw Slice Information", vr: VR::SH },
            E { tag: 0x70051042, alias: "Local Frame No", vr: VR::US },
            E { tag: 0x70051043, alias: "Volume Vector", vr: VR::DS },
            E { tag: 0x70051044, alias: "Volume Type", vr: VR::US },
            E { tag: 0x70051045, alias: "Relative Table Position of 4D Volume", vr: VR::DS },
            E { tag: 0x70051046, alias: "Absolute Table Position of 4D Volume", vr: VR::DS },
            E { tag: 0x70051047, alias: "Slice Pitch of 4D Volume", vr: VR::DS },
            E { tag: 0x70051048, alias: "Respiratory Gating Inf.", vr: VR::LO },
            E { tag: 0x70051049, alias: "Respiration Phase", vr: VR::SH },
            E { tag: 0x700510F1, alias: "Protect Mark for Image, Curve or Private Record", vr: VR::CS },
            E { tag: 0x700510F2, alias: "Protect Mark for Series Record", vr: VR::CS },
            E { tag: 0x700510F3, alias: "Protect Mark for Study Record", vr: VR::CS },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_CT_1.0",
        entries: &[
            E { tag: 0x00191001, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191002, alias: "Unknown", vr: VR::IS },
            E { tag: 0x00191003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191004, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191005, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00191006, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191007, alias: "Unknown", vr: VR::US },
            E { tag: 0x00191008, alias: "Orientation Head Feet", vr: VR::LT },
            E { tag: 0x00191009, alias: "View Direction", vr: VR::LT },
            E { tag: 0x0019100A, alias: "Orientation Supine Prone", vr: VR::LT },
            E { tag: 0x0019100B, alias: "Location", vr: VR::DS },
            E { tag: 0x0019100C, alias: "Scan View Direction", vr: VR::CS },
            E { tag: 0x0019100D, alias: "Time", vr: VR::TM },
            E { tag: 0x0019100E, alias: "Unknown", vr: VR::DS },
            E { tag: 0x7FF11001, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11002, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11003, alias: "Unknown", vr: VR::IS },
            E { tag: 0x7FF11004, alias: "Unknown", vr: VR::IS },
            E { tag: 0x7FF11005, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11007, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11008, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF11009, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF1100A, alias: "Unknown", vr: VR::LT },
            E { tag: 0x7FF1100B, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF1100C, alias: "Unknown", vr: VR::US },
            E { tag: 0x7FF1100D, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_MR3",
        entries: &[
            E { tag: 0x00290089, alias: "Unknown", vr: VR::LO },
            E { tag: 0x700D1000, alias: "Scale Factor", vr: VR::DS },
            E { tag: 0x700D1001, alias: "Acquisition Order", vr: VR::OB },
            E { tag: 0x700D1002, alias: "Orientation Vector", vr: VR::DS },
            E { tag: 0x700D1003, alias: "Flip Flag", vr: VR::SS },
            E { tag: 0x700D1004, alias: "Rotate Information", vr: VR::OB },
            E { tag: 0x700D1005, alias: "FOV", vr: VR::DS },
            E { tag: 0x700D1006, alias: "Image Matrix", vr: VR::US },
            E { tag: 0x700D1007, alias: "Image Information", vr: VR::OB },
            E { tag: 0x700D1008, alias: "Original Data", vr: VR::OB },
            E { tag: 0x700D1009, alias: "Original Data Flag", vr: VR::SS },
            E { tag: 0x700D100C, alias: "Receiver Gain Correction Check Flag", vr: VR::CS },
            E { tag: 0x700D1020, alias: "Identification Flag of 3D GDC", vr: VR::SH },
            E { tag: 0x700D1080, alias: "Number of PAC channel", vr: VR::US },
            E { tag: 0x700D1081, alias: "Reference mode", vr: VR::US },
            E { tag: 0x700D1082, alias: "Gain value group for MRS", vr: VR::SQ },
            E { tag: 0x700D1083, alias: "Gain value of each channel for MRS", vr: VR::FL },
            E { tag: 0x700D1084, alias: "Phase value of each channel for MRS", vr: VR::FL },
            E { tag: 0x700D1085, alias: "Reference gain value of each channel for MRS", vr: VR::FL },
            E { tag: 0x700D1086, alias: "Reference gain value of each channel for MRS", vr: VR::FL },
            E { tag: 0x700D1087, alias: "Receiver gain of reference scan", vr: VR::FL },
            E { tag: 0x700D1088, alias: "Flag of water Sat pulse", vr: VR::UL },
            E { tag: 0x700D1089, alias: "Selected contrast TE", vr: VR::FL },
            E { tag: 0x700D108A, alias: "Raw Data Set Sequence", vr: VR::SQ },
            E { tag: 0x700D108B, alias: "Multi Slice number", vr: VR::DS },
            E { tag: 0x700D108C, alias: "Multi Coverage number", vr: VR::DS },
            E { tag: 0x700D108D, alias: "Raw Data Sequence", vr: VR::SQ },
            E { tag: 0x700D108E, alias: "Coil Channel No.", vr: VR::DS },
            E { tag: 0x700D108F, alias: "MRS Raw Data", vr: VR::OB },
            E { tag: 0x700D1090, alias: "MRS Ref Raw Data", vr: VR::OF },
            E { tag: 0x700D1091, alias: "Receiver gain of prescan", vr: VR::FL },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_MR3^10",
        entries: &[
            E { tag: 0x00090002, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00110003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110004, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00110005, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110006, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110007, alias: "Unknown", vr: VR::US },
            E { tag: 0x00110008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0011000C, alias: "Unknown", vr: VR::SS },
            E { tag: 0x0011000E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190015, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190017, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019001E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190020, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190021, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190022, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190023, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190024, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190025, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190027, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019002F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::OF },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190033, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190045, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190047, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190049, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019004F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190050, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190051, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190054, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019005E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190064, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190065, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190069, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019006B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190070, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190071, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190073, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190075, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007B, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190085, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190088, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019008D, alias: "Unknown", vr: VR::TM },
            E { tag: 0x0019008E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190096, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CA, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900CB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CC, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900D8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D9, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900DA, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900DC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EA, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900EB, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::FD },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::UI },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00210004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210008, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00210009, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021000A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0021000B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000C, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021000D, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0021000F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210012, alias: "Unknown", vr: VR::DS },
            E { tag: 0x00210014, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00210015, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210016, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00210018, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021001A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0021001D, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0021001F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210022, alias: "Unknown", vr: VR::US },
            E { tag: 0x00210023, alias: "Unknown", vr: VR::SS },
            E { tag: 0x00290001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290005, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00290006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290007, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0029001C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00290021, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0029004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00290050, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290051, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290052, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290054, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00290067, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00290068, alias: "Unknown", vr: VR::SS },
            E { tag: 0x0029006E, alias: "Unknown", vr: VR::OB },
            E { tag: 0x700D0010, alias: "2nd Flip Angle [degree]", vr: VR::DS },
            E { tag: 0x700D0011, alias: "Acquisition Inner Matrix", vr: VR::US },
            E { tag: 0x700D0012, alias: "MP2RAGE Flag", vr: VR::US },
            E { tag: 0x700D0013, alias: "Inversion efficiency of inversion recovery pulse", vr: VR::FL },
            E { tag: 0x700D0014, alias: "Number of dummy shot", vr: VR::SL },
            E { tag: 0x700D0015, alias: "FFE total repetition time[s]", vr: VR::FL },
            E { tag: 0x700D0016, alias: "PAS Name", vr: VR::LO },
            E { tag: 0x700D0017, alias: "Intended Processing", vr: VR::LT },
            E { tag: 0x700D0018, alias: "Scanned Orientation IDs", vr: VR::SS },
            E { tag: 0x700D0019, alias: "PAS Reproduct Information", vr: VR::OB },
            E { tag: 0x700D0080, alias: "Unknown", vr: VR::US },
            E { tag: 0xE2010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE3010000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010001, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010002, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010003, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010004, alias: "Unknown", vr: VR::US },
            E { tag: 0xE4010005, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010007, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010008, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010012, alias: "Unknown", vr: VR::FL },
            E { tag: 0xE4010013, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4010020, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010021, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xE4010022, alias: "Unknown", vr: VR::LO },
            E { tag: 0xE4010023, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4010024, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SQ },
            E { tag: 0xEE010000, alias: "Unknown", vr: VR::US },
            E { tag: 0xEE010001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xEE010002, alias: "Unknown", vr: VR::US },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_MR3^11",
        entries: &[
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::OB },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190024, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190025, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190026, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190029, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019002C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019002E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190030, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190031, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190035, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190036, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190037, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190038, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190039, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003A, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019003F, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190040, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190041, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190042, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190043, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190044, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190046, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190048, alias: "Unknown", vr: VR::LT },
            E { tag: 0x00190049, alias: "Unknown", vr: VR::LT },
            E { tag: 0x0019004C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019004E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190055, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190057, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190058, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190059, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019005C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019005F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190060, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190061, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190062, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190063, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190067, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190068, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019006C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019006D, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019006E, alias: "Unknown", vr: VR::SH },
            E { tag: 0x0019006F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190070, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190071, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190072, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190073, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190077, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190078, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019007D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019007F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190080, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190083, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190084, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008C, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190091, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190092, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190094, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B3, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900B7, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900B9, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900BB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BC, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900BE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900BF, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C1, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900E5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900E6, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900F3, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F5, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x001900F6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F7, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F8, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900F9, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FF, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_MR3^12",
        entries: &[
            E { tag: 0x00190000, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190001, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190002, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190006, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190007, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::UL },
            E { tag: 0x0019000D, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019000E, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190010, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190011, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190012, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190013, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190018, alias: "Unknown", vr: VR::UL },
            E { tag: 0x00190028, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019002B, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190032, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190034, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190052, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190076, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190079, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190080, alias: "Unknown", vr: VR::FD },
            E { tag: 0x00190085, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190086, alias: "Unknown", vr: VR::LO },
            E { tag: 0x00190087, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x00190088, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190089, alias: "Unknown", vr: VR::SL },
            E { tag: 0x0019008A, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019008F, alias: "Unknown", vr: VR::OB },
            E { tag: 0x00190093, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190094, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190095, alias: "Unknown", vr: VR::UI },
            E { tag: 0x00190098, alias: "Unknown", vr: VR::FL },
            E { tag: 0x00190099, alias: "Unknown", vr: VR::LO },
            E { tag: 0x0019009A, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019009B, alias: "Unknown", vr: VR::IS },
            E { tag: 0x0019009C, alias: "Unknown", vr: VR::DS },
            E { tag: 0x0019009D, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019009E, alias: "Unknown", vr: VR::FL },
            E { tag: 0x0019009F, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A0, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A4, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900A8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AA, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AB, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AD, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AE, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900AF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900B3, alias: "Unknown", vr: VR::FD },
            E { tag: 0x001900B4, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900C2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900C3, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900C4, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C5, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C6, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900C7, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900CD, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900CE, alias: "Unknown", vr: VR::UL },
            E { tag: 0x001900CF, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D2, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D3, alias: "Unknown", vr: VR::SS },
            E { tag: 0x001900D4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D5, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D6, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900D7, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900D8, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900DE, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900DF, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E0, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900E1, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E2, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900E3, alias: "Unknown", vr: VR::TM },
            E { tag: 0x001900E4, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EB, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EC, alias: "Unknown", vr: VR::US },
            E { tag: 0x001900EF, alias: "Unknown", vr: VR::DS },
            E { tag: 0x001900F0, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900F1, alias: "Unknown", vr: VR::LO },
            E { tag: 0x001900FA, alias: "Unknown", vr: VR::LT },
            E { tag: 0x001900FC, alias: "Unknown", vr: VR::FL },
            E { tag: 0x001900FD, alias: "Unknown", vr: VR::SL },
            E { tag: 0x001900FE, alias: "Unknown", vr: VR::SL },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_MR3^13",
        entries: &[
            E { tag: 0x00190002, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190003, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190004, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190005, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190008, alias: "Unknown", vr: VR::SL },
            E { tag: 0x00190009, alias: "Unknown", vr: VR::SQ },
            E { tag: 0x0019000A, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000B, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000C, alias: "Unknown", vr: VR::FD },
            E { tag: 0x0019000F, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190014, alias: "Unknown", vr: VR::SH },
            E { tag: 0x00190019, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001B, alias: "Unknown", vr: VR::US },
            E { tag: 0x0019001F, alias: "Unknown", vr: VR::US },
            E { tag: 0x00190022, alias: "Unknown", vr: VR::SL },
            E { tag: 0xE4030000, alias: "Unknown", vr: VR::SH },
            E { tag: 0xE4030001, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030002, alias: "Unknown", vr: VR::OB },
            E { tag: 0xE4030003, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_OT3",
        entries: &[
            E { tag: 0x00091000, alias: "HIS/RIS Study ID", vr: VR::LO },
            E { tag: 0x70191071, alias: "Indicator of creator", vr: VR::CS },
            E { tag: 0x70191072, alias: "ID of quality assurance reviewer", vr: VR::LO },
            E { tag: 0x70191073, alias: "Name of quality assurance reviewer", vr: VR::PN },
            E { tag: 0x70191074, alias: "Date of completion of quality assurance", vr: VR::DA },
            E { tag: 0x70191075, alias: "Time of completion of quality assurance", vr: VR::TM },
            E { tag: 0x70191076, alias: "Host name of quality assurance station", vr: VR::LO },
            E { tag: 0x70191077, alias: "Station name and version of quality assurance station", vr: VR::CS },
            E { tag: 0x70191078, alias: "Flag of completion of quality assurance", vr: VR::CS },
            E { tag: 0x70191079, alias: "Status of quality assurance review", vr: VR::CS },
            E { tag: 0x7019107A, alias: "Original SOP Instance UID", vr: VR::UI },
            E { tag: 0x7019107B, alias: "Original Series Instance UID", vr: VR::UI },
            E { tag: 0x7019107C, alias: "Original Study Instance UID", vr: VR::UI },
            E { tag: 0x7019107D, alias: "Rotation information for image display from original image", vr: VR::US },
            E { tag: 0x7019107E, alias: "Sequence of LR mark", vr: VR::SQ },
            E { tag: 0x7019107F, alias: "Rotation status at creation of LR mark", vr: VR::US },
            E { tag: 0x70191080, alias: "Original image data on area of original LR mark", vr: VR::UN },
            E { tag: 0x70191081, alias: "X start position of LR mark", vr: VR::SS },
            E { tag: 0x70191082, alias: "Y start position of LR mark", vr: VR::SS },
            E { tag: 0x70191083, alias: "Size of LR mark in X direction", vr: VR::US },
            E { tag: 0x70191084, alias: "Size of LR mark in Y direction", vr: VR::US },
            E { tag: 0x70191085, alias: "Previous Patient ID", vr: VR::LO },
            E { tag: 0x70191086, alias: "Previous modality", vr: VR::CS },
        ],
    },
    P {
        creator: "TOSHIBA_MEC_XA3",
        entries: &[
            E { tag: 0x70791010, alias: "", vr: VR::LO },
            E { tag: 0x70791021, alias: "", vr: VR::SH },
            E { tag: 0x70791022, alias: "", vr: VR::IS },
            E { tag: 0x70791023, alias: "", vr: VR::IS },
            E { tag: 0x70791024, alias: "", vr: VR::DS },
            E { tag: 0x70791025, alias: "", vr: VR::SH },
            E { tag: 0x70791026, alias: "", vr: VR::DS },
            E { tag: 0x70791027, alias: "", vr: VR::US },
            E { tag: 0x70791028, alias: "", vr: VR::US },
            E { tag: 0x70791029, alias: "", vr: VR::US },
            E { tag: 0x7079102A, alias: "", vr: VR::US },
            E { tag: 0x7079102B, alias: "", vr: VR::UN },
            E { tag: 0x7079102C, alias: "", vr: VR::SH },
            E { tag: 0x7079102D, alias: "", vr: VR::SS },
            E { tag: 0x7079102E, alias: "", vr: VR::SH },
            E { tag: 0x7079102F, alias: "", vr: VR::SH },
            E { tag: 0x70791030, alias: "", vr: VR::US },
            E { tag: 0x70791031, alias: "", vr: VR::US },
            E { tag: 0x70791032, alias: "", vr: VR::SH },
            E { tag: 0x70791033, alias: "", vr: VR::US },
            E { tag: 0x70791034, alias: "", vr: VR::SH },
            E { tag: 0x70791035, alias: "", vr: VR::SH },
            E { tag: 0x70791036, alias: "", vr: VR::SH },
            E { tag: 0x70791037, alias: "", vr: VR::US },
            E { tag: 0x70791038, alias: "", vr: VR::SS },
            E { tag: 0x70791039, alias: "", vr: VR::SS },
            E { tag: 0x7079103A, alias: "", vr: VR::US },
            E { tag: 0x7079103B, alias: "", vr: VR::SH },
            E { tag: 0x7079103C, alias: "", vr: VR::DS },
            E { tag: 0x7079103D, alias: "", vr: VR::SS },
            E { tag: 0x7079103E, alias: "", vr: VR::SH },
            E { tag: 0x7079103F, alias: "", vr: VR::US },
            E { tag: 0x70791040, alias: "", vr: VR::SH },
            E { tag: 0x70791041, alias: "", vr: VR::SS },
            E { tag: 0x70791042, alias: "", vr: VR::SH },
            E { tag: 0x70791043, alias: "", vr: VR::SH },
            E { tag: 0x70791044, alias: "", vr: VR::US },
            E { tag: 0x70791045, alias: "", vr: VR::SH },
            E { tag: 0x70791046, alias: "", vr: VR::SS },
            E { tag: 0x70791047, alias: "", vr: VR::SS },
            E { tag: 0x70791048, alias: "", vr: VR::US },
            E { tag: 0x70791049, alias: "", vr: VR::US },
            E { tag: 0x7079104A, alias: "", vr: VR::US },
            E { tag: 0x7079104B, alias: "", vr: VR::LO },
            E { tag: 0x7079104C, alias: "", vr: VR::OB },
            E { tag: 0x7079104D, alias: "", vr: VR::LO },
            E { tag: 0x7079104E, alias: "", vr: VR::SL },
            E { tag: 0x7079104F, alias: "", vr: VR::SH },
            E { tag: 0x70791050, alias: "", vr: VR::US },
            E { tag: 0x70791051, alias: "", vr: VR::US },
            E { tag: 0x70791052, alias: "", vr: VR::US },
            E { tag: 0x70791053, alias: "", vr: VR::SH },
            E { tag: 0x70791054, alias: "", vr: VR::US },
            E { tag: 0x70791055, alias: "", vr: VR::UL },
            E { tag: 0x70791056, alias: "", vr: VR::US },
            E { tag: 0x70791057, alias: "", vr: VR::US },
            E { tag: 0x70791058, alias: "", vr: VR::US },
            E { tag: 0x70791059, alias: "", vr: VR::US },
            E { tag: 0x7079105A, alias: "", vr: VR::US },
            E { tag: 0x7079105B, alias: "", vr: VR::US },
            E { tag: 0x7079105C, alias: "", vr: VR::US },
            E { tag: 0x7079105D, alias: "", vr: VR::SH },
            E { tag: 0x7079105E, alias: "", vr: VR::US },
            E { tag: 0x7079105F, alias: "", vr: VR::SS },
            E { tag: 0x70791060, alias: "", vr: VR::US },
            E { tag: 0x70791061, alias: "", vr: VR::SS },
            E { tag: 0x70791062, alias: "", vr: VR::US },
            E { tag: 0x70791063, alias: "", vr: VR::US },
            E { tag: 0x70791064, alias: "", vr: VR::US },
            E { tag: 0x70791065, alias: "", vr: VR::US },
            E { tag: 0x70791066, alias: "", vr: VR::US },
            E { tag: 0x70791067, alias: "", vr: VR::US },
            E { tag: 0x70791068, alias: "", vr: VR::US },
            E { tag: 0x70791069, alias: "", vr: VR::US },
            E { tag: 0x7079106A, alias: "", vr: VR::US },
            E { tag: 0x7079106B, alias: "", vr: VR::US },
            E { tag: 0x7079106C, alias: "", vr: VR::US },
            E { tag: 0x7079106D, alias: "", vr: VR::DS },
            E { tag: 0x7079106E, alias: "", vr: VR::LO },
            E { tag: 0x7079106F, alias: "", vr: VR::DS },
            E { tag: 0x70791070, alias: "", vr: VR::DS },
            E { tag: 0x70791071, alias: "", vr: VR::DS },
            E { tag: 0x70791072, alias: "", vr: VR::DS },
            E { tag: 0x70791073, alias: "", vr: VR::SL },
            E { tag: 0x70791074, alias: "", vr: VR::SL },
            E { tag: 0x70791075, alias: "", vr: VR::SH },
            E { tag: 0x70791076, alias: "", vr: VR::US },
            E { tag: 0x70791077, alias: "", vr: VR::SL },
            E { tag: 0x70791078, alias: "", vr: VR::US },
            E { tag: 0x70791079, alias: "", vr: VR::US },
            E { tag: 0x7079107B, alias: "Unknown", vr: VR::SH },
            E { tag: 0x70791080, alias: "", vr: VR::LO },
            E { tag: 0x70791081, alias: "", vr: VR::SS },
            E { tag: 0x70791082, alias: "", vr: VR::SS },
        ],
    },
    P {
        creator: "TOSHIBA_SR",
        entries: &[
            E { tag: 0x70151010, alias: "Unknown", vr: VR::OB },
            E { tag: 0x70151060, alias: "Unknown", vr: VR::OB },
        ],
    },
    P {
        creator: "Thinking Systems Private Creator",
        entries: &[
            E { tag: 0x7FDD0042, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7FDD0097, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "V1",
        entries: &[
            E { tag: 0x00111001, alias: "User Data", vr: VR::OB },
            E { tag: 0x00111002, alias: "Normalization Coefficient", vr: VR::DS },
            E { tag: 0x00111003, alias: "Receiving Gain", vr: VR::DS },
            E { tag: 0x00111004, alias: "Mean Image Noise", vr: VR::DS },
        ],
    },
    P {
        creator: "VEPRO BROKER 1.0",
        entries: &[
            E { tag: 0x00571010, alias: "VEPRO Data replace sequence", vr: VR::SQ },
        ],
    },
    P {
        creator: "VEPRO BROKER 1.0 DATA REPLACE",
        entries: &[
            E { tag: 0x00571020, alias: "Original data sequence", vr: VR::SQ },
            E { tag: 0x00571030, alias: "Replaced data sequence", vr: VR::SQ },
            E { tag: 0x00571040, alias: "Date of data replacement", vr: VR::DA },
            E { tag: 0x00571041, alias: "Time of data replacement", vr: VR::TM },
            E { tag: 0x00571042, alias: "Dicom receive node", vr: VR::LO },
            E { tag: 0x00571043, alias: "Application name", vr: VR::LO },
            E { tag: 0x00571044, alias: "Computer name", vr: VR::LO },
        ],
    },
    P {
        creator: "VEPRO DICOM RECEIVE DATA 1.0",
        entries: &[
            E { tag: 0x00591040, alias: "Receive date", vr: VR::DA },
            E { tag: 0x00591041, alias: "Receive time", vr: VR::TM },
            E { tag: 0x00591042, alias: "Receive node", vr: VR::ST },
            E { tag: 0x00591043, alias: "Receive application", vr: VR::ST },
            E { tag: 0x00591050, alias: "Receive local computer", vr: VR::ST },
            E { tag: 0x00591051, alias: "Receive local ae title", vr: VR::ST },
            E { tag: 0x00591060, alias: "Receive remote computer", vr: VR::ST },
            E { tag: 0x00591061, alias: "Receive remote ae title", vr: VR::ST },
            E { tag: 0x00591070, alias: "Receive original transfer syntax", vr: VR::UI },
        ],
    },
    P {
        creator: "VEPRO DICOM TRANSFER 1.0",
        entries: &[
            E { tag: 0x00591010, alias: "Dicom Transfer Info", vr: VR::SQ },
        ],
    },
    P {
        creator: "VEPRO VIF 3.0 DATA",
        entries: &[
            E { tag: 0x00551030, alias: "Icon data", vr: VR::OB },
            E { tag: 0x00551065, alias: "Image hash value", vr: VR::OB },
        ],
    },
    P {
        creator: "VEPRO VIM 5.0 DATA",
        entries: &[
            E { tag: 0x00551010, alias: "VIFHEADER2", vr: VR::OB },
            E { tag: 0x00551020, alias: "VIMDATA2", vr: VR::OB },
            E { tag: 0x00551030, alias: "ICONDATA2", vr: VR::OB },
            E { tag: 0x00551051, alias: "VEPRO UID ??", vr: VR::UI },
            E { tag: 0x00551065, alias: "??", vr: VR::OB },
        ],
    },
    P {
        creator: "Varian Medical Systems VISION 3253",
        entries: &[
            E { tag: 0x32531000, alias: "XML stream.", vr: VR::UN },
            E { tag: 0x32531001, alias: "Data length of the XML stream", vr: VR::UN },
            E { tag: 0x32531002, alias: "Extended Interface Format tag", vr: VR::UN },
        ],
    },
    P {
        creator: "Viewing Protocol",
        entries: &[
            E { tag: 0x00651093, alias: "Unknown", vr: VR::CS },
        ],
    },
    P {
        creator: "Vital Images SW 3.4",
        entries: &[
            E { tag: 0x56531010, alias: "Saved Workflow", vr: VR::OB },
            E { tag: 0x56531011, alias: "Saved Workflow File Name", vr: VR::LO },
            E { tag: 0x56531012, alias: "Saved Workflow File Data", vr: VR::OB },
            E { tag: 0x56531013, alias: "Saved Workflow File Length", vr: VR::SL },
            E { tag: 0x56531014, alias: "Saved Workflow File Sequence", vr: VR::SQ },
            E { tag: 0x56531015, alias: "Image Sequence", vr: VR::SQ },
            E { tag: 0x56531016, alias: "Volume Interpolated Slices", vr: VR::SL },
            E { tag: 0x56531017, alias: "Volume SOP Instance UID", vr: VR::UI },
            E { tag: 0x56531018, alias: "Saved Workflow Mark", vr: VR::SH },
            E { tag: 0x56531019, alias: "Unknown", vr: VR::UI },
            E { tag: 0x56531022, alias: "Unknown", vr: VR::UI },
            E { tag: 0x56531023, alias: "Unknown", vr: VR::LO },
        ],
    },
    P {
        creator: "Voxar 2.16.124.113543.6003.1999.12.20.12.5.0",
        entries: &[
            E { tag: 0x11351000, alias: "Private Voxar Capture UID", vr: VR::UI },
            E { tag: 0x11351001, alias: "Private Voxar Saved State", vr: VR::OB },
            E { tag: 0x11351002, alias: "Private Voxar Icon Image Sequence", vr: VR::SQ },
            E { tag: 0x11351003, alias: "Private Voxar Top Right Display Demographics", vr: VR::ST },
            E { tag: 0x11351004, alias: "Private Voxar Bottom Left Display Demographics", vr: VR::ST },
            E { tag: 0x11351005, alias: "Private Voxar Span State Schema", vr: VR::IS },
            E { tag: 0x11351006, alias: "Private Voxar Live Image Restored Size", vr: VR::UL },
            E { tag: 0x11351007, alias: "Private Voxar CaptureType", vr: VR::UI },
            E { tag: 0x11351008, alias: "Private Voxar Capture Source Series UID Sequence", vr: VR::SQ },
            E { tag: 0x11351009, alias: "Private Voxar Capture Source Series UID", vr: VR::UI },
            E { tag: 0x11351010, alias: "Private Voxar Background Gray For Capture", vr: VR::IS },
            E { tag: 0x11351011, alias: "Private Voxar Calcium Scoring Report Schema", vr: VR::IS },
            E { tag: 0x11351012, alias: "Private Voxar Source Series Number", vr: VR::IS },
            E { tag: 0x11351013, alias: "Private Voxar Bottom Left Display Demographics", vr: VR::LO },
        ],
    },
    P {
        creator: "WG12 Supplement 43",
        entries: &[
            E { tag: 0x00091001, alias: "Event Timer Sequence", vr: VR::SQ },
            E { tag: 0x00091002, alias: "Event Time Interval", vr: VR::FD },
            E { tag: 0x00091003, alias: "Event Code Sequence", vr: VR::SQ },
            E { tag: 0x00191001, alias: "Focus Depth(s)", vr: VR::FD },
            E { tag: 0x00191003, alias: "Excluded Intervals Sequence", vr: VR::SQ },
            E { tag: 0x00191004, alias: "Exclusion Start Datetime", vr: VR::DT },
            E { tag: 0x00191005, alias: "Exclusion Duration", vr: VR::FD },
            E { tag: 0x00191006, alias: "US Image Description Sequence", vr: VR::SQ },
            E { tag: 0x00191007, alias: "Image Data Type Sequence", vr: VR::SQ },
            E { tag: 0x00191008, alias: "Data Type", vr: VR::CS },
            E { tag: 0x00191009, alias: "Transducer Scan Geometry Code Sequence", vr: VR::SQ },
            E { tag: 0x0019100B, alias: "Aliased Data Type", vr: VR::CS },
            E { tag: 0x0019100C, alias: "Position Measuring Device Used", vr: VR::CS },
            E { tag: 0x0019100D, alias: "Transducer Scanning Configuration Code Sequence", vr: VR::SQ },
            E { tag: 0x0019100E, alias: "Transducer Beam Steering Code Sequence", vr: VR::SQ },
            E { tag: 0x0019100F, alias: "Transducer Access Code Sequence", vr: VR::SQ },
            E { tag: 0x00211001, alias: "Image Position (Volume)", vr: VR::FD },
            E { tag: 0x00211002, alias: "Image Orientation (Volume)", vr: VR::FD },
            E { tag: 0x00211007, alias: "Ultrasound Acquisition Geometry", vr: VR::CS },
            E { tag: 0x00211008, alias: "Apex Position", vr: VR::FD },
            E { tag: 0x00211009, alias: "Volume to Transducer Mapping Matrix", vr: VR::FD },
            E { tag: 0x0021100A, alias: "Volume to Table Mapping Matrix", vr: VR::FD },
            E { tag: 0x0021100C, alias: "Patient Frame of Reference Source", vr: VR::CS },
            E { tag: 0x0021100D, alias: "Temporal Position Time Offset", vr: VR::FD },
            E { tag: 0x0021100E, alias: "Plane Position (Volume) Sequence", vr: VR::SQ },
            E { tag: 0x0021100F, alias: "Plane Orientation (Volume) Sequence", vr: VR::SQ },
            E { tag: 0x00211010, alias: "Temporal Position Sequence", vr: VR::SQ },
            E { tag: 0x00211011, alias: "Dimension Organization Type", vr: VR::CS },
            E { tag: 0x00291001, alias: "Data Frame Assignment Sequence", vr: VR::SQ },
            E { tag: 0x00291002, alias: "Data Path Assignment", vr: VR::CS },
            E { tag: 0x00291003, alias: "Bits Mapped to Color Lookup Table", vr: VR::US },
            E { tag: 0x00291004, alias: "Opacity 1 LUT Sequence", vr: VR::SQ },
            E { tag: 0x00291005, alias: "Opacity 1 LUT Transfer Function", vr: VR::CS },
            E { tag: 0x00291006, alias: "Opacity Constant", vr: VR::FD },
            E { tag: 0x00291007, alias: "Opacity Lookup Table Descriptor", vr: VR::US },
            E { tag: 0x00291008, alias: "Opacity Lookup Table Data", vr: VR::OW },
            E { tag: 0x0029100B, alias: "Enhanced Palette Color Lookup Table Sequence", vr: VR::SQ },
            E { tag: 0x0029100C, alias: "Opacity 2 LUT Sequence", vr: VR::SQ },
            E { tag: 0x0029100D, alias: "Opacity 2 LUT Transfer Function", vr: VR::CS },
            E { tag: 0x0029100E, alias: "Data Path ID", vr: VR::CS },
            E { tag: 0x0029100F, alias: "RGB LUT Transfer Function", vr: VR::CS },
            E { tag: 0x00291010, alias: "Alpha LUT Transfer Function", vr: VR::CS },
            E { tag: 0x00411001, alias: "Performed Protocol Type", vr: VR::CS },
        ],
    },
    P {
        creator: "agfa/originalStudyUID",
        entries: &[
            E { tag: 0x7FDD0013, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "agfa/seriesHeader",
        entries: &[
            E { tag: 0x7FDB0012, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7FDB0053, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7FDB0054, alias: "Unknown", vr: VR::UN },
            E { tag: 0x7FDB0055, alias: "Unknown", vr: VR::UN },
        ],
    },
    P {
        creator: "http://www.gemedicalsystems.com/it_solutions/bamwallthickness/1.0",
        entries: &[
            E { tag: 0x31191010, alias: "BAM WallThickness Session Date/Time", vr: VR::DT },
            E { tag: 0x31191020, alias: "BAM WallThickness Session Creator", vr: VR::PN },
            E { tag: 0x31191030, alias: "BAM WallThickness Session Completion Flag", vr: VR::CS },
            E { tag: 0x31191040, alias: "BAM WallThickness File Sequence", vr: VR::SQ },
            E { tag: 0x31191050, alias: "BAM WallThickness File Name", vr: VR::ST },
            E { tag: 0x31191060, alias: "BAM WallThickness File Content", vr: VR::OB },
        ],
    },
    P {
        creator: "http://www.gemedicalsystems.com/it_solutions/orthoview/2.1",
        entries: &[
            E { tag: 0x31171010, alias: "OrthoView Session Date/Time", vr: VR::DT },
            E { tag: 0x31171020, alias: "OrthoView Session Creator", vr: VR::PN },
            E { tag: 0x31171030, alias: "OrthoView Session Completion Flag", vr: VR::CS },
            E { tag: 0x31171040, alias: "OrthoView File Sequence", vr: VR::SQ },
            E { tag: 0x31171050, alias: "OrthoView File Name", vr: VR::ST },
            E { tag: 0x31171060, alias: "OrthoView File Content", vr: VR::OB },
        ],
    },
    P {
        creator: "http://www.gemedicalsystems.com/it_solutions/rad_pacs/",
        entries: &[
            E { tag: 0x31151001, alias: "Reference to pacs study", vr: VR::UT },
            E { tag: 0x31151002, alias: "Reference to pacs image", vr: VR::UT },
            E { tag: 0x31151003, alias: "Pacs examnotes flag", vr: VR::CS },
        ],
    },
    P {
        creator: "syngoDynamics",
        entries: &[
            E { tag: 0x002110AE, alias: "Unknown", vr: VR::OB },
        ],
    },];
