
import json
import zipfile

# NOTE: The `PDF` class provides a thin wrappers to data fetched from the
# API by the API client and should not be initialized directly.


class _BaseResource:
    """
    A base resource used to wrap documents fetched from the API with dot
    notation access to attributes and methods for access to related API
    endpoints.
    """

    def __init__(self, client, document):

        # The API client used to fetch the resource
        self._client = client

        # The document representing the resource's data
        self._document = document

    def __getattr__(self, name):

        if '_document' in self.__dict__:
            return self.__dict__['_document'].get(name, None)

        raise AttributeError(
            f"'{self.__class__.__name__}' has no attribute '{name}'"
        )

    def __getitem__(self, name):
        return self.__dict__['_document'][name]

    def __contains__(self, name):
        return name in self.__dict__['_document']

    def get(self, name, default=None):
        return self.__dict__['_document'].get(name, default)


class PDF(_BaseResource):
    """
    A PDF generated by DInk (specifically the store key and UID for the PDF).
    """

    def __str__(self):
        return f'PDF: {self.store_key}'

    @classmethod
    def create(cls,
        client,
        template_html,
        document_args,
        global_vars=None,
        assets=None
    ):
        """
        Create one or more PDFs. Returns a map of created PDFs with each key
        in the document args being assigned a PDF.

        NOTE: If there was an error generating a PDF then the PDF will not
        feature `store_key` and `uid` attributes but an `error` attribute.
        """

        if isinstance(assets, zipfile.ZipFile):

            # Convert the assets archive to a file we can send to the server
            f = io.BytesIO()
            assets.write(f)
            f.seek(0)
            asset = f

        results = client(
            'put',
            f'pdfs',
            files={'assets': assets} if assets else None,
            data={
                'template_html': template_html,
                'document_args': json.dumps(document_args),
                'global_vars': json.dumps(global_vars) \
                        if global_vars else None
            }
        )

        return {k: cls(client, v) for k, v in results.items()}
