import io
import json

import requests

from . import exceptions

__all__ = ['Client']


class Client:
    """
    A client for the DInk API.
    """

    def __init__(self,
        api_key,
        api_base_url='https://api.dink.run',
        timeout=None
    ):

        # A key used to authenticate API calls to an account
        self._api_key = api_key

        # The base URL to use when calling the API
        self._api_base_url = api_base_url

        # The period of time before requests to the API should timeout
        self._timeout = timeout

        # NOTE: Rate limiting information is only available after a request
        # has been made.

        # The maximum number of requests per second that can be made with the
        # given API key.
        self._rate_limit = None

        # The time (seconds since epoch) when the current rate limit will
        # reset.
        self._rate_limit_reset = None

        # The number of requests remaining within the current limit before the
        # next reset.
        self._rate_limit_remaining = None

    @property
    def rate_limit(self):
        return self._rate_limit

    @property
    def rate_limit_reset(self):
        return self._rate_limit_reset

    @property
    def rate_limit_remaining(self):
        return self._rate_limit_remaining

    def __call__(self,
        method,
        path,
        params=None,
        data=None,
        json_type_body=None,
        files=None,
        download=False
    ):
        """Call the API"""

        # Build headers
        headers = {'X-DInk-APIKey': self._api_key}

        if json_type_body:
            headers['Content-Type'] = 'application/json'

        if not download:
            headers['Accept'] = 'application/json'

        if params:
            # Filter out parameters set to `None`
            params = {k: v for k, v in params.items() if v is not None}

        # Make the request
        r = getattr(requests, method.lower())(
            f'{self._api_base_url}/{path}',
            headers=headers,
            params=params,
            data=data,
            json=json_type_body,
            files=files,
            timeout=self._timeout
        )

        # Update the rate limit
        if 'X-DInk-RateLimit-Limit' in r.headers:
            self._rate_limit = int(r.headers['X-DInk-RateLimit-Limit'])
            self._rate_limit_reset \
                    = float(r.headers['X-DInk-RateLimit-Reset'])
            self._rate_limit_remaining \
                    = int(r.headers['X-DInk-RateLimit-Remaining'])

        # Handle a successful response
        if r.status_code in [200, 204]:

            if download:
                return io.BytesIO(r.content)

            if r.headers.get('Content-Type', '')\
                    .startswith('application/json'):

                return r.json()

            return None

        # Raise an error related to the response
        try:
            error = r.json()

        except json.decoder.JSONDecodeError:
            error = {}

        if not isinstance(error, dict):
            error = {}

        error_cls = exceptions.DInkException.get_class_by_status_code(
            r.status_code
        )

        raise error_cls(
            r.status_code,
            error.get('hint'),
            error.get('arg_errors')
        )


