"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleJwksCache = exports.SimplePenaltyBox = exports.isJwk = exports.isJwks = exports.assertIsJwk = exports.assertIsJwks = exports.fetchJwk = exports.fetchJwks = void 0;
const error_1 = require("./error");
const https_1 = require("./https");
const safe_json_parse_1 = require("./safe-json-parse");
const optionalJwkFieldNames = [
    'alg',
];
const mandatoryJwkFieldNames = [
    'e',
    'kid',
    'kty',
    'n',
    'use',
];
async function fetchJwks(jwksUri) {
    const jwks = await https_1.fetchJson(jwksUri);
    assertIsJwks(jwks);
    return jwks;
}
exports.fetchJwks = fetchJwks;
async function fetchJwk(jwksUri, decomposedJwt) {
    if (!decomposedJwt.header.kid) {
        throw new error_1.JwtWithoutValidKidError('JWT header does not have valid kid claim');
    }
    const jwk = (await fetchJwks(jwksUri)).keys.find((key) => key.kid === decomposedJwt.header.kid);
    if (!jwk) {
        throw new error_1.KidNotFoundInJwksError(`JWK for kid "${decomposedJwt.header.kid}" not found in the JWKS`);
    }
    return jwk;
}
exports.fetchJwk = fetchJwk;
function assertIsJwks(jwks) {
    if (!jwks) {
        throw new error_1.JwksValidationError('JWKS empty');
    }
    if (!safe_json_parse_1.isJsonObject(jwks)) {
        throw new error_1.JwksValidationError('JWKS should be an object');
    }
    if (!Object.keys(jwks).includes('keys')) {
        throw new error_1.JwksValidationError('JWKS does not include keys');
    }
    if (!Array.isArray(jwks.keys)) {
        throw new error_1.JwksValidationError('JWKS keys should be an array');
    }
    for (const jwk of jwks.keys) {
        assertIsJwk(jwk);
    }
}
exports.assertIsJwks = assertIsJwks;
function assertIsJwk(jwk) {
    if (!jwk) {
        throw new error_1.JwkValidationError('JWK empty');
    }
    if (!safe_json_parse_1.isJsonObject(jwk)) {
        throw new error_1.JwkValidationError('JWK should be an object');
    }
    for (const field of mandatoryJwkFieldNames) {
        // disable eslint rule because `field` is trusted
        if (typeof jwk[field] !== 'string') {
            throw new error_1.JwkValidationError(`JWK ${field} should be a string`);
        }
    }
    for (const field of optionalJwkFieldNames) {
        // disable eslint rule because `field` is trusted
        if (field in jwk && typeof jwk[field] !== 'string') {
            throw new error_1.JwkValidationError(`JWK ${field} should be a string`);
        }
    }
}
exports.assertIsJwk = assertIsJwk;
function isJwks(jwks) {
    try {
        assertIsJwks(jwks);
        return true;
    }
    catch {
        return false;
    }
}
exports.isJwks = isJwks;
function isJwk(jwk) {
    try {
        assertIsJwk(jwk);
        return true;
    }
    catch {
        return false;
    }
}
exports.isJwk = isJwk;
class SimplePenaltyBox {
    constructor(props) {
        var _a;
        this.waitingUris = new Map();
        this.waitSeconds = (_a = props === null || props === void 0 ? void 0 : props.waitSeconds) !== null && _a !== void 0 ? _a : 10;
    }
    async wait(jwksUri) {
        // SimplePenaltyBox does not actually wait but bluntly throws an error
        // Any waiting and retries are expected to be done upstream (e.g. in the browser / app)
        if (this.waitingUris.has(jwksUri)) {
            throw new error_1.WaitPeriodNotYetEndedJwkError('Not allowed to fetch JWKS yet, still waiting for back off period to end');
        }
    }
    release(jwksUri) {
        const i = this.waitingUris.get(jwksUri);
        if (i) {
            clearTimeout(i);
            this.waitingUris.delete(jwksUri);
        }
    }
    registerFailedAttempt(jwksUri) {
        const i = setTimeout(() => {
            this.waitingUris.delete(jwksUri);
        }, this.waitSeconds * 1000).unref();
        this.waitingUris.set(jwksUri, i);
    }
    registerSuccessfulAttempt(jwksUri) {
        this.release(jwksUri);
    }
}
exports.SimplePenaltyBox = SimplePenaltyBox;
class SimpleJwksCache {
    constructor(props) {
        var _a, _b;
        this.jwksCache = new Map();
        this.fetchingJwks = new Map();
        this.penaltyBox = (_a = props === null || props === void 0 ? void 0 : props.penaltyBox) !== null && _a !== void 0 ? _a : new SimplePenaltyBox();
        this.fetcher = (_b = props === null || props === void 0 ? void 0 : props.fetcher) !== null && _b !== void 0 ? _b : new https_1.SimpleJsonFetcher();
    }
    addJwks(jwksUri, jwks) {
        this.jwksCache.set(jwksUri, jwks);
    }
    async getJwks(jwksUri) {
        const existingFetch = this.fetchingJwks.get(jwksUri);
        if (existingFetch) {
            return existingFetch;
        }
        const jwksPromise = this.fetcher.fetch(jwksUri).then((res) => {
            assertIsJwks(res);
            return res;
        });
        this.fetchingJwks.set(jwksUri, jwksPromise);
        let jwks;
        try {
            jwks = await jwksPromise;
        }
        finally {
            this.fetchingJwks.delete(jwksUri);
        }
        this.jwksCache.set(jwksUri, jwks);
        return jwks;
    }
    getCachedJwk(jwksUri, decomposedJwt) {
        if (typeof decomposedJwt.header.kid !== 'string') {
            throw new error_1.JwtWithoutValidKidError('JWT header does not have valid kid claim');
        }
        if (!this.jwksCache.has(jwksUri)) {
            throw new error_1.JwksNotAvailableInCacheError(`JWKS for uri ${jwksUri} not yet available in cache`);
        }
        const jwk = this.jwksCache
            .get(jwksUri)
            .keys.find((key) => key.kid === decomposedJwt.header.kid);
        if (!jwk) {
            throw new error_1.KidNotFoundInJwksError(`JWK for kid ${decomposedJwt.header.kid} not found in the JWKS`);
        }
        return jwk;
    }
    async getJwk(jwksUri, decomposedJwt) {
        var _a;
        if (typeof decomposedJwt.header.kid !== 'string') {
            throw new error_1.JwtWithoutValidKidError('JWT header does not have valid kid claim');
        }
        // Try to get JWK from cache:
        let jwk = (_a = this.jwksCache
            .get(jwksUri)) === null || _a === void 0 ? void 0 : _a.keys.find((key) => key.kid === decomposedJwt.header.kid);
        if (jwk) {
            return jwk;
        }
        // Await any wait period that is currently in effect
        // This prevents us from flooding the JWKS URI with requests
        await this.penaltyBox.wait(jwksUri, decomposedJwt.header.kid);
        // Fetch the JWKS and (try to) locate the JWK
        const jwks = await this.getJwks(jwksUri);
        jwk = jwks.keys.find((key) => key.kid === decomposedJwt.header.kid);
        // If the JWK could not be located, someone might be messing around with us
        // Register the failed attempt with the penaltyBox, so it can enforce a wait period
        // before trying again next time (instead of flooding the JWKS URI with requests)
        if (!jwk) {
            this.penaltyBox.registerFailedAttempt(jwksUri, decomposedJwt.header.kid);
            throw new error_1.KidNotFoundInJwksError(`JWK for kid "${decomposedJwt.header.kid}" not found in the JWKS`);
        }
        else {
            this.penaltyBox.registerSuccessfulAttempt(jwksUri, decomposedJwt.header.kid);
        }
        return jwk;
    }
}
exports.SimpleJwksCache = SimpleJwksCache;
//# sourceMappingURL=data:application/json;base64,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