"""
Generator dla WebSocket API
"""

import os
from pathlib import Path
from typing import Dict, Any, Set
from jinja2 import Environment

from ..core.analyzer import ApiSpec
from ..utils.file_utils import FileManager


class WebSocketGenerator:
    """Generator kodu WebSocket"""

    def __init__(self):
        self.file_manager = FileManager()
        self.templates = self._load_templates()

    def _load_templates(self) -> Dict[str, str]:
        """Ładuje szablony Jinja2 dla WebSocket"""
        server_template = """# WebSocket server generated by text2api

import asyncio
import json
import logging
import os
import signal
import sys
from datetime import datetime
from typing import Dict, Set, Any, Optional

import websockets
from websockets.server import WebSocketServerProtocol

{% if api_spec.database_required %}
import sqlite3
import threading
{% endif %}


class {{ api_spec.name|title }}WebSocketServer:
    # {{ api_spec.description }}

    def __init__(self, host: str = "0.0.0.0", port: int = 8765):
        self.host = host
        self.port = port
        self.clients: Set[WebSocketServerProtocol] = set()
        self.rooms: Dict[str, Set[WebSocketServerProtocol]] = {}
        self.server = None
        
        # Setup logging
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            handlers=[
                logging.StreamHandler(),
                logging.FileHandler('{{ api_spec.name|lower }}_server.log')
            ]
        )
        self.logger = logging.getLogger('{{ api_spec.name }}WebSocketServer')
        
        # Initialize database if needed
        {% if api_spec.database_required %}
        self.db_path = os.getenv('DATABASE_URL', '{{ api_spec.name|lower }}.db')
        self.db_lock = threading.Lock()
        self._init_database()
        {% endif %}
        
        # Register signal handlers for graceful shutdown
        signal.signal(signal.SIGINT, self._handle_shutdown)
        signal.signal(signal.SIGTERM, self._handle_shutdown)
        
        self.logger.info("{{ api_spec.name|title }} WebSocket Server initialized")
    
    {% if api_spec.database_required %}
    def _init_database(self):
        # Initialize database tables
        with self.db_lock:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            # Create tables for each model
            {% for model in api_spec.models %}
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS {{ model.name|lower }} (
                    id TEXT PRIMARY KEY,
                    {% for field in model.fields %}
                    {{ field.name }} {% if field.type == 'string' %}TEXT{% elif field.type == 'integer' %}INTEGER{% else %}TEXT{% endif %}{% if not loop.last %},{% endif %}
                    {% endfor %}
                    created_at TEXT,
                    updated_at TEXT
                )
            ''')
            {% endfor %}
            
            conn.commit()
            conn.close()
    
    def _get_db_connection(self):
        return sqlite3.connect(self.db_path)
    {% endif %}
    
    async def register(self, websocket: WebSocketServerProtocol):
        # Register a new client connection
        self.clients.add(websocket)
        self.logger.info(f"New client connected: {websocket.remote_address}")
    
    async def unregister(self, websocket: WebSocketServerProtocol):
        # Unregister a client connection
        if websocket in self.clients:
            self.clients.remove(websocket)
            
            # Remove from all rooms
            for room, clients in list(self.rooms.items()):
                if websocket in clients:
                    clients.remove(websocket)
                    self.logger.info(f"Client {websocket.remote_address} left room '{room}'")
                    
                    # Remove empty rooms
                    if not clients:
                        del self.rooms[room]
            
            self.logger.info(f"Client disconnected: {websocket.remote_address}")
    
    async def join_room(self, websocket: WebSocketServerProtocol, room: str):
        # Add client to a room
        if room not in self.rooms:
            self.rooms[room] = set()
        self.rooms[room].add(websocket)
        self.logger.info(f"Client {websocket.remote_address} joined room '{room}'")
    
    async def leave_room(self, websocket: WebSocketServerProtocol, room: str):
        # Remove client from a room
        if room in self.rooms and websocket in self.rooms[room]:
            self.rooms[room].remove(websocket)
            self.logger.info(f"Client {websocket.remote_address} left room '{room}'")
            
            # Remove empty rooms
            if not self.rooms[room]:
                del self.rooms[room]
    
    async def broadcast(self, message: str, room: Optional[str] = None):
        # Broadcast a message to all clients or clients in a specific room
        targets = self.rooms.get(room, set()) if room else self.clients
        
        if targets:
            await asyncio.wait([client.send(message) for client in targets])
    
    async def handle_message(self, websocket: WebSocketServerProtocol, message: str):
        # Handle incoming WebSocket messages
        try:
            data = json.loads(message)
            action = data.get('action')
            
            if action == 'join_room':
                await self.join_room(websocket, data['room'])
            elif action == 'leave_room':
                await self.leave_room(websocket, data.get('room', ''))
            elif action == 'broadcast':
                await self.broadcast(json.dumps({
                    'type': 'broadcast',
                    'from': str(websocket.remote_address),
                    'message': data.get('message', '')
                }), data.get('room'))
            else:
                await websocket.send(json.dumps({
                    'type': 'error',
                    'message': f'Unknown action: {action}'
                }))
                
        except json.JSONDecodeError:
            await websocket.send(json.dumps({
                'type': 'error',
                'message': 'Invalid JSON format'
            }))
        except Exception as e:
            self.logger.error(f"Error handling message: {str(e)}", exc_info=True)
            await websocket.send(json.dumps({
                'type': 'error',
                'message': 'Internal server error'
            }))
    
    async def handle_client(self, websocket: WebSocketServerProtocol, path: str):
        # Handle a new WebSocket connection
        await self.register(websocket)
        try:
            async for message in websocket:
                await self.handle_message(websocket, message)
        except websockets.exceptions.ConnectionClosed:
            pass
        finally:
            await self.unregister(websocket)
    
    async def start(self):
        # Start the WebSocket server
        self.server = await websockets.serve(
            self.handle_client,
            self.host,
            self.port,
            ping_interval=30,
            ping_timeout=10,
            close_timeout=5
        )
        
        self.logger.info(f"WebSocket server started on ws://{self.host}:{self.port}")
        return self.server
    
    async def stop(self):
        # Stop the WebSocket server
        if self.server:
            self.server.close()
            await self.server.wait_closed()
            self.logger.info("WebSocket server stopped")
    
    def _handle_shutdown(self, signum, frame):
        # Handle shutdown signals
        self.logger.info("Shutdown signal received. Stopping server...")
        asyncio.create_task(self.stop())


def main():
    # Parse command line arguments
    import argparse
    
    parser = argparse.ArgumentParser(description='{{ api_spec.name }} WebSocket Server')
    parser.add_argument('--host', default='0.0.0.0', help='Host to bind to')
    parser.add_argument('--port', type=int, default=8765, help='Port to listen on')
    args = parser.parse_args()
    
    # Create and start server
    server = {{ api_spec.name|title }}WebSocketServer(host=args.host, port=args.port)
    
    try:
        asyncio.get_event_loop().run_until_complete(server.start())
        asyncio.get_event_loop().run_forever()
    except KeyboardInterrupt:
        asyncio.get_event_loop().run_until_complete(server.stop())
    finally:
        asyncio.get_event_loop().close()


if __name__ == '__main__':
    main()
"""

        requirements_txt = """websockets>=10.0
{% if api_spec.database_required %}
sqlite3>=2.6.0
{% endif %}
"""

        readme_md = """# {{ api_spec.name }} WebSocket Server

This is an auto-generated WebSocket server for {{ api_spec.name }}.

## Setup

1. Install dependencies:
   ```bash
   pip install -r requirements.txt
   ```

2. Run the server:
   ```bash
   python server.py
   ```

## API

### Connecting

Connect to the WebSocket server at `ws://localhost:8765`.

### Available Actions

- **Join Room**:
  ```json
  {
    "action": "join_room",
    "room": "room_name"
  }
  ```

- **Leave Room**:
  ```json
  {
    "action": "leave_room",
    "room": "room_name"
  }
  ```

- **Broadcast Message**:
  ```json
  {
    "action": "broadcast",
    "message": "Hello, world!",
    "room": "optional_room_name"
  }
  ```
"""

        return {
            'server.py': server_template,
            'requirements.txt': requirements_txt,
            'README.md': readme_md
        }

    async def generate(self, api_spec: ApiSpec, output_path: Path) -> Dict[str, str]:
        """
        Generuje kod WebSocket na podstawie specyfikacji API
        
        Args:
            api_spec: Specyfikacja API
            output_path: Ścieżka do katalogu wyjściowego
            
        Returns:
            Słownik zawierający wygenerowane pliki
        """
        generated_files = {}
        env = Environment()
        
        # Create output directory
        output_path.mkdir(parents=True, exist_ok=True)
        
        # Generate server.py
        server_template = env.from_string(self.templates['server.py'])
        server_content = server_template.render(api_spec=api_spec)
        
        server_file = output_path / 'server.py'
        await self.file_manager.write_file(server_file, server_content)
        generated_files['server.py'] = str(server_file)
        
        # Generate requirements.txt
        requirements_template = env.from_string(self.templates['requirements.txt'])
        requirements_content = requirements_template.render(api_spec=api_spec)
        
        requirements_file = output_path / 'requirements.txt'
        await self.file_manager.write_file(requirements_file, requirements_content)
        generated_files['requirements.txt'] = str(requirements_file)
        
        # Generate README.md
        readme_template = env.from_string(self.templates['README.md'])
        readme_content = readme_template.render(api_spec=api_spec)
        
        readme_file = output_path / 'README.md'
        await self.file_manager.write_file(readme_file, readme_content)
        generated_files['README.md'] = str(readme_file)
        
        return generated_files
