"""
Generator dla GraphQL API
"""

from pathlib import Path
from typing import Dict
from jinja2 import Environment

from ..core.analyzer import ApiSpec
from ..utils.file_utils import FileManager

class GraphQLGenerator:
    """Generator kodu GraphQL z Graphene"""

    def __init__(self):
        self.file_manager = FileManager()
        self.templates = self._load_templates()

    def _load_templates(self) -> Dict[str, str]:
        """Ładuje szablony Jinja2 dla GraphQL"""
        template = """# GraphQL application generated by text2api

import graphene
from graphene import ObjectType, String
from flask import Flask, jsonify
from flask_graphql import GraphQLView
from flask_cors import CORS
import os

# Flask app initialization
app = Flask(__name__)
app.config['SECRET_KEY'] = os.getenv('SECRET_KEY', 'dev-secret-key')

# CORS
CORS(app)

# Database configuration (if needed)
{% if api_spec.database_required %}
from flask_sqlalchemy import SQLAlchemy
from datetime import datetime

app.config['SQLALCHEMY_DATABASE_URI'] = os.getenv(
    'DATABASE_URL', 
    'sqlite:///{{ api_spec.name }}.db'
)
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
db = SQLAlchemy(app)

# Database models
{% for model in api_spec.models %}
class {{ model.name }}(db.Model):
    __tablename__ = '{{ model.name|lower }}'
    
    id = db.Column(db.Integer, primary_key=True)
    {% for field in model.fields %}
    {{ field.name }} = db.Column({{ field.type }}{% if field.required %}, nullable=False{% endif %})
    {% endfor %}
    
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    def to_dict(self):
        return {
            'id': self.id,
            {% for field in model.fields %}
            '{{ field.name }}': self.{{ field.name }},
            {% endfor %}
            'created_at': self.created_at.isoformat(),
            'updated_at': self.updated_at.isoformat()
        }

{% endfor %}

# Create tables
with app.app_context():
    db.create_all()
{% endif %}

# GraphQL Types
class Query(ObjectType):
    # Add your queries here
    hello = String(description='A simple greeting')
    
    def resolve_hello(self, info):
        return 'Hello, GraphQL!'

# Schema
schema = graphene.Schema(query=Query)

# Routes
app.add_url_rule(
    '/graphql',
    view_func=GraphQLView.as_view(
        'graphql',
        schema=schema,
        graphiql=True
    )
)

@app.route('/')
def index():
    return jsonify({
        'name': '{{ api_spec.name }}',
        'version': '1.0.0',
        'docs': '/graphql',
        'status': 'running'
    })

if __name__ == '__main__':
    app.run(debug=True)"""
        return {'app.py': template}

    def generate(self, api_spec: ApiSpec, output_path: Path) -> Dict[str, str]:
        """
        Generuje kod GraphQL na podstawie specyfikacji API
        
        Args:
            api_spec: Specyfikacja API
            output_path: Ścieżka do katalogu wyjściowego
            
        Returns:
            Słownik zawierający wygenerowane pliki
        """
        env = Environment()
        template = env.from_string(self.templates['app.py'])
        rendered = template.render(api_spec=api_spec)
        
        output_file = output_path / 'app.py'
        self.file_manager.write_file(output_file, rendered)
        
        return {
            'app.py': str(output_file)
        }
