"""
Generator dla Flask REST API
"""

from pathlib import Path
from typing import Dict
from jinja2 import Environment, DictLoader

from ..core.analyzer import ApiSpec
from ..utils.file_utils import FileManager


class FlaskGenerator:
    """Generator kodu Flask"""

    def __init__(self):
        self.file_manager = FileManager()
        self.templates = self._load_templates()

    def _load_templates(self) -> Environment:
        """Ładuje szablony Jinja2 dla Flask"""

        templates = {
            'app.py': '''"""
Flask application generated by text2api
"""

from flask import Flask, request, jsonify, g
from flask_restx import Api, Resource, fields
from flask_cors import CORS
from functools import wraps
import os
{% if api_spec.database_required %}
from flask_sqlalchemy import SQLAlchemy
from flask_migrate import Migrate
from datetime import datetime
{% endif %}
{% if api_spec.auth_type %}
from flask_jwt_extended import JWTManager, create_access_token, jwt_required, get_jwt_identity
from werkzeug.security import generate_password_hash, check_password_hash
{% endif %}

# Flask app initialization
app = Flask(__name__)
app.config['SECRET_KEY'] = os.getenv('SECRET_KEY', 'dev-secret-key-change-in-production')

# CORS
CORS(app)

# Flask-RESTX for API documentation
api = Api(
    app,
    version='1.0',
    title='{{ api_spec.name }}',
    description='{{ api_spec.description }}',
    doc='/docs/'
)

{% if api_spec.database_required %}
# Database configuration
app.config['SQLALCHEMY_DATABASE_URI'] = os.getenv(
    'DATABASE_URL', 
    'sqlite:///{{ api_spec.name }}.db'
)
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

db = SQLAlchemy(app)
migrate = Migrate(app, db)
{% endif %}

{% if api_spec.auth_type %}
# JWT configuration
app.config['JWT_SECRET_KEY'] = os.getenv('JWT_SECRET', 'jwt-secret-change-in-production')
jwt = JWTManager(app)
{% endif %}

{% if api_spec.database_required %}
# Database models
{% for model in api_spec.models %}
class {{ model.name }}(db.Model):
    __tablename__ = '{{ model.name.lower() }}s'

    {% for field in model.fields %}
    {{ field.name }} = db.Column(
        {% if field.type == 'integer' %}db.Integer{% elif field.type == 'string' %}db.String(255){% elif field.type == 'boolean' %}db.Boolean{% elif field.type == 'datetime' %}db.DateTime{% else %}db.String(255){% endif %}{% if field.get('primary_key') %}, primary_key=True{% endif %}{% if field.get('auto_now_add') %}, default=datetime.utcnow{% endif %}
    )
    {% endfor %}

    def to_dict(self):
        return {
            {% for field in model.fields %}
            '{{ field.name }}': {% if field.type == 'datetime' %}self.{{ field.name }}.isoformat() if self.{{ field.name }} else None{% else %}self.{{ field.name }}{% endif %},
            {% endfor %}
        }

    def __repr__(self):
        return f'<{{ model.name }} {self.id}>'

{% endfor %}
{% endif %}

# API Models for documentation
{% for model in api_spec.models %}
{{ model.name.lower() }}_model = api.model('{{ model.name }}', {
    {% for field in model.fields %}
    '{{ field.name }}': fields.{% if field.type == 'integer' %}Integer{% elif field.type == 'string' %}String{% elif field.type == 'boolean' %}Boolean{% elif field.type == 'datetime' %}DateTime{% else %}String{% endif %}{% if field.get('required', True) %}(required=True{% else %}(required=False{% endif %}, description='{{ field.get("description", "") }}'),
    {% endfor %}
})

{% endfor %}

# Namespaces
{% set ns_names = [] %}
{% for endpoint in api_spec.endpoints %}
{% set ns_name = endpoint.path.split('/')[1] %}
{% if ns_name not in ns_names %}
{% set _ = ns_names.append(ns_name) %}
{{ ns_name }}_ns = api.namespace('{{ ns_name }}', description='{{ ns_name.title() }} operations')
{% endif %}
{% endfor %}

{% if api_spec.auth_type %}
# Authentication helpers
def auth_required(f):
    @wraps(f)
    @jwt_required()
    def decorated_function(*args, **kwargs):
        return f(*args, **kwargs)
    return decorated_function
{% endif %}

# Routes
@app.route('/')
def root():
    return jsonify({
        'message': '{{ api_spec.description }}',
        'version': '1.0.0',
        'docs': '/docs/'
    })

@app.route('/health')
def health():
    return jsonify({'status': 'healthy'})

{% if api_spec.auth_type %}
# Authentication endpoints
@api.route('/auth/login')
class Login(Resource):
    @api.expect(api.model('LoginData', {
        'username': fields.String(required=True),
        'password': fields.String(required=True)
    }))
    def post(self):
        """User login"""
        data = request.get_json()

        # TODO: Implement actual user authentication
        # For demo purposes, accept any credentials
        if data.get('username') and data.get('password'):
            access_token = create_access_token(identity=data['username'])
            return {'access_token': access_token}

        return {'message': 'Invalid credentials'}, 401

@api.route('/auth/register')
class Register(Resource):
    @api.expect(api.model('RegisterData', {
        'username': fields.String(required=True),
        'password': fields.String(required=True),
        'email': fields.String(required=True)
    }))
    def post(self):
        """User registration"""
        data = request.get_json()

        # TODO: Implement actual user registration
        return {'message': 'User registered successfully'}, 201
{% endif %}

# API endpoints
{% for endpoint in api_spec.endpoints %}
{% set ns_name = endpoint.path.split('/')[1] %}
@{{ ns_name }}_ns.route('{{ endpoint.path.replace("/" + ns_name, "") or "/" }}')
class {{ endpoint.name | title }}(Resource):
    {% if endpoint.method.value == 'GET' %}
    @{{ ns_name }}_ns.marshal_list_with({{ api_spec.models[0].name.lower() }}_model)
    {% if api_spec.auth_type %}@auth_required{% endif %}
    def get(self{% for param in endpoint.parameters %}{% if '{' + param.name + '}' not in endpoint.path %}, {{ param.name }}=None{% endif %}{% endfor %}):
        """{{ endpoint.description }}"""
        try:
            {% if 'list' in endpoint.name or endpoint.path.count('/') <= 2 %}
            # List items
            {% if api_spec.database_required %}
            items = {{ api_spec.models[0].name }}.query.all()
            return [item.to_dict() for item in items]
            {% else %}
            # Mock response
            return []
            {% endif %}
            {% else %}
            # Get single item
            {% if api_spec.database_required %}
            {% for param in endpoint.parameters %}
            {% if param.name == 'id' %}
            item = {{ api_spec.models[0].name }}.query.get_or_404({{ param.name }})
            return item.to_dict()
            {% endif %}
            {% endfor %}
            {% else %}
            # Mock response
            return {'id': 1, 'message': 'Item found'}
            {% endif %}
            {% endif %}
        except Exception as e:
            return {'error': str(e)}, 500

    {% elif endpoint.method.value == 'POST' %}
    @{{ ns_name }}_ns.expect({{ api_spec.models[0].name.lower() }}_model)
    @{{ ns_name }}_ns.marshal_with({{ api_spec.models[0].name.lower() }}_model, code=201)
    {% if api_spec.auth_type %}@auth_required{% endif %}
    def post(self):
        """{{ endpoint.description }}"""
        try:
            data = request.get_json()

            {% if api_spec.database_required %}
            # Create new item
            new_item = {{ api_spec.models[0].name }}(**data)
            db.session.add(new_item)
            db.session.commit()
            return new_item.to_dict(), 201
            {% else %}
            # Mock response
            return {'id': 1, 'message': 'Item created', **data}, 201
            {% endif %}
        except Exception as e:
            {% if api_spec.database_required %}
            db.session.rollback()
            {% endif %}
            return {'error': str(e)}, 500

    {% elif endpoint.method.value == 'PUT' %}
    @{{ ns_name }}_ns.expect({{ api_spec.models[0].name.lower() }}_model)
    @{{ ns_name }}_ns.marshal_with({{ api_spec.models[0].name.lower() }}_model)
    {% if api_spec.auth_type %}@auth_required{% endif %}
    def put(self{% for param in endpoint.parameters %}{% if '{' + param.name + '}' in endpoint.path %}, {{ param.name }}{% endif %}{% endfor %}):
        """{{ endpoint.description }}"""
        try:
            data = request.get_json()

            {% if api_spec.database_required %}
            {% for param in endpoint.parameters %}
            {% if param.name == 'id' %}
            item = {{ api_spec.models[0].name }}.query.get_or_404({{ param.name }})

            for key, value in data.items():
                if hasattr(item, key):
                    setattr(item, key, value)

            db.session.commit()
            return item.to_dict()
            {% endif %}
            {% endfor %}
            {% else %}
            # Mock response
            return {'id': {{ endpoint.parameters[0].name if endpoint.parameters else 1 }}, 'message': 'Item updated', **data}
            {% endif %}
        except Exception as e:
            {% if api_spec.database_required %}
            db.session.rollback()
            {% endif %}
            return {'error': str(e)}, 500

    {% elif endpoint.method.value == 'DELETE' %}
    {% if api_spec.auth_type %}@auth_required{% endif %}
    def delete(self{% for param in endpoint.parameters %}{% if '{' + param.name + '}' in endpoint.path %}, {{ param.name }}{% endif %}{% endfor %}):
        """{{ endpoint.description }}"""
        try:
            {% if api_spec.database_required %}
            {% for param in endpoint.parameters %}
            {% if param.name == 'id' %}
            item = {{ api_spec.models[0].name }}.query.get_or_404({{ param.name }})
            db.session.delete(item)
            db.session.commit()
            return {'message': 'Item deleted'}, 204
            {% endif %}
            {% endfor %}
            {% else %}
            # Mock response
            return {'message': 'Item deleted'}, 204
            {% endif %}
        except Exception as e:
            {% if api_spec.database_required %}
            db.session.rollback()
            {% endif %}
            return {'error': str(e)}, 500
    {% endif %}

{% endfor %}

{% if api_spec.database_required %}
# Create tables
@app.before_first_request
def create_tables():
    db.create_all()
{% endif %}

# Error handlers
@app.errorhandler(404)
def not_found(error):
    return jsonify({'error': 'Not found'}), 404

@app.errorhandler(500)
def internal_error(error):
    return jsonify({'error': 'Internal server error'}), 500

if __name__ == '__main__':
    app.run(
        host=os.getenv('HOST', '0.0.0.0'),
        port=int(os.getenv('PORT', 5000)),
        debug=os.getenv('DEBUG', 'False').lower() == 'true'
    )
''',

            'requirements.txt': '''flask==3.0.0
flask-restx==1.3.0
flask-cors==4.0.0
{% if api_spec.database_required %}
flask-sqlalchemy==3.1.1
flask-migrate==4.0.5
{% endif %}
{% if api_spec.auth_type %}
flask-jwt-extended==4.6.0
{% endif %}
python-dotenv==1.0.0
gunicorn==21.2.0
''',

            'config.py': '''"""
Configuration for {{ api_spec.name }}
"""

import os
from datetime import timedelta

class Config:
    SECRET_KEY = os.getenv('SECRET_KEY', 'dev-secret-key')

    {% if api_spec.database_required %}
    # Database
    SQLALCHEMY_DATABASE_URI = os.getenv('DATABASE_URL', 'sqlite:///app.db')
    SQLALCHEMY_TRACK_MODIFICATIONS = False
    {% endif %}

    {% if api_spec.auth_type %}
    # JWT
    JWT_SECRET_KEY = os.getenv('JWT_SECRET', 'jwt-secret-key')
    JWT_ACCESS_TOKEN_EXPIRES = timedelta(hours=1)
    {% endif %}

class DevelopmentConfig(Config):
    DEBUG = True
    {% if api_spec.database_required %}
    SQLALCHEMY_DATABASE_URI = os.getenv('DEV_DATABASE_URL', 'sqlite:///dev.db')
    {% endif %}

class ProductionConfig(Config):
    DEBUG = False
    {% if api_spec.database_required %}
    SQLALCHEMY_DATABASE_URI = os.getenv('DATABASE_URL')
    {% endif %}

class TestingConfig(Config):
    TESTING = True
    {% if api_spec.database_required %}
    SQLALCHEMY_DATABASE_URI = 'sqlite:///:memory:'
    {% endif %}

config = {
    'development': DevelopmentConfig,
    'production': ProductionConfig,
    'testing': TestingConfig,
    'default': DevelopmentConfig
}
''',

            '.env.example': '''# Environment variables for {{ api_spec.name }}

FLASK_APP=app.py
FLASK_ENV=development
SECRET_KEY=your-secret-key-here
{% if api_spec.database_required %}
DATABASE_URL=postgresql://user:password@localhost:5432/{{ api_spec.name }}
{% endif %}
{% if api_spec.auth_type %}
JWT_SECRET=your-jwt-secret-here
{% endif %}
HOST=0.0.0.0
PORT=5000
DEBUG=True
'''
        }

        return Environment(loader=DictLoader(templates))

    async def generate(self, api_spec: ApiSpec, output_path: Path) -> Dict[str, str]:
        """Generuje kod Flask"""

        # Utwórz folder wyjściowy
        await self.file_manager.ensure_directory(output_path)

        generated_files = {}

        # Generuj app.py
        app_content = self.templates.get_template('app.py').render(api_spec=api_spec)
        app_path = output_path / "app.py"
        await self.file_manager.write_file(app_path, app_content)
        generated_files['app.py'] = str(app_path)

        # Generuj requirements.txt
        req_content = self.templates.get_template('requirements.txt').render(api_spec=api_spec)
        req_path = output_path / "requirements.txt"
        await self.file_manager.write_file(req_path, req_content)
        generated_files['requirements.txt'] = str(req_path)

        # Generuj config.py
        config_content = self.templates.get_template('config.py').render(api_spec=api_spec)
        config_path = output_path / "config.py"
        await self.file_manager.write_file(config_path, config_content)
        generated_files['config.py'] = str(config_path)

        # Generuj .env.example
        env_content = self.templates.get_template('.env.example').render(api_spec=api_spec)
        env_path = output_path / ".env.example"
        await self.file_manager.write_file(env_path, env_content)
        generated_files['.env.example'] = str(env_path)

        # Generuj dodatkowe pliki
        await self._generate_additional_files(api_spec, output_path, generated_files)

        return generated_files

    async def _generate_additional_files(self, api_spec: ApiSpec, output_path: Path, generated_files: Dict[str, str]):
        """Generuje dodatkowe pliki dla Flask"""

        # wsgi.py dla produkcji
        wsgi_content = f'''"""
WSGI entry point for {api_spec.name}
"""

from app import app

if __name__ == "__main__":
    app.run()
'''

        wsgi_path = output_path / "wsgi.py"
        await self.file_manager.write_file(wsgi_path, wsgi_content)
        generated_files['wsgi.py'] = str(wsgi_path)

        # Jeśli używa bazy danych, stwórz migracje
        if api_spec.database_required:
            migrations_dir = output_path / "migrations"
            await self.file_manager.ensure_directory(migrations_dir)

            # Podstawowy plik migracji
            init_migration = '''"""
Initial migration

Revision ID: 001
Create Date: 2024-01-01 00:00:00.000000
"""

from alembic import op
import sqlalchemy as sa

# revision identifiers
revision = '001'
down_revision = None
branch_labels = None
depends_on = None

def upgrade():
    # Create tables based on models
    pass

def downgrade():
    # Drop tables
    pass
'''

            migration_path = migrations_dir / "001_initial.py"
            await self.file_manager.write_file(migration_path, init_migration)
            generated_files['migrations/001_initial.py'] = str(migration_path)