"""
Generator dla FastAPI
"""

import os
from pathlib import Path
from typing import Dict, List, Any
from jinja2 import Environment, FileSystemLoader, DictLoader

from ..core.analyzer import ApiSpec, Endpoint, Field, HttpMethod
from ..utils.file_utils import FileManager


class FastAPIGenerator:
    """Generator kodu FastAPI"""

    def __init__(self):
        self.file_manager = FileManager()
        self.templates = self._load_templates()

    def _load_templates(self) -> Environment:
        """Ładuje szablony Jinja2"""

        templates = {
            'main.py': '''"""
FastAPI application generated by text2api
"""

from fastapi import FastAPI, HTTPException, Depends, Query, Path
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel, Field
from typing import List, Optional, Dict, Any
from datetime import datetime
import uvicorn
{% if api_spec.database_required %}
from sqlalchemy import create_engine, Column, Integer, String, DateTime, Boolean
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker, Session
from databases import Database
import os
{% endif %}
{% if api_spec.auth_type %}
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from jose import JWTError, jwt
from passlib.context import CryptContext
{% endif %}

# FastAPI app initialization
app = FastAPI(
    title="{{ api_spec.name }}",
    description="{{ api_spec.description }}",
    version="1.0.0"
)

# CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

{% if api_spec.database_required %}
# Database configuration
DATABASE_URL = os.getenv("DATABASE_URL", "sqlite:///./{{ api_spec.name }}.db")
database = Database(DATABASE_URL)
engine = create_engine(DATABASE_URL)
SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)
Base = declarative_base()

# Database dependency
def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

@app.on_event("startup")
async def startup():
    await database.connect()

@app.on_event("shutdown")
async def shutdown():
    await database.disconnect()
{% endif %}

{% if api_spec.auth_type %}
# Authentication setup
security = HTTPBearer()
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
SECRET_KEY = os.getenv("SECRET_KEY", "your-secret-key-here")
ALGORITHM = "HS256"

def verify_token(credentials: HTTPAuthorizationCredentials = Depends(security)):
    try:
        payload = jwt.decode(credentials.credentials, SECRET_KEY, algorithms=[ALGORITHM])
        return payload
    except JWTError:
        raise HTTPException(status_code=401, detail="Invalid authentication credentials")
{% endif %}

# Pydantic models
{% for model in api_spec.models %}
class {{ model.name }}Base(BaseModel):
    {% for field in model.fields %}
    {% if field.name != 'id' and not field.get('auto_now_add') and not field.get('auto_now') %}
    {{ field.name }}: {% if field.get('required', True) %}{{ field.type | python_type }}{% else %}Optional[{{ field.type | python_type }}] = None{% endif %}
    {% endif %}
    {% endfor %}

class {{ model.name }}Create({{ model.name }}Base):
    pass

class {{ model.name }}Response({{ model.name }}Base):
    {% for field in model.fields %}
    {{ field.name }}: {{ field.type | python_type }}
    {% endfor %}

    class Config:
        from_attributes = True

{% endfor %}

{% if api_spec.database_required %}
# SQLAlchemy models
{% for model in api_spec.models %}
class {{ model.name }}DB(Base):
    __tablename__ = "{{ model.name.lower() }}s"

    {% for field in model.fields %}
    {{ field.name }} = Column(
        {% if field.type == 'integer' %}Integer{% elif field.type == 'string' %}String{% elif field.type == 'boolean' %}Boolean{% elif field.type == 'datetime' %}DateTime{% else %}String{% endif %}{% if field.get('primary_key') %}, primary_key=True, index=True{% endif %}{% if field.get('auto_now_add') %}, default=datetime.utcnow{% endif %}
    )
    {% endfor %}

{% endfor %}

# Create tables
Base.metadata.create_all(bind=engine)
{% endif %}

# Routes
@app.get("/")
async def root():
    return {"message": "{{ api_spec.description }}", "version": "1.0.0"}

@app.get("/health")
async def health_check():
    return {"status": "healthy", "timestamp": datetime.utcnow()}

{% for endpoint in api_spec.endpoints %}
@app.{{ endpoint.method.value.lower() }}("{{ endpoint.path }}")
async def {{ endpoint.name }}(
    {% for param in endpoint.parameters %}
    {% if '{' + param.name + '}' in endpoint.path %}
    {{ param.name }}: {{ param.type | python_type }} = Path(..., description="{{ param.description }}"),
    {% else %}
    {{ param.name }}: {% if param.required %}{{ param.type | python_type }}{% else %}Optional[{{ param.type | python_type }}] = None{% endif %} = Query({% if not param.required %}None{% else %}...{% endif %}, description="{{ param.description }}"),
    {% endif %}
    {% endfor %}
    {% if endpoint.request_body and endpoint.method.value in ['POST', 'PUT', 'PATCH'] %}
    {% set model_name = endpoint.request_body[0].name.title() if endpoint.request_body else 'Item' %}
    request_data: Dict[str, Any],
    {% endif %}
    {% if api_spec.database_required %}
    db: Session = Depends(get_db){% if api_spec.auth_type %},{% endif %}
    {% endif %}
    {% if api_spec.auth_type %}
    current_user: dict = Depends(verify_token)
    {% endif %}
):
    """{{ endpoint.description }}"""
    try:
        {% if endpoint.method.value == 'GET' %}
        {% if 'list' in endpoint.name or endpoint.path.count('/') == 1 %}
        # List items
        {% if api_spec.database_required %}
        items = db.query({{ api_spec.models[0].name }}DB).all()
        return {"items": items, "total": len(items)}
        {% else %}
        # Mock response
        return {"items": [], "total": 0}
        {% endif %}
        {% else %}
        # Get single item
        {% if api_spec.database_required %}
        {% for param in endpoint.parameters %}
        {% if param.name == 'id' %}
        item = db.query({{ api_spec.models[0].name }}DB).filter({{ api_spec.models[0].name }}DB.id == {{ param.name }}).first()
        if not item:
            raise HTTPException(status_code=404, detail="Item not found")
        return item
        {% endif %}
        {% endfor %}
        {% else %}
        # Mock response
        return {"id": 1, "message": "Item found"}
        {% endif %}
        {% endif %}

        {% elif endpoint.method.value == 'POST' %}
        # Create new item
        {% if api_spec.database_required %}
        new_item = {{ api_spec.models[0].name }}DB(**request_data)
        db.add(new_item)
        db.commit()
        db.refresh(new_item)
        return new_item
        {% else %}
        # Mock response
        return {"id": 1, "message": "Item created", **request_data}
        {% endif %}

        {% elif endpoint.method.value == 'PUT' %}
        # Update item
        {% if api_spec.database_required %}
        {% for param in endpoint.parameters %}
        {% if param.name == 'id' %}
        item = db.query({{ api_spec.models[0].name }}DB).filter({{ api_spec.models[0].name }}DB.id == {{ param.name }}).first()
        if not item:
            raise HTTPException(status_code=404, detail="Item not found")

        for key, value in request_data.items():
            setattr(item, key, value)

        db.commit()
        db.refresh(item)
        return item
        {% endif %}
        {% endfor %}
        {% else %}
        # Mock response
        return {"id": {{ endpoint.parameters[0].name if endpoint.parameters else 1 }}, "message": "Item updated", **request_data}
        {% endif %}

        {% elif endpoint.method.value == 'DELETE' %}
        # Delete item
        {% if api_spec.database_required %}
        {% for param in endpoint.parameters %}
        {% if param.name == 'id' %}
        item = db.query({{ api_spec.models[0].name }}DB).filter({{ api_spec.models[0].name }}DB.id == {{ param.name }}).first()
        if not item:
            raise HTTPException(status_code=404, detail="Item not found")

        db.delete(item)
        db.commit()
        return {"message": "Item deleted"}
        {% endif %}
        {% endfor %}
        {% else %}
        # Mock response
        return {"message": "Item deleted"}
        {% endif %}

        {% endif %}
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Internal server error: {str(e)}")

{% endfor %}

if __name__ == "__main__":
    uvicorn.run(app, host="0.0.0.0", port=8000)
''',

            'requirements.txt': '''fastapi==0.104.1
uvicorn[standard]==0.24.0
pydantic==2.5.0
{% if api_spec.database_required %}
sqlalchemy==2.0.23
databases[postgresql]==0.8.0
databases[sqlite]==0.8.0
alembic==1.13.1
{% endif %}
{% if api_spec.auth_type %}
python-jose[cryptography]==3.3.0
passlib[bcrypt]==1.7.4
python-multipart==0.0.6
{% endif %}
python-dotenv==1.0.0
''',

            'models.py': '''"""
Database models for {{ api_spec.name }}
"""

{% if api_spec.database_required %}
from sqlalchemy import Column, Integer, String, DateTime, Boolean, Text
from sqlalchemy.ext.declarative import declarative_base
from datetime import datetime

Base = declarative_base()

{% for model in api_spec.models %}
class {{ model.name }}(Base):
    __tablename__ = "{{ model.name.lower() }}s"

    {% for field in model.fields %}
    {{ field.name }} = Column(
        {% if field.type == 'integer' %}Integer{% elif field.type == 'string' %}String(255){% elif field.type == 'text' %}Text{% elif field.type == 'boolean' %}Boolean{% elif field.type == 'datetime' %}DateTime{% else %}String(255){% endif %}{% if field.get('primary_key') %}, primary_key=True, index=True{% endif %}{% if field.get('auto_now_add') %}, default=datetime.utcnow{% endif %}{% if field.get('auto_now') %}, onupdate=datetime.utcnow{% endif %}
    )
    {% endfor %}

    def __repr__(self):
        return f"<{{ model.name }}(id={self.id})>"

{% endfor %}
{% endif %}
''',

            '.env.example': '''# Environment variables for {{ api_spec.name }}

# Database
{% if api_spec.database_required %}
DATABASE_URL=postgresql://user:password@localhost:5432/{{ api_spec.name }}
# Alternative: sqlite:///./{{ api_spec.name }}.db
{% endif %}

# Security
{% if api_spec.auth_type %}
SECRET_KEY=your-secret-key-here-change-in-production
JWT_SECRET=your-jwt-secret-here
{% endif %}

# API Configuration
DEBUG=True
HOST=0.0.0.0
PORT=8000
'''
        }

        return Environment(loader=DictLoader(templates))

    async def generate(self, api_spec: ApiSpec, output_path: Path) -> Dict[str, str]:
        """Generuje kod FastAPI"""

        # Utwórz folder wyjściowy
        await self.file_manager.ensure_directory(output_path)

        generated_files = {}

        # Dodaj custom filtry do templates
        self.templates.filters['python_type'] = self._python_type_filter

        # Generuj main.py
        main_content = self.templates.get_template('main.py').render(api_spec=api_spec)
        main_path = output_path / "main.py"
        await self.file_manager.write_file(main_path, main_content)
        generated_files['main.py'] = str(main_path)

        # Generuj requirements.txt
        req_content = self.templates.get_template('requirements.txt').render(api_spec=api_spec)
        req_path = output_path / "requirements.txt"
        await self.file_manager.write_file(req_path, req_content)
        generated_files['requirements.txt'] = str(req_path)

        # Generuj models.py (jeśli potrzebna baza danych)
        if api_spec.database_required:
            models_content = self.templates.get_template('models.py').render(api_spec=api_spec)
            models_path = output_path / "models.py"
            await self.file_manager.write_file(models_path, models_content)
            generated_files['models.py'] = str(models_path)

        # Generuj .env.example
        env_content = self.templates.get_template('.env.example').render(api_spec=api_spec)
        env_path = output_path / ".env.example"
        await self.file_manager.write_file(env_path, env_content)
        generated_files['.env.example'] = str(env_path)

        # Generuj dodatkowe pliki utility
        await self._generate_utils(api_spec, output_path, generated_files)

        return generated_files

    def _python_type_filter(self, field_type: str) -> str:
        """Konwertuje typy pól na typy Python"""

        type_mapping = {
            'string': 'str',
            'integer': 'int',
            'float': 'float',
            'boolean': 'bool',
            'datetime': 'datetime',
            'array': 'List[Any]',
            'object': 'Dict[str, Any]'
        }

        return type_mapping.get(field_type, 'str')

    async def _generate_utils(self, api_spec: ApiSpec, output_path: Path, generated_files: Dict[str, str]):
        """Generuje pliki utility"""

        # config.py
        config_content = f'''"""
Configuration for {api_spec.name}
"""

import os
from pydantic import BaseSettings

class Settings(BaseSettings):
    app_name: str = "{api_spec.name}"
    debug: bool = os.getenv("DEBUG", "False").lower() == "true"
    host: str = os.getenv("HOST", "0.0.0.0")
    port: int = int(os.getenv("PORT", "8000"))

    {"database_url: str = os.getenv('DATABASE_URL', 'sqlite:///./app.db')" if api_spec.database_required else ""}
    {"secret_key: str = os.getenv('SECRET_KEY', 'change-me-in-production')" if api_spec.auth_type else ""}

    class Config:
        env_file = ".env"

settings = Settings()
'''

        config_path = output_path / "config.py"
        await self.file_manager.write_file(config_path, config_content)
        generated_files['config.py'] = str(config_path)

        # database.py (jeśli potrzebna baza danych)
        if api_spec.database_required:
            db_content = '''"""
Database configuration and utilities
"""

from sqlalchemy import create_engine
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker
from config import settings

engine = create_engine(settings.database_url)
SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)
Base = declarative_base()

def get_database():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()
'''

            db_path = output_path / "database.py"
            await self.file_manager.write_file(db_path, db_content)
            generated_files['database.py'] = str(db_path)

        # auth.py (jeśli potrzebna autoryzacja)
        if api_spec.auth_type:
            auth_content = '''"""
Authentication utilities
"""

from datetime import datetime, timedelta
from jose import JWTError, jwt
from passlib.context import CryptContext
from fastapi import HTTPException, Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from config import settings

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
security = HTTPBearer()

def create_access_token(data: dict, expires_delta: timedelta = None):
    to_encode = data.copy()
    if expires_delta:
        expire = datetime.utcnow() + expires_delta
    else:
        expire = datetime.utcnow() + timedelta(minutes=15)

    to_encode.update({"exp": expire})
    encoded_jwt = jwt.encode(to_encode, settings.secret_key, algorithm="HS256")
    return encoded_jwt

def verify_password(plain_password, hashed_password):
    return pwd_context.verify(plain_password, hashed_password)

def get_password_hash(password):
    return pwd_context.hash(password)

def verify_token(credentials: HTTPAuthorizationCredentials = Depends(security)):
    try:
        payload = jwt.decode(credentials.credentials, settings.secret_key, algorithms=["HS256"])
        return payload
    except JWTError:
        raise HTTPException(status_code=401, detail="Invalid authentication credentials")
'''

            auth_path = output_path / "auth.py"
            await self.file_manager.write_file(auth_path, auth_content)
            generated_files['auth.py'] = str(auth_path)