import decimal
import uuid

from easytradesdk.Serializer import DeserializableObject
from obspy import UTCDateTime


class BackTestConfig(DeserializableObject):

    def __init__(self, startDate=None, endDate=None):
        """
        :param startDate:  回测开始日期, yyyy-MM-dd
        :param endDate:    回测结束日期, yyyy-MM-dd
        :param endDate:    回测编号
        """

        if startDate is not None:
            self.startDate = startDate
            self.startTimeMills = int(UTCDateTime.strptime(startDate + " 00:00:00", "%Y-%m-%d %H:%M:%S").timestamp) * 1000
        if endDate is not None:
            self.endDate = endDate
            self.endTimeStampMills = int(UTCDateTime.strptime(endDate + " 23:59:59", "%Y-%m-%d %H:%M:%S").timestamp) * 1000

        self.backTestCode = None
        self.executeClass = None
        self.executePeriod = None
        self.strategyParams = None
        self.dataSourceConfig = None
        self.dataFetcherConfig = None
        self.cleanBackTestOrders = True
        self.klineConfigs = []
        self.positionConfigs = {}
        self.slippageConfigs = {}

    def configDataSource(self, dataSourceConfig):
        self.dataSourceConfig = dataSourceConfig
        return self

    def configDataFetcher(self, dataFetcherConfig):
        self.dataFetcherConfig = dataFetcherConfig
        return self

    def configBackTestCode(self, backTestCode=None):
        if self.startDate is None:
            raise Exception("please config startDate first")

        if self.endDate is None:
            raise Exception("please config endDate first")

        if backTestCode is None:
            self.backTestCode = self.startDate.replace("-", "") + "-" + self.endDate.replace("-", "") + "-" + str(uuid.uuid1()).replace("-", "")
        else:
            self.backTestCode = self.startDate.replace("-", "") + "-" + self.endDate.replace("-", "") + "-" + backTestCode
        return self

    def configKlines(self, tc, symbol, klinePeriod):
        """
        配置回测是要获取的 K 线
        :param tc:          交易所
        :param symbol:      交易标的
        :param klinePeriod: K 线周期
        :return:
        """
        if tc is None or symbol is None or klinePeriod is None:
            raise Exception("invalid params")
        self.klineConfigs.append({"tc": tc, "symbol": symbol, "klinePeriod": klinePeriod})
        return self

    def configPosition(self, tc, symbol, initialAmount=decimal.Decimal('0'), initialHolding=decimal.Decimal('0')):
        """
            配置仓位信息
            :param tc:              交易所
            :param symbol:          交易标的
            :param initialAmount:   初始金额
            :param initialHolding:  初始持仓
            :return:
        """

        if tc is None or symbol is None:
            raise Exception("invalid params")

        _key = tc + ":" + symbol

        if _key not in self.positionConfigs:
            self.positionConfigs[_key] = {"tc": tc, "symbol": symbol, "initialAmount": initialAmount, "initialHolding": initialHolding}

        return self

    def configSlippage(self, side, slippageType, value):
        """
            交易滑点配置
            :param side:          buy or sell
            :param slippageType:  滑点类型，1: 百分比，2：具体数值
            :param value:
            :return:
        """

        if side is None or side != "buy" or side != "sell":
            raise Exception("invalid side")

        if slippageType != 1 and slippageType != 2:
            raise Exception("invalid slippageType")

        self.slippageConfigs[side] = {"slippageType": slippageType, "value": value}
        return self

    def configStrategyParams(self, params: dict):
        """
            配置策略参数
            :param params:
            :return:
        """
        self.strategyParams = params
        return self

    def build(self, clazz, executePeriod, cleanBackTestOrders=True):
        self.executeClass = clazz
        self.executePeriod = executePeriod
        self.cleanBackTestOrders = cleanBackTestOrders
        return self
