"""
Add profiles based on the covering arrays generated by the ACTS to the sfera_automation.json.
We can then use the resulting sfera_automation.json in the next step.
"""

from csv import reader
from json import loads
from logging import getLogger
from pathlib import Path
from typing import Sequence

from better_safe_than_sorry.shared import (
    OVERRIDE_OPTION,
    PREFIX_OPTION,
    PROFILE_OPTION,
    SFERA_OPTION,
    Profile,
    Rule,
)
from better_safe_than_sorry.utils.custom_export import export_to_file, export_to_json
from typer import Argument, Option, Typer, echo

app = Typer(
    help="Commands to calculate covering arrays using the Automated Combinatorial Testing for Software."
)
logger = getLogger(__name__)


def csv_to_covering_arrays(file: Path) -> (Sequence[Rule], Sequence[Sequence[str]]):
    """
    Convert the ACTS output file (csv). Returns the used Rules and the covering arrays.
    """
    with file.open() as f:
        # filter out comments starting with '#'
        filtered_input = [
            row for row in f.readlines() if not row.lstrip().startswith("#")
        ]
    csvreader = reader(filtered_input)

    # Read rules
    rules = next(csvreader)

    # Read covering arrays
    covering_arrays = []
    for row in csvreader:
        covering_arrays.append(row)

    return rules, covering_arrays


@app.command(name="convert-acts-covering-arrays-into-sfera-profiles")
def cli(
    input_file: Path = Argument(
        "acts_output.csv",
        dir_okay=False,
        exists=True,
        help="Specify path to output.csv of ACTS.",
    ),
    sfera_automation_file: Path = SFERA_OPTION,
    prefix: str = PREFIX_OPTION,
    output_file: Path = Option(
        "custom_sfera_automation.json",
        "-o",
        "--output",
        dir_okay=False,
        help="Specify path where to store the resulting json.",
    ),
    override: bool = OVERRIDE_OPTION,
) -> None:
    """
    Adds the output of an ACTS model as new profiles to a sfera_automation.json
    """
    output_json = loads(sfera_automation_file.read_text())

    # Load existing profiles
    output_profiles = output_json["profiles"]

    # Convert CSV input to list of rules and covering arrays
    rules, covering_arrays = csv_to_covering_arrays(input_file)

    # Add a new profile per covering array
    for i, covering_array in enumerate(covering_arrays):
        # New profile
        profile = []

        for j, val in enumerate(covering_array):
            # Check if rule j should be applied according to the covering array
            if val.strip() == "true":
                profile.append(rules[j])
            # We could also check for don't care values (val="*") here

        # Add new profile to output, according to the naming scheme 'prefix{1..n}'
        output_profiles[prefix + str(i + 1)] = profile

    # Write output to file
    export_to_json(output_file, output_json, override)


@app.command(name='convert-a-profile-into-acts-input')
def cli(
    profile_name: str = PROFILE_OPTION,
    input_file: Path = Argument(
        "sfera_automation.json",
        help="Specify path to sfera_automation.json which should be converted.",
        dir_okay=False,
        exists=True,
    ),
    output_file: Path = Option(
        "sfera_automation_acts.txt",
        "-o",
        "--output",
        dir_okay=False,
        help="Specify path where to store the ACTS model output.",
    ),
    override: bool = OVERRIDE_OPTION,
) -> None:
    """
    Converts a sfera_automation profile to an ACTS model.
    """
    echo(f"Load rules from {input_file}")
    # Load data from sfera_automation.json
    data = loads(input_file.read_text())

    # Load all profiles
    profiles = data["profiles"]

    # Check if selected profile exists
    if profile_name not in profiles:
        echo("Could not find profile '" + profile_name + "' in the input!\nAborting.")
        return

    # Load selected profile
    selected_profile: Profile = profiles[profile_name]

    # Build header for the output
    output = _build_header(profile_name)

    # Add rules to the output
    for rule in selected_profile:
        output += "\n" + rule + " (boolean) : true, false"
        logger.info("Adding rule " + rule)

    # Write output to file
    export_to_file(output_file, output, override)
    echo("Done!")


def _build_header(profile_name: str) -> str:
    """
    Build the header for the ACTS input file
    :param profile_name:
    :return:
    """
    return f"""[System]
-- specify system name
Name: {profile_name}

[Parameter]
-- general syntax is parameter_name : value1, value2, ...
"""


if __name__ == "__main__":
    app()
