import { Connections, ISecurityGroup, IVpc, SubnetSelection } from "@aws-cdk/aws-ec2";
import { Grant, IGrantable } from "@aws-cdk/aws-iam";
import { IKey } from "@aws-cdk/aws-kms";
import { Credentials, Endpoint, IClusterEngine, IParameterGroup, IServerlessCluster, ISubnetGroup, RotationMultiUserOptions, RotationSingleUserOptions, ServerlessScalingOptions } from "@aws-cdk/aws-rds";
import { ISecret, SecretAttachmentTargetProps, SecretRotation } from "@aws-cdk/aws-secretsmanager";
import { Construct, Duration, RemovalPolicy, Resource } from "@aws-cdk/core";
/**
 * Properties to configure an Aurora Serverless Cluster.
 *
 * @stability stable
 */
export interface ServerlessClusterFromSnapshotProps {
    /**
     * What kind of database to start.
     *
     * @stability stable
     */
    readonly engine: IClusterEngine;
    /**
     * Credentials for the administrative user.
     *
     * @default - A username of 'admin' and SecretsManager-generated password
     * @stability stable
     */
    readonly credentials?: Credentials;
    /**
     * An optional identifier for the cluster.
     *
     * @default - A name is automatically generated.
     * @stability stable
     */
    readonly clusterIdentifier?: string;
    /**
     * The identifier for the DB snapshot or DB cluster snapshot to restore from.
     *
     * You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.
     *
     * After you restore a DB cluster with a SnapshotIdentifier property, you must specify the same SnapshotIdentifier property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the SnapshotIdentifier property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified SnapshotIdentifier property, and the original DB cluster is deleted.
     *
     * @stability stable
     */
    readonly snapshotIdentifier: string;
    /**
     * The number of days during which automatic DB snapshots are retained.
     *
     * Automatic backup retention cannot be disabled on serverless clusters.
     * Must be a value from 1 day to 35 days.
     *
     * @default Duration.days(1)
     * @stability stable
     */
    readonly backupRetention?: Duration;
    /**
     * Name of a database which is automatically created inside the cluster.
     *
     * @default - Database is not created in cluster.
     * @stability stable
     */
    readonly defaultDatabaseName?: string;
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if removalPolicy is RETAIN, false otherwise
     * @stability stable
     */
    readonly deletionProtection?: boolean;
    /**
     * Whether to enable the Data API.
     *
     * @default false
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html
     * @stability stable
     */
    readonly enableDataApi?: boolean;
    /**
     * The VPC that this Aurora Serverless cluster has been created in.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * Where to place the instances within the VPC.
     *
     * @default - the VPC default strategy if not specified.
     * @stability stable
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Scaling configuration of an Aurora Serverless database cluster.
     *
     * @default - Serverless cluster is automatically paused after 5 minutes of being idle.
     * minimum capacity: 2 ACU
     * maximum capacity: 16 ACU
     * @stability stable
     */
    readonly scaling?: ServerlessScalingOptions;
    /**
     * The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Security group.
     *
     * @default - a new security group is created.
     * @stability stable
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * The KMS key for storage encryption.
     *
     * If you specify the SnapshotIdentifier property, the StorageEncrypted property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified KmsKeyId property is used.
     *
     * @default - the default master key will be used for storage encryption
     * @stability stable
     */
    readonly storageEncryptionKey?: IKey;
    /**
     * Additional parameters to pass to the database engine.
     *
     * @default - no parameter group.
     * @stability stable
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     * @stability stable
     */
    readonly subnetGroup?: ISubnetGroup;
}
/**
 * A Serverless Cluster restored from a snapshot.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
export declare class ServerlessClusterFromSnapshot extends Resource implements IServerlessCluster {
    /**
     * Identifier of the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections.
     *
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * The secret attached to this cluster.
     *
     * @stability stable
     */
    readonly secret?: ISecret;
    /**
     * @stability stable
     */
    protected enableDataApi?: boolean;
    private readonly subnetGroup;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ServerlessClusterFromSnapshotProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @stability stable
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     *
     * @stability stable
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): SecretRotation;
    private renderScalingConfiguration;
    /**
     * The ARN of the cluster.
     *
     * @stability stable
     */
    get clusterArn(): string;
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantDataApiAccess(grantee: IGrantable): Grant;
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget(): SecretAttachmentTargetProps;
}
