from cachetools import Cache, FIFOCache
from flask import Flask
from logging import Logger
from pypomes_core import (
    APP_PREFIX, env_get_int, env_get_str
)
from typing import Any, Final

from .iam_common import _IAM_SERVERS, IamServer, _iam_lock
from .iam_pomes import user_token

KEYCLOAK_CLIENT_ID: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_CLIENT_ID")
KEYCLOAK_CLIENT_SECRET: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_CLIENT_SECRET")
KEYCLOAK_CLIENT_TIMEOUT: Final[int] = env_get_int(key=f"{APP_PREFIX}_KEYCLOAK_CLIENT_TIMEOUT")

KEYCLOAK_ENDPOINT_CALLBACK: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_ENDPOINT_CALLBACK",
                                                     def_value="/iam/ijud:callback")
KEYCLOAK_ENDPOINT_EXCHANGE: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_ENDPOINT_EXCHANGE",
                                                     def_value="/iam/ijud:exchange-token")
KEYCLOAK_ENDPOINT_LOGIN: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_ENDPOINT_LOGIN",
                                                  def_value="/iam/ijud:login")
KEYCLOAK_ENDPOINT_LOGOUT: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_ENDPOINT_LOGOUT",
                                                   def_value="/iam/ijud:logout")
KEYCLOAK_ENDPOINT_TOKEN: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_ENDPOINT_TOKEN",
                                                  def_value="/iam/ijud:get-token")

KEYCLOAK_PUBLIC_KEY_LIFETIME: Final[int] = env_get_int(key=f"{APP_PREFIX}_KEYCLOAK_PUBLIC_KEY_LIFETIME",
                                                       def_value=86400)  # 24 hours
KEYCLOAK_REALM: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_REALM")
KEYCLOAK_RECIPIENT_ATTR: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_RECIPIENT_ATTR",
                                                  def_value="preferred_username")
KEYCLOAK_URL_AUTH_BASE: Final[str] = env_get_str(key=f"{APP_PREFIX}_KEYCLOAK_URL_AUTH_BASE")


def keycloak_setup(flask_app: Flask,
                   base_url: str = KEYCLOAK_URL_AUTH_BASE,
                   realm: str = KEYCLOAK_REALM,
                   client_id: str = KEYCLOAK_CLIENT_ID,
                   client_secret: str = KEYCLOAK_CLIENT_SECRET,
                   client_timeout: int = KEYCLOAK_CLIENT_TIMEOUT,
                   public_key_lifetime: int | None = KEYCLOAK_PUBLIC_KEY_LIFETIME,
                   recipient_attribute: str | None = KEYCLOAK_RECIPIENT_ATTR,
                   callback_endpoint: str | None = KEYCLOAK_ENDPOINT_CALLBACK,
                   login_endpoint: str | None = KEYCLOAK_ENDPOINT_LOGIN,
                   logout_endpoint: str | None = KEYCLOAK_ENDPOINT_LOGOUT,
                   token_endpoint: str | None = KEYCLOAK_ENDPOINT_TOKEN,
                   exchange_endpoint: str | None = KEYCLOAK_ENDPOINT_EXCHANGE) -> None:
    """
    Configure the Keycloak IAM.

    This should be invoked only once, before the first access to a Keycloak service.

    :param flask_app: the Flask application
    :param base_url: base URL to request Keycloak services
    :param realm: the Keycloak realm
    :param client_id: the client's identification with JusBR
    :param client_secret: the client's password with JusBR
    :param client_timeout: timeout for login authentication (in seconds,defaults to no timeout)
    :param public_key_lifetime: how long to use Keycloak's public key, before refreshing it (in seconds)
    :param recipient_attribute: attribute in the token's payload holding the token's subject
    :param callback_endpoint: endpoint for the callback from the front end
    :param login_endpoint: endpoint for redirecting user to Keycloak's login page
    :param logout_endpoint: endpoint for terminating user access to Keycloak
    :param token_endpoint: endpoint for retrieving Keycloak's authentication token
    :param exchange_endpoint: endpoint for requesting token exchange
    """
    from .iam_services import (
        service_login, service_logout, service_callback, service_exchange, service_token
    )

    # configure the Keycloak registry
    cache: Cache = FIFOCache(maxsize=1048576)
    cache["users"] = {}
    with _iam_lock:
        _IAM_SERVERS[IamServer.IAM_KEYCLOAK] = {
            "base-url": f"{base_url}/realms/{realm}",
            "client-id": client_id,
            "client-secret": client_secret,
            "client-timeout": client_timeout,
            "public-key": None,
            "pk-lifetime": public_key_lifetime,
            "pk-expiration": 0,
            "recipient-attr": recipient_attribute,
            "cache": cache
        }

    # establish the endpoints
    if callback_endpoint:
        flask_app.add_url_rule(rule=callback_endpoint,
                               endpoint="keycloak-callback",
                               view_func=service_callback,
                               methods=["GET"])
    if login_endpoint:
        flask_app.add_url_rule(rule=login_endpoint,
                               endpoint="keycloak-login",
                               view_func=service_login,
                               methods=["GET"])
    if logout_endpoint:
        flask_app.add_url_rule(rule=logout_endpoint,
                               endpoint="keycloak-logout",
                               view_func=service_logout,
                               methods=["GET"])
    if token_endpoint:
        flask_app.add_url_rule(rule=token_endpoint,
                               endpoint="keycloak-token",
                               view_func=service_token,
                               methods=["GET"])
    if exchange_endpoint:
        flask_app.add_url_rule(rule=exchange_endpoint,
                               endpoint="keycloak-exchange",
                               view_func=service_exchange,
                               methods=["POST"])


def keycloak_get_token(user_id: str,
                       errors: list[str] = None,
                       logger: Logger = None) -> str:
    """
    Retrieve a Keycloak authentication token for *user_id*.

    :param user_id: the user's identification
    :param errors: incidental errors
    :param logger: optional logger
    :return: the uthentication tokem
    """
    # declare the return variable
    result: str

    # retrieve the token
    args: dict[str, Any] = {"user-id": user_id}
    with _iam_lock:
        result = user_token(iam_server=IamServer.IAM_KEYCLOAK,
                            args=args,
                            errors=errors,
                            logger=logger)
    return result
