#
# This file is part of AceQL Python Client SDK.
# AceQL Python Client SDK: Remote SQL access over HTTP with AceQL HTTP.
# Copyright (C) 2021,  KawanSoft SAS
# (http://www.kawansoft.com). All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
##
from aceql.proxy_auth import ProxyAuth


class ConnectionOptions(object):
    """Allows to create the Connection Options for the current session."""

    def __init__(self, *, proxies: dict = None, auth: ProxyAuth = None, session_id: str = None,
                 gzip_result: bool = False, timeout=None, request_headers: dict = None):
        """
            Creates the Connection Options for the current session

            Parameters
            ----------
            proxies : dict
                The proxy to use, can  be an authenticated proxy.
            auth : ProxyAuth
                aceql.ProxyAuth instance with proxy (username, password).
            session_id : str
                A session_id generated by AceQL login method on server. To be used for authentication without password.
            gzip_result : bool
                Says if Remote ResultSet will be zipped before returned to client side.
            timeout
                the HTTP connection timeout as defined in Requests library. None means no timeout is used
                (default value).
                See https://docs.python-requests.org/en/master/user/advanced/#timeouts.
            request_headers : dict
                the request headers to send to server for validation.

            Returns
            -------
            ConnectionOptions
                An object containing all connections options for this session
        """

        self.__proxies = proxies
        self.__auth = auth
        self.__session_id = session_id
        self.__gzip_result = gzip_result
        self.__timeout = timeout
        self.__request_headers = request_headers

    @property
    def proxies(self) -> str:
        return self.__proxies

    @property
    def auth(self) -> str:
        return self.__auth

    @property
    def session_id(self) -> str:
        return self.__session_id

    @property
    def gzip_result(self) -> bool:
        return self.__gzip_result

    @property
    def timeout(self):
        return self.__timeout

    @property
    def request_headers(self) -> dict:
        return self.__request_headers

    def __str__(self):
        """ The string representation."""
        return "ConnectionOptions [proxies=" + str(self.proxies) + ", auth=" + str(self.auth) \
               + ", session_id=" + str(self.session_id) \
               + ", gzip_result=" + str(self.gzip_result) + ", timeout=" + str(self.timeout) \
               + ", request_headers=" + str(self.request_headers) + "]"
