"""
Fixtures for ansible-sign tests
"""

import fileinput
import gnupg
import os
from pathlib import Path
import pytest
import shutil

from ansible_sign.signing import GPGSigner


def _gpg_home_with_secret_key(tmp_path, no_protection=False):
    """
    Creates a GPG home (in a temporary directory) and generates a private key
    inside of that GPG home.

    Returns the path to the GPG home.
    """
    home = tmp_path / "gpg-home"
    home.mkdir()
    gpg = gnupg.GPG(gnupghome=home)
    key_params = gpg.gen_key_input(
        key_length=2048,
        name_real="TEMPORARY ansible-sign TEST key",
        name_comment="Generated by ansible-sign test fixture",
        name_email="foo@example.com",
        passphrase="doYouEvenPassphrase" if no_protection is False else None,
        no_protection=no_protection,
    )
    gpg.gen_key(key_params)
    return home


@pytest.fixture
def gpg_home_with_secret_key(tmp_path):
    yield _gpg_home_with_secret_key(tmp_path)


@pytest.fixture
def gpg_home_with_secret_key_no_pass(tmp_path):
    yield _gpg_home_with_secret_key(tmp_path, no_protection=True)


@pytest.fixture
def gpg_home_with_hao_pubkey(tmp_path):
    home = tmp_path / "gpg-home-with-hao-pubkey"
    home.mkdir()
    gpg = gnupg.GPG(gnupghome=home)
    pubkey = open(os.path.join(os.path.dirname(os.path.abspath(__file__)), "fixtures", "gpgkeys", "hao_pubkey.txt"), "r").read()
    gpg.import_keys(pubkey)
    yield home


@pytest.fixture
def unsigned_project_with_checksum_manifest(tmp_path):
    """
    Creates a project directory (at a temporary location) with a generated, but
    unsigned, checksum manifest, ready for signing.

    Uses the 'manifest-success' checksum fixture directory as its base.
    """
    project = tmp_path / "project_root"
    shutil.copytree(
        os.path.join(
            os.path.dirname(os.path.abspath(__file__)),
            "fixtures",
            "checksum",
            "manifest-success",
        ),
        project,
        dirs_exist_ok=True,
    )
    yield project


@pytest.fixture
def unsigned_project_with_broken_checksum_manifest(unsigned_project_with_checksum_manifest):
    """
    Creates a project directory (at a temporary location) with a broken
    (syntactically invalid) checksum file.

    Uses the 'manifest-success' checksum fixture directory as its base and
    modifies the checksum manifest after copying.
    """
    manifest = unsigned_project_with_checksum_manifest / ".ansible-sign" / "sha256sum.txt"
    with fileinput.input(files=manifest, inplace=True) as f:
        for idx, line in enumerate(f):
            line = line.strip()
            if idx == 0:
                print(line.replace("  ", ""))
            else:
                print(line)
    yield unsigned_project_with_checksum_manifest


@pytest.fixture
def unsigned_project_with_broken_symlink(unsigned_project_with_checksum_manifest):
    """
    Creates a project directory (at a temporary location) with a broken
    symlink in the project root. This triggers a distlib.manifest bug and
    allows us to test our handling of it.
    """
    symlink = unsigned_project_with_checksum_manifest / "broken-symlink.txt"
    symlink.symlink_to(Path("/does/not/exist/and/never/will"))
    yield unsigned_project_with_checksum_manifest


@pytest.fixture
def unsigned_project_with_modified_checksum_manifest(unsigned_project_with_checksum_manifest):
    """
    Creates a project directory (at a temporary location) with a changed
    checksum file that contains wrong hashes.

    Uses the 'manifest-success' checksum fixture directory as its base and
    modifies the checksum manifest after copying.
    """
    manifest = unsigned_project_with_checksum_manifest / ".ansible-sign" / "sha256sum.txt"
    with fileinput.input(files=manifest, inplace=True) as f:
        for idx, line in enumerate(f):
            line = line.strip()
            if idx % 2 == 0:
                # Change some of the checksum lines.
                print(line.replace("2", "3").replace("a", "b").replace("7", "a").replace("c", "2"))
            else:
                print(line)
    yield unsigned_project_with_checksum_manifest


def _sign_project(gpg_home_with_secret_key, unsigned_project_with_checksum_manifest):
    """
    GPG-sign a project. Usually the arguments are temp directories produced by
    other fixtures above.
    """
    out = unsigned_project_with_checksum_manifest / ".ansible-sign" / "sha256sum.txt.sig"
    manifest_path = unsigned_project_with_checksum_manifest / ".ansible-sign" / "sha256sum.txt"
    signer = GPGSigner(
        manifest_path=manifest_path,
        output_path=out,
        passphrase="doYouEvenPassphrase",
        gpg_home=gpg_home_with_secret_key,
    )
    result = signer.sign()
    assert result.success is True
    assert os.path.exists(out)

    # now signed
    return (unsigned_project_with_checksum_manifest, gpg_home_with_secret_key)


@pytest.fixture
def signed_project_and_gpg(
    gpg_home_with_secret_key,
    unsigned_project_with_checksum_manifest,
):
    """
    Sign a project that has a valid manifest.
    """
    yield _sign_project(gpg_home_with_secret_key, unsigned_project_with_checksum_manifest)


@pytest.fixture
def signed_project_broken_manifest(
    gpg_home_with_secret_key,
    unsigned_project_with_broken_checksum_manifest,
):
    """
    Sign a project that has a broken manifest.
    """
    yield _sign_project(gpg_home_with_secret_key, unsigned_project_with_broken_checksum_manifest)


@pytest.fixture
def signed_project_modified_manifest(
    gpg_home_with_secret_key,
    unsigned_project_with_modified_checksum_manifest,
):
    """
    Sign a project that has a modified manifest.
    """
    yield _sign_project(gpg_home_with_secret_key, unsigned_project_with_modified_checksum_manifest)


@pytest.fixture
def signed_project_missing_manifest(
    gpg_home_with_secret_key,
    unsigned_project_with_checksum_manifest,
):
    """
    Sign a project that has a broken manifest.
    """
    (project, gpghome) = _sign_project(gpg_home_with_secret_key, unsigned_project_with_checksum_manifest)
    manifest = project / ".ansible-sign" / "sha256sum.txt"
    os.remove(manifest)
    yield (project, gpghome)


@pytest.fixture
def signed_project_with_different_gpg_home(
    gpg_home_with_secret_key,
    gpg_home_with_hao_pubkey,
    unsigned_project_with_checksum_manifest,
):
    """
    Sign a project but 'lose' the key it was signed with by returning an
    unrelated gnupg home directory.
    """
    (project, gpghome) = _sign_project(gpg_home_with_secret_key, unsigned_project_with_checksum_manifest)
    yield (project, gpg_home_with_hao_pubkey)
