# SPDX-License-Identifier: Apache-2.0
#
# Copyright (C) 2019, Arm Limited and contributors.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import os
import os.path
import abc

from lisa.wlgen.rta import RTAPhase, PeriodicWload
from lisa.tests.base import TestBundleBase, TestBundle, ResultBundle, RTATestBundle, AggregatedResultBundle
from lisa.trace import requires_events
from lisa.target import Target
from lisa.utils import ArtifactPath, kwargs_forwarded_to
from lisa.analysis.frequency import FrequencyAnalysis
from lisa.analysis.tasks import TasksAnalysis


class SchedTuneItemBase(RTATestBundle, TestBundle):
    """
    Abstract class enabling rtapp execution in a schedtune group

    :param boost: The boost level to set for the cgroup
    :type boost: int

    :param prefer_idle: The prefer_idle flag to set for the cgroup
    :type prefer_idle: bool
    """

    def __init__(self, res_dir, plat_info, boost, prefer_idle):
        super().__init__(res_dir, plat_info)
        self.boost = boost
        self.prefer_idle = prefer_idle

    @property
    def cgroup_configuration(self):
        return self.get_cgroup_configuration(self.plat_info, self.boost, self.prefer_idle)

    @classmethod
    def get_cgroup_configuration(cls, plat_info, boost, prefer_idle):
        attributes = {
            'boost': boost,
            'prefer_idle': int(prefer_idle)
        }
        return {'name': 'lisa_test',
                'controller': 'schedtune',
                'attributes': attributes}

    @classmethod
    # Not annotated, to prevent exekall from picking it up. See
    # SchedTuneBase.from_target
    def _from_target(cls, target, *, res_dir, boost, prefer_idle, collector=None):
        plat_info = target.plat_info
        rtapp_profile = cls.get_rtapp_profile(plat_info)
        cgroup_config = cls.get_cgroup_configuration(plat_info, boost, prefer_idle)
        cls.run_rtapp(target, res_dir, rtapp_profile, collector=collector, cg_cfg=cgroup_config)

        return cls(res_dir, plat_info, boost, prefer_idle)


class SchedTuneBase(TestBundleBase):
    """
    Abstract class enabling the aggregation of ``SchedTuneItemBase``

    :param test_bundles: a list of test bundles generated by
        multiple ``SchedTuneItemBase`` instances
    :type test_bundles: list
    """

    def __init__(self, res_dir, plat_info, test_bundles):
        super().__init__(res_dir, plat_info)

        self.test_bundles = test_bundles

    @classmethod
    @kwargs_forwarded_to(
        SchedTuneItemBase._from_target,
        ignore=[
            'boost',
            'prefer_idle',
        ]
    )
    def _from_target(cls, target: Target, *, res_dir: ArtifactPath = None,
        collector=None, **kwargs) -> 'SchedTuneBase':
        """
        Creates a SchedTuneBase bundle from the target.
        """
        return cls(res_dir, target.plat_info,
            list(cls._create_test_bundles(target, res_dir, **kwargs))
        )

    @classmethod
    @abc.abstractmethod
    def _create_test_bundles(cls, target, res_dir, **kwargs):
        """
        Collects and yields a :class:`lisa.tests.base.ResultBundle` per test
        item.
        """
        pass

    @classmethod
    def _create_test_bundle_item(cls, target, res_dir, item_cls,
                                 boost, prefer_idle, **kwargs):
        """
        Creates and returns a TestBundle for a given item class, and a given
        schedtune configuration
        """
        item_dir = ArtifactPath.join(res_dir, f'boost_{boost}_prefer_idle_{int(prefer_idle)}')
        os.makedirs(item_dir)

        logger = cls.get_logger()
        logger.info(f'Running {item_cls.__name__} with boost={boost}, prefer_idle={prefer_idle}')
        return item_cls.from_target(target,
            boost=boost,
            prefer_idle=prefer_idle,
            res_dir=item_dir,
            **kwargs,
        )


class SchedTuneFreqItem(SchedTuneItemBase):
    """
    Runs a tiny RT rtapp task pinned to a big CPU at a given boost level and
    checks the frequency selection was performed accordingly.
    """

    @classmethod
    def _get_rtapp_profile(cls, plat_info):
        cpu = plat_info['capacity-classes'][-1][0]
        return {
            'stune': RTAPhase(
                prop_wload=PeriodicWload(
                    # very small task, no impact on freq w/o boost
                    duty_cycle_pct=1,
                    duration=10,
                    period=cls.TASK_PERIOD,
                ),
                # pin to big CPU, to focus on frequency selection
                prop_cpus=[cpu],
                # RT tasks have the boost holding feature so the frequency
                # should be more stable, and we shouldn't go to max freq in
                # Android
                prop_policy='SCHED_FIFO'
            )
        }

    @FrequencyAnalysis.df_cpu_frequency.used_events
    @requires_events(SchedTuneItemBase.trace_window.used_events, "cpu_frequency")
    def trace_window(self, trace):
        """
        Set the boundaries of the trace window to ``cpu_frequency`` events
        before/after the task's start/end time
        """
        rta_start, rta_stop = super().trace_window(trace)

        cpu = self.plat_info['capacity-classes'][-1][0]
        freq_df = trace.ana.frequency.df_cpu_frequency(cpu)

        # Find the frequency events before and after the task runs
        freq_start = freq_df[freq_df.index < rta_start].index[-1]
        freq_stop = freq_df[freq_df.index > rta_stop].index[0]

        return (freq_start, freq_stop)

    @FrequencyAnalysis.get_average_cpu_frequency.used_events
    def test_stune_frequency(self, freq_margin_pct=10) -> ResultBundle:
        """
        Test that frequency selection followed the boost

        :param: freq_margin_pct: Allowed margin between estimated and measured
            average frequencies
        :type freq_margin_pct: int

        Compute the expected frequency given the boost level and compare to the
        real average frequency from the trace.
        Check that the difference between expected and measured frequencies is
        no larger than ``freq_margin_pct``.
        """
        kernel_version = self.plat_info['kernel']['version']
        if kernel_version.parts[:2] < (4, 14):
            self.get_logger().warning(f'This test requires the RT boost hold, but it may be disabled in {kernel_version}')

        cpu = self.plat_info['capacity-classes'][-1][0]
        freqs = self.plat_info['freqs'][cpu]
        max_freq = max(freqs)

        # Estimate the target frequency, including sugov's margin, and round
        # into a real OPP
        boost = self.boost
        target_freq = min(max_freq, max_freq * boost / 80)
        target_freq = list(filter(lambda f: f >= target_freq, freqs))[0]

        # Get the real average frequency
        avg_freq = self.trace.ana.frequency.get_average_cpu_frequency(cpu)

        distance = abs(target_freq - avg_freq) * 100 / target_freq
        res = ResultBundle.from_bool(distance < freq_margin_pct)
        res.add_metric("target freq", target_freq, 'kHz')
        res.add_metric("average freq", avg_freq, 'kHz')
        res.add_metric("boost", boost, '%')

        return res


class SchedTuneFrequencyTest(SchedTuneBase):
    """
    Runs multiple ``SchedTuneFreqItem`` tests at various boost levels ranging
    from 20% to 100%, then checks all succedeed.
    """

    @classmethod
    def _create_test_bundles(cls, target, res_dir, **kwargs):
        for boost in range(20, 101, 20):
            yield cls._create_test_bundle_item(
                target=target,
                res_dir=res_dir,
                item_cls=SchedTuneFreqItem,
                boost=boost,
                prefer_idle=False,
                **kwargs
            )

    def test_stune_frequency(self, freq_margin_pct=10) -> AggregatedResultBundle:
        """
        .. seealso:: :meth:`SchedTuneFreqItem.test_stune_frequency`
        """
        item_res_bundles = [
            item.test_stune_frequency(freq_margin_pct)
            for item in self.test_bundles
        ]
        return AggregatedResultBundle(item_res_bundles, 'boost')


class SchedTunePlacementItem(SchedTuneItemBase):
    """
    Runs a tiny RT-App task marked 'prefer_idle' at a given boost level and
    tests if it was placed on big-enough CPUs.
    """

    @classmethod
    def _get_rtapp_profile(cls, plat_info):
        return {
            'stune': RTAPhase(
                prop_wload=PeriodicWload(
                    duty_cycle_pct=1,
                    duration=3,
                    period=cls.TASK_PERIOD,
                )
            )
        }

    @TasksAnalysis.df_task_total_residency.used_events
    def test_stune_task_placement(self, bad_cpu_margin_pct=10) -> ResultBundle:
        """
        Test that the task placement satisfied the boost requirement

        Check that top-app tasks spend no more than ``bad_cpu_margin_pct`` of
        their time on CPUs that don't have enough capacity to serve their
        boost.
        """
        assert len(self.rtapp_tasks) == 1
        task = self.rtapp_tasks[0]
        df = self.trace.ana.tasks.df_task_total_residency(task)

        # Find CPUs without enough capacity to meet the boost
        boost = self.boost
        cpu_caps = self.plat_info['cpu-capacities']['rtapp']
        ko_cpus = list(filter(lambda x: (cpu_caps[x] / 10.24) < boost, cpu_caps))

        # Count how much time was spend on wrong CPUs
        time_ko = 0
        total_time = 0
        for cpu in cpu_caps:
            t = df['runtime'][cpu]
            if cpu in ko_cpus:
                time_ko += t
            total_time += t

        pct_ko = time_ko * 100 / total_time
        res = ResultBundle.from_bool(pct_ko < bad_cpu_margin_pct)
        res.add_metric("time spent on inappropriate CPUs", pct_ko, '%')
        res.add_metric("boost", boost, '%')

        return res


class SchedTunePlacementTest(SchedTuneBase):
    """
    Runs multiple ``SchedTunePlacementItem`` tests with prefer_idle set and
    typical top-app boost levels, then checks all succedeed.
    """

    @classmethod
    def _create_test_bundles(cls, target, res_dir, **kwargs):
        # Typically top-app tasks are boosted by 10%, or 50% during touchboost
        for boost in [10, 50]:
            yield cls._create_test_bundle_item(
                target=target,
                res_dir=res_dir,
                item_cls=SchedTunePlacementItem,
                boost=boost,
                prefer_idle=True,
                **kwargs
            )

    def test_stune_task_placement(self, margin_pct=10) -> AggregatedResultBundle:
        """
        .. seealso:: :meth:`SchedTunePlacementItem.test_stune_task_placement`
        """
        item_res_bundles = [
            item.test_stune_task_placement(margin_pct)
            for item in self.test_bundles
        ]
        return AggregatedResultBundle(item_res_bundles, 'boost')

# vim :set tabstop=4 shiftwidth=4 textwidth=80 expandtab
