import logging
import os
from inspect import stack
from pathlib import Path

import click
from click import BadParameter


_logger: logging.Logger = logging.getLogger(__name__)


@click.command('dp-demo')
@click.argument('file', type=str)
@click.pass_context
def dp_demo_date_predictability(ctx_context, file):
    """
    Demonstration of how to use commands related to Date Predictability.

    FILE is the path to persist the outputs generated by the Date Predictability commands, either an existing directory
    or an non-existing 'xlsx' file in an existing directory
    """
    _logger.debug(f"Function '{stack()[0].filename} - {stack()[0].function}' is called")

    p_the_file_path: Path = Path(file)
    if not p_the_file_path.is_absolute():
         p_the_file_path = Path(os.getcwd()).joinpath(p_the_file_path)

    # If the file path is an existing directory
    if p_the_file_path.is_dir():
        if p_the_file_path.exists():
            p_the_file_path = p_the_file_path.joinpath("com.enovation.toolbox.predictability.bean.xlsx")
        else:
            raise BadParameter(f"Directory '{p_the_file_path}' does not exist. Please make sure you provide a path to "
                               f"an existing directory.")

    # If the file path is an existing file
    elif p_the_file_path.is_file():
        raise BadParameter(f"File '{p_the_file_path}' does exist. Please make sure you provide a path to a file that "
                           f"does not exist.")

    # If the file path is neither an existing directory, nor an existing file
    else:
        if not p_the_file_path.resolve().parents[0].is_dir():
            raise BadParameter(f"Directory '{p_the_file_path.resolve().parents[0]}' does not exist. Please make sure "
                               f"you provide a path pointing to an existing directory.")
        elif p_the_file_path.suffix != ".xlsx":
            raise BadParameter(f"File '{p_the_file_path.name}' should have extension 'xlsx', not "
                               f"'{p_the_file_path.suffix}'. Please make sure you provide the proper extension.")

    click.echo(f"**************************"
               f"\n* Copy past in your terminal the command generated below, that will do the following:"
               f"\n*\t- Load a demo csv file '{Path(__file__).resolve().parents[0].joinpath('dp_demo.csv')}'"
               f"\n*\t- Rename columns"
               f"\n*\t- Compute date predictability"
               f"\n*\t- Persist the outputs into an excel spreadsheet '{p_the_file_path}'"
               f"\n*\t- Launch a dash server to graph the outputs"
               f"\n**************************")
    click.echo(
        f"enov --verbose "
               
        f"load-csv -c 'Project and phase' -c 'Projection date' -c 'Projected ETA' "
        f"\"{Path(__file__).resolve().parents[0].joinpath('dp_demo.csv')}\" df_raw "
       
        f"rename-columns -o df_renamed df_raw "
        f"'{{\"Project and phase\":\"key\",\"Projection date\":\"date\",\"Projected ETA\":\"measure\"}}' "
        
        f"dp_compute -o df_predictability df_renamed "
        
        f"dp_persist df_predictability \"{p_the_file_path}\" "

        f"dp_graph df_predictability")

    click.echo(f"**************************"
               f"\n* In case 'enov' is not recognized, copy past the following command..."
               f"\n**************************")
    click.echo(
        f"python3 \"{Path(__file__).resolve().parents[5].joinpath('enov.py')}\" --verbose "
               
        f"load-csv -c 'Project and phase' -c 'Projection date' -c 'Projected ETA' "
        f"\"{Path(__file__).resolve().parents[0].joinpath('dp_demo.csv')}\" df_raw "
       
        f"rename-columns -o df_renamed df_raw "
        f"'{{\"Project and phase\":\"key\",\"Projection date\":\"date\",\"Projected ETA\":\"measure\"}}' "
        
        f"dp_compute -o df_predictability df_renamed "
        
        f"dp_persist df_predictability \"{p_the_file_path}\" "

        f"dp_graph df_predictability")

    _logger.debug(f"Function '{stack()[0].filename} - {stack()[0].function}' is returning")
