import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as iam from '@aws-cdk/aws-iam';
import { Construct, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { EnableScalingProps, IScalableTableAttribute } from './scalable-attribute-api';
export interface Attribute {
    /**
     * The name of an attribute.
     */
    readonly name: string;
    /**
     * The data type of an attribute.
     */
    readonly type: AttributeType;
}
export interface TableOptions {
    /**
     * Partition key attribute definition.
     */
    readonly partitionKey: Attribute;
    /**
     * Table sort key attribute definition.
     *
     * @default no sort key
     */
    readonly sortKey?: Attribute;
    /**
     * The read capacity for the table. Careful if you add Global Secondary Indexes, as
     * those will share the table's provisioned throughput.
     *
     * Can only be provided if billingMode is Provisioned.
     *
     * @default 5
     */
    readonly readCapacity?: number;
    /**
     * The write capacity for the table. Careful if you add Global Secondary Indexes, as
     * those will share the table's provisioned throughput.
     *
     * Can only be provided if billingMode is Provisioned.
     *
     * @default 5
     */
    readonly writeCapacity?: number;
    /**
     * Specify how you are charged for read and write throughput and how you manage capacity.
     *
     * @default PROVISIONED if `replicationRegions` is not specified, PAY_PER_REQUEST otherwise
     */
    readonly billingMode?: BillingMode;
    /**
     * Whether point-in-time recovery is enabled.
     * @default - point-in-time recovery is disabled
     */
    readonly pointInTimeRecovery?: boolean;
    /**
     * Whether server-side encryption with an AWS managed customer master key is enabled.
     * @default - server-side encryption is enabled with an AWS owned customer master key
     */
    readonly serverSideEncryption?: boolean;
    /**
     * The name of TTL attribute.
     * @default - TTL is disabled
     */
    readonly timeToLiveAttribute?: string;
    /**
     * When an item in the table is modified, StreamViewType determines what information
     * is written to the stream for this table.
     *
     * @default - streams are disabled unless `replicationRegions` is specified
     */
    readonly stream?: StreamViewType;
    /**
     * The removal policy to apply to the DynamoDB Table.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Regions where replica tables will be created
     *
     * @default - no replica tables are created
     * @experimental
     */
    readonly replicationRegions?: string[];
}
export interface TableProps extends TableOptions {
    /**
     * Enforces a particular physical table name.
     * @default <generated>
     */
    readonly tableName?: string;
}
export interface SecondaryIndexProps {
    /**
     * The name of the secondary index.
     */
    readonly indexName: string;
    /**
     * The set of attributes that are projected into the secondary index.
     * @default ALL
     */
    readonly projectionType?: ProjectionType;
    /**
     * The non-key attributes that are projected into the secondary index.
     * @default - No additional attributes
     */
    readonly nonKeyAttributes?: string[];
}
export interface GlobalSecondaryIndexProps extends SecondaryIndexProps {
    /**
     * The attribute of a partition key for the global secondary index.
     */
    readonly partitionKey: Attribute;
    /**
     * The attribute of a sort key for the global secondary index.
     * @default - No sort key
     */
    readonly sortKey?: Attribute;
    /**
     * The read capacity for the global secondary index.
     *
     * Can only be provided if table billingMode is Provisioned or undefined.
     *
     * @default 5
     */
    readonly readCapacity?: number;
    /**
     * The write capacity for the global secondary index.
     *
     * Can only be provided if table billingMode is Provisioned or undefined.
     *
     * @default 5
     */
    readonly writeCapacity?: number;
}
export interface LocalSecondaryIndexProps extends SecondaryIndexProps {
    /**
     * The attribute of a sort key for the local secondary index.
     */
    readonly sortKey: Attribute;
}
/**
 * An interface that represents a DynamoDB Table - either created with the CDK, or an existing one.
 */
export interface ITable extends IResource {
    /**
     * Arn of the dynamodb table.
     *
     * @attribute
     */
    readonly tableArn: string;
    /**
     * Table name of the dynamodb table.
     *
     * @attribute
     */
    readonly tableName: string;
    /**
     * ARN of the table's stream, if there is one.
     *
     * @attribute
     */
    readonly tableStreamArn?: string;
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Metric for the number of Errors executing all Lambdas
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed read capacity units
     *
     * @param props properties of a metric
     */
    metricConsumedReadCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed write capacity units
     *
     * @param props properties of a metric
     */
    metricConsumedWriteCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the system errors
     *
     * @param props properties of a metric
     */
    metricSystemErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the user errors
     *
     * @param props properties of a metric
     */
    metricUserErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the conditional check failed requests
     *
     * @param props properties of a metric
     */
    metricConditionalCheckFailedRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the successful request latency
     *
     * @param props properties of a metric
     */
    metricSuccessfulRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Reference to a dynamodb table.
 */
export interface TableAttributes {
    /**
     * The ARN of the dynamodb table.
     * One of this, or {@link tabeName}, is required.
     *
     * @default - no table arn
     */
    readonly tableArn?: string;
    /**
     * The table name of the dynamodb table.
     * One of this, or {@link tabeArn}, is required.
     *
     * @default - no table name
     */
    readonly tableName?: string;
    /**
     * The ARN of the table's stream.
     *
     * @default - no table stream
     */
    readonly tableStreamArn?: string;
}
declare abstract class TableBase extends Resource implements ITable {
    /**
     * @attribute
     */
    abstract readonly tableArn: string;
    /**
     * @attribute
     */
    abstract readonly tableName: string;
    /**
     * @attribute
     */
    abstract readonly tableStreamArn?: string;
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Table
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed read capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedReadCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the consumed write capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedWriteCapacityUnits(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the system errors this table
     *
     * @default sum over a minute
     */
    metricSystemErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the user errors this table
     *
     * @default sum over a minute
     */
    metricUserErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the conditional check failed requests this table
     *
     * @default sum over a minute
     */
    metricConditionalCheckFailedRequests(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the successful request latency this table
     *
     * @default avg over a minute
     */
    metricSuccessfulRequestLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    protected abstract get hasIndex(): boolean;
}
/**
 * Provides a DynamoDB table.
 */
export declare class Table extends TableBase {
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee: iam.IGrantable): iam.Grant;
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope: Construct, id: string, tableName: string): ITable;
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope: Construct, id: string, tableArn: string): ITable;
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope: Construct, id: string, attrs: TableAttributes): ITable;
    /**
     * @attribute
     */
    readonly tableArn: string;
    /**
     * @attribute
     */
    readonly tableName: string;
    /**
     * @attribute
     */
    readonly tableStreamArn: string | undefined;
    private readonly table;
    private readonly keySchema;
    private readonly attributeDefinitions;
    private readonly globalSecondaryIndexes;
    private readonly localSecondaryIndexes;
    private readonly secondaryIndexNames;
    private readonly nonKeyAttributes;
    private readonly tablePartitionKey;
    private readonly tableSortKey?;
    private readonly billingMode;
    private readonly tableScaling;
    private readonly indexScaling;
    private readonly scalingRole;
    constructor(scope: Construct, id: string, props: TableProps);
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props: GlobalSecondaryIndexProps): void;
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props: LocalSecondaryIndexProps): void;
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName: string, props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName: string, props: EnableScalingProps): IScalableTableAttribute;
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    protected validate(): string[];
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    private validateProvisioning;
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    private validateIndexName;
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    private validateNonKeyAttributes;
    private buildIndexKeySchema;
    private buildIndexProjection;
    private findKey;
    private addKey;
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    private registerAttribute;
    /**
     * Return the role that will be used for AutoScaling
     */
    private makeScalingRole;
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    private createReplicaTables;
    /**
     * Whether this table has indexes
     */
    protected get hasIndex(): boolean;
}
export declare enum AttributeType {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    BINARY = "B",
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    NUMBER = "N",
    /** Up to 400KiB of UTF-8 encoded text */
    STRING = "S"
}
/**
 * DyanmoDB's Read/Write capacity modes.
 */
export declare enum BillingMode {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    PAY_PER_REQUEST = "PAY_PER_REQUEST",
    /**
     * Explicitly specified Read/Write capacity units.
     */
    PROVISIONED = "PROVISIONED"
}
export declare enum ProjectionType {
    /** Only the index and primary keys are projected into the index. */
    KEYS_ONLY = "KEYS_ONLY",
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    INCLUDE = "INCLUDE",
    /** All of the table attributes are projected into the index. */
    ALL = "ALL"
}
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
export declare enum StreamViewType {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    NEW_IMAGE = "NEW_IMAGE",
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    OLD_IMAGE = "OLD_IMAGE",
    /** Both the new and the old item images of the item are written to the stream. */
    NEW_AND_OLD_IMAGES = "NEW_AND_OLD_IMAGES",
    /** Only the key attributes of the modified item are written to the stream. */
    KEYS_ONLY = "KEYS_ONLY"
}
export {};
