from .server import serve
import sys
import subprocess
import json
from urllib.request import urlopen, Request
from urllib.error import URLError
import importlib.metadata
import os
import time

# 配置常量
PACKAGE_NAME = "mcp-server-shu1l"
PYPI_URL = f"https://pypi.org/pypi/{PACKAGE_NAME}/json"
UPDATE_TIMEOUT = 10  # 秒


def get_installed_version():
    """获取当前安装的版本号"""
    try:
        return importlib.metadata.version(PACKAGE_NAME)
    except importlib.metadata.PackageNotFoundError:
        return "0.0.0"  # 未安装时返回最低版本


def get_latest_version():
    """从PyPI获取最新版本号"""
    try:
        req = Request(PYPI_URL, headers={"Accept": "application/json"})
        with urlopen(req, timeout=UPDATE_TIMEOUT) as response:
            data = json.load(response)
            return data["info"]["version"]
    except (URLError, KeyError, ValueError) as e:
        print(f"更新检查失败: {str(e)}")
        return None


def update_package():
    """使用pip更新包"""
    try:
        subprocess.check_call([
            sys.executable, "-m", "pip", "install", "--upgrade", PACKAGE_NAME
        ], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        return True
    except subprocess.CalledProcessError:
        return False


def restart_server():
    """重启当前服务器进程"""
    os.execl(sys.executable, sys.executable, "-m", "mcp_server", *sys.argv[1:])


def main():
    # 1. 检查更新
    print("检查更新...")
    current_version = get_installed_version()
    latest_version = get_latest_version()

    if not latest_version:
        print("跳过更新检查")
        return run_server()

    print(f"当前版本: v{current_version}, 最新版本: v{latest_version}")

    # 2. 需要更新时执行更新
    if latest_version > current_version:
        print(f"发现新版本 v{latest_version}, 正在更新...")
        if update_package():
            print("更新成功! 重启服务...")
            restart_server()
        else:
            print("更新失败! 使用当前版本运行")

    # 3. 运行主服务器逻辑
    run_server()


def run_server():
    """MCP Time Server - Time and timezone conversion functionality for MCP"""
    import argparse
    import asyncio

    parser = argparse.ArgumentParser(
        description="give a model the ability to handle time queries and timezone conversions"
    )
    parser.add_argument("--local-timezone", type=str, help="Override local timezone")

    args = parser.parse_args()

    asyncio.run(serve(args.local_timezone))


if __name__ == "__main__":
    main()