#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# CERT UEFI Parser
#
# Copyright 2025 Carnegie Mellon University.
#
# NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL
# IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO,
# WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED
# FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY
# KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
#
# Licensed under a BSD (SEI)-style license, please see license.txt or contact
# permission@sei.cmu.edu for full terms.
#
# [DISTRIBUTION STATEMENT A] This material has been approved for public release and
# unlimited distribution.  Please see Copyright notice for non-US Government use and
# distribution.
#
# This Software includes and/or makes use of Third-Party Software each subject to its own
# license.
#
# DM25-1401
"""
A database of GUIDs seen in UEFI ROMs.
"""

from .efiguid import EfiGuid, EfiGuidDb

GUID_DATABASE = EfiGuidDb({
    "00000000-0000-0000-0000-000000000000": EfiGuid(
        "00000000-0000-0000-0000-000000000000", "ZERO_GUID",
        description = """
        This GUID has lots of aliases, and it routinely reused to signify missing
        data in various contexts, which sometimes get their own aliases.
        """,
        aliases = ['EFI_PART_TYPE_UNUSED_GUID', 'FU_EFI_SIGNATURE_GUID_ZERO', 'NULL_GUID', 'NULL_ROM_FILE_GUID', 'TPM_DEVICE_INTERFACE_NONE', 'gEfiPartTypeUnusedGuid', 'gEfiTpmDeviceInstanceNoneGuid', 'ZeroGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/ZeroGuid.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/Gpt.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Csm/CsmSupportLib/LegacyPlatform.c",
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/SecurityPkg.dec",
            "https://github.com/torvalds/linux/blob/master/include/linux/efi.h",
        ],
        vendors = ['EDK2', 'Linux'],
        filenames = [],
        hashes = [],
        junk = []),

    "00000000-0ed6-4e09-bbf9-28325db6ba30": EfiGuid(
        "00000000-0ed6-4e09-bbf9-28325db6ba30", "BSODDxeRuntime",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00026aeb-f334-4c15-a7f0-e1e897e9fe91": EfiGuid(
        "00026aeb-f334-4c15-a7f0-e1e897e9fe91", "NvmeRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['NvmeRecovery'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '8091cabbb4de557d84e3dbd55126455a',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'e7a706c2421a55cde70e476d1d37f831', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "000ac7b6-f44d-4220-bd8b-521d30dea80f": EfiGuid(
        "000ac7b6-f44d-4220-bd8b-521d30dea80f", "HpGraphicsDetectDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "000f42d0-283c-46ab-9287-acb6c9941fcd": EfiGuid(
        "000f42d0-283c-46ab-9287-acb6c9941fcd", "SmmSupvErrorReport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0013be6b-2198-43f1-93ba-2a7ed7b1e1cc": EfiGuid(
        "0013be6b-2198-43f1-93ba-2a7ed7b1e1cc", "LENOVO_SYSTEM_USB_SWITCH_DXE_GUID",
        description = "",
        aliases = ['SystemUsbSwitchDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00160f8d-2b35-4df2-bbe0-b272a8d631f0": EfiGuid(
        "00160f8d-2b35-4df2-bbe0-b272a8d631f0", "FirmwarePerformanceDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/Acpi/FirmwarePerformanceDataTableDxe/FirmwarePerformanceDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "001d24d1-4ab1-b202-42d1-0fb9f4e4ad36": EfiGuid(
        "001d24d1-4ab1-b202-42d1-0fb9f4e4ad36", "BiosPowerOnPortingSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00214cc1-06d1-45fe-9700-dca5726ad7bf": EfiGuid(
        "00214cc1-06d1-45fe-9700-dca5726ad7bf", "ArmVirtPlatformLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0029231b-72ef-4fa5-bd37-7018a046fb0d": EfiGuid(
        "0029231b-72ef-4fa5-bd37-7018a046fb0d", "LenovoMultipleKeyboardService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0029de6a-e024-4eb8-a91d-9f23aa1f4e92": EfiGuid(
        "0029de6a-e024-4eb8-a91d-9f23aa1f4e92", "NetworkStackSetupScreen",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0029f8d3-76a1-4232-afe1-917c7b5e7cfd": EfiGuid(
        "0029f8d3-76a1-4232-afe1-917c7b5e7cfd", "SmmHeciDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SmmHeciDrv'],
        hashes = [
            '2b4ca0bd16a20893deff9632c23c2cca', '48bba53caf662f85dd73a1a247b45dc7',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'c0bdac2284b1e46c7c9c1509ddbea006',
        ],
        junk = []),

    "003443be-0997-42d9-bb92-fcdf6926dcee": EfiGuid(
        "003443be-0997-42d9-bb92-fcdf6926dcee", "FchSmbusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00364a4c-6a0a-4f08-8ffd-0937f3dbb13e": EfiGuid(
        "00364a4c-6a0a-4f08-8ffd-0937f3dbb13e", "IdeBusBoard",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['IdeBusBoard'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0049858f-8ca7-4ccd-918b-d952cbf32975": EfiGuid(
        "0049858f-8ca7-4ccd-918b-d952cbf32975", "VirtioFdtDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/VirtioFdtDxe/VirtioFdtDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "004e6cc2-d60f-4726-be22-d3d8020c4687": EfiGuid(
        "004e6cc2-d60f-4726-be22-d3d8020c4687", "AsusSmartHsioDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00504624-8a59-4eeb-bd0f-6b36e96128e0": EfiGuid(
        "00504624-8a59-4eeb-bd0f-6b36e96128e0", "FU_EFI_FIRMWARE_VOLUME_GUID_NVRAM_EVSA2",
        description = "",
        aliases = ['FpNvStorage'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['FpNvStorage']),

    "0053d9d6-2659-4599-a26b-ef4536e631a9": EfiGuid(
        "0053d9d6-2659-4599-a26b-ef4536e631a9", "SHELL_ALIAS_VARIABLE_GUID",
        description = "",
        aliases = ['ShellAliasGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellAliasGuid.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['ShellAlias', 'ShellAliasGuid']),

    "0062f7d1-f7db-44fa-abee-f7a3cc636e0b": EfiGuid(
        "0062f7d1-f7db-44fa-abee-f7a3cc636e0b", "PhStallPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0065b106-eb27-4a3c-851c-ec77d51edcd6": EfiGuid(
        "0065b106-eb27-4a3c-851c-ec77d51edcd6", "AbsoluteActivateDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0065d394-9951-4144-82a3-0afc8579c251": EfiGuid(
        "0065d394-9951-4144-82a3-0afc8579c251", "EFI_PEI_RSC_HANDLER_PPI_GUID",
        description = "",
        aliases = ['EfiPeiRscHandlerPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/ReportStatusCodeHandler.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0067835f-9a50-433a-8cbb-852078197814": EfiGuid(
        "0067835f-9a50-433a-8cbb-852078197814", "EfiCpuCsrAccess",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00720665-67eb-4a99-baf7-d3c33a1c7cc9": EfiGuid(
        "00720665-67eb-4a99-baf7-d3c33a1c7cc9", "EFI_TCP4_SERVICE_BINDING_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTcp4ServiceBindingProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Tcp4.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0072f4b5-74d5-4cc8-aff6-cc9469d33f7b": EfiGuid(
        "0072f4b5-74d5-4cc8-aff6-cc9469d33f7b", "UsbMediaCardReaderConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00781ca1-5de3-405f-abb8-379c3c076984": EfiGuid(
        "00781ca1-5de3-405f-abb8-379c3c076984", "AmiRomLayoutGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "007a2b20-8383-4cd2-af81-2994473434d8": EfiGuid(
        "007a2b20-8383-4cd2-af81-2994473434d8", "UsbOcUpdateDxeBigPineKey",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "007af6dc-bb4a-4d2a-8b57-a96a307275b5": EfiGuid(
        "007af6dc-bb4a-4d2a-8b57-a96a307275b5", "FJFP_PBA_64",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0086af89-ae1a-4435-a698-0815e850a004": EfiGuid(
        "0086af89-ae1a-4435-a698-0815e850a004", "AsusSetupReserveVariableItem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0090f816-86a1-4638-97c8-4d1bd79e7e5f": EfiGuid(
        "0090f816-86a1-4638-97c8-4d1bd79e7e5f", "SmcPeiReportStatusConOut",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00a2b266-b5cf-43a8-8547-cbebfdec894c": EfiGuid(
        "00a2b266-b5cf-43a8-8547-cbebfdec894c", "SmmSvcCallback",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00a480ce-d751-4b11-89fb-7dcebbb22f4d": EfiGuid(
        "00a480ce-d751-4b11-89fb-7dcebbb22f4d", "DellPmPolicyProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00b46ec9-2712-486b-a6a4-e2933581c28b": EfiGuid(
        "00b46ec9-2712-486b-a6a4-e2933581c28b", "HstiPlatformDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['HstiPlatformDxe'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '2949649fbed4688a06e89bf439cea6a1',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '540ab45d79fb417ade67cb3aac048180', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7f662f93e41d560c5fb667dce9000eb7',
            '869f701de5296248126c8c8a03a479c3', '953dc1131b7a579176b9fc937eee5846',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ee40c5722cf3b43608a71e2a12c72193',
        ],
        junk = []),

    "00b8e6b5-83ce-448a-8ac6-ccbc9d5f3be3": EfiGuid(
        "00b8e6b5-83ce-448a-8ac6-ccbc9d5f3be3", "FchTaishanSmmInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00c782d2-db8c-4a0b-9298-01d31f744206": EfiGuid(
        "00c782d2-db8c-4a0b-9298-01d31f744206", "PupdrDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PupdrDxe'],
        hashes = [
            '2b4ca0bd16a20893deff9632c23c2cca', '3555862f32138b911338fae3789c3921',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '48bba53caf662f85dd73a1a247b45dc7',
            '82060a1dae992f24609d2f673cb58cfb', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b4fa3a413d748be6699112adc0c14b0d', 'c0bdac2284b1e46c7c9c1509ddbea006',
        ],
        junk = []),

    "00c7d289-1347-4de0-bf42-0e269d0ef34a": EfiGuid(
        "00c7d289-1347-4de0-bf42-0e269d0ef34a", "PCH_SPI_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PCH_SPI_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "00c86db8-013b-4ff4-b8e9-208f4fcf1c00": EfiGuid(
        "00c86db8-013b-4ff4-b8e9-208f4fcf1c00", "LibSignal",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00ca959f-6cfa-4db1-95bc-e46c47514390": EfiGuid(
        "00ca959f-6cfa-4db1-95bc-e46c47514390", "EFI_TLS_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTlsProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Tls.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "00cf32a8-495c-3ed8-7c68-e9bb86810ee0": EfiGuid(
        "00cf32a8-495c-3ed8-7c68-e9bb86810ee0", "RedfishPlatformCredentialLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmulatorPkg/Library/RedfishPlatformCredentialLib/RedfishPlatformCredentialLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "00d24382-8231-4b18-a4f0-2d94d8fe2e81": EfiGuid(
        "00d24382-8231-4b18-a4f0-2d94d8fe2e81", "SmmPciExpressLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/SmmPciExpressLib/SmmPciExpressLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "00d6b14b-7dd0-4062-8821-e5f96a2a1b00": EfiGuid(
        "00d6b14b-7dd0-4062-8821-e5f96a2a1b00", "FspReservedMemoryResourceHobMiscGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00d6e846-49c7-4517-ac79-a70e8bc0553c": EfiGuid(
        "00d6e846-49c7-4517-ac79-a70e8bc0553c", "AmdMemChanXLatSspPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00dbd91d-55e9-420f-9639-5e9f8437b44f": EfiGuid(
        "00dbd91d-55e9-420f-9639-5e9f8437b44f", "EFI_EXTENDED_SAL_STATUS_CODE_SERVICES_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiExtendedSalStatusCodeServicesProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00dc20a3-66a2-4d14-bbd7-5ba938e556de": EfiGuid(
        "00dc20a3-66a2-4d14-bbd7-5ba938e556de", "LenovoSmbios",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00e245ab-f12e-43f0-96db-09aaae8d2ea6": EfiGuid(
        "00e245ab-f12e-43f0-96db-09aaae8d2ea6", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['1ded807f42af76804c9a0bbb6e24a2a9'],
        junk = []),

    "00e98021-f4fe-46cc-ab2d-894c373afa01": EfiGuid(
        "00e98021-f4fe-46cc-ab2d-894c373afa01", "DXE_ENHANCED_SPEEDSTEP_PROTOCOL_GUID",
        description = "",
        aliases = ['DxeEnhancedSpeedstepProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00eeab3b-4719-432d-a2dc-c97a2c62bfdd": EfiGuid(
        "00eeab3b-4719-432d-a2dc-c97a2c62bfdd", "DellPermDevConfigPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00f02769-ae45-41cf-aa58-6377b87ba99a": EfiGuid(
        "00f02769-ae45-41cf-aa58-6377b87ba99a", "DeviceLayoutLoadPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "00f93d8c-00a6-42d0-9327-11ce309b944a": EfiGuid(
        "00f93d8c-00a6-42d0-9327-11ce309b944a", "OpalPasswordSupportLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "010216cd-9c09-4eb5-b7da-d0a2865092d4": EfiGuid(
        "010216cd-9c09-4eb5-b7da-d0a2865092d4", "ProjectDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0102add8-037f-46fc-a9f2-41d352d2f695": EfiGuid(
        "0102add8-037f-46fc-a9f2-41d352d2f695", "TouchPad_Elan_2nd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "010574d3-9ca7-4265-a536-d10e7a8cdcb0": EfiGuid(
        "010574d3-9ca7-4265-a536-d10e7a8cdcb0", "OemSolPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01190654-fed0-40d3-ba7f-2925539e5830": EfiGuid(
        "01190654-fed0-40d3-ba7f-2925539e5830", "BaseS3SmbusLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseS3SmbusLib/BaseS3SmbusLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "011a7546-db4e-4119-a216-9a3167b6af56": EfiGuid(
        "011a7546-db4e-4119-a216-9a3167b6af56", "ComputraceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01237498-4e20-42e5-bf75-1cd3b20f7ce3": EfiGuid(
        "01237498-4e20-42e5-bf75-1cd3b20f7ce3", "PlatformStatusCodeHandlerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01239999-fc0e-4b6e-9e79-d54d5db6cd20": EfiGuid(
        "01239999-fc0e-4b6e-9e79-d54d5db6cd20", "EfiServerMgmtSetupVariable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0127b951-e840-480a-a083-087a9ae17353": EfiGuid(
        "0127b951-e840-480a-a083-087a9ae17353", "LEMSetVariableCtlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013295ab-6680-4c41-9c98-c41aa6f5188b": EfiGuid(
        "013295ab-6680-4c41-9c98-c41aa6f5188b", "FlabLoadUadmDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013464b6-ac20-4e86-a2f9-23cc4650ddc2": EfiGuid(
        "013464b6-ac20-4e86-a2f9-23cc4650ddc2", "MemoryDiagnostic",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0135229a-ebb5-4a21-957d-1d20057cf751": EfiGuid(
        "0135229a-ebb5-4a21-957d-1d20057cf751", "TCMPEI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['TCMPEI'],
        hashes = [
            '4002d2fb07cfbaed2e86d9608824378a', '5163727ab3ed5f9e02330a0c98095da0',
            '7b3f08b9dac97b5334c6851820b9a1b0',
        ],
        junk = []),

    "0135229a-ebb5-4a21-957d-1d20057cf752": EfiGuid(
        "0135229a-ebb5-4a21-957d-1d20057cf752", "TCMDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['TCMDXE'],
        hashes = [
            '4002d2fb07cfbaed2e86d9608824378a', '5163727ab3ed5f9e02330a0c98095da0',
            '633e80e7347dc01c21c5133227df2aba', '7b3f08b9dac97b5334c6851820b9a1b0',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2',
        ],
        junk = []),

    "01359d99-9446-456d-ada4-50a711c03ada": EfiGuid(
        "01359d99-9446-456d-ada4-50a711c03ada", "CpuInitPei",
        description = "",
        aliases = ['CpuPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CpuInitPei', 'CpuInitPeim'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '3fa6831c92f133013c5bb0942f307f72', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "01359d99-9446-456d-ada4-50a711c03adb": EfiGuid(
        "01359d99-9446-456d-ada4-50a711c03adb", "BaseCacheLib",
        description = "",
        aliases = ['CpuPeim'],
        urls = [],
        vendors = [],
        filenames = ['BaseCacheLib', 'CpuPeim'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3555862f32138b911338fae3789c3921',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "01368881-c4ad-4b1d-b631-d57a8ec8db6b": EfiGuid(
        "01368881-c4ad-4b1d-b631-d57a8ec8db6b", "AmiGlobalVariableGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013a1234-db4e-3123-cea6-9a31982cae96": EfiGuid(
        "013a1234-db4e-3123-cea6-9a31982cae96", "DiskSanitizer",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013a65ac-db4e-3123-cea6-9a31982cae96": EfiGuid(
        "013a65ac-db4e-3123-cea6-9a31982cae96", "HddDiagnostic",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013a65ac-db4e-4119-cea6-9a31982cae96": EfiGuid(
        "013a65ac-db4e-4119-cea6-9a31982cae96", "ABCTDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "013a65ac-deac-3123-cea6-9a3198223e96": EfiGuid(
        "013a65ac-deac-3123-cea6-9a3198223e96", "MemTest",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0154dd51-9079-4a10-895c-9c0772815788": EfiGuid(
        "0154dd51-9079-4a10-895c-9c0772815788", "PlatformBootManagerStringPack",
        description = "",
        aliases = ['PlatformBootManagerStringPackGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0155db6a-3c05-46f6-9341-fefa4085e61e": EfiGuid(
        "0155db6a-3c05-46f6-9341-fefa4085e61e", "FjPostScreenMfgErrorCheckDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01631791-f34e-4c0a-b15d-0b6ce22b27a8": EfiGuid(
        "01631791-f34e-4c0a-b15d-0b6ce22b27a8", "TcgPeiAfterMem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['TcgPeiAfterMem'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '136ef7c1715a5ab03a93c66f1f9447e1', '3fa6831c92f133013c5bb0942f307f72',
            '6f7959935a87f450b05674b1e62ba7fa', '7457f65a90b756398ae360b93c326a5e',
            '756c574dcde55419446fb64425ebd1ff', '7d13012b3ccc7c61a0c88ca8969676f6',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9e80820994976fe9edc2b53346ea4f71',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'd4a0cd284c2f726df5db17c605809651', 'dbcaa4b015b1aec94558985a97dce7c2',
            'ea61df93fb3257dc78c2415817c1f871',
        ],
        junk = []),

    "01637772-b022-4047-bd52-35454c1c6b9f": EfiGuid(
        "01637772-b022-4047-bd52-35454c1c6b9f", "Armani_EcCommunicationSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0167ccc4-d0f7-4f21-a3ef-9e64b7cdce8b": EfiGuid(
        "0167ccc4-d0f7-4f21-a3ef-9e64b7cdce8b", "LENOVO_SYSTEM_SCSI_BUS_DXE_GUID",
        description = "",
        aliases = ['ScsiBus'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Scsi/ScsiBusDxe/ScsiBusDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['LENOVO_SYSTEM_SCSI_BUS_DXE_GUID', 'ScsiBus'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'bdbe423029a61611b078326db4953356',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['ScsiBus']),

    "0167ccc4-d0f7-4f21-a3ef-9e64b7cdce8c": EfiGuid(
        "0167ccc4-d0f7-4f21-a3ef-9e64b7cdce8c", "LENOVO_SYSTEM_AHCI_BUS_DXE_GUID",
        description = "",
        aliases = ['SystemAhciBusDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0170f60c-1d40-4651-956d-f0bd9879d527": EfiGuid(
        "0170f60c-1d40-4651-956d-f0bd9879d527", "Virtio10",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Virtio10Dxe/Virtio10.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0174b2b5-f505-4b12-aa60-59dff8d6ea37": EfiGuid(
        "0174b2b5-f505-4b12-aa60-59dff8d6ea37", "SHELL_NETWORK2_HII_GUID",
        description = "",
        aliases = ['ShellNetwork2HiiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellLibHiiGuid.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "017cb4b7-b80c-4040-b6c8-ea982bbb25b7": EfiGuid(
        "017cb4b7-b80c-4040-b6c8-ea982bbb25b7", "AMI_SMBUS_HOB_GUID",
        description = "",
        aliases = ['AmiSmbusHob'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "017ea01c-4988-4cd9-8a6c-3480e386a325": EfiGuid(
        "017ea01c-4988-4cd9-8a6c-3480e386a325", "LegacyStr",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "017f2629-2d3e-4ea9-9e59-88da4d98c027": EfiGuid(
        "017f2629-2d3e-4ea9-9e59-88da4d98c027", "SpiProtectionPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01806607-245b-47a8-952a-dc8c5c6a5316": EfiGuid(
        "01806607-245b-47a8-952a-dc8c5c6a5316", "OemDeviceDetect",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0182244e-f95d-43fc-91ec-60594ef47599": EfiGuid(
        "0182244e-f95d-43fc-91ec-60594ef47599", "Lpc47m18xDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01882424-63b1-4e30-90e8-4c2c344e89c0": EfiGuid(
        "01882424-63b1-4e30-90e8-4c2c344e89c0", "RecoveryCpuInitPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "018a5c7a-12eb-429d-9def-6fcc410b04e8": EfiGuid(
        "018a5c7a-12eb-429d-9def-6fcc410b04e8", "IioCfgUpdateDxeLightningRidgeEXECB4",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "018aa4f0-773e-46f3-b62d-2c70ea32e0d0": EfiGuid(
        "018aa4f0-773e-46f3-b62d-2c70ea32e0d0", "EfiSystemPartCapsuleStorage",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "018e1925-d6a2-4a2a-8958-817610a15adf": EfiGuid(
        "018e1925-d6a2-4a2a-8958-817610a15adf", "PeiS3LibNull",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "018e4d93-e8cf-4f51-9170-2658eca4c9e3": EfiGuid(
        "018e4d93-e8cf-4f51-9170-2658eca4c9e3", "AmdMiniRasStxKrkServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "018f3936-3a40-459d-9c82-3f1f78a90164": EfiGuid(
        "018f3936-3a40-459d-9c82-3f1f78a90164", "DellControlVaultDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0197ef5e-2ffc-4089-8e55-f70400b18146": EfiGuid(
        "0197ef5e-2ffc-4089-8e55-f70400b18146", "FSP_V_UPD_FFS_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "019fb1ca-d411-4948-b73c-4c054aba9e8e": EfiGuid(
        "019fb1ca-d411-4948-b73c-4c054aba9e8e", "FastBootFunctionEnabledHob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01a660d9-8009-4330-ba89-71b076cd5d0a": EfiGuid(
        "01a660d9-8009-4330-ba89-71b076cd5d0a", "EFI_SPI_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['EFI_SPI_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cfba6c85608b8135fb21024783573bf6', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "01ab1829-cecd-4cfa-a18c-ea75d66f3e74": EfiGuid(
        "01ab1829-cecd-4cfa-a18c-ea75d66f3e74", "INTEL_MEBX_PROTOCOL_GUID",
        description = "",
        aliases = ['IntelMebxProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['MEBX_PROTOCOL_GUID']),

    "01afca2f-5089-4559-bef3-6163ae15ae90": EfiGuid(
        "01afca2f-5089-4559-bef3-6163ae15ae90", "FRUSelfRepair",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01b95206-cd66-4c0d-a867-ed42960e07dc": EfiGuid(
        "01b95206-cd66-4c0d-a867-ed42960e07dc", "SYSTEM_FIRMWARE_DEVICE_SMM_PROTOCOL_GUID",
        description = "",
        aliases = ['SystemFirmwareDeviceSmmProtocol'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01c827c2-7765-42e9-b31c-e03e89a09113": EfiGuid(
        "01c827c2-7765-42e9-b31c-e03e89a09113", "AmdSocFp5RvPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01cf676f-6f01-4fa3-9789-b6322d890394": EfiGuid(
        "01cf676f-6f01-4fa3-9789-b6322d890394", "AmdCcxZen4Pei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01d127ea-f6f1-4ef6-9415-8a000093f89d": EfiGuid(
        "01d127ea-f6f1-4ef6-9415-8a000093f89d", "UEfiCorebootPayloadPkgTokenSpace",
        description = "",
        aliases = ['gUefiPayloadPkgTokenSpaceGuid', 'UEfiCorebootPayloadPkgTokenSpaceGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/UefiPayloadPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "01d6940e-06a2-455c-a0ca-dd380b3cea31": EfiGuid(
        "01d6940e-06a2-455c-a0ca-dd380b3cea31", "SDSecureFlash",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01d8f749-fd74-4f70-a393-6fed1d2c8d32": EfiGuid(
        "01d8f749-fd74-4f70-a393-6fed1d2c8d32", "AsusWmiToProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01def8e6-e998-4e27-8998-9cfab292bc50": EfiGuid(
        "01def8e6-e998-4e27-8998-9cfab292bc50", "SA_MISC_PEI_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01ed6c55-0d2e-4859-bb57-3044737a3679": EfiGuid(
        "01ed6c55-0d2e-4859-bb57-3044737a3679", "PhConSplitterDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01f34d25-4de2-23ad-3ff3-36353ff323f1": EfiGuid(
        "01f34d25-4de2-23ad-3ff3-36353ff323f1", "EFI_PEI_PCD_PPI_GUID",
        description = "",
        aliases = ['EfiPeiPcdPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/PiPcd.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_PCD_PPI_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "01f7ec2a-8f6e-4c51-94c2-de0e1c223f73": EfiGuid(
        "01f7ec2a-8f6e-4c51-94c2-de0e1c223f73", "RasAcpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "01fb5d53-4ff9-4ad9-adf5-7dc63659a1b2": EfiGuid(
        "01fb5d53-4ff9-4ad9-adf5-7dc63659a1b2", "PeiPciEnumeration",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02005acc-ad91-42b9-8636-403b4f5799c2": EfiGuid(
        "02005acc-ad91-42b9-8636-403b4f5799c2", "LenovoFlashDeviceInterfaceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02049744-32d9-4338-a64b-fdc42c836f9a": EfiGuid(
        "02049744-32d9-4338-a64b-fdc42c836f9a", "FchDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02049744-32d9-4338-cccb-fdc42c836f9c": EfiGuid(
        "02049744-32d9-4338-cccb-fdc42c836f9c", "FchPromontoryPlusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02061e01-bebd-4df5-81b6-dfc4f23d1d00": EfiGuid(
        "02061e01-bebd-4df5-81b6-dfc4f23d1d00", "SensorI2C",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "020bc7c9-80e5-476d-b187-0fb754850cab": EfiGuid(
        "020bc7c9-80e5-476d-b187-0fb754850cab", "PvnvramDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "020bcb33-7ea7-4e48-9af8-1b6ac52ce83f": EfiGuid(
        "020bcb33-7ea7-4e48-9af8-1b6ac52ce83f", "MeBackDoorSMI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "021722d8-522b-4079-852a-fe44c2c13f49": EfiGuid(
        "021722d8-522b-4079-852a-fe44c2c13f49", "SataController",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/SataControllerDxe/SataControllerDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "021b6156-3cc8-4e99-85ee-13d8a871edf2": EfiGuid(
        "021b6156-3cc8-4e99-85ee-13d8a871edf2", "ArmVirtMemoryInitPeiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/Library/ArmVirtMemoryInitPeiLib/ArmVirtMemoryInitPeiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "021bd2ca-51d2-11e3-8e56-b75417c70b44": EfiGuid(
        "021bd2ca-51d2-11e3-8e56-b75417c70b44", "UsbDeviceProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "021d723a-501c-4d1e-b792-0d3c4651b848": EfiGuid(
        "021d723a-501c-4d1e-b792-0d3c4651b848", "OemWwanSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "022218b8-fe5e-4ebc-bc96-74058a4e7e83": EfiGuid(
        "022218b8-fe5e-4ebc-bc96-74058a4e7e83", "ErrorGlobeBorder",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0226cd3f-69b5-4150-acbe-bfbf18e303d5": EfiGuid(
        "0226cd3f-69b5-4150-acbe-bfbf18e303d5", "EFI_WATCHDOG_TIMER_HOB_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02281cda-5ae2-4a27-93da-18ed4a545fbf": EfiGuid(
        "02281cda-5ae2-4a27-93da-18ed4a545fbf", "MRCErrorHandler",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02337e07-2cf2-46be-bcf7-834945fcceb9": EfiGuid(
        "02337e07-2cf2-46be-bcf7-834945fcceb9", "EfiUnbootablePartitionGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "023b3aa9-098f-4027-b83f-b6407da06569": EfiGuid(
        "023b3aa9-098f-4027-b83f-b6407da06569", "AmdPspFtpmDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0246e0b3-e8ea-4e03-b2a5-39fa937fd122": EfiGuid(
        "0246e0b3-e8ea-4e03-b2a5-39fa937fd122", "SsidPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "024d3127-7b60-48f4-a6fe-726e19cd4ceb": EfiGuid(
        "024d3127-7b60-48f4-a6fe-726e19cd4ceb", "PeiPlatformHelperLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "024dee41-33e7-11d3-9d69-0008c781f39f": EfiGuid(
        "024dee41-33e7-11d3-9d69-0008c781f39f", "EFI_PART_TYPE_LEGACY_MBR_GUID",
        description = "",
        aliases = ['EfiPartTypeLegacyMbrGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/Gpt.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02509c17-dd0b-4915-81f6-1de4ba954b14": EfiGuid(
        "02509c17-dd0b-4915-81f6-1de4ba954b14", "FanTableSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0250b201-273b-4a64-9cac-4fe9fb56f65e": EfiGuid(
        "0250b201-273b-4a64-9cac-4fe9fb56f65e", "AmdRAIDCoreDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0250d7ab-aade-4691-8909-8dc184822d1d": EfiGuid(
        "0250d7ab-aade-4691-8909-8dc184822d1d", "SmbiosTpmDeviceInfo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0253f9fa-129a-4a8d-b12e-7dc2b6376302": EfiGuid(
        "0253f9fa-129a-4a8d-b12e-7dc2b6376302", "dpDynamicCommand",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ShellPkg/DynamicCommand/DpDynamicCommand/DpDynamicCommand.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0255ded3-bf9c-4155-8ddb-6e8609ee0482": EfiGuid(
        "0255ded3-bf9c-4155-8ddb-6e8609ee0482", "AmdGopPlatformToDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "025bbfc7-e6a9-4b8b-82ad-6815a1aeaf4a": EfiGuid(
        "025bbfc7-e6a9-4b8b-82ad-6815a1aeaf4a", "LENOVO_MNP_DXE_BIN_GUID",
        description = "",
        aliases = ['MnpDxe'],
        urls = ['https://github.com/tianocore/edk2/blob/master/NetworkPkg/MnpDxe/MnpDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['LENOVO_MNP_DXE_BIN_GUID', 'MnpDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['MnpDxe']),

    "025f738b-4ebd-4d55-b728-5f421b601f1f": EfiGuid(
        "025f738b-4ebd-4d55-b728-5f421b601f1f", "PlatformInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PlatformInfoDxe'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '633e80e7347dc01c21c5133227df2aba',
            '811505684bd99c18f5ecbee1f725ff66', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b6cce665f85751125c3db2050ffc98d2',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "025f738b-4ebd-4d55-b728-5f421b601f20": EfiGuid(
        "025f738b-4ebd-4d55-b728-5f421b601f20", "PlatformCpuInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0273146c-96c4-45a1-a7af-78e0524a0ae2": EfiGuid(
        "0273146c-96c4-45a1-a7af-78e0524a0ae2", "AMI_BOARD_INFO_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiBoardInfoProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['AMI_BOARD_INFO_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "02805010-2591-4ed3-827b-a218f34ae0d7": EfiGuid(
        "02805010-2591-4ed3-827b-a218f34ae0d7", "DxePlatformHelperLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "028080a3-8958-4a62-a1a8-0fa1da162007": EfiGuid(
        "028080a3-8958-4a62-a1a8-0fa1da162007", "SmmCryptLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/CryptoPkg/Library/BaseCryptLib/SmmCryptLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02814228-57c8-405c-8a67-c928e4b50992": EfiGuid(
        "02814228-57c8-405c-8a67-c928e4b50992", "EepromInterfaceCoreSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0284f846-2cb7-4960-8db2-47aa71513ad5": EfiGuid(
        "0284f846-2cb7-4960-8db2-47aa71513ad5", "menu_selection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02955b8d-df90-4acb-b242-78d06b3fa6ca": EfiGuid(
        "02955b8d-df90-4acb-b242-78d06b3fa6ca", "TrackPointJYT",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "029d4c32-51e8-4866-bf51-fd0ed9a2dc55": EfiGuid(
        "029d4c32-51e8-4866-bf51-fd0ed9a2dc55", "DellBiosDimmLocation",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02a6de33-3ea9-4c17-8ea2-5681cc7afded": EfiGuid(
        "02a6de33-3ea9-4c17-8ea2-5681cc7afded", "VMDDxeEfi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02ac0805-155f-473a-a302-0d89d6e1e6cc": EfiGuid(
        "02ac0805-155f-473a-a302-0d89d6e1e6cc", "AsusSLP2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02b01ad5-7e59-43e8-a6d8-238180613a5a": EfiGuid(
        "02b01ad5-7e59-43e8-a6d8-238180613a5a", "EmuVariableRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02b2e27a-e8b5-4a42-82c3-55b43222eb9b": EfiGuid(
        "02b2e27a-e8b5-4a42-82c3-55b43222eb9b", "HpGFIStrsSubComp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02b3d5f2-ac28-11d3-9a2d-0090273fc14d": EfiGuid(
        "02b3d5f2-ac28-11d3-9a2d-0090273fc14d", "EFI_TCP_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTcpProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02b92d47-5227-4c18-9ff5-888550c395b9": EfiGuid(
        "02b92d47-5227-4c18-9ff5-888550c395b9", "MktmeLateInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02ba5cfc-d092-4404-9e52-a1b018525056": EfiGuid(
        "02ba5cfc-d092-4404-9e52-a1b018525056", "SiliconDataInitSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02bd55c2-ab1d-4b75-b0fd-9a63ae09b31d": EfiGuid(
        "02bd55c2-ab1d-4b75-b0fd-9a63ae09b31d", "BaseMemoryLibOptDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseMemoryLibOptDxe/BaseMemoryLibOptDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02c0ab0d-d5f0-4cad-a4bf-0cc40a554bbd": EfiGuid(
        "02c0ab0d-d5f0-4cad-a4bf-0cc40a554bbd", "HpInitNVSWksSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02c38de3-fb92-4b29-9430-ece6d0c0f3b0": EfiGuid(
        "02c38de3-fb92-4b29-9430-ece6d0c0f3b0", "FjMfgServicesDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02c40333-ced5-45ba-8836-b47e4c36489e": EfiGuid(
        "02c40333-ced5-45ba-8836-b47e4c36489e", "PlatformFanControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02c5a102-58ea-4e53-8be1-6ed1acaa708a": EfiGuid(
        "02c5a102-58ea-4e53-8be1-6ed1acaa708a", "AmdMemoryHobInfoPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02c8defd-9ee4-43e0-b8e7-d3245ad624af": EfiGuid(
        "02c8defd-9ee4-43e0-b8e7-d3245ad624af", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "02ce967a-dd7e-4ffc-9ee7-810cf0470880": EfiGuid(
        "02ce967a-dd7e-4ffc-9ee7-810cf0470880", "EFI_END_OF_DXE_EVENT_GROUP_GUID",
        description = "",
        aliases = ['EfiEndOfDxeEventGroupGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/EventGroup.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02d46af0-78a0-43fd-a9d6-25b094e0ba93": EfiGuid(
        "02d46af0-78a0-43fd-a9d6-25b094e0ba93", "LEN0130Driver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02d7dfe6-cb36-4ff9-9ed9-31806b797251": EfiGuid(
        "02d7dfe6-cb36-4ff9-9ed9-31806b797251", "FchSmmDiagDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02e800be-8f01-4aa6-946b-d71388e1833f": EfiGuid(
        "02e800be-8f01-4aa6-946b-d71388e1833f", "EFI_MTFTP4_SERVICE_BINDING_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02eea107-98db-400e-9830-460a1542d799": EfiGuid(
        "02eea107-98db-400e-9830-460a1542d799", "IP6_CONFIG_NVDATA_GUID",
        description = "",
        aliases = ['Ip6ConfigNvDataGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/NetworkPkg/Include/Guid/Ip6ConfigHii.h",
            "https://github.com/tianocore/edk2/blob/master/NetworkPkg/NetworkPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02eec6cf-fdf5-4ff0-a6af-df569bcd6828": EfiGuid(
        "02eec6cf-fdf5-4ff0-a6af-df569bcd6828", "HpDimmIdDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02eec6cf-fdf5-4ff0-a6af-df569bcd6829": EfiGuid(
        "02eec6cf-fdf5-4ff0-a6af-df569bcd6829", "HpDimmIdSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "02f04694-2c0a-4f1e-b0ce-64be25890b03": EfiGuid(
        "02f04694-2c0a-4f1e-b0ce-64be25890b03", "DebugAgentTimerLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Library/DebugAgentTimerLibNull/DebugAgentTimerLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "02f5dd2d-ad22-4ad0-8388-d1265eabdcb7": EfiGuid(
        "02f5dd2d-ad22-4ad0-8388-d1265eabdcb7", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "0301be48-7d25-443c-aa77-d5e753db416a": EfiGuid(
        "0301be48-7d25-443c-aa77-d5e753db416a", "EfiRsaBmcCommandsProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0302dcb7-a6a3-467c-b85e-f21db7e8533b": EfiGuid(
        "0302dcb7-a6a3-467c-b85e-f21db7e8533b", "SmmDispatchEngine",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03068297-dda5-4ccd-bbfa-7e09ae025177": EfiGuid(
        "03068297-dda5-4ccd-bbfa-7e09ae025177", "I2cTouchPanelDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0317f4ce-ad43-4461-8d14-b9f2543fd839": EfiGuid(
        "0317f4ce-ad43-4461-8d14-b9f2543fd839", "SataDriveInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0319c56b-c43a-42f1-80be-ca5bd1d5c928": EfiGuid(
        "0319c56b-c43a-42f1-80be-ca5bd1d5c928", "GRAPHICS_PEI_PRE_MEM_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0325a554-05be-466b-bc8c-70be3c9dafb1": EfiGuid(
        "0325a554-05be-466b-bc8c-70be3c9dafb1", "AmiTcg2InfoProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0325b5a1-0937-4a4f-b8af-ec3f80ee6b35": EfiGuid(
        "0325b5a1-0937-4a4f-b8af-ec3f80ee6b35", "SataControllerDxe",
        description = "",
        aliases = ['SataControllerLite'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0329d610-4269-d28f-61bf-b9a2d9fa9693": EfiGuid(
        "0329d610-4269-d28f-61bf-b9a2d9fa9693", "PERF_AMT_POST_MEM_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "032de3f1-1362-45a5-a5c2-44a0e63cc263": EfiGuid(
        "032de3f1-1362-45a5-a5c2-44a0e63cc263", "FjGabiSystemTableSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03327d04-c807-4b76-965f-b30046f15391": EfiGuid(
        "03327d04-c807-4b76-965f-b30046f15391", "VtioSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "033ff0be-abfb-40ac-87e3-c22a69bd7fa7": EfiGuid(
        "033ff0be-abfb-40ac-87e3-c22a69bd7fa7", "TransparentUnlockDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03417bf3-6d95-41df-88ef-0dda6e86dc34": EfiGuid(
        "03417bf3-6d95-41df-88ef-0dda6e86dc34", "AsfPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0341cf17-bc8f-4a20-ac28-6c3c324cd417": EfiGuid(
        "0341cf17-bc8f-4a20-ac28-6c3c324cd417", "ME_POLICY_HOB_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0347e9d2-3778-46dc-b8b5-f1687c703973": EfiGuid(
        "0347e9d2-3778-46dc-b8b5-f1687c703973", "FjWwanRfSettingUIDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "034aa642-91b0-4160-a0f7-d31a28e013cb": EfiGuid(
        "034aa642-91b0-4160-a0f7-d31a28e013cb", "H19DisplayControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03583ff6-cb36-4940-947e-b9b39f4afaf7": EfiGuid(
        "03583ff6-cb36-4940-947e-b9b39f4afaf7", "EFI_SMBIOS_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmbiosProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Smbios.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_SMBIOS_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0365e9e2-a03e-49be-8b2b-02b7e3ae953c": EfiGuid(
        "0365e9e2-a03e-49be-8b2b-02b7e3ae953c", "KeyMonFilter",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['KeyMonFilter'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b930f337647cc93e2809f678c893ad69',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cae37d13982bb26c35ffaada2841d171',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'dab68811108b555173f8f096c9737d20',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ea61df93fb3257dc78c2415817c1f871', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "0369593a-ba9b-457d-b46f-31d83a96d6fe": EfiGuid(
        "0369593a-ba9b-457d-b46f-31d83a96d6fe", "DehSmmProtocolVer3",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "036b6292-2793-4dc0-9a7f-d6b5f034558c": EfiGuid(
        "036b6292-2793-4dc0-9a7f-d6b5f034558c", "FjSystemHooks",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "036f84e1-7f37-428c-a79e-575fdfaa84ec": EfiGuid(
        "036f84e1-7f37-428c-a79e-575fdfaa84ec", "EFI_ERROR_SECTION_IOMMU_DMAR_GUID",
        description = "",
        aliases = ['EfiIommuDMArErrorSectionGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/Cper.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiIommuDMArErrorSection', 'EfiIommuDMArErrorSectionGuid']),

    "03721ef2-1801-4c91-a213-147323f75132": EfiGuid(
        "03721ef2-1801-4c91-a213-147323f75132", "AsfTableAsl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0379be4e-d706-437d-b037-edb82fb772a4": EfiGuid(
        "0379be4e-d706-437d-b037-edb82fb772a4", "EFI_DEVICE_PATH_UTILITIES_GUID",
        description = "",
        aliases = ['EFI_DEVICE_PATH_UTILITIES_PROTOCOL_GUID', 'EfiDevicePathUtilitiesProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Protocol/DevicePathUtilities.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DevicePathUtilities.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiDevicePathUtilitiesProtocol', 'EfiDevicePathUtilitiesProtocolGuid', 'EFI_DEVICE_PATH_UTILITIES_PROTOCOL_GUID']),

    "037b547a-97e8-4ff2-8cd1-dcb7a1b4915a": EfiGuid(
        "037b547a-97e8-4ff2-8cd1-dcb7a1b4915a", "SetupConfigUpdateDxeArcherCityModular",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0387f555-20a8-4fc2-bb94-cd30da1b4008": EfiGuid(
        "0387f555-20a8-4fc2-bb94-cd30da1b4008", "EfiWindowsInt10Workaround",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "038ce287-b806-45b6-a819-514daf4b91b9": EfiGuid(
        "038ce287-b806-45b6-a819-514daf4b91b9", "DELL_ERROR_LOG_CONFIG_GUID",
        description = "",
        aliases = ['DellErrorLogConfig'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "038ef9a9-5969-4793-964f-ff12fa18393e": EfiGuid(
        "038ef9a9-5969-4793-964f-ff12fa18393e", "AmdCpmOemInitPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0393d0c4-6b0c-4b96-b4c3-8c7eb718f348": EfiGuid(
        "0393d0c4-6b0c-4b96-b4c3-8c7eb718f348", "CsrPseudoOffsetInitSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0397b0c9-22e8-459e-a4ff-99bc65270929": EfiGuid(
        "0397b0c9-22e8-459e-a4ff-99bc65270929", "Tpm2StartupLocalityHob",
        description = "",
        aliases = ['Tpm2StartupLocalityHobGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/SecurityPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "03a8e402-2ab7-46b3-b591-740383796b82": EfiGuid(
        "03a8e402-2ab7-46b3-b591-740383796b82", "AmdMemChanXLatDummyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03adf4a1-a27a-45e3-b211-3177c6c2e7ed": EfiGuid(
        "03adf4a1-a27a-45e3-b211-3177c6c2e7ed", "SmbiosBasic",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03af477a-8336-0142-8a65-b4bd93b1a1a9": EfiGuid(
        "03af477a-8336-0142-8a65-b4bd93b1a1a9", "FirmwareExtension",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03bf5679-5749-4bc5-a53f-47471da767c8": EfiGuid(
        "03bf5679-5749-4bc5-a53f-47471da767c8", "LENOVO_ISSC_SMM_GUID",
        description = "",
        aliases = ['IsscSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03c1f5c8-48f1-416e-a6b6-992df3bbaca6": EfiGuid(
        "03c1f5c8-48f1-416e-a6b6-992df3bbaca6", "A01SmmServiceBody",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03c277b7-6584-441c-b58f-58bc62b6742e": EfiGuid(
        "03c277b7-6584-441c-b58f-58bc62b6742e", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "03c4e603-ac28-11d3-9a2d-0090273fc14d": EfiGuid(
        "03c4e603-ac28-11d3-9a2d-0090273fc14d", "EFI_PXE_BASE_CODE_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPxeBaseCodeProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/PxeBaseCode.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "03c4e624-ac28-11d3-9a2d-0090293fc14d": EfiGuid(
        "03c4e624-ac28-11d3-9a2d-0090293fc14d", "EFI_PXE_DHCP4_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPxeDhcp4Protocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03c70b0d-67e6-5c16-8e57-312df246a961": EfiGuid(
        "03c70b0d-67e6-5c16-8e57-312df246a961", "BootChimeAudio",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03d05ee4-cdeb-458c-9dfc-993f09bdf405": EfiGuid(
        "03d05ee4-cdeb-458c-9dfc-993f09bdf405", "ArmGicLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Drivers/ArmGic/ArmGicLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "03d56eec-e196-4815-b91a-c4885a839043": EfiGuid(
        "03d56eec-e196-4815-b91a-c4885a839043", "CompalGlobalNvsDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03d73468-3d39-4c9e-97f1-1fa925bd7d9e": EfiGuid(
        "03d73468-3d39-4c9e-97f1-1fa925bd7d9e", "SystemSecureVariableStorageDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03da99b3-ddf4-4c7e-8cca-216fc3f1f311": EfiGuid(
        "03da99b3-ddf4-4c7e-8cca-216fc3f1f311", "BaseFspSecPlatformLibNull",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03dbb540-e186-4615-8a7f-a427863b4e56": EfiGuid(
        "03dbb540-e186-4615-8a7f-a427863b4e56", "PoofAnimationState1",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03e0a38b-3fbe-49cb-b311-726611213182": EfiGuid(
        "03e0a38b-3fbe-49cb-b311-726611213182", "EgsFhfPolicyOverrideDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03e4d2d5-f088-4f47-9694-df5bea3b8c23": EfiGuid(
        "03e4d2d5-f088-4f47-9694-df5bea3b8c23", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "03e5cf63-bebb-4041-b7e7-bf546120f1c5": EfiGuid(
        "03e5cf63-bebb-4041-b7e7-bf546120f1c5", "VTD_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03ebdb4b-96ba-4f40-8329-7f3aa8865707": EfiGuid(
        "03ebdb4b-96ba-4f40-8329-7f3aa8865707", "LENOVO_EC_IO_SMM_GUID",
        description = "",
        aliases = ['EcIoSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03f3331b-f12d-494f-bf37-e55a657f2497": EfiGuid(
        "03f3331b-f12d-494f-bf37-e55a657f2497", "BaseSortLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/BaseSortLib/BaseSortLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "03fa15bf-a5c8-4006-81b5-b836e7a70030": EfiGuid(
        "03fa15bf-a5c8-4006-81b5-b836e7a70030", "GnbSummitRouting",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "03fdf171-1d67-4ace-a904-3e36d338fa74": EfiGuid(
        "03fdf171-1d67-4ace-a904-3e36d338fa74", "ME_PLATFORM_READY_TO_BOOT_GUID",
        description = "",
        aliases = ['SeCPlatformReadyToBootGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['SeCPlatformReadyToBoot', 'SeCPlatformReadyToBootGuid']),

    "04062bad-2117-46df-aeb9-8e468d8326e5": EfiGuid(
        "04062bad-2117-46df-aeb9-8e468d8326e5", "OemCheckRtcBattery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0412a7a1-c050-42c2-877a-77c379f9f5f1": EfiGuid(
        "0412a7a1-c050-42c2-877a-77c379f9f5f1", "FirewireOhciDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04132c8d-0a22-4fa8-826e-8bbfefdb836c": EfiGuid(
        "04132c8d-0a22-4fa8-826e-8bbfefdb836c", "EFI_FFS_SECTION_ALIGNMENT_PADDING_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Guid/FfsSectionAlignmentPadding.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0419e4c6-4e8e-4436-8509-38b7ae50bffd": EfiGuid(
        "0419e4c6-4e8e-4436-8509-38b7ae50bffd", "ReadyBootSMI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0419f582-0625-4531-8a33-85a9965c95bc": EfiGuid(
        "0419f582-0625-4531-8a33-85a9965c95bc", "WinNtBusDriverGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "041ce993-6677-4720-a665-375b80ffe012": EfiGuid(
        "041ce993-6677-4720-a665-375b80ffe012", "DELL_FP_THUNK_GUID",
        description = "",
        aliases = ['DellFpThunk'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "041fd986-b016-47ce-afc5-2269edd4ad64": EfiGuid(
        "041fd986-b016-47ce-afc5-2269edd4ad64", "ASROCK_ACPIS4_DXE_GUID",
        description = "",
        aliases = ['ACPIS4Dxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04249ac0-0088-439f-a74e-a7042a062f5d": EfiGuid(
        "04249ac0-0088-439f-a74e-a7042a062f5d", "GRAPHICS_PEI_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04356799-81b7-4e08-a38d-d978fa47ba42": EfiGuid(
        "04356799-81b7-4e08-a38d-d978fa47ba42", "EFI_EXTENDED_SAL_SM_COM_LAYER_SERVICES_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiExtendedSalSmComLayerServicesProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04398faf-b8d6-4c35-99c9-037911233dd9": EfiGuid(
        "04398faf-b8d6-4c35-99c9-037911233dd9", "H19QRCodeCreater",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "043a3340-7cd0-4338-b7fe-7a6aeb790a28": EfiGuid(
        "043a3340-7cd0-4338-b7fe-7a6aeb790a28", "HidKeyboardDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "043df38f-32e1-4893-911f-37a7fe91f723": EfiGuid(
        "043df38f-32e1-4893-911f-37a7fe91f723", "CapsuleProcessorDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "044310ab-77fd-402a-af1a-87d4120e7329": EfiGuid(
        "044310ab-77fd-402a-af1a-87d4120e7329", "FirmwarePerformanceSmm",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/Acpi/FirmwarePerformanceDataTableSmm/FirmwarePerformanceSmm.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "044d9982-aa57-45d7-bb7d-62966af7356b": EfiGuid(
        "044d9982-aa57-45d7-bb7d-62966af7356b", "DellStatusCodeHandlerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0455ceaf-fe5c-442f-ba23-34dfdc77505a": EfiGuid(
        "0455ceaf-fe5c-442f-ba23-34dfdc77505a", "NewRsmSampleDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "045b6f49-9102-4590-a0a5-35311b0aefc3": EfiGuid(
        "045b6f49-9102-4590-a0a5-35311b0aefc3", "EfiDpsdRSA1024AndSHA256SignatureVerificationProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "045d5b2d-f7a8-48ab-af69-a48752bf0f43": EfiGuid(
        "045d5b2d-f7a8-48ab-af69-a48752bf0f43", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "045e78d8-3d5d-4ee5-90aa-42121e1eee49": EfiGuid(
        "045e78d8-3d5d-4ee5-90aa-42121e1eee49", "CbsBasePeiRV",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04609bf5-e1c3-44fc-e4de-7f9489362184": EfiGuid(
        "04609bf5-e1c3-44fc-e4de-7f9489362184", "BiosConnectDaInterfaceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04624e48-d24e-4670-bdfd-124dd92f3dad": EfiGuid(
        "04624e48-d24e-4670-bdfd-124dd92f3dad", "PxeDriverI225",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04627b9b-385e-4744-9021-e66319f20394": EfiGuid(
        "04627b9b-385e-4744-9021-e66319f20394", "AMI_PB_KEY_RSA2048_GUID",
        description = "",
        aliases = ['AmiPbKeyRsa2048'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "046388b4-430e-4e61-88f6-51ea21db2632": EfiGuid(
        "046388b4-430e-4e61-88f6-51ea21db2632", "SecMemEncryptSevLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/BaseMemEncryptSevLib/SecMemEncryptSevLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "04688d96-3468-4d52-9727-357b9aed6727": EfiGuid(
        "04688d96-3468-4d52-9727-357b9aed6727", "FchPromontoryPlusCbsPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0468a601-c535-46fd-a95d-bbab991b178c": EfiGuid(
        "0468a601-c535-46fd-a95d-bbab991b178c", "EFI_POWER_ON_HOB_GUID",
        description = "",
        aliases = ['EfiPowerOnHobGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0469cf6e-1a81-4fff-ac00-1608472ea307": EfiGuid(
        "0469cf6e-1a81-4fff-ac00-1608472ea307", "MsdmUpdateSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "046c44db-59df-41b3-af53-ef707b930e9a": EfiGuid(
        "046c44db-59df-41b3-af53-ef707b930e9a", "FtRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0471bc5b-64be-43bf-8fa1-435af1161384": EfiGuid(
        "0471bc5b-64be-43bf-8fa1-435af1161384", "CrystalRidgeMeasurement",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0477cc85-f610-47b9-b322-42e1a93d9ee7": EfiGuid(
        "0477cc85-f610-47b9-b322-42e1a93d9ee7", "FjGpioAbstractionReferenceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "047ca5c8-efa6-64ca-98ec-a01df3bcc6a8": EfiGuid(
        "047ca5c8-efa6-64ca-98ec-a01df3bcc6a8", "A01BootDeviceServiceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "048520e2-46bc-829d-5683-329bf1162cd2": EfiGuid(
        "048520e2-46bc-829d-5683-329bf1162cd2", "BiosPowerOnSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04877be8-fac0-448e-801d-f0f895ccb012": EfiGuid(
        "04877be8-fac0-448e-801d-f0f895ccb012", "SmbiosDmiEditSsi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04970e59-fccc-47cc-b945-7976ee7dbb7a": EfiGuid(
        "04970e59-fccc-47cc-b945-7976ee7dbb7a", "OKRConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04a0d644-5599-430a-a4e1-c8fee8cbf46a": EfiGuid(
        "04a0d644-5599-430a-a4e1-c8fee8cbf46a", "SmbiosType142",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04a76c80-06b9-445e-b73e-cb8c61a6a964": EfiGuid(
        "04a76c80-06b9-445e-b73e-cb8c61a6a964", "SIO791",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04adeead-61ff-4d31-b6ba-64f8bf901f5a": EfiGuid(
        "04adeead-61ff-4d31-b6ba-64f8bf901f5a", "FU_EFI_FIRMWARE_VOLUME_GUID_APPLE_BOOT",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04afc94a-73af-432f-becb-b794568ac985": EfiGuid(
        "04afc94a-73af-432f-becb-b794568ac985", "DellOnboardNicSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04b37fe8-f6ae-480b-bdd5-37d98c5e89aa": EfiGuid(
        "04b37fe8-f6ae-480b-bdd5-37d98c5e89aa", "EDKII_VAR_ERROR_FLAG_GUID",
        description = "",
        aliases = ['EdkiiVarErrorFlagGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/VarErrorFlag.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "04c04e7f-0276-4d0d-a73f-1da3c8b708b2": EfiGuid(
        "04c04e7f-0276-4d0d-a73f-1da3c8b708b2", "ACER_SMM_SERVICE_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04c332fd-14d3-411c-a1ef-6512e260339e": EfiGuid(
        "04c332fd-14d3-411c-a1ef-6512e260339e", "FjNvsAreaSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04dd0ecd-4844-426d-ae59-1ef632c5ea4c": EfiGuid(
        "04dd0ecd-4844-426d-ae59-1ef632c5ea4c", "SystemDxeToSmmEventDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04dd1ee1-4c38-41c7-83c6-4ed6bb692cfd": EfiGuid(
        "04dd1ee1-4c38-41c7-83c6-4ed6bb692cfd", "AaeonDioDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04e03541-4663-417d-93f6-976378247d61": EfiGuid(
        "04e03541-4663-417d-93f6-976378247d61", "CsmSupportLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Csm/CsmSupportLib/CsmSupportLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "04e86dfa-6fa9-4749-a069-5a708f416af7": EfiGuid(
        "04e86dfa-6fa9-4749-a069-5a708f416af7", "OemPDAutoFlash",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04eaaaa1-29a1-11d7-8838-00500473d4eb": EfiGuid(
        "04eaaaa1-29a1-11d7-8838-00500473d4eb", "ASROCK_USBRT_GUID",
        description = "",
        aliases = ['UsbRtSmm'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['ASROCK_USBRT_GUID', 'UsbRt', 'UsbRtSmm'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '48bba53caf662f85dd73a1a247b45dc7', '4afaa764f57449bc440e58fa11422e02',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '633e80e7347dc01c21c5133227df2aba', '6f1b81ae0adfe7f90c141e463e9e9344',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '7dffb9d276837abbf9e319d968a966a1',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['USBRT']),

    "04f5b0f1-73e2-4cc4-9741-662dbc3ab78b": EfiGuid(
        "04f5b0f1-73e2-4cc4-9741-662dbc3ab78b", "ErrorCodeHandlerSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04f75442-a593-4281-bd4b-095935b7d2f8": EfiGuid(
        "04f75442-a593-4281-bd4b-095935b7d2f8", "wifi_1bar",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "04f7f9b7-1248-497c-b0eb-8f748abfcaf0": EfiGuid(
        "04f7f9b7-1248-497c-b0eb-8f748abfcaf0", "DellVideoProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05041a6b-4dee-47bb-9e58-5944d1870ee5": EfiGuid(
        "05041a6b-4dee-47bb-9e58-5944d1870ee5", "UsbConfigGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "050eb8c6-c12e-4b86-892b-40985e8b3137": EfiGuid(
        "050eb8c6-c12e-4b86-892b-40985e8b3137", "UefiDevicePathLibDevicePathProtocol",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/UefiDevicePathLibDevicePathProtocol/UefiDevicePathLibDevicePathProtocol.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "05105279-dce6-4d27-ba3c-0c6f3e8de66d": EfiGuid(
        "05105279-dce6-4d27-ba3c-0c6f3e8de66d", "SetBoardIdVar",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "051274f4-a724-4732-be00-82793a3d499a": EfiGuid(
        "051274f4-a724-4732-be00-82793a3d499a", "AmdUsb4Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0515bc05-2959-4e91-89c6-6b3a3f1fcb65": EfiGuid(
        "0515bc05-2959-4e91-89c6-6b3a3f1fcb65", "TCG_MPDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05161583-2ab3-43d1-8087-e890997b0f80": EfiGuid(
        "05161583-2ab3-43d1-8087-e890997b0f80", "DellPlatformFlexBay",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "051a0b77-c066-4610-9904-bf25c296d6e6": EfiGuid(
        "051a0b77-c066-4610-9904-bf25c296d6e6", "LENOVO_EN25_QH32_FLASH_PART_DXE_GUID",
        description = "",
        aliases = ['LenovoEn25Qh32FlashPartDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "052e6eb0-f240-42c5-8309-45874545c6b4": EfiGuid(
        "052e6eb0-f240-42c5-8309-45874545c6b4", "BOOT_NOW_COUNT_GUID",
        description = "",
        aliases = ['BootNowCount'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05302b01-e898-494e-9fb1-9e02dc5d7ccc": EfiGuid(
        "05302b01-e898-494e-9fb1-9e02dc5d7ccc", "EcMudPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0535c332-6d4d-4d89-9468-f3627bb386ce": EfiGuid(
        "0535c332-6d4d-4d89-9468-f3627bb386ce", "H19MpmRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0541150c-e33b-4daf-a263-02e4bb4bf1cf": EfiGuid(
        "0541150c-e33b-4daf-a263-02e4bb4bf1cf", "SecurityErrorHandlingDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "054f2504-e2bc-4641-83fc-502588fe1f28": EfiGuid(
        "054f2504-e2bc-4641-83fc-502588fe1f28", "CpuInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0558caea-fef3-4b6d-915e-8742efe6dee1": EfiGuid(
        "0558caea-fef3-4b6d-915e-8742efe6dee1", "SmbusLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0559e886-ac78-4bcc-899a-e7830b5d6462": EfiGuid(
        "0559e886-ac78-4bcc-899a-e7830b5d6462", "SystemSmbiosBcpSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0565365c-2fe1-4f88-b3be-624c04623a20": EfiGuid(
        "0565365c-2fe1-4f88-b3be-624c04623a20", "MicrocodeUpdateDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0565963b-3dfd-4712-9cfd-614c5edbe592": EfiGuid(
        "0565963b-3dfd-4712-9cfd-614c5edbe592", "MsiBoardPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "056623dc-1285-4eaf-9446-75c3b00f78c6": EfiGuid(
        "056623dc-1285-4eaf-9446-75c3b00f78c6", "FchMultiFchDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05687f4a-3ca7-4d19-9bc5-e180cea3569f": EfiGuid(
        "05687f4a-3ca7-4d19-9bc5-e180cea3569f", "AMI_TCG_PLATFORM_PPI_GUID",
        description = "",
        aliases = ['AmiPlatformSecurityChipGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "056bca18-2f19-41ee-84ea-83746cb5069a": EfiGuid(
        "056bca18-2f19-41ee-84ea-83746cb5069a", "CrystalRidgeSMM",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "056e7324-a718-465b-9a84-228f06642b4f": EfiGuid(
        "056e7324-a718-465b-9a84-228f06642b4f", "EFI_PLATFORM_DRIVER_GUID",
        description = "",
        aliases = ['DxePlatform'],
        urls = [],
        vendors = [],
        filenames = ['DxePlatform', 'EFI_PLATFORM_DRIVER_GUID'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = ['PlatformDxe']),

    "0571dc0c-d3b8-422c-a9ef-ac446483bdfe": EfiGuid(
        "0571dc0c-d3b8-422c-a9ef-ac446483bdfe", "VisualCSM",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "057a449a-1fdc-4c06-bfc9-f53f6a99bb92": EfiGuid(
        "057a449a-1fdc-4c06-bfc9-f53f6a99bb92", "EFI_PEI_PCI_CFG2_PPI_GUID",
        description = "",
        aliases = ['EfiPciCfg2PpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/PciCfg2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_PCI_CFG2_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EfiPciCfg2Ppi', 'EfiPciCfg2PpiGuid']),

    "057c712a-84f0-4f4a-94cb-713eef002e2f": EfiGuid(
        "057c712a-84f0-4f4a-94cb-713eef002e2f", "PeiNt32PeCoffExtraActionLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05810525-fdec-4006-9f1f-37609b3675fa": EfiGuid(
        "05810525-fdec-4006-9f1f-37609b3675fa", "ArmMtlNullLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Library/ArmMtlNullLib/ArmMtlNullLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "05839aac-361d-47d8-b2ba-50d5f4b09ad8": EfiGuid(
        "05839aac-361d-47d8-b2ba-50d5f4b09ad8", "AsusRecoveryFailedPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0583d694-af8b-4baa-9583-813cedf40843": EfiGuid(
        "0583d694-af8b-4baa-9583-813cedf40843", "CsmInt10Block",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0584fc67-72b9-4d46-ae3e-ad330452d9b4": EfiGuid(
        "0584fc67-72b9-4d46-ae3e-ad330452d9b4", "InputDeviceAggregatorDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05885d44-9588-428e-9eb0-1ca7e54ca96b": EfiGuid(
        "05885d44-9588-428e-9eb0-1ca7e54ca96b", "TdxPostMem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0589e077-93c8-470e-9b90-958b4e2fe686": EfiGuid(
        "0589e077-93c8-470e-9b90-958b4e2fe686", "DevUpdateProgress",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05913b1b-db0c-4679-94b1-f8aa09b4a971": EfiGuid(
        "05913b1b-db0c-4679-94b1-f8aa09b4a971", "FjPcieCardReaderRTS5250",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05984e1a-d8bb-5d8a-a8e6-90e6fb2ab7da": EfiGuid(
        "05984e1a-d8bb-5d8a-a8e6-90e6fb2ab7da", "AlertUI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05abe46f-bfa5-4c68-b305-3d545c58d805": EfiGuid(
        "05abe46f-bfa5-4c68-b305-3d545c58d805", "Vby1PeiPreGop",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05abe46f-bfa5-4c68-b305-3d545c58d806": EfiGuid(
        "05abe46f-bfa5-4c68-b305-3d545c58d806", "Vby1PeiPostBuffer",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05ad34ba-6f02-4214-952e-4da0398e2bb9": EfiGuid(
        "05ad34ba-6f02-4214-952e-4da0398e2bb9", "DXE_SERVICES_TABLE_GUID",
        description = "",
        aliases = ['EfiDxeServicesTableGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/DxeServices.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiDxeServicesTable', 'EfiDxeServicesTableGuid', 'EFI_DXE_SERVICES_TABLE_GUID']),

    "05b0a258-308a-445e-b669-a854549d225c": EfiGuid(
        "05b0a258-308a-445e-b669-a854549d225c", "SmmCpuProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05b17ef3-26cb-444a-82f7-92705cfb6b1e": EfiGuid(
        "05b17ef3-26cb-444a-82f7-92705cfb6b1e", "DellSetupFormSets",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05be67b1-0748-4022-97e6-7c0d2f992aa6": EfiGuid(
        "05be67b1-0748-4022-97e6-7c0d2f992aa6", "FmpDependencyDeviceLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/FmpDevicePkg/Library/FmpDependencyDeviceLibNull/FmpDependencyDeviceLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "05c99a21-c70f-4ad2-8a5f-35df3343f51e": EfiGuid(
        "05c99a21-c70f-4ad2-8a5f-35df3343f51e", "EFI_DEVICE_PATH_FROM_TEXT_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDevicePathFromTextProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DevicePathFromText.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "05ca01fc-0fc1-11dc-9011-00173153eba8": EfiGuid(
        "05ca01fc-0fc1-11dc-9011-00173153eba8", "OEM_ROM_HOLE_0_GUID",
        description = "",
        aliases = ['OemRomHole_0'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05ca01fd-0fc1-11dc-9011-00173153eba8": EfiGuid(
        "05ca01fd-0fc1-11dc-9011-00173153eba8", "OEM_ROM_HOLE_1_GUID",
        description = "",
        aliases = ['OemRomHole_1'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05cdf411-f3c1-4b12-aa64-21416e18a13c": EfiGuid(
        "05cdf411-f3c1-4b12-aa64-21416e18a13c", "AmdCpmSharedBSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05db6cc9-e9e4-4209-9d46-aa52c4052937": EfiGuid(
        "05db6cc9-e9e4-4209-9d46-aa52c4052937", "CNVISetupSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05dda141-2dfa-48bb-b09d-d922344947fe": EfiGuid(
        "05dda141-2dfa-48bb-b09d-d922344947fe", "ApobZpRvPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05dfca46-141f-11df-8508-e38c0891c4e2": EfiGuid(
        "05dfca46-141f-11df-8508-e38c0891c4e2", "HfsPlusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05e9cf0f-4be3-87f4-7265-b796a76959c3": EfiGuid(
        "05e9cf0f-4be3-87f4-7265-b796a76959c3", "S5MaxPowerSavingsSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05f7aa70-a64a-432c-8cee-4cdecb8671d7": EfiGuid(
        "05f7aa70-a64a-432c-8cee-4cdecb8671d7", "A01ODMDxeDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05fc0c8e-276a-422d-bae1-645cfd7b786b": EfiGuid(
        "05fc0c8e-276a-422d-bae1-645cfd7b786b", "Cf9Reset",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05fe1bb1-41d3-43dd-abb0-91860b4ab185": EfiGuid(
        "05fe1bb1-41d3-43dd-abb0-91860b4ab185", "AsusBackupSync",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "05ffb44d-ee01-40e1-9866-fd27fd4fbe92": EfiGuid(
        "05ffb44d-ee01-40e1-9866-fd27fd4fbe92", "ThunderboltDeviceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "060644b6-6a7f-4a97-900c-49645da956d7": EfiGuid(
        "060644b6-6a7f-4a97-900c-49645da956d7", "FchYuntaiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "060cc026-4c0d-4dda-8f41-595fef00a502": EfiGuid(
        "060cc026-4c0d-4dda-8f41-595fef00a502", "MEMORY_STATUS_CODE_RECORD_GUID",
        description = "",
        aliases = ['MemoryStatusCodeRecordGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/MemoryStatusCodeRecord.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0619f5c2-4858-4caa-a86a-73a21a18df6b": EfiGuid(
        "0619f5c2-4858-4caa-a86a-73a21a18df6b", "GenericWatchdogDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Drivers/GenericWatchdogDxe/GenericWatchdogDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0625fea6-f95f-498d-8789-3ea053a06d7e": EfiGuid(
        "0625fea6-f95f-498d-8789-3ea053a06d7e", "Cf9Reset",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "062a6204-8c80-4a27-9d09-96654c965894": EfiGuid(
        "062a6204-8c80-4a27-9d09-96654c965894", "AsusHwmSensorPortingDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "062acc82-1d1e-4f61-aa94-8b0c47236a3d": EfiGuid(
        "062acc82-1d1e-4f61-aa94-8b0c47236a3d", "BootScriptSaveOnS3SaveStateThunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "062fdfe6-2c0f-462c-9572-92018e6e52e3": EfiGuid(
        "062fdfe6-2c0f-462c-9572-92018e6e52e3", "HpIcicleRepSet",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0633a0f1-78fe-4139-b878-0045e81cb8ab": EfiGuid(
        "0633a0f1-78fe-4139-b878-0045e81cb8ab", "EfiVolatileMemModeVariable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06384570-1680-4620-9d00-6ad9e3ccb19f": EfiGuid(
        "06384570-1680-4620-9d00-6ad9e3ccb19f", "LENOVO_MTM_FORMAT_SMM_GUID",
        description = "",
        aliases = ['LenovoMtmFormatSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0639408b-19a6-4b5d-bafb-12a2f5114032": EfiGuid(
        "0639408b-19a6-4b5d-bafb-12a2f5114032", "Acoustic",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['Acoustic'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "064b4c5b-c5ef-4eff-85df-65518ef1314d": EfiGuid(
        "064b4c5b-c5ef-4eff-85df-65518ef1314d", "SmmServicesTableLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/SmmServicesTableLib/SmmServicesTableLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0650f140-439a-2d4f-9807-c92669339399": EfiGuid(
        "0650f140-439a-2d4f-9807-c92669339399", "AppleEvent",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0653586a-9ed4-400c-a342-9efaf99ce54e": EfiGuid(
        "0653586a-9ed4-400c-a342-9efaf99ce54e", "SecuredCorePCDeviceIdentifier",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "066785b1-edb8-46dc-842f-574404b8692f": EfiGuid(
        "066785b1-edb8-46dc-842f-574404b8692f", "EFI_PEI_CAPSULE_PPI_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "067b1f5f-cf26-44c5-8554-93d777912d42": EfiGuid(
        "067b1f5f-cf26-44c5-8554-93d777912d42", "ConfidentialComputingSevSnpBlob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "067e2381-7234-4798-b49c-d5fecbff6d07": EfiGuid(
        "067e2381-7234-4798-b49c-d5fecbff6d07", "VariableAuthSmmRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06818c78-1c8a-4591-9833-dc0e4b9566e4": EfiGuid(
        "06818c78-1c8a-4591-9833-dc0e4b9566e4", "TdtAm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['TdtAm'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '186cc72a8ef4d9cb1062b2cea44ef726',
            '3dbf73930867371066435e0908e50535', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', 'b218fecd176e986c1491046e58c9e01b',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
            'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "06830ac0-75d7-4933-b6af-4d3aa317d2af": EfiGuid(
        "06830ac0-75d7-4933-b6af-4d3aa317d2af", "FJPhysicalPresencePage",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0683fb88-664c-4ba6-9ed4-1c0916ee43a4": EfiGuid(
        "0683fb88-664c-4ba6-9ed4-1c0916ee43a4", "UserAuthenticationDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0684c591-19e8-499d-978a-d715636f1dab": EfiGuid(
        "0684c591-19e8-499d-978a-d715636f1dab", "AmdRasRvDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0693baa5-35d0-4360-83f6-1df26d3c53ce": EfiGuid(
        "0693baa5-35d0-4360-83f6-1df26d3c53ce", "SiCpuInitPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06968b6e-f620-4e8d-a6a6-52552354dd75": EfiGuid(
        "06968b6e-f620-4e8d-a6a6-52552354dd75", "LENOVO_DXE_RICOH_INIT_GUID",
        description = "",
        aliases = ['LenovoDxeRicohInit'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "069887de-145c-404c-9b23-e262a88789af": EfiGuid(
        "069887de-145c-404c-9b23-e262a88789af", "RfVlan",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06b10249-4d38-ff37-0ca5-13afb6d625cc": EfiGuid(
        "06b10249-4d38-ff37-0ca5-13afb6d625cc", "RedfishContentCodingLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/RedfishPkg/Library/RedfishContentCodingLibNull/RedfishContentCodingLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "06b5947e-ff53-457d-98bc-c5abc777fd5a": EfiGuid(
        "06b5947e-ff53-457d-98bc-c5abc777fd5a", "SetupMouseDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06be9c49-717c-46c8-9848-e626b79faa36": EfiGuid(
        "06be9c49-717c-46c8-9848-e626b79faa36", "FchYuntaiSmmInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06d1f4f5-b131-4f7c-a983-20e4be9e8a3d": EfiGuid(
        "06d1f4f5-b131-4f7c-a983-20e4be9e8a3d", "DellPSBFuseDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06d20d84-a032-4e25-969a-346d255e46d1": EfiGuid(
        "06d20d84-a032-4e25-969a-346d255e46d1", "CpuCsrAccess",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06de824a-a4e2-4295-a3f6-03b3fef5b109": EfiGuid(
        "06de824a-a4e2-4295-a3f6-03b3fef5b109", "LENOVO_SECURITY_VARIABLE_DXE_GUID",
        description = "",
        aliases = ['LenovoSecurityVariableDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06e81c58-4ad7-44bc-8390-f10265f72480": EfiGuid(
        "06e81c58-4ad7-44bc-8390-f10265f72480", "PCD_PPI_GUID",
        description = "",
        aliases = ['PcdPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/Pcd.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "06ecaf57-1a95-44dd-ae09-ab9b9bd681e9": EfiGuid(
        "06ecaf57-1a95-44dd-ae09-ab9b9bd681e9", "MpmAsfDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06f00ede-198c-0217-2416-4299bce072ca": EfiGuid(
        "06f00ede-198c-0217-2416-4299bce072ca", "OemMfgdoneDcfg",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "06f73f04-36c0-4cbc-a3b0-9f16fb603350": EfiGuid(
        "06f73f04-36c0-4cbc-a3b0-9f16fb603350", "OemPxeLoader",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07013588-c789-4e12-a7c3-88fafae79f7c": EfiGuid(
        "07013588-c789-4e12-a7c3-88fafae79f7c", "EfiSocketProcessorCoreVar",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0702269d-380c-4873-bfeb-32f5bc44dd16": EfiGuid(
        "0702269d-380c-4873-bfeb-32f5bc44dd16", "AmtLockI2cConInDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0703f912-bf8d-4e2a-be07-ab272525c592": EfiGuid(
        "0703f912-bf8d-4e2a-be07-ab272525c592", "gEfiMmPeiMmramMemoryReserveGuid",
        description = "",
        aliases = ['EfiSmmPeiMmramMemoryReserve'],
        urls = ['https://github.com/tianocore/edk2/blob/master/StandaloneMmPkg/StandaloneMmPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0718ad81-f26a-4850-a6ec-f268e309d707": EfiGuid(
        "0718ad81-f26a-4850-a6ec-f268e309d707", "Tpm20PlatformDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['Tpm20PlatformDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "071a3dbe-cff4-4b73-83f0-598c13dcfdd5": EfiGuid(
        "071a3dbe-cff4-4b73-83f0-598c13dcfdd5", "Slp21Markers",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0723f88b-810f-4e44-92a0-a5c83b433698": EfiGuid(
        "0723f88b-810f-4e44-92a0-a5c83b433698", "LENOVO_IBEX_PEAK_FLASH_CONTROLLER_DXE_GUID",
        description = "",
        aliases = ['PchFlashControllerDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07261c8f-9f27-4409-bd78-2bb0b691f331": EfiGuid(
        "07261c8f-9f27-4409-bd78-2bb0b691f331", "DetectWlan",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "072a957a-2a7a-4db1-a35d-21c9795acd6e": EfiGuid(
        "072a957a-2a7a-4db1-a35d-21c9795acd6e", "SetupUtilitySilicon1",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07309a59-462f-4251-bf5f-324e80cd5b05": EfiGuid(
        "07309a59-462f-4251-bf5f-324e80cd5b05", "WtTimestamp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0732bd39-d6b0-4039-b6c2-9654466de525": EfiGuid(
        "0732bd39-d6b0-4039-b6c2-9654466de525", "MEUD_ERROR_GUID",
        description = "",
        aliases = ['MeudError'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0736595d-88ab-483e-9886-8132018980fe": EfiGuid(
        "0736595d-88ab-483e-9886-8132018980fe", "SioChip1InitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0738958d-a35a-4175-ace9-d32445501d34": EfiGuid(
        "0738958d-a35a-4175-ace9-d32445501d34", "FjTpmPhysicalPresenceLockSkip",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0739ce30-2e1a-449a-b575-fd64f1b4321d": EfiGuid(
        "0739ce30-2e1a-449a-b575-fd64f1b4321d", "SystemSetupCapsuleFromInternetDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "073e2576-f6c1-4b91-92a9-d4675dda34b1": EfiGuid(
        "073e2576-f6c1-4b91-92a9-d4675dda34b1", "PTT_PASS_THRU_PROTOCOL_GUID",
        description = "",
        aliases = ['PttPassThruProtocolGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "073e7e01-2611-4e85-b896-a3b6767cba00": EfiGuid(
        "073e7e01-2611-4e85-b896-a3b6767cba00", "AMITSE_PASSWORD_PROMPT_ENTER_GUID",
        description = "",
        aliases = ['AmiTsePasswordPromptEnterGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "074993a4-19a1-4e0d-b892-8fbcc6d79f35": EfiGuid(
        "074993a4-19a1-4e0d-b892-8fbcc6d79f35", "SaveMemoryConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "074e1e48-8132-47a1-8c2c-3f14ad9a66dc": EfiGuid(
        "074e1e48-8132-47a1-8c2c-3f14ad9a66dc", "EFI_GLOBAL_NVS_AREA_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiGlobalNvsAreaProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_GLOBAL_NVS_AREA_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', '9f603f855d176a1ea5758fa617d3e0d2',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f8c34791a7a721ee330771cb43d68b91', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "07525079-2616-4599-93fa-5ee9e3f0295c": EfiGuid(
        "07525079-2616-4599-93fa-5ee9e3f0295c", "CsmPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0755bb58-3362-4e3c-8d4c-8e0f28366e21": EfiGuid(
        "0755bb58-3362-4e3c-8d4c-8e0f28366e21", "EcGbeSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0761eacb-21fe-4645-9287-d641935f160e": EfiGuid(
        "0761eacb-21fe-4645-9287-d641935f160e", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "0764aad8-9d82-441f-80f3-21afa56f807c": EfiGuid(
        "0764aad8-9d82-441f-80f3-21afa56f807c", "Afu32MOfbd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07683700-036b-440b-8eba-637f9ecb3deb": EfiGuid(
        "07683700-036b-440b-8eba-637f9ecb3deb", "SioSmbusAccessSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "076e4ff4-afb4-4b74-8d94-0517bec86361": EfiGuid(
        "076e4ff4-afb4-4b74-8d94-0517bec86361", "FchSmbusPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "076fbf50-ef83-4b77-912c-d5bb77075ae3": EfiGuid(
        "076fbf50-ef83-4b77-912c-d5bb77075ae3", "CompalThermalDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07709c98-414a-4039-b7d0-29c21f105e92": EfiGuid(
        "07709c98-414a-4039-b7d0-29c21f105e92", "AcpiPlatformFeatures",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AcpiPlatformFeatures'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0770e7a0-9c3a-499c-a311-2f8385080556": EfiGuid(
        "0770e7a0-9c3a-499c-a311-2f8385080556", "LENOVO_BOARD_INFO_SMM_GUID",
        description = "",
        aliases = ['BoardInfoSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07755871-ca72-42d1-af0e-5beb971f7e4c": EfiGuid(
        "07755871-ca72-42d1-af0e-5beb971f7e4c", "RtcAlarmSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "077700fc-1f81-4fd2-8ea8-be9258884d45": EfiGuid(
        "077700fc-1f81-4fd2-8ea8-be9258884d45", "PostScreenInfo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "077a3100-1667-45fa-b98c-37651b5073ee": EfiGuid(
        "077a3100-1667-45fa-b98c-37651b5073ee", "TpmSmbiosDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['TpmSmbiosDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0e6eebbc21300ba0ebec9f192d5beabf', '1ded807f42af76804c9a0bbb6e24a2a9',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '60785be6bfb0d1daa94a8bb214b774d1', '633e80e7347dc01c21c5133227df2aba',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '8091cabbb4de557d84e3dbd55126455a',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '97070b6e86ea62d42df9c0d42179c050',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b6cce665f85751125c3db2050ffc98d2',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'e7a706c2421a55cde70e476d1d37f831', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "077f6fc5-2d7c-4f58-b12c-f5287e5f5843": EfiGuid(
        "077f6fc5-2d7c-4f58-b12c-f5287e5f5843", "DellTagsDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0780f2b5-eea8-4a94-97b8-59c8275d5692": EfiGuid(
        "0780f2b5-eea8-4a94-97b8-59c8275d5692", "Ccg4ControllerDetectPortingSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "078485f1-0c1f-4b1b-afb3-4a09c0ef87a1": EfiGuid(
        "078485f1-0c1f-4b1b-afb3-4a09c0ef87a1", "It8728SmmFeatures",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0784924f-e296-11d4-9a49-0090273fc14d": EfiGuid(
        "0784924f-e296-11d4-9a49-0090273fc14d", "EFI_DRIVER_DIAGNOSTICS_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDriverDiagnosticsProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DriverDiagnostics.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0787c598-f9c6-4b7f-8c85-0c96c4e1e531": EfiGuid(
        "0787c598-f9c6-4b7f-8c85-0c96c4e1e531", "FjHookFormBrowserDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07893dfb-115c-4c5e-b54e-9a4e83ee2e70": EfiGuid(
        "07893dfb-115c-4c5e-b54e-9a4e83ee2e70", "InstallSLICTable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "078f54d4-cc22-4048-9e94-879c214d562f": EfiGuid(
        "078f54d4-cc22-4048-9e94-879c214d562f", "DefaultFvPadFileName",
        description = "",
        aliases = ['DefaultFvPadFileNameGuid'],
        urls = [],
        vendors = [],
        filenames = ['DefaultFvPadFileName'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "079c90b0-1b9c-4c3c-9316-efc7f680a67b": EfiGuid(
        "079c90b0-1b9c-4c3c-9316-efc7f680a67b", "PlatformSmbiosDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "079e8e98-ae93-4b9a-8a71-1dc869f23e09": EfiGuid(
        "079e8e98-ae93-4b9a-8a71-1dc869f23e09", "ShellSortTestApp",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ShellPkg/Application/ShellSortTestApp/ShellSortTestApp.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "07a50b13-7539-41cb-b924-9e72b870c6d3": EfiGuid(
        "07a50b13-7539-41cb-b924-9e72b870c6d3", "UsbVhcDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07a9330a-f347-11d4-9a49-0090273fc14d": EfiGuid(
        "07a9330a-f347-11d4-9a49-0090273fc14d", "LENOVO_SYSTEM_METRONOME_DXE_GUID",
        description = "",
        aliases = ['LegacyMetronome'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['LegacyMetronome', 'Metronome']),

    "07ad733a-0ddb-4244-bd82-071f39e2a420": EfiGuid(
        "07ad733a-0ddb-4244-bd82-071f39e2a420", "PlatformSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07b348f3-2187-4035-aeaf-f07a85d9754b": EfiGuid(
        "07b348f3-2187-4035-aeaf-f07a85d9754b", "AmdEdidOverrideDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07b37006-9302-408e-b416-b524d110dd7f": EfiGuid(
        "07b37006-9302-408e-b416-b524d110dd7f", "AppleHidInterface",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07b58ad3-eb7b-4233-9044-9c9e65a4973b": EfiGuid(
        "07b58ad3-eb7b-4233-9044-9c9e65a4973b", "RZ616_MtkSuppDex",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07bde155-616c-4291-bd0b-24411f3bb908": EfiGuid(
        "07bde155-616c-4291-bd0b-24411f3bb908", "AmdRAIDCoreDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07bef20e-5e47-4a69-b228-27c6c870eae1": EfiGuid(
        "07bef20e-5e47-4a69-b228-27c6c870eae1", "AmdSocFp7r2PhxDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07d1e0c7-1262-481f-95c0-3e05da38f766": EfiGuid(
        "07d1e0c7-1262-481f-95c0-3e05da38f766", "FchTacomaSmmInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07d25bbb-f832-41bb-bba0-612e9f033067": EfiGuid(
        "07d25bbb-f832-41bb-bba0-612e9f033067", "RuntimeDxeReportStatusCodeLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/RuntimeDxeReportStatusCodeLib/RuntimeDxeReportStatusCodeLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "07d279a1-34e4-4168-993b-178b3acc68ef": EfiGuid(
        "07d279a1-34e4-4168-993b-178b3acc68ef", "AodSetupDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07d75280-27d4-4d69-90d0-5643e238b341": EfiGuid(
        "07d75280-27d4-4d69-90d0-5643e238b341", "EFI_PCI_PLATFORM_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPciPlatformProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/PciPlatform.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "07dccae1-a9ae-4ef4-aa10-468b1c37becc": EfiGuid(
        "07dccae1-a9ae-4ef4-aa10-468b1c37becc", "OemDxeShp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07dfa0d2-2ac5-4cab-ac14-305c624887e4": EfiGuid(
        "07dfa0d2-2ac5-4cab-ac14-305c624887e4", "PlatformTokenSpace",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07edef55-24ec-43f3-8e3e-8c0831e21e68": EfiGuid(
        "07edef55-24ec-43f3-8e3e-8c0831e21e68", "EzTpmUpdStTpmFwUpdDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07f1b357-4270-4122-a32a-7eab3c013bac": EfiGuid(
        "07f1b357-4270-4122-a32a-7eab3c013bac", "DfciManager",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "07f6a3d8-9320-4bcb-ba8b-4fa6055d909d": EfiGuid(
        "07f6a3d8-9320-4bcb-ba8b-4fa6055d909d", "AmdPspP2CmboxV2Ssp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08018188-42cd-bb48-100f-5387d53ded3d": EfiGuid(
        "08018188-42cd-bb48-100f-5387d53ded3d", "EFI_ACPI_6_0_NFIT_GUID_RAM_DISK_SUPPORTING_VIRTUAL_CD_REGION_PERSISTENT",
        description = "",
        aliases = ['EFI_PERSISTENT_VIRTUAL_CD_GUID', 'EfiPersistentVirtualCdGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Protocol/DevicePath.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EFI_ACPI_6_1_NFIT_GUID_RAM_DISK_SUPPORTING_VIRTUAL_CD_REGION_PERSISTENT', 'EFI_ACPI_6_2_NFIT_GUID_RAM_DISK_SUPPORTING_VIRTUAL_CD_REGION_PERSISTENT', 'EFI_ACPI_6_3_NFIT_GUID_RAM_DISK_SUPPORTING_VIRTUAL_CD_REGION_PERSISTENT', 'EfiPersistentVirtualCd', 'EFI_PERSISTENT_VIRTUAL_CD_GUID']),

    "081603b4-0f1d-4022-b6fd-4ce35e09a1a6": EfiGuid(
        "081603b4-0f1d-4022-b6fd-4ce35e09a1a6", "EFI_UNIX_NETWORK_GUID",
        description = "",
        aliases = ['EfiUnixNetwork'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "081c9a43-431f-475d-b84c-93ad6714b80c": EfiGuid(
        "081c9a43-431f-475d-b84c-93ad6714b80c", "DiskIdm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "081cca80-ae21-afc8-1ac8-3091a8f1cbc5": EfiGuid(
        "081cca80-ae21-afc8-1ac8-3091a8f1cbc5", "SmcOobPlatformPolicySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "082198eb-1731-478a-901c-5a86a9c64910": EfiGuid(
        "082198eb-1731-478a-901c-5a86a9c64910", "DellPermDevConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08226c3f-bbf8-431b-875e-7361444a23f9": EfiGuid(
        "08226c3f-bbf8-431b-875e-7361444a23f9", "AmdMemSmbiosV2RnPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0823a088-1248-4285-9616-0111cc76d3c5": EfiGuid(
        "0823a088-1248-4285-9616-0111cc76d3c5", "SmbusRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0831ab36-d5d0-4280-bed2-a7e27e9fefc9": EfiGuid(
        "0831ab36-d5d0-4280-bed2-a7e27e9fefc9", "HpSioInitPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0833a84d-8a39-431f-bd1c-30965aae71dd": EfiGuid(
        "0833a84d-8a39-431f-bd1c-30965aae71dd", "ProcessorStartup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0840ff86-f690-4fd8-a891-8bf944b0f324": EfiGuid(
        "0840ff86-f690-4fd8-a891-8bf944b0f324", "DellSmmKeyboardBacklightProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08448b41-7f83-49be-82a7-0e84790ab133": EfiGuid(
        "08448b41-7f83-49be-82a7-0e84790ab133", "LePassKeyDeviceManager",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08464531-4c99-4c4c-a887-8d8ba4bbb063": EfiGuid(
        "08464531-4c99-4c4c-a887-8d8ba4bbb063", "Ps2MouseDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Isa/Ps2MouseDxe/Ps2MouseDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "08497e75-0bf6-40e2-8abf-b98dfb464c93": EfiGuid(
        "08497e75-0bf6-40e2-8abf-b98dfb464c93", "SpiFlashPeiLibNull",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "084a92b2-036a-414c-adea-620acf547664": EfiGuid(
        "084a92b2-036a-414c-adea-620acf547664", "RasMiscDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0859ba18-7dd7-4ed7-a88e-109c63917bdd": EfiGuid(
        "0859ba18-7dd7-4ed7-a88e-109c63917bdd", "EFI_PLATFORM_MEMTEST_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPlatformMemtestProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "085a786c-5cff-4762-81cf-d7cbfeca77d4": EfiGuid(
        "085a786c-5cff-4762-81cf-d7cbfeca77d4", "AmdPlatformRasBrhSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "085ddd80-ce41-4fa8-b298-6071e8f62bc5": EfiGuid(
        "085ddd80-ce41-4fa8-b298-6071e8f62bc5", "SmcOobPlatformPolicyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "085e8cc2-8ec9-4666-bd2a-49d481e95fa7": EfiGuid(
        "085e8cc2-8ec9-4666-bd2a-49d481e95fa7", "FjEvteLaunch",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "086147fa-4c0d-4781-ad27-0a3725f03f2b": EfiGuid(
        "086147fa-4c0d-4781-ad27-0a3725f03f2b", "DxeDg2OpregionInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0872e0a5-6d38-4567-964c-7a3b48053a4f": EfiGuid(
        "0872e0a5-6d38-4567-964c-7a3b48053a4f", "FdmInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08747359-ba88-42f8-9d25-f71d93b6cd26": EfiGuid(
        "08747359-ba88-42f8-9d25-f71d93b6cd26", "EcFlashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "087568d7-5a70-4ddc-84fd-92e358e7bf2c": EfiGuid(
        "087568d7-5a70-4ddc-84fd-92e358e7bf2c", "NvmePciHcPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "087e9760-1fb5-49f7-879a-853d2b214cc7": EfiGuid(
        "087e9760-1fb5-49f7-879a-853d2b214cc7", "LENOVO_SLP2_DXE_GUID",
        description = "",
        aliases = ['LenovoSlp2Dxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08804377-af7a-4496-8a7b-175900e9ab46": EfiGuid(
        "08804377-af7a-4496-8a7b-175900e9ab46", "EDKII_REDFISH_CREDENTIAL_PROTOCOL_GUID",
        description = "",
        aliases = ['gEdkIIRedfishCredentialProtocolGuid', 'EdkIIRedfishCredentialProtocol'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/RedfishPkg/Include/Protocol/EdkIIRedfishCredential.h",
            "https://github.com/tianocore/edk2/blob/master/RedfishPkg/RedfishPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0885f288-418c-4be1-a6af-8bad61da08fe": EfiGuid(
        "0885f288-418c-4be1-a6af-8bad61da08fe", "AMITSE_DRIVER_HEALTH_ENB_GUID",
        description = "",
        aliases = ['AmiTseDriverHealthEnb'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "088e4724-e941-4cbd-9df3-158b3eacc450": EfiGuid(
        "088e4724-e941-4cbd-9df3-158b3eacc450", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "08a2ca63-3b65-472c-874e-5e138e947324": EfiGuid(
        "08a2ca63-3b65-472c-874e-5e138e947324", "AsusEcDxeRt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08a92691-926c-414f-a0b5-e785d90b9853": EfiGuid(
        "08a92691-926c-414f-a0b5-e785d90b9853", "Memory_OK_DXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08abe065-c359-4b95-8d59-c1b58eb657b5": EfiGuid(
        "08abe065-c359-4b95-8d59-c1b58eb657b5", "DELL_INTEL_LOM_SMM_GUID",
        description = "",
        aliases = ['IntelLomSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08b2e586-35a8-4a3f-a9ed-e48134891601": EfiGuid(
        "08b2e586-35a8-4a3f-a9ed-e48134891601", "SmcInBandSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08b97689-86af-4a36-9e35-117b4d2ef26a": EfiGuid(
        "08b97689-86af-4a36-9e35-117b4d2ef26a", "AfscDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08bb308d-c7d8-4d7b-b4d4-bc4bb53768a6": EfiGuid(
        "08bb308d-c7d8-4d7b-b4d4-bc4bb53768a6", "OemVariableInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08bf43a1-cd5f-44e4-8275-7d3b79228fa3": EfiGuid(
        "08bf43a1-cd5f-44e4-8275-7d3b79228fa3", "Lpit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['Lpit', 'WrapperAcpi'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '3dbf73930867371066435e0908e50535', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'b218fecd176e986c1491046e58c9e01b', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "08c8bae1-92a6-49d5-ffff-ffffba9d8dc2": EfiGuid(
        "08c8bae1-92a6-49d5-ffff-ffffba9d8dc2", "SplashDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08cb0301-f702-4f88-a1a2-b51c2d99e80c": EfiGuid(
        "08cb0301-f702-4f88-a1a2-b51c2d99e80c", "SioCashDrawerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08dd466b-69fb-4c41-90b2-551161c5ecac": EfiGuid(
        "08dd466b-69fb-4c41-90b2-551161c5ecac", "AmiAgesaChipsetDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08e56a30-62ed-41c6-9240-b7455ee653d7": EfiGuid(
        "08e56a30-62ed-41c6-9240-b7455ee653d7", "DELL_LEGACY_VIDEO_ROM_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08efd15d-ec55-4023-b648-7ba40df7d05d": EfiGuid(
        "08efd15d-ec55-4023-b648-7ba40df7d05d", "PeiRamBoot",
        description = "",
        aliases = ['PeiRamBootPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['PeiRamBoot', 'PeiRamBootPei'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fb0d5db7a0d624b39b616db11024c0a',
            '0fc6e822ed386a76a1ac3b30c3afd4d4', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '3dbf73930867371066435e0908e50535', '3fa6831c92f133013c5bb0942f307f72',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '6f7959935a87f450b05674b1e62ba7fa',
            '73c9d09abcd1f4bf7ddbfe437773286e', '7457f65a90b756398ae360b93c326a5e',
            '756c574dcde55419446fb64425ebd1ff', '7d13012b3ccc7c61a0c88ca8969676f6',
            '81460325c53ef849a2e3c2a5e104304b', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9e80820994976fe9edc2b53346ea4f71', 'ac2383f4616ea50d075c37b988c3cb83',
            'b218fecd176e986c1491046e58c9e01b', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd4a0cd284c2f726df5db17c605809651', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "08f2c63b-08de-4ccd-8670-acfe644a1c48": EfiGuid(
        "08f2c63b-08de-4ccd-8670-acfe644a1c48", "DELL_PCH_S3_SUPPORT_GUID",
        description = "",
        aliases = ['PchS3Support'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DELL_PCH_S3_SUPPORT_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['PchS3Support']),

    "08f4e928-0f5f-46d4-8410-479fda279db6": EfiGuid(
        "08f4e928-0f5f-46d4-8410-479fda279db6", "SSA_BIOS_RESULTS_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08f74baa-ea36-41d9-9521-21a70f8780bc": EfiGuid(
        "08f74baa-ea36-41d9-9521-21a70f8780bc", "EFI_DISK_INFO_SCSI_INTERFACE_GUID",
        description = "",
        aliases = ['EfiDiskInfoScsiInterfaceGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DiskInfo.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "08fa7ec9-60d4-47c8-b299-9f83b4a9dac3": EfiGuid(
        "08fa7ec9-60d4-47c8-b299-9f83b4a9dac3", "AcpiOverride",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "08fbd32f-867f-452b-b134-705161f73333": EfiGuid(
        "08fbd32f-867f-452b-b134-705161f73333", "LENOVO_CRYPT_SERVICE_SMM_GUID",
        description = "",
        aliases = ['LenovoCryptServiceSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0903dd14-2ca0-458a-b5eb-0c0ca30d785c": EfiGuid(
        "0903dd14-2ca0-458a-b5eb-0c0ca30d785c", "EFI_SMBIOS_BOARD_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiSmbiosBoardProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_SMBIOS_BOARD_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0914de08-434d-4f9f-93b3-6fa48aeaf7e0": EfiGuid(
        "0914de08-434d-4f9f-93b3-6fa48aeaf7e0", "DELL_SPI_PART_MICRON_GUID",
        description = "",
        aliases = ['DellSpiPartMicron'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0916e322-3740-31ce-ad62-bd172cecca36": EfiGuid(
        "0916e322-3740-31ce-ad62-bd172cecca36", "ASROCK_RAID_SETUP_GUID",
        description = "",
        aliases = ['AsrockRaidSetup'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "091aaea3-09a9-40f6-9864-0139388a9f8c": EfiGuid(
        "091aaea3-09a9-40f6-9864-0139388a9f8c", "XhciPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0922e604-f5ec-42ef-980d-a35e9a2b1844": EfiGuid(
        "0922e604-f5ec-42ef-980d-a35e9a2b1844", "IpSecConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09286371-5ff2-4065-b3d2-b81c882898ab": EfiGuid(
        "09286371-5ff2-4065-b3d2-b81c882898ab", "HpIcicleSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0939a4cc-dc2b-48a0-909b-30994d39d82c": EfiGuid(
        "0939a4cc-dc2b-48a0-909b-30994d39d82c", "ReportStatusCodeRouterSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "093e0245-bda6-48c7-a0b4-c3c9bfb87c79": EfiGuid(
        "093e0245-bda6-48c7-a0b4-c3c9bfb87c79", "WLANDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "093e0fae-a6c4-4f50-9f1b-d41e2b89c19a": EfiGuid(
        "093e0fae-a6c4-4f50-9f1b-d41e2b89c19a", "EFI_CERT_SHA512_GUID",
        description = "",
        aliases = ['EfiCertSha512Guid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/ImageAuthentication.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "094325e0-4da2-4912-990b-d6db19077573": EfiGuid(
        "094325e0-4da2-4912-990b-d6db19077573", "HeciAccessDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "094788da-6ffa-4031-b435-ff9623656034": EfiGuid(
        "094788da-6ffa-4031-b435-ff9623656034", "PspDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0948381e-0b3d-43a9-8e03-f2489f2b9e2e": EfiGuid(
        "0948381e-0b3d-43a9-8e03-f2489f2b9e2e", "FchSmmDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09576e91-6d3f-11d2-8e39-00a0c969723b": EfiGuid(
        "09576e91-6d3f-11d2-8e39-00a0c969723b", "EFI_DEVICE_PATH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDevicePathProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Protocol/DevicePath.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DevicePath.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_DEVICE_PATH_PROTOCOL_GUID'],
        hashes = ['82060a1dae992f24609d2f673cb58cfb', 'b4fa3a413d748be6699112adc0c14b0d'],
        junk = []),

    "09576e92-6d3f-11d2-8e39-00a0c969723b": EfiGuid(
        "09576e92-6d3f-11d2-8e39-00a0c969723b", "EFI_FILE_INFO_ID",
        description = "",
        aliases = ['EFI_FILE_INFO_IDEFI_GUID', 'EfiFileInfoGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://github.com/torvalds/linux/blob/master/include/linux/efi.h",
        ],
        vendors = ['EDK2', 'Linux'],
        filenames = [],
        hashes = [],
        junk = ['EfiFileInfo', 'EfiFileInfoGuid']),

    "09576e93-6d3f-11d2-8e39-00a0c969723b": EfiGuid(
        "09576e93-6d3f-11d2-8e39-00a0c969723b", "EFI_FILE_SYSTEM_INFO_ID_GUID",
        description = "",
        aliases = ['EfiFileSystemInfoGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiFileSystemInfo', 'EfiFileSystemInfoGuid']),

    "095e3853-2f3d-4061-bbdf-0ec0d478fd16": EfiGuid(
        "095e3853-2f3d-4061-bbdf-0ec0d478fd16", "OemServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "096111bf-b355-4180-ba39-737153339f0e": EfiGuid(
        "096111bf-b355-4180-ba39-737153339f0e", "ServiceBatteryRemoval",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09633ad2-2a5e-4562-90a8-24c673c4097c": EfiGuid(
        "09633ad2-2a5e-4562-90a8-24c673c4097c", "AmdMemFp8Pei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09767db6-412a-45ba-8026-f087cae210e3": EfiGuid(
        "09767db6-412a-45ba-8026-f087cae210e3", "DtsDxePolicyInit_FILE_GUID",
        description = "",
        aliases = ['DTSPolicy'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "097cee1c-a7f0-40cb-ab0d-f0774ae90e2b": EfiGuid(
        "097cee1c-a7f0-40cb-ab0d-f0774ae90e2b", "FjSetupPower",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09813137-b2a5-4462-8a2a-48f77eca31bf": EfiGuid(
        "09813137-b2a5-4462-8a2a-48f77eca31bf", "SmbiosDataUpdateDxeExpertWorkStationRP",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09831032-6fa3-4484-af4f-0a000a8d3a82": EfiGuid(
        "09831032-6fa3-4484-af4f-0a000a8d3a82", "PL180MciDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0983616a-49bc-4732-b531-4af98d2056f0": EfiGuid(
        "0983616a-49bc-4732-b531-4af98d2056f0", "EarlyFdtPL011SerialPortLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/Library/FdtPL011SerialPortLib/EarlyFdtPL011SerialPortLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "098515fc-9f73-4be4-b542-d38be9e91e96": EfiGuid(
        "098515fc-9f73-4be4-b542-d38be9e91e96", "FlashUtilitySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "098d0689-4245-4f65-80c9-7f3202c5f44e": EfiGuid(
        "098d0689-4245-4f65-80c9-7f3202c5f44e", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "098ed901-c6bf-4d20-91e2-a39f0c084a8d": EfiGuid(
        "098ed901-c6bf-4d20-91e2-a39f0c084a8d", "QuickSpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0996199f-2ce2-4d97-830b-077a7b28588a": EfiGuid(
        "0996199f-2ce2-4d97-830b-077a7b28588a", "WpbtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0997e770-756e-4139-bf3f-7fc2a308177e": EfiGuid(
        "0997e770-756e-4139-bf3f-7fc2a308177e", "DELL_SMM_S3_GUID",
        description = "",
        aliases = ['DellSmmS3'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "099cfa27-134f-449a-af4a-d29c441cb521": EfiGuid(
        "099cfa27-134f-449a-af4a-d29c441cb521", "AsusAmlBuffDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "099e1f39-0102-43c3-8cec-8e897b6562f7": EfiGuid(
        "099e1f39-0102-43c3-8cec-8e897b6562f7", "EventLogSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "099fd87f-4b39-43f6-ab47-f801f99209f7": EfiGuid(
        "099fd87f-4b39-43f6-ab47-f801f99209f7", "DELL_DCP_REGISTER_GUID",
        description = "",
        aliases = ['DellDcpRegister'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09a2b815-be29-45ef-bfbf-58eaac5e2978": EfiGuid(
        "09a2b815-be29-45ef-bfbf-58eaac5e2978", "INTERRUPT_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09a9d5ac-5204-4214-96e5-94992e752bcd": EfiGuid(
        "09a9d5ac-5204-4214-96e5-94992e752bcd", "gEfiEventNotificationTypePeiGuid",
        description = "",
        aliases = ['EfiEventNotificationTypePei'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "09b17d99-bb07-49a8-b0d2-06d6afcbe3ab": EfiGuid(
        "09b17d99-bb07-49a8-b0d2-06d6afcbe3ab", "OmapDmaLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09b5d8da-9a25-47af-a810-45a6d549e3f7": EfiGuid(
        "09b5d8da-9a25-47af-a810-45a6d549e3f7", "FjI2CPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09b8ba84-3dd3-49a6-a05a-3134a5f07bad": EfiGuid(
        "09b8ba84-3dd3-49a6-a05a-3134a5f07bad", "ARM_SCMI_PERFORMANCE_PROTOCOL_GUID",
        description = "",
        aliases = ['gArmScmiPerformanceProtocolGuid', 'ArmScmiPerformanceProtocol'],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/ArmPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "09b98059-2d74-43c5-9709-ad0a0051867d": EfiGuid(
        "09b98059-2d74-43c5-9709-ad0a0051867d", "DELL_AHCI_DXE_GUID",
        description = "",
        aliases = ['DellAhciDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09bd2ed8-c485-4d9c-b9a1-7b064c0309d2": EfiGuid(
        "09bd2ed8-c485-4d9c-b9a1-7b064c0309d2", "UsbSxCallback",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09c960b3-da83-47be-bbc4-81267ba9b002": EfiGuid(
        "09c960b3-da83-47be-bbc4-81267ba9b002", "FlashBootFlag",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09ce2c2d-4d83-44b8-97d0-1085eb5490b8": EfiGuid(
        "09ce2c2d-4d83-44b8-97d0-1085eb5490b8", "ASROCK_RAID_LOADER_GUID",
        description = "",
        aliases = ['AsrockRaidLoader'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09cefa99-0d07-487f-a651-fb44f094b1c7": EfiGuid(
        "09cefa99-0d07-487f-a651-fb44f094b1c7", "SP804TimerLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09d13410-2718-463a-8b15-da93c3ca0a64": EfiGuid(
        "09d13410-2718-463a-8b15-da93c3ca0a64", "UsbCdcEcm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09d445be-3c89-4e4f-abe0-51fa84c2e4ff": EfiGuid(
        "09d445be-3c89-4e4f-abe0-51fa84c2e4ff", "ScGeneralConfigGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09d5b53f-f4b0-4f59-a0b1-7b57d35c0e05": EfiGuid(
        "09d5b53f-f4b0-4f59-a0b1-7b57d35c0e05", "EFI_NIC_IP4_CONFIG_NVDATA_GUID",
        description = "",
        aliases = ['NicIp4ConfigNvDataGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09df88e2-0e05-4f28-a55f-0d62bfbcf59f": EfiGuid(
        "09df88e2-0e05-4f28-a55f-0d62bfbcf59f", "AmdFabricRvPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09ea8911-be0d-4230-a003-edc693b48e11": EfiGuid(
        "09ea8911-be0d-4230-a003-edc693b48e11", "SA_PEI_INIT_PPI_GUID",
        description = "",
        aliases = ['VlvPeiInitPpiGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SA_PEI_INIT_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['VlvPeiInitPpi', 'VlvPeiInitPpiGuid', 'VLV_PEI_INIT_PPI_GUID']),

    "09ea894a-be0d-4230-a003-edc693b48e95": EfiGuid(
        "09ea894a-be0d-4230-a003-edc693b48e95", "PCH_INIT_PPI_GUID",
        description = "",
        aliases = ['PchInitPpiGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09eb7679-2152-47f3-b832-ee9b947b29f5": EfiGuid(
        "09eb7679-2152-47f3-b832-ee9b947b29f5", "SmartUsbProtection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09ecc29d-dbbe-49fb-a649-4bf640e2ebd6": EfiGuid(
        "09ecc29d-dbbe-49fb-a649-4bf640e2ebd6", "SA_OVERCLOCKING_PRE_MEM_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "09ee81d3-f15e-43f4-85b4-cb9873da5d6b": EfiGuid(
        "09ee81d3-f15e-43f4-85b4-cb9873da5d6b", "ArmMmCommunication",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Drivers/MmCommunicationDxe/MmCommunication.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "09fdcb1e-e08b-4b64-890c-70e3174be07a": EfiGuid(
        "09fdcb1e-e08b-4b64-890c-70e3174be07a", "FpgaErrorRecord",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a056e30-b2c9-44f3-ffff-ffff94a524a6": EfiGuid(
        "0a056e30-b2c9-44f3-ffff-ffff94a524a6", "XnotePowerButtonSmiHandler",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a18976f-6ca6-405b-af4b-27b8f7f6db98": EfiGuid(
        "0a18976f-6ca6-405b-af4b-27b8f7f6db98", "AmiUsbIadProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a1c30d2-7821-4fd2-b3c1-d24fa4c84b6a": EfiGuid(
        "0a1c30d2-7821-4fd2-b3c1-d24fa4c84b6a", "DellCfgChangeReport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a1d4fd8-4704-4501-85eb-93399492cbed": EfiGuid(
        "0a1d4fd8-4704-4501-85eb-93399492cbed", "DevShell",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a21ac47-844f-4307-b265-ce4af3c17421": EfiGuid(
        "0a21ac47-844f-4307-b265-ce4af3c17421", "IntelGopVbtmipiHrFfd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['IntelGopVbtmipiHrFfd'],
        hashes = [
            '3555862f32138b911338fae3789c3921', '39a2ecdc1427b5f2c143fd59ef976f2c',
            '82060a1dae992f24609d2f673cb58cfb', 'b4fa3a413d748be6699112adc0c14b0d',
        ],
        junk = []),

    "0a249bb5-5918-45f5-b220-76a3b6c89529": EfiGuid(
        "0a249bb5-5918-45f5-b220-76a3b6c89529", "FjLvdsFwDownPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a24a50a-c148-42b6-9302-8ed31d334e73": EfiGuid(
        "0a24a50a-c148-42b6-9302-8ed31d334e73", "ASRockNetFtp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a2fbd15-1c25-407e-8915-60c5652bc2aa": EfiGuid(
        "0a2fbd15-1c25-407e-8915-60c5652bc2aa", "SystemFirmwareUpdateDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SignedCapsulePkg/Universal/SystemFirmwareUpdate/SystemFirmwareUpdateDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0a32a803-acdf-4c89-8293-91011548cd91": EfiGuid(
        "0a32a803-acdf-4c89-8293-91011548cd91", "MicrocodeMeasurementDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a457a6c-178e-438c-ac96-d9c0ec52be66": EfiGuid(
        "0a457a6c-178e-438c-ac96-d9c0ec52be66", "DellTcgPeiPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a496aac-8c56-4da0-a960-e417247a6183": EfiGuid(
        "0a496aac-8c56-4da0-a960-e417247a6183", "BroadcomLomSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a4d622d-01f4-4974-b3f5-2bfe9888ef92": EfiGuid(
        "0a4d622d-01f4-4974-b3f5-2bfe9888ef92", "SLP20OEMPublicKeyVariableGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a4fb373-359f-4a34-b354-ab7b2eb72552": EfiGuid(
        "0a4fb373-359f-4a34-b354-ab7b2eb72552", "AmdSocAm4SmPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a53b507-988b-475c-bf76-33de106d9484": EfiGuid(
        "0a53b507-988b-475c-bf76-33de106d9484", "PCIE_RP_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a58b5c0-0e01-429c-a8ad-eb7349a5cfa5": EfiGuid(
        "0a58b5c0-0e01-429c-a8ad-eb7349a5cfa5", "DellPlatformVTdPolicyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a5ea2e1-be0b-44a0-a775-f429c9a018a0": EfiGuid(
        "0a5ea2e1-be0b-44a0-a775-f429c9a018a0", "PlatformEarlyInit",
        description = "",
        aliases = ['PlatformInit'],
        urls = [],
        vendors = [],
        filenames = ['PlatformEarlyInit'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3555862f32138b911338fae3789c3921',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "0a602c5b-05a0-40c4-9181-edcd891d0001": EfiGuid(
        "0a602c5b-05a0-40c4-9181-edcd891d0001", "OememDxeBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a602c5b-05a0-40c4-9181-edcd891d0002": EfiGuid(
        "0a602c5b-05a0-40c4-9181-edcd891d0002", "OememSmiBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a602c5b-05a0-40c4-9181-edcd891d0003": EfiGuid(
        "0a602c5b-05a0-40c4-9181-edcd891d0003", "OememPeiBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a66e322-3740-4cce-ad62-bd172cecca35": EfiGuid(
        "0a66e322-3740-4cce-ad62-bd172cecca35", "LENOVO_SYSTEM_SCSI_DISK_DXE_GUID",
        description = "",
        aliases = ['ScsiDisk'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Scsi/ScsiDiskDxe/ScsiDiskDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['LENOVO_SYSTEM_SCSI_DISK_DXE_GUID', 'ScsiDisk'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'bdbe423029a61611b078326db4953356',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['ScsiDisk']),

    "0a7521e1-09f1-44ae-9b5f-2781b78971d6": EfiGuid(
        "0a7521e1-09f1-44ae-9b5f-2781b78971d6", "DELL_SIMPLE_BOOT_FLAG_SRC_GUID",
        description = "",
        aliases = ['SimpleBootFlag'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a77e089-1590-4442-bbea-115e19e6358a": EfiGuid(
        "0a77e089-1590-4442-bbea-115e19e6358a", "EfiOsWdtPolicyProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a7a6fc0-ad10-445f-bcb0-704ad17cdb23": EfiGuid(
        "0a7a6fc0-ad10-445f-bcb0-704ad17cdb23", "Ucsi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a814161-de67-46e7-a813-b7f8f489afcd": EfiGuid(
        "0a814161-de67-46e7-a813-b7f8f489afcd", "UpdateSmbios",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a845224-8675-4241-8ae9-4d94c93deb90": EfiGuid(
        "0a845224-8675-4241-8ae9-4d94c93deb90", "PauseKey",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PauseKey'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '1ded807f42af76804c9a0bbb6e24a2a9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '633e80e7347dc01c21c5133227df2aba', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0a8badd5-03b8-4d19-b128-7b8f0edaa596": EfiGuid(
        "0a8badd5-03b8-4d19-b128-7b8f0edaa596", "EFI_CONFIG_KEYWORD_HANDLER_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiConfigKeywordHandlerProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/HiiConfigKeyword.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0a9db95a-0e4a-4816-8639-5be4ffa9c909": EfiGuid(
        "0a9db95a-0e4a-4816-8639-5be4ffa9c909", "AmdFabricPhxSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a9eafb0-faf9-48aa-91f9-6cca33061276": EfiGuid(
        "0a9eafb0-faf9-48aa-91f9-6cca33061276", "BctBaseSmmMDN",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0a9f795e-b9fc-4117-bc4e-138695d16562": EfiGuid(
        "0a9f795e-b9fc-4117-bc4e-138695d16562", "AmdSocFp5RvDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0aa10093-e97a-4e28-8e77-7cb37041b291": EfiGuid(
        "0aa10093-e97a-4e28-8e77-7cb37041b291", "DeviceBlacklistWmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0aa31bc6-3379-41e8-825a-53f82cc0f254": EfiGuid(
        "0aa31bc6-3379-41e8-825a-53f82cc0f254", "TPM32BIN",
        description = "",
        aliases = ['AmiTpm32Bin'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['TPM32BIN'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f3a7c093df5e8da1491e4b41973919a7',
            'f8c34791a7a721ee330771cb43d68b91', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0ab697ce-b920-48ac-a265-ec5624edcdd7": EfiGuid(
        "0ab697ce-b920-48ac-a265-ec5624edcdd7", "DellDevPasswordProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0abba7dc-e516-4167-bbf5-4d9d1c739416": EfiGuid(
        "0abba7dc-e516-4167-bbf5-4d9d1c739416", "FU_EFIVAR_GUID_FWUPDATE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0abd8284-6da3-4616-971a-83a5148067ba": EfiGuid(
        "0abd8284-6da3-4616-971a-83a5148067ba", "IsaFloppyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ac12ab3-dd33-4460-87f5-82694d3d7402": EfiGuid(
        "0ac12ab3-dd33-4460-87f5-82694d3d7402", "SmcOFBDNull",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ac2d35d-1c77-1033-a6f8-7ca55df7d0aa": EfiGuid(
        "0ac2d35d-1c77-1033-a6f8-7ca55df7d0aa", "CpuPolicyPei",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CpuPolicyPei'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "0ac742bf-07c6-4c87-a87d-a3b2918afd00": EfiGuid(
        "0ac742bf-07c6-4c87-a87d-a3b2918afd00", "TileDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ac7d82c-9bad-419f-b8ff-b39ba18737fd": EfiGuid(
        "0ac7d82c-9bad-419f-b8ff-b39ba18737fd", "Rtk8852CEWifiSupDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ad3d31b-b3d8-4506-ae71-2ef11006d90f": EfiGuid(
        "0ad3d31b-b3d8-4506-ae71-2ef11006d90f", "UefiAcpiBoardInfo",
        description = "",
        aliases = ['UefiAcpiBoardInfoGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/UefiPayloadPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0ad4f13b-d197-44ec-ffff-ffff62c8c689": EfiGuid(
        "0ad4f13b-d197-44ec-ffff-ffff62c8c689", "XnoteVariableDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ad6c423-4732-4cf3-9ce3-0a5416d634a5": EfiGuid(
        "0ad6c423-4732-4cf3-9ce3-0a5416d634a5", "DxeRsa2048Sha256GuidedSectionExtractLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/DxeRsa2048Sha256GuidedSectionExtractLib/DxeRsa2048Sha256GuidedSectionExtractLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0adfb62d-ff74-484c-8944-f85c4bea87a8": EfiGuid(
        "0adfb62d-ff74-484c-8944-f85c4bea87a8", "AMI_EFIKEYCODE_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiEfiKeycodeProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ae01967-8b3b-4572-85bd-efd5c0f251eb": EfiGuid(
        "0ae01967-8b3b-4572-85bd-efd5c0f251eb", "AmdPspRomArmor3Smm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ae8ce5d-e448-4437-a8d7-ebf5f194f731": EfiGuid(
        "0ae8ce5d-e448-4437-a8d7-ebf5f194f731", "EFI_DXE_IPL_PPI_GUID",
        description = "",
        aliases = ['EfiDxeIplPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/DxeIpl.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0aef5579-3700-41d2-a53e-dabd55e22432": EfiGuid(
        "0aef5579-3700-41d2-a53e-dabd55e22432", "DellUsbBusSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0af0b742-63ec-45bd-8db6-71ad7f2fe8e8": EfiGuid(
        "0af0b742-63ec-45bd-8db6-71ad7f2fe8e8", "SHELL_DRIVER1_HII_GUID",
        description = "",
        aliases = ['ShellDriver1HiiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellLibHiiGuid.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0af7c79c-65b5-4319-b0ae-44ec484e4ad7": EfiGuid(
        "0af7c79c-65b5-4319-b0ae-44ec484e4ad7", "EFI_HASH_ALGORTIHM_MD5_GUID",
        description = "",
        aliases = ['EfiHashAlgorithmMD5Guid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Hash.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiHashAlgorithmMD5', 'EfiHashAlgorithmMD5Guid']),

    "0afd2fa7-5ef2-4b3b-a65e-b6f276d082a1": EfiGuid(
        "0afd2fa7-5ef2-4b3b-a65e-b6f276d082a1", "AdvBootOptionPolicyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b04b2ed-861c-42cd-a22f-c3aafaccb896": EfiGuid(
        "0b04b2ed-861c-42cd-a22f-c3aafaccb896", "BiosVideoDxe",
        description = "",
        aliases = ['GraphicsOutputDxe'],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/GraphicsOutputDxe/GraphicsOutputDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['FbGop']),

    "0b06277e-3d59-4def-b2a4-74f509661046": EfiGuid(
        "0b06277e-3d59-4def-b2a4-74f509661046", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['1edacc01f9473c204c191bc39f10036f'],
        junk = []),

    "0b0d3c7e-473a-22e2-8428-3dae5cceac64": EfiGuid(
        "0b0d3c7e-473a-22e2-8428-3dae5cceac64", "CableDetectDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b1e645f-5e5c-48a1-ffff-ffff5804e747": EfiGuid(
        "0b1e645f-5e5c-48a1-ffff-ffff5804e747", "XnotePlatformResetRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b227b84-9466-46bd-8aba-8eea99fad22c": EfiGuid(
        "0b227b84-9466-46bd-8aba-8eea99fad22c", "AsusEcNvramDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b234df1-0080-412b-8716-8a81a3a11faa": EfiGuid(
        "0b234df1-0080-412b-8716-8a81a3a11faa", "DELL_EDIAGS_DXE_GUID",
        description = "",
        aliases = ['DellEdiagsDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b24790b-6c79-4ce4-8e09-3864c49be256": EfiGuid(
        "0b24790b-6c79-4ce4-8e09-3864c49be256", "SioWdtSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b280816-52e7-4e51-aa57-11bd41cbefc3": EfiGuid(
        "0b280816-52e7-4e51-aa57-11bd41cbefc3", "EFI_PCI_HOTPLUG_DEVICE_GUID",
        description = "",
        aliases = ['EfiPciHotplugDevice'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b2b4f68-3566-40fc-b7e6-fa819993840a": EfiGuid(
        "0b2b4f68-3566-40fc-b7e6-fa819993840a", "OemUniWillVariableDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b2cfbf2-3e08-4c4e-a74d-59748a9f930f": EfiGuid(
        "0b2cfbf2-3e08-4c4e-a74d-59748a9f930f", "LegacyRegionDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b2e406a-ed5b-4668-bcd1-2b1db01841fc": EfiGuid(
        "0b2e406a-ed5b-4668-bcd1-2b1db01841fc", "DELL_ACPI_OEM_CONFIG_GUID",
        description = "",
        aliases = ['DellAcpiOemConfig'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b3afdec-f177-4548-9c7e-4d09edf4a2fa": EfiGuid(
        "0b3afdec-f177-4548-9c7e-4d09edf4a2fa", "EcdSmartPowerOn",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b4ae6be-6da6-4908-8a71-7e6a8a33b11c": EfiGuid(
        "0b4ae6be-6da6-4908-8a71-7e6a8a33b11c", "MeFwVersion",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b4bdcff-74b2-45ad-91e1-8f6634c8a1dc": EfiGuid(
        "0b4bdcff-74b2-45ad-91e1-8f6634c8a1dc", "SIOBasicIOPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b590774-bc67-49f4-a7db-e82e89e6b5d6": EfiGuid(
        "0b590774-bc67-49f4-a7db-e82e89e6b5d6", "EfiPeiMpInitLibUpDepPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b64aab0-5429-11d4-9816-00a0c91fadcf": EfiGuid(
        "0b64aab0-5429-11d4-9816-00a0c91fadcf", "EFI_BIS_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiBisProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Bis.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0b6a8de0-6281-40a2-8e9a-09f1a18a6d40": EfiGuid(
        "0b6a8de0-6281-40a2-8e9a-09f1a18a6d40", "LENOVO_SYSTEM_USB_SWITCH_SMM_GUID",
        description = "",
        aliases = ['SystemUsbSwitchSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b6e5233-a65c-44c9-9407-d9ab83bfc8bd": EfiGuid(
        "0b6e5233-a65c-44c9-9407-d9ab83bfc8bd", "EFI_CERT_SHA224_GUID",
        description = "",
        aliases = ['EfiCertSha224Guid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/ImageAuthentication.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0b6f5ca7-4f53-445a-b76e-2e365b806366": EfiGuid(
        "0b6f5ca7-4f53-445a-b76e-2e365b806366", "ArmVirtualizationTokenSpaceGuid",
        description = "",
        aliases = ['ArmVirtTokenSpaceGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/ArmVirtPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['ArmVirtTokenSpace']),

    "0b74f980-2325-4af3-9664-0c3abe237d52": EfiGuid(
        "0b74f980-2325-4af3-9664-0c3abe237d52", "BootPerformanceLogSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b7bf76c-721a-44fb-a758-52e13ca457f1": EfiGuid(
        "0b7bf76c-721a-44fb-a758-52e13ca457f1", "DELL_SPI_PART_MXIC_GUID",
        description = "",
        aliases = ['DellSpiPartMxic'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b7c6357-a4f5-48b9-aa30-0ce7f9e66a72": EfiGuid(
        "0b7c6357-a4f5-48b9-aa30-0ce7f9e66a72", "RestoreVariablesSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b7e694d-b909-4097-9c03-5e728489f709": EfiGuid(
        "0b7e694d-b909-4097-9c03-5e728489f709", "HybridGraphicsConfigGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b81b13e-e595-4d35-9e7b-a3c85e01a23f": EfiGuid(
        "0b81b13e-e595-4d35-9e7b-a3c85e01a23f", "HhmDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0b9303a2-5bf6-4725-8414-7edcd343d27d": EfiGuid(
        "0b9303a2-5bf6-4725-8414-7edcd343d27d", "RZ616_MtkWiFiDex",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ba38e92-154a-4be2-a5e0-be80571cdb71": EfiGuid(
        "0ba38e92-154a-4be2-a5e0-be80571cdb71", "DELL_DTWLAN_SMM_GUID",
        description = "",
        aliases = ['DellDtwlanSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ba38ebd-e190-4df7-8ec4-0a6e2b43772d": EfiGuid(
        "0ba38ebd-e190-4df7-8ec4-0a6e2b43772d", "LockBoxNullLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/LockBoxNullLib/LockBoxNullLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0ba8263c-a8bd-4aad-b402-6a6af2f7e77d": EfiGuid(
        "0ba8263c-a8bd-4aad-b402-6a6af2f7e77d", "BLOCK_DEVICE_RECOVERY_CAPSULE_GUID",
        description = "",
        aliases = ['BlockDeviceCapsuleGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ba8630d-b1f9-4ced-887d-dd013a0c6b4a": EfiGuid(
        "0ba8630d-b1f9-4ced-887d-dd013a0c6b4a", "UsbTypeCChargingPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0baed1b4-b14b-40f1-9d67-b8cd96f923e4": EfiGuid(
        "0baed1b4-b14b-40f1-9d67-b8cd96f923e4", "AsusTCONDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bb12cac-cfa9-4839-a4c2-1797248109d0": EfiGuid(
        "0bb12cac-cfa9-4839-a4c2-1797248109d0", "DELL_LEGACY16_ROM_GUID",
        description = "",
        aliases = ['DellLegacy16Rom'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bb51cdd-a750-442f-a15e-7deadfba30ff": EfiGuid(
        "0bb51cdd-a750-442f-a15e-7deadfba30ff", "Mtftp4Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bb60340-8865-4584-a32d-7f839903df4e": EfiGuid(
        "0bb60340-8865-4584-a32d-7f839903df4e", "MmioSerialUart1",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bbc6cab-f228-4f20-8c6a-847cdee3ff24": EfiGuid(
        "0bbc6cab-f228-4f20-8c6a-847cdee3ff24", "OFCDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bbc88de-17d1-4dba-ba54-734b01de421f": EfiGuid(
        "0bbc88de-17d1-4dba-ba54-734b01de421f", "OEMPlatformGopPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bc22608-ecc5-47b9-a038-4f0a68372e44": EfiGuid(
        "0bc22608-ecc5-47b9-a038-4f0a68372e44", "DELL_DEVICE_NAME_CONFIG_GUID",
        description = "",
        aliases = ['DellDeviceNameConfig'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bc7e05c-c88a-4254-bcbb-d42638d28aad": EfiGuid(
        "0bc7e05c-c88a-4254-bcbb-d42638d28aad", "PlatformMilestoneSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bcaf6f4-5a78-4818-82bc-5594d175943c": EfiGuid(
        "0bcaf6f4-5a78-4818-82bc-5594d175943c", "FchSmbusPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bcb2808-fea5-6047-9a60-8767a46a72a1": EfiGuid(
        "0bcb2808-fea5-6047-9a60-8767a46a72a1", "AppleUdp4Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bd7ebbf-f1a9-4c0b-9941-636d102aa48a": EfiGuid(
        "0bd7ebbf-f1a9-4c0b-9941-636d102aa48a", "FchBixbyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bda00b0-05d6-4bb8-bfc7-058ad13615cf": EfiGuid(
        "0bda00b0-05d6-4bb8-bfc7-058ad13615cf", "ArmCrashDumpDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Drivers/ArmCrashDumpDxe/ArmCrashDumpDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0be3d6ae-6d2c-41ad-9869-a97364e7c1f1": EfiGuid(
        "0be3d6ae-6d2c-41ad-9869-a97364e7c1f1", "FjIbvSfuSecFlashAbstractionSmmProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0be98320-5634-4aab-b2b7-fcbc9777151e": EfiGuid(
        "0be98320-5634-4aab-b2b7-fcbc9777151e", "FjTerraSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bf06027-29a4-447d-8fa4-d8a289f1dddc": EfiGuid(
        "0bf06027-29a4-447d-8fa4-d8a289f1dddc", "CxlManager",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bf13afb-98b6-4b77-96e3-0868917c2d2e": EfiGuid(
        "0bf13afb-98b6-4b77-96e3-0868917c2d2e", "I2cBusConfigurationManagement",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0bf70067-d53b-42df-b770-e92c91c61411": EfiGuid(
        "0bf70067-d53b-42df-b770-e92c91c61411", "EFI_TDT_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTdtProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_TDT_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '8add95e4f71aa9680d87091104649f78', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bdbe423029a61611b078326db4953356',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "0bf89093-3e5e-457a-8ca6-62352915a3b4": EfiGuid(
        "0bf89093-3e5e-457a-8ca6-62352915a3b4", "PHashDPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c008b0d-596a-42d1-809c-d2e7b649177e": EfiGuid(
        "0c008b0d-596a-42d1-809c-d2e7b649177e", "HwErrorHandler",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c04b0ff-227d-479a-935a-f6e5a8b5198c": EfiGuid(
        "0c04b0ff-227d-479a-935a-f6e5a8b5198c", "EfiPeiPlatformTypeLightningRidgeExecB2Ppi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c070ba5-3f07-4e9d-883f-057c505fe3b8": EfiGuid(
        "0c070ba5-3f07-4e9d-883f-057c505fe3b8", "BiosAuditLogWmiReporter",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c0ac8c1-e368-4d20-85fe-23efb3db094e": EfiGuid(
        "0c0ac8c1-e368-4d20-85fe-23efb3db094e", "IntelPchAcpiTimerLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c0f3b43-44de-4907-b478-225f6f6289dc": EfiGuid(
        "0c0f3b43-44de-4907-b478-225f6f6289dc", "USB_KEYBOARD_LAYOUT_PACKAGE_GUID",
        description = "",
        aliases = ['UsbKeyboardLayoutPackageGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/UsbKeyBoardLayout.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0c0fa9b6-f3af-49bd-89c5-6c2d43969072": EfiGuid(
        "0c0fa9b6-f3af-49bd-89c5-6c2d43969072", "MmioSerialUart",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AmiSerialUart0', 'MmioSerialUart'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '28e65e1a6689d8a2a4a612a06cbd2a58', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '409221b30602fc13c8cbedf952c60297',
            '633e80e7347dc01c21c5133227df2aba', '811505684bd99c18f5ecbee1f725ff66',
            '82060a1dae992f24609d2f673cb58cfb', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b6cce665f85751125c3db2050ffc98d2',
            'cae37d13982bb26c35ffaada2841d171', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'dab68811108b555173f8f096c9737d20', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "0c109319-c149-450e-a3e3-b9badd9dc3a4": EfiGuid(
        "0c109319-c149-450e-a3e3-b9badd9dc3a4", "EfiPeiMmConfigurationPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c1f6580-9c57-4a32-8499-227c2ea27e7d": EfiGuid(
        "0c1f6580-9c57-4a32-8499-227c2ea27e7d", "DellSmmServicesProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c2070f9-02a7-4e8e-b584-84ea922363fb": EfiGuid(
        "0c2070f9-02a7-4e8e-b584-84ea922363fb", "PeiNt32OemHookStatusCodeLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c215b39-425d-42e0-a1b4-367d574956ec": EfiGuid(
        "0c215b39-425d-42e0-a1b4-367d574956ec", "FjDmiSystemData",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c2c4003-6551-4eea-b006-0fecb4bb300b": EfiGuid(
        "0c2c4003-6551-4eea-b006-0fecb4bb300b", "RTL8111E",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c34b372-2622-4a13-a46e-bfd0deb48bff": EfiGuid(
        "0c34b372-2622-4a13-a46e-bfd0deb48bff", "I2cBusDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/I2c/I2cDxe/I2cBusDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0c375a90-4c4c-4428-8ea0-531be8959bf7": EfiGuid(
        "0c375a90-4c4c-4428-8ea0-531be8959bf7", "FlashDriverSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['FlashDriverSmm'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0c3798aa-d22e-4dd6-8cb4-3c9cfb1d86d3": EfiGuid(
        "0c3798aa-d22e-4dd6-8cb4-3c9cfb1d86d3", "ASROCK_SIOSLPSMI_GUID",
        description = "",
        aliases = ['NCT6126DSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c396fca-6bda-4a15-b6a3-a6fa4544bdb7": EfiGuid(
        "0c396fca-6bda-4a15-b6a3-a6fa4544bdb7", "LENOVO_EC_FW_UPDATE_DXE_GUID",
        description = "",
        aliases = ['EcFwUpdateDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c3b7b59-28e5-4c99-85e5-d0116dbfaaf2": EfiGuid(
        "0c3b7b59-28e5-4c99-85e5-d0116dbfaaf2", "IsctWakeReason",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['IsctWakeReason'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '186cc72a8ef4d9cb1062b2cea44ef726',
            '3dbf73930867371066435e0908e50535', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', 'b21bd44978f0935538daa78003d23135',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cd7db3eb49885c95779ed44627512a82',
            'd09f3cd0ba6435ccf1611206ed293c92', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "0c4d10cf-d949-49b4-bd13-47a4ae22efce": EfiGuid(
        "0c4d10cf-d949-49b4-bd13-47a4ae22efce", "QemuVirtMemInfoPeiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/Library/QemuVirtMemInfoLib/QemuVirtMemInfoPeiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0c4ee8ac-4bcb-43b4-9f05-e07523a9fc97": EfiGuid(
        "0c4ee8ac-4bcb-43b4-9f05-e07523a9fc97", "AfterMemoryDummyDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AfterMemoryDummyDriver'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '7f662f93e41d560c5fb667dce9000eb7', '869f701de5296248126c8c8a03a479c3',
            '953dc1131b7a579176b9fc937eee5846', 'b49cea81c680d9848212ed2aca8e6771',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189',
        ],
        junk = []),

    "0c4f81c5-309f-4941-85a2-8c6a44b4f4f6": EfiGuid(
        "0c4f81c5-309f-4941-85a2-8c6a44b4f4f6", "AmdVariableProtectionSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c556bff-b16a-439d-a3ec-1164378e2c2a": EfiGuid(
        "0c556bff-b16a-439d-a3ec-1164378e2c2a", "AmdPspPeiV2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c5fce90-1c03-4ed2-9efe-b1d02e72b3b0": EfiGuid(
        "0c5fce90-1c03-4ed2-9efe-b1d02e72b3b0", "menu_bottom_mid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c8050b7-d2f0-42f8-852b-d1deb2f2e9b0": EfiGuid(
        "0c8050b7-d2f0-42f8-852b-d1deb2f2e9b0", "SetupIoDetect",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c80c256-3428-4d1c-bbeb-79fcc812aff6": EfiGuid(
        "0c80c256-3428-4d1c-bbeb-79fcc812aff6", "RestorePasswordSMI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c81d263-7ce3-4f61-a175-fda4fd240985": EfiGuid(
        "0c81d263-7ce3-4f61-a175-fda4fd240985", "HpAltModePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c85df8a-4212-4cc1-a4aa-1a7a36e8da97": EfiGuid(
        "0c85df8a-4212-4cc1-a4aa-1a7a36e8da97", "FpgaPlatformEarlyInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c8823d5-93fb-485c-aafa-fa7b6bf0a27f": EfiGuid(
        "0c8823d5-93fb-485c-aafa-fa7b6bf0a27f", "CompalEepromSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c88b3b5-6a1d-4657-aa88-1b7d92ff3699": EfiGuid(
        "0c88b3b5-6a1d-4657-aa88-1b7d92ff3699", "CompalCMFCCommonSwSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c88f1b7-dd08-4e29-be2d-d04688a42c74": EfiGuid(
        "0c88f1b7-dd08-4e29-be2d-d04688a42c74", "OemKbLightDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c95a916-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a916-a006-11d4-bcfa-0080c73c8881", "WinNtThunkDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c95a928-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a928-a006-11d4-bcfa-0080c73c8881", "EFI_WIN_NT_VIRTUAL_DISKS_GUID",
        description = "",
        aliases = ['EfiWinNtVirtualDisksGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c95a92f-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a92f-a006-11d4-bcfa-0080c73c8881", "EFI_WIN_NT_PHYSICAL_DISKS_GUID",
        description = "",
        aliases = ['EfiWinNtPhysicalDisksGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c95a935-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a935-a006-11d4-bcfa-0080c73c8881", "EFI_WIN_NT_FILE_SYSTEM_GUID",
        description = "",
        aliases = ['EfiWinNtFileSystemGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c95a93d-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a93d-a006-11d4-bcfa-0080c73c8881", "EFI_WIN_NT_SERIAL_PORT_GUID",
        description = "",
        aliases = ['LOCAL_EFI_WIN_NT_SERIAL_PORT_GUID', 'EfiWinNtSerialPortGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Library/UefiHandleParsingLib/UefiHandleParsingLib.c",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['LOCAL_EFI_WIN_NT_SERIAL_PORT_GUID']),

    "0c95a940-a006-11d4-bcfa-0080c73c8881": EfiGuid(
        "0c95a940-a006-11d4-bcfa-0080c73c8881", "WinNtBusDriverDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c989d41-f4b4-4244-9d7f-e9ffb4163273": EfiGuid(
        "0c989d41-f4b4-4244-9d7f-e9ffb4163273", "FastBootOption",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0c9c3169-6f39-409e-990b-5b8b48b9d2b9": EfiGuid(
        "0c9c3169-6f39-409e-990b-5b8b48b9d2b9", "SyncUpNvramToPostFlag",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ca970e1-43fa-4402-bc0a-81af336bffd6": EfiGuid(
        "0ca970e1-43fa-4402-bc0a-81af336bffd6", "DxeImageVerificationLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/DxeImageVerificationLib/DxeImageVerificationLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0cabb327-11fe-416b-ae80-2de5df60f77d": EfiGuid(
        "0cabb327-11fe-416b-ae80-2de5df60f77d", "IioPolicyHob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cacec37-b79b-4a78-b3c5-ea61eaf6980d": EfiGuid(
        "0cacec37-b79b-4a78-b3c5-ea61eaf6980d", "BiosInitToFactorySupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cb84598-824b-4833-b705-bdefdcbba1d5": EfiGuid(
        "0cb84598-824b-4833-b705-bdefdcbba1d5", "BaseCryptLibUnitTestHost",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/CryptoPkg/Test/UnitTest/Library/BaseCryptLib/TestBaseCryptLibHost.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0cc252d2-c106-4661-b5bd-3147a4f81f92": EfiGuid(
        "0cc252d2-c106-4661-b5bd-3147a4f81f92", "EFI_PRINT2S_PROTOCOL_GUID",
        description = "",
        aliases = ['gEfiPrint2SProtocolGuid', 'EfiPrint_2sProtocol'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/Print2.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0cc2c3fc-41f0-4830-acf7-0ba8ec78045a": EfiGuid(
        "0cc2c3fc-41f0-4830-acf7-0ba8ec78045a", "SpiAccessPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cca1898-d936-4386-872b-734fe850dc49": EfiGuid(
        "0cca1898-d936-4386-872b-734fe850dc49", "FjIbvSfuOverrideAbstractionDxeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ccf27cf-e68d-4279-96b0-8a4e1cdff10c": EfiGuid(
        "0ccf27cf-e68d-4279-96b0-8a4e1cdff10c", "SettingsManagerDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cd3d2fa-b21d-47b5-ba95-d8c00b38f324": EfiGuid(
        "0cd3d2fa-b21d-47b5-ba95-d8c00b38f324", "DELL_AI13_ROM_GUID",
        description = "",
        aliases = ['DellAi13Rom'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cd80a60-46d9-48b3-a8b9-61e5598ab054": EfiGuid(
        "0cd80a60-46d9-48b3-a8b9-61e5598ab054", "MultiPdtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cd9632b-a19c-4488-8917-4e1c8e75dc1a": EfiGuid(
        "0cd9632b-a19c-4488-8917-4e1c8e75dc1a", "ncm865x64",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cda5d94-951a-4c61-8dd5-e5bf34ba69ec": EfiGuid(
        "0cda5d94-951a-4c61-8dd5-e5bf34ba69ec", "SlotDataUpdateDxeLightningRidgeEXECB1",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ce47609-0e04-4ee5-a192-f410a1995e0a": EfiGuid(
        "0ce47609-0e04-4ee5-a192-f410a1995e0a", "b57undix64",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ce5ea44-bed3-49ca-aecc-b50ab9c5b0e9": EfiGuid(
        "0ce5ea44-bed3-49ca-aecc-b50ab9c5b0e9", "DellRecoverySiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cea87a5-e257-4ca9-aec4-6956f3099956": EfiGuid(
        "0cea87a5-e257-4ca9-aec4-6956f3099956", "CPU_POWER_MGMT_PSYS_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cefd645-588e-4ed7-9943-054808cd7572": EfiGuid(
        "0cefd645-588e-4ed7-9943-054808cd7572", "SiFirmwareVersionDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0cf7aee6-3bd4-48a3-97e2-e7e5b83f67fd": EfiGuid(
        "0cf7aee6-3bd4-48a3-97e2-e7e5b83f67fd", "CbsBasePeiRS",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d024887-b489-4e9f-884b-a4b6bb8a576b": EfiGuid(
        "0d024887-b489-4e9f-884b-a4b6bb8a576b", "ODMCompuTraceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d05c550-ac28-46d8-8ae6-da2bfb2cb548": EfiGuid(
        "0d05c550-ac28-46d8-8ae6-da2bfb2cb548", "USB_LEGACY_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d05d4db-4a1d-4dcd-89b5-87b36c6dd180": EfiGuid(
        "0d05d4db-4a1d-4dcd-89b5-87b36c6dd180", "SDEmmcInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d1b9c8e-f77b-4632-8343-91f43d9a8560": EfiGuid(
        "0d1b9c8e-f77b-4632-8343-91f43d9a8560", "EfiCpuHtCapable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d1ce46b-72d9-4ba7-95da-23511865e661": EfiGuid(
        "0d1ce46b-72d9-4ba7-95da-23511865e661", "CryptoPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/CryptoPkg/Driver/CryptoPei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0d1ed2f7-e92b-4562-92dd-5c82ec917eae": EfiGuid(
        "0d1ed2f7-e92b-4562-92dd-5c82ec917eae", "_CRB_PEI_BUID",
        description = "",
        aliases = ['CrbPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CrbPei', '_CRB_PEI_BUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0ebecbce655f6f780edb7d73c1d80d13', '0fb0d5db7a0d624b39b616db11024c0a',
            '0fc6e822ed386a76a1ac3b30c3afd4d4', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '3dbf73930867371066435e0908e50535', '3fa6831c92f133013c5bb0942f307f72',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f7959935a87f450b05674b1e62ba7fa', '73c9d09abcd1f4bf7ddbfe437773286e',
            '7457f65a90b756398ae360b93c326a5e', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d13012b3ccc7c61a0c88ca8969676f6', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7a706c2421a55cde70e476d1d37f831',
            'ea61df93fb3257dc78c2415817c1f871', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0d1ed2f7-e92b-4562-92dd-5c82ec917ebf": EfiGuid(
        "0d1ed2f7-e92b-4562-92dd-5c82ec917ebf", "EcsPlatformPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "0d22d223-e9a7-00c5-be84-3fcaf25ec1a7": EfiGuid(
        "0d22d223-e9a7-00c5-be84-3fcaf25ec1a7", "InstallMadtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d244df9-6ce3-4133-a1cf-53200ab663ac": EfiGuid(
        "0d244df9-6ce3-4133-a1cf-53200ab663ac", "FspsWrapperPeim",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/IntelFsp2WrapperPkg/FspsWrapperPeim/FspsWrapperPeim.inf'],
        vendors = ['EDK2'],
        filenames = ['FspsWrapperPeim'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '60785be6bfb0d1daa94a8bb214b774d1', '7a20bc8c8925db2963205a36be03b335',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'eee15c9794a4040563b41004ebccc91a',
        ],
        junk = []),

    "0d24a235-9c12-446c-8ecb-efc1f5280209": EfiGuid(
        "0d24a235-9c12-446c-8ecb-efc1f5280209", "DELL_JPEG_DECODER_GUID",
        description = "",
        aliases = ['DellJpegDecoder'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d275c72-10c6-4d8c-8b2c-6610e69e3403": EfiGuid(
        "0d275c72-10c6-4d8c-8b2c-6610e69e3403", "Uart16550SerialDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d28c529-87d4-4298-8a54-40f22a9fe24a": EfiGuid(
        "0d28c529-87d4-4298-8a54-40f22a9fe24a", "DELL_DA_HDD_PROTECTION_GUID",
        description = "",
        aliases = ['DellDaHddProtection'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d2b85c0-d468-4adb-93bf-e53ae203c034": EfiGuid(
        "0d2b85c0-d468-4adb-93bf-e53ae203c034", "OemErpPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d3fb176-9569-4d51-a3ef-7d61c64feaba": EfiGuid(
        "0d3fb176-9569-4d51-a3ef-7d61c64feaba", "SECURITYPKG_TOKEN_SPACE_GUID",
        description = "",
        aliases = ['EfiSecurityPkgTokenSpaceGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/Include/Guid/SecurityPkgTokenSpace.h",
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/SecurityPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiSecurityPkgTokenSpace', 'EfiSecurityPkgTokenSpaceGuid']),

    "0d42e9ab-5dde-4cac-a485-0215c252717f": EfiGuid(
        "0d42e9ab-5dde-4cac-a485-0215c252717f", "AmdSmmControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d4bbf18-c2cc-4c23-bd63-bfdad4c710d0": EfiGuid(
        "0d4bbf18-c2cc-4c23-bd63-bfdad4c710d0", "Tcg2Acpi",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Tcg/Tcg2Acpi/Tcg2Acpi.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0d510a4f-a81b-473f-8707-b7fdfbc045ba": EfiGuid(
        "0d510a4f-a81b-473f-8707-b7fdfbc045ba", "OPAL_PASSWORD_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Tcg/Opal/OpalPassword/OpalHii.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0d512092-b3ca-4443-8a1f-8fa1a651a774": EfiGuid(
        "0d512092-b3ca-4443-8a1f-8fa1a651a774", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['3b120a77bbb1ca0feff2532e35537452'],
        junk = []),

    "0d51905b-b77e-452a-a2c0-eca0cc8d514a": EfiGuid(
        "0d51905b-b77e-452a-a2c0-eca0cc8d514a", "EDKII_NON_DISCOVERABLE_DEVICE_PROTOCOL_GUID",
        description = "",
        aliases = ['EdkiiNonDiscoverableDeviceProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/NonDiscoverableDevice.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0d57b171-a6d2-49c9-95ad-abc02546e605": EfiGuid(
        "0d57b171-a6d2-49c9-95ad-abc02546e605", "DellSystemUsbPortConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d58ae36-31af-4cf3-a74d-74891a1597f2": EfiGuid(
        "0d58ae36-31af-4cf3-a74d-74891a1597f2", "AssetIDDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d648466-36bd-42c6-b287-7c3baa2575c0": EfiGuid(
        "0d648466-36bd-42c6-b287-7c3baa2575c0", "LENOVO_SETUP_UNDER_OS_DXE_GUID",
        description = "",
        aliases = ['LenovoSetupUnderOsDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d759f81-43b0-4e0a-b613-61550c6157c2": EfiGuid(
        "0d759f81-43b0-4e0a-b613-61550c6157c2", "SystemGraphicsOutputBltPipeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d79a645-1d91-40a6-a81f-61e6982b32b4": EfiGuid(
        "0d79a645-1d91-40a6-a81f-61e6982b32b4", "EfiNt32PkgTokenSpaceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d8039ff-49e9-4cc9-a806-bb7c31b0bcb0": EfiGuid(
        "0d8039ff-49e9-4cc9-a806-bb7c31b0bcb0", "AmiTpm20PlatformPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AmiTpm20PlatformPei'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '3555862f32138b911338fae3789c3921', '35a55f8ce610ca18a6d9480a0fc484df',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3fa6831c92f133013c5bb0942f307f72',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '6f7959935a87f450b05674b1e62ba7fa', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7d7236b9ad7c65c2e13957e6027586fa', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0d81fdc5-cb98-4b9f-b93b-70a9c0663abe": EfiGuid(
        "0d81fdc5-cb98-4b9f-b93b-70a9c0663abe", "DELL_DCCS_SMM_DRIVER_GUID",
        description = "",
        aliases = ['DellDccsSmmDriver'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d82a9ec-1289-4fd4-ac0b-4c6b1a25abc6": EfiGuid(
        "0d82a9ec-1289-4fd4-ac0b-4c6b1a25abc6", "SwitchableGraphicsDxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SwitchableGraphicsDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'cbd75246fd41b231527edfcacc8d5803',
            'cc80253042928592caf3fafdd1915db0', 'cfba6c85608b8135fb21024783573bf6',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "0d8e6e4e-b029-475f-9122-60a3fedba8c0": EfiGuid(
        "0d8e6e4e-b029-475f-9122-60a3fedba8c0", "DxeIoLibEsal",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d8fa117-1239-4cd6-ad0a-331663824b13": EfiGuid(
        "0d8fa117-1239-4cd6-ad0a-331663824b13", "DellOnboardNicPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d966d65-8f25-4574-8eaf-6c0463f38742": EfiGuid(
        "0d966d65-8f25-4574-8eaf-6c0463f38742", "UsbPortConfigSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d98d4c6-59b2-44cc-8e28-4540130e347d": EfiGuid(
        "0d98d4c6-59b2-44cc-8e28-4540130e347d", "LaunchPadDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d9a1427-e02a-437d-926b-aa521fd722ba": EfiGuid(
        "0d9a1427-e02a-437d-926b-aa521fd722ba", "EFI_PCI_LAN_INFO_GUID",
        description = "",
        aliases = ['EfiPciLanInfoGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0d9e8db6-5a5a-4a5b-8621-85ffc9ae4583": EfiGuid(
        "0d9e8db6-5a5a-4a5b-8621-85ffc9ae4583", "DellStealthModePolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0da55bc9-45f8-4bb4-8719-5224f18a4d45": EfiGuid(
        "0da55bc9-45f8-4bb4-8719-5224f18a4d45", "EFI_WIRELESS_MAC_CONNECTION_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiWiFiProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/WiFi.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiWiFiProtocol']),

    "0db48a36-4e54-ea9c-9b09-1ea5be3a660b": EfiGuid(
        "0db48a36-4e54-ea9c-9b09-1ea5be3a660b", "EFI_REST_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiRestProtocolGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Rest.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0dba19da-5197-4e71-a62f-4f8cb3cdb55f": EfiGuid(
        "0dba19da-5197-4e71-a62f-4f8cb3cdb55f", "IoTrapBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dbcec22-0abe-4bf4-bfee-a04d1ca6f643": EfiGuid(
        "0dbcec22-0abe-4bf4-bfee-a04d1ca6f643", "PlatformInitDxeHedt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dbf0b49-604c-40d4-9121-77ac41942626": EfiGuid(
        "0dbf0b49-604c-40d4-9121-77ac41942626", "FixedBootOrder",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dc02196-cec8-4a0e-9dcd-b0de59d86204": EfiGuid(
        "0dc02196-cec8-4a0e-9dcd-b0de59d86204", "Wcn685xLauncher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dc65adc-a973-4130-8df0-2adbeb9e4a31": EfiGuid(
        "0dc65adc-a973-4130-8df0-2adbeb9e4a31", "FIRMWARE_PERFORMANCE_S3_POINTER_GUID",
        description = "",
        aliases = ['FirmwarePerformanceS3PointerGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/FirmwarePerformance.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0dc73aed-cbf6-4a25-a68d-59c80f44c7c3": EfiGuid(
        "0dc73aed-cbf6-4a25-a68d-59c80f44c7c3", "EfiDFUVerGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dca3d4d-12da-4728-bf7e-86ceb928d067": EfiGuid(
        "0dca3d4d-12da-4728-bf7e-86ceb928d067", "EFI_NIC_IP4_CONFIG_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiNicIp4ConfigProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dca793a-ea96-42d8-bd7b-dc7f684e38c1": EfiGuid(
        "0dca793a-ea96-42d8-bd7b-dc7f684e38c1", "ROM_LAYOUT_FFS_GUID",
        description = "",
        aliases = ['LegacyRomLayout'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['ROM_LAYOUT_FFS_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['RomLayout']),

    "0dce384d-007c-4ba5-94bd-0f6eb64d2aa9": EfiGuid(
        "0dce384d-007c-4ba5-94bd-0f6eb64d2aa9", "PEI_NT_AUTOSCAN_PPI_GUID",
        description = "",
        aliases = ['PeiNtAutoScanPpiGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dcf3594-318c-4596-b00f-be61842de3e2": EfiGuid(
        "0dcf3594-318c-4596-b00f-be61842de3e2", "SystemBootTypePeiPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0dde9636-8321-4edf-9f14-0bfca3b473f5": EfiGuid(
        "0dde9636-8321-4edf-9f14-0bfca3b473f5", "DELL_INTRUSION_DETECT_SMM_GUID",
        description = "",
        aliases = ['DellIntrusionDetectSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ddf623a-9f60-4af2-b5c7-efe4a637290e": EfiGuid(
        "0ddf623a-9f60-4af2-b5c7-efe4a637290e", "FjDtDiagnosticsDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0de19141-7aeb-46ca-9f87-2d19febc99c3": EfiGuid(
        "0de19141-7aeb-46ca-9f87-2d19febc99c3", "OemBootMiscDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0de2ce25-446a-45a7-bfc9-37da26344b37": EfiGuid(
        "0de2ce25-446a-45a7-bfc9-37da26344b37", "EFI_PEI_DEVICE_RECOVERY_MODULE_PPI_GUID",
        description = "",
        aliases = ['EfiPeiDeviceRecoveryModulePpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/DeviceRecoveryModule.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0de8bacf-e00a-4538-be0d-81af9374fcc9": EfiGuid(
        "0de8bacf-e00a-4538-be0d-81af9374fcc9", "EMULATED_EEPROM_SMM_PROTOCOL_GUID",
        description = "",
        aliases = ['EmulatedEepromSmmProtocol'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ded86ee-6e79-4764-aa83-37a472f48123": EfiGuid(
        "0ded86ee-6e79-4764-aa83-37a472f48123", "ChkrecoveryFile",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e00b084-2d16-4a27-b172-b1f68c2cc55d": EfiGuid(
        "0e00b084-2d16-4a27-b172-b1f68c2cc55d", "MicrocodeUpdates",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e06a80e-d726-4baf-a0cf-211260fe69d8": EfiGuid(
        "0e06a80e-d726-4baf-a0cf-211260fe69d8", "OemSSIDUpdate",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e0885b3-07a2-41aa-82e7-806a47a9215e": EfiGuid(
        "0e0885b3-07a2-41aa-82e7-806a47a9215e", "Parade_Retimer_FmpDxe_FwUpd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e109c7b-8790-4b9d-981c-9d4b079e98d4": EfiGuid(
        "0e109c7b-8790-4b9d-981c-9d4b079e98d4", "MsiSGSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e135e4d-e63e-45c6-a9a2-e9363477ddfc": EfiGuid(
        "0e135e4d-e63e-45c6-a9a2-e9363477ddfc", "AmdNbioBaseMdnDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e1b2f3c-dd5f-44ce-8d37-5aec2b339b5c": EfiGuid(
        "0e1b2f3c-dd5f-44ce-8d37-5aec2b339b5c", "BiosConnectSOSLauncher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e1c2f09-a27d-47e8-b4a0-15b22959faa0": EfiGuid(
        "0e1c2f09-a27d-47e8-b4a0-15b22959faa0", "TIANO_COMPRESS_SMM_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e1d2972-65af-4ac1-bfa3-cef4ab0c38fe": EfiGuid(
        "0e1d2972-65af-4ac1-bfa3-cef4ab0c38fe", "EfiCapsuleCrashGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e205c8a-8586-4dec-9f5c-4f9e394aefe8": EfiGuid(
        "0e205c8a-8586-4dec-9f5c-4f9e394aefe8", "UefiShellCEntryLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ShellPkg/Library/UefiShellCEntryLib/UefiShellCEntryLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0e2d6bb1-c624-446d-9982-693cd181a607": EfiGuid(
        "0e2d6bb1-c624-446d-9982-693cd181a607", "EFI_SMM_TCO_DISPATCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmmTcoDispatchProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e2daf63-8a4f-4026-a899-de2d7f46e5ec": EfiGuid(
        "0e2daf63-8a4f-4026-a899-de2d7f46e5ec", "SgTpvPei",
        description = "",
        aliases = ['TpvPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SgTpvPei'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '0ebecbce655f6f780edb7d73c1d80d13',
            '0fc6e822ed386a76a1ac3b30c3afd4d4', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '2949649fbed4688a06e89bf439cea6a1',
            '3fa6831c92f133013c5bb0942f307f72', '4002d2fb07cfbaed2e86d9608824378a',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d13012b3ccc7c61a0c88ca8969676f6', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b49cea81c680d9848212ed2aca8e6771',
            'b930f337647cc93e2809f678c893ad69', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd4a0cd284c2f726df5db17c605809651', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0e2eda9f-4bb6-4140-aed6-7dbceb63c829": EfiGuid(
        "0e2eda9f-4bb6-4140-aed6-7dbceb63c829", "MtkWifiSupDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e359f52-c6de-4ea4-9a05-857de1b7d320": EfiGuid(
        "0e359f52-c6de-4ea4-9a05-857de1b7d320", "FjWifi6eSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e3606d2-1dc3-4e6f-be65-394982a26547": EfiGuid(
        "0e3606d2-1dc3-4e6f-be65-394982a26547", "gPlatformVirtualKeyboardProtocolGuid",
        description = "",
        aliases = ['PlatformVirtualKeyboardProtocol'],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0e3e9703-9fff-4e23-8b2f-e3a124d09e0e": EfiGuid(
        "0e3e9703-9fff-4e23-8b2f-e3a124d09e0e", "OCPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['OCPei'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '0ebecbce655f6f780edb7d73c1d80d13',
            '0fc6e822ed386a76a1ac3b30c3afd4d4', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '3fa6831c92f133013c5bb0942f307f72',
            '540ab45d79fb417ade67cb3aac048180', '54b7b82c79116e989585483d47eff1b4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7f662f93e41d560c5fb667dce9000eb7', '869f701de5296248126c8c8a03a479c3',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '953dc1131b7a579176b9fc937eee5846',
            '9e80820994976fe9edc2b53346ea4f71', 'ac2383f4616ea50d075c37b988c3cb83',
            'b49cea81c680d9848212ed2aca8e6771', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd4a0cd284c2f726df5db17c605809651', 'dbcaa4b015b1aec94558985a97dce7c2',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
        ],
        junk = []),

    "0e3fefde-4a6c-4e4f-b77f-4b456150430f": EfiGuid(
        "0e3fefde-4a6c-4e4f-b77f-4b456150430f", "PegaSetKeyboardSetting",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e3fefde-4a6c-4e4f-b77f-4b46c950430f": EfiGuid(
        "0e3fefde-4a6c-4e4f-b77f-4b46c950430f", "PegaSMBIOS",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e4d805d-746c-4ebc-8795-31a286cca620": EfiGuid(
        "0e4d805d-746c-4ebc-8795-31a286cca620", "TcgPeiPolicyHobGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e511f9d-5b86-4389-9325-9f8217220fbb": EfiGuid(
        "0e511f9d-5b86-4389-9325-9f8217220fbb", "CpuInstallEfiMemoryPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e51393a-e865-4c3e-8c5f-442fe09146a5": EfiGuid(
        "0e51393a-e865-4c3e-8c5f-442fe09146a5", "DellSmmVariableProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e5870e4-0525-40ad-95a8-0fff155b8fc0": EfiGuid(
        "0e5870e4-0525-40ad-95a8-0fff155b8fc0", "AmiSmbiosElogSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e714b8d-eeaa-4fba-83ab-2b8005d417c3": EfiGuid(
        "0e714b8d-eeaa-4fba-83ab-2b8005d417c3", "EnableM2PCIeCardDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e7383b1-83ee-41a4-939e-24c886f03ad6": EfiGuid(
        "0e7383b1-83ee-41a4-939e-24c886f03ad6", "DELL_VIRT_RTC_SMM_GUID",
        description = "",
        aliases = ['DellVirtRtcSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e78d1d6-5691-4793-8ff7-465cb7f9124e": EfiGuid(
        "0e78d1d6-5691-4793-8ff7-465cb7f9124e", "DellNvmePwDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e84fc69-29cc-4c6d-92ac-6d476921850f": EfiGuid(
        "0e84fc69-29cc-4c6d-92ac-6d476921850f", "UpdateDriverDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0e8c545b-a2ee-470d-8e26-bda1a13c0aa3": EfiGuid(
        "0e8c545b-a2ee-470d-8e26-bda1a13c0aa3", "LAST_ENUM_LANGUAGE_GUID",
        description = "",
        aliases = ['LastEnumLangGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['LastEnumLang', 'LastEnumLangGuid']),

    "0e923c25-13cd-430b-8714-ffe85652a97b": EfiGuid(
        "0e923c25-13cd-430b-8714-ffe85652a97b", "VmgExitLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/VmgExitLib/VmgExitLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0e93c52b-4b73-5c32-86d5-69250a0ba723": EfiGuid(
        "0e93c52b-4b73-5c32-86d5-69250a0ba723", "AppleThemeFileNames",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ea3d651-49b4-47d9-bdc3-e4a379f3de76": EfiGuid(
        "0ea3d651-49b4-47d9-bdc3-e4a379f3de76", "DELL_SMM_HWM_DRIVER_GUID",
        description = "",
        aliases = ['HwmInitSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ea4c03f-d91b-4929-aaa5-b2fc8d69e2f4": EfiGuid(
        "0ea4c03f-d91b-4929-aaa5-b2fc8d69e2f4", "CapsuleUpdatePolicyLibOnProtocol",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/FmpDevicePkg/Library/CapsuleUpdatePolicyLibOnProtocol/CapsuleUpdatePolicyLibOnProtocol.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0ea4e9cb-819d-4b37-9362-ae1783481ac0": EfiGuid(
        "0ea4e9cb-819d-4b37-9362-ae1783481ac0", "FjDtPowerFailureRecoveryPlatformSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ea50c2b-4c0d-4ca3-97c1-0014db754473": EfiGuid(
        "0ea50c2b-4c0d-4ca3-97c1-0014db754473", "UefiLegacyBootMarker",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ea82aa2-6c36-4fd5-bc90-ffa3ecb5e0ce": EfiGuid(
        "0ea82aa2-6c36-4fd5-bc90-ffa3ecb5e0ce", "PciSegmentInfoLibAcpiBoardInfo",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/Library/PciSegmentInfoLibAcpiBoardInfo/PciSegmentInfoLibAcpiBoardInfo.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0eab05c1-766a-4805-a039-3081de0210c7": EfiGuid(
        "0eab05c1-766a-4805-a039-3081de0210c7", "FmpDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0eb4819c-d878-4c8d-913b-404947aeb89f": EfiGuid(
        "0eb4819c-d878-4c8d-913b-404947aeb89f", "PlatformStorageInformationDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0eb4fac0-a3ec-4fed-9783-a8d0ffd1ae33": EfiGuid(
        "0eb4fac0-a3ec-4fed-9783-a8d0ffd1ae33", "DELL_SM_BIOS_STRUC_DB_GUID",
        description = "",
        aliases = ['DellSmBiosStrucDb'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0eb84da1-267a-40b4-8347-1f48694c8b47": EfiGuid(
        "0eb84da1-267a-40b4-8347-1f48694c8b47", "PeCoffExtraActionLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BasePeCoffExtraActionLibNull/BasePeCoffExtraActionLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0eb9d3ed-c4d3-4d9a-b291-ac3bc9c3c8fb": EfiGuid(
        "0eb9d3ed-c4d3-4d9a-b291-ac3bc9c3c8fb", "SmmMeSpiRegionBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ec1c157-dd5e-426b-8764-68c1da088c49": EfiGuid(
        "0ec1c157-dd5e-426b-8764-68c1da088c49", "OSDSMIFunctionPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ec2019d-bc70-4a4e-a239-501ef5d6f742": EfiGuid(
        "0ec2019d-bc70-4a4e-a239-501ef5d6f742", "InstallPlatformSsdt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ecc666b-4662-47f9-9dd5-d096ff7da49e": EfiGuid(
        "0ecc666b-4662-47f9-9dd5-d096ff7da49e", "PEI_SEC_PERFORMANCE_PPI_GUID",
        description = "",
        aliases = ['PeiSecPerformancePpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Ppi/SecPerformance.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0ece530f-7bdd-4405-9eb9-61c9868d8abd": EfiGuid(
        "0ece530f-7bdd-4405-9eb9-61c9868d8abd", "AaeonSmbiosDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ecedd30-67ec-4570-9efb-308de53ee93d": EfiGuid(
        "0ecedd30-67ec-4570-9efb-308de53ee93d", "AmiTseOemPortingVar23",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0edc9494-2743-4ba5-8818-0aef5213f188": EfiGuid(
        "0edc9494-2743-4ba5-8818-0aef5213f188", "EFI_EXTENDED_SAL_CACHE_SERVICES_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiExtendedSalCacheServicesProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0edef0b3-ef91-4935-859e-2338dffac099": EfiGuid(
        "0edef0b3-ef91-4935-859e-2338dffac099", "MpdtUpdateDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ee0dbdb-55fd-45cc-a790-894bd347707a": EfiGuid(
        "0ee0dbdb-55fd-45cc-a790-894bd347707a", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "0eec96bc-0b82-4573-9791-c414e4dcee64": EfiGuid(
        "0eec96bc-0b82-4573-9791-c414e4dcee64", "IhisiService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0eecb4a5-7b9f-4707-81d7-9c045e98ca95": EfiGuid(
        "0eecb4a5-7b9f-4707-81d7-9c045e98ca95", "AmiPlatformInfoUncompressedFfsSection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ef3c867-459b-41d8-8541-dd7dc277ebc0": EfiGuid(
        "0ef3c867-459b-41d8-8541-dd7dc277ebc0", "PdHostInterfaceRtkPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ef53039-3a38-42d1-bcec-ce966e87061a": EfiGuid(
        "0ef53039-3a38-42d1-bcec-ce966e87061a", "PeiFrb",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ef84c20-d178-490b-9fcd-122739faaefc": EfiGuid(
        "0ef84c20-d178-490b-9fcd-122739faaefc", "FchSmmDiagDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ef8a3b1-388a-4b62-8be6-c7877d50aedf": EfiGuid(
        "0ef8a3b1-388a-4b62-8be6-c7877d50aedf", "UefiPxeBcDxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['UefiPxeBcDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7',
        ],
        junk = []),

    "0ef98d3a-3e33-497a-a401-77be3eb74f38": EfiGuid(
        "0ef98d3a-3e33-497a-a401-77be3eb74f38", "EFI_ACPI_S3_CONTEXT_GUID",
        description = "",
        aliases = ['EfiAcpiS3ContextGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/AcpiS3Context.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0efc6282-f1e5-469a-8a70-194a8761f9aa": EfiGuid(
        "0efc6282-f1e5-469a-8a70-194a8761f9aa", "XenAcpiPlatformDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/XenAcpiPlatformDxe/XenAcpiPlatformDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0f03d285-6614-4ddc-86dc-dae5e375443a": EfiGuid(
        "0f03d285-6614-4ddc-86dc-dae5e375443a", "EndOfFirstPowerOn",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f0b1735-87a0-4193-b266-538c38af48ce": EfiGuid(
        "0f0b1735-87a0-4193-b266-538c38af48ce", "EFI_IFR_TIANO_GUID",
        description = "",
        aliases = ['EfiIfrTianoGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Common/MdeModuleHii.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/MdeModuleHii.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0f0da838-e678-432e-9aee-cb606e89b3da": EfiGuid(
        "0f0da838-e678-432e-9aee-cb606e89b3da", "FchSmbusPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f17cecc-653a-c343-9cfa-faa27a07efe5": EfiGuid(
        "0f17cecc-653a-c343-9cfa-faa27a07efe5", "AppleCrypto",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f23c1f8-4bac-470c-b6b8-b392d544290a": EfiGuid(
        "0f23c1f8-4bac-470c-b6b8-b392d544290a", "FmpCapsuleUpdate",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['FmpCapsuleUpdate'],
        hashes = ['7d7236b9ad7c65c2e13957e6027586fa'],
        junk = []),

    "0f2a0816-d319-4ee7-a6b8-d58524e4428f": EfiGuid(
        "0f2a0816-d319-4ee7-a6b8-d58524e4428f", "NullDmaLib",
        description = "",
        aliases = ['CoherentDmaLib'],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Library/CoherentDmaLib/CoherentDmaLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0f2fadab-e401-63ed-7e29-69f215c60511": EfiGuid(
        "0f2fadab-e401-63ed-7e29-69f215c60511", "OemUSBGetSetupData",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f411be5-b10c-4dda-b28c-868fc24789f8": EfiGuid(
        "0f411be5-b10c-4dda-b28c-868fc24789f8", "AmdVariableProtection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f492340-21b3-4e23-9140-8c14476292cd": EfiGuid(
        "0f492340-21b3-4e23-9140-8c14476292cd", "SetupRepSetSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f500be6-ece4-4ed8-9081-9aa9a523fb7b": EfiGuid(
        "0f500be6-ece4-4ed8-9081-9aa9a523fb7b", "HSTI_PUBLISH_COMPLETE_PROTOCOL_GUID",
        description = "",
        aliases = ['HstiPublishCompleteProtocolGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f5dd251-6c8b-4752-8110-fcdf57eedfb2": EfiGuid(
        "0f5dd251-6c8b-4752-8110-fcdf57eedfb2", "DxeReportStatusConOut",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f5ef786-17a0-40c6-bc18-1b3272a00987": EfiGuid(
        "0f5ef786-17a0-40c6-bc18-1b3272a00987", "IccInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f647b12-5fa2-4b5e-9d61-2ab3c51f224e": EfiGuid(
        "0f647b12-5fa2-4b5e-9d61-2ab3c51f224e", "BctBaseSmmSSP",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f6499b1-e9ad-493d-b9c2-2f90815c6cbc": EfiGuid(
        "0f6499b1-e9ad-493d-b9c2-2f90815c6cbc", "EFI_PHYSICAL_PRESENCE_DATA_GUID",
        description = "",
        aliases = ['EfiPhysicalPresenceGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/Include/Guid/PhysicalPresenceData.h",
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/SecurityPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiPhysicalPresence', 'EfiPhysicalPresenceGuid']),

    "0f69f6d7-0e4b-43a6-bfc2-6871694369b0": EfiGuid(
        "0f69f6d7-0e4b-43a6-bfc2-6871694369b0", "WdtAppPei",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f72230b-8762-4546-9b67-ed8929857a66": EfiGuid(
        "0f72230b-8762-4546-9b67-ed8929857a66", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['136ef7c1715a5ab03a93c66f1f9447e1'],
        junk = []),

    "0f729f33-25c1-41a7-86b2-23a737a91823": EfiGuid(
        "0f729f33-25c1-41a7-86b2-23a737a91823", "IntelSnbGopVbt",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['IntelSnbGopVbt'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac6277674bc0b5eb2156e1391dfb7143', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cfba6c85608b8135fb21024783573bf6',
            'd293013d9f585157787fce8c228e8897', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe',
        ],
        junk = []),

    "0f7bde7c-aaf3-48d0-931f-475750dde209": EfiGuid(
        "0f7bde7c-aaf3-48d0-931f-475750dde209", "OemPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f7bde7c-aaf3-48d0-931f-475750dde210": EfiGuid(
        "0f7bde7c-aaf3-48d0-931f-475750dde210", "OemWwanPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f7ec77a-1ee1-400f-a99d-7cbd1feb181e": EfiGuid(
        "0f7ec77a-1ee1-400f-a99d-7cbd1feb181e", "PcatPciRootBridge",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f886e75-3029-4ecf-b694-22894d3f2143": EfiGuid(
        "0f886e75-3029-4ecf-b694-22894d3f2143", "ASROCK_PLEDDXE_GUID",
        description = "",
        aliases = ['PLEDDXE'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f8e91fd-9964-4a65-b052-c4639c5a9ef0": EfiGuid(
        "0f8e91fd-9964-4a65-b052-c4639c5a9ef0", "SioRfTestWmiSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f92508b-f17c-4efb-8722-bca5531f8b53": EfiGuid(
        "0f92508b-f17c-4efb-8722-bca5531f8b53", "CpuDeviceInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f99e33c-ca0c-4aa2-887d-b57ec9050278": EfiGuid(
        "0f99e33c-ca0c-4aa2-887d-b57ec9050278", "SaveMemoryConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0f9d89e8-9259-4f76-a5af-0c89e34023df": EfiGuid(
        "0f9d89e8-9259-4f76-a5af-0c89e34023df", "EFI_FIRMWARE_CONTENTS_SIGNED_GUID",
        description = "",
        aliases = ['EfiFirmwareContentsSignedGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/FirmwareContentsSigned.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0f9dd4b0-599e-4d66-8565-22fbea1e028c": EfiGuid(
        "0f9dd4b0-599e-4d66-8565-22fbea1e028c", "AmdFabricPhxPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fa00c23-caca-5515-6ad3-b7d87540dfa3": EfiGuid(
        "0fa00c23-caca-5515-6ad3-b7d87540dfa3", "BLSeedTest",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fa79b12-719a-41ec-8c57-7c24f4771162": EfiGuid(
        "0fa79b12-719a-41ec-8c57-7c24f4771162", "VerifyHpRomSignature",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0faaecb1-226e-4782-aace-7db9bcbf4daf": EfiGuid(
        "0faaecb1-226e-4782-aace-7db9bcbf4daf", "EFI_FTP4_SERVICE_BINDING_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiFtp4ServiceBindingProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Ftp4.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "0fad5644-7bdf-4a75-b568-287ae2ebd3a6": EfiGuid(
        "0fad5644-7bdf-4a75-b568-287ae2ebd3a6", "SmcSwSmiFlashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0faf9d33-e3b7-49de-9b04-3a8b1e732afd": EfiGuid(
        "0faf9d33-e3b7-49de-9b04-3a8b1e732afd", "Armani_BatteryInfoSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fb3bcf9-0a1d-4598-a521-e5c84e95d97a": EfiGuid(
        "0fb3bcf9-0a1d-4598-a521-e5c84e95d97a", "EfiTraceHubStatusCodeHandlePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fb7c852-adca-4853-8d0f-fba71b1ce11a": EfiGuid(
        "0fb7c852-adca-4853-8d0f-fba71b1ce11a", "EFI_FORM_BROWSER_COMPATIBILITY_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiFormBrowserCompatibilityProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['EFI_FORM_BROWSER_PROTOCOL_GUID']),

    "0fba43b8-acdb-4ee2-ab31-0fa4d5b6c3c5": EfiGuid(
        "0fba43b8-acdb-4ee2-ab31-0fa4d5b6c3c5", "FAST_BOOT_TSE_PROTOCOL_GUID",
        description = "",
        aliases = ['FastBootTseProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fbee984-33bf-4ac0-a871-b352ad59e337": EfiGuid(
        "0fbee984-33bf-4ac0-a871-b352ad59e337", "DisableLTEB",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fc38c56-500a-4654-89a7-f43c1a3843f5": EfiGuid(
        "0fc38c56-500a-4654-89a7-f43c1a3843f5", "AmiPlatformInfoFfsFile",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fc50878-1633-432a-bde4-841357fc15e9": EfiGuid(
        "0fc50878-1633-432a-bde4-841357fc15e9", "AmiScsiPassThruInitProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fc63daf-8483-4772-8e79-3d69d8477de4": EfiGuid(
        "0fc63daf-8483-4772-8e79-3d69d8477de4", "LINUX_FILESYSTEM_PARTITION",
        description = """
        """,
        aliases = [],
        urls = ['https://en.wikipedia.org/wiki/GUID_Partition_Table'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fc9013a-0568-4ba9-9b7e-c9c390a6609b": EfiGuid(
        "0fc9013a-0568-4ba9-9b7e-c9c390a6609b", "EFI_LEGACY_REGION_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiLegacyRegionProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_LEGACY_REGION_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "0fd41e0b-c4df-4b4b-863d-17c0dbd6ef56": EfiGuid(
        "0fd41e0b-c4df-4b4b-863d-17c0dbd6ef56", "SbSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fd57366-4708-3ee0-392b-14a873782a25": EfiGuid(
        "0fd57366-4708-3ee0-392b-14a873782a25", "PciePowerManagementSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fd60162-a235-47d3-bf7e-735e027d48b2": EfiGuid(
        "0fd60162-a235-47d3-bf7e-735e027d48b2", "AmdFabricRmbSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fd96974-23aa-4cdc-b9cb-98d17750322a": EfiGuid(
        "0fd96974-23aa-4cdc-b9cb-98d17750322a", "EFI_HII_STRING_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiHiiStringProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/HiiString.h",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_HII_STRING_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0fdb6f0b-5581-4999-af7d-a571e3131bd2": EfiGuid(
        "0fdb6f0b-5581-4999-af7d-a571e3131bd2", "FjGabiDeviceFirmwareApiSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fdb764b-e669-4c69-83ac-5edd99a2711e": EfiGuid(
        "0fdb764b-e669-4c69-83ac-5edd99a2711e", "ReadOnlyVariableOnReadOnlyVariable2Thunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fe159b5-076f-4c36-bf26-d724f2831252": EfiGuid(
        "0fe159b5-076f-4c36-bf26-d724f2831252", "LENOVO_BDS_CTRL_GUID",
        description = "",
        aliases = ['BdsCtrl'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0fe9da53-043d-4265-a94d-fd77fede2eb4": EfiGuid(
        "0fe9da53-043d-4265-a94d-fd77fede2eb4", "TcgPlatformSetupPeiPolicy",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['TcgPlatformSetupPeiPolicy'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0ebecbce655f6f780edb7d73c1d80d13',
            '0fc6e822ed386a76a1ac3b30c3afd4d4', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '3555862f32138b911338fae3789c3921', '35a55f8ce610ca18a6d9480a0fc484df',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3fa6831c92f133013c5bb0942f307f72',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '6f7959935a87f450b05674b1e62ba7fa', '7457f65a90b756398ae360b93c326a5e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7d7236b9ad7c65c2e13957e6027586fa', '7dffb9d276837abbf9e319d968a966a1',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9e80820994976fe9edc2b53346ea4f71', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "0febe434-a6af-4166-bc2f-de2c5952c87d": EfiGuid(
        "0febe434-a6af-4166-bc2f-de2c5952c87d", "DellAbsoluteDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ff2ddd0-efc9-4f49-997a-cb5944e697d3": EfiGuid(
        "0ff2ddd0-efc9-4f49-997a-cb5944e697d3", "OPAL_PASSWORD_NOTIFY_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "0ffbce19-324c-4690-a009-98c6ae2eb186": EfiGuid(
        "0ffbce19-324c-4690-a009-98c6ae2eb186", "PEI_CAPSULE_ON_FAT_USB_DISK_GUID",
        description = "",
        aliases = ['RECOVERY_ON_FAT_USB_DISK_GUID', 'RecoveryOnFatUsbDiskGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/RecoveryDevice.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['RecoveryOnFatUsbDisk', 'RecoveryOnFatUsbDiskGuid', 'RECOVERY_ON_FAT_USB_DISK_GUID']),

    "0ffc90bd-b0b8-4538-9d8a-23ab12345678": EfiGuid(
        "0ffc90bd-b0b8-4538-9d8a-23ab12345678", "SctPdmCustomFileDialogDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10064aec-4e12-440d-8126-d71d07837423": EfiGuid(
        "10064aec-4e12-440d-8126-d71d07837423", "PeiCallback",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1009a840-c625-42b3-a94b-b90f991b1b1a": EfiGuid(
        "1009a840-c625-42b3-a94b-b90f991b1b1a", "NhltInstallTable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "100c2cfa-b586-4198-9b4c-1683d195b1da": EfiGuid(
        "100c2cfa-b586-4198-9b4c-1683d195b1da", "MMC",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['MMCHS']),

    "100cb8aa-7bfc-4d2b-90be-0a48c9ed5316": EfiGuid(
        "100cb8aa-7bfc-4d2b-90be-0a48c9ed5316", "BoardMilestoneHookDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1011b5d2-261a-424f-977c-c8a4b39353b8": EfiGuid(
        "1011b5d2-261a-424f-977c-c8a4b39353b8", "StaticSkuDataDxeArcherCityModular",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10149626-5583-4364-822f-a459b8121302": EfiGuid(
        "10149626-5583-4364-822f-a459b8121302", "LENOVO_SYSTEM_FIRMWARE_DEVICE_DXE_GUID",
        description = "",
        aliases = ['SystemFirmwareDeviceDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1015ea63-7421-417d-bb51-e5193061c551": EfiGuid(
        "1015ea63-7421-417d-bb51-e5193061c551", "DxeIpmiUsb",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10164673-d365-4be2-8513-1497cc07611d": EfiGuid(
        "10164673-d365-4be2-8513-1497cc07611d", "ScPolicyPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1019f54a-2560-41b2-87b0-6750b98f3eff": EfiGuid(
        "1019f54a-2560-41b2-87b0-6750b98f3eff", "CompatImageLoaderDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/CompatImageLoaderDxe/CompatImageLoaderDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "101df54a-10ff-4e0d-94c0-a89d5546dd6c": EfiGuid(
        "101df54a-10ff-4e0d-94c0-a89d5546dd6c", "AmdFabricStxhPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "102287b4-6b12-4d41-91e1-ebee1f3aa614": EfiGuid(
        "102287b4-6b12-4d41-91e1-ebee1f3aa614", "UefiDebugLibDebugPortProtocol",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/UefiDebugLibDebugPortProtocol/UefiDebugLibDebugPortProtocol.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "102491f2-8866-44ba-ae11-8af20feb97ed": EfiGuid(
        "102491f2-8866-44ba-ae11-8af20feb97ed", "OemChipsetDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['OemChipsetDxe'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '0fb0d5db7a0d624b39b616db11024c0a',
            '136ef7c1715a5ab03a93c66f1f9447e1', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '3dbf73930867371066435e0908e50535',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7f662f93e41d560c5fb667dce9000eb7', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '953dc1131b7a579176b9fc937eee5846', '9e80820994976fe9edc2b53346ea4f71',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd4a0cd284c2f726df5db17c605809651', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "1034183f-af0a-4716-be0e-4cb5319d6193": EfiGuid(
        "1034183f-af0a-4716-be0e-4cb5319d6193", "PciHotPlugNonRpDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "103d3c97-ce0f-4df0-97b0-194c61bd540c": EfiGuid(
        "103d3c97-ce0f-4df0-97b0-194c61bd540c", "FchEspiCmdDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1051ec65-f6fc-41ea-ba51-344c37f60cdf": EfiGuid(
        "1051ec65-f6fc-41ea-ba51-344c37f60cdf", "BoardConfigInitPreMem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['BoardConfigInitPreMem'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1054b2fa-3df0-41fc-a7ce-d59379f3988e": EfiGuid(
        "1054b2fa-3df0-41fc-a7ce-d59379f3988e", "AsusSIBoardPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "105884e1-57da-4f36-9104-8d092339d234": EfiGuid(
        "105884e1-57da-4f36-9104-8d092339d234", "DellThermInfoConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "105ff0ea-a0c0-48a8-b8f7-e8b4d62a1019": EfiGuid(
        "105ff0ea-a0c0-48a8-b8f7-e8b4d62a1019", "FmpAuthenticationLibRsa2048Sha256",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/FmpAuthenticationLibRsa2048Sha256/FmpAuthenticationLibRsa2048Sha256.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1062cb92-72a9-4fdc-9694-b1fe817efaaf": EfiGuid(
        "1062cb92-72a9-4fdc-9694-b1fe817efaaf", "DustFilterAlertDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10654683-e6cb-49f6-99ed-1b23a2cefc09": EfiGuid(
        "10654683-e6cb-49f6-99ed-1b23a2cefc09", "MeSpiRegionBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1065e739-23c2-493f-b1ad-dc6d8df77cf4": EfiGuid(
        "1065e739-23c2-493f-b1ad-dc6d8df77cf4", "HpGenSwSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1068e0ed-5c8e-4724-b011-2c5f95065df2": EfiGuid(
        "1068e0ed-5c8e-4724-b011-2c5f95065df2", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0b8de120c00ed155bbf748f6e8438a2f'],
        junk = []),

    "106a2fd5-11ff-42ea-afba-b4969a702616": EfiGuid(
        "106a2fd5-11ff-42ea-afba-b4969a702616", "BaseOobLibrary",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "106c877f-c2ba-4c46-876c-bdfe6171cd7e": EfiGuid(
        "106c877f-c2ba-4c46-876c-bdfe6171cd7e", "DebugCommunicationLibUsb3Pei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SourceLevelDebugPkg/Library/DebugCommunicationLibUsb3/DebugCommunicationLibUsb3Pei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "106f3545-b788-4cb5-9d2a-ce0cdb208df5": EfiGuid(
        "106f3545-b788-4cb5-9d2a-ce0cdb208df5", "EfiHiiThunkProducer",
        description = "",
        aliases = ['EfiHiiThunkProducerGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "107a772b-d5e1-11d4-9a46-0090273fc14d": EfiGuid(
        "107a772b-d5e1-11d4-9a46-0090273fc14d", "EFI_DRIVER_CONFIGURATION_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDriverConfigurationProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DriverConfiguration.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "107a772c-d5e1-11d4-9a46-0090273fc14d": EfiGuid(
        "107a772c-d5e1-11d4-9a46-0090273fc14d", "EFI_COMPONENT_NAME_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiComponentNameProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/ComponentName.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "10811f41-037e-4ebe-9872-771d7646fffa": EfiGuid(
        "10811f41-037e-4ebe-9872-771d7646fffa", "LEMSAPAMLock",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "108343b1-3f5b-4908-9525-e25e62d5729a": EfiGuid(
        "108343b1-3f5b-4908-9525-e25e62d5729a", "LegacyPciUcrDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10921d66-281d-496e-a3b4-a8d47fd6de9e": EfiGuid(
        "10921d66-281d-496e-a3b4-a8d47fd6de9e", "HpVpinSelectionSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1093a6f0-37ed-49ba-9dcc-4f6999807315": EfiGuid(
        "1093a6f0-37ed-49ba-9dcc-4f6999807315", "MmioSerialUart2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10952220-aa32-11df-a438-0002a5d5c51b": EfiGuid(
        "10952220-aa32-11df-a438-0002a5d5c51b", "PL35xSmc",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "109d5fc6-56e6-481a-88ef-0cb828fbe0f6": EfiGuid(
        "109d5fc6-56e6-481a-88ef-0cb828fbe0f6", "EdkiiSystemCapsuleLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SignedCapsulePkg/Library/EdkiiSystemCapsuleLib/EdkiiSystemCapsuleLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "10b12add-f5e2-cc78-5ca0-b77f76223acd": EfiGuid(
        "10b12add-f5e2-cc78-5ca0-b77f76223acd", "AmiCpuFeaturesDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AmiCpuFeaturesDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9e80820994976fe9edc2b53346ea4f71', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "10b1ae5f-f4d1-4c15-89d0-a8c26728f580": EfiGuid(
        "10b1ae5f-f4d1-4c15-89d0-a8c26728f580", "INSYDE_BDS_DXE_DISPATCH_START_NOTIFY_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10b7f979-741c-467e-8764-8dd50d8ad6c5": EfiGuid(
        "10b7f979-741c-467e-8764-8dd50d8ad6c5", "SdioInt13",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SdioInt13'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54b7b82c79116e989585483d47eff1b4',
            '60785be6bfb0d1daa94a8bb214b774d1', '6f1b81ae0adfe7f90c141e463e9e9344',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'b218fecd176e986c1491046e58c9e01b', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'cae37d13982bb26c35ffaada2841d171',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'eee15c9794a4040563b41004ebccc91a',
            'f8c34791a7a721ee330771cb43d68b91', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "10b91e23-11ba-4cb5-9012-af5016005dd3": EfiGuid(
        "10b91e23-11ba-4cb5-9012-af5016005dd3", "SelfHealingDetectCorruptionDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10ba6bbe-a97e-41c3-9a07-607ad9bd60e5": EfiGuid(
        "10ba6bbe-a97e-41c3-9a07-607ad9bd60e5", "EfiVlv2VariableGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10c22623-db6f-4721-aa30-4c12af4230a7": EfiGuid(
        "10c22623-db6f-4721-aa30-4c12af4230a7", "IdeRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['IdeRecovery'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3555862f32138b911338fae3789c3921',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '60785be6bfb0d1daa94a8bb214b774d1', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9e80820994976fe9edc2b53346ea4f71', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'b218fecd176e986c1491046e58c9e01b', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "10c22623-db6f-4721-aa30-9c12af4230f8": EfiGuid(
        "10c22623-db6f-4721-aa30-9c12af4230f8", "ExtRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10d72ad4-f4d5-4f35-a5a3-001f6ed81e1e": EfiGuid(
        "10d72ad4-f4d5-4f35-a5a3-001f6ed81e1e", "DxeSerialTextOut",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10d93a56-2d64-47d3-828d-7a5066915292": EfiGuid(
        "10d93a56-2d64-47d3-828d-7a5066915292", "AmdRasBrApeiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10db0a54-f6f9-4ca2-a75e-f9aacae70970": EfiGuid(
        "10db0a54-f6f9-4ca2-a75e-f9aacae70970", "BiosGuardConfigGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10e26bc1-226a-4cd7-ae0a-95f56009c919": EfiGuid(
        "10e26bc1-226a-4cd7-ae0a-95f56009c919", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "10e26df1-8775-4ee1-b50a-3ae82893703a": EfiGuid(
        "10e26df1-8775-4ee1-b50a-3ae82893703a", "SeCfTPMPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10e88749-28d1-49a0-ac9e-0324e40211a8": EfiGuid(
        "10e88749-28d1-49a0-ac9e-0324e40211a8", "WMISwSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10ee5462-b207-4a4f-abd8-cb522ecaa3a4": EfiGuid(
        "10ee5462-b207-4a4f-abd8-cb522ecaa3a4", "Udp4Dxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['Udp4Dxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7',
        ],
        junk = []),

    "10ee54ae-b207-4a4f-abd8-cb522ecaa3a4": EfiGuid(
        "10ee54ae-b207-4a4f-abd8-cb522ecaa3a4", "Udp6Dxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['Udp6Dxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7',
        ],
        junk = []),

    "10f312d4-4a25-4d75-a1f3-1a264adcdbcf": EfiGuid(
        "10f312d4-4a25-4d75-a1f3-1a264adcdbcf", "DellSoftTAALoadDefault",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10f432de-deec-4631-80cd-47f65d8f80bb": EfiGuid(
        "10f432de-deec-4631-80cd-47f65d8f80bb", "EFI_PEI_PERFORMANCE_HOB_GUID",
        description = "",
        aliases = ['EfiPeiPerformanceHob'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10fb6e0f-537f-41d2-9214-7d00eebfd6a9": EfiGuid(
        "10fb6e0f-537f-41d2-9214-7d00eebfd6a9", "LnvSyncMFGDone2BiosGuard",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "10fe7e3b-dbe5-4cfa-9025-4002cfddbb89": EfiGuid(
        "10fe7e3b-dbe5-4cfa-9025-4002cfddbb89", "PCH_EMMC_TUNING_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11052d8b-c1ac-4244-a564-b5a71ff7822a": EfiGuid(
        "11052d8b-c1ac-4244-a564-b5a71ff7822a", "BiosGuardStoreStatusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11148d39-5926-4022-91ac-cb252af74530": EfiGuid(
        "11148d39-5926-4022-91ac-cb252af74530", "Usb_Lan_RT8152B",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1118396e-54cc-4389-bfd1-97d2ae13e12c": EfiGuid(
        "1118396e-54cc-4389-bfd1-97d2ae13e12c", "FjGabiCoreSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1126215d-b99c-4b09-9ca8-60c611ace29d": EfiGuid(
        "1126215d-b99c-4b09-9ca8-60c611ace29d", "OemEcFeature",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11335c60-2618-4af4-af9f-15bd35d0577b": EfiGuid(
        "11335c60-2618-4af4-af9f-15bd35d0577b", "FjLidSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11354a0c-781e-44a1-a787-c0178c8d570f": EfiGuid(
        "11354a0c-781e-44a1-a787-c0178c8d570f", "P2sbConfigGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1136d4dd-d9b7-4801-8352-e7582d32a05a": EfiGuid(
        "1136d4dd-d9b7-4801-8352-e7582d32a05a", "CISDIntelDRS",
        description = "",
        aliases = ['IntelDRS'],
        urls = [],
        vendors = [],
        filenames = ['CISDIntelDRS'],
        hashes = [
            '15c3afd565e35a7b87854f203ea15633', '82060a1dae992f24609d2f673cb58cfb',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'cae37d13982bb26c35ffaada2841d171', 'dab68811108b555173f8f096c9737d20',
        ],
        junk = []),

    "1137c217-b5bc-4e9a-b328-1e7bcd530520": EfiGuid(
        "1137c217-b5bc-4e9a-b328-1e7bcd530520", "DellThermalDebugSmmDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11399a01-0423-49cc-8368-85291533c35d": EfiGuid(
        "11399a01-0423-49cc-8368-85291533c35d", "FprSynapticsMetallicaDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "113b2126-fc8a-11e3-bd6c-b8e8562cbafa": EfiGuid(
        "113b2126-fc8a-11e3-bd6c-b8e8562cbafa", "EFI_BOOT_MANAGER_POLICY_CONNECT_ALL_GUID",
        description = "",
        aliases = ['EfiBootManagerPolicyConnectAllGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/BootManagerPolicy.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "113fd31a-be8b-418b-b034-7eafe5c60c99": EfiGuid(
        "113fd31a-be8b-418b-b034-7eafe5c60c99", "EsaInterfacesForTseProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1144265e-f049-45b9-8778-752f12a1365a": EfiGuid(
        "1144265e-f049-45b9-8778-752f12a1365a", "IntelTechSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11472071-a5ca-4281-9779-df2c839a455a": EfiGuid(
        "11472071-a5ca-4281-9779-df2c839a455a", "DxeSndwBeep",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "114b7105-6cc9-453c-badc-16df227bb4ef": EfiGuid(
        "114b7105-6cc9-453c-badc-16df227bb4ef", "TrEESmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "114ba15a-6189-180d-bfc0-2053b3480949": EfiGuid(
        "114ba15a-6189-180d-bfc0-2053b3480949", "EfiBoardSelectGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "114ca60c-d965-4c13-bef7-c4062248e1fa": EfiGuid(
        "114ca60c-d965-4c13-bef7-c4062248e1fa", "LENOVO_EC_IO_DXE_GUID",
        description = "",
        aliases = ['EcIoDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "114da5ef-2cf1-4e12-9bbb-c470b55205d9": EfiGuid(
        "114da5ef-2cf1-4e12-9bbb-c470b55205d9", "EFI_ADAPTER_INFO_SAN_MAC_ADDRESS_GUID",
        description = "",
        aliases = ['EfiAdapterInfoSanMacAddressGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/AdapterInformation.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1150e536-25e0-47ea-a54f-a4fc4af34e3c": EfiGuid(
        "1150e536-25e0-47ea-a54f-a4fc4af34e3c", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "11527125-78b2-4d3e-a0df-41e75c221f5a": EfiGuid(
        "11527125-78b2-4d3e-a0df-41e75c221f5a", "CpuS3",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1156efc6-ea32-4396-b5d5-26932e83c313": EfiGuid(
        "1156efc6-ea32-4396-b5d5-26932e83c313", "EFI_SPI_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSpiProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_SPI_PROTOCOL_GUID'],
        hashes = ['a9f4f993c8cee3cd83fecdb334fdf666', 'c0bdac2284b1e46c7c9c1509ddbea006'],
        junk = []),

    "115f8f4f-7899-4154-9c1d-b6e12b320be1": EfiGuid(
        "115f8f4f-7899-4154-9c1d-b6e12b320be1", "EcCommunicationSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11650c26-915a-40ac-829c-a1e1b6333d79": EfiGuid(
        "11650c26-915a-40ac-829c-a1e1b6333d79", "PwrButtonCtrlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11668fca-73ba-4b58-85a2-98ac4df59a2c": EfiGuid(
        "11668fca-73ba-4b58-85a2-98ac4df59a2c", "SystemSmmCommunicationBufferManagerDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11681f03-6b8b-4cad-83e7-588a64bc64ea": EfiGuid(
        "11681f03-6b8b-4cad-83e7-588a64bc64ea", "FjPowerOverEthernetD2927",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "116e1acf-2533-4cc2-820a-bbc10a2ab07c": EfiGuid(
        "116e1acf-2533-4cc2-820a-bbc10a2ab07c", "CpuSpSmi",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CpuSpSMI', 'CpuSpSmi'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "116ed1e8-f9c6-4112-a49c-87ada570dec1": EfiGuid(
        "116ed1e8-f9c6-4112-a49c-87ada570dec1", "AudioPlayback",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AudioPlayback'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '13bf9b32863d8378b9c802c9508c96e9', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '28e65e1a6689d8a2a4a612a06cbd2a58', '2949649fbed4688a06e89bf439cea6a1',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '633e80e7347dc01c21c5133227df2aba', '6f1b81ae0adfe7f90c141e463e9e9344',
            '7a20bc8c8925db2963205a36be03b335', '7dffb9d276837abbf9e319d968a966a1',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cbd75246fd41b231527edfcacc8d5803',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f3a7c093df5e8da1491e4b41973919a7',
            'f8c34791a7a721ee330771cb43d68b91', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "11726d68-f147-48df-ab16-0489818bd570": EfiGuid(
        "11726d68-f147-48df-ab16-0489818bd570", "AmiHpetTimer",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "117342b2-9824-4170-8afb-e568e62c0694": EfiGuid(
        "117342b2-9824-4170-8afb-e568e62c0694", "AmdSsdtPpcvOverride",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11777581-2b67-4075-8eb4-f691a47ecec7": EfiGuid(
        "11777581-2b67-4075-8eb4-f691a47ecec7", "ProcMemInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1177b5a8-7ba6-4c20-a3b4-7d6519d8670a": EfiGuid(
        "1177b5a8-7ba6-4c20-a3b4-7d6519d8670a", "ECHeartbeatPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11793215-cd3a-422a-a229-249417282dfa": EfiGuid(
        "11793215-cd3a-422a-a229-249417282dfa", "FchDxeMmio",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11799aa2-1c5d-4e5b-8a5d-32d20c39b9d3": EfiGuid(
        "11799aa2-1c5d-4e5b-8a5d-32d20c39b9d3", "AfuCapsuleOnDiskPostFlag",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1181e16d-af11-4c52-847e-516dd09bd376": EfiGuid(
        "1181e16d-af11-4c52-847e-516dd09bd376", "DELL_CENTURY_ROLLOVER_SMM_GUID",
        description = "",
        aliases = ['DellCenturyRolloverSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11822c42-79e4-48c9-9f73-700d3556b0c0": EfiGuid(
        "11822c42-79e4-48c9-9f73-700d3556b0c0", "MyAsusAutoInstallItem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1183fb2d-dfbd-40cd-945b-3fa630879bcd": EfiGuid(
        "1183fb2d-dfbd-40cd-945b-3fa630879bcd", "FjEvteServicesDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1188f1fc-06e9-49b8-a615-f5a0886fcf89": EfiGuid(
        "1188f1fc-06e9-49b8-a615-f5a0886fcf89", "UhciInitPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "118c6187-b0d3-4fd4-8b21-a4ae732416ab": EfiGuid(
        "118c6187-b0d3-4fd4-8b21-a4ae732416ab", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "118fe494-0699-42aa-aaca-62e849359b17": EfiGuid(
        "118fe494-0699-42aa-aaca-62e849359b17", "FspRsc2PostCodeMapServicePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1191bbf5-dcb9-44f4-827e-95359744c987": EfiGuid(
        "1191bbf5-dcb9-44f4-827e-95359744c987", "PlatformStage2Pei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "119bfa16-911b-4f1f-b1b2-69f43b759448": EfiGuid(
        "119bfa16-911b-4f1f-b1b2-69f43b759448", "DellPcdUpdatePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "119f3764-a7c2-4329-b25c-e6305e743049": EfiGuid(
        "119f3764-a7c2-4329-b25c-e6305e743049", "DELL_SMM_SECURITY_VAULT_GUID",
        description = "",
        aliases = ['DellSmmSecurityVault'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11a03a8e-a9c5-4db9-90cf-4434b01586c4": EfiGuid(
        "11a03a8e-a9c5-4db9-90cf-4434b01586c4", "BootPerformanceLog",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11a51daf-6834-4ba1-ad5e-4a58633b343e": EfiGuid(
        "11a51daf-6834-4ba1-ad5e-4a58633b343e", "AmdPspDxeV2Rmb",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11a687c0-edbf-4acf-8fd5-32a0bd5d76aa": EfiGuid(
        "11a687c0-edbf-4acf-8fd5-32a0bd5d76aa", "RZ6xxLauncher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11a6edf6-a9be-426d-a6cc-b22fe51d9224": EfiGuid(
        "11a6edf6-a9be-426d-a6cc-b22fe51d9224", "PciHotPlugInitDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/PciHotPlugInitDxe/PciHotPlugInit.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "11ade65f-956b-4d0e-b368-86dbc92f01aa": EfiGuid(
        "11ade65f-956b-4d0e-b368-86dbc92f01aa", "VariableSmiInt15Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11af0d83-0784-4c1e-9a1b-9cf43127555e": EfiGuid(
        "11af0d83-0784-4c1e-9a1b-9cf43127555e", "CrbModernStandby",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11b1e470-255e-40e3-9517-30513e3f14d3": EfiGuid(
        "11b1e470-255e-40e3-9517-30513e3f14d3", "GsensorExtendModeDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11b34006-d85b-4d0a-a290-d5a571310ef7": EfiGuid(
        "11b34006-d85b-4d0a-a290-d5a571310ef7", "PCD_PROTOCOL_GUID",
        description = "",
        aliases = ['PcdProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Pcd.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "11bfbeb0-fe77-4362-84fb-4f961e7bb2ed": EfiGuid(
        "11bfbeb0-fe77-4362-84fb-4f961e7bb2ed", "H2OFormDialogDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11c4d07d-b44f-4630-bcd8-eff786e24a74": EfiGuid(
        "11c4d07d-b44f-4630-bcd8-eff786e24a74", "CypressCCGxSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11caa2f5-4552-4ac3-95d0-8b5db1ff6d4f": EfiGuid(
        "11caa2f5-4552-4ac3-95d0-8b5db1ff6d4f", "OemDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11cdb71a-aa09-4b1b-bc58-f99838a4980d": EfiGuid(
        "11cdb71a-aa09-4b1b-bc58-f99838a4980d", "fTPMTisPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11d1ec21-e568-4eb0-8e1d-a0809772b606": EfiGuid(
        "11d1ec21-e568-4eb0-8e1d-a0809772b606", "DellEnhancedVersionProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11d8ac35-fb8a-44d1-8d09-0b5606d321b9": EfiGuid(
        "11d8ac35-fb8a-44d1-8d09-0b5606d321b9", "DSDT",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11d92dfb-3ca9-4f93-ba2e-4780ed3e03b5": EfiGuid(
        "11d92dfb-3ca9-4f93-ba2e-4780ed3e03b5", "VirtioBlkDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/VirtioBlkDxe/VirtioBlk.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "11dd0f6c-209b-47ce-8a98-418c9b7cc756": EfiGuid(
        "11dd0f6c-209b-47ce-8a98-418c9b7cc756", "DELL_DXE_SW_SMI_GUID",
        description = "",
        aliases = ['DxeSwSmi'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11e32c34-60b5-4991-8dea-63d3e8c876de": EfiGuid(
        "11e32c34-60b5-4991-8dea-63d3e8c876de", "UsbRndisDriverSrc",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11ee12f3-4b62-172f-37a6-a6bedab1ff62": EfiGuid(
        "11ee12f3-4b62-172f-37a6-a6bedab1ff62", "Pca9545aPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11f6c2ea-e21f-5acf-a5df-32c12e5e0e70": EfiGuid(
        "11f6c2ea-e21f-5acf-a5df-32c12e5e0e70", "PlatformPreInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "11fbfdfb-10d2-43e6-b5b1-b4386edccb9a": EfiGuid(
        "11fbfdfb-10d2-43e6-b5b1-b4386edccb9a", "EFI_ME_RC_INFO_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSeCRcInfoProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_ME_RC_INFO_PROTOCOL_GUID'],
        hashes = ['82060a1dae992f24609d2f673cb58cfb', 'b4fa3a413d748be6699112adc0c14b0d'],
        junk = ['EfiSeCRcInfoProtocol', 'EfiSeCRcInfoProtocolGuid']),

    "12025686-3984-466c-980b-8b5e89da0319": EfiGuid(
        "12025686-3984-466c-980b-8b5e89da0319", "AmiReadyToLoadDxeCorePpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12067096-18af-41dd-b4f1-d29592283062": EfiGuid(
        "12067096-18af-41dd-b4f1-d29592283062", "SystemDeviceModePrivateWmiSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1206f7ca-a475-4624-a83e-e6fc9bb38e49": EfiGuid(
        "1206f7ca-a475-4624-a83e-e6fc9bb38e49", "SmmControl2Dxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/SmmControl2Dxe/SmmControl2Dxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "120992f2-0339-402c-bab6-16122db2ec7e": EfiGuid(
        "120992f2-0339-402c-bab6-16122db2ec7e", "PeiMemoryDiagnosticTest",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "120d28aa-6630-46f0-8157-c0adc2383bf5": EfiGuid(
        "120d28aa-6630-46f0-8157-c0adc2383bf5", "EFI_AMI_LEGACYBOOT_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiLegacyBootProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1216bd2a-c05e-4c6d-91e7-eb8e37ad3b7a": EfiGuid(
        "1216bd2a-c05e-4c6d-91e7-eb8e37ad3b7a", "ASROCK_A_DEFAULT_DXE_GUID",
        description = "",
        aliases = ['aDefaultDXE'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12199730-7d13-4c92-97d5-1562515a1e48": EfiGuid(
        "12199730-7d13-4c92-97d5-1562515a1e48", "OemThermalPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "121d5264-990f-4716-8061-c2144a7d4d0d": EfiGuid(
        "121d5264-990f-4716-8061-c2144a7d4d0d", "DxeDgOpregionInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1224b1b9-cba1-41ca-82a7-dcf5ee6aebed": EfiGuid(
        "1224b1b9-cba1-41ca-82a7-dcf5ee6aebed", "SYSTEM_USB_DATABASE_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12270524-d586-42de-a1d0-d88007edafa9": EfiGuid(
        "12270524-d586-42de-a1d0-d88007edafa9", "DpsdSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['DpsdSetup'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "123099f5-3416-4ae8-aea0-2b7d1e7f8232": EfiGuid(
        "123099f5-3416-4ae8-aea0-2b7d1e7f8232", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['3555862f32138b911338fae3789c3921'],
        junk = []),

    "12345678-930a-4a95-ab04-2e6cfdff6631": EfiGuid(
        "12345678-930a-4a95-ab04-2e6cfdff6631", "TcgPeiAftermem",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['TcgPeiAftermem'],
        hashes = [
            '85e7ac10a2c876e04f9714b035fc8be0', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'cc80253042928592caf3fafdd1915db0',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "12345678-aabb-ccdd-eeff-112233445566": EfiGuid(
        "12345678-aabb-ccdd-eeff-112233445566", "MY_GUID",
        description = "",
        aliases = ['My'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "123ac7b6-f44d-4220-bd8b-521d30dea80f": EfiGuid(
        "123ac7b6-f44d-4220-bd8b-521d30dea80f", "M2CarrierSupportDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "123dd843-57c9-4158-8418-ce68b3944ce7": EfiGuid(
        "123dd843-57c9-4158-8418-ce68b3944ce7", "BaseCacheMaintenanceLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseCacheMaintenanceLib/BaseCacheMaintenanceLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "124a2e7a-1949-483e-899f-6032904ca0a7": EfiGuid(
        "124a2e7a-1949-483e-899f-6032904ca0a7", "LENOVO_SYSTEM_SMM_AHCI_ASPI_LEGACY_RT_GUID",
        description = "",
        aliases = ['SystemSmmAhciAspiLegacyRt'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "124ed7da-3dbe-49db-8aa8-ca584cc063e4": EfiGuid(
        "124ed7da-3dbe-49db-8aa8-ca584cc063e4", "DirtyShutdown",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1259f60d-b754-468e-a789-4db85d55e87e": EfiGuid(
        "1259f60d-b754-468e-a789-4db85d55e87e", "EFI_SWAP_ADDRESS_RANGE_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSwapAddressRangeProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/SwapAddressRange.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "125f2de1-fb85-440c-a54c-4d99358a8d38": EfiGuid(
        "125f2de1-fb85-440c-a54c-4d99358a8d38", "EFI_ACPI_S3_SAVE_GUID",
        description = "",
        aliases = ['EfiAcpiS3SaveProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['EfiAcpiS3SaveProtocol', 'EfiAcpiS3SaveProtocolGuid']),

    "126a762d-5758-4fca-8531-201a7f57f850": EfiGuid(
        "126a762d-5758-4fca-8531-201a7f57f850", "LENOVO_BOOT_MENU_DXE_GUID",
        description = "",
        aliases = ['LenovoSetupStartupDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "126f424e-f45b-4406-801e-2aacf404167f": EfiGuid(
        "126f424e-f45b-4406-801e-2aacf404167f", "AmiSetPhysicalPresenceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1279e288-24cd-47e9-96ba-d7a38c17bd64": EfiGuid(
        "1279e288-24cd-47e9-96ba-d7a38c17bd64", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "127c1c4e-9135-46e3-b006-f9808b0559a5": EfiGuid(
        "127c1c4e-9135-46e3-b006-f9808b0559a5", "Slp20Markers",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "127d145b-410d-4dd7-bfaf-15a3d3b60b94": EfiGuid(
        "127d145b-410d-4dd7-bfaf-15a3d3b60b94", "SetupRepSetDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "127d4e48-85c0-4fba-b678-b14e9492aadf": EfiGuid(
        "127d4e48-85c0-4fba-b678-b14e9492aadf", "FchKernPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12826089-d1ec-4505-b3ae-8e4cf3a4a78b": EfiGuid(
        "12826089-d1ec-4505-b3ae-8e4cf3a4a78b", "SetBoardId",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "128fb770-5e79-4176-9e51-9bb268a17dd1": EfiGuid(
        "128fb770-5e79-4176-9e51-9bb268a17dd1", "PciHostBridgeDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Pci/PciHostBridgeDxe/PciHostBridgeDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['PciHostBridgeDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '60785be6bfb0d1daa94a8bb214b774d1', '7a20bc8c8925db2963205a36be03b335',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'eee15c9794a4040563b41004ebccc91a',
        ],
        junk = []),

    "12900e5e-b88b-446c-ae19-18a0856b717f": EfiGuid(
        "12900e5e-b88b-446c-ae19-18a0856b717f", "SioSmbusAccessDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12963e55-5826-469e-a934-a3cbb3076ec5": EfiGuid(
        "12963e55-5826-469e-a934-a3cbb3076ec5", "DELL_SMM_SB_ACPI_GUID",
        description = "",
        aliases = ['DellSmmSbAcpi'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "129f6aa7-ab69-4cb8-aed1-40985001115e": EfiGuid(
        "129f6aa7-ab69-4cb8-aed1-40985001115e", "SerialMuxControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12a5dc4a-88aa-89da-89d0-d4547191e3f4": EfiGuid(
        "12a5dc4a-88aa-89da-89d0-d4547191e3f4", "RstUefiDriverSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['RstUefiDriverSupport'],
        hashes = [
            '002681fae113306a057042387f62f790', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db',
        ],
        junk = []),

    "12a5dc4b-88aa-89da-89d0-d4547191e3f4": EfiGuid(
        "12a5dc4b-88aa-89da-89d0-d4547191e3f4", "SdMmcOverride",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12a9420e-546a-484b-85e7-6f6b27bf9b40": EfiGuid(
        "12a9420e-546a-484b-85e7-6f6b27bf9b40", "FjCapsuleResetHookSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12a94570-a22f-4069-b52e-b71eaee4e06f": EfiGuid(
        "12a94570-a22f-4069-b52e-b71eaee4e06f", "FjOemGlobalNVSDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12aedbea-392d-4e2a-8789-5f6dc6b23661": EfiGuid(
        "12aedbea-392d-4e2a-8789-5f6dc6b23661", "OemBadgingSupportDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12b0aa24-ac37-43f2-a84e-3cca6437bf6d": EfiGuid(
        "12b0aa24-ac37-43f2-a84e-3cca6437bf6d", "AmdLegacyInterrupt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12b6c042-7c62-4666-a472-89ed5133a98c": EfiGuid(
        "12b6c042-7c62-4666-a472-89ed5133a98c", "FjIbvInterexchangeDataStorageProtocolSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12bfca88-7a2f-4ab5-9a5d-c40ca68bf75f": EfiGuid(
        "12bfca88-7a2f-4ab5-9a5d-c40ca68bf75f", "BootOrderSection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "12c293c3-3745-4726-8fa2-2a9ef18679ad": EfiGuid(
        "12c293c3-3745-4726-8fa2-2a9ef18679ad", "AaeonBfpiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12c67be1-ad2e-4f13-a95f-6edc2c4392de": EfiGuid(
        "12c67be1-ad2e-4f13-a95f-6edc2c4392de", "MEPeiPolicyInit",
        description = "",
        aliases = ['MePolicyInitPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['MEPeiPolicyInit', 'MePeiPolicyInit'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '3fa6831c92f133013c5bb0942f307f72', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "12cadd30-6007-4c83-89d7-ff237f7b947a": EfiGuid(
        "12cadd30-6007-4c83-89d7-ff237f7b947a", "SetupCompleteRedrawGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12cd8a4d-93fd-e81c-72fc-d931ebbc04d4": EfiGuid(
        "12cd8a4d-93fd-e81c-72fc-d931ebbc04d4", "DellTouchScreen",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12d1d3a2-99f7-420b-bc69-8bb1d492a355": EfiGuid(
        "12d1d3a2-99f7-420b-bc69-8bb1d492a355", "Logoid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12d43aae-ddc1-4615-b0bc-e115f9d17fc1": EfiGuid(
        "12d43aae-ddc1-4615-b0bc-e115f9d17fc1", "FjPowerButtonSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12d58591-e491-4e89-a081-3a3ce413181c": EfiGuid(
        "12d58591-e491-4e89-a081-3a3ce413181c", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "12e082f4-0dce-444b-b89a-ef3328428d64": EfiGuid(
        "12e082f4-0dce-444b-b89a-ef3328428d64", "ArmaniSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12e1529b-b6a0-4d7f-8b15-5036c9bc2de3": EfiGuid(
        "12e1529b-b6a0-4d7f-8b15-5036c9bc2de3", "Fji210LanControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12e2306c-1ebd-3140-b92e-efa9099e82d2": EfiGuid(
        "12e2306c-1ebd-3140-b92e-efa9099e82d2", "CacheManager",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12e5a97f-98d1-4c1f-87da-fb67cffbd9da": EfiGuid(
        "12e5a97f-98d1-4c1f-87da-fb67cffbd9da", "EfiLanDriverDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12e7691f-1eee-4ed4-8793-69e2801d602a": EfiGuid(
        "12e7691f-1eee-4ed4-8793-69e2801d602a", "LENOVO_FP_PBA_GUID",
        description = "",
        aliases = ['LenovoFpPba'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12e7691f-1eee-4ed4-8793-69e2801d6123": EfiGuid(
        "12e7691f-1eee-4ed4-8793-69e2801d6123", "AsixPxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12f38e73-b34d-4559-99e5-ae2dcd002156": EfiGuid(
        "12f38e73-b34d-4559-99e5-ae2dcd002156", "BaseFspWrapperPlatformLibSample",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/IntelFsp2WrapperPkg/Library/BaseFspWrapperPlatformLibSample/BaseFspWrapperPlatformLibSample.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "12f70db2-8585-49a6-a1e5-20f0363b05b0": EfiGuid(
        "12f70db2-8585-49a6-a1e5-20f0363b05b0", "WCN7850",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12f75401-5415-4ff3-a981-a39bee9f369e": EfiGuid(
        "12f75401-5415-4ff3-a981-a39bee9f369e", "SerialRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "12ff2998-429f-400b-b9ee-fea8287a1dfe": EfiGuid(
        "12ff2998-429f-400b-b9ee-fea8287a1dfe", "FjAbtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1305a712-33a6-4fa7-ba59-aeac3362931a": EfiGuid(
        "1305a712-33a6-4fa7-ba59-aeac3362931a", "PlatformFspLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "130b8ba5-e63e-44a0-85db-4d4e571c526a": EfiGuid(
        "130b8ba5-e63e-44a0-85db-4d4e571c526a", "IioCfgUpdateDxeNeonCityEPECB",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1310ba34-d7b4-4a80-a645-8c0417c6aa55": EfiGuid(
        "1310ba34-d7b4-4a80-a645-8c0417c6aa55", "AmiMemoryInfoConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AmiMemoryInfoConfig'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1310ba66-d7b4-77b0-a645-8c8887c6aa29": EfiGuid(
        "1310ba66-d7b4-77b0-a645-8c8887c6aa29", "AmiSvrMemoryInfoConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1314216c-cb8d-421c-b854-06231386e642": EfiGuid(
        "1314216c-cb8d-421c-b854-06231386e642", "PlatformInfo",
        description = "",
        aliases = ['PlatformInfoDxe'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['PlatformInfo', 'PlatformInfoDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1314b450-e856-438e-8131-cf823cab3550": EfiGuid(
        "1314b450-e856-438e-8131-cf823cab3550", "ExpansionSlotConfigPortingPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13173319-b270-4669-8592-3bb2b31e9e29": EfiGuid(
        "13173319-b270-4669-8592-3bb2b31e9e29", "ArmVirtPL031FdtClientLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/Library/ArmVirtPL031FdtClientLib/ArmVirtPL031FdtClientLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1317f0d5-7842-475c-b1ca-6edc20dcbe7d": EfiGuid(
        "1317f0d5-7842-475c-b1ca-6edc20dcbe7d", "HashLibTpm2",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/HashLibTpm2/HashLibTpm2.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1318fed0-2765-42b5-803e-3215a3c093c5": EfiGuid(
        "1318fed0-2765-42b5-803e-3215a3c093c5", "AmiTseOemPortingVar27",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "131aac21-2145-49e6-a3e1-a97bce090020": EfiGuid(
        "131aac21-2145-49e6-a3e1-a97bce090020", "AcpiSpcr",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13222252-931b-4552-9577-68556113afd0": EfiGuid(
        "13222252-931b-4552-9577-68556113afd0", "FjSmmPcieCardReaderRTS5250",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1323c7f8-dad5-4126-a54b-7a05fbf41515": EfiGuid(
        "1323c7f8-dad5-4126-a54b-7a05fbf41515", "DELL_SMM_ACCESS_GUID",
        description = "",
        aliases = ['SmmAccess'],
        urls = [],
        vendors = [],
        filenames = ['DELL_SMM_ACCESS_GUID', 'SmmAccess'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['SmmAccess']),

    "1323c999-dad5-5432-a54b-7a05fbf41515": EfiGuid(
        "1323c999-dad5-5432-a54b-7a05fbf41515", "PciHostBridgeWrapBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "132ec73b-3025-4fbf-b193-8acb50efbd89": EfiGuid(
        "132ec73b-3025-4fbf-b193-8acb50efbd89", "HkUpdate",
        description = "",
        aliases = ['QFlash'],
        urls = [],
        vendors = [],
        filenames = ['HkUpdate'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "132ec73b-bd26-4fbf-b193-8acb50efbd89": EfiGuid(
        "132ec73b-bd26-4fbf-b193-8acb50efbd89", "ASROCK_USER_DEF_SETUP_DXE_GUID",
        description = "",
        aliases = ['UserDefSetupDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "132fd006-2915-40ea-b779-0f0ddd01465f": EfiGuid(
        "132fd006-2915-40ea-b779-0f0ddd01465f", "PxeDriverRtk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "133ead80-fb48-11e5-939d-6431503b939a": EfiGuid(
        "133ead80-fb48-11e5-939d-6431503b939a", "TransparentUnlockDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "134bedba-7acf-4295-a4b4-66565e3e97d1": EfiGuid(
        "134bedba-7acf-4295-a4b4-66565e3e97d1", "OemAcpiNvs",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13524551-5600-43b4-a5df-f4b1942f339a": EfiGuid(
        "13524551-5600-43b4-a5df-f4b1942f339a", "MeResiliencyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1353de63-b74a-4bef-80fd-2c5cfa83040b": EfiGuid(
        "1353de63-b74a-4bef-80fd-2c5cfa83040b", "GifDecoderDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "135902e7-9709-4b41-8fd2-4069daf0546a": EfiGuid(
        "135902e7-9709-4b41-8fd2-4069daf0546a", "TcEfiGlobalVariableGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1361dec9-a270-4013-9701-a540f56c58c9": EfiGuid(
        "1361dec9-a270-4013-9701-a540f56c58c9", "MctpSmbusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "136a3048-752a-4bf6-a757-0936119538ed": EfiGuid(
        "136a3048-752a-4bf6-a757-0936119538ed", "UiStringPack",
        description = "",
        aliases = ['UiStringPackGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1374882c-b994-48db-8d1b-93e817541fbd": EfiGuid(
        "1374882c-b994-48db-8d1b-93e817541fbd", "CertificateStorageDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "137b3044-f6d7-473e-a625-9fb92505c180": EfiGuid(
        "137b3044-f6d7-473e-a625-9fb92505c180", "EfiSpiBusProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13863f79-d94b-4205-bb0f-e4e06aaa5a4e": EfiGuid(
        "13863f79-d94b-4205-bb0f-e4e06aaa5a4e", "DelayUefiRaidOprom",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1388066e-3a57-4efa-98f3-c12f3a958a29": EfiGuid(
        "1388066e-3a57-4efa-98f3-c12f3a958a29", "EFI_PEI_SECURITY_PPI_GUID",
        description = "",
        aliases = ['EfiPeiSecurityPpiGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['PEI_SECURITY_PPI_GUID']),

    "138f81c1-0a5f-4f5b-bff9-f9b5bcba6abf": EfiGuid(
        "138f81c1-0a5f-4f5b-bff9-f9b5bcba6abf", "HpSecureBootModeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "138f9cf4-f0e7-4721-8f49-f5ffecf42d40": EfiGuid(
        "138f9cf4-f0e7-4721-8f49-f5ffecf42d40", "EfiPeiMpInitLibMpDepPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1390954d-da95-4227-9328-7282c217daa8": EfiGuid(
        "1390954d-da95-4227-9328-7282c217daa8", "EFI_SMM_BASE_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmmBaseProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_SMM_BASE_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "1398661b-1eb6-417b-848e-ccc388526099": EfiGuid(
        "1398661b-1eb6-417b-848e-ccc388526099", "IntelUndiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "139aa2a0-311f-42cc-911a-579a2447b37d": EfiGuid(
        "139aa2a0-311f-42cc-911a-579a2447b37d", "XhciDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13a3f0f6-264a-3ef0-f2e0-dec512342f34": EfiGuid(
        "13a3f0f6-264a-3ef0-f2e0-dec512342f34", "EFI_PCD_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPcdProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/PiPcd.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PCD_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "13ac6dd0-73d0-11d4-b06b-00aa00bd6de7": EfiGuid(
        "13ac6dd0-73d0-11d4-b06b-00aa00bd6de7", "LENOVO_SYSTEM_EBC_DXE_GUID",
        description = "",
        aliases = ['EbcDxe'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/EbcDxe/EbcDxe.inf",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EbcDxe', 'LENOVO_SYSTEM_EBC_DXE_GUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EbcDxe']),

    "13ac6dd1-73d0-11d4-b06b-00aa00bd6de7": EfiGuid(
        "13ac6dd1-73d0-11d4-b06b-00aa00bd6de7", "EFI_EBC_INTERPRETER_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_EBC_PROTOCOL_GUID', 'EfiEbcProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Ebc.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiEbcProtocol', 'EfiEbcProtocolGuid']),

    "13b00aae-4fd2-964e-b30f-94627017370a": EfiGuid(
        "13b00aae-4fd2-964e-b30f-94627017370a", "BmpConvert",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13b03f00-18b8-48da-8b1c-b290c69baafe": EfiGuid(
        "13b03f00-18b8-48da-8b1c-b290c69baafe", "EzFileBrowser",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13b22491-b2f8-4d19-8769-641778f3ee28": EfiGuid(
        "13b22491-b2f8-4d19-8769-641778f3ee28", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "13bead28-488d-9112-5b0e-40b67edd800c": EfiGuid(
        "13bead28-488d-9112-5b0e-40b67edd800c", "AmdRasSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13c4bbe0-422c-4cac-804f-facdba6f1a28": EfiGuid(
        "13c4bbe0-422c-4cac-804f-facdba6f1a28", "SuperIoExPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13c5506e-8231-415e-9ebc-88dd115e3818": EfiGuid(
        "13c5506e-8231-415e-9ebc-88dd115e3818", "SmbiosElog",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13d3b72a-8047-47b8-a4c4-4e814f6b25a1": EfiGuid(
        "13d3b72a-8047-47b8-a4c4-4e814f6b25a1", "EfiFastBootDxe",
        description = "",
        aliases = ['DnxDxe'],
        urls = [],
        vendors = [],
        filenames = ['EfiFastBootDxe'],
        hashes = [
            '2b4ca0bd16a20893deff9632c23c2cca', '48bba53caf662f85dd73a1a247b45dc7',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'c0bdac2284b1e46c7c9c1509ddbea006',
        ],
        junk = []),

    "13e58e55-e1e1-4ca7-be88-2d8fe72664dc": EfiGuid(
        "13e58e55-e1e1-4ca7-be88-2d8fe72664dc", "DashManagementPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13e828e8-41a9-4d36-98de-fd3968e2c18e": EfiGuid(
        "13e828e8-41a9-4d36-98de-fd3968e2c18e", "LnvPdrSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13ec936c-b16c-4e6e-9587-fe3dd3f239b6": EfiGuid(
        "13ec936c-b16c-4e6e-9587-fe3dd3f239b6", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['1edacc01f9473c204c191bc39f10036f'],
        junk = []),

    "13ecd928-87ab-4460-bbe0-b520f9eb1d32": EfiGuid(
        "13ecd928-87ab-4460-bbe0-b520f9eb1d32", "IconNetBoot",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13f13249-ac31-4373-8b2b-afc5755a6fcd": EfiGuid(
        "13f13249-ac31-4373-8b2b-afc5755a6fcd", "BaseCacheMaintenanceLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseCacheMaintenanceLibNull/BaseCacheMaintenanceLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "13f37218-70df-45dd-8883-d4e6f5be4255": EfiGuid(
        "13f37218-70df-45dd-8883-d4e6f5be4255", "NvidiaGpuAcpitables",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13f40f6e-50c1-4b73-b1e2-6e72d21cb04a": EfiGuid(
        "13f40f6e-50c1-4b73-b1e2-6e72d21cb04a", "EfiUsbLegacyPlatformProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "13f4ea8e-bff1-43bf-8f44-80bcc96040f1": EfiGuid(
        "13f4ea8e-bff1-43bf-8f44-80bcc96040f1", "FpgaDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13f74ce7-cfed-4e97-9ee6-042b3d2d977f": EfiGuid(
        "13f74ce7-cfed-4e97-9ee6-042b3d2d977f", "FjCryptoDESSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "13fa7698-c831-49c7-87ea-8f43fcc25196": EfiGuid(
        "13fa7698-c831-49c7-87ea-8f43fcc25196", "EFI_EVENT_VIRTUAL_ADDRESS_CHANGE_GUID",
        description = "",
        aliases = ['EfiEventVirtualAddressChangeGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1400f9e6-4bab-4b68-a53f-58fd240818e6": EfiGuid(
        "1400f9e6-4bab-4b68-a53f-58fd240818e6", "Reset_Modify",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14045170-ca65-47bc-9c15-2de36d44aee9": EfiGuid(
        "14045170-ca65-47bc-9c15-2de36d44aee9", "SmcInBandDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1405ad5e-5bed-11e3-866a-047d7b99e097": EfiGuid(
        "1405ad5e-5bed-11e3-866a-047d7b99e097", "Slp20ServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1407443b-119b-40bc-9467-3c0a36740005": EfiGuid(
        "1407443b-119b-40bc-9467-3c0a36740005", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "140e8004-16e1-4de1-a352-c6ef51110ecf": EfiGuid(
        "140e8004-16e1-4de1-a352-c6ef51110ecf", "ArmSmcLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Library/ArmSmcLibNull/ArmSmcLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1410c6ac-9f4b-495b-9c23-8a5aeb0165e9": EfiGuid(
        "1410c6ac-9f4b-495b-9c23-8a5aeb0165e9", "SmmSwDispatch2OnSmmSwDispatchThunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SmmSwDispatch2OnSmmSwDispatchThunk'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
            'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "1410c6ad-9f4b-495b-9c23-8a5aeb0165e9": EfiGuid(
        "1410c6ad-9f4b-495b-9c23-8a5aeb0165e9", "SmmUsbDispatch2OnSmmUsbDispatchThunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SmmUsbDispatch2OnSmmUsbDispatchThunk'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
            'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "14186114-b990-4734-898e-5f86fae49784": EfiGuid(
        "14186114-b990-4734-898e-5f86fae49784", "DellHwmIoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "141b0f7f-e241-4659-91bf-4505d79a7714": EfiGuid(
        "141b0f7f-e241-4659-91bf-4505d79a7714", "FjCpuInfo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1421d662-67bf-4a23-8b32-607e1c9e3aee": EfiGuid(
        "1421d662-67bf-4a23-8b32-607e1c9e3aee", "MacAddressPassThroughDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "142204e2-c7b1-4af9-a729-923758d96d03": EfiGuid(
        "142204e2-c7b1-4af9-a729-923758d96d03", "LEGX16",
        description = "",
        aliases = ['AmiLegx16Bin'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['LEGX16'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f3a7c093df5e8da1491e4b41973919a7',
            'f8c34791a7a721ee330771cb43d68b91', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "14257b56-bda2-4faf-8e4f-c885df75583c": EfiGuid(
        "14257b56-bda2-4faf-8e4f-c885df75583c", "IccPlatform",
        description = "",
        aliases = ['IccPlatformDxe'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['IccPlatform', 'IccPlatformDxe'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '61205618135add3dd7e62e5bc90886a4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1428f772-b64a-441e-b8c3-9ebdd7f893c7": EfiGuid(
        "1428f772-b64a-441e-b8c3-9ebdd7f893c7", "QEMU_KERNEL_LOADER_FS_MEDIA_GUID",
        description = "",
        aliases = ['gQemuKernelLoaderFsMediaGuid', 'QemuKernelLoaderFsMedia'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Include/Guid/QemuKernelLoaderFsMedia.h",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/OvmfPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "143b5044-7c1b-4904-9778-ea16f1f3d554": EfiGuid(
        "143b5044-7c1b-4904-9778-ea16f1f3d554", "PlatformBdsLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "143b7632-b81b-4cb7-abd3-b625a5b9bffe": EfiGuid(
        "143b7632-b81b-4cb7-abd3-b625a5b9bffe", "EFI_EXT_SCSI_PASS_THRU_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiExtScsiPassThruProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/ScsiPassThruExt.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "143c715e-e903-4667-bf15-03f5a542f58e": EfiGuid(
        "143c715e-e903-4667-bf15-03f5a542f58e", "ClientronShowSystemInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1440f976-2332-4df3-b77f-62f8c6c1effa": EfiGuid(
        "1440f976-2332-4df3-b77f-62f8c6c1effa", "AmdNbioBaseRmbDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1448c340-0202-4711-9c4b-2d063aa6475b": EfiGuid(
        "1448c340-0202-4711-9c4b-2d063aa6475b", "ProgressBarEmptyLeftEndcap",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "144c3319-5f75-4a9f-a939-882a7ab29979": EfiGuid(
        "144c3319-5f75-4a9f-a939-882a7ab29979", "FjM2WlanEnableDisableDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1450464f-eb36-4ac6-aceb-daacc8fc3218": EfiGuid(
        "1450464f-eb36-4ac6-aceb-daacc8fc3218", "BoardSelectPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "145372bc-66b9-476d-81bc-2127c376bb66": EfiGuid(
        "145372bc-66b9-476d-81bc-2127c376bb66", "FFS.pad",
        description = "",
        aliases = ['FFS_pad'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14575e7c-03d7-4a62-9c01-4ebfbd48ba48": EfiGuid(
        "14575e7c-03d7-4a62-9c01-4ebfbd48ba48", "SmbiosTypeAdd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "145971e9-ad52-4094-a8c8-be5b3fecc82d": EfiGuid(
        "145971e9-ad52-4094-a8c8-be5b3fecc82d", "CpuPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "145e152b-3dde-4eb4-99a7-77083c2bcbec": EfiGuid(
        "145e152b-3dde-4eb4-99a7-77083c2bcbec", "DellFmpMcu",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14610837-4e97-4427-96e0-21d9b2956996": EfiGuid(
        "14610837-4e97-4427-96e0-21d9b2956996", "EsalVariableDxeSal",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14702e6f-73fb-46d5-bf5b-f0ad781a17b7": EfiGuid(
        "14702e6f-73fb-46d5-bf5b-f0ad781a17b7", "AuthenticationPromptProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1471b886-a01f-4835-903c-5c7e792075c2": EfiGuid(
        "1471b886-a01f-4835-903c-5c7e792075c2", "OememPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1478454a-4584-4cca-b0d2-120ace129dbb": EfiGuid(
        "1478454a-4584-4cca-b0d2-120ace129dbb", "DELL_MFG_MODE_SMM_DRIVER_GUID",
        description = "",
        aliases = ['DellMfgModeSmmDriver'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "147b4839-5dbe-413f-917f-dfeb687c6312": EfiGuid(
        "147b4839-5dbe-413f-917f-dfeb687c6312", "PchResetPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14814579-d905-4a3b-8874-410b518674e7": EfiGuid(
        "14814579-d905-4a3b-8874-410b518674e7", "RstNvmExpressSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1484ebe8-2681-45f1-a2e5-12ecad893b62": EfiGuid(
        "1484ebe8-2681-45f1-a2e5-12ecad893b62", "ArmJunoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14862240-96e5-4986-ab8e-9cec8bc2ab57": EfiGuid(
        "14862240-96e5-4986-ab8e-9cec8bc2ab57", "ShowBmcIp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "148c8945-f4a2-4586-a508-42f5470b2d47": EfiGuid(
        "148c8945-f4a2-4586-a508-42f5470b2d47", "DellSpdSmbusAccessDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14982a4f-b0ed-45b8-a811-5a7a9bc232df": EfiGuid(
        "14982a4f-b0ed-45b8-a811-5a7a9bc232df", "EFI_HII_SET_KEYBOARD_LAYOUT_EVENT_GUID",
        description = "",
        aliases = ['EfiHiiKeyBoardLayoutGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/HiiKeyBoardLayout.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiHiiKeyBoardLayout', 'EfiHiiKeyBoardLayoutGuid']),

    "149a6b53-fafa-44f9-8baf-f50c68a05236": EfiGuid(
        "149a6b53-fafa-44f9-8baf-f50c68a05236", "ScInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['ScInitDxe'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '633e80e7347dc01c21c5133227df2aba',
            '811505684bd99c18f5ecbee1f725ff66', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b6cce665f85751125c3db2050ffc98d2',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "149e1cbd-a263-46db-895c-4b0201d242b1": EfiGuid(
        "149e1cbd-a263-46db-895c-4b0201d242b1", "CrServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14a7c46f-bc02-4047-9f18-a5d725d8bd19": EfiGuid(
        "14a7c46f-bc02-4047-9f18-a5d725d8bd19", "EfiDFUResultGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14bae438-5c05-483e-afdb-d88325cc0e93": EfiGuid(
        "14bae438-5c05-483e-afdb-d88325cc0e93", "DellKeyManagementServiceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14bb6da5-0c47-4f2c-9348-8dc272619998": EfiGuid(
        "14bb6da5-0c47-4f2c-9348-8dc272619998", "VmwSvgaDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14bd268f-76fc-4edb-87f0-f0d4ebb256b4": EfiGuid(
        "14bd268f-76fc-4edb-87f0-f0d4ebb256b4", "VfsFingerPrintDevice",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14d159a2-52ed-465d-9353-f239c5464bf5": EfiGuid(
        "14d159a2-52ed-465d-9353-f239c5464bf5", "DellSmmLomProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14e7d9be-7834-477e-a942-ff5f823ee249": EfiGuid(
        "14e7d9be-7834-477e-a942-ff5f823ee249", "DELL_SPI_PART_ATMEL_GUID",
        description = "",
        aliases = ['DellSpiPartAtmel'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14e9734f-3644-4f3f-adfe-ab80bf5b48bc": EfiGuid(
        "14e9734f-3644-4f3f-adfe-ab80bf5b48bc", "SMBGlobalAcpiNvs",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14f95e01-d562-432e-844a-95a43905107e": EfiGuid(
        "14f95e01-d562-432e-844a-95a43905107e", "GuidBase",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "14fc52be-01dc-426c-91ae-a23c3e220ae8": EfiGuid(
        "14fc52be-01dc-426c-91ae-a23c3e220ae8", "EFI_SMM_SX_DISPATCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmmSxDispatchProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_SMM_SX_DISPATCH_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "14ff38a8-acba-4228-a7d7-a73260c7559b": EfiGuid(
        "14ff38a8-acba-4228-a7d7-a73260c7559b", "PiSmmStatusCodeOnFrameworkSmmStatusCodeThunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1501f737-c56d-4721-988f-7437f3d22fd9": EfiGuid(
        "1501f737-c56d-4721-988f-7437f3d22fd9", "H19SmmReadyToLockHook",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "150360ef-99be-4e43-94bb-bd4026ca3457": EfiGuid(
        "150360ef-99be-4e43-94bb-bd4026ca3457", "SPI_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1504ba7b-58f8-4d12-9638-b494a7044376": EfiGuid(
        "1504ba7b-58f8-4d12-9638-b494a7044376", "LENOVO_SYSTEM_SMM_SERVICES_SMM_GUID",
        description = "",
        aliases = ['LenovoSystemSmmServicesSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "150ce416-ee63-46b6-8ba3-7322bbe04637": EfiGuid(
        "150ce416-ee63-46b6-8ba3-7322bbe04637", "SA_PLATFORM_POLICY_PPI_GUID",
        description = "",
        aliases = ['SaPlatformPolicyPpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SA_PLATFORM_POLICY_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cfba6c85608b8135fb21024783573bf6', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "150ec16f-0bb6-42a8-b605-79794b89d15b": EfiGuid(
        "150ec16f-0bb6-42a8-b605-79794b89d15b", "DELL_EDIAGS_CONFIG_GUID",
        description = "",
        aliases = ['DellEdiagsConfig'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "151252ac-f77d-4c44-9977-a48cbeec9cca": EfiGuid(
        "151252ac-f77d-4c44-9977-a48cbeec9cca", "SmmHooks",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "151664d4-6001-478f-8c97-c51420fc2b75": EfiGuid(
        "151664d4-6001-478f-8c97-c51420fc2b75", "FchI2cMasterDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "151c8eae-7f2c-472c-9e54-9828194f6a88": EfiGuid(
        "151c8eae-7f2c-472c-9e54-9828194f6a88", "EFI_DISK_IO2_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDiskIo2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DiskIo2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "152656b6-564b-4c8c-877e-b67fb70dfddc": EfiGuid(
        "152656b6-564b-4c8c-877e-b67fb70dfddc", "FjHobFlexIOPresence",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15344673-d365-4be2-8513-1497cc07611d": EfiGuid(
        "15344673-d365-4be2-8513-1497cc07611d", "PCH_PLATFORM_POLICY_PPI_GUID",
        description = "",
        aliases = ['PchPlatformPolicyPpiGuid'],
        urls = [],
        vendors = [],
        filenames = ['PCH_PLATFORM_POLICY_PPI_GUID'],
        hashes = ['a9f4f993c8cee3cd83fecdb334fdf666', 'c0bdac2284b1e46c7c9c1509ddbea006'],
        junk = []),

    "15380de9-1d51-407f-be3e-ca71ec05c60d": EfiGuid(
        "15380de9-1d51-407f-be3e-ca71ec05c60d", "AsusSetupDxeVariableItem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15446019-9170-436a-a981-cc7521e9d7f9": EfiGuid(
        "15446019-9170-436a-a981-cc7521e9d7f9", "Usb4PlatformPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "154774ec-4350-40d4-af66-7d1837bcd559": EfiGuid(
        "154774ec-4350-40d4-af66-7d1837bcd559", "EfiHeciTrustedChannelSmmProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0e6eebbc21300ba0ebec9f192d5beabf'],
        junk = []),

    "1547b4f3-3e8a-4fef-81c8-328ed647ab1a": EfiGuid(
        "1547b4f3-3e8a-4fef-81c8-328ed647ab1a", "SYSTEM_ROM_FILE_GUID",
        description = "",
        aliases = ['Csm16'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Csm/Csm16/Csm16.inf",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Csm/CsmSupportLib/LegacyPlatform.c",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['Csm16']),

    "15488fa6-2391-4c9b-6fb7-6fe0e0f410bc": EfiGuid(
        "15488fa6-2391-4c9b-6fb7-6fe0e0f410bc", "DellPeiPolicyInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "154cab4a-52b5-46cd-99c3-4368abbacffd": EfiGuid(
        "154cab4a-52b5-46cd-99c3-4368abbacffd", "MetronomeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['Metronome']),

    "154d6d26-54b8-45bc-ba3a-cbaa20c02a6a": EfiGuid(
        "154d6d26-54b8-45bc-ba3a-cbaa20c02a6a", "SmmCpuPlatformHookLibQemu",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/SmmCpuPlatformHookLibQemu/SmmCpuPlatformHookLibQemu.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "15510d47-fadc-41f4-8b9d-fc323c821fe2": EfiGuid(
        "15510d47-fadc-41f4-8b9d-fc323c821fe2", "VerboseDisableCursor",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1551a247-bb31-4393-8bb4-10509ae2f18f": EfiGuid(
        "1551a247-bb31-4393-8bb4-10509ae2f18f", "ChargeLedDebugPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15551af8-56a3-43df-b0bd-22422ad2f08d": EfiGuid(
        "15551af8-56a3-43df-b0bd-22422ad2f08d", "DeviceStatusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1555acf3-bd07-4685-b668-a86945a4124d": EfiGuid(
        "1555acf3-bd07-4685-b668-a86945a4124d", "DELL_CPU_PEI_BEFORE_MEM_GUID",
        description = "",
        aliases = ['CpuPeiBeforeMem'],
        urls = [],
        vendors = [],
        filenames = ['DELL_CPU_PEI_BEFORE_MEM_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['CpuPeiBeforeMem']),

    "1555acf3-bd07-4e88-b3a0-b77f78eb34fe": EfiGuid(
        "1555acf3-bd07-4e88-b3a0-b77f78eb34fe", "CPU_PEI_BEFORE_MEM_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "156c3b97-64de-4af7-80d1-1b2600dc8f54": EfiGuid(
        "156c3b97-64de-4af7-80d1-1b2600dc8f54", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['1e9a8e238836fceb393a1318e4bfadc9'],
        junk = []),

    "15721c79-5720-4838-a544-052563708e24": EfiGuid(
        "15721c79-5720-4838-a544-052563708e24", "AlderLakeSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "157c666c-7c74-4e4a-b639-7bba21487ce1": EfiGuid(
        "157c666c-7c74-4e4a-b639-7bba21487ce1", "QFanDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15853d7c-3ddf-43e0-a1cb-ebf85b8f872c": EfiGuid(
        "15853d7c-3ddf-43e0-a1cb-ebf85b8f872c", "EFI_DEFERRED_IMAGE_LOAD_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDeferredImageLoadProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DeferredImageLoad.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "158b59e5-de4b-4964-8306-40c03f5b5d57": EfiGuid(
        "158b59e5-de4b-4964-8306-40c03f5b5d57", "FjMeDataRecoveryEventLogDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "158dc712-f15a-44dc-93bb-1675045be066": EfiGuid(
        "158dc712-f15a-44dc-93bb-1675045be066", "HashLibBaseCryptoRouterDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/HashLibBaseCryptoRouter/HashLibBaseCryptoRouterDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "158def5a-f656-419c-b027-7a3192c079d2": EfiGuid(
        "158def5a-f656-419c-b027-7a3192c079d2", "SHELL_VARIABLE_GUID",
        description = "",
        aliases = ['ShellVariableGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellVariableGuid.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "158e2079-23f3-4e83-8a35-657a76408c0a": EfiGuid(
        "158e2079-23f3-4e83-8a35-657a76408c0a", "ApobSspPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1597ab4f-d542-4efe-9af7-b244ec544c0b": EfiGuid(
        "1597ab4f-d542-4efe-9af7-b244ec544c0b", "EdkiiPeiPciDevicePpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15a02416-dc90-4f8c-b687-093ffa8bb966": EfiGuid(
        "15a02416-dc90-4f8c-b687-093ffa8bb966", "AMIProjectDxePriori",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15a450be-5d20-47e9-8b3b-856058ccd408": EfiGuid(
        "15a450be-5d20-47e9-8b3b-856058ccd408", "AsusNct6796DHwmPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15a5baf6-1c91-467d-9dfb-319d178d4bb4": EfiGuid(
        "15a5baf6-1c91-467d-9dfb-319d178d4bb4", "UniversalPayloadExtraData",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15af363b-7c41-4d1a-800d-8bcee473ff3b": EfiGuid(
        "15af363b-7c41-4d1a-800d-8bcee473ff3b", "AdlMiscellaneous",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15afbe86-e7f9-42f9-a02d-e95043e62f98": EfiGuid(
        "15afbe86-e7f9-42f9-a02d-e95043e62f98", "AdlSmbios",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15b26f43-a389-4bae-bde3-4bb0719b7d4f": EfiGuid(
        "15b26f43-a389-4bae-bde3-4bb0719b7d4f", "SerialPortLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15b985c5-7103-4f35-b59d-2235fc5f3ffe": EfiGuid(
        "15b985c5-7103-4f35-b59d-2235fc5f3ffe", "UsbTypeCDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['UsbTypeC', 'UsbTypeCDxe'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '1ded807f42af76804c9a0bbb6e24a2a9',
            '28e65e1a6689d8a2a4a612a06cbd2a58', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '633e80e7347dc01c21c5133227df2aba', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e7a706c2421a55cde70e476d1d37f831', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "15b9b6da-00a9-4de7-b8e8-ed7afb88f16e": EfiGuid(
        "15b9b6da-00a9-4de7-b8e8-ed7afb88f16e", "CpuPolicyInitDxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CpuPolicyInitDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "15c5e761-58d8-461a-9173-cab020916264": EfiGuid(
        "15c5e761-58d8-461a-9173-cab020916264", "VgaMiniPort",
        description = "",
        aliases = ['VAminiPort'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15c75527-302a-4872-bbed-4730a0abffca": EfiGuid(
        "15c75527-302a-4872-bbed-4730a0abffca", "LegacyBootEvent",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15c80344-f980-4bf5-aaa0-bfbe027aef16": EfiGuid(
        "15c80344-f980-4bf5-aaa0-bfbe027aef16", "LENOVO_EC_SERVICE_GUID",
        description = "",
        aliases = ['LenovoEcService'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15ccacbe-2a4a-45ed-9ec2-53135f98ab24": EfiGuid(
        "15ccacbe-2a4a-45ed-9ec2-53135f98ab24", "BmcAcpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15cf24a9-f9ab-46d0-8dcf-83664c632fd8": EfiGuid(
        "15cf24a9-f9ab-46d0-8dcf-83664c632fd8", "MemDetect",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15d101a9-60c5-4577-9672-de91842d01d5": EfiGuid(
        "15d101a9-60c5-4577-9672-de91842d01d5", "FjRTS52500S",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15d9a694-3d2a-4184-9672-ba55c3070e07": EfiGuid(
        "15d9a694-3d2a-4184-9672-ba55c3070e07", "PeiMemEncryptSevLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/BaseMemEncryptSevLib/PeiMemEncryptSevLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "15dd5676-2679-4e24-9caa-85b22dd893eb": EfiGuid(
        "15dd5676-2679-4e24-9caa-85b22dd893eb", "LENOVO_SECURE_FLASH_VERIFY_SMM_GUID",
        description = "",
        aliases = ['LenovoSecureFlashVerifySmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15e4c005-52c0-444c-8c94-56ffa04c7b36": EfiGuid(
        "15e4c005-52c0-444c-8c94-56ffa04c7b36", "SmuV12Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15eb6248-fb67-4690-b368-270b73b6ed2c": EfiGuid(
        "15eb6248-fb67-4690-b368-270b73b6ed2c", "OemUniversalDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15ef0bf5-b7ed-4fb6-93d5-08bb4de87abb": EfiGuid(
        "15ef0bf5-b7ed-4fb6-93d5-08bb4de87abb", "HbmMemMap",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "15fe2940-b426-479a-a002-5454a34c7a6e": EfiGuid(
        "15fe2940-b426-479a-a002-5454a34c7a6e", "FlashMapBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1601e050-be0c-41b7-8f96-9f48f72b7e26": EfiGuid(
        "1601e050-be0c-41b7-8f96-9f48f72b7e26", "OnboardControllerCtrl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16036a73-e8ef-46d0-953c-9b8e96527d13": EfiGuid(
        "16036a73-e8ef-46d0-953c-9b8e96527d13", "Reset",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/ResetRuntimeDxe/ResetRuntimeDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "160d11d8-93d8-40a2-b1f8-7f7e80e0c8c7": EfiGuid(
        "160d11d8-93d8-40a2-b1f8-7f7e80e0c8c7", "DellSecureBootDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1612ccdf-2549-466a-bf6f-d06daae60958": EfiGuid(
        "1612ccdf-2549-466a-bf6f-d06daae60958", "AppleKeyMapAggregator",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "161be597-e9c5-49db-ae50-c462ab54eeda": EfiGuid(
        "161be597-e9c5-49db-ae50-c462ab54eeda", "POWER_MANAGEMENT_ACPI_TABLE_STORAGE_GUID",
        description = "",
        aliases = ['PowerManagementAcpiTables2'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['POWER_MANAGEMENT_ACPI_TABLE_STORAGE_GUID'],
        hashes = [
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = ['PowerManagementAcpiTables2']),

    "161f757e-ed55-4a75-a6f0-ed2df16e0842": EfiGuid(
        "161f757e-ed55-4a75-a6f0-ed2df16e0842", "LanguageDefaultsAndWmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16259edb-3a84-4fc1-34ba-ba5ac23256dc": EfiGuid(
        "16259edb-3a84-4fc1-34ba-ba5ac23256dc", "OemWwanDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16271fca-55d9-4a33-93fc-5a3eb128de21": EfiGuid(
        "16271fca-55d9-4a33-93fc-5a3eb128de21", "MiscSubclassDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['MiscSubclassDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "16271fca-55d9-4a33-93fc-5a3eb128deb6": EfiGuid(
        "16271fca-55d9-4a33-93fc-5a3eb128deb6", "_CRB_DXE_GUID",
        description = "",
        aliases = ['CrbDxe'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CrbDxe', '_CRB_DXE_GUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "16271fca-55d9-4a33-93fc-5a3eb128dec7": EfiGuid(
        "16271fca-55d9-4a33-93fc-5a3eb128dec7", "EcsPlatformDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "16271fca-55d9-4a33-93fc-5a3eb128dee6": EfiGuid(
        "16271fca-55d9-4a33-93fc-5a3eb128dee6", "OemDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "162793aa-0fe7-4010-9818-45f55c5ccc3e": EfiGuid(
        "162793aa-0fe7-4010-9818-45f55c5ccc3e", "AmdCpmInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16287ba4-5b9d-4d98-919f-7b7b78cb2be0": EfiGuid(
        "16287ba4-5b9d-4d98-919f-7b7b78cb2be0", "AMI_LOAD_CSM_GUID",
        description = "",
        aliases = ['AmiLoadCsm'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "162cca7d-db17-4cd4-99c4-15f16282206e": EfiGuid(
        "162cca7d-db17-4cd4-99c4-15f16282206e", "GoodixTouchpadUpdate",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "163699ca-ffd9-4efa-9901-a7b2df3de7d1": EfiGuid(
        "163699ca-ffd9-4efa-9901-a7b2df3de7d1", "Class3Smm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "163ba792-f811-4fba-ae8c-101732f0328f": EfiGuid(
        "163ba792-f811-4fba-ae8c-101732f0328f", "AdlSemaThermalControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16449edb-3a84-4fc1-88ba-ba5ac23256dc": EfiGuid(
        "16449edb-3a84-4fc1-88ba-ba5ac23256dc", "OemWwanPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "164b86bf-b265-4033-92f9-bcb6d4adc0c9": EfiGuid(
        "164b86bf-b265-4033-92f9-bcb6d4adc0c9", "OobProvisionDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "164ce56b-5ebc-4984-b0da-c12b4f21d2ec": EfiGuid(
        "164ce56b-5ebc-4984-b0da-c12b4f21d2ec", "AmdRasRvSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "164e5590-fb7a-47b9-8687-3e91da499d97": EfiGuid(
        "164e5590-fb7a-47b9-8687-3e91da499d97", "AmiAmdUefiRaid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "165e05f4-ccd6-418c-9bd9-6d6ff9b1b996": EfiGuid(
        "165e05f4-ccd6-418c-9bd9-6d6ff9b1b996", "HpCommonDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16639ade-3d0e-45f2-b158-9186fc416a1e": EfiGuid(
        "16639ade-3d0e-45f2-b158-9186fc416a1e", "EfiFileSystemCapsuleStorage",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "166c533a-8f1e-4d34-a60e-0f68d8d61308": EfiGuid(
        "166c533a-8f1e-4d34-a60e-0f68d8d61308", "OemKey",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "166cae80-98de-45b9-bb47-85307145d874": EfiGuid(
        "166cae80-98de-45b9-bb47-85307145d874", "AmdNbioGfxRMBPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "166cd554-8aae-4617-8fdd-a2e3a5afd89e": EfiGuid(
        "166cd554-8aae-4617-8fdd-a2e3a5afd89e", "SystemSetupSecurityDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "166d1713-23a0-4f6a-9821-acd94f77bde3": EfiGuid(
        "166d1713-23a0-4f6a-9821-acd94f77bde3", "SbSocBixbyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "166fd043-ea13-4848-bb3c-6fa295b94627": EfiGuid(
        "166fd043-ea13-4848-bb3c-6fa295b94627", "DELL_VARIABLE_SMM_GUID",
        description = "",
        aliases = ['DellVariable2Smm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16791c86-47ba-bfc7-468f-d9b50d509487": EfiGuid(
        "16791c86-47ba-bfc7-468f-d9b50d509487", "AmdPsppControlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "167cc1f7-f4ab-46cc-8773-6278056c14eb": EfiGuid(
        "167cc1f7-f4ab-46cc-8773-6278056c14eb", "BootMaintDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1682fe44-bd7a-4407-b7c7-dca37ca3922d": EfiGuid(
        "1682fe44-bd7a-4407-b7c7-dca37ca3922d", "EFI_TLS_CONFIGURATION_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTlsConfigurationProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/TlsConfig.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "168d1a6e-f4a5-448a-9e95-795661bb3067": EfiGuid(
        "168d1a6e-f4a5-448a-9e95-795661bb3067", "ArmPciCpuIo2Dxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Drivers/ArmPciCpuIo2Dxe/ArmPciCpuIo2Dxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "168e8ffc-b8de-48e4-9cfc-d8a667abef11": EfiGuid(
        "168e8ffc-b8de-48e4-9cfc-d8a667abef11", "GnbSocRenoirPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "168e8ffc-b8de-48e4-9cfc-d8a667abef65": EfiGuid(
        "168e8ffc-b8de-48e4-9cfc-d8a667abef65", "GnbSocRavenPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16958446-19b7-480b-b047-7485ad3f716d": EfiGuid(
        "16958446-19b7-480b-b047-7485ad3f716d", "FDT_HOB_GUID",
        description = "",
        aliases = ['FdtHobGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec",
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Include/Guid/FdtHob.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "16a4add0-ef11-4c86-b159-88a2a8c4501c": EfiGuid(
        "16a4add0-ef11-4c86-b159-88a2a8c4501c", "wifi_3bars",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16a66e14-40a3-4a0c-b569-1c6822373345": EfiGuid(
        "16a66e14-40a3-4a0c-b569-1c6822373345", "AmdSocFp7r2PhxPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16ad4fe0-b5b1-11df-8cbf-0002a5d5c51b": EfiGuid(
        "16ad4fe0-b5b1-11df-8cbf-0002a5d5c51b", "PL310L2Cache",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16b14e2b-dad9-4618-ace2-7d324a26f9a0": EfiGuid(
        "16b14e2b-dad9-4618-ace2-7d324a26f9a0", "FjGpioResetPDPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16b45da2-7d70-4aea-a58d-760e9ecb841d": EfiGuid(
        "16b45da2-7d70-4aea-a58d-760e9ecb841d", "FU_EFI_FIRMWARE_VOLUME_GUID_PFH1",
        description = "",
        aliases = ['FD_Drv_X86'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['FD_Drv_X86']),

    "16b6109e-194c-440f-94f8-c7ccccc32deb": EfiGuid(
        "16b6109e-194c-440f-94f8-c7ccccc32deb", "EfiCseEndofPostProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16c368fe-f174-4881-92ce-388699d34d95": EfiGuid(
        "16c368fe-f174-4881-92ce-388699d34d95", "DELL_SMM_GPIO_POLICY_GUID",
        description = "",
        aliases = ['SmmGpioPolicy'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16c58600-554b-4587-8c62-a40997cfe206": EfiGuid(
        "16c58600-554b-4587-8c62-a40997cfe206", "ASRockNetSmtpBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16c8a6d0-fe8a-4082-a208-cf89c4290433": EfiGuid(
        "16c8a6d0-fe8a-4082-a208-cf89c4290433", "UefiSystemTableInfo",
        description = "",
        aliases = ['UefiSystemTableInfoGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/UefiPayloadPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "16d0a23e-c09c-407d-a14a-ad058fdd0ca1": EfiGuid(
        "16d0a23e-c09c-407d-a14a-ad058fdd0ca1", "DELL_ACPI_GUID",
        description = "",
        aliases = ['ACPI'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['ACPI', 'DELL_ACPI_GUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "16d11030-71ba-4e5e-a9f9-b475a549048a": EfiGuid(
        "16d11030-71ba-4e5e-a9f9-b475a549048a", "EfiRedirElogProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16dbcd0a-ed77-442c-a4ab-3dcade378a1c": EfiGuid(
        "16dbcd0a-ed77-442c-a4ab-3dcade378a1c", "ThunderboltNhi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16e9ba4e-9ee9-48a8-a9a0-8e5f529911da": EfiGuid(
        "16e9ba4e-9ee9-48a8-a9a0-8e5f529911da", "Pca6107Pei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "16faa37e-4b6a-4891-9028-242de65a3b70": EfiGuid(
        "16faa37e-4b6a-4891-9028-242de65a3b70", "REDFISH_EFI_INFORMATION_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1705e39e-c353-11e2-a187-047d7b99e097": EfiGuid(
        "1705e39e-c353-11e2-a187-047d7b99e097", "SetupMenuService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17088572-377f-44ef-8f4e-b09fff46a070": EfiGuid(
        "17088572-377f-44ef-8f4e-b09fff46a070", "CPU_MICROCODE_FILE_GUID",
        description = "",
        aliases = ['Microcode'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CPU_MICROCODE_FILE_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['Microcode']),

    "170d9568-c969-4fa1-a533-db6779bfbe62": EfiGuid(
        "170d9568-c969-4fa1-a533-db6779bfbe62", "LENOVO_EN25_QH128_FLASH_PART_SMM_GUID",
        description = "",
        aliases = ['LenovoEn25Qh128FlashPartSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "170e13c0-bf1b-4218-871d-2abdc6f887bc": EfiGuid(
        "170e13c0-bf1b-4218-871d-2abdc6f887bc", "EFI_OEM_BADGING_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiOEMBadgingProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "171148f3-6cf6-4c58-85d6-73013e19d90e": EfiGuid(
        "171148f3-6cf6-4c58-85d6-73013e19d90e", "DriveEraseInterfaceCoreDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17119241-1153-970d-6509-75dcdfa41774": EfiGuid(
        "17119241-1153-970d-6509-75dcdfa41774", "SbSocRavenDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "171272dd-45cf-45e8-bcd9-f3891bf22975": EfiGuid(
        "171272dd-45cf-45e8-bcd9-f3891bf22975", "BiosAuditLogDefaultsAndWmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "171e9188-31d3-40f5-b10c-539b2db940cd": EfiGuid(
        "171e9188-31d3-40f5-b10c-539b2db940cd", "EFI_SHELLPKG_TOKEN_SPACE_GUID",
        description = "",
        aliases = ['EfiShellPkgTokenSpaceGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellPkgTokenSpace.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "171e9398-269c-4081-9099-3844e260466c": EfiGuid(
        "171e9398-269c-4081-9099-3844e260466c", "EfiPlatformTypeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "171f43dc-c4d9-47a6-9641-65ddcdd5aa30": EfiGuid(
        "171f43dc-c4d9-47a6-9641-65ddcdd5aa30", "UsbRtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['UsbRtDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0e6eebbc21300ba0ebec9f192d5beabf', '1ded807f42af76804c9a0bbb6e24a2a9',
            '20f1da0b4063ef2319439a473c0d4ae3', '2b4ca0bd16a20893deff9632c23c2cca',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '8091cabbb4de557d84e3dbd55126455a',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '97070b6e86ea62d42df9c0d42179c050',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b4fa3a413d748be6699112adc0c14b0d',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'e7a706c2421a55cde70e476d1d37f831', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1722efd4-b7f0-41e8-ad21-0da8fd6297a3": EfiGuid(
        "1722efd4-b7f0-41e8-ad21-0da8fd6297a3", "AmtLockUsbConInDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "172b4edf-df64-48e5-acae-bdb51878fdab": EfiGuid(
        "172b4edf-df64-48e5-acae-bdb51878fdab", "AsfPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "173220da-f287-4d34-84c4-12ea968b927e": EfiGuid(
        "173220da-f287-4d34-84c4-12ea968b927e", "FjGabiEntrySmiDispatcherSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17390333-4ce6-48a8-afb9-0ef55b666b74": EfiGuid(
        "17390333-4ce6-48a8-afb9-0ef55b666b74", "DELL_RUNTIME_ACPI_DXE_GUID",
        description = "",
        aliases = ['RunTimeAcpiDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "174005f5-3663-4305-9062-904b2be2d07a": EfiGuid(
        "174005f5-3663-4305-9062-904b2be2d07a", "ApobPhxPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17407e5a-af6c-4ee8-98a8-00210453cdd9": EfiGuid(
        "17407e5a-af6c-4ee8-98a8-00210453cdd9", "APPLE_SMC_IO_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "174a8f8a-7b53-48b4-aca9-d955ab1b5e90": EfiGuid(
        "174a8f8a-7b53-48b4-aca9-d955ab1b5e90", "SrSetupPure",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "174cf46d-b167-4e6a-b1cd-d41e24efa0f9": EfiGuid(
        "174cf46d-b167-4e6a-b1cd-d41e24efa0f9", "AsusBackupSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1755e35e-3b29-4f3a-ad20-80788159da1f": EfiGuid(
        "1755e35e-3b29-4f3a-ad20-80788159da1f", "LEMPasswordStoreProtocolSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17565311-4b71-4340-88aa-dc9f4422e53a": EfiGuid(
        "17565311-4b71-4340-88aa-dc9f4422e53a", "SMM_BIOS_GUARD_PROTOCOL_GUID",
        description = "",
        aliases = ['SmmBiosGuardProtocol'],
        urls = [],
        vendors = [],
        filenames = ['SMM_BIOS_GUARD_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '869f701de5296248126c8c8a03a479c3',
            '953dc1131b7a579176b9fc937eee5846', 'b49cea81c680d9848212ed2aca8e6771',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189',
        ],
        junk = []),

    "175a5bb0-1508-4714-a1cf-34bd56c825df": EfiGuid(
        "175a5bb0-1508-4714-a1cf-34bd56c825df", "FjHobFlexIOSave",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "176652ee-2b20-4ce8-a61b-1c152726fec4": EfiGuid(
        "176652ee-2b20-4ce8-a61b-1c152726fec4", "HeciLegacyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1767ceed-db82-47cd-bf2b-68458a8ccfff": EfiGuid(
        "1767ceed-db82-47cd-bf2b-68458a8ccfff", "EFI_MEMORY_SUBCLASS_DRIVER_GUID",
        description = "",
        aliases = ['EfiMemorySubclassDriver'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17689034-f11b-468b-8cc4-e114c77f41b1": EfiGuid(
        "17689034-f11b-468b-8cc4-e114c77f41b1", "AsusPTTDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17706d27-83fe-4770-875f-4cef4cb8f63d": EfiGuid(
        "17706d27-83fe-4770-875f-4cef4cb8f63d", "AmiAhciPlatformPolicyProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17772369-d262-4b90-9f31-bdc41f2663a5": EfiGuid(
        "17772369-d262-4b90-9f31-bdc41f2663a5", "DELL_MEBX_SETUP_BINARY_GUID",
        description = "",
        aliases = ['mebx_main'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['mebx_main']),

    "177b2c74-9674-45f4-aaeb-43f5506ae0fe": EfiGuid(
        "177b2c74-9674-45f4-aaeb-43f5506ae0fe", "AsusPostMessageDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "177d39d2-43b8-40c8-9ae1-3c5198d6941e": EfiGuid(
        "177d39d2-43b8-40c8-9ae1-3c5198d6941e", "PEI_TCG_PPI_GUID",
        description = "",
        aliases = ['PeiTcgPpiGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['PEI_TCG_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9e80820994976fe9edc2b53346ea4f71', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "177e63c1-aed6-4d88-917e-9b4fb9fe4092": EfiGuid(
        "177e63c1-aed6-4d88-917e-9b4fb9fe4092", "PhoenixSmmCoreServicesProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17851fbf-45c4-4ff7-a2a0-c3b12d63c27e": EfiGuid(
        "17851fbf-45c4-4ff7-a2a0-c3b12d63c27e", "SdBlockIoPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Sd/SdBlockIoPei/SdBlockIoPei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17865dc0-0b8b-4da8-8b42-7c46b85cca4d": EfiGuid(
        "17865dc0-0b8b-4da8-8b42-7c46b85cca4d", "PCH_RESET_CALLBACK_PPI_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17985e6f-e778-4d94-aefa-c5dd2b77e186": EfiGuid(
        "17985e6f-e778-4d94-aefa-c5dd2b77e186", "QemuFwCfgAcpiPlatform",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/AcpiPlatformDxe/QemuFwCfgAcpiPlatformDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17a0a3d7-c0a5-4635-bbd5-072187dfe2ee": EfiGuid(
        "17a0a3d7-c0a5-4635-bbd5-072187dfe2ee", "EmbeddedGpioProtocolGuid",
        description = "",
        aliases = ['gEmbeddedGpioProtocolGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17a269de-7c35-4132-9ea0-a3eb5ab87f1e": EfiGuid(
        "17a269de-7c35-4132-9ea0-a3eb5ab87f1e", "PowerLostNotifyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17a8a22c-8365-4540-9866-daf6dabeabef": EfiGuid(
        "17a8a22c-8365-4540-9866-daf6dabeabef", "McBankErrorInjection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17aaf4d7-0e18-4b99-87c6-94c8b5f25fb4": EfiGuid(
        "17aaf4d7-0e18-4b99-87c6-94c8b5f25fb4", "FchShastaSmmInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17ae8af9-6644-4342-a786-4ca69645d137": EfiGuid(
        "17ae8af9-6644-4342-a786-4ca69645d137", "HspFtpmAcpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17af0060-a98c-4a03-90ad-6abed75045bd": EfiGuid(
        "17af0060-a98c-4a03-90ad-6abed75045bd", "CbsBaseDxeRV",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17bf72a7-ba65-4837-866d-979b2f2c075e": EfiGuid(
        "17bf72a7-ba65-4837-866d-979b2f2c075e", "RasClvPatrolScrubProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17c7fc8c-8c5d-497e-9c0e-c21255b30e04": EfiGuid(
        "17c7fc8c-8c5d-497e-9c0e-c21255b30e04", "StandaloneMmReportStatusCodeLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/SmmReportStatusCodeLib/StandaloneMmReportStatusCodeLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17ca9b37-5bab-492c-a09c-7121fbe34ce6": EfiGuid(
        "17ca9b37-5bab-492c-a09c-7121fbe34ce6", "LockBoxBaseLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/LockBoxLib/LockBoxBaseLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17ccf600-3aa4-4ff7-82d0-19cbeb78f443": EfiGuid(
        "17ccf600-3aa4-4ff7-82d0-19cbeb78f443", "MePolicyHelper",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17d09362-326c-4920-a50c-76f8a116a093": EfiGuid(
        "17d09362-326c-4920-a50c-76f8a116a093", "SeCPolicyInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SeCPolicyInitDxe'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '0fb0d5db7a0d624b39b616db11024c0a',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2b4ca0bd16a20893deff9632c23c2cca', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '633e80e7347dc01c21c5133227df2aba',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dab68811108b555173f8f096c9737d20',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "17d290e7-49be-41fc-bf62-912c17152fa8": EfiGuid(
        "17d290e7-49be-41fc-bf62-912c17152fa8", "DellSimulatedECDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17e2c90e-ad1f-443a-8c94-6e50c0e98607": EfiGuid(
        "17e2c90e-ad1f-443a-8c94-6e50c0e98607", "BaseS3PciSegmentLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseS3PciSegmentLib/BaseS3PciSegmentLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "17e9ec9e-4036-4237-b074-d1f29464599c": EfiGuid(
        "17e9ec9e-4036-4237-b074-d1f29464599c", "OemDataRegionFlashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17ed0da9-eac5-4613-bde7-c3a506917676": EfiGuid(
        "17ed0da9-eac5-4613-bde7-c3a506917676", "LoadPartialDefault",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17ee496a-d8e4-4b9a-94d1-ce8272300850": EfiGuid(
        "17ee496a-d8e4-4b9a-94d1-ce8272300850", "EfiPeiBootInRecoveryModePpiGuid",
        description = "",
        aliases = ['gEfiPeiBootInRecoveryModePpiGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec'],
        vendors = ['EDK2'],
        filenames = ['EfiPeiBootInRecoveryModePpiGuid'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "17f76387-de12-4e59-a107-485bb133b4d6": EfiGuid(
        "17f76387-de12-4e59-a107-485bb133b4d6", "SecureBioDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "17fe14b6-9aba-43c3-beb0-d12f81aaa883": EfiGuid(
        "17fe14b6-9aba-43c3-beb0-d12f81aaa883", "AmdCpmDiscreteUSB4Smm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "180636a5-871b-496d-b8f3-e83ef196d100": EfiGuid(
        "180636a5-871b-496d-b8f3-e83ef196d100", "menu_top_right",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1807040d-5934-41a2-a088-8e0f777f71ab": EfiGuid(
        "1807040d-5934-41a2-a088-8e0f777f71ab", "NvramDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['NVRAMDxe', 'NvramDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "180d765f-c489-4f7a-8a3f-596018499eaf": EfiGuid(
        "180d765f-c489-4f7a-8a3f-596018499eaf", "DTbtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1810ab4a-2314-4df6-81eb-67c6ec058591": EfiGuid(
        "1810ab4a-2314-4df6-81eb-67c6ec058591", "EFI_S3_BOOT_SCRIPT_TABLE_BASE_GUID",
        description = "",
        aliases = ['BootScriptTableBaseGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['BootScriptHeaderData', 'BootScriptTableBase']),

    "1812efd5-0dfb-44d7-9ad6-74fcbb33b192": EfiGuid(
        "1812efd5-0dfb-44d7-9ad6-74fcbb33b192", "UiAppWrapper",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "181e874d-c089-4c99-8ec2-6d6761348220": EfiGuid(
        "181e874d-c089-4c99-8ec2-6d6761348220", "AMI_PLL_OVER_VOTAGE_FLAG_HOB_GUID",
        description = "",
        aliases = ['AmiPllOverVotageFlagHob'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18231874-2933-4a4c-aaf1-65b4a752a409": EfiGuid(
        "18231874-2933-4a4c-aaf1-65b4a752a409", "FwHealthReporterDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1826b10e-ece1-4bd1-b351-bc469d17fc59": EfiGuid(
        "1826b10e-ece1-4bd1-b351-bc469d17fc59", "DellPlatformSwSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1827cfc7-4e61-4273-b796-d35f4b0c88fc": EfiGuid(
        "1827cfc7-4e61-4273-b796-d35f4b0c88fc", "LENOVO_HIDDEN_SETTING_VARIABLE_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "182f61f3-a6b7-4b56-ab1c-6ef6dde09798": EfiGuid(
        "182f61f3-a6b7-4b56-ab1c-6ef6dde09798", "FjGabiFlashBiosRegionCtrlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1830a6dd-e03d-4bc0-b115-94d91950fe4a": EfiGuid(
        "1830a6dd-e03d-4bc0-b115-94d91950fe4a", "SioDynamicSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SioDynamicSetup'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '633e80e7347dc01c21c5133227df2aba', '811505684bd99c18f5ecbee1f725ff66',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a9258521b7c65199bd268280867cc628',
            'b6cce665f85751125c3db2050ffc98d2', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "1831a379-2d48-45bd-9744-d4059d93815d": EfiGuid(
        "1831a379-2d48-45bd-9744-d4059d93815d", "dp",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ShellPkg/DynamicCommand/DpDynamicCommand/DpApp.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1834a306-0d70-470e-901f-2370ff30ddab": EfiGuid(
        "1834a306-0d70-470e-901f-2370ff30ddab", "DxeSpiNorFlash",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1834c334-6f84-4147-b72e-16a1af7bef02": EfiGuid(
        "1834c334-6f84-4147-b72e-16a1af7bef02", "AsusPEFirmwareBiosGuardVS2015Bin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18355ec0-ef43-4cd5-9132-d2b33c5e5897": EfiGuid(
        "18355ec0-ef43-4cd5-9132-d2b33c5e5897", "FjUsbTypecPwrLimitCtrlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "183bb3e1-a1e5-4445-8ac9-0e83b6547e0e": EfiGuid(
        "183bb3e1-a1e5-4445-8ac9-0e83b6547e0e", "CpuFeaturesPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/CpuFeatures/CpuFeaturesPei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "18435cd7-8003-4ced-afa4-ecbc440c0f30": EfiGuid(
        "18435cd7-8003-4ced-afa4-ecbc440c0f30", "FwBlockServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1849bda2-6952-4e86-a1db-559a3c479df1": EfiGuid(
        "1849bda2-6952-4e86-a1db-559a3c479df1", "EDKII_FIRMWARE_MANAGEMENT_PROGRESS_PROTOCOL_GUID",
        description = "",
        aliases = ['gEdkiiFirmwareManagementProgressProtocolGuid', 'EdkiiFirmwareManagementProgressProtocol'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/FirmwareManagementProgress.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "18578e75-d073-4203-90d2-8788a871e44d": EfiGuid(
        "18578e75-d073-4203-90d2-8788a871e44d", "LENOVO_COMPUTRACE_SMI_SERVICES_GUID",
        description = "",
        aliases = ['LenovoComputraceSmiServices'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1859044f-c25a-4aa8-965a-2410d06a603f": EfiGuid(
        "1859044f-c25a-4aa8-965a-2410d06a603f", "ApobBrhPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18633bfc-1735-4217-8ac9-17239282d3f8": EfiGuid(
        "18633bfc-1735-4217-8ac9-17239282d3f8", "EFI_BTT_ABSTRACTION_GUID",
        description = "",
        aliases = ['EfiBttAbstractionGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/Btt.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1879bfdd-47ff-485b-88fb-4fcd206e49f4": EfiGuid(
        "1879bfdd-47ff-485b-88fb-4fcd206e49f4", "AmdSocFp7r2RmbDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "188daed4-939d-4faa-9f47-f01d5c16dd82": EfiGuid(
        "188daed4-939d-4faa-9f47-f01d5c16dd82", "H19ErrorLog",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "189641f9-4f75-423f-9427-6457e8ed095d": EfiGuid(
        "189641f9-4f75-423f-9427-6457e8ed095d", "DashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18968e73-3c45-4333-b97f-8764c23923ac": EfiGuid(
        "18968e73-3c45-4333-b97f-8764c23923ac", "FjAutoBiosUpdateInterfaceProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18a031ab-b443-4d1a-a5c0-0c09261e9f71": EfiGuid(
        "18a031ab-b443-4d1a-a5c0-0c09261e9f71", "EFI_DRIVER_BINDING_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDriverBindingProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DriverBinding.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "18a3b667-14af-4ae7-840f-cd6317ea0cbd": EfiGuid(
        "18a3b667-14af-4ae7-840f-cd6317ea0cbd", "NbRsRouting",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18a3c6dc-5eea-48c8-a1c1-b53389f98999": EfiGuid(
        "18a3c6dc-5eea-48c8-a1c1-b53389f98999", "EFI_MM_SW_DISPATCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_SMM_SW_DISPATCH2_PROTOCOL_GUID', 'gEfiMmSwDispatchProtocolGuid', 'EfiSmmSwDispatch2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MmSwDispatch.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SmmSwDispatch2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_MM_SW_DISPATCH_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EfiSmmSwDispatch2Protocol', 'EfiSmmSwDispatch2ProtocolGuid', 'EFI_SMM_SW_DISPATCH2_PROTOCOL_GUID']),

    "18af0e1a-cdb4-48c2-b111-14cb070417bf": EfiGuid(
        "18af0e1a-cdb4-48c2-b111-14cb070417bf", "SyncCBSconfigPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18b12c83-7718-4d83-ada4-87f2fe698dd4": EfiGuid(
        "18b12c83-7718-4d83-ada4-87f2fe698dd4", "ArmSmcPsciResetSystemLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Library/ArmSmcPsciResetSystemLib/ArmSmcPsciResetSystemLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "18bd198c-ecf5-40f2-98ed-c388c3fd9136": EfiGuid(
        "18bd198c-ecf5-40f2-98ed-c388c3fd9136", "DRAMPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18c040d4-15e4-4148-b7b3-582764710bbc": EfiGuid(
        "18c040d4-15e4-4148-b7b3-582764710bbc", "FprValidityRaptorDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18cb2cf0-40cd-11e5-b970-0800200c9a66": EfiGuid(
        "18cb2cf0-40cd-11e5-b970-0800200c9a66", "DellDptfChipsetDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18cb87ae-14f1-4829-9c10-a21449e40366": EfiGuid(
        "18cb87ae-14f1-4829-9c10-a21449e40366", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "18cc225e-8799-4945-9f92-ebb96423d6c1": EfiGuid(
        "18cc225e-8799-4945-9f92-ebb96423d6c1", "FjGabiFlashCommonGbeRegionCtrlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18cc325e-2799-4545-9f92-ebb9e423d6c1": EfiGuid(
        "18cc325e-2799-4545-9f92-ebb9e423d6c1", "FjGabiFlashCommonEcRegionCtrlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18d543e8-c59a-40dc-aa28-50b832c67d02": EfiGuid(
        "18d543e8-c59a-40dc-aa28-50b832c67d02", "CutOffStrToLimitedLength",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18d5ff31-7a47-424d-b500-7d792d3e33ac": EfiGuid(
        "18d5ff31-7a47-424d-b500-7d792d3e33ac", "DellDtDiagSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18d6fc7e-d862-48de-b56d-fa311421bf29": EfiGuid(
        "18d6fc7e-d862-48de-b56d-fa311421bf29", "EfiTpmCallBackProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18da03a8-fae4-410a-a8c2-34c91913f515": EfiGuid(
        "18da03a8-fae4-410a-a8c2-34c91913f515", "AmdPspIntrusionDetectionSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18ebdb3e-95c2-4725-ab63-74b478c475de": EfiGuid(
        "18ebdb3e-95c2-4725-ab63-74b478c475de", "AmdCpmSocAm4RvDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18ef8946-68f5-49e6-b202-ce90c3eef1c9": EfiGuid(
        "18ef8946-68f5-49e6-b202-ce90c3eef1c9", "IchSmmDispatcherDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "18f3779f-12d1-4ee8-b726-aa0c314d1fa5": EfiGuid(
        "18f3779f-12d1-4ee8-b726-aa0c314d1fa5", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['2b4ca0bd16a20893deff9632c23c2cca'],
        junk = []),

    "18fe7632-f5c8-4e63-8de8-17a55c5913bd": EfiGuid(
        "18fe7632-f5c8-4e63-8de8-17a55c5913bd", "gStandaloneMmPkgTokenSpaceGuid",
        description = "",
        aliases = ['StandaloneSmmPkgTokenSpace'],
        urls = ['https://github.com/tianocore/edk2/blob/master/StandaloneMmPkg/StandaloneMmPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1905fcf3-ddb6-49bb-a785-1e50c807160d": EfiGuid(
        "1905fcf3-ddb6-49bb-a785-1e50c807160d", "SystemSlotInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19123709-f770-4faa-858e-3bd5fb18464d": EfiGuid(
        "19123709-f770-4faa-858e-3bd5fb18464d", "OemSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1916f73e-c824-4f91-ac9c-bf1fd28e4431": EfiGuid(
        "1916f73e-c824-4f91-ac9c-bf1fd28e4431", "DELL_SERVICE_MENU_GUID",
        description = "",
        aliases = ['DellServiceMenu'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1925903d-3faa-4a91-a257-448f4513b309": EfiGuid(
        "1925903d-3faa-4a91-a257-448f4513b309", "AmdNbioPciePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "192afab3-55f8-44b8-b49e-275a9dfdd03f": EfiGuid(
        "192afab3-55f8-44b8-b49e-275a9dfdd03f", "SetupFunction",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19342d99-ff4e-4798-a3e0-fa36c961ccfe": EfiGuid(
        "19342d99-ff4e-4798-a3e0-fa36c961ccfe", "PdtCapsuleDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1944f611-fbcc-40bc-88c8-850d0fe514e3": EfiGuid(
        "1944f611-fbcc-40bc-88c8-850d0fe514e3", "WCN6855",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "194d473a-01f6-48ae-9429-fd395813be47": EfiGuid(
        "194d473a-01f6-48ae-9429-fd395813be47", "AdlinkNxpPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "194ee52b-02fe-418d-aa10-2e61e51cb894": EfiGuid(
        "194ee52b-02fe-418d-aa10-2e61e51cb894", "DELL_MONOTONIC_COUNTER_GUID",
        description = "",
        aliases = ['DellMonotonicCounter'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19618bce-55ae-09c6-37e9-4ce04084c7a1": EfiGuid(
        "19618bce-55ae-09c6-37e9-4ce04084c7a1", "httpDynamicCommand",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ShellPkg/DynamicCommand/HttpDynamicCommand/HttpDynamicCommand.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1964a5a5-91df-4b2a-b3c4-8b4bf238d0bf": EfiGuid(
        "1964a5a5-91df-4b2a-b3c4-8b4bf238d0bf", "SystemVirtualKeyboardDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1967167b-a13d-4e45-801b-a605d11946d5": EfiGuid(
        "1967167b-a13d-4e45-801b-a605d11946d5", "AmiTseOemPortingVar15",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1967dd9b-b72c-4328-8c80-d4acfc83fdf8": EfiGuid(
        "1967dd9b-b72c-4328-8c80-d4acfc83fdf8", "PciHotPlugDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "196bf9e3-20d7-4b7b-89f9-31c27208c9b9": EfiGuid(
        "196bf9e3-20d7-4b7b-89f9-31c27208c9b9", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "196bfe1f-51fc-43e6-8e14-d421552bf0c7": EfiGuid(
        "196bfe1f-51fc-43e6-8e14-d421552bf0c7", "FjFlashMapRt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "196ca3d8-9a5a-4735-b328-8ffc1d93d188": EfiGuid(
        "196ca3d8-9a5a-4735-b328-8ffc1d93d188", "TcgPlatformSetupPolicy",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['TcgPlatformSetupPolicy'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '7dffb9d276837abbf9e319d968a966a1',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "196f7f0b-8946-4333-a2b7-ec1bf4c97498": EfiGuid(
        "196f7f0b-8946-4333-a2b7-ec1bf4c97498", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "197db236-f856-4924-90f8-cdf12fb875f3": EfiGuid(
        "197db236-f856-4924-90f8-cdf12fb875f3", "Microcode",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['Microcode'],
        hashes = [
            '81460325c53ef849a2e3c2a5e104304b', 'b218fecd176e986c1491046e58c9e01b',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
        ],
        junk = []),

    "197e8644-1c26-7644-b5b4-978ed7301ad7": EfiGuid(
        "197e8644-1c26-7644-b5b4-978ed7301ad7", "PlanarSelfRepair",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "198251a4-de06-442a-aab7-b0ee1e6db8ff": EfiGuid(
        "198251a4-de06-442a-aab7-b0ee1e6db8ff", "GenericVar",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1983e991-5033-43da-8ea1-16d03ae895c8": EfiGuid(
        "1983e991-5033-43da-8ea1-16d03ae895c8", "MemoryMarginToolHookSmmShp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1988a1d8-04fc-4d68-b58d-636f36f30d82": EfiGuid(
        "1988a1d8-04fc-4d68-b58d-636f36f30d82", "SoftStrapsPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19984b85-18a1-468c-af3e-c3d93f18c9fa": EfiGuid(
        "19984b85-18a1-468c-af3e-c3d93f18c9fa", "AsusSetupRestorePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "199c1ef0-6400-41c5-b0a4-ffbf219dcbae": EfiGuid(
        "199c1ef0-6400-41c5-b0a4-ffbf219dcbae", "QuarkPlatformTokenSpace",
        description = "",
        aliases = ['QuarkPlatformTokenSpaceGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "199fd111-0785-4132-a9b3-1f66573f0060": EfiGuid(
        "199fd111-0785-4132-a9b3-1f66573f0060", "GenericComponentsSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19a84692-4aad-c04b-90e9-e017360a9860": EfiGuid(
        "19a84692-4aad-c04b-90e9-e017360a9860", "DiskImage",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19ad5244-fd6b-4e5c-826a-414646d6da6a": EfiGuid(
        "19ad5244-fd6b-4e5c-826a-414646d6da6a", "EfiGlkVariableGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19afd36b-f501-4446-9c9e-0457db6e7888": EfiGuid(
        "19afd36b-f501-4446-9c9e-0457db6e7888", "SetupItemLinkageDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19b058df-f97e-475f-92c7-bb26e0594fc8": EfiGuid(
        "19b058df-f97e-475f-92c7-bb26e0594fc8", "DashManagerDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19b1db23-e657-4d6d-b973-6f4d08f07aa4": EfiGuid(
        "19b1db23-e657-4d6d-b973-6f4d08f07aa4", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "19b23409-85bd-44b0-897f-ecfe8aa7a145": EfiGuid(
        "19b23409-85bd-44b0-897f-ecfe8aa7a145", "SmbiosElogType15",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19b4beae-54da-4aa9-98b0-41ed75d805ba": EfiGuid(
        "19b4beae-54da-4aa9-98b0-41ed75d805ba", "EcRotTpmRecoveryDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19b6f165-7855-42e8-aa6f-2361ca87e024": EfiGuid(
        "19b6f165-7855-42e8-aa6f-2361ca87e024", "LENOVO_SMM_VIDEO_INT10_GUID",
        description = "",
        aliases = ['LenovoSmmVideoInt10'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19be1e4b-1a9a-44c1-8f12-32dd0470516a": EfiGuid(
        "19be1e4b-1a9a-44c1-8f12-32dd0470516a", "DxeRuntimeCapsuleLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/DxeCapsuleLibFmp/DxeRuntimeCapsuleLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "19c2a1dc-73cb-46bf-a420-4c5558f958e8": EfiGuid(
        "19c2a1dc-73cb-46bf-a420-4c5558f958e8", "DmiEditorDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19c4b49d-28da-4232-996a-d175bbf6aff7": EfiGuid(
        "19c4b49d-28da-4232-996a-d175bbf6aff7", "SystemBiosSelfHealingPremiumDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19cb87ab-2cb9-4665-8360-ddcf6054f79d": EfiGuid(
        "19cb87ab-2cb9-4665-8360-ddcf6054f79d", "EFI_PCI_HOTPLUG_REQUEST_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPciHotPlugRequestProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/PciHotPlugRequest.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "19cbbb97-ff61-45ff-8c3f-dfa66dd118c8": EfiGuid(
        "19cbbb97-ff61-45ff-8c3f-dfa66dd118c8", "UefiRuntimeServicesTableLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/UefiRuntimeServicesTableLib/UefiRuntimeServicesTableLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "19ce57cc-628e-4f96-93d4-3c21f823c343": EfiGuid(
        "19ce57cc-628e-4f96-93d4-3c21f823c343", "LEMPhyPresenceSet20",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19d17940-ba8d-4fa7-a704-f33d9fafab9d": EfiGuid(
        "19d17940-ba8d-4fa7-a704-f33d9fafab9d", "LibStringlist",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19d92c2d-f089-4b7c-abe6-333a1205ed89": EfiGuid(
        "19d92c2d-f089-4b7c-abe6-333a1205ed89", "AmiSmmMultiPlatformProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19d96d3f-6a6a-47d2-b195-7b2432da3be2": EfiGuid(
        "19d96d3f-6a6a-47d2-b195-7b2432da3be2", "ADD_BOOT_OPTION_GUID",
        description = "",
        aliases = ['AddBootOption'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19dbbdc1-dedf-4ded-8684-2476b99fac91": EfiGuid(
        "19dbbdc1-dedf-4ded-8684-2476b99fac91", "AST2500DxeInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "19df145a-b1d4-453f-8507-38816676d7f6": EfiGuid(
        "19df145a-b1d4-453f-8507-38816676d7f6", "AtaBusDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Ata/AtaBusDxe/AtaBusDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "19e3bbba-beb1-43e8-b32d-9acbb22c7639": EfiGuid(
        "19e3bbba-beb1-43e8-b32d-9acbb22c7639", "BasePostCodeLibDebug",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BasePostCodeLibDebug/BasePostCodeLibDebug.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "19fe2fe9-470e-4c7e-8d55-6c7a08da855a": EfiGuid(
        "19fe2fe9-470e-4c7e-8d55-6c7a08da855a", "SetupDefault",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a10742f-fa80-4b79-9da6-357058cc397b": EfiGuid(
        "1a10742f-fa80-4b79-9da6-357058cc397b", "ArrowCursor",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a1241e6-8f19-41a9-bc0e-e8ef39e06546": EfiGuid(
        "1a1241e6-8f19-41a9-bc0e-e8ef39e06546", "EFI_HII_IMAGE_EX_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiHiiImageExProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/HiiImageEx.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1a1d1204-e385-4943-a892-5334ce3809ac": EfiGuid(
        "1a1d1204-e385-4943-a892-5334ce3809ac", "DELL_CSM_OEM_ROM_GUID",
        description = "",
        aliases = ['DellCsmOemRom'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a1e4886-9517-440e-9fde-3be44cee2136": EfiGuid(
        "1a1e4886-9517-440e-9fde-3be44cee2136", "CpuDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/CpuDxe/CpuDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['CpuDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '60785be6bfb0d1daa94a8bb214b774d1', '7a20bc8c8925db2963205a36be03b335',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'eee15c9794a4040563b41004ebccc91a',
        ],
        junk = []),

    "1a20b273-2cbf-4e64-a4a0-16a2af252c27": EfiGuid(
        "1a20b273-2cbf-4e64-a4a0-16a2af252c27", "AmdFchWheaSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a20be1f-33ad-450c-b49a-7123fca8b7f9": EfiGuid(
        "1a20be1f-33ad-450c-b49a-7123fca8b7f9", "ArmCacheMaintenanceLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmPkg/Library/ArmCacheMaintenanceLib/ArmCacheMaintenanceLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1a2614a0-89e7-11e3-2990-31d281feb1dc": EfiGuid(
        "1a2614a0-89e7-11e3-2990-31d281feb1dc", "NgnRasProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a2b4139-0da4-416c-ade3-85877b318266": EfiGuid(
        "1a2b4139-0da4-416c-ade3-85877b318266", "SmmControlBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a3113d2-91a2-41fd-a83f-ad5e6b90d43d": EfiGuid(
        "1a3113d2-91a2-41fd-a83f-ad5e6b90d43d", "AmdDashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a345bc0-cc35-9abc-1cea-1cad7d33addf": EfiGuid(
        "1a345bc0-cc35-9abc-1cea-1cad7d33addf", "SmcLsiRaidOOBSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a3558ec-13bb-4451-a589-74dfbd9a27ad": EfiGuid(
        "1a3558ec-13bb-4451-a589-74dfbd9a27ad", "LidPoller",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a36e4e7-fab6-476a-8e75-695a0576fdd7": EfiGuid(
        "1a36e4e7-fab6-476a-8e75-695a0576fdd7", "EFI_PEI_DECOMPRESS_PPI_GUID",
        description = "",
        aliases = ['EfiPeiDecompressPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/Decompress.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_DECOMPRESS_PPI_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9e80820994976fe9edc2b53346ea4f71',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ee40c5722cf3b43608a71e2a12c72193', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1a481e8e-342f-40aa-af31-f4fb7c99d428": EfiGuid(
        "1a481e8e-342f-40aa-af31-f4fb7c99d428", "AmtInt16",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a4d88c8-015e-42c2-9d4a-c4212bd881ac": EfiGuid(
        "1a4d88c8-015e-42c2-9d4a-c4212bd881ac", "MfgDoneSyncItem",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a57c46d-b93e-4036-8f33-1174ca091857": EfiGuid(
        "1a57c46d-b93e-4036-8f33-1174ca091857", "SystemDefaultVariableManagerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a682da2-3ee3-4793-8521-4db8a69939fc": EfiGuid(
        "1a682da2-3ee3-4793-8521-4db8a69939fc", "AmdNbioAlibMDNDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a6853c8-f362-4f68-a77e-0b304a194c05": EfiGuid(
        "1a6853c8-f362-4f68-a77e-0b304a194c05", "UseSocketDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a6f63f1-aead-45fc-882c-03885004a6d6": EfiGuid(
        "1a6f63f1-aead-45fc-882c-03885004a6d6", "DellAdvSysMgmtSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a6fd644-ccc2-4bf8-a41d-7f975be577ba": EfiGuid(
        "1a6fd644-ccc2-4bf8-a41d-7f975be577ba", "AplFlashUpdateAreas",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AplFlashUpdateAreas'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '633e80e7347dc01c21c5133227df2aba',
            '811505684bd99c18f5ecbee1f725ff66', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b6cce665f85751125c3db2050ffc98d2',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "1a763b74-1400-4245-bf89-56910fa7be48": EfiGuid(
        "1a763b74-1400-4245-bf89-56910fa7be48", "FjSystemInfo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a7beae8-0587-4eb5-b35d-d0d262201800": EfiGuid(
        "1a7beae8-0587-4eb5-b35d-d0d262201800", "DellAcpiResetProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a7e4468-2f55-4a56-903c-01265eb7622b": EfiGuid(
        "1a7e4468-2f55-4a56-903c-01265eb7622b", "LENOVO_TCP_DXE_BIN_GUID",
        description = "",
        aliases = ['TcpDxe'],
        urls = ['https://github.com/tianocore/edk2/blob/master/NetworkPkg/TcpDxe/TcpDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['LENOVO_TCP_DXE_BIN_GUID', 'TcpDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['TcpDxe']),

    "1a819e49-d8ee-48cb-9a9c-0aa0d2810a38": EfiGuid(
        "1a819e49-d8ee-48cb-9a9c-0aa0d2810a38", "DXE_PCH_POLICY_UPDATE_PROTOCOL_GUID",
        description = "",
        aliases = ['DxePchPolicyUpdateProtocolGuid'],
        urls = [],
        vendors = [],
        filenames = ['DXE_PCH_POLICY_UPDATE_PROTOCOL_GUID'],
        hashes = [
            '81460325c53ef849a2e3c2a5e104304b', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
            'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "1a8dc70e-1d6d-4e33-bf0a-7fe6cd38744e": EfiGuid(
        "1a8dc70e-1d6d-4e33-bf0a-7fe6cd38744e", "SiInitPch",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a8e9d96-66e6-461b-95d6-882c984d0b00": EfiGuid(
        "1a8e9d96-66e6-461b-95d6-882c984d0b00", "TbtPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0dc2a67fa77b54e8cedc419081e3117c'],
        junk = []),

    "1a926325-2764-47c8-9e1c-4f83b1723336": EfiGuid(
        "1a926325-2764-47c8-9e1c-4f83b1723336", "VirtualEcDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1a931fb8-c466-454a-b684-73ada2cb050b": EfiGuid(
        "1a931fb8-c466-454a-b684-73ada2cb050b", "AsusPostMessagePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aa34862-c1d3-49d1-9281-9f261545c09c": EfiGuid(
        "1aa34862-c1d3-49d1-9281-9f261545c09c", "FjSystemDataFvDriverSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aa6d900-89d1-4c21-9c50-edc7390a67c3": EfiGuid(
        "1aa6d900-89d1-4c21-9c50-edc7390a67c3", "SlotDataUpdateDxeNeonCityFPGA",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aa719f3-10a8-47bb-983f-3a03babbb1eb": EfiGuid(
        "1aa719f3-10a8-47bb-983f-3a03babbb1eb", "JGPIOCtrlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aaf6a9b-cb1b-469e-9420-0d3912b8e376": EfiGuid(
        "1aaf6a9b-cb1b-469e-9420-0d3912b8e376", "MrcHooksChipServicesPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ab12a1b-eebd-33f0-8b3c-a145b8be3ff0": EfiGuid(
        "1ab12a1b-eebd-33f0-8b3c-a145b8be3ff0", "AmiDtsPolicyBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ab9b2fc-7a50-4fa2-a190-eeb35797571d": EfiGuid(
        "1ab9b2fc-7a50-4fa2-a190-eeb35797571d", "OpromUpdateDxeSierra",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1abcb349-299b-46e8-99d1-582d91b3ab54": EfiGuid(
        "1abcb349-299b-46e8-99d1-582d91b3ab54", "RstEraseDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aced566-76ed-4218-bc81-767f1f977a89": EfiGuid(
        "1aced566-76ed-4218-bc81-767f1f977a89", "EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID_31",
        description = "",
        aliases = ['EfiNetworkInterfaceIdentifierProtocolGuid_31'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['1']),

    "1ad5b195-44d9-4917-9bae-e14784979306": EfiGuid(
        "1ad5b195-44d9-4917-9bae-e14784979306", "Fji210LanDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ae42876-008f-4161-b2b7-1c0d15c5ef43": EfiGuid(
        "1ae42876-008f-4161-b2b7-1c0d15c5ef43", "DefaultDataFile",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ae856c8-1c5a-45f0-8fc2-0093292db915": EfiGuid(
        "1ae856c8-1c5a-45f0-8fc2-0093292db915", "IteOnlySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aea4e7d-d6f8-491e-bb5c-4bec10c98c99": EfiGuid(
        "1aea4e7d-d6f8-491e-bb5c-4bec10c98c99", "FchSmmDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aec7edf-3287-4669-b114-33a1198a60c0": EfiGuid(
        "1aec7edf-3287-4669-b114-33a1198a60c0", "OemAcpiDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1aec89fc-db8d-48ef-a6cc-c87a0b63e934": EfiGuid(
        "1aec89fc-db8d-48ef-a6cc-c87a0b63e934", "XnoteSwSmiServiceSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1afe6bd0-c9c5-44d4-b7bd-8f5e7d0f2560": EfiGuid(
        "1afe6bd0-c9c5-44d4-b7bd-8f5e7d0f2560", "DELL_DIAGS_SB_CONTROL_SMM_GUID",
        description = "",
        aliases = ['DellDiagsSbControlSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b037083-5f15-4437-be09-6c68ac675b85": EfiGuid(
        "1b037083-5f15-4437-be09-6c68ac675b85", "MtkWiFiDxeGen2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b04374d-fa9c-420f-ac62-fee6d45e8443": EfiGuid(
        "1b04374d-fa9c-420f-ac62-fee6d45e8443", "FspDebugServicePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b05de41-c93b-4bb4-ad47-2a78ac0fc9e4": EfiGuid(
        "1b05de41-c93b-4bb4-ad47-2a78ac0fc9e4", "HSTI_PROTOCOL_GUID",
        description = "",
        aliases = ['HstiProtocolGuid'],
        urls = [],
        vendors = [],
        filenames = ['HSTI_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '869f701de5296248126c8c8a03a479c3',
            '953dc1131b7a579176b9fc937eee5846', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b49cea81c680d9848212ed2aca8e6771', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ee40c5722cf3b43608a71e2a12c72193',
        ],
        junk = []),

    "1b06c65d-466e-35c9-154d-4d91bc4b9f8f": EfiGuid(
        "1b06c65d-466e-35c9-154d-4d91bc4b9f8f", "AmdPspDtpmPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b06fa46-e65b-4257-8855-5828b7591e61": EfiGuid(
        "1b06fa46-e65b-4257-8855-5828b7591e61", "FingerPrintHDDPasswordChecking",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b08a1db-f91a-4fa1-a911-255c417f1cf7": EfiGuid(
        "1b08a1db-f91a-4fa1-a911-255c417f1cf7", "ServerMgmtSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b0fb9bf-699d-4fdd-a7c3-2546681bf63b": EfiGuid(
        "1b0fb9bf-699d-4fdd-a7c3-2546681bf63b", "EFI_WIRELESS_MAC_CONNECTION_II_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiWiFi2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/WiFi2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiWiFi2Protocol']),

    "1b1183fa-1823-46a7-8872-9c578755409d": EfiGuid(
        "1b1183fa-1823-46a7-8872-9c578755409d", "EFI_MM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_SMM_POWER_BUTTON_DISPATCH2_PROTOCOL_GUID', 'gEfiMmPowerButtonDispatchProtocolGuid', 'EfiSmmPowerButtonDispatch2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MmPowerButtonDispatch.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SmmPowerButtonDispatch2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_MM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9e80820994976fe9edc2b53346ea4f71', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'b218fecd176e986c1491046e58c9e01b', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EfiSmmPowerButtonDispatch2Protocol', 'EfiSmmPowerButtonDispatch2ProtocolGuid', 'EFI_SMM_POWER_BUTTON_DISPATCH2_PROTOCOL_GUID']),

    "1b1924c3-6321-484f-83ef-8894b21de258": EfiGuid(
        "1b1924c3-6321-484f-83ef-8894b21de258", "AmiPeiCrbInfoPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b2042f7-5ce0-4360-b98f-2973cae22b97": EfiGuid(
        "1b2042f7-5ce0-4360-b98f-2973cae22b97", "FjSysmanTeutatesSmmBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b25af84-1ea8-4b52-894e-bfa6880b97ff": EfiGuid(
        "1b25af84-1ea8-4b52-894e-bfa6880b97ff", "PcAtSerialPortLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/PcAtChipsetPkg/Library/SerialIoLib/SerialIoLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1b26c724-c7d4-4de6-b0b1-c622bc2937ff": EfiGuid(
        "1b26c724-c7d4-4de6-b0b1-c622bc2937ff", "SmartCoverPortingDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b290126-5760-424e-8aa2-3faf4d0d7978": EfiGuid(
        "1b290126-5760-424e-8aa2-3faf4d0d7978", "EmuGopDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Bhyve/BhyveRfbDxe/BhyveRfbDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1b2c4952-d778-4b64-bda1-15a36f5fa545": EfiGuid(
        "1b2c4952-d778-4b64-bda1-15a36f5fa545", "Slp20PubKey",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b31a273-217e-3377-23d1-ab0cf19b9d47": EfiGuid(
        "1b31a273-217e-3377-23d1-ab0cf19b9d47", "ChgbootDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b346098-ab0b-4ae7-ddbe-88ab3cd6220d": EfiGuid(
        "1b346098-ab0b-4ae7-ddbe-88ab3cd6220d", "BatteryIdm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b417388-b6b7-4026-b849-30060477a2ad": EfiGuid(
        "1b417388-b6b7-4026-b849-30060477a2ad", "BootDeviceOptionsWmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b45cc0a-156a-428a-62af-49864da0e6e6": EfiGuid(
        "1b45cc0a-156a-428a-62af-49864da0e6e6", "PEI_APRIORI_FILE_NAME_GUID",
        description = "",
        aliases = [],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/AprioriFileName.h",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1b45cc0a-156a-428a-af62-49864da0e6e6": EfiGuid(
        "1b45cc0a-156a-428a-af62-49864da0e6e6", "EFI_PEI_APRIORI_FILE_NAME_GUID",
        description = "",
        aliases = ['gPeiAprioriFileNameGuid', 'AprioriPei'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_APRIORI_FILE_NAME_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9e80820994976fe9edc2b53346ea4f71',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ee40c5722cf3b43608a71e2a12c72193', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['AprioriPei']),

    "1b49372d-cbbd-44d3-842c-8e900390da8c": EfiGuid(
        "1b49372d-cbbd-44d3-842c-8e900390da8c", "H19BatteryControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b496714-7685-402c-a5ba-37b8113d4e74": EfiGuid(
        "1b496714-7685-402c-a5ba-37b8113d4e74", "DellFreeFallSensorSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b4ae0f8-ed1f-4fd1-9b18-b082290f86f5": EfiGuid(
        "1b4ae0f8-ed1f-4fd1-9b18-b082290f86f5", "EfiPlatformTypeLightningRidgeExrpProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b52c630-86f6-4aee-9f0c-474dc6be49b6": EfiGuid(
        "1b52c630-86f6-4aee-9f0c-474dc6be49b6", "TB16_PC2_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b59ccda-7db2-4a55-afc2-4364f824d288": EfiGuid(
        "1b59ccda-7db2-4a55-afc2-4364f824d288", "McBankErrorInjection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b5b480a-2b45-48ae-aa0a-d82b92a7738b": EfiGuid(
        "1b5b480a-2b45-48ae-aa0a-d82b92a7738b", "AppAdapterSgx3v0",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b5c27fe-f01c-4fbc-aeae-341b2e992a17": EfiGuid(
        "1b5c27fe-f01c-4fbc-aeae-341b2e992a17", "FspSFirmwareFileSystemFvGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b6664cc-57b4-4e47-a477-2c9dd2413667": EfiGuid(
        "1b6664cc-57b4-4e47-a477-2c9dd2413667", "DellPchPtssTablePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b69bd70-9fad-4a0e-8a3e-dc15aa40fc56": EfiGuid(
        "1b69bd70-9fad-4a0e-8a3e-dc15aa40fc56", "LEMBIOSLock",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b6bc809-c986-4937-934f-1ea58622fe50": EfiGuid(
        "1b6bc809-c986-4937-934f-1ea58622fe50", "AMITSE_BOOT_ORDER_CHANGE_GUID",
        description = "",
        aliases = ['AmiTseBootOrderChangeGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b6e9d50-12e4-4b55-81d9-1abfec59d4fc": EfiGuid(
        "1b6e9d50-12e4-4b55-81d9-1abfec59d4fc", "ASROCK_WAKEUP_CTRL_SMM_GUID",
        description = "",
        aliases = ['WakeCtrlSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b76b808-b90a-42a4-b0a0-be61b121e114": EfiGuid(
        "1b76b808-b90a-42a4-b0a0-be61b121e114", "FjTpmMisc",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b799e86-28cc-4dbf-a763-f7f33ffd2772": EfiGuid(
        "1b799e86-28cc-4dbf-a763-f7f33ffd2772", "SmuV13Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b838190-4625-4ead-abc9-cd5e6af18fe0": EfiGuid(
        "1b838190-4625-4ead-abc9-cd5e6af18fe0", "EFI_HII_EXPORT_DATABASE_GUID",
        description = "",
        aliases = ['EfiHiiExportDatabase'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b852ce9-6bcb-4c9d-be8a-ba34fa8fbd77": EfiGuid(
        "1b852ce9-6bcb-4c9d-be8a-ba34fa8fbd77", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0fb0d5db7a0d624b39b616db11024c0a'],
        junk = []),

    "1b8b5f2b-605a-4d09-afda-842c3933a050": EfiGuid(
        "1b8b5f2b-605a-4d09-afda-842c3933a050", "AmiFchUart1Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b8ddea4-deb2-4152-91c2-b73cb16ce464": EfiGuid(
        "1b8ddea4-deb2-4152-91c2-b73cb16ce464", "PEI_ATA_POLICY_PPI_GUID",
        description = "",
        aliases = ['PeiAtaPolicyPpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1b9cd864-da7c-483e-8d69-d35cb4ad27c9": EfiGuid(
        "1b9cd864-da7c-483e-8d69-d35cb4ad27c9", "EfiWheaPlatformNonStandardErrorSection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ba0062e-c779-4582-8566-336ae8f78f09": EfiGuid(
        "1ba0062e-c779-4582-8566-336ae8f78f09", "EFI_FFS_VOLUME_TOP_FILE_GUID",
        description = "",
        aliases = ['ResetVector', 'SecCore', 'XenResetVector', 'gEfiFirmwareVolumeTopFileGuid', 'SecCore_VolumeTopFile'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/BaseTools/Source/C/Include/Guid/PiFirmwareFileSystem.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/FirmwareFileSystem2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Bhyve/ResetVector/ResetVector.inf",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/ResetVector/ResetVector.inf",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/XenResetVector/XenResetVector.inf",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/ResetVector/FixupVtf/Vtf.inf",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/ResetVector/Vtf0/Bin/ResetVector.inf",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/ResetVector/Vtf0/Vtf0.inf",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/SecCore/SecCore.inf",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_FFS_VOLUME_TOP_FILE_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EfiFirmwareVolumeTopFile', 'EfiFirmwareVolumeTopFileGuid', 'FspSecCore', 'FspWrapperSecCore', 'ResetVector', 'SecCore']),

    "1ba0ada4-526b-4f11-a9bd-3f83ee17ea30": EfiGuid(
        "1ba0ada4-526b-4f11-a9bd-3f83ee17ea30", "CopyRight",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ba41b68-465f-4b22-8406-2b82173dae50": EfiGuid(
        "1ba41b68-465f-4b22-8406-2b82173dae50", "DellSpecialBootControlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1bad4160-ce02-4bdd-9d04-dcf7d7b72e0f": EfiGuid(
        "1bad4160-ce02-4bdd-9d04-dcf7d7b72e0f", "IioErrorHandler",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1bad711c-d451-4241-b1f3-8537812e0c70": EfiGuid(
        "1bad711c-d451-4241-b1f3-8537812e0c70", "EFI_ME_BIOS_EXTENSION_SETUP_GUID",
        description = "",
        aliases = ['EfiMeBiosExtensionSetup'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1bb13967-8b24-411b-9828-18d285a9cc4d": EfiGuid(
        "1bb13967-8b24-411b-9828-18d285a9cc4d", "AmiPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1bb737ef-427a-4144-8b3b-b76ef38515e6": EfiGuid(
        "1bb737ef-427a-4144-8b3b-b76ef38515e6", "SdMmcPciHcPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Pci/SdMmcPciHcPei/SdMmcPciHcPei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1bbec6eb-f0c1-40a4-98bb-62cf3e87d0e0": EfiGuid(
        "1bbec6eb-f0c1-40a4-98bb-62cf3e87d0e0", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "1bd014a7-a164-4066-ab57-5ce39b505818": EfiGuid(
        "1bd014a7-a164-4066-ab57-5ce39b505818", "BtBus",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['BtBus'],
        hashes = [
            '2b4ca0bd16a20893deff9632c23c2cca', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b4fa3a413d748be6699112adc0c14b0d',
            'c0bdac2284b1e46c7c9c1509ddbea006',
        ],
        junk = []),

    "1be14579-d805-4c3b-8874-410b818674e9": EfiGuid(
        "1be14579-d805-4c3b-8874-410b818674e9", "RealtekPxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1be65202-9318-492d-a551-08df2bd60aee": EfiGuid(
        "1be65202-9318-492d-a551-08df2bd60aee", "DELL_AMT_PLATFORM_POLICY_GUID",
        description = "",
        aliases = ['AmtPlatformPolicy'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DELL_AMT_PLATFORM_POLICY_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '9a05d9b82e4f2c467423be3959175608',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'd293013d9f585157787fce8c228e8897',
            'ee8ffebc11a82e29543a142d51a8797d', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['AmtPlatformPolicy']),

    "1be9a477-92e6-4ba4-9496-c9de8e152534": EfiGuid(
        "1be9a477-92e6-4ba4-9496-c9de8e152534", "SbSocStarshipSp3Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1bf06aea-5bec-4a8d-9576-749b09562d30": EfiGuid(
        "1bf06aea-5bec-4a8d-9576-749b09562d30", "EFI_PROCESSOR_PRODUCER_GUID",
        description = "",
        aliases = ['ProcessorProducerGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['PROCESSOR_PRODUCER_GUID']),

    "1bfc532e-f48a-4ebe-b2fb-2b286d70a6eb": EfiGuid(
        "1bfc532e-f48a-4ebe-b2fb-2b286d70a6eb", "IconUsbHD",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c015629-00cf-473d-be1c-3561024f0569": EfiGuid(
        "1c015629-00cf-473d-be1c-3561024f0569", "AdlSemaDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c0c34f6-d380-41fa-a049-8ad06c1a66aa": EfiGuid(
        "1c0c34f6-d380-41fa-a049-8ad06c1a66aa", "EFI_EDID_DISCOVERED_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiEdidDiscoveredProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/EdidDiscovered.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1c0d835f-c0c5-4848-b42b-95fedbcc815c": EfiGuid(
        "1c0d835f-c0c5-4848-b42b-95fedbcc815c", "GnbBristolRouting",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c110d74-159b-46b3-b47e-7b7ad22fa6d6": EfiGuid(
        "1c110d74-159b-46b3-b47e-7b7ad22fa6d6", "DellResetReduction",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c132679-bb95-479c-8a82-6de72a52d6fd": EfiGuid(
        "1c132679-bb95-479c-8a82-6de72a52d6fd", "AaeonPowerButtonSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c14f9be-034f-478c-ad0a-d4d178c11629": EfiGuid(
        "1c14f9be-034f-478c-ad0a-d4d178c11629", "OemNetworkDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c178237-6897-459e-9d36-67ce8ef94f76": EfiGuid(
        "1c178237-6897-459e-9d36-67ce8ef94f76", "EFI_KMS_FORMAT_MD5SHA_128_GUID",
        description = "",
        aliases = ['EfiKmsFormatMd5sha128Guid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Kms.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1c1eab81-9229-42ac-b940-757c498a09ed": EfiGuid(
        "1c1eab81-9229-42ac-b940-757c498a09ed", "AmiTseOemPortingVar22",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c2b04a1-2c42-4d3b-d080-6a65335005e4": EfiGuid(
        "1c2b04a1-2c42-4d3b-d080-6a65335005e4", "EFI_SYSTEM_PART_CAPSULE_STORAGE_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c324c08-eb94-4715-9180-f5a453c57269": EfiGuid(
        "1c324c08-eb94-4715-9180-f5a453c57269", "OemGOPCallBack",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c33c958-fea4-435f-93ef-0a9c14ae87e4": EfiGuid(
        "1c33c958-fea4-435f-93ef-0a9c14ae87e4", "DELL_MFG_AUTHENTICATION_GUID",
        description = "",
        aliases = ['DellMfgAuthentication'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c377111-e07e-4b33-9b5b-30817d1f1be3": EfiGuid(
        "1c377111-e07e-4b33-9b5b-30817d1f1be3", "DellIntrusionDetectSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c3f1d99-4d35-4bbe-b75e-a2c92c7228da": EfiGuid(
        "1c3f1d99-4d35-4bbe-b75e-a2c92c7228da", "OemWwanSmBiosType133",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c4c501a-8cdc-4d1f-8639-27e9049494c1": EfiGuid(
        "1c4c501a-8cdc-4d1f-8639-27e9049494c1", "HpCommonAcpiArea",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c505528-32d7-4d70-8818-9a489b414aba": EfiGuid(
        "1c505528-32d7-4d70-8818-9a489b414aba", "PcieErrorLog",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c58aac6-76c9-d94d-a14c-0059e13b96a3": EfiGuid(
        "1c58aac6-76c9-d94d-a14c-0059e13b96a3", "AppleMtftp4Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c5c6e7e-552a-443e-9a04-7408adee99d3": EfiGuid(
        "1c5c6e7e-552a-443e-9a04-7408adee99d3", "FirmwareConfigDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c5fdaf2-9fb3-431b-8ecd-b7d35cbefae9": EfiGuid(
        "1c5fdaf2-9fb3-431b-8ecd-b7d35cbefae9", "EfiAfterPlatformLocksEvent",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c606e42-f267-4a2f-95ea-a081b62e3f4b": EfiGuid(
        "1c606e42-f267-4a2f-95ea-a081b62e3f4b", "H19CheckPointHddService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c675c27-6fb1-4170-94e7-3dffe8beadba": EfiGuid(
        "1c675c27-6fb1-4170-94e7-3dffe8beadba", "LaunchPad3Drv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c6a6dbe-ca59-44e4-8d42-68fac524a89c": EfiGuid(
        "1c6a6dbe-ca59-44e4-8d42-68fac524a89c", "SystemAudioDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c6b2faf-d8bd-44d1-a91e-7321b4c2f3d1": EfiGuid(
        "1c6b2faf-d8bd-44d1-a91e-7321b4c2f3d1", "LENOVO_SYSTEM_BOOT_SCRIPT_SAVE_DXE_GUID",
        description = "",
        aliases = ['ScriptSaveDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['ScriptSaveDxe']),

    "1c728baf-96b6-46e6-b549-e50c45898e3a": EfiGuid(
        "1c728baf-96b6-46e6-b549-e50c45898e3a", "TamperEventProtection",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c733cb5-4c8f-44d9-bbf0-95477089eb08": EfiGuid(
        "1c733cb5-4c8f-44d9-bbf0-95477089eb08", "DELL_SM_BIOS_STRUC_DA_GUID",
        description = "",
        aliases = ['DellSmBiosStrucDa'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c747f6b-0a58-49ae-8ea3-0327a4fa10e3": EfiGuid(
        "1c747f6b-0a58-49ae-8ea3-0327a4fa10e3", "PeiServicesTablePointerLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/PeiServicesTablePointerLib/PeiServicesTablePointerLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1c7b1fd9-373c-4569-8dc1-8db2fffa7639": EfiGuid(
        "1c7b1fd9-373c-4569-8dc1-8db2fffa7639", "NistDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c8020b7-d2f2-42f8-852b-cbc232ccc9bc": EfiGuid(
        "1c8020b7-d2f2-42f8-852b-cbc232ccc9bc", "AsusROGLiveServiceSupportDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c83550a-7e5b-4a30-9e93-e2153771aa25": EfiGuid(
        "1c83550a-7e5b-4a30-9e93-e2153771aa25", "FjDmiCpu",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c871d93-ba95-474d-ba2a-397ccf9d2691": EfiGuid(
        "1c871d93-ba95-474d-ba2a-397ccf9d2691", "OemTurnOnAmp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1c8b7f78-1699-40e6-af33-9b995d16b043": EfiGuid(
        "1c8b7f78-1699-40e6-af33-9b995d16b043", "PiSmmCommunicationPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/PiSmmCommunication/PiSmmCommunicationPei.inf'],
        vendors = ['EDK2'],
        filenames = ['PiSmmCommunicationPei'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0ebecbce655f6f780edb7d73c1d80d13',
            '0fb0d5db7a0d624b39b616db11024c0a', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '136ef7c1715a5ab03a93c66f1f9447e1', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '3555862f32138b911338fae3789c3921',
            '35a55f8ce610ca18a6d9480a0fc484df', '39a2ecdc1427b5f2c143fd59ef976f2c',
            '3dbf73930867371066435e0908e50535', '3fa6831c92f133013c5bb0942f307f72',
            '4002d2fb07cfbaed2e86d9608824378a', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '6f7959935a87f450b05674b1e62ba7fa', '73c9d09abcd1f4bf7ddbfe437773286e',
            '7457f65a90b756398ae360b93c326a5e', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d13012b3ccc7c61a0c88ca8969676f6', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9e80820994976fe9edc2b53346ea4f71', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'b218fecd176e986c1491046e58c9e01b', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1c98780a-c67d-4d9b-a9d8-4ac0487a6d6e": EfiGuid(
        "1c98780a-c67d-4d9b-a9d8-4ac0487a6d6e", "PcdRecoveryPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PcdRecoveryPei'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '15c3afd565e35a7b87854f203ea15633', '1ded807f42af76804c9a0bbb6e24a2a9',
            '20f1da0b4063ef2319439a473c0d4ae3', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'cae37d13982bb26c35ffaada2841d171', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'e7a706c2421a55cde70e476d1d37f831', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1ca0e202-fe9e-4776-9faa-570c19617a06": EfiGuid(
        "1ca0e202-fe9e-4776-9faa-570c19617a06", "EfiPciCallbackProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ca5f829-068c-4e7a-b069-16d6d39865ae": EfiGuid(
        "1ca5f829-068c-4e7a-b069-16d6d39865ae", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['9c78983f2bf3fb5dfbf9652704b6322f'],
        junk = []),

    "1ca7e5f3-2385-427d-a4af-f6e28bb8bbec": EfiGuid(
        "1ca7e5f3-2385-427d-a4af-f6e28bb8bbec", "AmdMemAm5Pei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cae7577-d662-4727-baec-777631d3a958": EfiGuid(
        "1cae7577-d662-4727-baec-777631d3a958", "SystemVspCmosPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cb1adda-d8dd-4c6e-8def-a0dacb2aec83": EfiGuid(
        "1cb1adda-d8dd-4c6e-8def-a0dacb2aec83", "RetimerCapsuleUpdate",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cbc61dc-3d35-41ed-8e92-9147a0d68e77": EfiGuid(
        "1cbc61dc-3d35-41ed-8e92-9147a0d68e77", "AmdMemFp8StxDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cbed347-7de6-bc48-ac68-3758598124d2": EfiGuid(
        "1cbed347-7de6-bc48-ac68-3758598124d2", "PeiEmuPeCoffGetEntryPointLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmulatorPkg/Library/PeiEmuPeCoffGetEntryPointLib/PeiEmuPeCoffGetEntryPointLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1cbfc9b6-7f1e-4e9e-801f-fbce4d92a76d": EfiGuid(
        "1cbfc9b6-7f1e-4e9e-801f-fbce4d92a76d", "AmiTseBeforeTimeOutGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cc640e2-28ee-4d0c-8211-5d0e30967ee6": EfiGuid(
        "1cc640e2-28ee-4d0c-8211-5d0e30967ee6", "DellRecoveryMgmtDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cc6fa21-4a9d-46b4-9ebf-9e42f5d044f2": EfiGuid(
        "1cc6fa21-4a9d-46b4-9ebf-9e42f5d044f2", "FjCapsuleResetHookSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cde0a03-d14e-458b-9ff7-a8ea2750cf68": EfiGuid(
        "1cde0a03-d14e-458b-9ff7-a8ea2750cf68", "IntelGopVbt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['IntelGopVbt'],
        hashes = [
            '2b4ca0bd16a20893deff9632c23c2cca', '48bba53caf662f85dd73a1a247b45dc7',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'c0bdac2284b1e46c7c9c1509ddbea006',
        ],
        junk = []),

    "1ce12314-afbc-11f0-8a3e-ab44b8ee3120": EfiGuid(
        "1ce12314-afbc-11f0-8a3e-ab44b8ee3120", "PpmPolicyInitDxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['PpmPolicyInitDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cfba6c85608b8135fb21024783573bf6', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "1ce1e5bc-7ceb-42f2-81e5-8aadf180f57b": EfiGuid(
        "1ce1e5bc-7ceb-42f2-81e5-8aadf180f57b", "LINUX_EFI_RANDOM_SEED_TABLE_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/torvalds/linux/blob/master/include/linux/efi.h'],
        vendors = ['Linux'],
        filenames = [],
        hashes = [],
        junk = []),

    "1ce34cff-dfe9-4d9d-b264-166ffcb8315a": EfiGuid(
        "1ce34cff-dfe9-4d9d-b264-166ffcb8315a", "MebxSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ce692f0-3317-4574-8b57-4800ed091f40": EfiGuid(
        "1ce692f0-3317-4574-8b57-4800ed091f40", "DELL_DXE_COMPUTRACE_GUID",
        description = "",
        aliases = ['DellDxeComputrace'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ce76f5c-3b9d-4a4c-904a-733172eb5301": EfiGuid(
        "1ce76f5c-3b9d-4a4c-904a-733172eb5301", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['01f2619337409777ae00804717c4dabc'],
        junk = []),

    "1ce7c25e-02e0-4cb8-9410-766eee8b8a97": EfiGuid(
        "1ce7c25e-02e0-4cb8-9410-766eee8b8a97", "FjMfgTpmDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ce9b08d-7649-4ee6-aef5-729a4da7e469": EfiGuid(
        "1ce9b08d-7649-4ee6-aef5-729a4da7e469", "EcIoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cebdd17-1325-42bb-8a7a-22857cf10a5e": EfiGuid(
        "1cebdd17-1325-42bb-8a7a-22857cf10a5e", "AmdFabricRsDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cec16ec-1c60-449b-9775-23ca66b93f52": EfiGuid(
        "1cec16ec-1c60-449b-9775-23ca66b93f52", "AmdBoardIdPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cee181a-df5c-4391-bca8-cde5a5a1a0ba": EfiGuid(
        "1cee181a-df5c-4391-bca8-cde5a5a1a0ba", "OemHooks",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cf369c2-3c8f-4d75-aaed-45cfeb8e9b6c": EfiGuid(
        "1cf369c2-3c8f-4d75-aaed-45cfeb8e9b6c", "UsbPortDisable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cf3f8b3-c5b1-49a2-aa59-5eef92ffa63c": EfiGuid(
        "1cf3f8b3-c5b1-49a2-aa59-5eef92ffa63c", "EFI_IA32_X64_ERROR_TYPE_BUS_CHECK_GUID",
        description = "",
        aliases = ['EfiIa32X64ErrorTypeBusCheckGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Guid/Cper.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1cf40d19-eead-4c73-93db-bbb8b6acf929": EfiGuid(
        "1cf40d19-eead-4c73-93db-bbb8b6acf929", "UserIdentificationDxe",
        description = "",
        aliases = ['SystemNUserUserManagerDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cf59bf9-e3f6-4d9d-bb94-8b857278bbc6": EfiGuid(
        "1cf59bf9-e3f6-4d9d-bb94-8b857278bbc6", "GnbDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1cf6c0c8-fa85-43ee-94af-db27c54174c9": EfiGuid(
        "1cf6c0c8-fa85-43ee-94af-db27c54174c9", "RAIDCorePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d000ae9-756b-4937-b736-1f3d96a5a8f0": EfiGuid(
        "1d000ae9-756b-4937-b736-1f3d96a5a8f0", "Dptf",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['Dptf'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '0fb0d5db7a0d624b39b616db11024c0a',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2b4ca0bd16a20893deff9632c23c2cca', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '633e80e7347dc01c21c5133227df2aba',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dab68811108b555173f8f096c9737d20',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "1d005f4c-4dfa-41b5-ab0c-be91fe121459": EfiGuid(
        "1d005f4c-4dfa-41b5-ab0c-be91fe121459", "BaseLibUnitTestsHost",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Test/UnitTest/Library/BaseLib/BaseLibUnitTestsHost.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1d0cfb9b-eb00-43a6-819c-d218df8dc4b4": EfiGuid(
        "1d0cfb9b-eb00-43a6-819c-d218df8dc4b4", "BootMode",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d12b73e-7dde-4910-93c5-2bdb041c6849": EfiGuid(
        "1d12b73e-7dde-4910-93c5-2bdb041c6849", "FjIbvBiosPasswordAbstractionDxeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d154ae4-5bdb-4341-90e2-32531679cf14": EfiGuid(
        "1d154ae4-5bdb-4341-90e2-32531679cf14", "TypeCEventMediator",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d201235-2f40-4fbc-8650-8502092d62ab": EfiGuid(
        "1d201235-2f40-4fbc-8650-8502092d62ab", "LENOVO_EAIA_DXE_GUID",
        description = "",
        aliases = ['LenovoEaiaDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d202cab-c8ab-4d5c-94f7-3cfcc0d3d335": EfiGuid(
        "1d202cab-c8ab-4d5c-94f7-3cfcc0d3d335", "EFI_SMM_CPU_SERVICE_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmmCpuServiceProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/Include/Protocol/SmmCpuService.h",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/UefiCpuPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_SMM_CPU_SERVICE_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '869f701de5296248126c8c8a03a479c3',
            '953dc1131b7a579176b9fc937eee5846', 'b49cea81c680d9848212ed2aca8e6771',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189',
        ],
        junk = []),

    "1d26adc3-b011-ee2c-2177-89bbaacc3392": EfiGuid(
        "1d26adc3-b011-ee2c-2177-89bbaacc3392", "AMI_BEFORE_CPU_RC_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiBeforeCpuRcProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d292105-2653-11e5-b46f-b8e8562cbafa": EfiGuid(
        "1d292105-2653-11e5-b46f-b8e8562cbafa", "AppleCapsuleRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d3de7f0-0807-424f-aa69-11a54e19a46f": EfiGuid(
        "1d3de7f0-0807-424f-aa69-11a54e19a46f", "EFI_ATA_PASS_THRU_PROTOCOL_GUID",
        description = "",
        aliases = ['gEfiAtaPassThruProtocolGuid', 'EfiExtScsiPassThruProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/AtaPassThru.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiExtScsiPassThruProtocol']),

    "1d3e9cb8-43af-490b-830a-3516aa532047": EfiGuid(
        "1d3e9cb8-43af-490b-830a-3516aa532047", "EDKII_FAULT_TOLERANT_WRITE_GUID",
        description = "",
        aliases = ['EdkiiFaultTolerantWriteGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/FaultTolerantWrite.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1d4b3c7b-7d01-4cff-af3a-0dafb53f6321": EfiGuid(
        "1d4b3c7b-7d01-4cff-af3a-0dafb53f6321", "FjGabiSystemDataLogoAbstraction",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d4b3c7b-7d01-4cff-af3a-0dafb53f6322": EfiGuid(
        "1d4b3c7b-7d01-4cff-af3a-0dafb53f6322", "FjGabiSystemDataLogoAbstractionFTS",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d564ec9-9373-49a4-9e3f-e4d7b9974c84": EfiGuid(
        "1d564ec9-9373-49a4-9e3f-e4d7b9974c84", "DxeDebugPrintErrorLevelLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/DxeDebugPrintErrorLevelLib/DxeDebugPrintErrorLevelLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1d57b5d5-bab4-4d2b-b7eb-0eb41d7b189c": EfiGuid(
        "1d57b5d5-bab4-4d2b-b7eb-0eb41d7b189c", "LibGlob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d58e9fd-ba80-4a47-8190-08ae9bb8a0ed": EfiGuid(
        "1d58e9fd-ba80-4a47-8190-08ae9bb8a0ed", "DellTagsPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d677a58-c753-4af1-b552-efe142df8f57": EfiGuid(
        "1d677a58-c753-4af1-b552-efe142df8f57", "QemuBootOrderLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/QemuBootOrderLib/QemuBootOrderLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1d6f730f-5a55-4078-869b-e0a18324bdc8": EfiGuid(
        "1d6f730f-5a55-4078-869b-e0a18324bdc8", "TemplateSec",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d6f853e-0006-40d8-9b4b-79618a5733b0": EfiGuid(
        "1d6f853e-0006-40d8-9b4b-79618a5733b0", "AmiTseOemPortingVar3Guid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d6fe4a5-8e70-4d10-84cd-b1d33214f9b5": EfiGuid(
        "1d6fe4a5-8e70-4d10-84cd-b1d33214f9b5", "HpPopupsSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d7add6e-b2da-4b0b-b29f-49cb42f46356": EfiGuid(
        "1d7add6e-b2da-4b0b-b29f-49cb42f46356", "EFI_MEMORY_PRODUCER_GUID",
        description = "",
        aliases = ['EfiMemoryProducer'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d82219f-386f-4f26-afac-613f242c0760": EfiGuid(
        "1d82219f-386f-4f26-afac-613f242c0760", "DELL_SMM_STATUS_CODE_GUID",
        description = "",
        aliases = ['DellSmmStatusCode'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1d85cd7f-f43d-11d2-9a0c-0090273fc14d": EfiGuid(
        "1d85cd7f-f43d-11d2-9a0c-0090273fc14d", "EFI_UNICODE_COLLATION_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiUnicodeCollationProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/UnicodeCollation.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1d88c542-9df7-424a-aa90-02b61f286938": EfiGuid(
        "1d88c542-9df7-424a-aa90-02b61f286938", "DELL_WDT_PEI_GUID",
        description = "",
        aliases = ['WdtPei'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DELL_WDT_PEI_GUID', 'WdtPei'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '3fa6831c92f133013c5bb0942f307f72', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['WdtPei']),

    "1da02c3c-62e5-438b-b0db-9e5128379661": EfiGuid(
        "1da02c3c-62e5-438b-b0db-9e5128379661", "EarlySmmDevices",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1da2723f-52df-432a-8d03-6e8fa8acc107": EfiGuid(
        "1da2723f-52df-432a-8d03-6e8fa8acc107", "PxeBcPcdProducerLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/PxeBcPcdProducerLib/PxeBcPcdProducerLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1da353a3-6400-4241-9ab0-e3e65c690ef7": EfiGuid(
        "1da353a3-6400-4241-9ab0-e3e65c690ef7", "IchSmbusArpDisabled",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1da69afd-c297-4ee2-a086-0a41eb2b4dc8": EfiGuid(
        "1da69afd-c297-4ee2-a086-0a41eb2b4dc8", "XnoteGopVbtPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1da97072-bddc-4b30-99f1-72a0b56fff2a": EfiGuid(
        "1da97072-bddc-4b30-99f1-72a0b56fff2a", "EFI_MONOTONIC_COUNTER_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiMonotonicCounterArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MonotonicCounter.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_MONOTONIC_COUNTER_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['EFI_MONTONIC_COUNTER_ARCH_PROTOCOL_GUID']),

    "1dace8ee-cd97-491e-8a0c-305d6437323c": EfiGuid(
        "1dace8ee-cd97-491e-8a0c-305d6437323c", "AsusOnBoardDeviceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1db184ae-81f5-4e72-8544-2bab0c2cac5c": EfiGuid(
        "1db184ae-81f5-4e72-8544-2bab0c2cac5c", "AMI_BBS_DEVICE_PATH_GUID",
        description = "",
        aliases = ['AmiBbsDevicePath'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1db29ae0-9dcb-43bc-8d87-5da14964dde2": EfiGuid(
        "1db29ae0-9dcb-43bc-8d87-5da14964dde2", "EFI_USER_INFO_ACCESS_SETUP_NORMAL_GUID",
        description = "",
        aliases = ['EfiUserInfoAccessSetupNormalGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/UserManager.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1db43ec9-df5f-4cf5-aaf0-0e85db4e149a": EfiGuid(
        "1db43ec9-df5f-4cf5-aaf0-0e85db4e149a", "BootGuardDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['BootGuardDxe'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '953dc1131b7a579176b9fc937eee5846',
            'b49cea81c680d9848212ed2aca8e6771', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
        ],
        junk = []),

    "1dbd1503-0a60-4230-aaa3-8016d8c3de2f": EfiGuid(
        "1dbd1503-0a60-4230-aaa3-8016d8c3de2f", "EfiSmmIpmiTransportProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dcfbaca-6ada-4c0d-86ed-af658bdfec0c": EfiGuid(
        "1dcfbaca-6ada-4c0d-86ed-af658bdfec0c", "AmiPlatformToDriverAgentProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dcff17c-aa53-4b78-b234-864027555035": EfiGuid(
        "1dcff17c-aa53-4b78-b234-864027555035", "LibUefi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dd1d619-f9b8-463e-8681-d1dc7c07b72c": EfiGuid(
        "1dd1d619-f9b8-463e-8681-d1dc7c07b72c", "EDKII_NON_DISCOVERABLE_SDHCI_DEVICE_GUID",
        description = "",
        aliases = ['EdkiiNonDiscoverableSdhciDeviceGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/NonDiscoverableDevice.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1dda5978-b29a-4ea7-aefb-8b0baa982e22": EfiGuid(
        "1dda5978-b29a-4ea7-aefb-8b0baa982e22", "ExReportStatusCodeRouterPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ddbb0e2-6fb5-4b1f-a14c-f2e5b5e59168": EfiGuid(
        "1ddbb0e2-6fb5-4b1f-a14c-f2e5b5e59168", "aqnicdxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ddbfd6a-3423-462f-9150-a7ffa66ff0ca": EfiGuid(
        "1ddbfd6a-3423-462f-9150-a7ffa66ff0ca", "StaticSkuDataDxeLightningRidgeEXRP",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1de0b8c2-ffb6-4bdf-97f5-0ffb33979038": EfiGuid(
        "1de0b8c2-ffb6-4bdf-97f5-0ffb33979038", "BaseReportStatusCodeLibNull",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseReportStatusCodeLibNull/BaseReportStatusCodeLibNull.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1de25879-6e2a-4d72-a768-288ccb9fa719": EfiGuid(
        "1de25879-6e2a-4d72-a768-288ccb9fa719", "EfiMemoryConfigDataHob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1de2ab60-eb98-4d6b-9dce-a7ef2202b3ae": EfiGuid(
        "1de2ab60-eb98-4d6b-9dce-a7ef2202b3ae", "AmdAgesaParameterGroupPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1de64b8e-138b-4258-b7dd-f2d8ec142a9e": EfiGuid(
        "1de64b8e-138b-4258-b7dd-f2d8ec142a9e", "AmiFwUpdateBmc",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1de6ef1d-db01-4c27-ad29-c582887e2e18": EfiGuid(
        "1de6ef1d-db01-4c27-ad29-c582887e2e18", "OemSataInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1df18da0-a18b-11df-8c3a-0002a5d5c51b": EfiGuid(
        "1df18da0-a18b-11df-8c3a-0002a5d5c51b", "SataSiI3132",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dfb7bfa-bf8e-4d11-9766-2fb0d0442310": EfiGuid(
        "1dfb7bfa-bf8e-4d11-9766-2fb0d0442310", "ASROCK_AMI_AGESA_DXE_GUID",
        description = "",
        aliases = ['AmiAgesaDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dfb7bfa-bf8e-4d12-9766-b0d014120415": EfiGuid(
        "1dfb7bfa-bf8e-4d12-9766-b0d014120415", "UpdateHwSignature",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dfc7675-cb00-4bf8-b8f0-e1c993814d2e": EfiGuid(
        "1dfc7675-cb00-4bf8-b8f0-e1c993814d2e", "FjWirelessLanDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1dffe9f3-7b5f-4b44-8ebd-39a739eba903": EfiGuid(
        "1dffe9f3-7b5f-4b44-8ebd-39a739eba903", "AcpiPlatform",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['AcpiPlatform'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '409221b30602fc13c8cbedf952c60297', '633e80e7347dc01c21c5133227df2aba',
            '811505684bd99c18f5ecbee1f725ff66', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a9258521b7c65199bd268280867cc628', 'b6cce665f85751125c3db2050ffc98d2',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "1e00830b-8ba5-4d24-8609-e74994e13a39": EfiGuid(
        "1e00830b-8ba5-4d24-8609-e74994e13a39", "Common_LilyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e107fe3-565d-4ae0-94b5-563d9f96bfc8": EfiGuid(
        "1e107fe3-565d-4ae0-94b5-563d9f96bfc8", "EfiSmcInBandLoadDefault",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e1cac3d-8460-4eda-9791-6d1b2da75e21": EfiGuid(
        "1e1cac3d-8460-4eda-9791-6d1b2da75e21", "NvmeSmartFeature",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e21c326-0d40-40f2-9f02-3a9b7838aa6b": EfiGuid(
        "1e21c326-0d40-40f2-9f02-3a9b7838aa6b", "AmiIpmiPkgTokenSpace",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e27d461-78f3-4f7d-b1c2-f72384f13a6e": EfiGuid(
        "1e27d461-78f3-4f7d-b1c2-f72384f13a6e", "VirtualRealTimeClockLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Library/VirtualRealTimeClockLib/VirtualRealTimeClockLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1e28dc28-d461-4c36-93f0-69f2fbe021bc": EfiGuid(
        "1e28dc28-d461-4c36-93f0-69f2fbe021bc", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['2949649fbed4688a06e89bf439cea6a1'],
        junk = []),

    "1e2acc41-e26a-483d-afc7-a056c34e087b": EfiGuid(
        "1e2acc41-e26a-483d-afc7-a056c34e087b", "EFI_PLATFORM_INFO_GUID",
        description = "",
        aliases = ['EfiPlatformInfoGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e2acc41-e26a-483d-afc7-a056c34e087c": EfiGuid(
        "1e2acc41-e26a-483d-afc7-a056c34e087c", "EfiNorthPeakGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e2ed096-30e2-4254-bd89-863bbef82325": EfiGuid(
        "1e2ed096-30e2-4254-bd89-863bbef82325", "EFI_TCG2_FINAL_EVENTS_TABLE_GUID",
        description = "",
        aliases = ['LINUX_EFI_TPM_FINAL_LOG_GUID', 'EfiTcg2FinalEventsTableGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Tcg2Protocol.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://github.com/torvalds/linux/blob/master/include/linux/efi.h",
        ],
        vendors = ['EDK2', 'Linux'],
        filenames = [],
        hashes = [],
        junk = []),

    "1e30e33d-1854-437a-bd68-fc1553aa8be4": EfiGuid(
        "1e30e33d-1854-437a-bd68-fc1553aa8be4", "CseEmmcSelectPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e321458-2cf2-4afd-8102-c86423ecd18d": EfiGuid(
        "1e321458-2cf2-4afd-8102-c86423ecd18d", "FjMiniCardModePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e43298f-3478-41a7-b577-86064635c728": EfiGuid(
        "1e43298f-3478-41a7-b577-86064635c728", "OptionRomPkgTokenSpaceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e463f9d-4cfb-4396-a784-68c350badcf1": EfiGuid(
        "1e463f9d-4cfb-4396-a784-68c350badcf1", "AmdSpiHcProtocolDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e469095-efc8-4147-97db-4d68b727e2e0": EfiGuid(
        "1e469095-efc8-4147-97db-4d68b727e2e0", "FwBlockService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e4eaab1-e637-443e-a5d6-56e60d97c619": EfiGuid(
        "1e4eaab1-e637-443e-a5d6-56e60d97c619", "UsbComboPeimPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e519311-3e75-4208-b05e-ebe17e3ff41f": EfiGuid(
        "1e519311-3e75-4208-b05e-ebe17e3ff41f", "ACPI_NOTIFY_INC_BRIGHTNESS_HOTKEY_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e5668e2-8481-11d4-bcf1-0080c73c8881": EfiGuid(
        "1e5668e2-8481-11d4-bcf1-0080c73c8881", "EFI_VARIABLE_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiVariableArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Variable.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_VARIABLE_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1e6ebc58-55e2-4fe4-a396-aa2c228ee7e2": EfiGuid(
        "1e6ebc58-55e2-4fe4-a396-aa2c228ee7e2", "PhPlatformSiSmmCodeCheckSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e753e16-dcef-47d0-9a38-7adecdb983ed": EfiGuid(
        "1e753e16-dcef-47d0-9a38-7adecdb983ed", "TcmLegX16FileGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e75e77f-8a15-4653-964d-542c157ef40a": EfiGuid(
        "1e75e77f-8a15-4653-964d-542c157ef40a", "SgPeiPolicyInit",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SgPeiPolicyInit'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'cbd75246fd41b231527edfcacc8d5803',
            'cc80253042928592caf3fafdd1915db0', 'cfba6c85608b8135fb21024783573bf6',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "1e77550e-e429-4fa9-bebf-b515acf7d919": EfiGuid(
        "1e77550e-e429-4fa9-bebf-b515acf7d919", "Ip6",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e82b556-4eb6-479f-955a-78a5fe0c0cb2": EfiGuid(
        "1e82b556-4eb6-479f-955a-78a5fe0c0cb2", "BroadcomWirelessDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e843ad6-e237-42fc-bda2-de78542e16dd": EfiGuid(
        "1e843ad6-e237-42fc-bda2-de78542e16dd", "SstSpiFlash",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e8d6bad-a79c-4696-878a-11471a7bac1c": EfiGuid(
        "1e8d6bad-a79c-4696-878a-11471a7bac1c", "OCDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['OCDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '20f1da0b4063ef2319439a473c0d4ae3',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '4002d2fb07cfbaed2e86d9608824378a', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '54a655d3479750be95c150698a9e80ff',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '86c4a8e759a9aa0eee18a5e7d480583f',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1e8dcce9-341c-4219-a790-f14c28b51ce4": EfiGuid(
        "1e8dcce9-341c-4219-a790-f14c28b51ce4", "L850Id0WAPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e8dcce9-341c-4219-a790-f14c28b51cee": EfiGuid(
        "1e8dcce9-341c-4219-a790-f14c28b51cee", "LENOVO_EAIA_PEI_GUID",
        description = "",
        aliases = ['LenovoEaiaPei'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e910ff4-3106-44f0-bb75-115a52fb9bac": EfiGuid(
        "1e910ff4-3106-44f0-bb75-115a52fb9bac", "DellVideoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e93e633-d65a-459e-ab84-93d9ec266d18": EfiGuid(
        "1e93e633-d65a-459e-ab84-93d9ec266d18", "EFI_TAPE_IO_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTapeIoProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/TapeIo.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1e93f29b-a3b2-f340-a605-de31ee3da031": EfiGuid(
        "1e93f29b-a3b2-f340-a605-de31ee3da031", "EdkTerminal",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e94f097-5acd-4089-b2e3-b9a5c879a70c": EfiGuid(
        "1e94f097-5acd-4089-b2e3-b9a5c879a70c", "ME_DATA_HOB_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e97097a-c884-4be4-a530-299f85bdc894": EfiGuid(
        "1e97097a-c884-4be4-a530-299f85bdc894", "MediaSanitize",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1e97e5f2-f850-0fd9-8191-c18fe43b0405": EfiGuid(
        "1e97e5f2-f850-0fd9-8191-c18fe43b0405", "WnameDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ea09f16-1a28-435b-8e8e-8ab7c1de04d3": EfiGuid(
        "1ea09f16-1a28-435b-8e8e-8ab7c1de04d3", "FjHiiString",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ea81bec-f01a-4d98-a201-4a61ce2fc022": EfiGuid(
        "1ea81bec-f01a-4d98-a201-4a61ce2fc022", "PERFORMANCE_EX_PROTOCOL_GUID",
        description = "",
        aliases = ['PerformanceExProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Guid/Performance.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1eaf8a37-cd36-4267-831e-ac69789c5cca": EfiGuid(
        "1eaf8a37-cd36-4267-831e-ac69789c5cca", "mSupplicantDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1eb667b3-3dd6-4f19-ba60-7c4638ac4062": EfiGuid(
        "1eb667b3-3dd6-4f19-ba60-7c4638ac4062", "AcerPortingDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ebe57f5-be42-45f0-a1f9-fa3dc633910b": EfiGuid(
        "1ebe57f5-be42-45f0-a1f9-fa3dc633910b", "PeiCrc32GuidedSectionExtractLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/PeiCrc32GuidedSectionExtractLib/PeiCrc32GuidedSectionExtractLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1ebe5ab9-2129-49e7-84d7-eeb9fce5dedd": EfiGuid(
        "1ebe5ab9-2129-49e7-84d7-eeb9fce5dedd", "EFI_EMMC_CARD_INFO_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiEmmcCardInfoProtocolGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ec0efc9-c93a-4b62-9b27-c059abd80e92": EfiGuid(
        "1ec0efc9-c93a-4b62-9b27-c059abd80e92", "VlvPlatformInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ec0f53a-fde0-4576-8f25-7a1a410f58eb": EfiGuid(
        "1ec0f53a-fde0-4576-8f25-7a1a410f58eb", "StatusCodePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ed1d253-e62b-47c2-acc3-50b859cbd511": EfiGuid(
        "1ed1d253-e62b-47c2-acc3-50b859cbd511", "Armani_BatteryHealthControlDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ed2d6f1-a9d2-476e-8e74-add95b051082": EfiGuid(
        "1ed2d6f1-a9d2-476e-8e74-add95b051082", "PCIE_DXE_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ed46147-76e3-4171-bce5-42d368d26aa6": EfiGuid(
        "1ed46147-76e3-4171-bce5-42d368d26aa6", "DebugPortTableDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1edc318f-4005-488d-af3a-9bb5179bc6f1": EfiGuid(
        "1edc318f-4005-488d-af3a-9bb5179bc6f1", "GmchMbiDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1edcbdf9-ffc6-4bd4-94f6-195d1de17056": EfiGuid(
        "1edcbdf9-ffc6-4bd4-94f6-195d1de17056", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "1edd13e6-d0cd-4432-a692-ff65c9b4f039": EfiGuid(
        "1edd13e6-d0cd-4432-a692-ff65c9b4f039", "SmmPerformanceLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/SmmPerformanceLib/SmmPerformanceLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1ee0e496-28b4-440b-a45a-7464bf4b32a2": EfiGuid(
        "1ee0e496-28b4-440b-a45a-7464bf4b32a2", "FjUsbSecurityV",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1eea2bfe-01cb-40cc-a34e-cb224c800aa2": EfiGuid(
        "1eea2bfe-01cb-40cc-a34e-cb224c800aa2", "Tpm2DeviceLibSeC",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ef02cf1-bf69-49c3-8fdc-5bb55d224cfa": EfiGuid(
        "1ef02cf1-bf69-49c3-8fdc-5bb55d224cfa", "DELL_POWER_BUTTON_CONFIG_GUID",
        description = "",
        aliases = ['DellPowerButtonConfig'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ef08355-4136-443d-be11-b4cb99524d80": EfiGuid(
        "1ef08355-4136-443d-be11-b4cb99524d80", "HpWsCommonSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ef46a7b-5f4d-4967-b3c4-a308d898eadd": EfiGuid(
        "1ef46a7b-5f4d-4967-b3c4-a308d898eadd", "AsusCalibrationDxeEntry",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f05933d-ff2e-4236-be92-56682d3e8fb4": EfiGuid(
        "1f05933d-ff2e-4236-be92-56682d3e8fb4", "FdCapsule",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f08f001-ac83-43d3-996b-47fae153cb2a": EfiGuid(
        "1f08f001-ac83-43d3-996b-47fae153cb2a", "DisableAbtSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f0f049e-3a68-4c97-865a-bc5eed7920e7": EfiGuid(
        "1f0f049e-3a68-4c97-865a-bc5eed7920e7", "AMI_PEI_NB_CPU_ONLY_RESET_PPI_GUID",
        description = "",
        aliases = ['AmiPeiNbCpuOnlyResetPpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f1829f7-54d7-4a08-94d3-8db412782531": EfiGuid(
        "1f1829f7-54d7-4a08-94d3-8db412782531", "SioPS2PowerOn",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f18c5b3-29ed-4d9e-a504-6d978e7ed569": EfiGuid(
        "1f18c5b3-29ed-4d9e-a504-6d978e7ed569", "QncS3CodeInLockBox",
        description = "",
        aliases = ['QncS3CodeInLockBoxGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f1dfc5b-5c64-49a3-a14a-ac2d3138b595": EfiGuid(
        "1f1dfc5b-5c64-49a3-a14a-ac2d3138b595", "FjIbvAbstractionDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f202fec-dda1-4bbf-8584-bd42e379ce99": EfiGuid(
        "1f202fec-dda1-4bbf-8584-bd42e379ce99", "DellDiagDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f21d964-807d-49f4-ae4e-f52663d1a16f": EfiGuid(
        "1f21d964-807d-49f4-ae4e-f52663d1a16f", "FjCryptoDESDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f28b56e-d26a-48fd-9fac-9e18a2c5bbdb": EfiGuid(
        "1f28b56e-d26a-48fd-9fac-9e18a2c5bbdb", "CnvUefiVariables",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['CnvUefiVariables'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '60785be6bfb0d1daa94a8bb214b774d1', '7a20bc8c8925db2963205a36be03b335',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'eee15c9794a4040563b41004ebccc91a',
        ],
        junk = []),

    "1f2ccb4f-d817-404e-98e7-80e4851fb33e": EfiGuid(
        "1f2ccb4f-d817-404e-98e7-80e4851fb33e", "GdbStub",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/GdbStub/GdbStub.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1f2d63e1-febd-4dc7-9cc5-ba2b1cef9c5b": EfiGuid(
        "1f2d63e1-febd-4dc7-9cc5-ba2b1cef9c5b", "FILE_EXPLORE_FORMSET_GUID",
        description = "",
        aliases = ['FileExploreFormSetGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f2ed27b-a01d-4867-b993-9b710e5926c5": EfiGuid(
        "1f2ed27b-a01d-4867-b993-9b710e5926c5", "StandaloneMmSmiHandlerProfileLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/SmmSmiHandlerProfileLib/StandaloneMmSmiHandlerProfileLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1f345462-cea5-4add-9c43-5ca6720b9da6": EfiGuid(
        "1f345462-cea5-4add-9c43-5ca6720b9da6", "IrqBoardInfoRvp",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f36527e-a97c-45f8-b24a-9d95b0a940fe": EfiGuid(
        "1f36527e-a97c-45f8-b24a-9d95b0a940fe", "AppleBrightnessControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f3a3278-82eb-4c0d-86f1-5bcda5846cb2": EfiGuid(
        "1f3a3278-82eb-4c0d-86f1-5bcda5846cb2", "PrePiHobLib",
        description = "",
        aliases = ['PrePiLib'],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Library/PrePiLib/PrePiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['PrePiLib']),

    "1f3cdfbe-f7ae-4453-8c48-c1ad35a4fd98": EfiGuid(
        "1f3cdfbe-f7ae-4453-8c48-c1ad35a4fd98", "SmbiosDmiEditProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f4c6f90-b06b-48d8-a201-bae5f1cd7d56": EfiGuid(
        "1f4c6f90-b06b-48d8-a201-bae5f1cd7d56", "EFI_PEI_STALL_PPI_GUID",
        description = "",
        aliases = ['EfiPeiStallPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/Stall.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_STALL_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['PEI_STALL_PPI_GUID']),

    "1f642910-3d7b-4627-8d18-dc62671e0554": EfiGuid(
        "1f642910-3d7b-4627-8d18-dc62671e0554", "BiosReservedMemoryPolicyPpiGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f6835a4-4a5d-4b02-86c5-b3fc87840e92": EfiGuid(
        "1f6835a4-4a5d-4b02-86c5-b3fc87840e92", "AsusOA3Bin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f73b18d-4630-43c1-a1de-6f80855d7da4": EfiGuid(
        "1f73b18d-4630-43c1-a1de-6f80855d7da4", "FORM_BROWSER_EXTENSION_PROTOCOL_GUID",
        description = "",
        aliases = ['gEdkiiFormBrowserExProtocolGuid', 'EfiFormBrowserExProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/FormBrowserEx.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EdkiiFormBrowserExProtocol', 'EfiFormBrowserExProtocol', 'EfiFormBrowserExProtocolGuid']),

    "1f77ffd9-c00d-4245-b2ac-f5f43e333c39": EfiGuid(
        "1f77ffd9-c00d-4245-b2ac-f5f43e333c39", "PtuLoader",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f78349d-7fba-4686-8098-fa017eda35fb": EfiGuid(
        "1f78349d-7fba-4686-8098-fa017eda35fb", "ArmShellCmdRunAxf",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f7a4aab-e6ab-416c-98b8-31f614fff198": EfiGuid(
        "1f7a4aab-e6ab-416c-98b8-31f614fff198", "DellSmmComputracePreInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f871708-1541-461f-91d5-4a90852dc913": EfiGuid(
        "1f871708-1541-461f-91d5-4a90852dc913", "FjGabiNvramMergeSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f8ece17-87a5-4196-b265-4f01fbf9aa2d": EfiGuid(
        "1f8ece17-87a5-4196-b265-4f01fbf9aa2d", "FjGabiLogoHandlerDxeBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1f91b73e-5b6a-4317-80e8-e7c36a3c7af4": EfiGuid(
        "1f91b73e-5b6a-4317-80e8-e7c36a3c7af4", "TestBaseSafeIntLibUefiShell",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Test/UnitTest/Library/BaseSafeIntLib/TestBaseSafeIntLibUefiShell.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1f9bf300-67fb-46ee-b6e7-2b6f6590531c": EfiGuid(
        "1f9bf300-67fb-46ee-b6e7-2b6f6590531c", "HpCertificateManagerDriverWmiSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fa1f39e-feff-4aae-bd7b-38a070a3b609": EfiGuid(
        "1fa1f39e-feff-4aae-bd7b-38a070a3b609", "PartitionDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/Disk/PartitionDxe/PartitionDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['PartitionDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "1fa4dafe-fa5d-4d75-bea6-5863862c520a": EfiGuid(
        "1fa4dafe-fa5d-4d75-bea6-5863862c520a", "TcgConfigDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Tcg/TcgConfigDxe/TcgConfigDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1faaa415-f8a9-4a33-b434-a7811fbec145": EfiGuid(
        "1faaa415-f8a9-4a33-b434-a7811fbec145", "SaveConfigMemDataBin_GUID",
        description = "",
        aliases = ['SaveConfigMemData'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SaveConfigMemData', 'SaveConfigMemDataBin_GUID'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "1fae70ac-8203-48f0-b9ef-e873ffecc1f2": EfiGuid(
        "1fae70ac-8203-48f0-b9ef-e873ffecc1f2", "PeiLegacyRegionPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fba8cfa-8194-49c4-bd8e-9f162a957ba1": EfiGuid(
        "1fba8cfa-8194-49c4-bd8e-9f162a957ba1", "WMISwSmiDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fbd2960-4130-41e5-94ac-d2cf037fb37c": EfiGuid(
        "1fbd2960-4130-41e5-94ac-d2cf037fb37c", "EFI_ADAPTER_INFO_NETWORK_BOOT_GUID",
        description = "",
        aliases = ['EfiAdapterInfoNetworkBootGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/AdapterInformation.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "1fd28f7c-8cfe-46c4-ae0f-0ceba37ff28b": EfiGuid(
        "1fd28f7c-8cfe-46c4-ae0f-0ceba37ff28b", "SystemSignaturesDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fd29be6-70d0-42a4-a6e7-e5d10e6ac376": EfiGuid(
        "1fd29be6-70d0-42a4-a6e7-e5d10e6ac376", "HDD_UNLOCKED_GUID",
        description = "",
        aliases = ['AmiHddPasswordVerified'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fd3ad5b-5e65-485e-9ebd-95b14d6eccc6": EfiGuid(
        "1fd3ad5b-5e65-485e-9ebd-95b14d6eccc6", "SetupConfigUpdateDxeXPV",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fdaabb0-ab7d-480c-91ff-428dc1546f3a": EfiGuid(
        "1fdaabb0-ab7d-480c-91ff-428dc1546f3a", "RTSMArmVExpressSecLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fdd3847-5848-4c79-9fbf-6d0086dfc899": EfiGuid(
        "1fdd3847-5848-4c79-9fbf-6d0086dfc899", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "1fdf71c4-d46d-45fb-bf85-56f1c529725a": EfiGuid(
        "1fdf71c4-d46d-45fb-bf85-56f1c529725a", "ComTypeControlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fe16506-5cc5-4d89-b8fb-6eb5e134887d": EfiGuid(
        "1fe16506-5cc5-4d89-b8fb-6eb5e134887d", "DxeWhea",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1feaee73-f8dc-439f-9fd3-a0ed92b5ebda": EfiGuid(
        "1feaee73-f8dc-439f-9fd3-a0ed92b5ebda", "SmbiosMFG",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1ff7913e-d890-4360-ac75-b8d8384bd1e0": EfiGuid(
        "1ff7913e-d890-4360-ac75-b8d8384bd1e0", "AmdSmmControl",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "1fff93c2-8c76-49e4-8ab3-43d92f5445ef": EfiGuid(
        "1fff93c2-8c76-49e4-8ab3-43d92f5445ef", "LogoJpg",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2000a832-c674-486a-81be-cb4670a4f7fd": EfiGuid(
        "2000a832-c674-486a-81be-cb4670a4f7fd", "DELL_REBOOT_DXE_GUID",
        description = "",
        aliases = ['DellRebootDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "201a92e1-2d0f-48e9-a3ab-93e1695a92f2": EfiGuid(
        "201a92e1-2d0f-48e9-a3ab-93e1695a92f2", "AppleHDA",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2022b972-b1b9-4b55-a603-7851cb062e34": EfiGuid(
        "2022b972-b1b9-4b55-a603-7851cb062e34", "LePassKeyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2029d12a-0a83-4277-86a2-712905201ae7": EfiGuid(
        "2029d12a-0a83-4277-86a2-712905201ae7", "SpiWmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "202a2922-8c27-4943-9855-26180bf9f113": EfiGuid(
        "202a2922-8c27-4943-9855-26180bf9f113", "VariableInfo",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Application/VariableInfo/VariableInfo.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "202a2b0e-9a31-4812-b291-8747df152439": EfiGuid(
        "202a2b0e-9a31-4812-b291-8747df152439", "LENOVO_SYSTEM_PS2_MOUSE_DXE_GUID",
        description = "",
        aliases = ['Ps2MouseDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['Ps2MouseDxe']),

    "204810e0-4941-4c66-b99a-6bee4f84453c": EfiGuid(
        "204810e0-4941-4c66-b99a-6bee4f84453c", "AtherosWiFi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "204c3d37-d83f-49ab-883f-9b5d6c647762": EfiGuid(
        "204c3d37-d83f-49ab-883f-9b5d6c647762", "FspTempRamExitGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20576248-c01a-439d-8cd6-418eaabe0a29": EfiGuid(
        "20576248-c01a-439d-8cd6-418eaabe0a29", "RasClvAdddcProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "205f7f0e-8eac-4914-8390-1b90dd7e2a27": EfiGuid(
        "205f7f0e-8eac-4914-8390-1b90dd7e2a27", "VariablePolicyLibRuntimeDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/VariablePolicyLib/VariablePolicyLibRuntimeDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "20605bba-7fb9-4279-959f-8dbf74cdb0c6": EfiGuid(
        "20605bba-7fb9-4279-959f-8dbf74cdb0c6", "VideoBios",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "206e7181-a935-4baf-949f-a875136cd46f": EfiGuid(
        "206e7181-a935-4baf-949f-a875136cd46f", "PldmBcc",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "208117f2-25f8-479d-b726-10c10bed6dc1": EfiGuid(
        "208117f2-25f8-479d-b726-10c10bed6dc1", "AppleLegacyRegion",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20830080-cc28-4169-9836-7f42b8d0c8c9": EfiGuid(
        "20830080-cc28-4169-9836-7f42b8d0c8c9", "GraphicsOutputDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/Console/GraphicsOutputDxe/GraphicsOutputDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2083b834-aa45-4785-b85f-1d717afd7b95": EfiGuid(
        "2083b834-aa45-4785-b85f-1d717afd7b95", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['5163727ab3ed5f9e02330a0c98095da0'],
        junk = []),

    "209097af-5acf-4e2d-a31b-72257e05dc40": EfiGuid(
        "209097af-5acf-4e2d-a31b-72257e05dc40", "ODMMemoryOverride",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "209674ef-6060-43e4-be87-e742e16beb00": EfiGuid(
        "209674ef-6060-43e4-be87-e742e16beb00", "NvmeControllerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2098f2de-af12-42e4-ad75-d88203ea0683": EfiGuid(
        "2098f2de-af12-42e4-ad75-d88203ea0683", "SecureBIOCamera_Sunplus2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20a1d839-853d-4067-b69c-552a0d05c1aa": EfiGuid(
        "20a1d839-853d-4067-b69c-552a0d05c1aa", "DellFormBrowser",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20a7378c-aa83-4ce1-821f-4740ee1b3f9f": EfiGuid(
        "20a7378c-aa83-4ce1-821f-4740ee1b3f9f", "PEI_OPERATOR_PRESENCE_PPI_GUID",
        description = "",
        aliases = ['PeiOperatorPresencePpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20b0f1c2-b0d8-4c5d-aad9-f44580dfdf8b": EfiGuid(
        "20b0f1c2-b0d8-4c5d-aad9-f44580dfdf8b", "EFI_PEI_AMI_KEYCODE_PPI_GUID",
        description = "",
        aliases = ['EfiPeiAmiKeycodePpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20b181e2-33e8-4211-b9d7-9b8696764e66": EfiGuid(
        "20b181e2-33e8-4211-b9d7-9b8696764e66", "WheaElog",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20c7c6be-56c6-46b7-b3ab-5588289aa803": EfiGuid(
        "20c7c6be-56c6-46b7-b3ab-5588289aa803", "XnoteSwSmiServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20c95b72-f4e3-4bb2-a289-19dbfd23f531": EfiGuid(
        "20c95b72-f4e3-4bb2-a289-19dbfd23f531", "PlatformInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20cbbcc9-5a4d-40ed-b1e0-80423ab1554f": EfiGuid(
        "20cbbcc9-5a4d-40ed-b1e0-80423ab1554f", "PB_DTSBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20d19705-e59d-4362-9956-fa533aff2429": EfiGuid(
        "20d19705-e59d-4362-9956-fa533aff2429", "SetBoardId",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20d6e759-4c4a-40c0-9533-2bf0066850fd": EfiGuid(
        "20d6e759-4c4a-40c0-9533-2bf0066850fd", "EfiQuiesceProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20d8fffe-15c3-4ea9-9d28-cfe2745d78f3": EfiGuid(
        "20d8fffe-15c3-4ea9-9d28-cfe2745d78f3", "CryptoDxe",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CryptoDxe'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7',
        ],
        junk = []),

    "20daf0fc-5548-44dc-a42a-60eaf0a22e47": EfiGuid(
        "20daf0fc-5548-44dc-a42a-60eaf0a22e47", "DXE_PLATFORM_TDT_POLICY_GUID",
        description = "",
        aliases = ['DxePlatformTdtPolicyGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DXE_PLATFORM_TDT_POLICY_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '8add95e4f71aa9680d87091104649f78',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bdbe423029a61611b078326db4953356',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'd293013d9f585157787fce8c228e8897', 'd805f342c43e187512b4b9fc30b58ed3',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "20dc4df3-aea8-4d12-9a20-c07eb9157e04": EfiGuid(
        "20dc4df3-aea8-4d12-9a20-c07eb9157e04", "OemSetupCallBack",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20de009a-b0b3-43da-8047-b5e2b19d6cc0": EfiGuid(
        "20de009a-b0b3-43da-8047-b5e2b19d6cc0", "FlashOemHooKDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20ded343-e5fa-49c2-b20f-c18798743d41": EfiGuid(
        "20ded343-e5fa-49c2-b20f-c18798743d41", "OemPowerMgmtSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20df5c15-5e32-42d7-bb67-3390a0e51ffd": EfiGuid(
        "20df5c15-5e32-42d7-bb67-3390a0e51ffd", "HpErrorLoggingPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20e28787-df32-4bda-b7e7-cbbda3371ef8": EfiGuid(
        "20e28787-df32-4bda-b7e7-cbbda3371ef8", "IDE_CONTROLLER_PROTOCOL_GUID",
        description = "",
        aliases = ['IdeControllerProtocol'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20e687be-b09b-4da1-bcfe-e804b3b97ce4": EfiGuid(
        "20e687be-b09b-4da1-bcfe-e804b3b97ce4", "CsmeHealthDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20ea66ca-17c3-4dd4-a51f-956cc9e4fc26": EfiGuid(
        "20ea66ca-17c3-4dd4-a51f-956cc9e4fc26", "AmdIspCamera",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "20f31568-d687-49ba-b326-ccd9d38ede16": EfiGuid(
        "20f31568-d687-49ba-b326-ccd9d38ede16", "AcpiRawLibArm",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/DynamicTablesPkg/Library/Acpi/Arm/AcpiRawLibArm/AcpiRawLibArm.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "20f8fd36-6d00-40fb-b704-d12c153c62eb": EfiGuid(
        "20f8fd36-6d00-40fb-b704-d12c153c62eb", "CcEventEntryHob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21073ef1-ff66-41ff-9bf4-a3e936db1901": EfiGuid(
        "21073ef1-ff66-41ff-9bf4-a3e936db1901", "FchHuashanPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21073fb3-ba6f-43eb-83f0-4a840c648165": EfiGuid(
        "21073fb3-ba6f-43eb-83f0-4a840c648165", "PlatformPeiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/ArmVirtPkg/Library/KvmtoolPlatformPeiLib/KvmtoolPlatformPeiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "21094ecb-9f20-4781-ae4b-50728b389a6e": EfiGuid(
        "21094ecb-9f20-4781-ae4b-50728b389a6e", "IchInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "210dcb72-bc14-4a19-a29b-f696eb7de12b": EfiGuid(
        "210dcb72-bc14-4a19-a29b-f696eb7de12b", "EzSetup",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2119bbd7-9432-4f47-b5e2-5c4ea31b6bdc": EfiGuid(
        "2119bbd7-9432-4f47-b5e2-5c4ea31b6bdc", "DxeIpl",
        description = "",
        aliases = ['PayloadEntry'],
        urls = ['https://github.com/tianocore/edk2/blob/master/UefiPayloadPkg/UefiPayloadEntry/UefiPayloadEntry.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "213a595c-f512-4935-90e3-2cead077c721": EfiGuid(
        "213a595c-f512-4935-90e3-2cead077c721", "DashDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21429b90-5f67-4e93-af55-1d314d646e12": EfiGuid(
        "21429b90-5f67-4e93-af55-1d314d646e12", "MemoryProfileInfo",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Application/MemoryProfileInfo/MemoryProfileInfo.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2145f72f-e6f1-4440-a828-59dc9aab5f89": EfiGuid(
        "2145f72f-e6f1-4440-a828-59dc9aab5f89", "EmmcDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Sd/EmmcDxe/EmmcDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2156e996-66de-4b27-9cc9-b09fac4d02be": EfiGuid(
        "2156e996-66de-4b27-9cc9-b09fac4d02be", "EFI_KMS_FORMAT_GENERIC_DYNAMIC_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Kms.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "215fdd18-bd50-4feb-890b-58ca0b4739e9": EfiGuid(
        "215fdd18-bd50-4feb-890b-58ca0b4739e9", "EFI_SIO_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSioProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SuperIo.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "21605810-54b3-4109-8aff-3b3fbf4af040": EfiGuid(
        "21605810-54b3-4109-8aff-3b3fbf4af040", "DellPasswordPolicyDxeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2161bf9c-8038-4f12-a758-30140660f22f": EfiGuid(
        "2161bf9c-8038-4f12-a758-30140660f22f", "StaticSkuDataDxeFischerLakeRP",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2167f964-7298-4a8f-9a2a-bfe498d600a8": EfiGuid(
        "2167f964-7298-4a8f-9a2a-bfe498d600a8", "ASROCK_HDD_READY_SMI_GUID",
        description = "",
        aliases = ['HddReadySmi'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "216a681c-9e51-415f-9ada-cb9fbcd05f5d": EfiGuid(
        "216a681c-9e51-415f-9ada-cb9fbcd05f5d", "DELL_FLASH_IO_DXE_GUID",
        description = "",
        aliases = ['DellFlashIoDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "217102b4-b465-4a1d-a2de-93dd385ec480": EfiGuid(
        "217102b4-b465-4a1d-a2de-93dd385ec480", "BaseIoLibIntrinsicArmVirt",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseIoLibIntrinsic/BaseIoLibIntrinsicArmVirt.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "21729246-5df1-46a1-a5c6-2484bcd706fb": EfiGuid(
        "21729246-5df1-46a1-a5c6-2484bcd706fb", "LEMSetVariablePreInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21782819-fda0-4ade-bd36-c95f079f057d": EfiGuid(
        "21782819-fda0-4ade-bd36-c95f079f057d", "AsusBackupAccess",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "217fa926-f494-4b3f-ac45-d213c2a0294d": EfiGuid(
        "217fa926-f494-4b3f-ac45-d213c2a0294d", "PostLogo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21834f44-9201-4aa3-9b15-afd794d93bc4": EfiGuid(
        "21834f44-9201-4aa3-9b15-afd794d93bc4", "OemWakeOnRtcPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "218596fc-6ebd-4d19-a79b-41b13448021c": EfiGuid(
        "218596fc-6ebd-4d19-a79b-41b13448021c", "OemSWBoardIDDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21877e2f-f86e-4e8a-9c9b-d7b152dd40d8": EfiGuid(
        "21877e2f-f86e-4e8a-9c9b-d7b152dd40d8", "EfiPeiPlatformTypeNeonCityEpecbPpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21891a9e-763e-4377-8841-8d5c90d88c51": EfiGuid(
        "21891a9e-763e-4377-8841-8d5c90d88c51", "NetworkStackSetupScreen",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "218e36a1-c2a4-4d42-8024-c9b80e3daf47": EfiGuid(
        "218e36a1-c2a4-4d42-8024-c9b80e3daf47", "FjFvFlashRt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "218f1930-10ab-43b3-9c66-b98483a1ae9d": EfiGuid(
        "218f1930-10ab-43b3-9c66-b98483a1ae9d", "OpalBlockSidDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "218f4b00-e4f0-4cad-8a90-577b833d0128": EfiGuid(
        "218f4b00-e4f0-4cad-8a90-577b833d0128", "BixbyDxeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "219144ea-d6b4-4d6a-b8f5-63ca8a09c8d0": EfiGuid(
        "219144ea-d6b4-4d6a-b8f5-63ca8a09c8d0", "CsSmmDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2197dd5b-b8d9-4897-92f1-18d75903e6af": EfiGuid(
        "2197dd5b-b8d9-4897-92f1-18d75903e6af", "HpSleepStateSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21a05fd5-db4a-4cfc-b84b-eb0dbb569934": EfiGuid(
        "21a05fd5-db4a-4cfc-b84b-eb0dbb569934", "IconGenericCD",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21a74f63-ca10-4dc6-a4eb-36c93b9a063a": EfiGuid(
        "21a74f63-ca10-4dc6-a4eb-36c93b9a063a", "PLATFORM_FLASH_SMM_PROTOCOL_GUID",
        description = "",
        aliases = ['PlatformFlashSmmProtocol'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21adc483-021f-4f3b-8dce-613cc981a269": EfiGuid(
        "21adc483-021f-4f3b-8dce-613cc981a269", "menu_dots_selected",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21af95e1-371f-4712-9c07-798e3cb019e4": EfiGuid(
        "21af95e1-371f-4712-9c07-798e3cb019e4", "LockSMRAMEntryDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21b0cb55-4330-4343-aff8-a68089124118": EfiGuid(
        "21b0cb55-4330-4343-aff8-a68089124118", "DellServiceMenuConfig",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21b564bf-53ea-40d1-85ae-effc93d28640": EfiGuid(
        "21b564bf-53ea-40d1-85ae-effc93d28640", "FchKeithMdnDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21be6180-733a-443b-df46-41c008047687": EfiGuid(
        "21be6180-733a-443b-df46-41c008047687", "AmdNbioPciePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21bf3644-1ce7-4e87-8c60-0a5c607173a6": EfiGuid(
        "21bf3644-1ce7-4e87-8c60-0a5c607173a6", "RemoteFWConfiguration",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21c3b115-4e0b-470c-85c7-e105a575c97b": EfiGuid(
        "21c3b115-4e0b-470c-85c7-e105a575c97b", "gEdkiiEmbeddedGpioPpiGuid",
        description = "",
        aliases = ['EdkiiEmbeddedGpioPpi'],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "21ccf0b7-246b-412c-a334-0b65a07b28df": EfiGuid(
        "21ccf0b7-246b-412c-a334-0b65a07b28df", "SmmBaseOnSmmBase2Thunk",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SmmBaseOnSmmBase2Thunk'],
        hashes = [
            '0e6eebbc21300ba0ebec9f192d5beabf', '0fb0d5db7a0d624b39b616db11024c0a',
            '13bf9b32863d8378b9c802c9508c96e9', '186cc72a8ef4d9cb1062b2cea44ef726',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '2b4ca0bd16a20893deff9632c23c2cca',
            '3dbf73930867371066435e0908e50535', '409221b30602fc13c8cbedf952c60297',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'b218fecd176e986c1491046e58c9e01b', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'ea61df93fb3257dc78c2415817c1f871',
            'f8c34791a7a721ee330771cb43d68b91', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "21d04395-96a0-4fd5-b477-a922648edaf3": EfiGuid(
        "21d04395-96a0-4fd5-b477-a922648edaf3", "AsusSampleDxeWrapperPkg",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21d56275-f984-405f-9f0b-ea084def0a7e": EfiGuid(
        "21d56275-f984-405f-9f0b-ea084def0a7e", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['8a42c7b29e7fcf02022c68ff430e9ed7'],
        junk = []),

    "21e173d6-c221-48ee-a18f-b73d9810fbf6": EfiGuid(
        "21e173d6-c221-48ee-a18f-b73d9810fbf6", "AsusApmPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21e34727-3881-4dee-8020-d8908a980311": EfiGuid(
        "21e34727-3881-4dee-8020-d8908a980311", "EpuHwModeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21e70404-df72-4122-b030-281306eb7be3": EfiGuid(
        "21e70404-df72-4122-b030-281306eb7be3", "IntelGopVbt5",
        description = "",
        aliases = ['VbtEdpTypeCGuid'],
        urls = [],
        vendors = [],
        filenames = ['IntelGopVbt5'],
        hashes = [
            '13bf9b32863d8378b9c802c9508c96e9', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '28e65e1a6689d8a2a4a612a06cbd2a58', '409221b30602fc13c8cbedf952c60297',
            '811505684bd99c18f5ecbee1f725ff66', '97012cbeef60b7e6dc93c362aa6fe780',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a9258521b7c65199bd268280867cc628',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'f8c34791a7a721ee330771cb43d68b91',
        ],
        junk = []),

    "21f2a221-0cf9-4e0a-9fd0-2daddc025e31": EfiGuid(
        "21f2a221-0cf9-4e0a-9fd0-2daddc025e31", "BctBaseSmmSTX",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21f302ad-6e94-471b-84bc-b14800403a1d": EfiGuid(
        "21f302ad-6e94-471b-84bc-b14800403a1d", "EFI_SMM_CPU_SAVE_STATE_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmmCpuSaveStateProtocolGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_SMM_CPU_SAVE_STATE_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cfba6c85608b8135fb21024783573bf6', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "21fee8db-0d29-477e-b5a9-96eb343ba99c": EfiGuid(
        "21fee8db-0d29-477e-b5a9-96eb343ba99c", "ADDITIONAL_SYSTEM_INFO_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "21ff1fee-d33a-4fce-a65e-955ea3c41f40": EfiGuid(
        "21ff1fee-d33a-4fce-a65e-955ea3c41f40", "PEI_QNC_MEMORY_INIT_PPI_GUID",
        description = "",
        aliases = ['QNCMemoryInitPpiGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['QNCMemoryInitPpi']),

    "22046d50-f390-498c-92e5-5ba4f8e7f8b6": EfiGuid(
        "22046d50-f390-498c-92e5-5ba4f8e7f8b6", "SBSATAIDE",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SBSATAIDE'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '8091cabbb4de557d84e3dbd55126455a',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "22074e71-bccc-4517-8757-7995edfd8032": EfiGuid(
        "22074e71-bccc-4517-8757-7995edfd8032", "PCH_PLATFORM_POLICY_PPI_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['PCH_PLATFORM_POLICY_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8add95e4f71aa9680d87091104649f78',
            '9cf5adeb713d71eb03527d7e52165d5d', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cfba6c85608b8135fb21024783573bf6', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "220ac432-1d43-49e5-a74f-4c9da67ad23b": EfiGuid(
        "220ac432-1d43-49e5-a74f-4c9da67ad23b", "HOT_PLUG_DEVICE_GUID",
        description = "",
        aliases = ['HotPlugDevice'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "220e57c8-4e71-493f-91b8-0f7f820a1da2": EfiGuid(
        "220e57c8-4e71-493f-91b8-0f7f820a1da2", "I2cMouseDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "220e73b6-6bdb-4413-8405-b974b108619a": EfiGuid(
        "220e73b6-6bdb-4413-8405-b974b108619a", "EFI_FIRMWARE_VOLUME2_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiFirmwareVolume2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/FirmwareVolume2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_FIRMWARE_VOLUME2_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "22137115-fa45-6e45-6c5e-36dc9a000171": EfiGuid(
        "22137115-fa45-6e45-6c5e-36dc9a000171", "MaximHr",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['MaximHr'],
        hashes = [
            '3555862f32138b911338fae3789c3921', '39a2ecdc1427b5f2c143fd59ef976f2c',
            '82060a1dae992f24609d2f673cb58cfb', 'b4fa3a413d748be6699112adc0c14b0d',
        ],
        junk = []),

    "221521ae-0a35-44cd-b580-5aedbb770b1d": EfiGuid(
        "221521ae-0a35-44cd-b580-5aedbb770b1d", "glyphs",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2216614e-cdbe-46e7-ba61-9bd2bc640393": EfiGuid(
        "2216614e-cdbe-46e7-ba61-9bd2bc640393", "AmiPspFlashSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22198fd5-4835-4842-bf31-eb957c7dd70d": EfiGuid(
        "22198fd5-4835-4842-bf31-eb957c7dd70d", "GetNetByAddr",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "221d6760-4ac0-4c12-ba96-9c63e4d9c1c8": EfiGuid(
        "221d6760-4ac0-4c12-ba96-9c63e4d9c1c8", "DELL_PEI_HWM_INIT_GUID",
        description = "",
        aliases = ['HwmInitPei'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "221f1d4f-034c-4bea-b2bb-b7a9672b06d7": EfiGuid(
        "221f1d4f-034c-4bea-b2bb-b7a9672b06d7", "_CRB_SMI_GUID",
        description = "",
        aliases = ['CrbSmi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CrbSmi', '_CRB_SMI_GUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "2224aee3-8d0b-480a-b272-2abe92cd4e78": EfiGuid(
        "2224aee3-8d0b-480a-b272-2abe92cd4e78", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['15e3c83eb7038ec850e8498e82078fb0'],
        junk = []),

    "222c386d-5abc-4fb4-b124-fbb82488acf4": EfiGuid(
        "222c386d-5abc-4fb4-b124-fbb82488acf4", "PlatformPei",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/PlatformPei/PlatformPei.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2236b8bc-e488-4424-beb9-97c8a1483693": EfiGuid(
        "2236b8bc-e488-4424-beb9-97c8a1483693", "PlatformOpalDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2237c798-93e7-4119-b10c-cf75cd22bfe0": EfiGuid(
        "2237c798-93e7-4119-b10c-cf75cd22bfe0", "AmdApcbZpDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2240157b-a45a-46eb-9a7e-1e79c4781fd7": EfiGuid(
        "2240157b-a45a-46eb-9a7e-1e79c4781fd7", "HWChangeWarning",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "224453ce-fed2-49cc-a42f-443ff9ccb6c7": EfiGuid(
        "224453ce-fed2-49cc-a42f-443ff9ccb6c7", "Ast2600VgaDriverX64",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "224fbfe4-adb6-4df2-b835-602182aeef20": EfiGuid(
        "224fbfe4-adb6-4df2-b835-602182aeef20", "Clock",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22597239-6107-df44-ad3f-5f053e92222e": EfiGuid(
        "22597239-6107-df44-ad3f-5f053e92222e", "EmuSnpDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmulatorPkg/EmuSnpDxe/EmuSnpDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "225c3a5a-5902-4307-968a-af8deffe61e8": EfiGuid(
        "225c3a5a-5902-4307-968a-af8deffe61e8", "UsbPwrCtrlSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "226824a3-1989-4d6c-bd17-c863845f7e99": EfiGuid(
        "226824a3-1989-4d6c-bd17-c863845f7e99", "DellSmiCpuDecodeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "226a500a-e14f-414a-a956-40e5762d3d1e": EfiGuid(
        "226a500a-e14f-414a-a956-40e5762d3d1e", "PrmLoaderDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "226d28e1-7672-48d5-addc-e5f91f398a3e": EfiGuid(
        "226d28e1-7672-48d5-addc-e5f91f398a3e", "SetupConfigUpdateDxeExpertWorkStationRP",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2272865b-ebf6-4047-b008-7889497f53bd": EfiGuid(
        "2272865b-ebf6-4047-b008-7889497f53bd", "SuperMDriverProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22766cba-d444-4a04-9e77-50bb240f00a6": EfiGuid(
        "22766cba-d444-4a04-9e77-50bb240f00a6", "VPMSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22819110-7f6f-4852-b4bb-13a770149b0c": EfiGuid(
        "22819110-7f6f-4852-b4bb-13a770149b0c", "FpgaFormSet",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "228aad82-e150-4288-b176-a1aad77d783e": EfiGuid(
        "228aad82-e150-4288-b176-a1aad77d783e", "FjIbvTraceAbstractionDxeProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "228f344d-b3de-43bb-a4d7-ea200b1b1482": EfiGuid(
        "228f344d-b3de-43bb-a4d7-ea200b1b1482", "EFI_SM_MONITOR_INIT_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiSmMonitorInitProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/Include/Protocol/SmMonitorInit.h",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/UefiCpuPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "228f7d4f-231c-6baa-bdcb-bea9875b04d3": EfiGuid(
        "228f7d4f-231c-6baa-bdcb-bea9875b04d3", "AmiCpmWrapperSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22951f5e-4f7b-405b-8ada-ecdc431c616f": EfiGuid(
        "22951f5e-4f7b-405b-8ada-ecdc431c616f", "UefiWindowsInt10Workaround",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "229832d3-7a30-4b36-b827-f40cb7d45436": EfiGuid(
        "229832d3-7a30-4b36-b827-f40cb7d45436", "EFI_PEI_REPORT_PROGRESS_CODE_PPI_GUID",
        description = "",
        aliases = ['EfiPeiStatusCodePpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/StatusCode.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiPeiStatusCodePpi', 'EfiPeiStatusCodePpiGuid', 'PEI_STATUS_CODE_PPI_GUID']),

    "229b7efd-da02-46b9-93f4-e20c009f94e9": EfiGuid(
        "229b7efd-da02-46b9-93f4-e20c009f94e9", "CpuS3DataDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/CpuS3DataDxe/CpuS3DataDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "229f268f-23b0-4436-83e3-8b7ae08a6bd8": EfiGuid(
        "229f268f-23b0-4436-83e3-8b7ae08a6bd8", "EfiMfgUuidHob",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22a5bb4d-f962-4d33-9cd6-fdad39acd153": EfiGuid(
        "22a5bb4d-f962-4d33-9cd6-fdad39acd153", "PlatformBoardIdPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22a66b9e-ac73-41a1-b3ac-7248eb55c339": EfiGuid(
        "22a66b9e-ac73-41a1-b3ac-7248eb55c339", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "22a8c2c1-9f75-4b9d-b037-22cc43e6929e": EfiGuid(
        "22a8c2c1-9f75-4b9d-b037-22cc43e6929e", "SioDummyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22aab6b3-fdad-4383-a6d4-1cbad157c895": EfiGuid(
        "22aab6b3-fdad-4383-a6d4-1cbad157c895", "AcpiI3cSlaveSsdt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22aaffea-2358-4c53-9af0-cc0ccb06e2ef": EfiGuid(
        "22aaffea-2358-4c53-9af0-cc0ccb06e2ef", "PanelResolution",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22b194b4-cc0e-46c7-9fce-da10d6ed1731": EfiGuid(
        "22b194b4-cc0e-46c7-9fce-da10d6ed1731", "PchSmbusArpEnabled",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22bdee84-c807-452e-b56e-f683fd76c989": EfiGuid(
        "22bdee84-c807-452e-b56e-f683fd76c989", "BuslogicDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22bf0b85-4eb3-479b-a9e9-a5f7cfcca787": EfiGuid(
        "22bf0b85-4eb3-479b-a9e9-a5f7cfcca787", "LENOVO_SMART_CARD_DETECT_DXE_GUID",
        description = "",
        aliases = ['LenovoSmartCardDetectDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22c71e5f-b32b-44e0-b705-b1adcfe70c0c": EfiGuid(
        "22c71e5f-b32b-44e0-b705-b1adcfe70c0c", "SureStartPoliciesPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22cef43a-ed2b-4795-af0d-918ff85b3573": EfiGuid(
        "22cef43a-ed2b-4795-af0d-918ff85b3573", "ServiceBodyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22d85435-f24a-43db-7d04-015606df21b1": EfiGuid(
        "22d85435-f24a-43db-7d04-015606df21b1", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "22da1234-1db3-4d56-8646-fd785a59337e": EfiGuid(
        "22da1234-1db3-4d56-8646-fd785a59337e", "AmiDfciPlatform",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22dc2b60-fe40-42ac-b01f-3ab1fad9aad8": EfiGuid(
        "22dc2b60-fe40-42ac-b01f-3ab1fad9aad8", "EmuVariableFvbRuntimeDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/EmuVariableFvbRuntimeDxe/Fvb.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "22df032e-b7b2-43ec-b903-bce4190aed43": EfiGuid(
        "22df032e-b7b2-43ec-b903-bce4190aed43", "RstPeim32",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22ea234f-e72a-11e4-91f9-28d2447c4829": EfiGuid(
        "22ea234f-e72a-11e4-91f9-28d2447c4829", "HttpUtilitiesDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/NetworkPkg/HttpUtilitiesDxe/HttpUtilitiesDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['HttpUtilitiesDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '1ded807f42af76804c9a0bbb6e24a2a9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '633e80e7347dc01c21c5133227df2aba', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "22eb683b-1ad2-4be9-ae1c-c4f1f281bdf7": EfiGuid(
        "22eb683b-1ad2-4be9-ae1c-c4f1f281bdf7", "PoweronFromKeyboardPortDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22f8b88d-4e9e-4da3-8cd5-74463593ef13": EfiGuid(
        "22f8b88d-4e9e-4da3-8cd5-74463593ef13", "DellPeiDiagLeds",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22fbd744-ca98-4f6d-9882-954201312da0": EfiGuid(
        "22fbd744-ca98-4f6d-9882-954201312da0", "H2OVarCheckRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22fdff5d-ef58-4e45-b515-a41067d5c80b": EfiGuid(
        "22fdff5d-ef58-4e45-b515-a41067d5c80b", "AssetQrCode",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22ff12f3-4b62-172f-37a6-a6bedab1ff63": EfiGuid(
        "22ff12f3-4b62-172f-37a6-a6bedab1ff63", "I2CcontrolPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "22ff5e69-1765-4a3d-ad07-d2029925ce39": EfiGuid(
        "22ff5e69-1765-4a3d-ad07-d2029925ce39", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "2304df80-e21d-4170-9c3c-113c878f7ac0": EfiGuid(
        "2304df80-e21d-4170-9c3c-113c878f7ac0", "X86QemuLoadImageLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Library/X86QemuLoadImageLib/X86QemuLoadImageLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "230f6679-f703-4dc2-b2b7-41c670bcc0d1": EfiGuid(
        "230f6679-f703-4dc2-b2b7-41c670bcc0d1", "BiosInfoRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "231be53d-b6e3-44d8-a8d5-e0566e55663d": EfiGuid(
        "231be53d-b6e3-44d8-a8d5-e0566e55663d", "EsrtOverrideDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "231cdc65-129e-4870-a882-79244127bde3": EfiGuid(
        "231cdc65-129e-4870-a882-79244127bde3", "SkipStoragePwdPrompt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "231d3a65-9442-4507-a4e4-e9d2176578f1": EfiGuid(
        "231d3a65-9442-4507-a4e4-e9d2176578f1", "DELL_SM_BIOS_STRUC_B1_GUID",
        description = "",
        aliases = ['DellSmBiosStrucB1'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2325f2fc-5683-4648-97c4-9a520dfbe325": EfiGuid(
        "2325f2fc-5683-4648-97c4-9a520dfbe325", "AMI_TCG_PERM_FLAGS_GUID",
        description = "",
        aliases = ['AmiTcgPermFlags'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "232d6306-b70c-48bb-88bb-9fa6d47c8208": EfiGuid(
        "232d6306-b70c-48bb-88bb-9fa6d47c8208", "RtcStatus",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "232fb018-f308-4081-b280-812e6164123b": EfiGuid(
        "232fb018-f308-4081-b280-812e6164123b", "PasswordMgrDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23321ffd-1f8e-4597-84ec-4dad0e2ba995": EfiGuid(
        "23321ffd-1f8e-4597-84ec-4dad0e2ba995", "SmmLockS3",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2332e35a-4fd6-47cd-97d3-8304e81b12f9": EfiGuid(
        "2332e35a-4fd6-47cd-97d3-8304e81b12f9", "ItkSmmVarsDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['ItkSmmVarsDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "2333a860-a228-462e-8121-7b49895c11c5": EfiGuid(
        "2333a860-a228-462e-8121-7b49895c11c5", "ASRockSIDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2336a93d-08d7-41c5-afc1-cdb60c94cda4": EfiGuid(
        "2336a93d-08d7-41c5-afc1-cdb60c94cda4", "CaseOpenPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2338337a-47b0-4c41-9ca7-0160fb94dff4": EfiGuid(
        "2338337a-47b0-4c41-9ca7-0160fb94dff4", "LENOVO_PLATFORM_FLASH_DXE_GUID",
        description = "",
        aliases = ['PlatformFlashDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "233c2592-1cec-494a-a097-15dc96379777": EfiGuid(
        "233c2592-1cec-494a-a097-15dc96379777", "FwVolDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "233df097-3218-47b2-9e09-fe58c2b20d22": EfiGuid(
        "233df097-3218-47b2-9e09-fe58c2b20d22", "LENOVO_PLATFORM_STAGE2_PEI_GUID",
        description = "",
        aliases = ['LenovoPlatformStage2Pei'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2342ca44-3b35-4a34-995b-cedeeb1a9576": EfiGuid(
        "2342ca44-3b35-4a34-995b-cedeeb1a9576", "Intel945Uga",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2345fb4d-0b26-4a14-aa2e-f7b955154495": EfiGuid(
        "2345fb4d-0b26-4a14-aa2e-f7b955154495", "StibpFeatureDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "234dfad6-8626-40fd-a1ef-e4b68cf57a3b": EfiGuid(
        "234dfad6-8626-40fd-a1ef-e4b68cf57a3b", "ProtectedDataDisplayPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "234e6146-1fc6-4508-8231-1294cc28da4c": EfiGuid(
        "234e6146-1fc6-4508-8231-1294cc28da4c", "FjGrasscarryLaunch",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2354d320-3eb3-4c81-99f4-cc4ae143463e": EfiGuid(
        "2354d320-3eb3-4c81-99f4-cc4ae143463e", "EfiEdkCompatibilityPkgTokenSpaceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "235be51a-d4d5-49ef-9c46-d313056b7401": EfiGuid(
        "235be51a-d4d5-49ef-9c46-d313056b7401", "UsbPolicySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "235c9ba7-2da4-4df4-92dd-5555fc7db2ac": EfiGuid(
        "235c9ba7-2da4-4df4-92dd-5555fc7db2ac", "OemACRecoveryPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2361a042-f0da-4006-b3c1-3a130fae8df0": EfiGuid(
        "2361a042-f0da-4006-b3c1-3a130fae8df0", "I2cTouchPanel",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2362ea9c-84e5-4dff-83bc-b5acecb57cbb": EfiGuid(
        "2362ea9c-84e5-4dff-83bc-b5acecb57cbb", "AMI_CSM_THUNK_PROTOCOL_GUID",
        description = "",
        aliases = ['AmiCsmThunkDriverGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2366c20f-e15a-11e3-8bf1-e4115b28bc50": EfiGuid(
        "2366c20f-e15a-11e3-8bf1-e4115b28bc50", "HttpDxe",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/NetworkPkg/HttpDxe/HttpDxe.inf'],
        vendors = ['EDK2'],
        filenames = ['HttpDxe'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '0e6eebbc21300ba0ebec9f192d5beabf', '13bf9b32863d8378b9c802c9508c96e9',
            '1a0f212de5a43dc1de07ea5e31e3adb0', '1ded807f42af76804c9a0bbb6e24a2a9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '633e80e7347dc01c21c5133227df2aba', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f8c34791a7a721ee330771cb43d68b91', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "2374eddf-f203-4fc0-a20e-61bad73089d6": EfiGuid(
        "2374eddf-f203-4fc0-a20e-61bad73089d6", "DELL_IO_TRAP_GUID",
        description = "",
        aliases = ['IoTrap'],
        urls = [],
        vendors = [],
        filenames = ['DELL_IO_TRAP_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['IoTrap']),

    "237cc158-6032-4832-b0ba-1f83c0fca609": EfiGuid(
        "237cc158-6032-4832-b0ba-1f83c0fca609", "CoreHotKeyProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2383608e-c6d0-4e3e-858d-45dfac3543d5": EfiGuid(
        "2383608e-c6d0-4e3e-858d-45dfac3543d5", "PciHostBridgeDxe",
        description = "",
        aliases = ['PciHostBridge'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2386622c-fc1a-4b5b-ae3a-c81fb30af128": EfiGuid(
        "2386622c-fc1a-4b5b-ae3a-c81fb30af128", "FjDxeLeg",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2386e231-a9cf-4f52-946a-6f6b6c133b46": EfiGuid(
        "2386e231-a9cf-4f52-946a-6f6b6c133b46", "DELL_SM_BIOS_DXE_DRIVER_GUID",
        description = "",
        aliases = ['DellSmBiosTableLoaderDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "238d654d-a6ad-41c8-a31b-67a3208625c3": EfiGuid(
        "238d654d-a6ad-41c8-a31b-67a3208625c3", "LENOVO_MX25_L3206_EFLASH_PART_SMM_GUID",
        description = "",
        aliases = ['LenovoMx25L3206EflashPartSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "239421f6-f025-429c-9889-ab854e00eee6": EfiGuid(
        "239421f6-f025-429c-9889-ab854e00eee6", "CheckRaid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "239b7f68-26e1-4b48-b966-85338aa7b637": EfiGuid(
        "239b7f68-26e1-4b48-b966-85338aa7b637", "OemSioPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "239e812c-ad4a-4d5a-8bb9-169131c344bf": EfiGuid(
        "239e812c-ad4a-4d5a-8bb9-169131c344bf", "AmdIdsDebugPrintPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23a089b3-eed5-4ac5-b2ab-43e3298c2343": EfiGuid(
        "23a089b3-eed5-4ac5-b2ab-43e3298c2343", "VariableSmm",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Universal/Variable/RuntimeDxe/VariableSmm.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "23a3e7ba-75d1-4cb9-9c8f-56fa4e48d99e": EfiGuid(
        "23a3e7ba-75d1-4cb9-9c8f-56fa4e48d99e", "EfiTraceHubDebugLibIa32",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23a464ad-cb83-48b8-94ab-1a6fefcfe522": EfiGuid(
        "23a464ad-cb83-48b8-94ab-1a6fefcfe522", "EFI_SIO_PPI_GUID",
        description = "",
        aliases = ['EfiSioPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/SuperIo.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "23a6afda-f2a5-45ec-beff-420639d345b9": EfiGuid(
        "23a6afda-f2a5-45ec-beff-420639d345b9", "DynamicAmlLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/DynamicTablesPkg/Library/Common/AmlLib/AmlLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "23a7b657-077f-4abc-ab7e-b70d8a389dbe": EfiGuid(
        "23a7b657-077f-4abc-ab7e-b70d8a389dbe", "DellVariableProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23af8bfa-1d89-47cb-876c-1be11fa0f4b0": EfiGuid(
        "23af8bfa-1d89-47cb-876c-1be11fa0f4b0", "AtaIdentifyData",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23af9a54-3d7c-444d-8318-e9cf752da349": EfiGuid(
        "23af9a54-3d7c-444d-8318-e9cf752da349", "DxeNt32PeCoffExtraActionLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23b461d0-3c94-490d-9a4b-4ae1916bd425": EfiGuid(
        "23b461d0-3c94-490d-9a4b-4ae1916bd425", "DellAtaAtapiIoProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23b4bd04-198c-4a11-b65a-a52b33c98c18": EfiGuid(
        "23b4bd04-198c-4a11-b65a-a52b33c98c18", "AmiHspFtpmAcpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23b5f74b-df6c-4b38-89e8-0fddfeb0f1ab": EfiGuid(
        "23b5f74b-df6c-4b38-89e8-0fddfeb0f1ab", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "23c29286-947d-4270-b061-7fe5d0758b63": EfiGuid(
        "23c29286-947d-4270-b061-7fe5d0758b63", "Legacy2Region",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23cace14-eba4-49f6-9681-c697ff0b649e": EfiGuid(
        "23cace14-eba4-49f6-9681-c697ff0b649e", "VirtioSerialDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23d1280d-43f0-4713-90b2-0e5e4221af4c": EfiGuid(
        "23d1280d-43f0-4713-90b2-0e5e4221af4c", "BatteryState1",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23d93eaa-0723-4559-b768-5e22df762de4": EfiGuid(
        "23d93eaa-0723-4559-b768-5e22df762de4", "LENOVO_TPM_ENABLER_DXE_GUID",
        description = "",
        aliases = ['LenovoTpmEnablerDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23eed05d-1b93-4a1a-8e1b-931d69e37952": EfiGuid(
        "23eed05d-1b93-4a1a-8e1b-931d69e37952", "Omap35xxBoardInterruptDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23f0f40f-686d-45e4-b26e-11f9f265cf66": EfiGuid(
        "23f0f40f-686d-45e4-b26e-11f9f265cf66", "BatteryInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23f0f40f-686d-45e4-b28e-11f9f245cf62": EfiGuid(
        "23f0f40f-686d-45e4-b28e-11f9f245cf62", "MpmBatteryInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "23f69cd0-ff46-4db6-b982-63edf1a901ff": EfiGuid(
        "23f69cd0-ff46-4db6-b982-63edf1a901ff", "AmiTseOemPortingGuid2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240286e6-55d1-4dd2-8e9a-ad26569c7abd": EfiGuid(
        "240286e6-55d1-4dd2-8e9a-ad26569c7abd", "BoardInitSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240612b5-a063-11d4-9a3a-0090273fc14d": EfiGuid(
        "240612b5-a063-11d4-9a3a-0090273fc14d", "IsaBusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240612b5-a063-11d4-9a3a-0090273fc18e": EfiGuid(
        "240612b5-a063-11d4-9a3a-0090273fc18e", "SmmCoreDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240612b7-a063-11d4-9a3a-0090273fc14c": EfiGuid(
        "240612b7-a063-11d4-9a3a-0090273fc14c", "LENOVO_SYSTEM_USB_SUPPORT_POLICY_SMM_GUID",
        description = "",
        aliases = ['SystemUsbSupportPolicySmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240612b7-a063-11d4-9a3a-0090273fc14d": EfiGuid(
        "240612b7-a063-11d4-9a3a-0090273fc14d", "LENOVO_SYSTEM_USB_BUS_DXE_GUID",
        description = "",
        aliases = ['UsbBusDxe', 'SystemUsbBusDxe'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Bus/Usb/UsbBusDxe/UsbBusDxe.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['UsbBusDxe']),

    "240612b7-a063-11d4-9a3a-0090273fc14e": EfiGuid(
        "240612b7-a063-11d4-9a3a-0090273fc14e", "LENOVO_SYSTEM_USB_SUPPORT_POLICY_DXE_GUID",
        description = "",
        aliases = ['SystemUsbSupportPolicyDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24092bed-2736-40a7-8d87-515e8594f6b3": EfiGuid(
        "24092bed-2736-40a7-8d87-515e8594f6b3", "FjGabiOsEntryGateDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240c89b9-5bb1-494d-a0a6-8f0e1935b45b": EfiGuid(
        "240c89b9-5bb1-494d-a0a6-8f0e1935b45b", "AmdFabricStxKrkSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "240e9bef-b64b-4760-a05f-50027564ef1d": EfiGuid(
        "240e9bef-b64b-4760-a05f-50027564ef1d", "VariableBackupRestoreSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2413d304-aa41-4079-a81e-31b8d7c3fd6b": EfiGuid(
        "2413d304-aa41-4079-a81e-31b8d7c3fd6b", "GpioV2PpiInitPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24169e35-2454-4940-92bc-82321a2c7562": EfiGuid(
        "24169e35-2454-4940-92bc-82321a2c7562", "VlvInitPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['ChvInitPeim', 'VlvInitPeim'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '186cc72a8ef4d9cb1062b2cea44ef726',
            '2b4ca0bd16a20893deff9632c23c2cca', '3555862f32138b911338fae3789c3921',
            '39a2ecdc1427b5f2c143fd59ef976f2c', '3dbf73930867371066435e0908e50535',
            '48bba53caf662f85dd73a1a247b45dc7', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'b218fecd176e986c1491046e58c9e01b',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'cae37d13982bb26c35ffaada2841d171',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "242b3d0c-5fb8-4a75-9cd3-710ddfe42703": EfiGuid(
        "242b3d0c-5fb8-4a75-9cd3-710ddfe42703", "AmdNbioDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "243373d3-c183-4f6a-9ae4-e38419fcdccc": EfiGuid(
        "243373d3-c183-4f6a-9ae4-e38419fcdccc", "WarmResetFlagSmi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2433ec61-17be-4b98-895c-333950c62318": EfiGuid(
        "2433ec61-17be-4b98-895c-333950c62318", "RtkUndiDxe_2057",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "243c8c8a-bbd0-4aa9-be17-cf9b583130ec": EfiGuid(
        "243c8c8a-bbd0-4aa9-be17-cf9b583130ec", "LENOVO_SMM_OEMINT15_GUID",
        description = "",
        aliases = ['SmmOEMInt15'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "243d3e8c-2780-4a25-9693-a410475bfcec": EfiGuid(
        "243d3e8c-2780-4a25-9693-a410475bfcec", "FrameBufferBltLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/FrameBufferBltLib/FrameBufferBltLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "243e170b-83bb-4e43-840a-f9ec857fd783": EfiGuid(
        "243e170b-83bb-4e43-840a-f9ec857fd783", "FjGabiPostFlash",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24400798-3807-4a42-b413-a1ecee205dd8": EfiGuid(
        "24400798-3807-4a42-b413-a1ecee205dd8", "DXE_SIGN_BIOS_AUTHENTICATE_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "244338f5-abda-4dbd-8782-1de293b5bbb4": EfiGuid(
        "244338f5-abda-4dbd-8782-1de293b5bbb4", "PlatformStatusCodeHandlerPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2443909f-892d-46d7-7977-4902259419aa": EfiGuid(
        "2443909f-892d-46d7-7977-4902259419aa", "HpRtxXhciSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24486226-f8c2-41f5-b9dd-783e9e56dea0": EfiGuid(
        "24486226-f8c2-41f5-b9dd-783e9e56dea0", "EfiMmioDeviceProtocolGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24503dd2-5a92-4db8-95b6-0f0bf6fea97c": EfiGuid(
        "24503dd2-5a92-4db8-95b6-0f0bf6fea97c", "AmdNbioPcieDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "245cb4da-8e15-4a1b-87e3-9878ffa07520": EfiGuid(
        "245cb4da-8e15-4a1b-87e3-9878ffa07520", "Legacy8259",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/8259InterruptControllerDxe/8259.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "245dca21-fb7b-11d3-8f01-00a0c969723b": EfiGuid(
        "245dca21-fb7b-11d3-8f01-00a0c969723b", "EFI_PXE_BASE_CODE_CALLBACK_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiPxeBaseCodeCallbackProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/PxeBaseCodeCallBack.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "246f9f0a-11e3-459a-ae06-372bdc4de806": EfiGuid(
        "246f9f0a-11e3-459a-ae06-372bdc4de806", "LENOVO_SYSTEM_STATUS_CODE_GENERIC_RT_GUID",
        description = "",
        aliases = ['LenovoSystemStatusCodeGenericRt'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "247323af-c8f1-4b8c-9087-aa4ba7b76d6a": EfiGuid(
        "247323af-c8f1-4b8c-9087-aa4ba7b76d6a", "ME_EOP_DONE_HOB_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2474dcb8-4bb4-49da-8783-7cd3d385ff07": EfiGuid(
        "2474dcb8-4bb4-49da-8783-7cd3d385ff07", "P2SB_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2476acb7-d4c7-48d3-ab73-7513b03bb5df": EfiGuid(
        "2476acb7-d4c7-48d3-ab73-7513b03bb5df", "AmdOemRasRsSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2477be9e-8e1a-431e-b705-14e663717377": EfiGuid(
        "2477be9e-8e1a-431e-b705-14e663717377", "PsmiComboBufferGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2480271c-09c6-4f36-ad75-5e1390bd9929": EfiGuid(
        "2480271c-09c6-4f36-ad75-5e1390bd9929", "QNCSmmDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24848d1d-a637-45dd-974f-beba0340ff96": EfiGuid(
        "24848d1d-a637-45dd-974f-beba0340ff96", "RemotePlatformErase",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2486829b-d3f3-47ec-827a-fc104907fc5c": EfiGuid(
        "2486829b-d3f3-47ec-827a-fc104907fc5c", "SmmGenericSio",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['SmmGenericSio'],
        hashes = [
            '002681fae113306a057042387f62f790', '0bf3064114ecb05222114c4d8d687938',
            '1ded807f42af76804c9a0bbb6e24a2a9', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '54a655d3479750be95c150698a9e80ff', '60785be6bfb0d1daa94a8bb214b774d1',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '8091cabbb4de557d84e3dbd55126455a',
            '86c4a8e759a9aa0eee18a5e7d480583f', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'e7a706c2421a55cde70e476d1d37f831',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'eee15c9794a4040563b41004ebccc91a',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "2493f533-a25c-4e65-b26c-cf4dd6b7903b": EfiGuid(
        "2493f533-a25c-4e65-b26c-cf4dd6b7903b", "SctMilestoneTaskSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "249cef4d-7b90-49f1-b67f-2033f0942623": EfiGuid(
        "249cef4d-7b90-49f1-b67f-2033f0942623", "DellTcg2Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "249edd7d-346d-4c90-b94f-16079ef061d5": EfiGuid(
        "249edd7d-346d-4c90-b94f-16079ef061d5", "AmdCpuPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24a2d66f-eedd-4086-9042-f26e4797ee69": EfiGuid(
        "24a2d66f-eedd-4086-9042-f26e4797ee69", "ROOT_BRIDGES_CONNECTED_EVENT_GROUP_GUID",
        description = "",
        aliases = ['RootBridgesConnectedEventGroupGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/Include/Guid/RootBridgesConnectedEventGroup.h",
            "https://github.com/tianocore/edk2/blob/master/OvmfPkg/OvmfPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "24a44caf-0bf2-4514-90c4-c794b3e778f5": EfiGuid(
        "24a44caf-0bf2-4514-90c4-c794b3e778f5", "MePolicyInitDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "24a6ab96-8a00-4cc6-bae7-63a9d3dd3b42": EfiGuid(
        "24a6ab96-8a00-4cc6-bae7-63a9d3dd3b42", "SuperIoExDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24b09abe-4e47-481c-a9ad-cef12c392327": EfiGuid(
        "24b09abe-4e47-481c-a9ad-cef12c392327", "Omap35xxTokenSpaceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24b14cb4-a14a-4e9a-aad2-853335fa3297": EfiGuid(
        "24b14cb4-a14a-4e9a-aad2-853335fa3297", "AmdCpmModernStandbyFeatureDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24b4980d-3f4d-4529-b326-cbe87e35013c": EfiGuid(
        "24b4980d-3f4d-4529-b326-cbe87e35013c", "AmdDmarControlPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24b9d62c-5a36-417b-94b6-38dbaea90dcf": EfiGuid(
        "24b9d62c-5a36-417b-94b6-38dbaea90dcf", "AcpiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Library/AcpiLib/AcpiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "24bb68e0-0922-44c4-84ea-95fd75e59de2": EfiGuid(
        "24bb68e0-0922-44c4-84ea-95fd75e59de2", "AsusGpnvVersion",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24be45e6-2b61-440d-8a75-e71fb6eb5b10": EfiGuid(
        "24be45e6-2b61-440d-8a75-e71fb6eb5b10", "Tdt3gBin_GUID",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24c33f98-505d-4177-90f7-a20cf8ff3020": EfiGuid(
        "24c33f98-505d-4177-90f7-a20cf8ff3020", "ReTimerFmpDummy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24c5dc2f-53e2-40ca-9ed6-a5d9a49f463b": EfiGuid(
        "24c5dc2f-53e2-40ca-9ed6-a5d9a49f463b", "EFI_HASH_ALGORITHM_SHA1_NOPAD_GUID",
        description = "",
        aliases = ['EfiHashAlgorithmSha1NoPadGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Hash.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "24c6f3e2-6acd-436b-a604-56a5cf742a55": EfiGuid(
        "24c6f3e2-6acd-436b-a604-56a5cf742a55", "BaseFspPlatformInfoLibSample",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24c856a9-46e7-4635-8017-0fdcca1ff9c9": EfiGuid(
        "24c856a9-46e7-4635-8017-0fdcca1ff9c9", "AmdFabricMdnSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24ccd374-3df6-4181-86f6-e3c66920a145": EfiGuid(
        "24ccd374-3df6-4181-86f6-e3c66920a145", "DELL_UPDATE_MEMORY_RECORD_GUID",
        description = "",
        aliases = ['UpdateMemoryRecord'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DELL_UPDATE_MEMORY_RECORD_GUID', 'UpdateMemoryRecord'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '13bf9b32863d8378b9c802c9508c96e9', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f8c34791a7a721ee330771cb43d68b91', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['UpdateMemoryRecord']),

    "24d7ac91-3a1a-44b7-8cd5-0800a66bcfe5": EfiGuid(
        "24d7ac91-3a1a-44b7-8cd5-0800a66bcfe5", "OemBadgingSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24dc0658-f2cd-4034-b50d-2634640c35ba": EfiGuid(
        "24dc0658-f2cd-4034-b50d-2634640c35ba", "AsusHardwareSignature",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24ddc83f-9f06-44ab-b696-60fe7df2c948": EfiGuid(
        "24ddc83f-9f06-44ab-b696-60fe7df2c948", "RstVmdPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24e24f91-2e6d-4525-bae9-b977de2005d6": EfiGuid(
        "24e24f91-2e6d-4525-bae9-b977de2005d6", "AepLogDrv",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24e70042-d5c5-4260-8c39-0ad3aa32e93d": EfiGuid(
        "24e70042-d5c5-4260-8c39-0ad3aa32e93d", "EFI_MM_END_OF_DXE_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_SMM_END_OF_DXE_PROTOCOL_GUID', 'gEfiMmEndOfDxeProtocolGuid', 'EfiSmmEndOfDxeProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MmEndOfDxe.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SmmEndOfDxe.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiSmmEndOfDxeProtocol', 'EfiSmmEndOfDxeProtocolGuid', 'EFI_SMM_END_OF_DXE_PROTOCOL_GUID']),

    "24e9a512-3a25-4cba-a0ae-67c053bdf3b6": EfiGuid(
        "24e9a512-3a25-4cba-a0ae-67c053bdf3b6", "DxeVideoTextOut",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "24f14d8a-d7a8-4991-91e0-96c3b7db8456": EfiGuid(
        "24f14d8a-d7a8-4991-91e0-96c3b7db8456", "VARIABLE_CLEANUP_HII_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Library/PlatformVarCleanupLib/PlatVarCleanupHii.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "24ff7176-8b2a-4138-9509-3d5e3059bcf4": EfiGuid(
        "24ff7176-8b2a-4138-9509-3d5e3059bcf4", "ExtendOperatingAmbientTemperatureModeDXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25053bb9-01ae-4af7-82a7-cda6d9545921": EfiGuid(
        "25053bb9-01ae-4af7-82a7-cda6d9545921", "TouchPad_Synaptics_2nd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25075106-f537-4bd4-afa5-cff0e6f3b2a6": EfiGuid(
        "25075106-f537-4bd4-afa5-cff0e6f3b2a6", "AmdFabricRvDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2509b2f1-a022-4cca-af70-f9d321fb6649": EfiGuid(
        "2509b2f1-a022-4cca-af70-f9d321fb6649", "EDKII_DEVICE_IDENTIFIER_TYPE_PCI_GUID",
        description = "",
        aliases = ['gEdkiiDeviceIdentifierTypePciGuid', 'EdkiiDeviceIdentifierTypePci'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Protocol/DeviceSecurity.h",
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/MdeModulePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2515f54f-3277-47da-86a5-484510dd08e1": EfiGuid(
        "2515f54f-3277-47da-86a5-484510dd08e1", "AmdPspDxeV2Mdn",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "251701e7-e266-4623-9968-738cd2231096": EfiGuid(
        "251701e7-e266-4623-9968-738cd2231096", "SERIAL_IRQ_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "251c7818-0dbf-e619-7fc2-d6ac43427da3": EfiGuid(
        "251c7818-0dbf-e619-7fc2-d6ac43427da3", "HASH_ALGORITHM_SM3_256_GUID",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/SecurityPkg/Include/Library/HashLib.h'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "25247a74-9440-47d5-bf0a-ed92a4d6eba4": EfiGuid(
        "25247a74-9440-47d5-bf0a-ed92a4d6eba4", "DELL_COMPUTRACE_BINARY_GUID",
        description = "",
        aliases = ['DellComputraceBinary'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2525b0f0-80c2-4af7-bc68-3be4bb42c11e": EfiGuid(
        "2525b0f0-80c2-4af7-bc68-3be4bb42c11e", "PermanentlyDisabled",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2525c3f2-e255-4375-a7c9-92d1054d62b8": EfiGuid(
        "2525c3f2-e255-4375-a7c9-92d1054d62b8", "MsiBoardECPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25264b72-7a80-4856-a7ec-15802270ee1b": EfiGuid(
        "25264b72-7a80-4856-a7ec-15802270ee1b", "CheckBootGuardKeyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "252e599f-d604-4bea-8feb-347668e93b8f": EfiGuid(
        "252e599f-d604-4bea-8feb-347668e93b8f", "FjRTS54xx",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25330d5d-5474-4eb8-8192-e6de3d5ed0b6": EfiGuid(
        "25330d5d-5474-4eb8-8192-e6de3d5ed0b6", "DellVariable2Smm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "253e85e9-993b-439b-b74c-6120f77b4723": EfiGuid(
        "253e85e9-993b-439b-b74c-6120f77b4723", "PlatformReset",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25462cda-221f-47df-ac1d-259cfaa4e326": EfiGuid(
        "25462cda-221f-47df-ac1d-259cfaa4e326", "DT_PLATFORM_DEFAULT_DTB_FILE_GUID",
        description = "",
        aliases = ['gDtPlatformDefaultDtbFileGuid', 'DtPlatformDefaultDtbFile'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec",
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Include/Guid/DtPlatformDefaultDtbFile.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2549016d-2b17-4d00-b5a4-1d443a0138f8": EfiGuid(
        "2549016d-2b17-4d00-b5a4-1d443a0138f8", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['8a42c7b29e7fcf02022c68ff430e9ed7'],
        junk = []),

    "254901ad-7db7-45f8-93c8-93d579398d9f": EfiGuid(
        "254901ad-7db7-45f8-93c8-93d579398d9f", "PeiPciSegmentLibPciCfg2",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/PeiPciSegmentLibPciCfg2/PeiPciSegmentLibPciCfg2.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "254b4a79-772c-45cc-05c9-17f84c2ea197": EfiGuid(
        "254b4a79-772c-45cc-05c9-17f84c2ea197", "HpAmdXhciDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "254baffe-f25e-45f1-a06f-5ef11443aca4": EfiGuid(
        "254baffe-f25e-45f1-a06f-5ef11443aca4", "AmdMcaZen5Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2554ef5e-c9ca-4a48-9d94-249eb1e87c2d": EfiGuid(
        "2554ef5e-c9ca-4a48-9d94-249eb1e87c2d", "LENOVO_CRYPT_SERVICE_GUID",
        description = "",
        aliases = ['LenovoCryptService'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2555a2a2-101b-4775-b11e-47674f446628": EfiGuid(
        "2555a2a2-101b-4775-b11e-47674f446628", "FjPostManagerProtocolAbstractionReference",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25566b03-b577-4cbf-958c-ed663ea24380": EfiGuid(
        "25566b03-b577-4cbf-958c-ed663ea24380", "EFI_MM_GPI_DISPATCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_SMM_GPI_DISPATCH2_PROTOCOL_GUID', 'gEfiMmGpiDispatchProtocolGuid', 'EfiSmmGpiDispatch2ProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MmGpiDispatch.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SmmGpiDispatch2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_MM_GPI_DISPATCH_PROTOCOL_GUID'],
        hashes = [
            '81460325c53ef849a2e3c2a5e104304b', 'ac2383f4616ea50d075c37b988c3cb83',
            'b218fecd176e986c1491046e58c9e01b', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'd4a0cd284c2f726df5db17c605809651',
            'dab68811108b555173f8f096c9737d20', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'ea61df93fb3257dc78c2415817c1f871', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = ['EfiSmmGpiDispatch2Protocol', 'EfiSmmGpiDispatch2ProtocolGuid', 'EFI_SMM_GPI_DISPATCH2_PROTOCOL_GUID']),

    "255d96a8-1579-4adf-8575-56d4b0467117": EfiGuid(
        "255d96a8-1579-4adf-8575-56d4b0467117", "DellTcg2EarlyPlatformPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25682ba8-b41d-4403-b034-253769e0dad5": EfiGuid(
        "25682ba8-b41d-4403-b034-253769e0dad5", "AcpiIortLibArm",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/DynamicTablesPkg/Library/Acpi/Arm/AcpiIortLibArm/AcpiIortLibArm.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "256a381e-0165-4909-8663-ee317819292e": EfiGuid(
        "256a381e-0165-4909-8663-ee317819292e", "AIMT",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "258a5f45-00d5-485b-84db-0b6d4aab5e2f": EfiGuid(
        "258a5f45-00d5-485b-84db-0b6d4aab5e2f", "TbtSsdt",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "258b415e-3e8e-4b82-8067-9efc7949eac8": EfiGuid(
        "258b415e-3e8e-4b82-8067-9efc7949eac8", "ResetSystemDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "258b77c1-5989-45a4-bfff-7daf147b2ce0": EfiGuid(
        "258b77c1-5989-45a4-bfff-7daf147b2ce0", "FjAddingFlexIoToSmBios",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "259311f9-a268-46c9-8db4-79acfb2b7dc1": EfiGuid(
        "259311f9-a268-46c9-8db4-79acfb2b7dc1", "EarlyDevices",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "259495ea-2dd9-4ea5-a111-61f58781499d": EfiGuid(
        "259495ea-2dd9-4ea5-a111-61f58781499d", "I2cMasterSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "259a90b1-c151-43e5-8a71-82b4e3201f4e": EfiGuid(
        "259a90b1-c151-43e5-8a71-82b4e3201f4e", "LegacyBootMaintUiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/OvmfPkg/Csm/LegacyBootMaintUiLib/LegacyBootMaintUiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "25a4fd4a-9703-4ba9-a190-b7c84efb3e57": EfiGuid(
        "25a4fd4a-9703-4ba9-a190-b7c84efb3e57", "FDT_VARIABLE_GUID",
        description = "",
        aliases = ['FdtVariableGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/EmbeddedPkg.dec",
            "https://github.com/tianocore/edk2/blob/master/EmbeddedPkg/Include/Guid/Fdt.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "25a8824e-6bbf-4fb2-a200-84b0f7bece6b": EfiGuid(
        "25a8824e-6bbf-4fb2-a200-84b0f7bece6b", "B57785",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25ac458a-cf60-476e-861a-211c757657a6": EfiGuid(
        "25ac458a-cf60-476e-861a-211c757657a6", "SnpDxePort",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25acf158-dd61-4e64-9a49-55851e9a26c7": EfiGuid(
        "25acf158-dd61-4e64-9a49-55851e9a26c7", "DELL_BIOS_BLK_IO_SRC_GUID",
        description = "",
        aliases = ['CsmBlockIo'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['CsmBlockIo', 'DELL_BIOS_BLK_IO_SRC_GUID'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '3dbf73930867371066435e0908e50535',
            '4002d2fb07cfbaed2e86d9608824378a', '409221b30602fc13c8cbedf952c60297',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '97012cbeef60b7e6dc93c362aa6fe780', '97070b6e86ea62d42df9c0d42179c050',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', '9f603f855d176a1ea5758fa617d3e0d2',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9258521b7c65199bd268280867cc628',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "25ae661d-3652-4767-a800-2c3d03f4097a": EfiGuid(
        "25ae661d-3652-4767-a800-2c3d03f4097a", "Armani_ProductInfoSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25b09472-f258-49ea-a1ce-8a68f3a4a54b": EfiGuid(
        "25b09472-f258-49ea-a1ce-8a68f3a4a54b", "InitOemSetupVariable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25b5312a-70da-41d7-9801-25d6af9c6250": EfiGuid(
        "25b5312a-70da-41d7-9801-25d6af9c6250", "PchResetPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['PchResetPeim'],
        hashes = [
            '0ebecbce655f6f780edb7d73c1d80d13', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '3fa6831c92f133013c5bb0942f307f72', '6f7959935a87f450b05674b1e62ba7fa',
            '7457f65a90b756398ae360b93c326a5e', '7d13012b3ccc7c61a0c88ca8969676f6',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '9e80820994976fe9edc2b53346ea4f71',
            'ac2383f4616ea50d075c37b988c3cb83', 'c7e102ca6520f991d823aa784d64ae46',
            'cb6725610355d5dfad06ffc8bb80b931', 'd4a0cd284c2f726df5db17c605809651',
            'dbcaa4b015b1aec94558985a97dce7c2',
        ],
        junk = []),

    "25bb7f51-752c-491e-bcb1-55ea608b3197": EfiGuid(
        "25bb7f51-752c-491e-bcb1-55ea608b3197", "CertificateStorageSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25bddda9-ccf5-4d13-9954-ec69d30babc4": EfiGuid(
        "25bddda9-ccf5-4d13-9954-ec69d30babc4", "AmdCpmZeroPowerOddPeim",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25bfeba1-a3ba-4be2-9248-886392f7b008": EfiGuid(
        "25bfeba1-a3ba-4be2-9248-886392f7b008", "AmdNbioPciePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25c0310a-cb9e-47e0-cbc0-f7f45e55ac9f": EfiGuid(
        "25c0310a-cb9e-47e0-cbc0-f7f45e55ac9f", "DellMfgBootListConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25c36cac-80a6-400f-b0b1-8aef82341801": EfiGuid(
        "25c36cac-80a6-400f-b0b1-8aef82341801", "OemReadyBoot",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25ca4430-495d-42ef-8157-4d0ae2124862": EfiGuid(
        "25ca4430-495d-42ef-8157-4d0ae2124862", "HidAbsolutePointer",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25d3605e-522f-4570-a197-56bddacb5fed": EfiGuid(
        "25d3605e-522f-4570-a197-56bddacb5fed", "BiosUpdatePlatformPolicyPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25ec8ad0-0006-4f3e-908a-0cb82609938e": EfiGuid(
        "25ec8ad0-0006-4f3e-908a-0cb82609938e", "NvmExpressSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25ecad7c-2c93-35b8-2e54-c71a4c02d3d1": EfiGuid(
        "25ecad7c-2c93-35b8-2e54-c71a4c02d3d1", "QualCommSupplicantDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25f200aa-d3cb-470a-bf51-e7d162d22e6f": EfiGuid(
        "25f200aa-d3cb-470a-bf51-e7d162d22e6f", "SHELL_DEBUG1_HII_GUID",
        description = "",
        aliases = ['ShellDebug1HiiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/Include/Guid/ShellLibHiiGuid.h",
            "https://github.com/tianocore/edk2/blob/master/ShellPkg/ShellPkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "25f384da-cb4f-4a89-9e12-1eb4be79d6c1": EfiGuid(
        "25f384da-cb4f-4a89-9e12-1eb4be79d6c1", "EcMemoryIntrusionTestModeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "25f49067-a65b-48f5-bbbe-35418c488836": EfiGuid(
        "25f49067-a65b-48f5-bbbe-35418c488836", "LENOVO_TCG_SERVICES_DXE_GUID",
        description = "",
        aliases = ['TcgDxeMainDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['TcgDxeMainDxe']),

    "25f6cc90-8c83-479b-95ab-d2d0adc818dc": EfiGuid(
        "25f6cc90-8c83-479b-95ab-d2d0adc818dc", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "25fc783f-41ad-4ba1-b190-6d4b3fb6cc5b": EfiGuid(
        "25fc783f-41ad-4ba1-b190-6d4b3fb6cc5b", "ThunderboltDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2601c96e-243b-4509-ab14-df2228e5b401": EfiGuid(
        "2601c96e-243b-4509-ab14-df2228e5b401", "DellSbSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2605c8b7-ef64-475d-90f5-e767339f4d3e": EfiGuid(
        "2605c8b7-ef64-475d-90f5-e767339f4d3e", "OemUsbLightBarDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26091df6-43bf-4907-b7dd-f7020849a63c": EfiGuid(
        "26091df6-43bf-4907-b7dd-f7020849a63c", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['14a40eed04ba11d0eacb77b25294ed69'],
        junk = []),

    "260aa875-0eed-4ee1-8a14-046b4fb17ef5": EfiGuid(
        "260aa875-0eed-4ee1-8a14-046b4fb17ef5", "DetectRecoveryModePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "261230e3-22a4-3e55-40c6-50486d9f8d67": EfiGuid(
        "261230e3-22a4-3e55-40c6-50486d9f8d67", "RTL8111dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2619ea76-4599-4978-8649-e7371170c256": EfiGuid(
        "2619ea76-4599-4978-8649-e7371170c256", "OpticalDiskIdm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "261e97b4-fbb5-4db1-b25c-13270ba7bb60": EfiGuid(
        "261e97b4-fbb5-4db1-b25c-13270ba7bb60", "EmulatedEepromDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "262b2e3f-da2d-4b81-8d1d-f938e851fded": EfiGuid(
        "262b2e3f-da2d-4b81-8d1d-f938e851fded", "SmcBoardInfoVariable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "263056b9-7100-4536-8240-c54f8e6ab3fe": EfiGuid(
        "263056b9-7100-4536-8240-c54f8e6ab3fe", "FastBootPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['FastBootPei'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0ebecbce655f6f780edb7d73c1d80d13',
            '0fb0d5db7a0d624b39b616db11024c0a', '0fc6e822ed386a76a1ac3b30c3afd4d4',
            '136ef7c1715a5ab03a93c66f1f9447e1', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '3555862f32138b911338fae3789c3921',
            '35a55f8ce610ca18a6d9480a0fc484df', '39a2ecdc1427b5f2c143fd59ef976f2c',
            '3dbf73930867371066435e0908e50535', '3fa6831c92f133013c5bb0942f307f72',
            '4002d2fb07cfbaed2e86d9608824378a', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '6f7959935a87f450b05674b1e62ba7fa',
            '73c9d09abcd1f4bf7ddbfe437773286e', '7457f65a90b756398ae360b93c326a5e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d13012b3ccc7c61a0c88ca8969676f6',
            '7d7236b9ad7c65c2e13957e6027586fa', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '9e80820994976fe9edc2b53346ea4f71',
            'a12b44297a063d752d860068ef2ce3db', 'a40b9558d367d413a0126b691b78e006',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'eee15c9794a4040563b41004ebccc91a', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "263631d7-5836-4b74-be48-ee22e92ce5d3": EfiGuid(
        "263631d7-5836-4b74-be48-ee22e92ce5d3", "WinNtConsoleDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2636fcac-b54e-4aa5-a720-49887ce228d2": EfiGuid(
        "2636fcac-b54e-4aa5-a720-49887ce228d2", "ClearCmosPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26452f27-45de-4a94-807a-0e6fdc1cb962": EfiGuid(
        "26452f27-45de-4a94-807a-0e6fdc1cb962", "EmuPeiGateDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26490f7a-7fa2-423c-8939-c6206329bc37": EfiGuid(
        "26490f7a-7fa2-423c-8939-c6206329bc37", "AcpiGtdtLibArm",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/DynamicTablesPkg/Library/Acpi/Arm/AcpiGtdtLibArm/AcpiGtdtLibArm.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "266e31cc-13c5-4807-b9dc-39a6ba88ff1a": EfiGuid(
        "266e31cc-13c5-4807-b9dc-39a6ba88ff1a", "CPU_INIT_DATA_HOB_GUID",
        description = "",
        aliases = ['CpuInitDataHobGuid'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26703ed8-9171-40a3-95c2-56436f8a5e56": EfiGuid(
        "26703ed8-9171-40a3-95c2-56436f8a5e56", "EfiBpCommonPkgTokenSpaceGuid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2673b0c0-1f9f-45ef-ffff-ffff92fa6f0a": EfiGuid(
        "2673b0c0-1f9f-45ef-ffff-ffff92fa6f0a", "XnoteReportStatusCodePei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "267ff286-b286-41b5-ae00-951ee5d39d09": EfiGuid(
        "267ff286-b286-41b5-ae00-951ee5d39d09", "BctBaseSmmRMB",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "268038b9-b691-404c-8e83-58014f3919fc": EfiGuid(
        "268038b9-b691-404c-8e83-58014f3919fc", "CheckCpuFanState",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26841bde-920a-4e7a-9fbe-637f477143a6": EfiGuid(
        "26841bde-920a-4e7a-9fbe-637f477143a6", "Ip4ConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['Ip4ConfigDxe'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '0fb0d5db7a0d624b39b616db11024c0a',
            '136ef7c1715a5ab03a93c66f1f9447e1', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '3dbf73930867371066435e0908e50535',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7f662f93e41d560c5fb667dce9000eb7', '81460325c53ef849a2e3c2a5e104304b',
            '869f701de5296248126c8c8a03a479c3', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '953dc1131b7a579176b9fc937eee5846', '9e80820994976fe9edc2b53346ea4f71',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b49cea81c680d9848212ed2aca8e6771', 'b68b9a378d9f7e39c59e5c68771086c8',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd4a0cd284c2f726df5db17c605809651', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'f9917106a00ba5725ae8e30c4becb2c7',
        ],
        junk = []),

    "2684d010-abe8-4e1e-a61d-ce6419a513c3": EfiGuid(
        "2684d010-abe8-4e1e-a61d-ce6419a513c3", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['38db553d1865539b33c3ba7db59e0be9'],
        junk = []),

    "2686340e-665c-427f-8819-05ba54f030f5": EfiGuid(
        "2686340e-665c-427f-8819-05ba54f030f5", "IdeController2",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2688b232-9c02-4c12-be1f-857c0ff2aae3": EfiGuid(
        "2688b232-9c02-4c12-be1f-857c0ff2aae3", "TcgDxeplatform",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['TcgDxeplatform'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '136ef7c1715a5ab03a93c66f1f9447e1', '13bf9b32863d8378b9c802c9508c96e9',
            '15c3afd565e35a7b87854f203ea15633', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '36d56c567fa4c101f6af6f56bec74299',
            '3c3abb9a4b4b6d9718f8ac83d8145e92', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '4afaa764f57449bc440e58fa11422e02',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '756c574dcde55419446fb64425ebd1ff',
            '7a20bc8c8925db2963205a36be03b335', '7b3f08b9dac97b5334c6851820b9a1b0',
            '7d7236b9ad7c65c2e13957e6027586fa', '7dffb9d276837abbf9e319d968a966a1',
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'ac6277674bc0b5eb2156e1391dfb7143',
            'b218fecd176e986c1491046e58c9e01b', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'ee8ffebc11a82e29543a142d51a8797d',
            'eee15c9794a4040563b41004ebccc91a', 'f199b62c1ea50fa7dc3f10fda43e62a2',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "268f33a9-cccd-48be-8817-86053ac32ed6": EfiGuid(
        "268f33a9-cccd-48be-8817-86053ac32ed6", "EFI_PEI_MM_ACCESS_PPI_GUID",
        description = "",
        aliases = ['PEI_SMM_ACCESS_PPI_GUID', 'PeiSmmAccessPpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdeModulePkg/Include/Ppi/SmmAccess.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/MmAccess.h",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_PEI_MM_ACCESS_PPI_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '953dc1131b7a579176b9fc937eee5846', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ea61df93fb3257dc78c2415817c1f871', 'ead4f1b7e116eaed945c115a76621189',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = ['PeiSmmAccessPpi', 'PeiSmmAccessPpiGuid', 'PEI_SMM_ACCESS_PPI_GUID']),

    "2696f127-44b2-4e14-8bff-804aa177fe41": EfiGuid(
        "2696f127-44b2-4e14-8bff-804aa177fe41", "SDJpegDecoder",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26a2481e-4424-46a2-9943-cc4039ead8f8": EfiGuid(
        "26a2481e-4424-46a2-9943-cc4039ead8f8", "DELL_ACPIS3_SAVE_GUID",
        description = "",
        aliases = ['S3Save'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['DELL_ACPIS3_SAVE_GUID', 'S3Save'],
        hashes = [
            '0b8de120c00ed155bbf748f6e8438a2f', '0fb0d5db7a0d624b39b616db11024c0a',
            '136ef7c1715a5ab03a93c66f1f9447e1', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1f0bf94fb4f5484c39510715a328e0e9',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '3dbf73930867371066435e0908e50535', '48bba53caf662f85dd73a1a247b45dc7',
            '4bb7a38c96ed64c4fb149d0ad4ec2d8d', '540ab45d79fb417ade67cb3aac048180',
            '54b7b82c79116e989585483d47eff1b4', '6f1b81ae0adfe7f90c141e463e9e9344',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c0bdac2284b1e46c7c9c1509ddbea006',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd4a0cd284c2f726df5db17c605809651', 'd805f342c43e187512b4b9fc30b58ed3',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e0f49f68b1fc98c1a09b18f35912ef2f',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f9917106a00ba5725ae8e30c4becb2c7',
            'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['S3Save']),

    "26baccb1-6f42-11d4-bce7-0080c73c8881": EfiGuid(
        "26baccb1-6f42-11d4-bce7-0080c73c8881", "EFI_CPU_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiCpuArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Cpu.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_CPU_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "26baccb2-6f42-11d4-bce7-0080c73c8881": EfiGuid(
        "26baccb2-6f42-11d4-bce7-0080c73c8881", "EFI_METRONOME_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiMetronomeArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Metronome.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_METRONOME_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "26baccb3-6f42-11d4-bce7-0080c73c8881": EfiGuid(
        "26baccb3-6f42-11d4-bce7-0080c73c8881", "EFI_TIMER_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiTimerArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Timer.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_TIMER_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "26c04cf3-f5fb-4968-8d57-c7fa0a932783": EfiGuid(
        "26c04cf3-f5fb-4968-8d57-c7fa0a932783", "DELL_SB_SERVICES_SMM_GUID",
        description = "",
        aliases = ['DellSbServicesSmm'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26c628f2-ced5-466c-8237-433ca4d24241": EfiGuid(
        "26c628f2-ced5-466c-8237-433ca4d24241", "MemoryEvContent",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26c9d769-9167-4537-8219-d9f5fc2378be": EfiGuid(
        "26c9d769-9167-4537-8219-d9f5fc2378be", "OemUsbTypeCDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26cc0fad-beb3-478a-91b2-0c188f726198": EfiGuid(
        "26cc0fad-beb3-478a-91b2-0c188f726198", "EFI_PEI_RECOVERY_BLOCK_IO2_PPI_GUID",
        description = "",
        aliases = ['EfiPeiVirtualBlockIo2PpiGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Ppi/BlockIo2.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiPeiVirtualBlockIo2Ppi']),

    "26cc7c04-4e9d-4fed-afba-cbf23334dc0d": EfiGuid(
        "26cc7c04-4e9d-4fed-afba-cbf23334dc0d", "AmtWrapperDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26cf084c-c9db-41bb-92c6-d197b8a1e4bf": EfiGuid(
        "26cf084c-c9db-41bb-92c6-d197b8a1e4bf", "MEMORY_CONFIG_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26dc4851-195f-4ae1-9a19-fbf883bbb35e": EfiGuid(
        "26dc4851-195f-4ae1-9a19-fbf883bbb35e", "AMI_APTIO_SIG_OWNER_GUID",
        description = "",
        aliases = ['AmiAptioSigOwner'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26ddba9f-5b0d-4e80-86b2-80dae4d01b0e": EfiGuid(
        "26ddba9f-5b0d-4e80-86b2-80dae4d01b0e", "LENOVO_FDISK_OEM_GUID",
        description = "",
        aliases = ['FdiskOemDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26ebec07-930b-4b92-81b7-2bc104d1cdbc": EfiGuid(
        "26ebec07-930b-4b92-81b7-2bc104d1cdbc", "Nct5124dPeiInit",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26eeb3de-b689-492e-80f0-be8bd7da4ba7": EfiGuid(
        "26eeb3de-b689-492e-80f0-be8bd7da4ba7", "EFI_MM_CONFIGURATION_PROTOCOL_GUID",
        description = "",
        aliases = ['EFI_SMM_CONFIGURATION_PROTOCOL_GUID', 'EfiSmmConfigurationProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/MmConfiguration.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/SmmConfiguration.h",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = ['EfiSmmConfigurationProtocol', 'EfiSmmConfigurationProtocolGuid', 'EFI_SMM_CONFIGURATION_PROTOCOL_GUID']),

    "26f8ab01-d3cd-489c-984f-dfdef768395b": EfiGuid(
        "26f8ab01-d3cd-489c-984f-dfdef768395b", "PEI_STATUS_CODE_MEMORY_PPI_GUID",
        description = "",
        aliases = ['PeiStatusCodeMemoryPpi'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26fd847a-dc93-4d93-917c-6041a3856cbc": EfiGuid(
        "26fd847a-dc93-4d93-917c-6041a3856cbc", "FchHuangshanDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "26fdeb7e-b8af-4ccf-aa97-02633ce48ca7": EfiGuid(
        "26fdeb7e-b8af-4ccf-aa97-02633ce48ca7", "EFI_PROCESSOR_SUBCLASS_GUID",
        description = "",
        aliases = ['EfiProcessorSubClassGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2700f72f-e0ea-4767-9a1e-d172f0704778": EfiGuid(
        "2700f72f-e0ea-4767-9a1e-d172f0704778", "PeiSelStatusCode",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "270279d6-2554-47be-97d0-6ae5ad18b973": EfiGuid(
        "270279d6-2554-47be-97d0-6ae5ad18b973", "OemEeprom",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2707e46d-dbd7-41c2-9c04-c9fdb8bad86c": EfiGuid(
        "2707e46d-dbd7-41c2-9c04-c9fdb8bad86c", "JpegDecoderDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2709a182-0dda-4a91-ba53-11032b713633": EfiGuid(
        "2709a182-0dda-4a91-ba53-11032b713633", "AmiPspOfbd",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "271565a5-6e05-4948-bae7-39af389ffb97": EfiGuid(
        "271565a5-6e05-4948-bae7-39af389ffb97", "DMBMPEI",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2717997b-abac-4733-adaf-9c19d8fbb7a1": EfiGuid(
        "2717997b-abac-4733-adaf-9c19d8fbb7a1", "CbmrDriver",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2719f233-dd1a-abcd-a9dc-aae07b6788f2": EfiGuid(
        "2719f233-dd1a-abcd-a9dc-aae07b6788f2", "SECURITY_ERROR_HANLDING_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "271b424e-a4cc-4e0e-90a2-7ea4841f12f3": EfiGuid(
        "271b424e-a4cc-4e0e-90a2-7ea4841f12f3", "ExportHiiDb",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "271dd6f2-54cb-45e6-8585-8c923c1ac706": EfiGuid(
        "271dd6f2-54cb-45e6-8585-8c923c1ac706", "EFI_PCH_S3_IMAGE_GUID",
        description = "",
        aliases = ['PchS3Peim'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['EFI_PCH_S3_IMAGE_GUID', 'PchS3Peim'],
        hashes = [
            '0fb0d5db7a0d624b39b616db11024c0a', '186cc72a8ef4d9cb1062b2cea44ef726',
            '3dbf73930867371066435e0908e50535', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '73c9d09abcd1f4bf7ddbfe437773286e', '756c574dcde55419446fb64425ebd1ff',
            '7dffb9d276837abbf9e319d968a966a1', '81460325c53ef849a2e3c2a5e104304b',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9eb1ebc7405f93821d9e87833372bee9', 'a6b226c54bf1345a1d1823552f2c15ed',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c39c99f8e0d4a8336f2ead75ebb1966c', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd293013d9f585157787fce8c228e8897',
            'd805f342c43e187512b4b9fc30b58ed3', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ea61df93fb3257dc78c2415817c1f871',
            'ebc573c6530cb4d022d1a02b8a8bf6ed', 'ee8ffebc11a82e29543a142d51a8797d',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = ['PchS3Peim']),

    "271f1343-20d6-4e14-9b62-3c0297f56f07": EfiGuid(
        "271f1343-20d6-4e14-9b62-3c0297f56f07", "SmmPowerManagement",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2721587e-2ae8-43a0-aaeb-19dda16c7764": EfiGuid(
        "2721587e-2ae8-43a0-aaeb-19dda16c7764", "FlexIoPortConfigSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27270327-d3ec-4237-974d-d71abb2ebfea": EfiGuid(
        "27270327-d3ec-4237-974d-d71abb2ebfea", "FjMasterPasswordDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27287751-867a-46cd-91af-416ff6ae3dfe": EfiGuid(
        "27287751-867a-46cd-91af-416ff6ae3dfe", "DELL_SYSTEM_ID_CONFIG_DXE_GUID",
        description = "",
        aliases = ['DellSystemIdConfigDxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "272ac110-0b60-4d07-a558-6d73e2438595": EfiGuid(
        "272ac110-0b60-4d07-a558-6d73e2438595", "PERF_SA_SEC_LOCK_POST_MEM_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "272c766a-654b-4e1f-96b4-398577b96376": EfiGuid(
        "272c766a-654b-4e1f-96b4-398577b96376", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "273820db-55d9-5b44-93fc-6a4ec138eec6": EfiGuid(
        "273820db-55d9-5b44-93fc-6a4ec138eec6", "DellSpecialBootControlDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "273d3086-65a3-463d-b5d6-aac7ccc18445": EfiGuid(
        "273d3086-65a3-463d-b5d6-aac7ccc18445", "PlatformTcg2Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "274365ef-0893-480c-9160-472a26d2dc0f": EfiGuid(
        "274365ef-0893-480c-9160-472a26d2dc0f", "FjGabiGpioAbstraction",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27470dba-64dd-4c71-9ce5-5e3272207598": EfiGuid(
        "27470dba-64dd-4c71-9ce5-5e3272207598", "SioSmiSmbAlertDispatcher",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "274f0c8f-9e57-41d8-9966-29ccd48d31c2": EfiGuid(
        "274f0c8f-9e57-41d8-9966-29ccd48d31c2", "SmmAccess",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2755590c-6f3c-42fa-9ea4-a3ba543cda25": EfiGuid(
        "2755590c-6f3c-42fa-9ea4-a3ba543cda25", "EFI_DEBUG_SUPPORT_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiDebugSupportProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/DebugSupport.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "27584f62-1a26-41ba-85be-74c0eeff46ae": EfiGuid(
        "27584f62-1a26-41ba-85be-74c0eeff46ae", "LANMacInfoDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27587b71-37f9-4a48-b570-58b6d14f6dec": EfiGuid(
        "27587b71-37f9-4a48-b570-58b6d14f6dec", "DELL_OROM_KB_ACCESS_GUID",
        description = "",
        aliases = ['DellOromKbAccess'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27592cb0-97cb-4e36-b0fe-804e389714b4": EfiGuid(
        "27592cb0-97cb-4e36-b0fe-804e389714b4", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['002681fae113306a057042387f62f790'],
        junk = []),

    "276e96ab-6101-4979-9980-e85bfd69102b": EfiGuid(
        "276e96ab-6101-4979-9980-e85bfd69102b", "Common_AcpiDriverSupport",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27723f8b-25a8-4da1-a3fc-7b30e9871dc7": EfiGuid(
        "27723f8b-25a8-4da1-a3fc-7b30e9871dc7", "SmcOobPlatformPolicy",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27755d2d-83df-4916-bdfe-4ff3cd657965": EfiGuid(
        "27755d2d-83df-4916-bdfe-4ff3cd657965", "POSTCODE0A_RGBKBCTRL_DXE",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2785ed8a-795f-43a1-ae2d-6bffeccba646": EfiGuid(
        "2785ed8a-795f-43a1-ae2d-6bffeccba646", "AcpiUpdateCpuInfo",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2786b637-5e93-4e57-b1d4-b69d8d496580": EfiGuid(
        "2786b637-5e93-4e57-b1d4-b69d8d496580", "FjSysmanTeutatesSmmFjFextBin",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "278d0017-1f07-4f5e-a3ec-21d04dcc3a6f": EfiGuid(
        "278d0017-1f07-4f5e-a3ec-21d04dcc3a6f", "IncompatiblePciDevice",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "279851d7-65a8-4009-a5b8-1f56bab7e99d": EfiGuid(
        "279851d7-65a8-4009-a5b8-1f56bab7e99d", "AmdMemAm5Dxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2799a453-fc10-4334-9e66-d3d6ef09d0ee": EfiGuid(
        "2799a453-fc10-4334-9e66-d3d6ef09d0ee", "AmiTseOemPortingVar5Guid",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "279d52cb-5bd9-4bf7-99ed-5d365f73bebb": EfiGuid(
        "279d52cb-5bd9-4bf7-99ed-5d365f73bebb", "AmdMiniRasRplServiceDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "279db133-b4c6-4a34-9be3-c98326dc27dc": EfiGuid(
        "279db133-b4c6-4a34-9be3-c98326dc27dc", "LEMHddSecurityEndProtocolHook",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27a5159d-5e61-4809-919a-422e887101ef": EfiGuid(
        "27a5159d-5e61-4809-919a-422e887101ef", "PcatSingleSegmentPciCfgPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27a71b1e-73ee-43d6-ace3-521a2dc5d092": EfiGuid(
        "27a71b1e-73ee-43d6-ace3-521a2dc5d092", "REPUBLISH_SEC_PPI_PPI_GUID",
        description = "",
        aliases = ['gRepublishSecPpiPpiGuid', 'RepublishSecPpiPpi'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/Include/Ppi/RepublishSecPpi.h",
            "https://github.com/tianocore/edk2/blob/master/UefiCpuPkg/UefiCpuPkg.dec",
        ],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "27a95d13-15fb-4a2e-91e2-c784bf0d20d3": EfiGuid(
        "27a95d13-15fb-4a2e-91e2-c784bf0d20d3", "PersistenceConfigDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27a9836b-cd74-4db2-9a8e-8a4fd52180b7": EfiGuid(
        "27a9836b-cd74-4db2-9a8e-8a4fd52180b7", "NvmeDeviceService",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27aabeca-4bc5-4e6e-980b-bbd6627da20e": EfiGuid(
        "27aabeca-4bc5-4e6e-980b-bbd6627da20e", "TypeCWmiMessages",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27abf055-b1b8-4c26-8048-748f37baa2df": EfiGuid(
        "27abf055-b1b8-4c26-8048-748f37baa2df", "EFI_EVENT_EXIT_BOOT_SERVICES_GUID",
        description = "",
        aliases = ['EfiEventExitBootServicesGuid'],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "27b1960a-ee02-47a8-87b0-1222f119257f": EfiGuid(
        "27b1960a-ee02-47a8-87b0-1222f119257f", "FchTaishanPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27ba7e67-d54d-4983-be18-9df07d2389dc": EfiGuid(
        "27ba7e67-d54d-4983-be18-9df07d2389dc", "SystemNvmeServiceOwnerDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27beda18-ae2b-43c2-af6b-74952441de28": EfiGuid(
        "27beda18-ae2b-43c2-af6b-74952441de28", "MonitorKeyDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27c10b10-114a-4d71-8c58-d67ed0f20d89": EfiGuid(
        "27c10b10-114a-4d71-8c58-d67ed0f20d89", "OemVGA",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27c4686b-32b8-4bec-a499-c5ba75639fc9": EfiGuid(
        "27c4686b-32b8-4bec-a499-c5ba75639fc9", "RsaSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27cfac87-46cc-11d4-9a38-0090273fc14d": EfiGuid(
        "27cfac87-46cc-11d4-9a38-0090273fc14d", "EFI_REAL_TIME_CLOCK_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiRealTimeClockArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/RealTimeClock.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_REAL_TIME_CLOCK_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "27cfac88-46cc-11d4-9a38-0090273fc14d": EfiGuid(
        "27cfac88-46cc-11d4-9a38-0090273fc14d", "EFI_RESET_ARCH_PROTOCOL_GUID",
        description = "",
        aliases = ['EfiResetArchProtocolGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/MdePkg/Include/Protocol/Reset.h",
            "https://github.com/tianocore/edk2/blob/master/MdePkg/MdePkg.dec",
            "https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h",
        ],
        vendors = ['EDK2'],
        filenames = ['EFI_RESET_ARCH_PROTOCOL_GUID'],
        hashes = [
            '7f662f93e41d560c5fb667dce9000eb7', '8091cabbb4de557d84e3dbd55126455a',
            '811505684bd99c18f5ecbee1f725ff66', '81460325c53ef849a2e3c2a5e104304b',
            '82060a1dae992f24609d2f673cb58cfb', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9e80820994976fe9edc2b53346ea4f71',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a9258521b7c65199bd268280867cc628',
            'a9f4f993c8cee3cd83fecdb334fdf666', 'aa8c4ab0a8c8bfc2ef11abfd93537b7e',
            'ac2383f4616ea50d075c37b988c3cb83', 'b218fecd176e986c1491046e58c9e01b',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b4fa3a413d748be6699112adc0c14b0d', 'b68b9a378d9f7e39c59e5c68771086c8',
            'b69bc34b7dc16a8f568aa0bac8c0013e', 'b6cce665f85751125c3db2050ffc98d2',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c5d29ea3bb1b6fc4a17c65126809fdd8',
            'c7013ed5b01fbe5cec1e0f7afcaccfee', 'c7e102ca6520f991d823aa784d64ae46',
            'cae37d13982bb26c35ffaada2841d171', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbc0e5e827cb6cd1a7e9156b42ce19c2', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd14eaff2e32dac7bc295a2f39f3a4ba',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ecd17f3e5e1dff15b7cff5d95b21f20c',
            'ee40c5722cf3b43608a71e2a12c72193', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "27d67720-ea68-48ae-93da-a3a074c90e30": EfiGuid(
        "27d67720-ea68-48ae-93da-a3a074c90e30", "BaseLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/BaseLib/BaseLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "27d71940-d1f7-40f2-aa19-18c2b0564462": EfiGuid(
        "27d71940-d1f7-40f2-aa19-18c2b0564462", "FjSysmanAmphionS3Resume",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27daee12-488e-4e98-adcf-38adc10a6b68": EfiGuid(
        "27daee12-488e-4e98-adcf-38adc10a6b68", "H19ServiceBodySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27dc3f10-ca4b-42d9-99f0-161ae542799d": EfiGuid(
        "27dc3f10-ca4b-42d9-99f0-161ae542799d", "GraphicAreas",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['GraphicAreas'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "27de9eb2-73b4-43b2-abd0-e87a5965424e": EfiGuid(
        "27de9eb2-73b4-43b2-abd0-e87a5965424e", "HddStandBySmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27e569d5-0afc-4d8f-8c90-783ac4a318ab": EfiGuid(
        "27e569d5-0afc-4d8f-8c90-783ac4a318ab", "SaAcpiTables",
        description = "",
        aliases = [],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['SaAcpiTables'],
        hashes = [
            '7dffb9d276837abbf9e319d968a966a1', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            'a6b226c54bf1345a1d1823552f2c15ed', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c7e102ca6520f991d823aa784d64ae46', 'cb6725610355d5dfad06ffc8bb80b931',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd293013d9f585157787fce8c228e8897', 'd4a0cd284c2f726df5db17c605809651',
            'd805f342c43e187512b4b9fc30b58ed3', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd47783067bb28fe370a065d80d103b6', 'dd579ab31868fff1b32afe1997484822',
            'df960d692cee44984c4b6e0b88e62b49', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ee8ffebc11a82e29543a142d51a8797d', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'fb738b9beaae1160994619f381a30d71',
        ],
        junk = []),

    "27e94a0b-6e3c-4b4b-b876-176ae521ce60": EfiGuid(
        "27e94a0b-6e3c-4b4b-b876-176ae521ce60", "DellHotSosDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27eedf2d-469e-478c-805e-993ccfb5c0c1": EfiGuid(
        "27eedf2d-469e-478c-805e-993ccfb5c0c1", "CrbPxeLoader",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27eeec71-d849-4b64-a04b-5643cceac876": EfiGuid(
        "27eeec71-d849-4b64-a04b-5643cceac876", "FchKunlunPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27f05af5-1644-4ef4-8944-48c4f75675a0": EfiGuid(
        "27f05af5-1644-4ef4-8944-48c4f75675a0", "RealTimeClockDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = ['RealTimeClock']),

    "27f3838f-ba27-4ffa-b374-35bea28a431b": EfiGuid(
        "27f3838f-ba27-4ffa-b374-35bea28a431b", "DELL_SMSC5544_DXE_GUID",
        description = "",
        aliases = ['DellSmsc5544Dxe'],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27f4917b-a707-4aad-9676-26df168cbf0d": EfiGuid(
        "27f4917b-a707-4aad-9676-26df168cbf0d", "LENOVO_PCH_SPI_SMM_GUID",
        description = "",
        aliases = ['PchSpiSmm'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = ['LENOVO_PCH_SPI_SMM_GUID', 'PchSpiSmm'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '136ef7c1715a5ab03a93c66f1f9447e1',
            '15c3afd565e35a7b87854f203ea15633', '1ded807f42af76804c9a0bbb6e24a2a9',
            '1f0bf94fb4f5484c39510715a328e0e9', '20f1da0b4063ef2319439a473c0d4ae3',
            '2949649fbed4688a06e89bf439cea6a1', '2cd4cb81c4b865a8751fba6f051a8e9a',
            '2f84c5a8e276b66cc9d014789977cacb', '30beb84b9d76fb63220a1cf1dc6e7c23',
            '35a55f8ce610ca18a6d9480a0fc484df', '4002d2fb07cfbaed2e86d9608824378a',
            '4afaa764f57449bc440e58fa11422e02', '5163727ab3ed5f9e02330a0c98095da0',
            '540ab45d79fb417ade67cb3aac048180', '54a655d3479750be95c150698a9e80ff',
            '54b7b82c79116e989585483d47eff1b4', '567f0a47affb9a5e87603a8c7d3239a5',
            '60785be6bfb0d1daa94a8bb214b774d1', '61205618135add3dd7e62e5bc90886a4',
            '6f1b81ae0adfe7f90c141e463e9e9344', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '82ae1a72660ab3fd9be7c1add0a512f9',
            '85e7ac10a2c876e04f9714b035fc8be0', '869f701de5296248126c8c8a03a479c3',
            '86c4a8e759a9aa0eee18a5e7d480583f', '87e7da52d3a3153f4372f3c998ba7ce8',
            '894d4ba28352c4328fc369d9aaaecb36', '8a42c7b29e7fcf02022c68ff430e9ed7',
            '8add95e4f71aa9680d87091104649f78', '9138917b3dae616958881ae2a907dfe1',
            '93afd415682ff830ae424c88ae1c9df9', '953dc1131b7a579176b9fc937eee5846',
            '9a05d9b82e4f2c467423be3959175608', '9c78983f2bf3fb5dfbf9652704b6322f',
            '9cf5adeb713d71eb03527d7e52165d5d', '9e80820994976fe9edc2b53346ea4f71',
            '9eb1ebc7405f93821d9e87833372bee9', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b21bd44978f0935538daa78003d23135',
            'b2a8331c64f95dba9412e6e0abbe51ef', 'b49cea81c680d9848212ed2aca8e6771',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b930f337647cc93e2809f678c893ad69', 'ba61ae91d36d08fd8787d6912e134591',
            'bc3eee79cadf2f7f815bf470a375b7ec', 'bcf062b22e970ae13f2825c8596635e6',
            'bdbe423029a61611b078326db4953356', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbd75246fd41b231527edfcacc8d5803',
            'cc2f312578af11ec3f6a43710e9a56c5', 'cc80253042928592caf3fafdd1915db0',
            'cccaf2fb68cd9dd0b4060e8c020d76c8', 'cd7db3eb49885c95779ed44627512a82',
            'cfba6c85608b8135fb21024783573bf6', 'd09f3cd0ba6435ccf1611206ed293c92',
            'd1b0f1bd9731a794ff1ca36ab45c7429', 'd293013d9f585157787fce8c228e8897',
            'd43db2fbdf6c056d2e0f0d00d4037c6c', 'd4a0cd284c2f726df5db17c605809651',
            'd7f470107292518544abfb6d2f80a513', 'd805f342c43e187512b4b9fc30b58ed3',
            'da95b83695eb74e17c88577b3919577a', 'dab68811108b555173f8f096c9737d20',
            'dbcaa4b015b1aec94558985a97dce7c2', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e2a9a88ae42a7df98ad75370c6a85c51', 'e7422cc52fc27bfd199a2acfafb80b46',
            'e7a706c2421a55cde70e476d1d37f831', 'e972ecdd1f3cb9c9a06e7d9fe9844d4a',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f1cb9f7d3dff3bfe3827aa02f7050cbe', 'f3a7c093df5e8da1491e4b41973919a7',
            'fb738b9beaae1160994619f381a30d71', 'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "27f51949-1577-4cf6-b2e2-ae9392a4ebb7": EfiGuid(
        "27f51949-1577-4cf6-b2e2-ae9392a4ebb7", "BootPriority",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27f85559-359f-4b25-9b73-3ee5de399121": EfiGuid(
        "27f85559-359f-4b25-9b73-3ee5de399121", "DellSmbusHcSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "27f9093f-527a-42ab-ae55-5c56da8d9ab8": EfiGuid(
        "27f9093f-527a-42ab-ae55-5c56da8d9ab8", "BootOrderDefaultSettings",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "280251c4-1d09-4035-9062-839acb5f18c1": EfiGuid(
        "280251c4-1d09-4035-9062-839acb5f18c1", "CpuMpPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "280bcc15-2cd7-467c-9647-1b30307d7ed5": EfiGuid(
        "280bcc15-2cd7-467c-9647-1b30307d7ed5", "AmiFriVariable",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "280da667-74b6-410e-8acd-d06c3f32ea20": EfiGuid(
        "280da667-74b6-410e-8acd-d06c3f32ea20", "IntelLANPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "280e42c3-826e-4573-9772-b74ef1086d95": EfiGuid(
        "280e42c3-826e-4573-9772-b74ef1086d95", "UefiScsiLib",
        description = "",
        aliases = [],
        urls = ['https://github.com/tianocore/edk2/blob/master/MdePkg/Library/UefiScsiLib/UefiScsiLib.inf'],
        vendors = ['EDK2'],
        filenames = [],
        hashes = [],
        junk = []),

    "2818256a-6bdb-4871-993c-95315854012e": EfiGuid(
        "2818256a-6bdb-4871-993c-95315854012e", "EdidOverrideDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2820908b-87f6-446b-a100-74eee36e2918": EfiGuid(
        "2820908b-87f6-446b-a100-74eee36e2918", "",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = ['0ebecbce655f6f780edb7d73c1d80d13'],
        junk = []),

    "28260fd2-975f-48b4-b1a8-8f7919f55746": EfiGuid(
        "28260fd2-975f-48b4-b1a8-8f7919f55746", "AmdOemRasBrhDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "282c8564-c985-4390-88e5-a740c4c5fb7f": EfiGuid(
        "282c8564-c985-4390-88e5-a740c4c5fb7f", "CustomSMBIOS",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = ['CustomSMBIOS'],
        hashes = [
            '002681fae113306a057042387f62f790', '0b8de120c00ed155bbf748f6e8438a2f',
            '0bf3064114ecb05222114c4d8d687938', '0e6eebbc21300ba0ebec9f192d5beabf',
            '0fb0d5db7a0d624b39b616db11024c0a', '136ef7c1715a5ab03a93c66f1f9447e1',
            '13bf9b32863d8378b9c802c9508c96e9', '15c3afd565e35a7b87854f203ea15633',
            '186cc72a8ef4d9cb1062b2cea44ef726', '1a0f212de5a43dc1de07ea5e31e3adb0',
            '1ded807f42af76804c9a0bbb6e24a2a9', '1f0bf94fb4f5484c39510715a328e0e9',
            '20f1da0b4063ef2319439a473c0d4ae3', '28e65e1a6689d8a2a4a612a06cbd2a58',
            '2949649fbed4688a06e89bf439cea6a1', '2b4ca0bd16a20893deff9632c23c2cca',
            '2cd4cb81c4b865a8751fba6f051a8e9a', '2f84c5a8e276b66cc9d014789977cacb',
            '30beb84b9d76fb63220a1cf1dc6e7c23', '35a55f8ce610ca18a6d9480a0fc484df',
            '36d56c567fa4c101f6af6f56bec74299', '3c3abb9a4b4b6d9718f8ac83d8145e92',
            '3dbf73930867371066435e0908e50535', '4002d2fb07cfbaed2e86d9608824378a',
            '409221b30602fc13c8cbedf952c60297', '48bba53caf662f85dd73a1a247b45dc7',
            '4afaa764f57449bc440e58fa11422e02', '4bb7a38c96ed64c4fb149d0ad4ec2d8d',
            '5163727ab3ed5f9e02330a0c98095da0', '540ab45d79fb417ade67cb3aac048180',
            '54a655d3479750be95c150698a9e80ff', '54b7b82c79116e989585483d47eff1b4',
            '567f0a47affb9a5e87603a8c7d3239a5', '60785be6bfb0d1daa94a8bb214b774d1',
            '61205618135add3dd7e62e5bc90886a4', '633e80e7347dc01c21c5133227df2aba',
            '6f1b81ae0adfe7f90c141e463e9e9344', '73c9d09abcd1f4bf7ddbfe437773286e',
            '756c574dcde55419446fb64425ebd1ff', '7a20bc8c8925db2963205a36be03b335',
            '7b3f08b9dac97b5334c6851820b9a1b0', '7d7236b9ad7c65c2e13957e6027586fa',
            '7dffb9d276837abbf9e319d968a966a1', '7f662f93e41d560c5fb667dce9000eb7',
            '8091cabbb4de557d84e3dbd55126455a', '811505684bd99c18f5ecbee1f725ff66',
            '81460325c53ef849a2e3c2a5e104304b', '82060a1dae992f24609d2f673cb58cfb',
            '82ae1a72660ab3fd9be7c1add0a512f9', '85e7ac10a2c876e04f9714b035fc8be0',
            '869f701de5296248126c8c8a03a479c3', '86c4a8e759a9aa0eee18a5e7d480583f',
            '87e7da52d3a3153f4372f3c998ba7ce8', '894d4ba28352c4328fc369d9aaaecb36',
            '8a42c7b29e7fcf02022c68ff430e9ed7', '8add95e4f71aa9680d87091104649f78',
            '9138917b3dae616958881ae2a907dfe1', '93afd415682ff830ae424c88ae1c9df9',
            '953dc1131b7a579176b9fc937eee5846', '97012cbeef60b7e6dc93c362aa6fe780',
            '97070b6e86ea62d42df9c0d42179c050', '9a05d9b82e4f2c467423be3959175608',
            '9c78983f2bf3fb5dfbf9652704b6322f', '9cf5adeb713d71eb03527d7e52165d5d',
            '9e80820994976fe9edc2b53346ea4f71', '9eb1ebc7405f93821d9e87833372bee9',
            '9f603f855d176a1ea5758fa617d3e0d2', 'a12b44297a063d752d860068ef2ce3db',
            'a40b9558d367d413a0126b691b78e006', 'a6b226c54bf1345a1d1823552f2c15ed',
            'a9258521b7c65199bd268280867cc628', 'a9f4f993c8cee3cd83fecdb334fdf666',
            'aa8c4ab0a8c8bfc2ef11abfd93537b7e', 'ac2383f4616ea50d075c37b988c3cb83',
            'ac6277674bc0b5eb2156e1391dfb7143', 'b218fecd176e986c1491046e58c9e01b',
            'b21bd44978f0935538daa78003d23135', 'b2a8331c64f95dba9412e6e0abbe51ef',
            'b49cea81c680d9848212ed2aca8e6771', 'b4fa3a413d748be6699112adc0c14b0d',
            'b68b9a378d9f7e39c59e5c68771086c8', 'b69bc34b7dc16a8f568aa0bac8c0013e',
            'b6cce665f85751125c3db2050ffc98d2', 'b930f337647cc93e2809f678c893ad69',
            'ba61ae91d36d08fd8787d6912e134591', 'bc3eee79cadf2f7f815bf470a375b7ec',
            'bcf062b22e970ae13f2825c8596635e6', 'bdbe423029a61611b078326db4953356',
            'c0bdac2284b1e46c7c9c1509ddbea006', 'c39c99f8e0d4a8336f2ead75ebb1966c',
            'c5d29ea3bb1b6fc4a17c65126809fdd8', 'c7013ed5b01fbe5cec1e0f7afcaccfee',
            'c7e102ca6520f991d823aa784d64ae46', 'cae37d13982bb26c35ffaada2841d171',
            'cb6725610355d5dfad06ffc8bb80b931', 'cbc0e5e827cb6cd1a7e9156b42ce19c2',
            'cbd75246fd41b231527edfcacc8d5803', 'cc2f312578af11ec3f6a43710e9a56c5',
            'cc80253042928592caf3fafdd1915db0', 'cccaf2fb68cd9dd0b4060e8c020d76c8',
            'cd7db3eb49885c95779ed44627512a82', 'cfba6c85608b8135fb21024783573bf6',
            'd09f3cd0ba6435ccf1611206ed293c92', 'd1b0f1bd9731a794ff1ca36ab45c7429',
            'd293013d9f585157787fce8c228e8897', 'd43db2fbdf6c056d2e0f0d00d4037c6c',
            'd4a0cd284c2f726df5db17c605809651', 'd7f470107292518544abfb6d2f80a513',
            'd805f342c43e187512b4b9fc30b58ed3', 'da95b83695eb74e17c88577b3919577a',
            'dab68811108b555173f8f096c9737d20', 'dbcaa4b015b1aec94558985a97dce7c2',
            'dd14eaff2e32dac7bc295a2f39f3a4ba', 'dd47783067bb28fe370a065d80d103b6',
            'dd579ab31868fff1b32afe1997484822', 'df960d692cee44984c4b6e0b88e62b49',
            'e0f49f68b1fc98c1a09b18f35912ef2f', 'e2a9a88ae42a7df98ad75370c6a85c51',
            'e7422cc52fc27bfd199a2acfafb80b46', 'e7a706c2421a55cde70e476d1d37f831',
            'e972ecdd1f3cb9c9a06e7d9fe9844d4a', 'ea61df93fb3257dc78c2415817c1f871',
            'ead4f1b7e116eaed945c115a76621189', 'ebc573c6530cb4d022d1a02b8a8bf6ed',
            'ecd17f3e5e1dff15b7cff5d95b21f20c', 'ee40c5722cf3b43608a71e2a12c72193',
            'ee8ffebc11a82e29543a142d51a8797d', 'eee15c9794a4040563b41004ebccc91a',
            'f199b62c1ea50fa7dc3f10fda43e62a2', 'f1cb9f7d3dff3bfe3827aa02f7050cbe',
            'f3a7c093df5e8da1491e4b41973919a7', 'f8c34791a7a721ee330771cb43d68b91',
            'f9917106a00ba5725ae8e30c4becb2c7', 'fb738b9beaae1160994619f381a30d71',
            'fce5101ae773ca70044c975c298bb274',
        ],
        junk = []),

    "28324ef7-6bb3-4bab-a2ec-18d5f7940f23": EfiGuid(
        "28324ef7-6bb3-4bab-a2ec-18d5f7940f23", "FjMaptDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "28374747-76ff-41b3-9740-381efaef13bc": EfiGuid(
        "28374747-76ff-41b3-9740-381efaef13bc", "PspPlatform",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2837c020-83f6-11df-8395-0800200c9a66": EfiGuid(
        "2837c020-83f6-11df-8395-0800200c9a66", "SYSTEM_AHCI_BUS_SMM_PROTOCOL_GUID",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "283c945f-3ff5-4588-ad80-86e8708b942c": EfiGuid(
        "283c945f-3ff5-4588-ad80-86e8708b942c", "DellSimulatedECPei",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "283cad13-a151-4d55-be2d-96ea57392a82": EfiGuid(
        "283cad13-a151-4d55-be2d-96ea57392a82", "UefiLanguageLib",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "283d62be-957d-4863-8041-7e9c22201709": EfiGuid(
        "283d62be-957d-4863-8041-7e9c22201709", "LfcNvsAcpi",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "283e7ad9-effa-468b-8289-887347d23300": EfiGuid(
        "283e7ad9-effa-468b-8289-887347d23300", "FjCardReaderSmm",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "283fa2ee-532c-484d-9383-9f93b36f0b7e": EfiGuid(
        "283fa2ee-532c-484d-9383-9f93b36f0b7e", "EFI_UPDATE_DATA_FILE_GUID",
        description = "",
        aliases = ['EfiUpdateDataFileGuid'],
        urls = ['https://raw.githubusercontent.com/mazzoo/efiguid/master/efi_guid_list.h'],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "28451aa4-b4c4-4aa9-be3a-1bbcc2e5553a": EfiGuid(
        "28451aa4-b4c4-4aa9-be3a-1bbcc2e5553a", "GptRecovery",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2846412b-f50f-4a5b-8c8b-76644324aa66": EfiGuid(
        "2846412b-f50f-4a5b-8c8b-76644324aa66", "FchSmbusDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2848a12f-3f86-47c7-81d8-d3fd47b205b2": EfiGuid(
        "2848a12f-3f86-47c7-81d8-d3fd47b205b2", "CbsSetupDxeRPL",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "28638cfa-ea88-456c-92a5-f249ca488535": EfiGuid(
        "28638cfa-ea88-456c-92a5-f249ca488535", "EfiRedirFruProtocol",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "2864ca7a-632c-456d-a4fb-b5b718566956": EfiGuid(
        "2864ca7a-632c-456d-a4fb-b5b718566956", "ReportStatusCodeRouterRuntimeDxe",
        description = "",
        aliases = [],
        urls = [],
        vendors = [],
        filenames = [],
        hashes = [],
        junk = []),

    "286bf25a-c2c3-408c-b3b4-25e6758b7317": EfiGuid(
        "286bf25a-c2c3-408c-b3b4-25e6758b7317", "TPM_DEVICE_INTERFACE_TPM20_DTPM",
        description = "",
        aliases = ['Tpm2InstanceLibDTpm', 'EfiTpmDeviceInstanceTpm20DtpmGuid'],
        urls = [
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/Include/Guid/TpmInstance.h",
            "https://github.com/tianocore/edk2/blob/master/SecurityPkg/Library/Tpm2DeviceLibDTpm/Tpm2