"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LoadBalancer);
            }
            throw error;
        }
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: props.crossZone ?? true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addListener);
            }
            throw error;
        }
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_ILoadBalancerTarget(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTarget);
            }
            throw error;
        }
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancing.LoadBalancer", version: "1.170.0" };
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "@aws-cdk/aws-elasticloadbalancing.ListenerPort", version: "1.170.0" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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