"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'create a subscription'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E'
            }
        }));
        test.done();
    },
    'with filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    whitelist: ['red', 'green'],
                    blacklist: ['white', 'orange'],
                    matchPrefixes: ['bl', 'ye'],
                }),
                price: sns.SubscriptionFilter.numericFilter({
                    whitelist: [100, 200],
                    between: { start: 300, stop: 350 },
                    greaterThan: 500,
                    lessThan: 1000,
                    betweenStrict: { start: 2000, stop: 3000 },
                    greaterThanOrEqualTo: 1000,
                    lessThanOrEqualTo: -2,
                })
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                color: [
                    'red',
                    'green',
                    { 'anything-but': ['white', 'orange'] },
                    { prefix: 'bl' },
                    { prefix: 'ye' }
                ],
                price: [
                    { numeric: ['=', 100] },
                    { numeric: ['=', 200] },
                    { numeric: ['>', 500] },
                    { numeric: ['>=', 1000] },
                    { numeric: ['<', 1000] },
                    { numeric: ['<=', -2] },
                    { numeric: ['>=', 300, '<=', 350] },
                    { numeric: ['>', 2000, '<', 3000] },
                ]
            },
        }));
        test.done();
    },
    'with existsFilter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                size: sns.SubscriptionFilter.existsFilter(),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                size: [{ exists: true }]
            },
        }));
        test.done();
    },
    'throws with raw delivery for protocol other than http, https or sqs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            rawMessageDelivery: true
        }), /Raw message delivery/);
        test.done();
    },
    'throws with more than 5 attributes in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const cond = { conditions: [] };
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: cond,
                b: cond,
                c: cond,
                d: cond,
                e: cond,
                f: cond,
            },
        }), /5 attribute names/);
        test.done();
    },
    'throws with more than 100 conditions in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: { conditions: [...Array.from(Array(2).keys())] },
                b: { conditions: [...Array.from(Array(10).keys())] },
                c: { conditions: [...Array.from(Array(6).keys())] },
            },
        }), /\(120\) must not exceed 100/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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