import pandas as pd
import glob
from fpdf import FPDF
from pathlib import Path
import os

def generate(invoices_path, pdfs_path, image_path, product_id, product_name, amount_purchased, price_per_unit, total_price):
    """
    Generate PDF invoices from Excel files.
    :param invoices_path: Path to the folder containing the Excel invoices.
    :param pdfs_path: Path to the folder where the PDF invoices will be saved.
    :param image_path: Path to the logo image.
    :param product_id: Column name for the product ID.
    :param product_name: Column name for the product name.
    :param amount_purchased: Column name for the amount purchased.
    :param price_per_unit: Column name for the price per unit.
    :param total_price: Column name for the total price.
    """
    
    filepaths = glob.glob(f"{invoices_path}/*.xlsx")
    print(filepaths)

    for filepath in filepaths:
        pdf = FPDF(orientation="P", unit="mm", format="A4")
        pdf.add_page()

        filename = Path(filepath).stem
        invoice_nr, date = filename.split("-")

        pdf.set_font(family="Times", style="B", size=16)
        pdf.cell(w=50, h=8, txt = f"Invoice Nr. {invoice_nr}", ln=1)

        pdf.set_font(family="Times", style="B", size=16)
        pdf.cell(w=50, h=8, txt = f"Date: {date}", ln=1)

        df = pd.read_excel(filepath, sheet_name="Sheet 1")

        # Add header
        columns = df.columns
        columns = [item.replace("_", " ").title() for item in columns]
        pdf.set_font(family="Times", size=10, style="B")
        pdf.cell(w=30, h=8, txt=columns[0], border = 1)
        pdf.cell(w=70, h=8, txt=columns[1], border = 1)
        pdf.cell(w=32, h=8, txt=columns[2], border = 1)
        pdf.cell(w=30, h=8, txt=columns[3], border = 1)
        pdf.cell(w=30, h=8, txt=columns[4], border = 1, ln=1)

        # Add rows
        for index, row in df.iterrows():
            pdf.set_font(family="Times", size=10)
            pdf.set_text_color(80, 80, 80)
            pdf.cell(w=30, h=8, txt=str(row[product_id]), border = 1)
            pdf.cell(w=70, h=8, txt=str(row[product_name]), border = 1)
            pdf.cell(w=32, h=8, txt=str(row[amount_purchased]), border = 1)
            pdf.cell(w=30, h=8, txt=str(row[price_per_unit]), border = 1)
            pdf.cell(w=30, h=8, txt=str(row[total_price]), border = 1, ln=1)

        final_total_price = df['total_price'].sum()
        pdf.cell(w=30, h=8, txt="", border = 1)
        pdf.cell(w=70, h=8, txt="", border = 1)
        pdf.cell(w=32, h=8, txt="", border = 1)
        pdf.cell(w=30, h=8, txt="", border = 1)
        pdf.cell(w=30, h=8, txt=str(final_total_price), border = 1, ln=1)

        # Total price sentence
        pdf.set_font(family="Times", size=10, style="B")
        pdf.cell(w=30, h=8, txt=f"The total price is {final_total_price}", ln=1)

        # Add company name and logo
        pdf.set_font(family="Times", size=14, style="B")
        pdf.cell(w=25, h=8, txt=f"PythonHow")
        pdf.image(image_path, w=10)

        os.makedirs(pdfs_path, exist_ok=True)
        pdf.output(f"{pdfs_path}/{filename}.pdf")
