"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToIotProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        extraApiGwProps = defaults.consolidateProps(extraApiGwProps, props.apiGatewayProps);
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        this.addResourceMethod(namedShadowResource, props, 'things/{thingName}/shadow?name={shadowName}', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        if (props.apiGatewayCreateApiKey === true) {
            // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
            defaults.addCfnSuppressRules(apiMethod, [
                {
                    id: "W59",
                    reason: "When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE",
                },
            ]);
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "1.155.0" };
//# sourceMappingURL=data:application/json;base64,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