"""
file: sample_directed_test.py
brief: Directed test case for specific scenarios using PyUVM

Module:      sample_design
Design File: sample_design.v
Project:     test_project

Description:
    Executes directed test scenarios with specific data patterns
    to verify corner cases, boundary conditions, and protocol
    compliance using PyUVM framework.

Test Scenarios:
    1. Minimum/maximum value handling
    2. Zero crossing behavior  
    3. Back-to-back transactions
    4. Protocol violations (negative testing)
"""

from pyuvm import *
import cocotb
from cocotb.triggers import Timer, RisingEdge
import logging

class SampleDirectedTest(uvm_test):
    """
    Directed Test Case
    
    Executes specific test scenarios with predetermined data
    patterns and timing to verify corner cases and boundary
    conditions.
    
    Test Coverage:
        - Boundary value testing (min/max)
        - Zero value handling
        - Back-to-back transaction timing
        - Protocol compliance checking
    """
    
    def __init__(self, name, parent):
        """
        Initialize directed test
        
        Args:
            name (str): Test name
            parent: Parent component
        """
        super().__init__(name, parent)
        self.env = None
        self.cfg = None
        
    def build_phase(self):
        """
        Build phase - creates environment with directed configuration
        
        Sets up test-specific configuration for directed scenarios.
        """
        super().build_phase()
        
        # Create configuration for directed test
        self.cfg = SampleCfg("cfg")
        self.cfg.num_transactions = 50  # Fewer transactions for directed test
        self.cfg.enable_errors = False  # No random errors
        self.cfg.randomize_delays = False  # Fixed timing
        
        # Store in ConfigDB
        ConfigDB().set(None, "*", "cfg", self.cfg)
        
        # Create environment
        self.env = SampleEnv("env", self)
        
        logging.info(f"Built {self.get_name()} for directed scenarios")
        
    async def run_phase(self):
        """
        Run phase - executes directed test sequences
        
        Runs multiple directed sequences testing specific scenarios.
        """
        self.raise_objection()
        
        logging.info(f"Starting {self.get_name()}")
        
        # Test Scenario 1: Boundary values
        logging.info("Scenario 1: Testing boundary values")
        seq = SampleBoundarySeq("boundary_seq")
        await seq.start(self.env.input_agent.sequencer)
        
        await Timer(100, units='ns')
        
        # Test Scenario 2: Zero crossing
        logging.info("Scenario 2: Testing zero crossing")
        seq = SampleZeroSeq("zero_seq")
        await seq.start(self.env.input_agent.sequencer)
        
        await Timer(100, units='ns')
        
        # Test Scenario 3: Back-to-back transactions
        logging.info("Scenario 3: Testing back-to-back transactions")
        seq = SampleBackToBackSeq("b2b_seq")
        await seq.start(self.env.input_agent.sequencer)
        
        await Timer(100, units='ns')
        
        logging.info(f"Completed {self.get_name()}")
        
        self.drop_objection()
        
    def report_phase(self):
        """
        Report phase - prints detailed test results
        
        Displays results for each test scenario.
        """
        super().report_phase()
        
        logging.info(f"\n{'='*60}")
        logging.info(f"Directed Test: {self.get_name()}")
        logging.info(f"All scenarios completed successfully")
        logging.info(f"{'='*60}\n")
