"""
file: sample_cfg.py
brief: Configuration object for sample PyUVM environment

class SampleCfg

brief: Configuration class for verification environment

::

    Module       : sample_design
    Design File  : sample_design.v
    Project      : test_project

Provides parameterized configuration knobs controlling stimulus generation,
agent activation, scoreboarding, and coverage collection.

Parameters:
    data_width (int): Width of data bus (default: 32)
    num_transactions (int): Number of transactions to generate (default: 100)
"""

from pyuvm import *

class SampleCfg(uvm_object):
    """
    Configuration Object
    
    Contains all configuration parameters for the verification environment.
    
    Attributes:
        is_active (bool): Enable active driving (True) or passive monitoring (False)
        num_transactions (int): Total number of transactions to generate
        data_width (int): Width of data bus in bits
        enable_coverage (bool): Enable functional coverage collection
        enable_scoreboard (bool): Enable scoreboard checking
        timeout_ns (int): Test timeout in nanoseconds
        enable_errors (bool): Enable error injection for negative testing
    """
    
    def __init__(self, name="SampleCfg"):
        """
        Initialize configuration with default values
        
        Args:
            name (str): Configuration object name
        """
        super().__init__(name)
        
        # Agent configuration
        self.is_active = True
        
        # Stimulus configuration
        self.num_transactions = 100
        self.data_width = 32
        
        # Environment control
        self.enable_coverage = True
        self.enable_scoreboard = True
        self.timeout_ns = 100000
        
        # Test configuration
        self.enable_errors = False
        self.randomize_delays = True
        self.min_delay = 1
        self.max_delay = 10
        
    def randomize(self):
        """
        Randomize configuration parameters
        
        Returns:
            bool: True if randomization successful
        """
        import random
        
        self.num_transactions = random.randint(50, 200)
        self.min_delay = random.randint(0, 5)
        self.max_delay = random.randint(self.min_delay, 20)
        
        return True
        
    def print_config(self):
        """
        Print current configuration values
        """
        print(f"\n{'='*60}")
        print(f"Configuration: {self.get_name()}")
        print(f"{'='*60}")
        print(f"Active Mode:      {self.is_active}")
        print(f"Transactions:     {self.num_transactions}")
        print(f"Data Width:       {self.data_width}")
        print(f"Coverage Enabled: {self.enable_coverage}")
        print(f"Scoreboard:       {self.enable_scoreboard}")
        print(f"Timeout (ns):     {self.timeout_ns}")
        print(f"{'='*60}\n")
