"""
Documentation generator module - handles Sphinx project setup and building.
"""
import os
import subprocess
import re
import sys
from pathlib import Path
from typing import Optional, List, Set


class DocGenerator:
    """Main documentation generator class."""
    
    def __init__(
        self,
        output_dir: str = "./docs",
        project_name: str = "SystemVerilog Documentation",
        author: str = "Your Name",
        version: str = "1.0",
        theme: str = "sphinx_rtd_theme"
    ):
        """
        Initialize documentation generator.
        
        Args:
            output_dir: Output directory for documentation
            project_name: Project name for documentation
            author: Author name
            version: Project version
            theme: Sphinx HTML theme
        """
        self.output_dir = Path(output_dir)
        self.source_dir = self.output_dir / "source"
        self.build_dir = self.output_dir / "build"
        self.project_name = project_name
        self.author = author
        self.version = version
        self.theme = theme
    
    def detect_python_imports(self, python_dirs: List[str]) -> Set[str]:
        """
        Scan Python files and extract all import statements.
        
        Args:
            python_dirs: List of directories containing Python files
            
        Returns:
            Set of imported module names
        """
        imports = set()
        
        for py_dir in python_dirs:
            if not py_dir:
                continue
                
            py_path = Path(py_dir)
            if not py_path.exists():
                continue
            
            # Find all .py files
            py_files = list(py_path.glob('*.py')) + list(py_path.glob('**/*.py'))
            
            for py_file in py_files:
                try:
                    with open(py_file, 'r', encoding='utf-8', errors='ignore') as f:
                        content = f.read()
                    
                    # Extract import statements
                    # Match: import module, from module import ...
                    import_patterns = [
                        r'^\s*import\s+([a-zA-Z_][a-zA-Z0-9_]*)',  # import module
                        r'^\s*from\s+([a-zA-Z_][a-zA-Z0-9_]*)\s+import',  # from module import
                    ]
                    
                    for pattern in import_patterns:
                        matches = re.finditer(pattern, content, re.MULTILINE)
                        for match in matches:
                            module_name = match.group(1)
                            # Skip relative imports (start with .)
                            if not module_name.startswith('.'):
                                imports.add(module_name)
                
                except Exception as e:
                    print(f"  Warning: Could not parse {py_file}: {e}")
        
        return imports
    
    def install_missing_dependencies(self, python_dirs: List[str], quiet: bool = False):
        """
        Detect and install missing Python dependencies needed for documentation.
        
        Args:
            python_dirs: List of directories containing Python files
            quiet: Suppress output
        """
        # Standard library modules that don't need installation
        stdlib_modules = {
            'os', 'sys', 're', 'math', 'random', 'logging', 'collections', 'enum',
            'typing', 'pathlib', 'subprocess', 'argparse', 'json', 'datetime',
            'time', 'threading', 'multiprocessing', 'queue', 'io', 'itertools',
            'functools', 'copy', 'pickle', 'csv', 'configparser', 'unittest',
            'abc', 'dataclasses', 'warnings', 'tempfile', 'shutil', 'glob'
        }
        
        # Modules that are already dependencies of sphinx-doc
        builtin_deps = {'sphinx', 'pyuvm', 'cocotb'}
        
        # Common package name mappings (PyPI name vs import name)
        package_mappings = {
            'vsc': 'pyvsc',
            'cv2': 'opencv-python',
            'PIL': 'Pillow',
            'yaml': 'pyyaml',
            'serial': 'pyserial',
        }
        
        if not quiet:
            print("🔍 Detecting Python dependencies...")
        
        # Detect imports
        imports = self.detect_python_imports(python_dirs)
        
        # Filter out standard library and built-in dependencies
        external_imports = imports - stdlib_modules - builtin_deps
        
        if not external_imports:
            if not quiet:
                print("✓ No additional dependencies needed")
            return
        
        if not quiet:
            print(f"  Found {len(external_imports)} external imports: {', '.join(sorted(external_imports))}")
        
        # Check which packages are missing
        missing_packages = []
        for module in external_imports:
            try:
                __import__(module)
            except ImportError:
                # Map to PyPI package name if needed
                package_name = package_mappings.get(module, module)
                missing_packages.append(package_name)
        
        if not missing_packages:
            if not quiet:
                print("✓ All dependencies already installed")
            return
        
        # Install missing packages one by one
        if not quiet:
            print(f"📦 Installing {len(missing_packages)} missing package(s)...")
        
        installed = []
        failed = []
        
        for package in missing_packages:
            if not quiet:
                print(f"   Installing {package}...", end=" ")
            try:
                subprocess.check_call(
                    [sys.executable, '-m', 'pip', 'install', package],
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL
                )
                installed.append(package)
                if not quiet:
                    print("✓")
            except subprocess.CalledProcessError:
                failed.append(package)
                if not quiet:
                    print("✗ (not found on PyPI or local module)")
        
        if not quiet:
            if installed:
                print(f"✓ Successfully installed: {', '.join(installed)}")
            if failed:
                print(f"⚠️  Could not install (likely local modules): {', '.join(failed)}")
                print("  These will be mocked during documentation build")
        
    def initialize_sphinx_dirs(self):
        """Create Sphinx directory structure."""
        self.source_dir.mkdir(parents=True, exist_ok=True)
        self.build_dir.mkdir(parents=True, exist_ok=True)
        (self.source_dir / "_static").mkdir(exist_ok=True)
        (self.source_dir / "_templates").mkdir(exist_ok=True)
        print(f"✓ Created Sphinx directories in {self.output_dir}")
        
    def generate_conf_py(self, sv_env_dir: Optional[str] = None, sv_tests_dir: Optional[str] = None, 
                         py_env_dir: Optional[str] = None, py_tests_dir: Optional[str] = None):
        """Generate conf.py configuration file."""
        conf_content = f"""# Configuration file for the Sphinx documentation builder.

import os
import sys

# Project information
project = '{self.project_name}'
copyright = '2025, {self.author}'
author = '{self.author}'
version = '{self.version}'
release = '{self.version}'

# General configuration
extensions = [
    'sphinx.ext.autodoc',       # Auto-generate from docstrings
    'sphinx.ext.viewcode',      # Add links to source code
    'sphinx.ext.napoleon',      # Google/NumPy style docstrings
    'sphinx_verilog_domain',
]

templates_path = ['_templates']
exclude_patterns = []

# HTML output options
html_theme = '{self.theme}'
html_static_path = ['_static']

# Theme customization options
html_theme_options = {{
    # Navigation and Display
    'navigation_depth': -1,              # Show all levels (-1 = unlimited)
    'collapse_navigation': False,        # Keep navigation expanded
    'sticky_navigation': True,           # Sidebar scrolls with content
    'includehidden': True,               # Include hidden toctrees
    'titles_only': False,                # Show subheadings in TOC
    'prev_next_buttons_location': 'both',  # Show prev/next at top and bottom
}}

# Expand content width to use more screen space
html_css_files = ['custom.css']

# Show TOC entries for objects (functions, classes, etc.)
toc_object_entries = True

# Syntax highlighting style
pygments_style = 'sphinx'  # Options: sphinx, monokai, solarized-dark, etc.

# SystemVerilog source directories
verilog_src_dirs = ["""
        
        if sv_env_dir:
            conf_content += f"'{sv_env_dir}', "
        if sv_tests_dir:
            conf_content += f"'{sv_tests_dir}', "
            
        conf_content += """]

# Python module paths for autodoc
# Set environment variable to indicate we're building documentation
os.environ['SPHINX_BUILD'] = '1'

"""
        
        # Calculate the correct Python path based on provided directories
        if py_env_dir or py_tests_dir:
            # Add parent directories of Python packages to the path
            # This allows importing as: import pyenv.module or import pytests.module
            conf_content += f"\n# Add parent directories of Python packages to path\n"
            added_paths = set()
            
            python_dirs = []
            if py_env_dir:
                python_dirs.append(py_env_dir)
            if py_tests_dir:
                python_dirs.append(py_tests_dir)
            
            for python_dir in python_dirs:
                # Get the parent directory of the Python package
                python_path = Path(python_dir).resolve()
                parent_path = python_path.parent
                
                # Avoid adding the same parent path multiple times
                if str(parent_path) not in added_paths:
                    added_paths.add(str(parent_path))
                    # Always use absolute path - works from any location
                    conf_content += f"sys.path.insert(0, r'{parent_path}')\n"
        else:
            # Fallback for backward compatibility
            conf_content += "\nsys.path.insert(0, os.path.abspath('../../..'))\n"
            
        conf_content += """
# Napoleon settings for Google/NumPy style docstrings
napoleon_google_docstring = True
napoleon_numpy_docstring = True
napoleon_include_init_with_doc = True
napoleon_include_private_with_doc = False
napoleon_include_special_with_doc = True
napoleon_use_admonition_for_examples = True
napoleon_use_admonition_for_notes = True
napoleon_use_admonition_for_references = True
napoleon_use_ivar = False
napoleon_use_param = True
napoleon_use_rtype = True

# Autodoc settings
autodoc_default_options = {
    'members': True,
    'member-order': 'bysource',
    'special-members': '__init__',
    'undoc-members': True,
    'show-inheritance': True,
}
"""
        
        conf_file = self.source_dir / "conf.py"
        conf_file.write_text(conf_content)
        print(f"✓ Generated {conf_file}")
        
    def generate_custom_css(self):
        """Generate custom CSS for wider content."""
        css_content = """/* Expand content width for better use of screen space */
.wy-nav-content {
    max-width: 99% !important;  /* Default is 800px, use 99% of screen */
}

/* Optional: Adjust sidebar width if needed */
.wy-nav-side {
    width: 300px;  /* Default is 300px */
}
"""
        css_file = self.source_dir / "_static" / "custom.css"
        css_file.write_text(css_content)
        print(f"✓ Generated custom CSS")
        
    def generate_index_rst(self, include_python: bool = False):
        """Generate index.rst file."""
        index_content = f"""{self.project_name}
{'=' * len(self.project_name)}

Welcome to the {self.project_name} UVM testbench documentation.

.. toctree::
   :maxdepth: 7
   :caption: Contents:

   environment
   testcases
"""
        
        if include_python:
            index_content += """   python_environment
   python_testcases
"""
        
        index_content += """
Indices and tables
==================

* :ref:`genindex`
* :ref:`modindex`
* :ref:`search`
"""
        
        index_file = self.source_dir / "index.rst"
        index_file.write_text(index_content)
        print(f"✓ Generated {index_file}")
        
    def build_html(self):
        """Build HTML documentation using sphinx-build."""
        print("\n📚 Building HTML documentation...")
        try:
            result = subprocess.run(
                ["sphinx-build", "-b", "html", str(self.source_dir), str(self.build_dir / "html")],
                capture_output=True,
                text=True
            )
            
            if result.returncode == 0:
                print("✓ HTML documentation built successfully")
                print(f"\n📖 Documentation available at: {self.build_dir / 'html' / 'index.html'}")
                return True
            else:
                print(f"✗ Build failed:\n{result.stderr}")
                return False
        except FileNotFoundError:
            print("✗ sphinx-build not found. Please install Sphinx: pip install sphinx")
            return False
            
    def open_in_browser(self):
        """Open generated documentation in web browser."""
        index_html = self.build_dir / "html" / "index.html"
        if index_html.exists():
            import webbrowser
            webbrowser.open(f"file://{index_html.absolute()}")
            print(f"✓ Opened documentation in browser")
        else:
            print(f"✗ Documentation not found at {index_html}")


# Re-export RSTGenerator from parser module for backwards compatibility
from .parser import RSTGenerator

__all__ = ["DocGenerator", "RSTGenerator"]
