# Sphinx Doc

Automated Sphinx documentation generator for SystemVerilog and Python UVM testbenches.

## Features

- 📄 **Automatic Documentation Generation**: Parses SystemVerilog (.sv, .svh) files and generates beautiful Sphinx documentation
- 🐍 **Python Support**: Also supports Python UVM (PyUVM) testbenches with autodoc
- 📦 **Complete Package**: Includes parser, generator, and Sphinx configuration
- 🎨 **Professional Theme**: Uses Read the Docs theme with customizable styling
- 🔍 **Detailed Extraction**: Extracts classes, modules, functions, tasks, variables, and comments
- 📚 **Collapsible Source**: Includes collapsible source code blocks for better readability
- 🔗 **Cross-References**: Automatic linking and indexing
- 🎯 **Language Auto-Detection**: Automatically detects SystemVerilog and Python files

## Installation

```bash
pip install sphinx-doc
```

## Sample Files for Testing

After installation, the package includes sample SystemVerilog and Python files in the `tests/` directory for quick testing and experimentation.

### Sample File Structure

The sample files include:
- **tests/env/**: SystemVerilog environment components (agent, env, sequence library)
- **tests/tests/**: SystemVerilog test cases (random and directed tests)
- **tests/pyenv/**: Python environment components (agent, env, config)
- **tests/pytests/**: Python test cases

### Try It Out

```bash
sphinx-doc -e "/path/to/tests/env" -t "/path/to/tests/tests" --view
```

## Quick Start

### Check Version

```bash
sphinx-doc --version
# or
sphinx-doc -V
```

### Generate Documentation (Auto-Detect Language)

```bash
# Generates RST and builds HTML automatically
sphinx-doc --env ./testbench/env --test ./testbench/tests

# Build and open in browser
sphinx-doc -e ./env -t ./tests --view

# Generate only RST files (skip HTML build)
sphinx-doc --env ./testbench/env --test ./testbench/tests --no-build
```

### Multiple Directories (Mixed Languages)

```bash
# Mix SystemVerilog and Python - auto-detected
sphinx-doc -e ./env -e ./pyenv -t ./tests -t ./pytests
```

### Full Options

```bash
sphinx-doc \
  --env ./path/to/environment \
  --test ./path/to/tests \
  --output ./documentation \
  --project-name "My UVM Testbench" \
  --author "Your Name" \
  --revision "1.0.0"
```

## Command Line Options

### Main Command: `sphinx-doc`

- `--env PATH` or `-e PATH`: Path to environment directory (auto-detects .sv/.svh or .py files). Can be used multiple times.
- `--test PATH` or `-t PATH`: Path to tests directory (auto-detects .sv/.svh or .py files). Can be used multiple times.
- `--output PATH`: Output directory for documentation (default: ./docs)
- `--project-name NAME`: Project name for documentation (default: auto-detected)
- `--author NAME`: Author name (default: "Your Name")
- `--revision VER`: Documentation revision/version (default: "1.0.0")
- `--theme THEME`: Sphinx HTML theme (default: sphinx_rtd_theme)
- `--no-build`: Skip HTML build (only generate RST files)
- `--view`: Open documentation in browser after building
- `--version` or `-V`: Show package version and exit

## Documentation Features

### SystemVerilog Support

- **Modules**: Full module extraction with ports and descriptions
- **Classes**: UVM classes with inheritance, variables, and methods
- **Interfaces**: Interface definitions with port tables
- **Functions/Tasks**: Complete signatures with collapsible source code
- **Comments**: Extracts block comments (`/* */`), line comments (`//`), and inline comments (`//<`)
- **Auto-Detection**: Automatically finds .sv and .svh files

### Python Support

- **Auto-documentation**: Uses Sphinx autodoc for Python modules
- **Docstring Styles**: Supports Google and NumPy docstring formats
- **Class/Function Documentation**: Automatic extraction from Python docstrings
- **Cross-linking**: Links between Python and SystemVerilog components
- **Auto-Detection**: Automatically finds .py files

### Generated Documentation Structure

```
docs/
├── build/
│   └── html/
│       └── index.html  (Main documentation)
└── source/
    ├── conf.py
    ├── index.rst
    ├── environment.rst
    ├── testcases.rst
    ├── sv_env/
    │   ├── component1.rst
    │   └── component2.rst
    └── sv_tests/
        ├── test1.rst
        └── test2.rst
```

## Example Project Structure

```
my_uvm_project/
├── env/
│   ├── my_env.sv
│   ├── my_agent.sv
│   └── my_driver.sv
├── tests/
│   ├── base_test.sv
│   └── my_test.sv
├── pyenv/                  (optional)
│   ├── my_env.py
│   └── my_agent.py
└── pytests/                (optional)
    └── test_my_env.py
```

## Comment Style Guide

### Block Comments (Before Construct)

```systemverilog
/*
 * This is a UVM driver for AXI protocol.
 * Handles write and read transactions.
 */
class axi_driver extends uvm_driver;
```

### Line Comments (Variable Description)

```systemverilog
// Transaction queue for storing incoming requests
mailbox #(axi_transaction) req_queue;
```

### Inline Comments (Same Line)

```systemverilog
bit [7:0] data;  //< 8-bit data payload
```

### Function/Task Comments (Inside Body)

```systemverilog
function void my_function();
  /* Brief: Performs initialization of driver components */
  // Function implementation
endfunction
```

## Customization

The generated `conf.py` can be customized for:

- Theme selection
- Color schemes
- Sidebar navigation
- Table of contents depth
- Code highlighting styles

## Building Documentation Manually

If you want to modify and rebuild:

```bash
cd docs
sphinx-build -b html source build/html
```

## Requirements

- Python 3.8+
- Sphinx 5.0+
- sphinx-rtd-theme 1.0+
- sphinx-verilog-domain 0.2+

## License

MIT License - See LICENSE file for details

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## Support

For issues and questions:
- GitHub Issues: https://github.com/SanCodex/sv-sphinx-doc/issues
- Documentation: https://sphinx-doc.readthedocs.io

## Changelog

### Version 1.0.1 (2025-01-01)
- Initial release
- SystemVerilog parsing support
- Python autodoc integration
- Collapsible source code
- Professional theme
