"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitDependencyCheck = void 0;
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cli_1 = require("./cli");
class CodeCommitDependencyCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { schedule, computeType, repository, preCheckCommand = 'echo "No preCheckCommand!"', version = '5.3.2', projectName, failOnCVSS = 0, paths = ['.'], excludes, suppressions, enableExperimental, reportsBucket, } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        const dependencyCheck = `dependency-check-${version}-release`;
        const cli = new cli_1.Cli();
        this.checkProject = new aws_codebuild_1.Project(this, 'CheckProject', {
            cache: aws_codebuild_1.Cache.local(aws_codebuild_1.LocalCacheMode.CUSTOM),
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    install: {
                        commands: [
                            'echo "[===== Install OWASP Dependency Check =====]"',
                            'wget -O public-key.asc https://bintray.com/user/downloadSubjectPublicKey?username=jeremy-long',
                            'gpg --keyid-format long --list-options show-keyring public-key.asc',
                            'gpg --import public-key.asc',
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip`,
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip.asc`,
                            `gpg --verify ${dependencyCheck}.zip.asc ${dependencyCheck}.zip`,
                            `unzip ${dependencyCheck}.zip -d /opt`,
                            'chmod +x /opt/dependency-check/bin/dependency-check.sh',
                            'export PATH="$PATH:/opt/dependency-check/bin"',
                        ],
                    },
                    pre_build: {
                        commands: [
                            `echo "[===== Clone repository: ${repositoryName} =====]"`,
                            `git clone "${repositoryCloneUrlHttp}"`,
                            `cd ${repositoryName}`,
                            `${preCheckCommand}`,
                            'SHA=$(git rev-parse HEAD)',
                            'cd ${CODEBUILD_SRC_DIR}',
                        ],
                    },
                    build: {
                        commands: [
                            `echo "[===== Scan repository: ${repositoryName} =====]"`,
                            'echo "[===== SHA: $SHA =====]"',
                            'mkdir reports',
                            cli.version(),
                            cli.scan({
                                projectName: projectName || repositoryName,
                                basedir: repositoryName,
                                paths,
                                failOnCVSS,
                                enableExperimental,
                                suppressions,
                                excludes,
                            }),
                        ],
                        finally: [
                            'echo "[===== Upload reports =====]"',
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            reportsBucket
                                ? `aws s3 cp reports/dependency-check-report.html s3://${reportsBucket.bucketName}/${repositoryName}/\${dt}_UTC/`
                                : 'echo "No reportsBuckets"',
                        ],
                    },
                },
                reports: {
                    dependencyCheckReport: {
                        files: ['reports/dependency-check-junit.xml'],
                    },
                },
                cache: {
                    paths: ['/opt/dependency-check/data/**/*'],
                },
            }),
        });
        this.checkProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        if (reportsBucket) {
            reportsBucket.grantWrite(this.checkProject);
        }
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.checkProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.checkProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.checkProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.checkProject.onBuildSucceeded(id, options);
    }
}
exports.CodeCommitDependencyCheck = CodeCommitDependencyCheck;
//# sourceMappingURL=data:application/json;base64,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