"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const assert_2 = require("assert");
const glue = require("../lib");
test('unpartitioned JSON table', () => {
    const app = new cdk.App();
    const dbStack = new cdk.Stack(app, 'db');
    const database = new glue.Database(dbStack, 'Database', {
        databaseName: 'database',
    });
    const tableStack = new cdk.Stack(app, 'table');
    const table = new glue.Table(tableStack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.UNENCRYPTED);
    assert_1.expect(tableStack).to(assert_1.haveResource('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        DeletionPolicy: 'Retain',
        UpdateReplacePolicy: 'Retain',
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: false,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('partitioned JSON table', () => {
    const app = new cdk.App();
    const dbStack = new cdk.Stack(app, 'db');
    const database = new glue.Database(dbStack, 'Database', {
        databaseName: 'database',
    });
    const tableStack = new cdk.Stack(app, 'table');
    const table = new glue.Table(tableStack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        partitionKeys: [{
                name: 'year',
                type: glue.Schema.SMALL_INT,
            }],
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.UNENCRYPTED);
    assert_2.equal(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: false,
            },
            PartitionKeys: [
                {
                    Name: 'year',
                    Type: 'smallint',
                },
            ],
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('compressed table', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        compressed: true,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: false,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: true,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('table.node.defaultChild', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    // WHEN
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        compressed: true,
        dataFormat: glue.DataFormat.JSON,
    });
    // THEN
    assert_2.ok(table.node.defaultChild instanceof glue.CfnTable);
});
test('encrypted table: SSE-S3', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.S3_MANAGED,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.S3_MANAGED);
    assert_2.equal(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
    }));
});
test('encrypted table: SSE-KMS (implicitly created key)', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.KMS,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.KMS);
    assert_2.equal(table.encryptionKey, table.bucket.encryptionKey);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
        Description: 'Created by Default/Table/Bucket',
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            'Fn::GetAtt': [
                                'TableBucketKey3E9F984A',
                                'Arn',
                            ],
                        },
                        SSEAlgorithm: 'aws:kms',
                    },
                },
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('encrypted table: SSE-KMS (explicitly created key)', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const encryptionKey = new kms.Key(stack, 'MyKey');
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.KMS,
        encryptionKey,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.KMS);
    assert_2.equal(table.encryptionKey, table.bucket.encryptionKey);
    assert_2.notEqual(table.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            'Fn::GetAtt': [
                                'MyKey6AB29FA6',
                                'Arn',
                            ],
                        },
                        SSEAlgorithm: 'aws:kms',
                    },
                },
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('encrypted table: SSE-KMS_MANAGED', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.KMS_MANAGED,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.KMS_MANAGED);
    assert_2.equal(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'aws:kms',
                    },
                },
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Name: 'table',
            Description: 'table generated by CDK',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('encrypted table: CSE-KMS (implicitly created key)', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
    assert_2.notEqual(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('encrypted table: CSE-KMS (explicitly created key)', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const encryptionKey = new kms.Key(stack, 'MyKey');
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
        encryptionKey,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
    assert_2.notEqual(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'TableBucketDA42407C',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('encrypted table: CSE-KMS (explicitly passed bucket and key)', () => {
    const stack = new cdk.Stack();
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const bucket = new s3.Bucket(stack, 'Bucket');
    const encryptionKey = new kms.Key(stack, 'MyKey');
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        bucket,
        encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
        encryptionKey,
        dataFormat: glue.DataFormat.JSON,
    });
    assert_2.equal(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
    assert_2.notEqual(table.encryptionKey, undefined);
    assert_2.equal(table.bucket.encryptionKey, undefined);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            Ref: 'DatabaseB269D8BB',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: true,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'Bucket83908E77',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('explicit s3 bucket and prefix', () => {
    const app = new cdk.App();
    const dbStack = new cdk.Stack(app, 'db');
    const stack = new cdk.Stack(app, 'app');
    const bucket = new s3.Bucket(stack, 'ExplicitBucket');
    const database = new glue.Database(dbStack, 'Database', {
        databaseName: 'database',
    });
    new glue.Table(stack, 'Table', {
        database,
        bucket,
        s3Prefix: 'prefix/',
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        dataFormat: glue.DataFormat.JSON,
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: false,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'ExplicitBucket0AA51A3F',
                            },
                            '/prefix/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('explicit s3 bucket and with empty prefix', () => {
    const app = new cdk.App();
    const dbStack = new cdk.Stack(app, 'db');
    const stack = new cdk.Stack(app, 'app');
    const bucket = new s3.Bucket(stack, 'ExplicitBucket');
    const database = new glue.Database(dbStack, 'Database', {
        databaseName: 'database',
    });
    new glue.Table(stack, 'Table', {
        database,
        bucket,
        s3Prefix: '',
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        dataFormat: glue.DataFormat.JSON,
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
        CatalogId: {
            Ref: 'AWS::AccountId',
        },
        DatabaseName: {
            'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4',
        },
        TableInput: {
            Description: 'table generated by CDK',
            Name: 'table',
            Parameters: {
                has_encrypted_data: false,
            },
            StorageDescriptor: {
                Columns: [
                    {
                        Name: 'col',
                        Type: 'string',
                    },
                ],
                Compressed: false,
                InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                Location: {
                    'Fn::Join': [
                        '',
                        [
                            's3://',
                            {
                                Ref: 'ExplicitBucket0AA51A3F',
                            },
                            '/',
                        ],
                    ],
                },
                OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                SerdeInfo: {
                    SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe',
                },
                StoredAsSubDirectories: false,
            },
            TableType: 'EXTERNAL_TABLE',
        },
    }));
});
test('grants: read only', () => {
    const stack = new cdk.Stack();
    const user = new iam.User(stack, 'User');
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        compressed: true,
        dataFormat: glue.DataFormat.JSON,
    });
    table.grantRead(user);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'glue:BatchDeletePartition',
                        'glue:BatchGetPartition',
                        'glue:GetPartition',
                        'glue:GetPartitions',
                        'glue:GetTable',
                        'glue:GetTables',
                        'glue:GetTableVersions',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':glue:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':table/',
                                {
                                    Ref: 'DatabaseB269D8BB',
                                },
                                '/',
                                {
                                    Ref: 'Table4C2D914F',
                                },
                            ],
                        ],
                    },
                },
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'TableBucketDA42407C',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'TableBucketDA42407C',
                                            'Arn',
                                        ],
                                    },
                                    '/',
                                ],
                            ],
                        },
                    ],
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'UserDefaultPolicy1F97781E',
        Users: [
            {
                Ref: 'User00B015A1',
            },
        ],
    }));
});
test('grants: write only', () => {
    const stack = new cdk.Stack();
    const user = new iam.User(stack, 'User');
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        compressed: true,
        dataFormat: glue.DataFormat.JSON,
    });
    table.grantWrite(user);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'glue:BatchCreatePartition',
                        'glue:BatchDeletePartition',
                        'glue:CreatePartition',
                        'glue:DeletePartition',
                        'glue:UpdatePartition',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':glue:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':table/',
                                {
                                    Ref: 'DatabaseB269D8BB',
                                },
                                '/',
                                {
                                    Ref: 'Table4C2D914F',
                                },
                            ],
                        ],
                    },
                },
                {
                    Action: [
                        's3:DeleteObject*',
                        's3:PutObject*',
                        's3:Abort*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'TableBucketDA42407C',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'TableBucketDA42407C',
                                            'Arn',
                                        ],
                                    },
                                    '/',
                                ],
                            ],
                        },
                    ],
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'UserDefaultPolicy1F97781E',
        Users: [
            {
                Ref: 'User00B015A1',
            },
        ],
    }));
});
test('grants: read and write', () => {
    const stack = new cdk.Stack();
    const user = new iam.User(stack, 'User');
    const database = new glue.Database(stack, 'Database', {
        databaseName: 'database',
    });
    const table = new glue.Table(stack, 'Table', {
        database,
        tableName: 'table',
        columns: [{
                name: 'col',
                type: glue.Schema.STRING,
            }],
        compressed: true,
        dataFormat: glue.DataFormat.JSON,
    });
    table.grantReadWrite(user);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'glue:BatchDeletePartition',
                        'glue:BatchGetPartition',
                        'glue:GetPartition',
                        'glue:GetPartitions',
                        'glue:GetTable',
                        'glue:GetTables',
                        'glue:GetTableVersions',
                        'glue:BatchCreatePartition',
                        'glue:BatchDeletePartition',
                        'glue:CreatePartition',
                        'glue:DeletePartition',
                        'glue:UpdatePartition',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':glue:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':table/',
                                {
                                    Ref: 'DatabaseB269D8BB',
                                },
                                '/',
                                {
                                    Ref: 'Table4C2D914F',
                                },
                            ],
                        ],
                    },
                },
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                        's3:DeleteObject*',
                        's3:PutObject*',
                        's3:Abort*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'TableBucketDA42407C',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'TableBucketDA42407C',
                                            'Arn',
                                        ],
                                    },
                                    '/',
                                ],
                            ],
                        },
                    ],
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'UserDefaultPolicy1F97781E',
        Users: [
            {
                Ref: 'User00B015A1',
            },
        ],
    }));
});
test('validate: at least one column', () => {
    expect(() => {
        createTable({
            columns: [],
            tableName: 'name',
        });
    }).toThrowError('you must specify at least one column for the table');
});
test('validate: unique column names', () => {
    expect(() => {
        createTable({
            tableName: 'name',
            columns: [{
                    name: 'col1',
                    type: glue.Schema.STRING,
                }, {
                    name: 'col1',
                    type: glue.Schema.STRING,
                }],
        });
    }).toThrowError("column names and partition keys must be unique, but 'col1' is duplicated");
});
test('validate: unique partition keys', () => {
    expect(() => {
        createTable({
            tableName: 'name',
            columns: [{
                    name: 'col1',
                    type: glue.Schema.STRING,
                }],
            partitionKeys: [{
                    name: 'p1',
                    type: glue.Schema.STRING,
                }, {
                    name: 'p1',
                    type: glue.Schema.STRING,
                }],
        });
    }).toThrowError("column names and partition keys must be unique, but 'p1' is duplicated");
});
test('validate: column names and partition keys are all unique', () => {
    expect(() => {
        createTable({
            tableName: 'name',
            columns: [{
                    name: 'col1',
                    type: glue.Schema.STRING,
                }],
            partitionKeys: [{
                    name: 'col1',
                    type: glue.Schema.STRING,
                }],
        });
    }).toThrowError("column names and partition keys must be unique, but 'col1' is duplicated");
});
test('validate: can not specify an explicit bucket and encryption', () => {
    expect(() => {
        createTable({
            tableName: 'name',
            columns: [{
                    name: 'col1',
                    type: glue.Schema.STRING,
                }],
            bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
            encryption: glue.TableEncryption.KMS,
        });
    }).toThrowError('you can not specify encryption settings if you also provide a bucket');
});
test('validate: can explicitly pass bucket if Encryption undefined', () => {
    assert_2.doesNotThrow(() => createTable({
        tableName: 'name',
        columns: [{
                name: 'col1',
                type: glue.Schema.STRING,
            }],
        bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
        encryption: undefined,
    }));
});
test('validate: can explicitly pass bucket if Unencrypted', () => {
    assert_2.doesNotThrow(() => createTable({
        tableName: 'name',
        columns: [{
                name: 'col1',
                type: glue.Schema.STRING,
            }],
        bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
        encryption: undefined,
    }));
});
test('validate: can explicitly pass bucket if ClientSideKms', () => {
    assert_2.doesNotThrow(() => createTable({
        tableName: 'name',
        columns: [{
                name: 'col1',
                type: glue.Schema.STRING,
            }],
        bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
        encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
    }));
});
test('Table.fromTableArn', () => {
    // GIVEN
    const stack = new cdk.Stack();
    // WHEN
    const table = glue.Table.fromTableArn(stack, 'boom', 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
    // THEN
    assert_2.deepEqual(table.tableArn, 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
    assert_2.deepEqual(table.tableName, 'tbl1');
});
function createTable(props) {
    const stack = new cdk.Stack();
    new glue.Table(stack, 'table', {
        ...props,
        database: new glue.Database(stack, 'db', {
            databaseName: 'database_name',
        }),
        dataFormat: glue.DataFormat.JSON,
    });
}
//# sourceMappingURL=data:application/json;base64,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