"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * Utility method to add a new GatewayRoute to the VirtualGateway
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualGatewayArn],
        });
    }
}
/**
 * VirtualGateway represents a newly defined App Mesh Virtual Gateway
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 */
class VirtualGateway extends VirtualGatewayBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.listeners = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualGatewayProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, VirtualGateway);
            }
            throw error;
        }
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.http().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = props.accessLog?.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, props.backendDefaults?.tlsClientPolicy),
                        },
                    }
                    : undefined,
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualGateway given an ARN
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualGatewayArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualGatewayArn = virtualGatewayArn;
                this.virtualGatewayName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualGateway given its attributes
     */
    static fromVirtualGatewayAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualGatewayAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVirtualGatewayAttributes);
            }
            throw error;
        }
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualGatewayName = attrs.virtualGatewayName;
                this.virtualGatewayArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualGateway`,
                    resourceName: this.virtualGatewayName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualGateway = VirtualGateway;
_a = JSII_RTTI_SYMBOL_1;
VirtualGateway[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualGateway", version: "1.182.0" };
//# sourceMappingURL=data:application/json;base64,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