# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['torch_points3d',
 'torch_points3d.applications',
 'torch_points3d.core',
 'torch_points3d.core.base_conv',
 'torch_points3d.core.common_modules',
 'torch_points3d.core.data_transform',
 'torch_points3d.core.initializer',
 'torch_points3d.core.losses',
 'torch_points3d.core.regularizer',
 'torch_points3d.core.schedulers',
 'torch_points3d.core.spatial_ops',
 'torch_points3d.datasets',
 'torch_points3d.datasets.classification',
 'torch_points3d.datasets.registration',
 'torch_points3d.datasets.segmentation',
 'torch_points3d.datasets.segmentation.forward',
 'torch_points3d.metrics',
 'torch_points3d.models',
 'torch_points3d.models.base_architectures',
 'torch_points3d.models.registration',
 'torch_points3d.models.segmentation',
 'torch_points3d.modules',
 'torch_points3d.modules.KPConv',
 'torch_points3d.modules.MinkowskiEngine',
 'torch_points3d.modules.PointCNN',
 'torch_points3d.modules.PointNet',
 'torch_points3d.modules.RSConv',
 'torch_points3d.modules.RandLANet',
 'torch_points3d.modules.pointnet2',
 'torch_points3d.utils',
 'torch_points3d.utils.model_building_utils',
 'torch_points3d.visualization']

package_data = \
{'': ['*'],
 'torch_points3d.applications': ['conf/kpconv/*',
                                 'conf/pointnet2/*',
                                 'conf/rsconv/*'],
 'torch_points3d.datasets.registration': ['urls/*'],
 'torch_points3d.modules.KPConv': ['kernels/dispositions/*']}

install_requires = \
['hydra-core>=0.11.2,<0.12.0',
 'matplotlib>=3.1,<4.0',
 'numba>=0.49.0,<0.50.0',
 'numpy>=1.18.2,<2.0.0',
 'pytorch_metric_learning>=0.9.75,<0.10.0',
 'scikit-image>=0.16.2,<0.17.0',
 'tensorboard>=2.1,<3.0',
 'torch-cluster==1.4.5',
 'torch-geometric==1.3.2',
 'torch-points-kernels==0.5.2',
 'torch-scatter==1.4.0',
 'torch-sparse==0.4.3',
 'torch==1.3.1',
 'torchnet>=0.0.4,<0.0.5',
 'tqdm>=4.40,<5.0',
 'wandb>=0.8.18,<0.9.0']

extras_require = \
{'MinkowskiEngine': ['MinkowskiEngine>=0.4.2,<0.5.0']}

setup_kwargs = {
    'name': 'torch-points3d',
    'version': '0.1.2',
    'description': 'Point Cloud Deep Learning Extension Library for PyTorch',
    'long_description': '<p align="center">\n  <img width="40%" src="https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/logo.png" />\n</p>\n\n[![codecov](https://codecov.io/gh/nicolas-chaulet/torch-points3d/branch/master/graph/badge.svg)](https://codecov.io/gh/nicolas-chaulet/torch-points3d) [![Actions Status](https://github.com/nicolas-chaulet/torch-points3d/workflows/unittest/badge.svg)](https://github.com/nicolas-chaulet/torch-points3d/actions) [![Documentation Status](https://readthedocs.org/projects/torch-points3d/badge/?version=latest)](https://torch-points3d.readthedocs.io/en/latest/?badge=latest)\n\nThis is a framework for running common deep learning models for point cloud analysis tasks against classic benchmark. It heavily relies on [Pytorch Geometric](https://pytorch-geometric.readthedocs.io/en/latest/notes/resources.html) and [Facebook Hydra](https://hydra.cc/).\n\nThe framework allows lean and yet complex model to be built with minimum effort and great reproducibility.\n\n## Project structure\n\n```bash\n├─ benchmark               # Output from various benchmark runs\n├─ conf                    # All configurations for training nad evaluation leave there\n├─ dashboard               # A collection of notebooks that allow result exploration and network debugging\n├─ docker                  # Docker image that can be used for inference or training\n├─ docs                    # All the doc\n├─ eval.py                 # Eval script\n├─ find_neighbour_dist.py  # Script to find optimal #neighbours within neighbour search operations\n├─ forward_scripts         # Script that runs a forward pass on possibly non annotated data\n├─ outputs                 # All outputs from your runs sorted by date\n├─ scripts                 # Some scripts to help manage the project\n├─ torch_points3d\n    ├─ core                # Core components\n    ├─ datasets            # All code related to datasets\n    ├─ metrics             # All metrics and trackers\n    ├─ models              # All models\n    ├─ modules             # Basic modules that can be used in a modular way\n    ├─ utils               # Various utils\n    └─ visualization       # Visualization\n├─ test\n└─ train.py                # Main script to launch a training\n```\n\nAs a general philosophy we have split datasets and models by task. For example, datasets has three subfolders:\n\n- segmentation\n- classification\n- registration\n\nwhere each folder contains the dataset related to each task.\n\n## Methods currently implemented\n\n- **[PointNet](https://github.com/nicolas-chaulet/torch-points3d/blob/master/torch_points3d/modules/PointNet/modules.py#L54)** from Charles R. Qi _et al._: [PointNet: Deep Learning on Point Sets for 3D Classification and Segmentation](https://arxiv.org/abs/1612.00593) (CVPR 2017)\n- **[PointNet++](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/pointnet2)** from Charles from Charles R. Qi _et al._: [PointNet++: Deep Hierarchical Feature Learning on Point Sets in a Metric Space](https://arxiv.org/abs/1706.02413)\n- **[RSConv](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/RSConv)** from Yongcheng Liu _et al._: [Relation-Shape Convolutional Neural Network for Point Cloud Analysis](https://arxiv.org/abs/1904.07601) (CVPR 2019)\n- **[RandLA-Net](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/RandLANet)** from Qingyong Hu _et al._: [RandLA-Net: Efficient Semantic Segmentation of Large-Scale Point Clouds](https://arxiv.org/abs/1911.11236)\n- **[PointCNN](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/PointCNN)** from Yangyan Li _et al._: [PointCNN: Convolution On X-Transformed Points](https://arxiv.org/abs/1801.07791) (NIPS 2018)\n- **[KPConv](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/KPConv)** from Hugues Thomas _et al._: [KPConv: Flexible and Deformable Convolution for Point Clouds](https://arxiv.org/abs/1801.07791) (ICCV 2019)\n- **[MinkowskiEngine](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/MinkowskiEngine)** from Christopher Choy _et al._: [4D Spatio-Temporal ConvNets: Minkowski Convolutional Neural Networks](https://arxiv.org/abs/1904.08755) (CVPR\'19)\n\n## Available datasets\n\n### Segmentation\n\n- **[Scannet](https://github.com/ScanNet/ScanNet)** from Angela Dai _et al._: [ScanNet: Richly-annotated 3D Reconstructions of Indoor Scenes](https://arxiv.org/abs/1702.04405)\n\n- **[S3DIS](http://buildingparser.stanford.edu/dataset.html)** from Iro Armeni _et al._: [Joint 2D-3D-Semantic Data for Indoor Scene Understanding](https://arxiv.org/abs/1702.01105)\n\n```\n* S3DIS 1x1\n* S3DIS Room\n* S3DIS Fused\n```\n\n- **[Shapenet](https://www.shapenet.org/)** from Iro Armeni _et al._: [ShapeNet: An Information-Rich 3D Model Repository](https://arxiv.org/abs/1512.03012)\n\n### Registration\n\n- **[3DMatch](http://3dmatch.cs.princeton.edu)** from Andy Zeng _et al._: [3DMatch: Learning Local Geometric Descriptors from RGB-D Reconstructions](https://arxiv.org/abs/1603.08182)\n\n### Classification\n\n- **[ModelNet](https://modelnet.cs.princeton.edu)** from Zhirong Wu _et al._: [3D ShapeNets: A Deep Representation for Volumetric Shapes](https://people.csail.mit.edu/khosla/papers/cvpr2015_wu.pdf)\n\n## Getting started\n\n### Requirements:\n\n- CUDA > 10\n- Python 3 + headers (python-dev)\n- [Poetry](https://poetry.eustace.io/) (Optional but highly recommended)\n\n### Setup repo\n\nClone the repo to your local machine\n\nRun the following command from the root of the repo\n\n```\npoetry install --no-root\n```\n\nThis will install all required dependencies in a new virtual environment.\n\nActivate it\n\n```bash\npoetry shell\n```\n\nYou can check that the install has been successful by running\n\n```bash\npython -m unittest -v\n```\n\nor from pypi\n\n```bash\npip install torch_points3d\n```\n\n#### [Minkowski Engine](https://github.com/StanfordVL/MinkowskiEngine)\n\nThe repository is supporting [Minkowski Engine](https://github.com/StanfordVL/MinkowskiEngine) which requires `openblas-dev` and `nvcc` if you have a CUDA device on your machine. First install `openblas`\n\n```bash\nsudo apt install libopenblas-dev\n```\n\nthen make sure that `nvcc` is in your path:\n\n```bash\nnvcc -V\n```\n\nIf it\'s not then locate it (`locate nvcc`) and add its location to your `PATH` variable. On my machine:\n\n```bash\nexport PATH="/usr/local/cuda-10.2/bin:$PATH"\n```\n\nYou are now in a position to install MinkowskiEngine with GPU support:\n\n```bash\npoetry install -E MinkowskiEngine --no-root\n```\n\n#### Pycuda\n\n```bash\npip install pycuda\n```\n\n## Train pointnet++ on part segmentation task for dataset shapenet\n\n```bash\npoetry run python train.py task=segmentation model_type=pointnet2 model_name=pointnet2_charlesssg dataset=shapenet-fixed\n```\n\nAnd you should see something like that\n\n![logging](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/logging.png)\n\nThe [config](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/conf/models/segmentation/pointnet2.yaml) for pointnet++ is a good example of how to define a model and is as follow:\n\n```yaml\n# PointNet++: Deep Hierarchical Feature Learning on Point Sets in a Metric Space (https://arxiv.org/abs/1706.02413)\n# Credit Charles R. Qi: https://github.com/charlesq34/pointnet2/blob/master/models/pointnet2_part_seg_msg_one_hot.py\n\npointnet2_onehot:\n  architecture: pointnet2.PointNet2_D\n  conv_type: \'DENSE\'\n  use_category: True\n  down_conv:\n    module_name: PointNetMSGDown\n    npoint: [1024, 256, 64, 16]\n    radii: [[0.05, 0.1], [0.1, 0.2], [0.2, 0.4], [0.4, 0.8]]\n    nsamples: [[16, 32], [16, 32], [16, 32], [16, 32]]\n    down_conv_nn:\n      [\n        [[FEAT, 16, 16, 32], [FEAT, 32, 32, 64]],\n        [[32 + 64, 64, 64, 128], [32 + 64, 64, 96, 128]],\n        [[128 + 128, 128, 196, 256], [128 + 128, 128, 196, 256]],\n        [[256 + 256, 256, 256, 512], [256 + 256, 256, 384, 512]],\n      ]\n  up_conv:\n    module_name: DenseFPModule\n    up_conv_nn:\n      [\n        [512 + 512 + 256 + 256, 512, 512],\n        [512 + 128 + 128, 512, 512],\n        [512 + 64 + 32, 256, 256],\n        [256 + FEAT, 128, 128],\n      ]\n    skip: True\n  mlp_cls:\n    nn: [128, 128]\n    dropout: 0.5\n```\n\n# Benchmark\n\n## S3DIS 1x1\n\n| Model Name                                                           | # params  | Speed Train / Test      | Cross Entropy | OAcc  | mIou  | mAcc  |\n| -------------------------------------------------------------------- | --------- | ----------------------- | ------------- | ----- | ----- | ----- |\n| [`pointnet2_original`](https://github.com/nicolas-chaulet/torch-points3d/blob/master/benchmark/s3dis_fold5/Pointnet2_original.md) | 3,026,829 | 04:29 / 01:07(RTX 2060) | 0.0512        | 85.26 | 45.58 | 73.11 |\n\n## Shapenet part segmentation\n\nThe data reported below correspond to the part segmentation problem for Shapenet for all categories. We report against mean instance IoU and mean class IoU (average of the mean instance IoU per class)\n\n| Model Name                                                            | Use Normals | # params  | Speed Train / Test      | Cross Entropy | CmIou  | ImIou |\n| --------------------------------------------------------------------- | ----------- | --------- | ----------------------- | ------------- | ------ | ----- |\n| [`pointnet2_charlesmsg`](https://github.com/nicolas-chaulet/torch-points3d/blob/master/benchmark/shapenet/pointnet2_charlesmsg.md) | Yes         | 1,733,946 | 15:07 / 01:20 (K80)     | 0.089         | 82.1   | 85.1  |\n| [`RSCNN_MSG`](https://github.com/nicolas-chaulet/torch-points3d/blob/master/benchmark/shapenet/rscnn_original.md)                  | No          | 3,488,417 | 05:40 / 0:24 (RTX 2060) | 0.04          | 82.811 | 85.3  |\n\n## Explore your experiments\n\nWe provide a [notebook](https://github.com/nicolas-chaulet/torch-points3d/blob/master/dashboard/dashboard.ipynb) based [pyvista](https://docs.pyvista.org/) and [panel](https://panel.holoviz.org/) that allows you to explore your past experiments visually. When using jupyter lab you will have to install an extension:\n\n```\njupyter labextension install @pyviz/jupyterlab_pyviz\n```\n\nRun through the notebook and you should see a dashboard starting that looks like the following:\n\n![dashboard](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/Dashboard_demo.gif)\n\n## Inference\n\n### Inference script\n\nWe provide a script for running a given pre trained model on custom data that may not be annotated. You will find an [example](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/forward.py) of this for the part segmentation task on Shapenet. Just like for the rest of the codebase most of the customization happens through config files and the provided example can be extended to other datasets. You can also easily create your own from there. Going back to the part segmentation task, say you have a folder full of point clouds that you know are Airplanes, and you have the checkpoint of a model trained on Airplanes and potentially other classes, simply edit the [config.yaml](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/conf/config.yaml) and [shapenet.yaml](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/conf/dataset/shapenet.yaml) and run the following command:\n\n```bash\npython forward_scripts/forward.py\n```\n\nThe result of the forward run will be placed in the specified `output_folder` and you can use the [notebook](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/notebooks/viz_shapenet.ipynb) provided to explore the results. Below is an example of the outcome of using a model trained on caps only to find the parts of airplanes and caps.\n\n![resexplore](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/inference_demo.gif)\n\n### Containerize your model with Docker\n\nFinally, for people interested in deploying their models to production environments, we provide a [Dockerfile](https://github.com/nicolas-chaulet/torch-points3d/blob/master/docker/Dockerfile) as well as a [build script](https://github.com/nicolas-chaulet/torch-points3d/blob/master/docker/build.sh). Say you have trained a network for semantic segmentation that gave the weight `<outputfolder/weights.pt>`, the following command will build a docker image for you:\n\n```bash\ncd docker\n./build.sh outputfolder/weights.pt\n```\n\nYou can then use it to run a forward pass on a all the point clouds in `input_path` and generate the results in `output_path`\n\n```bash\ndocker run -v /test_data:/in -v /test_data/out:/out pointnet2_charlesssg:latest python3 forward_scripts/forward.py dataset=shapenet data.forward_category=Cap input_path="/in" output_path="/out"\n```\n\nThe `-v` option mounts a local directory to the container\'s file system. For example in the command line above, `/test_data/out` will be mounted at the location `/out`. As a consequence, all files written in `/out` will be available in the folder `/test_data/out` on your machine.\n\n## Profiling\n\nWe advice to use [`snakeviz`](https://jiffyclub.github.io/snakeviz/) and [`cProfile`](https://docs.python.org/2/library/profile.html)\n\nUse cProfile to profile your code\n\n```\npoetry run python -m cProfile -o {your_name}.prof train.py ... debugging.profiling=True\n```\n\nAnd visualize results using snakeviz.\n\n```\nsnakeviz {your_name}.prof\n```\n\nIt is also possible to use [`torch.utils.bottleneck`](https://pytorch.org/docs/stable/bottleneck.html)\n\n```\npython -m torch.utils.bottleneck /path/to/source/script.py [args]\n```\n\n## Troubleshooting\n\n#### Undefined symbol / Updating Pytorch\n\nWhen we update the version of Pytorch that is used, the compiled packages need to be reinstalled, otherwise you will run into an error that looks like this:\n\n```\n... scatter_cpu.cpython-36m-x86_64-linux-gnu.so: undefined symbol: _ZN3c1012CUDATensorIdEv\n```\n\nThis can happen for the following libraries:\n\n- torch-points\n- torch-scatter\n- torch-cluster\n- torch-sparse\n\nAn easy way to fix this is to run the following command with the virtual env activated:\n\n```\npip uninstall torch-scatter torch-sparse torch-cluster torch-points-kernels -y\nrm -rf ~/.cache/pip\npoetry install\n```\n\n## Contributing\n\nContributions are welcome! The only asks are that you stick to the styling and that you add tests as you add more features!\n\nFor styling you can use [pre-commit hooks](https://ljvmiranda921.github.io/notebook/2018/06/21/precommits-using-black-and-flake8/) to help you:\n\n```\npre-commit install\n```\n\nA sequence of checks will be run for you and you may have to add the fixed files again to the stashed files.\n\nWhen it comes to docstrings we use [numpy style](https://numpydoc.readthedocs.io/en/latest/format.html) docstrings, for those who use\nVisual Studio Code, there is a great [extension](https://github.com/NilsJPWerner/autoDocstring) that can help with that. Install it and set the format to numpy and you should be good to go!\n\nFinaly, if you want to have a direct chat with us feel free to join our slack, just shoot us an email and we\'ll add you.\n',
    'author': 'Thomas Chaton',
    'author_email': 'thomas.chaton.ai@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
