from typing import Union, List, Tuple, Any, cast

import numpy as np

Mm = 8  # Bits per symbol
Nn = 255  # Symbols per block, equivalent to ((1 << Mm) - 1)
NRoots = 32  # Number of generator roots = number of parity symbols
Fcr = 112  # First consecutive root, index form
Prim = 11  # Primitive element, index form
IPrim = 116  # Prim-th root of 1, index form
A0 = Nn

ALPHA_TO = np.array([
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x87, 0x89, 0x95, 0xad, 0xdd, 0x3d, 0x7a, 0xf4,
    0x6f, 0xde, 0x3b, 0x76, 0xec, 0x5f, 0xbe, 0xfb, 0x71, 0xe2, 0x43, 0x86, 0x8b, 0x91, 0xa5, 0xcd,
    0x1d, 0x3a, 0x74, 0xe8, 0x57, 0xae, 0xdb, 0x31, 0x62, 0xc4, 0x0f, 0x1e, 0x3c, 0x78, 0xf0, 0x67,
    0xce, 0x1b, 0x36, 0x6c, 0xd8, 0x37, 0x6e, 0xdc, 0x3f, 0x7e, 0xfc, 0x7f, 0xfe, 0x7b, 0xf6, 0x6b,
    0xd6, 0x2b, 0x56, 0xac, 0xdf, 0x39, 0x72, 0xe4, 0x4f, 0x9e, 0xbb, 0xf1, 0x65, 0xca, 0x13, 0x26,
    0x4c, 0x98, 0xb7, 0xe9, 0x55, 0xaa, 0xd3, 0x21, 0x42, 0x84, 0x8f, 0x99, 0xb5, 0xed, 0x5d, 0xba,
    0xf3, 0x61, 0xc2, 0x03, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x07, 0x0e, 0x1c, 0x38, 0x70, 0xe0,
    0x47, 0x8e, 0x9b, 0xb1, 0xe5, 0x4d, 0x9a, 0xb3, 0xe1, 0x45, 0x8a, 0x93, 0xa1, 0xc5, 0x0d, 0x1a,
    0x34, 0x68, 0xd0, 0x27, 0x4e, 0x9c, 0xbf, 0xf9, 0x75, 0xea, 0x53, 0xa6, 0xcb, 0x11, 0x22, 0x44,
    0x88, 0x97, 0xa9, 0xd5, 0x2d, 0x5a, 0xb4, 0xef, 0x59, 0xb2, 0xe3, 0x41, 0x82, 0x83, 0x81, 0x85,
    0x8d, 0x9d, 0xbd, 0xfd, 0x7d, 0xfa, 0x73, 0xe6, 0x4b, 0x96, 0xab, 0xd1, 0x25, 0x4a, 0x94, 0xaf,
    0xd9, 0x35, 0x6a, 0xd4, 0x2f, 0x5e, 0xbc, 0xff, 0x79, 0xf2, 0x63, 0xc6, 0x0b, 0x16, 0x2c, 0x58,
    0xb0, 0xe7, 0x49, 0x92, 0xa3, 0xc1, 0x05, 0x0a, 0x14, 0x28, 0x50, 0xa0, 0xc7, 0x09, 0x12, 0x24,
    0x48, 0x90, 0xa7, 0xc9, 0x15, 0x2a, 0x54, 0xa8, 0xd7, 0x29, 0x52, 0xa4, 0xcf, 0x19, 0x32, 0x64,
    0xc8, 0x17, 0x2e, 0x5c, 0xb8, 0xf7, 0x69, 0xd2, 0x23, 0x46, 0x8c, 0x9f, 0xb9, 0xf5, 0x6d, 0xda,
    0x33, 0x66, 0xcc, 0x1f, 0x3e, 0x7c, 0xf8, 0x77, 0xee, 0x5b, 0xb6, 0xeb, 0x51, 0xa2, 0xc3, 0x00
], dtype=np.uint32)  # yapf: disable

INDEX_OF = np.array([
    0xff, 0x00, 0x01, 0x63, 0x02, 0xc6, 0x64, 0x6a, 0x03, 0xcd, 0xc7, 0xbc, 0x65, 0x7e, 0x6b, 0x2a,
    0x04, 0x8d, 0xce, 0x4e, 0xc8, 0xd4, 0xbd, 0xe1, 0x66, 0xdd, 0x7f, 0x31, 0x6c, 0x20, 0x2b, 0xf3,
    0x05, 0x57, 0x8e, 0xe8, 0xcf, 0xac, 0x4f, 0x83, 0xc9, 0xd9, 0xd5, 0x41, 0xbe, 0x94, 0xe2, 0xb4,
    0x67, 0x27, 0xde, 0xf0, 0x80, 0xb1, 0x32, 0x35, 0x6d, 0x45, 0x21, 0x12, 0x2c, 0x0d, 0xf4, 0x38,
    0x06, 0x9b, 0x58, 0x1a, 0x8f, 0x79, 0xe9, 0x70, 0xd0, 0xc2, 0xad, 0xa8, 0x50, 0x75, 0x84, 0x48,
    0xca, 0xfc, 0xda, 0x8a, 0xd6, 0x54, 0x42, 0x24, 0xbf, 0x98, 0x95, 0xf9, 0xe3, 0x5e, 0xb5, 0x15,
    0x68, 0x61, 0x28, 0xba, 0xdf, 0x4c, 0xf1, 0x2f, 0x81, 0xe6, 0xb2, 0x3f, 0x33, 0xee, 0x36, 0x10,
    0x6e, 0x18, 0x46, 0xa6, 0x22, 0x88, 0x13, 0xf7, 0x2d, 0xb8, 0x0e, 0x3d, 0xf5, 0xa4, 0x39, 0x3b,
    0x07, 0x9e, 0x9c, 0x9d, 0x59, 0x9f, 0x1b, 0x08, 0x90, 0x09, 0x7a, 0x1c, 0xea, 0xa0, 0x71, 0x5a,
    0xd1, 0x1d, 0xc3, 0x7b, 0xae, 0x0a, 0xa9, 0x91, 0x51, 0x5b, 0x76, 0x72, 0x85, 0xa1, 0x49, 0xeb,
    0xcb, 0x7c, 0xfd, 0xc4, 0xdb, 0x1e, 0x8b, 0xd2, 0xd7, 0x92, 0x55, 0xaa, 0x43, 0x0b, 0x25, 0xaf,
    0xc0, 0x73, 0x99, 0x77, 0x96, 0x5c, 0xfa, 0x52, 0xe4, 0xec, 0x5f, 0x4a, 0xb6, 0xa2, 0x16, 0x86,
    0x69, 0xc5, 0x62, 0xfe, 0x29, 0x7d, 0xbb, 0xcc, 0xe0, 0xd3, 0x4d, 0x8c, 0xf2, 0x1f, 0x30, 0xdc,
    0x82, 0xab, 0xe7, 0x56, 0xb3, 0x93, 0x40, 0xd8, 0x34, 0xb0, 0xef, 0x26, 0x37, 0x0c, 0x11, 0x44,
    0x6f, 0x78, 0x19, 0x9a, 0x47, 0x74, 0xa7, 0xc1, 0x23, 0x53, 0x89, 0xfb, 0x14, 0x5d, 0xf8, 0x97,
    0x2e, 0x4b, 0xb9, 0x60, 0x0f, 0xed, 0x3e, 0xe5, 0xf6, 0x87, 0xa5, 0x17, 0x3a, 0xa3, 0x3c, 0xb7
], dtype=np.int32)  # yapf: disable

GEN_POLY = np.array([
    0x00, 0xf9, 0x3b, 0x42, 0x04, 0x2b, 0x7e, 0xfb, 0x61, 0x1e, 0x03, 0xd5, 0x32, 0x42, 0xaa, 0x05,
    0x18, 0x05, 0xaa, 0x42, 0x32, 0xd5, 0x03, 0x1e, 0x61, 0xfb, 0x7e, 0x2b, 0x04, 0x42, 0x3b, 0xf9,
    0x00
], dtype=np.uint32)  # yapf: disable

TAL_TO_DUAL_BASIS = np.array([
    0x00, 0x7b, 0xaf, 0xd4, 0x99, 0xe2, 0x36, 0x4d, 0xfa, 0x81, 0x55, 0x2e, 0x63, 0x18, 0xcc, 0xb7,
    0x86, 0xfd, 0x29, 0x52, 0x1f, 0x64, 0xb0, 0xcb, 0x7c, 0x07, 0xd3, 0xa8, 0xe5, 0x9e, 0x4a, 0x31,
    0xec, 0x97, 0x43, 0x38, 0x75, 0x0e, 0xda, 0xa1, 0x16, 0x6d, 0xb9, 0xc2, 0x8f, 0xf4, 0x20, 0x5b,
    0x6a, 0x11, 0xc5, 0xbe, 0xf3, 0x88, 0x5c, 0x27, 0x90, 0xeb, 0x3f, 0x44, 0x09, 0x72, 0xa6, 0xdd,
    0xef, 0x94, 0x40, 0x3b, 0x76, 0x0d, 0xd9, 0xa2, 0x15, 0x6e, 0xba, 0xc1, 0x8c, 0xf7, 0x23, 0x58,
    0x69, 0x12, 0xc6, 0xbd, 0xf0, 0x8b, 0x5f, 0x24, 0x93, 0xe8, 0x3c, 0x47, 0x0a, 0x71, 0xa5, 0xde,
    0x03, 0x78, 0xac, 0xd7, 0x9a, 0xe1, 0x35, 0x4e, 0xf9, 0x82, 0x56, 0x2d, 0x60, 0x1b, 0xcf, 0xb4,
    0x85, 0xfe, 0x2a, 0x51, 0x1c, 0x67, 0xb3, 0xc8, 0x7f, 0x04, 0xd0, 0xab, 0xe6, 0x9d, 0x49, 0x32,
    0x8d, 0xf6, 0x22, 0x59, 0x14, 0x6f, 0xbb, 0xc0, 0x77, 0x0c, 0xd8, 0xa3, 0xee, 0x95, 0x41, 0x3a,
    0x0b, 0x70, 0xa4, 0xdf, 0x92, 0xe9, 0x3d, 0x46, 0xf1, 0x8a, 0x5e, 0x25, 0x68, 0x13, 0xc7, 0xbc,
    0x61, 0x1a, 0xce, 0xb5, 0xf8, 0x83, 0x57, 0x2c, 0x9b, 0xe0, 0x34, 0x4f, 0x02, 0x79, 0xad, 0xd6,
    0xe7, 0x9c, 0x48, 0x33, 0x7e, 0x05, 0xd1, 0xaa, 0x1d, 0x66, 0xb2, 0xc9, 0x84, 0xff, 0x2b, 0x50,
    0x62, 0x19, 0xcd, 0xb6, 0xfb, 0x80, 0x54, 0x2f, 0x98, 0xe3, 0x37, 0x4c, 0x01, 0x7a, 0xae, 0xd5,
    0xe4, 0x9f, 0x4b, 0x30, 0x7d, 0x06, 0xd2, 0xa9, 0x1e, 0x65, 0xb1, 0xca, 0x87, 0xfc, 0x28, 0x53,
    0x8e, 0xf5, 0x21, 0x5a, 0x17, 0x6c, 0xb8, 0xc3, 0x74, 0x0f, 0xdb, 0xa0, 0xed, 0x96, 0x42, 0x39,
    0x08, 0x73, 0xa7, 0xdc, 0x91, 0xea, 0x3e, 0x45, 0xf2, 0x89, 0x5d, 0x26, 0x6b, 0x10, 0xc4, 0xbf
], dtype=np.uint8)  # yapf: disable

TAL_TO_CONVENTIONAL = np.array([
    0x00, 0xcc, 0xac, 0x60, 0x79, 0xb5, 0xd5, 0x19, 0xf0, 0x3c, 0x5c, 0x90, 0x89, 0x45, 0x25, 0xe9,
    0xfd, 0x31, 0x51, 0x9d, 0x84, 0x48, 0x28, 0xe4, 0x0d, 0xc1, 0xa1, 0x6d, 0x74, 0xb8, 0xd8, 0x14,
    0x2e, 0xe2, 0x82, 0x4e, 0x57, 0x9b, 0xfb, 0x37, 0xde, 0x12, 0x72, 0xbe, 0xa7, 0x6b, 0x0b, 0xc7,
    0xd3, 0x1f, 0x7f, 0xb3, 0xaa, 0x66, 0x06, 0xca, 0x23, 0xef, 0x8f, 0x43, 0x5a, 0x96, 0xf6, 0x3a,
    0x42, 0x8e, 0xee, 0x22, 0x3b, 0xf7, 0x97, 0x5b, 0xb2, 0x7e, 0x1e, 0xd2, 0xcb, 0x07, 0x67, 0xab,
    0xbf, 0x73, 0x13, 0xdf, 0xc6, 0x0a, 0x6a, 0xa6, 0x4f, 0x83, 0xe3, 0x2f, 0x36, 0xfa, 0x9a, 0x56,
    0x6c, 0xa0, 0xc0, 0x0c, 0x15, 0xd9, 0xb9, 0x75, 0x9c, 0x50, 0x30, 0xfc, 0xe5, 0x29, 0x49, 0x85,
    0x91, 0x5d, 0x3d, 0xf1, 0xe8, 0x24, 0x44, 0x88, 0x61, 0xad, 0xcd, 0x01, 0x18, 0xd4, 0xb4, 0x78,
    0xc5, 0x09, 0x69, 0xa5, 0xbc, 0x70, 0x10, 0xdc, 0x35, 0xf9, 0x99, 0x55, 0x4c, 0x80, 0xe0, 0x2c,
    0x38, 0xf4, 0x94, 0x58, 0x41, 0x8d, 0xed, 0x21, 0xc8, 0x04, 0x64, 0xa8, 0xb1, 0x7d, 0x1d, 0xd1,
    0xeb, 0x27, 0x47, 0x8b, 0x92, 0x5e, 0x3e, 0xf2, 0x1b, 0xd7, 0xb7, 0x7b, 0x62, 0xae, 0xce, 0x02,
    0x16, 0xda, 0xba, 0x76, 0x6f, 0xa3, 0xc3, 0x0f, 0xe6, 0x2a, 0x4a, 0x86, 0x9f, 0x53, 0x33, 0xff,
    0x87, 0x4b, 0x2b, 0xe7, 0xfe, 0x32, 0x52, 0x9e, 0x77, 0xbb, 0xdb, 0x17, 0x0e, 0xc2, 0xa2, 0x6e,
    0x7a, 0xb6, 0xd6, 0x1a, 0x03, 0xcf, 0xaf, 0x63, 0x8a, 0x46, 0x26, 0xea, 0xf3, 0x3f, 0x5f, 0x93,
    0xa9, 0x65, 0x05, 0xc9, 0xd0, 0x1c, 0x7c, 0xb0, 0x59, 0x95, 0xf5, 0x39, 0x20, 0xec, 0x8c, 0x40,
    0x54, 0x98, 0xf8, 0x34, 0x2d, 0xe1, 0x81, 0x4d, 0xa4, 0x68, 0x08, 0xc4, 0xdd, 0x11, 0x71, 0xbd
], dtype=np.uint8)  # yapf: disable

BLOCK_LENGTH = Nn
DATA_LENGTH = Nn - NRoots
PARITY_LENGTH = NRoots


class UncorrectableError(ValueError):
    def __init__(self) -> None:
        super(UncorrectableError, self).__init__('Block contains uncorrectable errors')


def encode(data: Union[bytes, bytearray, memoryview], dual_basis: bool, interleaving: int = 1) -> bytes:
    """
    Calculates parity for data using Reed Solomon RS(255, 223)

    Args:
        data: Data to encode.
        dual_basis: If false, parity will be written in conventional form.
            If true, parity will be written in dual-basis representation.
        interleaving: Interleaving of input data.

    Returns:
        Encoded data.
    """
    error_message = f'Data must be exactly {DATA_LENGTH * interleaving} bytes long'
    if isinstance(data, memoryview):
        assert data.nbytes == DATA_LENGTH * interleaving, error_message
    else:
        assert len(data) == DATA_LENGTH * interleaving, error_message

    data_array = np.frombuffer(data, dtype=np.uint8)

    blocks = [data_array[index::interleaving] for index in range(interleaving)]

    encoded_blocks = [encode_block(block, dual_basis=dual_basis) for block in blocks]

    result = np.vstack(encoded_blocks).reshape((-1, ), order='F')

    return cast(bytes, result.tobytes())


def decode(data: Union[bytes, bytearray, memoryview],
           dual_basis: bool,
           interleaving: int = 1) -> Tuple[List[int], bytes]:
    """
    Decodes an RS(255, 223) encoded block

    Args:
        data: Data to decode.
        dual_basis: If false, block will be treated as conventional form. If true, block will be treated as dual-basis.
        interleaving: Interleaving of input data.

    Returns:
        List of the positions of any known erasures and decoded data.
    """
    if isinstance(data, memoryview):
        assert data.nbytes == Nn * interleaving, f'Data must be exactly {Nn * interleaving} bytes long'
    else:
        assert len(data) == Nn * interleaving, f'Data must be exactly {Nn * interleaving} bytes long'

    data_array = np.frombuffer(data, dtype=np.uint8)

    encoded_blocks = [data_array[index::interleaving] for index in range(interleaving)]

    status = []
    blocks = []

    for encoded_block in encoded_blocks:
        stat, block = decode_block(encoded_block, dual_basis=dual_basis)
        status.append(stat)
        blocks.append(block)

    result = np.vstack(blocks).reshape((-1, ), order='F')

    return status, result.tobytes()


def encode_block(data: Any, dual_basis: bool = False) -> Any:
    """
    Calculates parity for data using Reed Solomon RS(255, 223).

    Arguments:
        data: Data to encode.
        dual_basis: If false, parity will be written in conventional form.
            If true, parity will be written in dual-basis representation.

    Returns:
        Encoded data.
    """
    assert len(data.shape) == 1, 'Data must be one dimensional array'
    assert data.size >= DATA_LENGTH, f'Data must be at least {DATA_LENGTH} bytes long'

    if dual_basis:
        data = TAL_TO_CONVENTIONAL[data]

    parity = np.zeros(PARITY_LENGTH, dtype=np.uint8)

    for i in range(0, Nn - NRoots):
        feedback = INDEX_OF[data[i] ^ parity[0]]
        if feedback != A0:
            for j in range(1, NRoots):
                parity[j] ^= ALPHA_TO[(feedback + GEN_POLY[NRoots - j]) % Nn]

        parity[0:NRoots - 1] = parity[1:NRoots]

        if feedback != A0:
            parity[NRoots - 1] = ALPHA_TO[(feedback + GEN_POLY[0]) % Nn]
        else:
            parity[NRoots - 1] = 0

    if dual_basis:
        data = TAL_TO_DUAL_BASIS[data]
        parity = TAL_TO_DUAL_BASIS[parity]

    return np.concatenate((data, parity))


def decode_block(data: Any, dual_basis: bool = False) -> Tuple[int, Any]:
    """
    Decodes an RS(255, 223) encoded block

    Arguments:
        data: Encoded data. The first 223 bytes must be data, the last 32 bytes must be parity.
        dual_basis: If false, block will be treated as conventional form. If true, block will be treated as dual-basis.

    Returns:
        Number of corrected bytes and decoded data.
    """
    assert len(data.shape) == 1, 'Data must be one dimensional array'
    assert data.size >= BLOCK_LENGTH, f'Data must be at least {BLOCK_LENGTH} bytes long'

    if dual_basis:
        data = TAL_TO_CONVENTIONAL[data]
    else:
        data = np.copy(data)

    syndromes = np.full(NRoots, data[0], dtype=np.uint8)

    for j in range(1, Nn):
        for i in range(0, NRoots):
            if syndromes[i] == 0:
                syndromes[i] = data[j]
            else:
                syndromes[i] = data[j] ^ ALPHA_TO[(INDEX_OF[syndromes[i]] + (Fcr + i) * Prim) % Nn]

    # Convert syndromes to index form, checking for nonzero condition
    syn_error = 0
    for i in range(0, NRoots):
        syn_error |= syndromes[i]
        syndromes[i] = INDEX_OF[syndromes[i]]

    if syn_error == 0:
        # If syndrome is zero, block[] is a codeword and there are no
        # errors to correct. So return block[] unmodified.
        count = 0
    else:
        t = np.zeros(NRoots + 1, dtype=np.int32)
        omega = np.zeros(NRoots + 1, dtype=np.int32)
        root = np.zeros(NRoots, dtype=np.int32)
        reg = np.zeros(NRoots + 1, dtype=np.int32)
        loc = np.zeros(NRoots, dtype=np.int32)
        lambdax = np.zeros(NRoots + 1, dtype=np.int32)
        lambdax[0] = 1

        b = INDEX_OF[lambdax]

        #
        # Begin Berlekamp-Massey algorithm to determine error+erasure
        # locator polynomial
        #

        # r is the step number
        r = 0
        el = 0

        while r + 1 <= NRoots:
            r += 1
            # Compute discrepancy at the r-th step in poly-form
            discrepancy = 0
            for i in range(0, r):
                if (lambdax[i] != 0) and (syndromes[r - i - 1] != A0):
                    discrepancy ^= ALPHA_TO[(INDEX_OF[lambdax[i]] + syndromes[r - i - 1]) % Nn]

            discrepancy = INDEX_OF[discrepancy]

            if discrepancy == A0:
                b[1:NRoots + 1] = b[0:NRoots]

                b[0] = A0
            else:
                t[0] = lambdax[0]
                for i in range(0, NRoots):
                    if b[i] != A0:
                        t[i + 1] = lambdax[i + 1] ^ ALPHA_TO[(discrepancy + b[i]) % Nn]
                    else:
                        t[i + 1] = lambdax[i + 1]

                if 2 * el <= r - 1:
                    el = r - el

                    for i in range(NRoots):
                        b[i] = A0 if lambdax[i] == 0 else (INDEX_OF[lambdax[i]] - discrepancy + Nn) % Nn
                else:
                    b[1:NRoots + 1] = b[0:NRoots]

                    b[0] = A0

                lambdax[0:NRoots + 1] = t[0:NRoots + 1]

        lambdax = INDEX_OF[lambdax]
        degs_lambda = np.where(lambdax != A0)[0]
        if len(degs_lambda) != 0:
            deg_lambda = degs_lambda[-1]
        else:
            deg_lambda = np.int64(0)

        # Find roots of the error+erasure locator polynomial by Chien search
        reg[1:NRoots + 1] = lambdax[1:NRoots + 1]

        count = 0  # Number of roots of lambda(x)

        i = 1
        k = IPrim - 1
        while i <= Nn:
            q = 1  # lambda[0] is always 0
            for j in range(deg_lambda, 0, -1):
                if reg[j] != A0:
                    reg[j] = ((reg[j] + j) % Nn)
                    q ^= ALPHA_TO[reg[j]]

            if q != 0:
                # Not a root
                i += 1
                k = (k + IPrim) % Nn
                continue

            # Store root (index-form) and error location number
            root[count] = i
            loc[count] = k

            # Increment count
            count += 1

            # If we've already found max possible roots,
            # abort the search to save time

            if count == deg_lambda:
                break

            i += 1
            k = (k + IPrim) % Nn

        if deg_lambda != count:
            # deg(lambda) unequal to number of roots => uncorrectable error detected
            raise UncorrectableError()
        else:
            # Compute err+eras evaluator poly omega(x) = s(x)*lambda(x) (modulo x*NRoots). in index form.
            # Also find deg(omega).
            deg_omega = 0
            for i in range(0, NRoots):
                tmp = 0

                j = min(deg_lambda, i)
                while j >= 0:
                    if (syndromes[i - j] != A0) and (lambdax[j] != A0):
                        tmp ^= ALPHA_TO[(syndromes[i - j] + lambdax[j]) % Nn]
                    j -= 1

                if tmp != 0:
                    deg_omega = i

                omega[i] = INDEX_OF[tmp]
            omega[NRoots] = A0

            # Compute error values in poly-form. num1 = omega(inv(X(l))), num2
            # = inv(X(l))**(kFcr-1) and den = lambda_pr(inv(X(l))) all in
            # poly-form

            for j in range(0, count)[::-1]:
                num1 = 0
                for i in range(0, deg_omega + 1)[::-1]:
                    if omega[i] != A0:
                        num1 ^= ALPHA_TO[(omega[i] + (i * root[j])) % Nn]

                num2 = ALPHA_TO[((root[j] * (Fcr - 1)) + Nn) % Nn]
                den = 0

                # lambda[i+1] for i even is the formal derivative lambda_pr of lambda[i]

                # Find the minimum of deg_lambda and NRoots - 1
                start = deg_lambda if (deg_lambda < NRoots - 1) else (NRoots - 1)

                # If this minimum is odd, step it down to the next lowest integer.
                # We can do this with a bithack: AND the number with 0xFFFFFFFE
                start &= ~1

                # Step down through evens
                for i in range(start, -1, -2):
                    if lambdax[i + 1] != A0:
                        den ^= ALPHA_TO[(lambdax[i + 1] + (i * root[j])) % Nn]
                if den == 0:
                    count = -1
                else:
                    # Apply error to data
                    if num1 != 0:
                        data[loc[j]] ^= ALPHA_TO[(INDEX_OF[num1] + INDEX_OF[num2] + Nn - INDEX_OF[den]) % Nn]

    if dual_basis:
        data = TAL_TO_DUAL_BASIS[data]

    return count, data[:Nn - NRoots]
