import * as iam from '@aws-cdk/aws-iam';
import { IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Alias } from './alias';
/**
 * A KMS Key, either managed by this CDK app, or imported.
 *
 * @stability stable
 */
export interface IKey extends IResource {
    /**
     * The ARN of the key.
     *
     * @stability stable
     * @attribute true
     */
    readonly keyArn: string;
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @stability stable
     * @attribute true
     */
    readonly keyId: string;
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(alias: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @param statement The policy statement to add.
     * @param allowNoOp If this is set to `false` and there is no policy defined (i.e. external key), the operation will fail. Otherwise, it will no-op.
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement, allowNoOp?: boolean): iam.AddToResourcePolicyResult;
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
}
declare abstract class KeyBase extends Resource implements IKey {
    /**
     * The ARN of the key.
     */
    abstract readonly keyArn: string;
    abstract readonly keyId: string;
    /**
     * Optional policy document that represents the resource policy of this key.
     *
     * If specified, addToResourcePolicy can be used to edit this policy.
     * Otherwise this method will no-op.
     */
    protected abstract readonly policy?: iam.PolicyDocument;
    /**
     * Optional property to control trusting account identities.
     *
     * If specified, grants will default identity policies instead of to both
     * resource and identity policies. This matches the default behavior when creating
     * KMS keys via the API or console.
     */
    protected abstract readonly trustAccountIdentities: boolean;
    /**
     * Collection of aliases added to the key
     *
     * Tracked to determine whether or not the aliasName should be added to the end of its ID
     */
    private readonly aliases;
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(aliasName: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @param statement The policy statement to add.
     * @param allowNoOp If this is set to `false` and there is no policy defined (i.e. external key), the operation will fail. Otherwise, it will no-op.
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement, allowNoOp?: boolean): iam.AddToResourcePolicyResult;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    private granteeStackDependsOnKeyStack;
    private principalIsANewlyCreatedResource;
    private isGranteeFromAnotherRegion;
    private isGranteeFromAnotherAccount;
}
/**
 * Construction properties for a KMS Key object.
 *
 * @stability stable
 */
export interface KeyProps {
    /**
     * A description of the key.
     *
     * Use a description that helps your users decide
     * whether the key is appropriate for a particular task.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * Initial alias to add to the key.
     *
     * More aliases can be added later by calling `addAlias`.
     *
     * @default - No alias is added for the key.
     * @stability stable
     */
    readonly alias?: string;
    /**
     * Indicates whether AWS KMS rotates the key.
     *
     * @default false
     * @stability stable
     */
    readonly enableKeyRotation?: boolean;
    /**
     * Indicates whether the key is available for use.
     *
     * @default - Key is enabled.
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * Custom policy document to attach to the KMS key.
     *
     * NOTE - If the '@aws-cdk/aws-kms:defaultKeyPolicies' feature flag is set (the default for new projects),
     * this policy will *override* the default key policy and become the only key policy for the key. If the
     * feature flag is not set, this policy will be appended to the default key policy.
     *
     * @default - A policy document with permissions for the account root to
     * administer the key will be created.
     * @stability stable
     */
    readonly policy?: iam.PolicyDocument;
    /**
     * A list of principals to add as key administrators to the key policy.
     *
     * Key administrators have permissions to manage the key (e.g., change permissions, revoke), but do not have permissions
     * to use the key in cryptographic operations (e.g., encrypt, decrypt).
     *
     * These principals will be added to the default key policy (if none specified), or to the specified policy (if provided).
     *
     * @default []
     * @stability stable
     */
    readonly admins?: iam.IPrincipal[];
    /**
     * Whether the encryption key should be retained when it is removed from the Stack.
     *
     * This is useful when one wants to
     * retain access to data that was encrypted with a key that is being retired.
     *
     * @default RemovalPolicy.Retain
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * (deprecated) Whether the key usage can be granted by IAM policies.
     *
     * Setting this to true adds a default statement which delegates key
     * access control completely to the identity's IAM policy (similar
     * to how it works for other AWS resources). This matches the default behavior
     * when creating KMS keys via the API or console.
     *
     * If the '@aws-cdk/aws-kms:defaultKeyPolicies' feature flag is set (the default for new projects),
     * this flag will always be treated as 'true' and does not need to be explicitly set.
     *
     * @default - false, unless the '
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam
     * @deprecated redundant with the '
     * @aws-cdk /aws-kms:defaultKeyPolicies' feature flag
     */
    readonly trustAccountIdentities?: boolean;
}
/**
 * Defines a KMS key.
 *
 * @stability stable
 * @resource AWS::KMS::Key
 */
export declare class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope the construct that will "own" the imported key.
     * @param id the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     * @stability stable
     */
    static fromKeyArn(scope: Construct, id: string, keyArn: string): IKey;
    /**
     * The ARN of the key.
     *
     * @stability stable
     */
    readonly keyArn: string;
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @stability stable
     */
    readonly keyId: string;
    /**
     * Optional policy document that represents the resource policy of this key.
     *
     * If specified, addToResourcePolicy can be used to edit this policy.
     * Otherwise this method will no-op.
     *
     * @stability stable
     */
    protected readonly policy?: iam.PolicyDocument;
    /**
     * Optional property to control trusting account identities.
     *
     * If specified, grants will default identity policies instead of to both
     * resource and identity policies. This matches the default behavior when creating
     * KMS keys via the API or console.
     *
     * @stability stable
     */
    protected readonly trustAccountIdentities: boolean;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: KeyProps);
    /**
     * Grant admins permissions using this key to the given principal.
     *
     * Key administrators have permissions to manage the key (e.g., change permissions, revoke), but do not have permissions
     * to use the key in cryptographic operations (e.g., encrypt, decrypt).
     *
     * @stability stable
     */
    grantAdmin(grantee: iam.IGrantable): iam.Grant;
    /**
     * Adds the default key policy to the key. This policy gives the AWS account (root user) full access to the CMK,
     * which reduces the risk of the CMK becoming unmanageable and enables IAM policies to allow access to the CMK.
     * This is the same policy that is default when creating a Key via the KMS API or Console.
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     */
    private addDefaultAdminPolicy;
    /**
     * Grants the account admin privileges -- not full account access -- plus the GenerateDataKey action.
     * The GenerateDataKey action was added for interop with S3 in https://github.com/aws/aws-cdk/issues/3458.
     *
     * This policy is discouraged and deprecated by the '@aws-cdk/aws-kms:defaultKeyPolicies' feature flag.
     *
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @deprecated
     */
    private addLegacyAdminPolicy;
}
export {};
