# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['qmm']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.19.4,<2.0.0']

setup_kwargs = {
    'name': 'qmm',
    'version': '0.12.1',
    'description': 'Quadratic Majorize-Minimize Python toolbox',
    'long_description': "Q-MM: A Python toolbox for Quadratic Majorization-Minimization\n==============================================================\n\n[![DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.6373070.svg)](https://doi.org/10.5281/zenodo.6373070)\n![licence](https://img.shields.io/github/license/forieux/qmm)\n![pypi](https://img.shields.io/pypi/v/qmm)\n![status](https://img.shields.io/pypi/status/qmm)\n![version](https://img.shields.io/pypi/pyversions/qmm)\n![maintained](https://img.shields.io/maintenance/yes/2022)\n[![Documentation Status](https://readthedocs.org/projects/qmm/badge/?version=latest)](https://qmm.readthedocs.io/en/latest/?badge=latest)\n\n![image](./docs/qmm.png)\n\nQ-MM is a Python implementation of Majorize-Minimize Quadratic\noptimization algorithms. Algorithms provided here come from\n\n    [1] C. Labat and J. Idier, “Convergence of Conjugate Gradient Methods with a\n    Closed-Form Stepsize Formula,” J Optim Theory Appl, p. 18, 2008.\n\nand\n\n    [2] E. Chouzenoux, J. Idier, and S. Moussaoui, “A Majorize–Minimize Strategy\n    for Subspace Optimization Applied to Image Restoration,” IEEE Trans. on\n    Image Process., vol. 20, no. 6, pp. 1517–1528, Jun. 2011, doi:\n    10.1109/TIP.2010.2103083.\n\nSee [documentation](https://qmm.readthedocs.io/en/stable/index.html) for more\nbackground. If you use this code, please cite the references above and a\ncitation of this toolbox will also be appreciated, see [below](#citation). You\ncan also click ⭐ on the repo.\n\nQuadratic Majorize-Minimize\n---------------------------\n\nThe Q-MM optimization algorithms compute the minimizer of objective\nfunction like\n\nJ(x) = ∑ₖ μₖ ψₖ(Vₖ·x - ωₖ)\n\nwhere x is the unknown vector, Vₖ a linear operator, ωₖ a fixed data, μₖ\na scalar, ψₖ(u) = ∑ᵢφₖ(uᵢ), and φₖ a function that must be\ndifferentiable, even, coercive, φ(√·) concave, and 0 \\< φ\\'(u) / u \\<\n+∞.\n\nThe optimization is done thanks to quadratic sugorate function. In\nparticular, no linesearch or sub-iteration is necessary, and close form\nformula for the step are used with guaranteed convergence.\n\nA classical example, like in the figure below that show an image\ndeconvolution problem, is the resolution of an inverse problem with the\nminimization of\n\nJ(x) = \\|\\|² + μ ψ(V·x)\n\nwhere H is a low-pass forward model, V a regularization operator that\napproximate gradient (kind of high-pass filter) and ψ an edge preserving\nfunction like Huber. The above objective is obtained with k ∈ {1, 2},\nψ₁(·) = \\|\\|², V₁ = H, ω₁ = y, and ω₂ = 0.\n\n![image](./docs/image.png)\n\nFeatures\n--------\n\n-   The `mmmg`, Majorize-Minimize Memory Gradient algorithm. See\n    documentation and \\[2\\] for details.\n-   The `mmcg`, Majorize-Minimize Conjugate Gradient algorithm. See\n    documentation and \\[1\\] for details.\n-   **No linesearch**: the step is obtained from a close form formula\n    without sub-iteration.\n-   **No conjugacy choice**: a conjugacy strategy is not necessary\n    thanks to the subspace nature of the algorithms. The `mmcg`\n    algorithm use a Polak-Ribière formula.\n-   Generic and flexible: there is no restriction on the number of\n    regularizer, their type, ..., as well as for data adequacy.\n-   Provided base class for objectives and losses allowing easy and fast\n    implementation.\n-   Just one file if you like quick and dirty installation, but\n    available with `pip`.\n-   Comes with examples of implemented linear operator.\n\nInstallation and documentation\n------------------------------\n\nQ-MM is essentially just one file `qmm.py`. We recommend using poetry\nfor installation\n\n``` {.sourceCode .sh}\npoetry add qmm\n```\n\nThe package can also be installed with pip. More options are described\nin the [documentation](https://qmm.readthedocs.io/en/stable/index.html).\n\nQ-MM only depends on `numpy` and Python 3.6.\n\nExample\n-------\n\nThe `demo.py` presents an example on image deconvolution. The first step\nis to implement the operators `V` and the adjoint `Vᵀ` as callable\n(function or methods). The user is in charge of these operators and\nthese callable must accept a unique Numpy array `x` and a unique return\nvalue\n([partial](https://docs.python.org/fr/3.9/library/functools.html#functools.partial)\nin the `functools` module in the standard library is usefull here).\nThere is no constraints on the shape, everything is vectorized\ninternally.\n\nAfter import of `qmm`, user must instantiate `Potential` objects that\nimplement `φ` and `Objective` objects that implement `μ ψ(V·x - ω)`\n\n``` {.sourceCode .python}\nimport qmm\nphi = qmm.Huber(delta=10)  # φ\n\ndata_adeq = qmm.QuadObjective(H, Ht, HtH, data=data)  # ||y - H·x||²\nprior = qmm.Objective(V, Vt, phi, hyper=0.01)  # μ ψ(V·x) = μ ∑ᵢ φ(vᵢᵗ·x)\n```\n\nThen you can run the algorithm\n\n``` {.sourceCode .python}\nres = qmm.mmmg([data_adeq, prior], init, max_iter=200)\n```\n\nwhere `[data_adeq, prior]`{.sourceCode} means that the two objective\nfunctions are summed. For more details, see\n[documentation](https://qmm.readthedocs.io/en/stable/index.html).\n\nContribute\n----------\n\n-   Source code: <https://github.com/forieux/qmm>\n-   Issue tracker: <https://github.com/forieux/qmm/issues>\n\nAuthor\n------\n\nIf you are having issues, please let us know\n\norieux AT l2s.centralesupelec.fr\n\nMore information about me [here](https://pro.orieux.fr). F. Orieux and\nR. Abirizk are affiliated to the Signal and Systems Laboratory\n[L2S](https://l2s.centralesupelec.fr/).\n\nCitation\n--------\n\nQ-MM has a DOI with Zenodo\n[![DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.6373069.svg)](https://doi.org/10.5281/zenodo.6373069).\nSpecific version can also be cited. Citation can be\n\n    François Orieux, & Ralph Abirizk. (2022). Q-MM: The Quadratic Majorize-Minimize\n    Python toolbox (v0.12.0). Zenodo. https://doi.org/10.5281/zenodo.6373070\n\nA example of bibtex is\n\n    @software{francois_orieux_2022_6373070,\n      author       = {François Orieux and Ralph Abirizk},\n      title        = {Q-MM: The Quadratic Majorize-Minimize Python toolbox},\n      month        = mar,\n      year         = 2022,\n      publisher    = {Zenodo},\n      version      = {0.12.0},\n      doi          = {10.5281/zenodo.6373069},\n      url          = {https://doi.org/10.5281/zenodo.6373069}\n    }\n\nAcknowledgement\n---------------\n\nAuthor would like to thanks [J.\nIdier](https://pagespersowp.ls2n.fr/jeromeidier/en/jerome-idier-3/), [S.\nMoussaoui](https://scholar.google.fr/citations?user=Vkr8yxkAAAAJ&hl=fr)\nand [É. Chouzenoux](http://www-syscom.univ-mlv.fr/~chouzeno/). É.\nChouzenoux has also a Matlab package that implements 3MG for image\ndeconvolution that can be found on her\n[webpage](http://www-syscom.univ-mlv.fr/~chouzeno/Logiciel.html).\n\nLicense\n-------\n\nThe project is licensed under the GPLv3 license.\n",
    'author': 'François Orieux',
    'author_email': 'francois.orieux@universite-paris-saclay.fr',
    'maintainer': 'François Orieux',
    'maintainer_email': 'francois.orieux@universite-paris-saclay.fr',
    'url': 'https://qmm.readthedocs.io/en/stable/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.11',
}


setup(**setup_kwargs)
