"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
describe('Version', () => {
    describe('.isGreaterThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '10.0.9.2',
                    secondVersion: '10.1.9.2',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ], [
                'greater than',
                {
                    firstVersion: '2.0.0.1',
                    secondVersion: '2.0.0.0',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isGreaterThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isGreaterThan constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 1, 9, 2]);
        const result = lhs.isGreaterThan(lib_1.Version.parse('10.0.9.2'));
        expect(result).toEqual(true);
    });
    describe('.isLessThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'greater minor version',
                {
                    firstVersion: '10.1.9.2',
                    secondVersion: '10.0.9.2',
                    expectedValue: false,
                },
            ], [
                'greater patch version',
                {
                    firstVersion: '1.1.1.2',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '2.0.0.0',
                    secondVersion: '2.0.0.1',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isLessThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isEqual', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: true,
                },
            ], [
                'unequal',
                {
                    firstVersion: '2.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isEqual(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isLessThan using constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 0, 9, 2]);
        const result = lhs.isLessThan(lib_1.Version.parse('10.1.9.2'));
        expect(result).toEqual(true);
    });
    describe('constructor validation', () => {
        test.each([
            [
                'incorrect component count',
                {
                    version: [10, 1, 9],
                    expectedException: /Invalid version format/,
                },
            ], [
                'negative value',
                {
                    version: [10, -1, 9],
                    expectedException: /Invalid version format/,
                },
            ], [
                'decimal value',
                {
                    version: [10, 1, 9.2],
                    expectedException: /Invalid version format/,
                },
            ], [
                'correct value',
                {
                    version: [10, 1, 9, 2],
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => new lib_1.Version(version)).toThrow(expectedException);
            }
            else {
                expect(() => new lib_1.Version(version)).not.toThrow();
            }
        });
    });
    describe('.parse throws exception', () => {
        test.each([
            [
                'ending with .',
                {
                    version: '10.1.9.',
                    expectedException: /Invalid version format/,
                },
            ], [
                'empty string',
                {
                    version: '',
                    expectedException: /Invalid version format/,
                },
            ], [
                'negative value',
                {
                    version: '10.-1.9.2',
                    expectedException: /Invalid version format/,
                },
            ], [
                'correct version',
                {
                    version: '10.1.9.2',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => lib_1.Version.parse(version)).toThrow(expectedException);
            }
            else {
                expect(() => lib_1.Version.parse(version)).not.toThrow();
            }
        });
    });
});
describe('VersionQuery', () => {
    beforeEach(() => {
        stack = new core_1.Stack(undefined, undefined);
    });
    describe('constructor', () => {
        test('throws not implemented error for empty version string', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version', {
                    version: '',
                });
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '',
            }));
        });
        test('throws not implemented error for valid version string', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version', {
                    version: '1.2',
                });
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '1.2',
            }));
        });
        test('throws not implemented error without props', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version');
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '',
            }));
        });
    });
    // GIVEN
    const majorVersion = 1;
    const minorVersion = 2;
    const releaseVersion = 3;
    const patchVersion = 4;
    const expectedVersionString = `${majorVersion}.${minorVersion}.${releaseVersion}.${patchVersion}`;
    let version;
    function exactVersionTests() {
        test('does not create a custom resource', () => {
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource'));
        });
        test('does not create a lambda', () => {
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
        });
        test('populates version components', () => {
            // THEN
            expect(version.majorVersion).toEqual(1);
            expect(version.minorVersion).toEqual(2);
            expect(version.releaseVersion).toEqual(3);
        });
        test('provides linux installers', () => {
            // GIVEN
            const linuxFullVersionString = version.linuxFullVersionString();
            // THEN
            expect(version.linuxInstallers).toBeDefined();
            expect(linuxFullVersionString).toBeDefined();
            expect(linuxFullVersionString).toMatch(expectedVersionString);
            expect(version.linuxInstallers.repository).toBeDefined();
            expect(version.linuxInstallers.repository.s3Bucket.bucketName).toMatch('thinkbox-installers');
            expect(version.linuxInstallers.repository.objectKey).toMatch(`DeadlineRepository-${expectedVersionString}-linux-x64-installer.run`);
        });
    }
    describe('.exact()', () => {
        beforeEach(() => {
            version = lib_1.VersionQuery.exact(stack, 'version', {
                majorVersion,
                minorVersion,
                releaseVersion,
                patchVersion,
            });
        });
        exactVersionTests();
    });
    describe('.exactString()', () => {
        beforeEach(() => {
            version = lib_1.VersionQuery.exactString(stack, 'version', expectedVersionString);
        });
        exactVersionTests();
        test.each([
            [''],
            ['abc'],
            ['1'],
            ['1.2'],
            ['1.2.3'],
            ['1.2.3.4a'],
            ['a1.2.3.4a'],
            ['a1.2.3.4'],
            [' 1.2.3.4 '],
            ['a.b.c.d'],
            ['-1.0.2.3'],
            ['.1.0.2.3'],
        ])('throws an error on invalid versions %p', (versionStr) => {
            // WHEN
            function when() {
                lib_1.VersionQuery.exactString(stack, 'version', versionStr);
            }
            // THEN
            expect(when).toThrowError(new RegExp(`^Invalid version format. Expected format 'a.b.c.d', found '${versionStr}'$`));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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