"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let stage;
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    const RCS_RECIPE_NAME = 'rcs';
    const RCS_RECIPE = {
        description: 'rcs',
        title: 'rcs',
        buildArgs: {
            a: 'a',
            b: 'b',
        },
        target: undefined,
    };
    const LICENSE_FORWARDER_RECIPE_NAME = 'license-forwarder';
    const LICENSE_FORWARDER_RECIPE = {
        title: 'license-forwarder',
        description: 'license-forwarder',
        buildArgs: {
            c: 'c',
            d: 'd',
        },
        target: 'lf',
    };
    const MAJOR_VERSION = 10;
    const MINOR_VERSION = 1;
    const RELEASE_VERSION = 9;
    const PATCH_VERSION = 2;
    const FULL_VERSION_STRING = `${MAJOR_VERSION}.${MINOR_VERSION}.${RELEASE_VERSION}.${PATCH_VERSION}`;
    beforeEach(() => {
        app = new core_1.App();
        class TestStage extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        stage = new TestStage({
            path: STAGE_PATH,
            manifest: {
                schema: 1,
                version: FULL_VERSION_STRING,
                recipes: {
                    [RCS_RECIPE_NAME]: RCS_RECIPE,
                    [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
                },
            },
        });
        stack = new core_1.Stack(app, 'Stack');
    });
    /**
     * Tests that the remote connection server and license forwarder Docker recipes
     * create the correct {@link DockerImageAsset} instances using the build arguments,
     * and target from the supplied manifest.
     *
     * This is done by comparing the `sourceHash` property of the expected vs actual
     * DockerImageAsset instances.
     */
    test.each([
        [
            'remoteConnnectionServer',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: RCS_RECIPE.buildArgs,
                    target: RCS_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.remoteConnectionServer;
            },
        ],
        [
            'licenseForwarder',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: LICENSE_FORWARDER_RECIPE.buildArgs,
                    target: LICENSE_FORWARDER_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.licenseForwarder;
            },
        ],
    ])('has proper %p container image', (_imageName, getExpected, getActual) => {
        // GIVEN
        const expectedImage = getExpected();
        // WHEN
        const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        const actualImage = getActual(recipes);
        // THEN
        expect(actualImage.sourceHash).toEqual(expectedImage.sourceHash);
    });
    describe('provides the Deadline version with', () => {
        let version;
        beforeEach(() => {
            // WHEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            version = recipes.version;
        });
        test('majorVersion attribute', () => {
            // THEN
            expect(version.majorVersion).toEqual(MAJOR_VERSION);
        });
        test('minorVersion attribute', () => {
            // THEN
            expect(version.minorVersion).toEqual(MINOR_VERSION);
        });
        test('releaseVersion component', () => {
            // THEN
            expect(version.releaseVersion).toEqual(RELEASE_VERSION);
        });
        test('linuxFullVersionString() result', () => {
            // THEN
            expect(version.linuxFullVersionString()).toEqual(FULL_VERSION_STRING);
        });
        test('linuxInstallers', () => {
            var _a, _b, _c, _d, _e, _f;
            // THEN
            expect(version.linuxInstallers).toBeDefined();
            expect((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.patchVersion).toEqual(PATCH_VERSION);
            expect((_b = version.linuxInstallers) === null || _b === void 0 ? void 0 : _b.repository).toBeDefined();
            expect((_d = (_c = version.linuxInstallers) === null || _c === void 0 ? void 0 : _c.repository) === null || _d === void 0 ? void 0 : _d.s3Bucket.bucketName).toEqual('thinkbox-installers');
            expect((_f = (_e = version.linuxInstallers) === null || _e === void 0 ? void 0 : _e.repository) === null || _f === void 0 ? void 0 : _f.objectKey).toEqual(`Deadline/${FULL_VERSION_STRING}/Linux/DeadlineRepository-${FULL_VERSION_STRING}-linux-x64-installer.run`);
        });
    });
    test.each([
        ['rcs', {
                [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
            }],
        ['license-forwarder', {
                [RCS_RECIPE_NAME]: RCS_RECIPE,
            }],
    ])('manifest missing required recipe %s throws error', (_recipeName, recipes) => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        class StageWithPublicConstructor extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        const manifest = {
            recipes,
            schema: 1,
            version: '1.2.3.4',
        };
        // WHEN
        expect(() => {
            new lib_1.ThinkboxDockerRecipes(isolatedStack, 'Recipes', {
                stage: new StageWithPublicConstructor({
                    manifest,
                    path: '/some/path',
                }),
            });
        })
            // THEN
            .toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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