"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Stage', () => {
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    /**
     * A sub-class of Stage that makes the constructor public for testing purposes.
     *
     * This is to be able to specify a manifest rather than load it from the file-system via {@link Stage.fromDirectory}.
     */
    class StageWithPulicConstructor extends lib_1.Stage {
        constructor(props) {
            super(props);
        }
    }
    beforeEach(() => {
        jest.resetModules();
    });
    describe('.fromDirectory', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {},
        };
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        let stage;
        beforeAll(() => {
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            stage = require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        });
        afterAll(() => {
            readFileSync.mockRestore();
            jest.resetModules();
        });
        test('has dirPath', () => {
            // THEN
            expect(stage.dirPath).toBe(STAGE_PATH);
        });
        test('loads manifest.json from directory', () => {
            // THEN
            const expectedPath = path.join(STAGE_PATH, 'manifest.json');
            expect(readFileSync).toHaveBeenCalledWith(expectedPath, expect.objectContaining({
                encoding: 'utf8',
            }));
            expect(stage.manifest).toEqual(manifest);
        });
    });
    describe('.fromDirectory', () => {
        test('not supported version failure', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.8.0',
                recipes: {},
            };
            // WHEN
            const readFileSync = jest.spyOn(fs, 'readFileSync');
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            // THEN
            expect(() => {
                require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
            }).toThrow('Staged Deadline Version (10.1.8.0) is less than the minimum supported version (10.1.9.2)');
            readFileSync.mockRestore();
            jest.resetModules();
        });
    });
    test('has manifest', () => {
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {
                a: {
                    title: 'a-title',
                    description: 'a-description',
                    buildArgs: {
                        argone: 'a-argone-value',
                        argtwo: 'a-argtwo-value',
                    },
                },
                b: {
                    title: 'b-title',
                    description: 'b-description',
                    buildArgs: {
                        argone: 'b-argone-value',
                        argtwo: 'b-argtwo-value',
                    },
                },
            },
        };
        const stage = new StageWithPulicConstructor({
            manifest,
            path: STAGE_PATH,
        });
        // THEN
        expect(stage.manifest).toEqual(manifest);
    });
    describe('manifest validation', () => {
        test.each([
            [
                'mainfest wrong type',
                {
                    manifest: 1,
                    expectedError: /Expected object but got/,
                },
            ],
            [
                'manifest version not string',
                {
                    manifest: {
                        version: 1,
                        recipes: {},
                    },
                },
            ],
            [
                'missing schema',
                {
                    manifest: {
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "schema" key/,
                },
            ],
            [
                'wrong schema type', {
                    manifest: {
                        schema: 'a',
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Expected a numeric "schema" but got:/,
                },
            ],
            [
                'missing version', {
                    manifest: {
                        schema: 1,
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "version" key/,
                },
            ],
            [
                'incorrect version format', {
                    manifest: {
                        schema: 1,
                        version: '10.1.',
                        recipes: {},
                    },
                    expectedError: /Invalid version format/,
                },
            ],
            [
                'version not supported', {
                    manifest: {
                        schema: 1,
                        version: '10.1.0.0',
                        recipes: {},
                    },
                    expectedError: 'Staged Deadline Version (10.1.0.0) is less than the minimum supported version (10.1.9.2)',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { manifest, expectedError } = testcase;
            // WHEN
            function when() {
                new StageWithPulicConstructor({
                    path: STAGE_PATH,
                    manifest,
                });
            }
            expect(when).toThrow(expectedError);
        });
    });
    describe('valid recipes', () => {
        test.each([
            [
                'build args and no target',
                {
                    title: 'Eugene\'s Favorite Cookies',
                    description: 'C is for cookie. That\'s good enough for me',
                    buildArgs: {
                        flour: '3 cups',
                        butter: '1 cup',
                        sugar: '1/2 cup',
                        egg: '2',
                        bakingpowder: '1 tsp',
                        bakingsoda: '1 tsb',
                        vanilla: '1 tsp',
                        salt: '1 pinch',
                    },
                },
            ],
            [
                'buildArgs and a target',
                {
                    title: 'Josh\'s S\'mores',
                    description: 'a secret family recipe',
                    buildArgs: {
                        grahamcracker: '2',
                        marshmellow: '1',
                        chocolate: '2',
                        campfire: '1',
                    },
                    target: 'simplicity',
                },
            ],
            [
                'target and no buildArgs',
                {
                    title: 'Jericho\s special brownie batch with a strawberry milkshake',
                    description: 'a slight improvement to the original recipe',
                    target: 'target',
                },
            ],
            [
                'no target or buildArgs',
                {
                    title: 'Yash\'s Tequila Lime Shot (TLS)',
                    description: 'A sure-fire way to brighten your day',
                },
            ],
        ])('%s', (_scenario, recipe) => {
            // GIVEN
            const manifest = {
                recipes: {
                    recipeName: recipe,
                },
                schema: 1,
                version: '10.1.9.2',
            };
            // WHEN
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // THEN
            expect(stage.manifest.recipes.recipeName).toEqual(recipe);
        });
    });
    describe('.imageFromRecipe()', () => {
        let app;
        let stack;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
        });
        test('creates DockerImageAssets from existing recipes', () => {
            jest.resetModules();
            const mockDockerImageAssetConstructor = jest.fn();
            jest.mock('@aws-cdk/aws-ecr-assets', () => {
                class DockerImageAsset {
                    constructor(...args) {
                        mockDockerImageAssetConstructor(...args);
                    }
                }
                return {
                    DockerImageAsset,
                };
            });
            // Import the class under test now that the ec2 mock is in place
            const ReloadedStage = jest.requireActual('../lib').Stage; // eslint-disable-line
            class ReloadedStageWithPublicConstructor extends ReloadedStage {
                constructor(props) {
                    super(props);
                }
            }
            // GIVEN
            const recipeName = 'myname';
            const recipe = {
                description: 'some description',
                title: 'some title',
                buildArgs: {
                    a: 'a',
                    b: 'b',
                },
            };
            const stage = new ReloadedStageWithPublicConstructor({
                path: STAGE_PATH,
                manifest: {
                    version: '10.1.9.2',
                    schema: 1,
                    recipes: {
                        [recipeName]: recipe,
                    },
                },
            });
            // WHEN
            stage.imageFromRecipe(stack, 'Image', recipeName);
            // THEN
            expect(mockDockerImageAssetConstructor).toHaveBeenCalledWith(stack, 'Image', {
                ...recipe,
                directory: STAGE_PATH,
            });
            expect(mockDockerImageAssetConstructor.mock.calls[0][2]).not.toMatchObject({
                target: expect.anything(),
            });
        });
        test('throws exception when recipe does not exist', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const invalidRecipeName = 'this-recipe-does-not-exist';
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            function when() {
                stage.imageFromRecipe(stack, 'Image', invalidRecipeName);
            }
            // THEN
            expect(when).toThrowError(`No such recipe: ${invalidRecipeName}`);
        });
    });
    describe('.getVersion()', () => {
        test('returns a version in the success case', () => {
            var _a;
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            const version = stage.getVersion(stack, 'Version');
            const linuxFullVersionString = version.linuxFullVersionString();
            // THEN
            expect(version.majorVersion).toEqual(10);
            expect(version.minorVersion).toEqual(1);
            expect(version.releaseVersion).toEqual(9);
            expect(version.linuxInstallers).toBeDefined();
            expect((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.patchVersion).toEqual(2);
            expect(linuxFullVersionString).toBeDefined();
            expect(linuxFullVersionString).toEqual('10.1.9.2');
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhZ2UudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0YWdlLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCx5QkFBeUI7QUFDekIsNkJBQTZCO0FBRTdCLHdDQUd1QjtBQUV2QixnQ0FLZ0I7QUFFaEIsUUFBUSxDQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUU7SUFDckIsUUFBUTtJQUNSLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBRWxEOzs7O09BSUc7SUFDSCxNQUFNLHlCQUEwQixTQUFRLFdBQUs7UUFDM0MsWUFBWSxLQUFpQjtZQUMzQixLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDZixDQUFDO0tBQ0Y7SUFFRCxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDO0lBQ3RCLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGdCQUFnQixFQUFFLEdBQUcsRUFBRTtRQUM5QixRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQWE7WUFDekIsTUFBTSxFQUFFLENBQUM7WUFDVCxPQUFPLEVBQUUsVUFBVTtZQUNuQixPQUFPLEVBQUUsRUFBRTtTQUNaLENBQUM7UUFFRixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNwRCxJQUFJLEtBQVksQ0FBQztRQUVqQixTQUFTLENBQUMsR0FBRyxFQUFFO1lBQ2IsWUFBWSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDdkQsS0FBSyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsc0JBQXNCO1FBQ25GLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLEdBQUcsRUFBRTtZQUNaLFlBQVksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUMzQixJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDdEIsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxFQUFFLEdBQUcsRUFBRTtZQUN2QixPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDekMsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1lBQzlDLE9BQU87WUFDUCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUU1RCxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUMsb0JBQW9CLENBQ3ZDLFlBQVksRUFDWixNQUFNLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3RCLFFBQVEsRUFBRSxNQUFNO2FBQ2pCLENBQUMsQ0FDSCxDQUFDO1lBQ0YsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDM0MsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7UUFDOUIsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtZQUN6QyxRQUFRO1lBQ1IsTUFBTSxRQUFRLEdBQWE7Z0JBQ3pCLE1BQU0sRUFBRSxDQUFDO2dCQUNULE9BQU8sRUFBRSxVQUFVO2dCQUNuQixPQUFPLEVBQUUsRUFBRTthQUNaLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDcEQsWUFBWSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFFdkQsT0FBTztZQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLENBQUEsQ0FBQyxzQkFBc0I7WUFDMUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDBGQUEwRixDQUFDLENBQUM7WUFFdkcsWUFBWSxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQzNCLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUN0QixDQUFDLENBQUMsQ0FBQztJQUVMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7UUFDeEIsTUFBTSxRQUFRLEdBQWE7WUFDekIsTUFBTSxFQUFFLENBQUM7WUFDVCxPQUFPLEVBQUUsVUFBVTtZQUNuQixPQUFPLEVBQUU7Z0JBQ1AsQ0FBQyxFQUFFO29CQUNELEtBQUssRUFBRSxTQUFTO29CQUNoQixXQUFXLEVBQUUsZUFBZTtvQkFDNUIsU0FBUyxFQUFFO3dCQUNULE1BQU0sRUFBRSxnQkFBZ0I7d0JBQ3hCLE1BQU0sRUFBRSxnQkFBZ0I7cUJBQ3pCO2lCQUNGO2dCQUNELENBQUMsRUFBRTtvQkFDRCxLQUFLLEVBQUUsU0FBUztvQkFDaEIsV0FBVyxFQUFFLGVBQWU7b0JBQzVCLFNBQVMsRUFBRTt3QkFDVCxNQUFNLEVBQUUsZ0JBQWdCO3dCQUN4QixNQUFNLEVBQUUsZ0JBQWdCO3FCQUN6QjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sS0FBSyxHQUFHLElBQUkseUJBQXlCLENBQUM7WUFDMUMsUUFBUTtZQUNSLElBQUksRUFBRSxVQUFVO1NBQ2pCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUMzQyxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7UUFDbkMsSUFBSSxDQUFDLElBQUksQ0FBMkU7WUFDbEY7Z0JBQ0UscUJBQXFCO2dCQUNyQjtvQkFDRSxRQUFRLEVBQUUsQ0FBQztvQkFDWCxhQUFhLEVBQUUseUJBQXlCO2lCQUN6QzthQUNGO1lBQ0Q7Z0JBQ0UsNkJBQTZCO2dCQUM3QjtvQkFDRSxRQUFRLEVBQUU7d0JBQ1IsT0FBTyxFQUFFLENBQUM7d0JBQ1YsT0FBTyxFQUFFLEVBQUU7cUJBQ1o7aUJBQ0Y7YUFDRjtZQUNEO2dCQUNFLGdCQUFnQjtnQkFDaEI7b0JBQ0UsUUFBUSxFQUFFO3dCQUNSLE9BQU8sRUFBRSxVQUFVO3dCQUNuQixPQUFPLEVBQUUsRUFBRTtxQkFDWjtvQkFDRCxhQUFhLEVBQUUsbUNBQW1DO2lCQUNuRDthQUNGO1lBQ0Q7Z0JBQ0UsbUJBQW1CLEVBQUU7b0JBQ25CLFFBQVEsRUFBRTt3QkFDUixNQUFNLEVBQUUsR0FBRzt3QkFDWCxPQUFPLEVBQUUsVUFBVTt3QkFDbkIsT0FBTyxFQUFFLEVBQUU7cUJBQ1o7b0JBQ0QsYUFBYSxFQUFFLHNDQUFzQztpQkFDdEQ7YUFDRjtZQUNEO2dCQUNFLGlCQUFpQixFQUFFO29CQUNqQixRQUFRLEVBQUU7d0JBQ1IsTUFBTSxFQUFFLENBQUM7d0JBQ1QsT0FBTyxFQUFFLEVBQUU7cUJBQ1o7b0JBQ0QsYUFBYSxFQUFFLG9DQUFvQztpQkFDcEQ7YUFDRjtZQUNEO2dCQUNFLDBCQUEwQixFQUFFO29CQUMxQixRQUFRLEVBQUU7d0JBQ1IsTUFBTSxFQUFFLENBQUM7d0JBQ1QsT0FBTyxFQUFFLE9BQU87d0JBQ2hCLE9BQU8sRUFBRSxFQUFFO3FCQUNaO29CQUNELGFBQWEsRUFBRSx3QkFBd0I7aUJBQ3hDO2FBQ0Y7WUFDRDtnQkFDRSx1QkFBdUIsRUFBRTtvQkFDdkIsUUFBUSxFQUFFO3dCQUNSLE1BQU0sRUFBRSxDQUFDO3dCQUNULE9BQU8sRUFBRSxVQUFVO3dCQUNuQixPQUFPLEVBQUUsRUFBRTtxQkFDWjtvQkFDRCxhQUFhLEVBQUUsMEZBQTBGO2lCQUMxRzthQUNGO1NBQ0YsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRTtZQUMzQixNQUFNLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxHQUFHLFFBQVEsQ0FBQztZQUM3QyxPQUFPO1lBQ1AsU0FBUyxJQUFJO2dCQUNYLElBQUkseUJBQXlCLENBQUM7b0JBQzVCLElBQUksRUFBRSxVQUFVO29CQUNoQixRQUFRO2lCQUNULENBQUMsQ0FBQztZQUNMLENBQUM7WUFFRCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3RDLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRTtRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1I7Z0JBQ0UsMEJBQTBCO2dCQUMxQjtvQkFDRSxLQUFLLEVBQUUsNEJBQTRCO29CQUNuQyxXQUFXLEVBQUUsNkNBQTZDO29CQUMxRCxTQUFTLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLFFBQVE7d0JBQ2YsTUFBTSxFQUFFLE9BQU87d0JBQ2YsS0FBSyxFQUFFLFNBQVM7d0JBQ2hCLEdBQUcsRUFBRSxHQUFHO3dCQUNSLFlBQVksRUFBRSxPQUFPO3dCQUNyQixVQUFVLEVBQUUsT0FBTzt3QkFDbkIsT0FBTyxFQUFFLE9BQU87d0JBQ2hCLElBQUksRUFBRSxTQUFTO3FCQUNoQjtpQkFDRjthQUNGO1lBQ0Q7Z0JBQ0Usd0JBQXdCO2dCQUN4QjtvQkFDRSxLQUFLLEVBQUUsa0JBQWtCO29CQUN6QixXQUFXLEVBQUUsd0JBQXdCO29CQUNyQyxTQUFTLEVBQUU7d0JBQ1QsYUFBYSxFQUFFLEdBQUc7d0JBQ2xCLFdBQVcsRUFBRSxHQUFHO3dCQUNoQixTQUFTLEVBQUUsR0FBRzt3QkFDZCxRQUFRLEVBQUUsR0FBRztxQkFDZDtvQkFDRCxNQUFNLEVBQUUsWUFBWTtpQkFDckI7YUFDRjtZQUNEO2dCQUNFLHlCQUF5QjtnQkFDekI7b0JBQ0UsS0FBSyxFQUFFLDZEQUE2RDtvQkFDcEUsV0FBVyxFQUFFLDZDQUE2QztvQkFDMUQsTUFBTSxFQUFFLFFBQVE7aUJBQ2pCO2FBQ0Y7WUFDRDtnQkFDRSx3QkFBd0I7Z0JBQ3hCO29CQUNFLEtBQUssRUFBRSxpQ0FBaUM7b0JBQ3hDLFdBQVcsRUFBRSxzQ0FBc0M7aUJBQ3BEO2FBQ0Y7U0FDRixDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsU0FBaUIsRUFBRSxNQUFjLEVBQUUsRUFBRTtZQUM3QyxRQUFRO1lBQ1IsTUFBTSxRQUFRLEdBQWE7Z0JBQ3pCLE9BQU8sRUFBRTtvQkFDUCxVQUFVLEVBQUUsTUFBTTtpQkFDbkI7Z0JBQ0QsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsT0FBTyxFQUFFLFVBQVU7YUFDcEIsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLHlCQUF5QixDQUFDO2dCQUMxQyxRQUFRO2dCQUNSLElBQUksRUFBRSxVQUFVO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzVELENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFO1FBQ2xDLElBQUksR0FBUSxDQUFDO1FBQ2IsSUFBSSxLQUFZLENBQUM7UUFFakIsVUFBVSxDQUFDLEdBQUcsRUFBRTtZQUNkLFFBQVE7WUFDUixHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztZQUNoQixLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2xDLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGlEQUFpRCxFQUFFLEdBQUcsRUFBRTtZQUMzRCxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDcEIsTUFBTSwrQkFBK0IsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUM7WUFDbEQsSUFBSSxDQUFDLElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7Z0JBRXhDLE1BQU0sZ0JBQWdCO29CQUNwQixZQUFZLEdBQUcsSUFBVzt3QkFDeEIsK0JBQStCLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztvQkFDM0MsQ0FBQztpQkFDRjtnQkFFRCxPQUFPO29CQUNMLGdCQUFnQjtpQkFDakIsQ0FBQztZQUNKLENBQUMsQ0FBQyxDQUFDO1lBRUgsZ0VBQWdFO1lBQ2hFLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsc0JBQXNCO1lBRWhGLE1BQU0sa0NBQW1DLFNBQVEsYUFBYTtnQkFDNUQsWUFBbUIsS0FBaUI7b0JBQ2xDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDZixDQUFDO2FBQ0Y7WUFFRCxRQUFRO1lBQ1IsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDO1lBQzVCLE1BQU0sTUFBTSxHQUFXO2dCQUNyQixXQUFXLEVBQUUsa0JBQWtCO2dCQUMvQixLQUFLLEVBQUUsWUFBWTtnQkFDbkIsU0FBUyxFQUFFO29CQUNULENBQUMsRUFBRSxHQUFHO29CQUNOLENBQUMsRUFBRSxHQUFHO2lCQUNQO2FBQ0YsQ0FBQztZQUNGLE1BQU0sS0FBSyxHQUFHLElBQUksa0NBQWtDLENBQUM7Z0JBQ25ELElBQUksRUFBRSxVQUFVO2dCQUNoQixRQUFRLEVBQUU7b0JBQ1IsT0FBTyxFQUFFLFVBQVU7b0JBQ25CLE1BQU0sRUFBRSxDQUFDO29CQUNULE9BQU8sRUFBRTt3QkFDUCxDQUFDLFVBQVUsQ0FBQyxFQUFFLE1BQU07cUJBQ3JCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLEtBQUssQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxVQUFVLENBQUMsQ0FBQztZQUVsRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLCtCQUErQixDQUFDLENBQUMsb0JBQW9CLENBQzFELEtBQUssRUFDTCxPQUFPLEVBQ1A7Z0JBQ0UsR0FBRyxNQUFNO2dCQUNULFNBQVMsRUFBRSxVQUFVO2FBQ3RCLENBQ0YsQ0FBQztZQUNGLE1BQU0sQ0FBQywrQkFBK0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQztnQkFDekUsTUFBTSxFQUFFLE1BQU0sQ0FBQyxRQUFRLEVBQUU7YUFDMUIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO1lBQ3ZELFFBQVE7WUFDUixNQUFNLFFBQVEsR0FBYTtnQkFDekIsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsT0FBTyxFQUFFLFVBQVU7Z0JBQ25CLE9BQU8sRUFBRSxFQUFFO2FBQ1osQ0FBQztZQUNGLE1BQU0saUJBQWlCLEdBQUcsNEJBQTRCLENBQUM7WUFDdkQsTUFBTSxLQUFLLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQztnQkFDMUMsUUFBUTtnQkFDUixJQUFJLEVBQUUsVUFBVTthQUNqQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsU0FBUyxJQUFJO2dCQUNYLEtBQUssQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBQzNELENBQUM7WUFFRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1FBQ3BFLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRTtRQUM3QixJQUFJLENBQUMsdUNBQXVDLEVBQUUsR0FBRyxFQUFFOztZQUNqRCxRQUFRO1lBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztZQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDdEMsTUFBTSxRQUFRLEdBQWE7Z0JBQ3pCLE1BQU0sRUFBRSxDQUFDO2dCQUNULE9BQU8sRUFBRSxVQUFVO2dCQUNuQixPQUFPLEVBQUUsRUFBRTthQUNaLENBQUM7WUFDRixNQUFNLEtBQUssR0FBRyxJQUFJLHlCQUF5QixDQUFDO2dCQUMxQyxRQUFRO2dCQUNSLElBQUksRUFBRSxVQUFVO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUNuRCxNQUFNLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO1lBRWhFLE9BQU87WUFDUCxNQUFNLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUN6QyxNQUFNLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN4QyxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUUxQyxNQUFNLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQzlDLE1BQU0sT0FBQyxPQUFPLENBQUMsZUFBZSwwQ0FBRSxZQUFZLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFFekQsTUFBTSxDQUFDLHNCQUFzQixDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDN0MsTUFBTSxDQUFDLHNCQUFzQixDQUFDLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ3JELENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5cbmltcG9ydCB7XG4gIEFwcCxcbiAgU3RhY2ssXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQge1xuICBNYW5pZmVzdCxcbiAgUmVjaXBlLFxuICBTdGFnZSxcbiAgU3RhZ2VQcm9wcyxcbn0gZnJvbSAnLi4vbGliJztcblxuZGVzY3JpYmUoJ1N0YWdlJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBTVEFHRV9QQVRIID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJ2Fzc2V0cycpO1xuXG4gIC8qKlxuICAgKiBBIHN1Yi1jbGFzcyBvZiBTdGFnZSB0aGF0IG1ha2VzIHRoZSBjb25zdHJ1Y3RvciBwdWJsaWMgZm9yIHRlc3RpbmcgcHVycG9zZXMuXG4gICAqXG4gICAqIFRoaXMgaXMgdG8gYmUgYWJsZSB0byBzcGVjaWZ5IGEgbWFuaWZlc3QgcmF0aGVyIHRoYW4gbG9hZCBpdCBmcm9tIHRoZSBmaWxlLXN5c3RlbSB2aWEge0BsaW5rIFN0YWdlLmZyb21EaXJlY3Rvcnl9LlxuICAgKi9cbiAgY2xhc3MgU3RhZ2VXaXRoUHVsaWNDb25zdHJ1Y3RvciBleHRlbmRzIFN0YWdlIHtcbiAgICBjb25zdHJ1Y3Rvcihwcm9wczogU3RhZ2VQcm9wcykge1xuICAgICAgc3VwZXIocHJvcHMpO1xuICAgIH1cbiAgfVxuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIGplc3QucmVzZXRNb2R1bGVzKCk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCcuZnJvbURpcmVjdG9yeScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IG1hbmlmZXN0OiBNYW5pZmVzdCA9IHtcbiAgICAgIHNjaGVtYTogMSxcbiAgICAgIHZlcnNpb246ICcxMC4xLjkuMicsXG4gICAgICByZWNpcGVzOiB7fSxcbiAgICB9O1xuXG4gICAgY29uc3QgcmVhZEZpbGVTeW5jID0gamVzdC5zcHlPbihmcywgJ3JlYWRGaWxlU3luYycpO1xuICAgIGxldCBzdGFnZTogU3RhZ2U7XG5cbiAgICBiZWZvcmVBbGwoKCkgPT4ge1xuICAgICAgcmVhZEZpbGVTeW5jLm1vY2tSZXR1cm5WYWx1ZShKU09OLnN0cmluZ2lmeShtYW5pZmVzdCkpO1xuICAgICAgc3RhZ2UgPSByZXF1aXJlKCcuLi9saWInKS5TdGFnZS5mcm9tRGlyZWN0b3J5KFNUQUdFX1BBVEgpOyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lXG4gICAgfSk7XG5cbiAgICBhZnRlckFsbCgoKSA9PiB7XG4gICAgICByZWFkRmlsZVN5bmMubW9ja1Jlc3RvcmUoKTtcbiAgICAgIGplc3QucmVzZXRNb2R1bGVzKCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdoYXMgZGlyUGF0aCcsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFnZS5kaXJQYXRoKS50b0JlKFNUQUdFX1BBVEgpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnbG9hZHMgbWFuaWZlc3QuanNvbiBmcm9tIGRpcmVjdG9yeScsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIGNvbnN0IGV4cGVjdGVkUGF0aCA9IHBhdGguam9pbihTVEFHRV9QQVRILCAnbWFuaWZlc3QuanNvbicpO1xuXG4gICAgICBleHBlY3QocmVhZEZpbGVTeW5jKS50b0hhdmVCZWVuQ2FsbGVkV2l0aChcbiAgICAgICAgZXhwZWN0ZWRQYXRoLFxuICAgICAgICBleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgICAgICAgZW5jb2Rpbmc6ICd1dGY4JyxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgZXhwZWN0KHN0YWdlLm1hbmlmZXN0KS50b0VxdWFsKG1hbmlmZXN0KTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJy5mcm9tRGlyZWN0b3J5JywgKCkgPT4ge1xuICAgIHRlc3QoJ25vdCBzdXBwb3J0ZWQgdmVyc2lvbiBmYWlsdXJlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1hbmlmZXN0OiBNYW5pZmVzdCA9IHtcbiAgICAgICAgc2NoZW1hOiAxLFxuICAgICAgICB2ZXJzaW9uOiAnMTAuMS44LjAnLFxuICAgICAgICByZWNpcGVzOiB7fSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHJlYWRGaWxlU3luYyA9IGplc3Quc3B5T24oZnMsICdyZWFkRmlsZVN5bmMnKTtcbiAgICAgIHJlYWRGaWxlU3luYy5tb2NrUmV0dXJuVmFsdWUoSlNPTi5zdHJpbmdpZnkobWFuaWZlc3QpKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KCgpID0+IHtcbiAgICAgICAgcmVxdWlyZSgnLi4vbGliJykuU3RhZ2UuZnJvbURpcmVjdG9yeShTVEFHRV9QQVRIKSAvLyBlc2xpbnQtZGlzYWJsZS1saW5lXG4gICAgICB9KS50b1Rocm93KCdTdGFnZWQgRGVhZGxpbmUgVmVyc2lvbiAoMTAuMS44LjApIGlzIGxlc3MgdGhhbiB0aGUgbWluaW11bSBzdXBwb3J0ZWQgdmVyc2lvbiAoMTAuMS45LjIpJyk7XG5cbiAgICAgIHJlYWRGaWxlU3luYy5tb2NrUmVzdG9yZSgpO1xuICAgICAgamVzdC5yZXNldE1vZHVsZXMoKTtcbiAgICB9KTtcblxuICB9KTtcblxuICB0ZXN0KCdoYXMgbWFuaWZlc3QnLCAoKSA9PiB7XG4gICAgY29uc3QgbWFuaWZlc3Q6IE1hbmlmZXN0ID0ge1xuICAgICAgc2NoZW1hOiAxLFxuICAgICAgdmVyc2lvbjogJzEwLjEuOS4yJyxcbiAgICAgIHJlY2lwZXM6IHtcbiAgICAgICAgYToge1xuICAgICAgICAgIHRpdGxlOiAnYS10aXRsZScsXG4gICAgICAgICAgZGVzY3JpcHRpb246ICdhLWRlc2NyaXB0aW9uJyxcbiAgICAgICAgICBidWlsZEFyZ3M6IHtcbiAgICAgICAgICAgIGFyZ29uZTogJ2EtYXJnb25lLXZhbHVlJyxcbiAgICAgICAgICAgIGFyZ3R3bzogJ2EtYXJndHdvLXZhbHVlJyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBiOiB7XG4gICAgICAgICAgdGl0bGU6ICdiLXRpdGxlJyxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogJ2ItZGVzY3JpcHRpb24nLFxuICAgICAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICAgICAgYXJnb25lOiAnYi1hcmdvbmUtdmFsdWUnLFxuICAgICAgICAgICAgYXJndHdvOiAnYi1hcmd0d28tdmFsdWUnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH07XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2VXaXRoUHVsaWNDb25zdHJ1Y3Rvcih7XG4gICAgICBtYW5pZmVzdCxcbiAgICAgIHBhdGg6IFNUQUdFX1BBVEgsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWdlLm1hbmlmZXN0KS50b0VxdWFsKG1hbmlmZXN0KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ21hbmlmZXN0IHZhbGlkYXRpb24nLCAoKSA9PiB7XG4gICAgdGVzdC5lYWNoPFtzdHJpbmcsIHsgbWFuaWZlc3Q6IGFueSwgZXhwZWN0ZWRFcnJvcj86IHN0cmluZyB8IFJlZ0V4cCB8IHVuZGVmaW5lZCB9XT4oW1xuICAgICAgW1xuICAgICAgICAnbWFpbmZlc3Qgd3JvbmcgdHlwZScsXG4gICAgICAgIHtcbiAgICAgICAgICBtYW5pZmVzdDogMSxcbiAgICAgICAgICBleHBlY3RlZEVycm9yOiAvRXhwZWN0ZWQgb2JqZWN0IGJ1dCBnb3QvLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIFtcbiAgICAgICAgJ21hbmlmZXN0IHZlcnNpb24gbm90IHN0cmluZycsXG4gICAgICAgIHtcbiAgICAgICAgICBtYW5pZmVzdDoge1xuICAgICAgICAgICAgdmVyc2lvbjogMSxcbiAgICAgICAgICAgIHJlY2lwZXM6IHt9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgW1xuICAgICAgICAnbWlzc2luZyBzY2hlbWEnLFxuICAgICAgICB7XG4gICAgICAgICAgbWFuaWZlc3Q6IHtcbiAgICAgICAgICAgIHZlcnNpb246ICcxMC4xLjkuMicsXG4gICAgICAgICAgICByZWNpcGVzOiB7fSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGV4cGVjdGVkRXJyb3I6IC9NYW5pZmVzdCBjb250YWlucyBubyBcInNjaGVtYVwiIGtleS8sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgW1xuICAgICAgICAnd3Jvbmcgc2NoZW1hIHR5cGUnLCB7XG4gICAgICAgICAgbWFuaWZlc3Q6IHtcbiAgICAgICAgICAgIHNjaGVtYTogJ2EnLFxuICAgICAgICAgICAgdmVyc2lvbjogJzEwLjEuOS4yJyxcbiAgICAgICAgICAgIHJlY2lwZXM6IHt9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgZXhwZWN0ZWRFcnJvcjogL0V4cGVjdGVkIGEgbnVtZXJpYyBcInNjaGVtYVwiIGJ1dCBnb3Q6LyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBbXG4gICAgICAgICdtaXNzaW5nIHZlcnNpb24nLCB7XG4gICAgICAgICAgbWFuaWZlc3Q6IHtcbiAgICAgICAgICAgIHNjaGVtYTogMSxcbiAgICAgICAgICAgIHJlY2lwZXM6IHt9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgZXhwZWN0ZWRFcnJvcjogL01hbmlmZXN0IGNvbnRhaW5zIG5vIFwidmVyc2lvblwiIGtleS8sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgW1xuICAgICAgICAnaW5jb3JyZWN0IHZlcnNpb24gZm9ybWF0Jywge1xuICAgICAgICAgIG1hbmlmZXN0OiB7XG4gICAgICAgICAgICBzY2hlbWE6IDEsXG4gICAgICAgICAgICB2ZXJzaW9uOiAnMTAuMS4nLFxuICAgICAgICAgICAgcmVjaXBlczoge30sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBleHBlY3RlZEVycm9yOiAvSW52YWxpZCB2ZXJzaW9uIGZvcm1hdC8sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgW1xuICAgICAgICAndmVyc2lvbiBub3Qgc3VwcG9ydGVkJywge1xuICAgICAgICAgIG1hbmlmZXN0OiB7XG4gICAgICAgICAgICBzY2hlbWE6IDEsXG4gICAgICAgICAgICB2ZXJzaW9uOiAnMTAuMS4wLjAnLFxuICAgICAgICAgICAgcmVjaXBlczoge30sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBleHBlY3RlZEVycm9yOiAnU3RhZ2VkIERlYWRsaW5lIFZlcnNpb24gKDEwLjEuMC4wKSBpcyBsZXNzIHRoYW4gdGhlIG1pbmltdW0gc3VwcG9ydGVkIHZlcnNpb24gKDEwLjEuOS4yKScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIF0pKCclcycsIChfbmFtZSwgdGVzdGNhc2UpID0+IHtcbiAgICAgIGNvbnN0IHsgbWFuaWZlc3QsIGV4cGVjdGVkRXJyb3IgfSA9IHRlc3RjYXNlO1xuICAgICAgLy8gV0hFTlxuICAgICAgZnVuY3Rpb24gd2hlbigpIHtcbiAgICAgICAgbmV3IFN0YWdlV2l0aFB1bGljQ29uc3RydWN0b3Ioe1xuICAgICAgICAgIHBhdGg6IFNUQUdFX1BBVEgsXG4gICAgICAgICAgbWFuaWZlc3QsXG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICBleHBlY3Qod2hlbikudG9UaHJvdyhleHBlY3RlZEVycm9yKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ3ZhbGlkIHJlY2lwZXMnLCAoKSA9PiB7XG4gICAgdGVzdC5lYWNoKFtcbiAgICAgIFtcbiAgICAgICAgJ2J1aWxkIGFyZ3MgYW5kIG5vIHRhcmdldCcsXG4gICAgICAgIHtcbiAgICAgICAgICB0aXRsZTogJ0V1Z2VuZVxcJ3MgRmF2b3JpdGUgQ29va2llcycsXG4gICAgICAgICAgZGVzY3JpcHRpb246ICdDIGlzIGZvciBjb29raWUuIFRoYXRcXCdzIGdvb2QgZW5vdWdoIGZvciBtZScsXG4gICAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgICBmbG91cjogJzMgY3VwcycsXG4gICAgICAgICAgICBidXR0ZXI6ICcxIGN1cCcsXG4gICAgICAgICAgICBzdWdhcjogJzEvMiBjdXAnLFxuICAgICAgICAgICAgZWdnOiAnMicsXG4gICAgICAgICAgICBiYWtpbmdwb3dkZXI6ICcxIHRzcCcsXG4gICAgICAgICAgICBiYWtpbmdzb2RhOiAnMSB0c2InLFxuICAgICAgICAgICAgdmFuaWxsYTogJzEgdHNwJyxcbiAgICAgICAgICAgIHNhbHQ6ICcxIHBpbmNoJyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIFtcbiAgICAgICAgJ2J1aWxkQXJncyBhbmQgYSB0YXJnZXQnLFxuICAgICAgICB7XG4gICAgICAgICAgdGl0bGU6ICdKb3NoXFwncyBTXFwnbW9yZXMnLFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnYSBzZWNyZXQgZmFtaWx5IHJlY2lwZScsXG4gICAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgICBncmFoYW1jcmFja2VyOiAnMicsXG4gICAgICAgICAgICBtYXJzaG1lbGxvdzogJzEnLFxuICAgICAgICAgICAgY2hvY29sYXRlOiAnMicsXG4gICAgICAgICAgICBjYW1wZmlyZTogJzEnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdGFyZ2V0OiAnc2ltcGxpY2l0eScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgW1xuICAgICAgICAndGFyZ2V0IGFuZCBubyBidWlsZEFyZ3MnLFxuICAgICAgICB7XG4gICAgICAgICAgdGl0bGU6ICdKZXJpY2hvXFxzIHNwZWNpYWwgYnJvd25pZSBiYXRjaCB3aXRoIGEgc3RyYXdiZXJyeSBtaWxrc2hha2UnLFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnYSBzbGlnaHQgaW1wcm92ZW1lbnQgdG8gdGhlIG9yaWdpbmFsIHJlY2lwZScsXG4gICAgICAgICAgdGFyZ2V0OiAndGFyZ2V0JyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBbXG4gICAgICAgICdubyB0YXJnZXQgb3IgYnVpbGRBcmdzJyxcbiAgICAgICAge1xuICAgICAgICAgIHRpdGxlOiAnWWFzaFxcJ3MgVGVxdWlsYSBMaW1lIFNob3QgKFRMUyknLFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnQSBzdXJlLWZpcmUgd2F5IHRvIGJyaWdodGVuIHlvdXIgZGF5JyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgXSkoJyVzJywgKF9zY2VuYXJpbzogc3RyaW5nLCByZWNpcGU6IFJlY2lwZSkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1hbmlmZXN0OiBNYW5pZmVzdCA9IHtcbiAgICAgICAgcmVjaXBlczoge1xuICAgICAgICAgIHJlY2lwZU5hbWU6IHJlY2lwZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2NoZW1hOiAxLFxuICAgICAgICB2ZXJzaW9uOiAnMTAuMS45LjInLFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2VXaXRoUHVsaWNDb25zdHJ1Y3Rvcih7XG4gICAgICAgIG1hbmlmZXN0LFxuICAgICAgICBwYXRoOiBTVEFHRV9QQVRILFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFnZS5tYW5pZmVzdC5yZWNpcGVzLnJlY2lwZU5hbWUpLnRvRXF1YWwocmVjaXBlKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJy5pbWFnZUZyb21SZWNpcGUoKScsICgpID0+IHtcbiAgICBsZXQgYXBwOiBBcHA7XG4gICAgbGV0IHN0YWNrOiBTdGFjaztcblxuICAgIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGFwcCA9IG5ldyBBcHAoKTtcbiAgICAgIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjcmVhdGVzIERvY2tlckltYWdlQXNzZXRzIGZyb20gZXhpc3RpbmcgcmVjaXBlcycsICgpID0+IHtcbiAgICAgIGplc3QucmVzZXRNb2R1bGVzKCk7XG4gICAgICBjb25zdCBtb2NrRG9ja2VySW1hZ2VBc3NldENvbnN0cnVjdG9yID0gamVzdC5mbigpO1xuICAgICAgamVzdC5tb2NrKCdAYXdzLWNkay9hd3MtZWNyLWFzc2V0cycsICgpID0+IHtcblxuICAgICAgICBjbGFzcyBEb2NrZXJJbWFnZUFzc2V0IHtcbiAgICAgICAgICBjb25zdHJ1Y3RvciguLi5hcmdzOiBbYW55XSkge1xuICAgICAgICAgICAgbW9ja0RvY2tlckltYWdlQXNzZXRDb25zdHJ1Y3RvciguLi5hcmdzKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIERvY2tlckltYWdlQXNzZXQsXG4gICAgICAgIH07XG4gICAgICB9KTtcblxuICAgICAgLy8gSW1wb3J0IHRoZSBjbGFzcyB1bmRlciB0ZXN0IG5vdyB0aGF0IHRoZSBlYzIgbW9jayBpcyBpbiBwbGFjZVxuICAgICAgY29uc3QgUmVsb2FkZWRTdGFnZSA9IGplc3QucmVxdWlyZUFjdHVhbCgnLi4vbGliJykuU3RhZ2U7IC8vIGVzbGludC1kaXNhYmxlLWxpbmVcblxuICAgICAgY2xhc3MgUmVsb2FkZWRTdGFnZVdpdGhQdWJsaWNDb25zdHJ1Y3RvciBleHRlbmRzIFJlbG9hZGVkU3RhZ2Uge1xuICAgICAgICBwdWJsaWMgY29uc3RydWN0b3IocHJvcHM6IFN0YWdlUHJvcHMpIHtcbiAgICAgICAgICBzdXBlcihwcm9wcyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHJlY2lwZU5hbWUgPSAnbXluYW1lJztcbiAgICAgIGNvbnN0IHJlY2lwZTogUmVjaXBlID0ge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ3NvbWUgZGVzY3JpcHRpb24nLFxuICAgICAgICB0aXRsZTogJ3NvbWUgdGl0bGUnLFxuICAgICAgICBidWlsZEFyZ3M6IHtcbiAgICAgICAgICBhOiAnYScsXG4gICAgICAgICAgYjogJ2InLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICAgIGNvbnN0IHN0YWdlID0gbmV3IFJlbG9hZGVkU3RhZ2VXaXRoUHVibGljQ29uc3RydWN0b3Ioe1xuICAgICAgICBwYXRoOiBTVEFHRV9QQVRILFxuICAgICAgICBtYW5pZmVzdDoge1xuICAgICAgICAgIHZlcnNpb246ICcxMC4xLjkuMicsXG4gICAgICAgICAgc2NoZW1hOiAxLFxuICAgICAgICAgIHJlY2lwZXM6IHtcbiAgICAgICAgICAgIFtyZWNpcGVOYW1lXTogcmVjaXBlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgc3RhZ2UuaW1hZ2VGcm9tUmVjaXBlKHN0YWNrLCAnSW1hZ2UnLCByZWNpcGVOYW1lKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KG1vY2tEb2NrZXJJbWFnZUFzc2V0Q29uc3RydWN0b3IpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKFxuICAgICAgICBzdGFjayxcbiAgICAgICAgJ0ltYWdlJyxcbiAgICAgICAge1xuICAgICAgICAgIC4uLnJlY2lwZSxcbiAgICAgICAgICBkaXJlY3Rvcnk6IFNUQUdFX1BBVEgsXG4gICAgICAgIH0sXG4gICAgICApO1xuICAgICAgZXhwZWN0KG1vY2tEb2NrZXJJbWFnZUFzc2V0Q29uc3RydWN0b3IubW9jay5jYWxsc1swXVsyXSkubm90LnRvTWF0Y2hPYmplY3Qoe1xuICAgICAgICB0YXJnZXQ6IGV4cGVjdC5hbnl0aGluZygpLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCd0aHJvd3MgZXhjZXB0aW9uIHdoZW4gcmVjaXBlIGRvZXMgbm90IGV4aXN0JywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1hbmlmZXN0OiBNYW5pZmVzdCA9IHtcbiAgICAgICAgc2NoZW1hOiAxLFxuICAgICAgICB2ZXJzaW9uOiAnMTAuMS45LjInLFxuICAgICAgICByZWNpcGVzOiB7fSxcbiAgICAgIH07XG4gICAgICBjb25zdCBpbnZhbGlkUmVjaXBlTmFtZSA9ICd0aGlzLXJlY2lwZS1kb2VzLW5vdC1leGlzdCc7XG4gICAgICBjb25zdCBzdGFnZSA9IG5ldyBTdGFnZVdpdGhQdWxpY0NvbnN0cnVjdG9yKHtcbiAgICAgICAgbWFuaWZlc3QsXG4gICAgICAgIHBhdGg6IFNUQUdFX1BBVEgsXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgZnVuY3Rpb24gd2hlbigpIHtcbiAgICAgICAgc3RhZ2UuaW1hZ2VGcm9tUmVjaXBlKHN0YWNrLCAnSW1hZ2UnLCBpbnZhbGlkUmVjaXBlTmFtZSk7XG4gICAgICB9XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdCh3aGVuKS50b1Rocm93RXJyb3IoYE5vIHN1Y2ggcmVjaXBlOiAke2ludmFsaWRSZWNpcGVOYW1lfWApO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnLmdldFZlcnNpb24oKScsICgpID0+IHtcbiAgICB0ZXN0KCdyZXR1cm5zIGEgdmVyc2lvbiBpbiB0aGUgc3VjY2VzcyBjYXNlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG4gICAgICBjb25zdCBtYW5pZmVzdDogTWFuaWZlc3QgPSB7XG4gICAgICAgIHNjaGVtYTogMSxcbiAgICAgICAgdmVyc2lvbjogJzEwLjEuOS4yJyxcbiAgICAgICAgcmVjaXBlczoge30sXG4gICAgICB9O1xuICAgICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2VXaXRoUHVsaWNDb25zdHJ1Y3Rvcih7XG4gICAgICAgIG1hbmlmZXN0LFxuICAgICAgICBwYXRoOiBTVEFHRV9QQVRILFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHZlcnNpb24gPSBzdGFnZS5nZXRWZXJzaW9uKHN0YWNrLCAnVmVyc2lvbicpO1xuICAgICAgY29uc3QgbGludXhGdWxsVmVyc2lvblN0cmluZyA9IHZlcnNpb24ubGludXhGdWxsVmVyc2lvblN0cmluZygpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QodmVyc2lvbi5tYWpvclZlcnNpb24pLnRvRXF1YWwoMTApO1xuICAgICAgZXhwZWN0KHZlcnNpb24ubWlub3JWZXJzaW9uKS50b0VxdWFsKDEpO1xuICAgICAgZXhwZWN0KHZlcnNpb24ucmVsZWFzZVZlcnNpb24pLnRvRXF1YWwoOSk7XG5cbiAgICAgIGV4cGVjdCh2ZXJzaW9uLmxpbnV4SW5zdGFsbGVycykudG9CZURlZmluZWQoKTtcbiAgICAgIGV4cGVjdCh2ZXJzaW9uLmxpbnV4SW5zdGFsbGVycz8ucGF0Y2hWZXJzaW9uKS50b0VxdWFsKDIpO1xuXG4gICAgICBleHBlY3QobGludXhGdWxsVmVyc2lvblN0cmluZykudG9CZURlZmluZWQoKTtcbiAgICAgIGV4cGVjdChsaW51eEZ1bGxWZXJzaW9uU3RyaW5nKS50b0VxdWFsKCcxMC4xLjkuMicpO1xuICAgIH0pO1xuICB9KTtcbn0pO1xuIl19