"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = exports.Version = void 0;
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * Will return undefined if the linuxInstallers property is undefined.
     *
     * @inheritdoc true
     */
    linuxFullVersionString() {
        if (!this.linuxInstallers) {
            return undefined;
        }
        return this.fullVersionString(this.linuxInstallers.patchVersion);
    }
    /**
     *
     */
    fullVersionString(patchVersion) {
        return `${this.releaseVersionString}.${patchVersion}`;
    }
    /**
     * Helper to concatenate the major, minor, and release version values into a release version string.
     */
    get releaseVersionString() {
        return `${this.majorVersion}.${this.minorVersion}.${this.releaseVersion}`;
    }
}
/**
 *
 */
VersionQueryBase.INSTALLER_BUCKET = 'thinkbox-installers';
/**
 * This class is reposonsible to do basic operations on version format.
 */
class Version {
    /**
     *
     */
    constructor(components) {
        // validations
        if (components.length != 4) {
            throw new Error('Invalid version format. Version should contain exactly 4 components.');
        }
        components.forEach((component) => {
            if (component < 0) {
                throw new RangeError('Invalid version format. None of the version components can be negative.');
            }
        });
        this.components = components;
    }
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse.
     */
    static parse(version) {
        if (!Version.validateVersionFormat(version)) {
            throw new TypeError(`Invalid version format. Expected format 'a.b.c.d', found '${version}'`);
        }
        return new Version(version.split('.').map(x => parseInt(x)));
    }
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    static validateVersionFormat(version) {
        /**
         * Regex: ^\d+(?:\.\d+){3}$
         * Matches a sequence of '.' separated numbers with exactly 4 digits.
         * - ^ asserts position at start of a line.
         * - \d+ Matches one or more digits.
         * - (?:\.\d+) Matches a dot and the following one or more digits.
         * - {3} Matches previous pattern exactly 3 times.
         * - $ asserts position at the end of a line
         */
        if (version.match(/^\d+(?:\.\d+){3}$/)) {
            return true;
        }
        return false;
    }
    /**
     * The major version number.
     *
     * @inheritdoc true
     */
    get majorVersion() {
        return this.components[0];
    }
    /**
     * The minor version number.
     *
     * @inheritdoc true
     */
    get minorVersion() {
        return this.components[1];
    }
    /**
     * The release version number.
     *
     * @inheritdoc true
     */
    get releaseVersion() {
        return this.components[2];
    }
    /**
     * The patch version number.
     *
     * @inheritdoc true
     */
    get patchVersion() {
        return this.components[3];
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     */
    isGreaterThan(version) {
        return this.compare(version) > 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     */
    isLessThan(version) {
        return this.compare(version) < 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     */
    isEqual(version) {
        return this.compare(version) == 0;
    }
    /**
     * The method returns the version components in dot separated string format.
     */
    toString() {
        return this.components.join('.');
    }
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    compare(version) {
        if (this.components.length != version.components.length) {
            throw new TypeError('Component count in both the versions should be same.');
        }
        for (let i = 0; i < version.components.length; i++) {
            const diff = this.components[i] - version.components[i];
            if (diff != 0) {
                return diff;
            }
        }
        return 0;
    }
}
exports.Version = Version;
/**
 * This variable holds the value for minimum supported deadline version.
 */
Version.MINIMUM_SUPPORTED_DEADLINE_VERSION = new Version([10, 1, 9, 2]);
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        throw new Error(`MethodNotSupportedException: This method is currently not implemented. Input: ${JSON.stringify(props)}`);
    }
    /**
     * Specify a Deadline version from a fully-qualified Deadline patch version.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @param versionComponents The individual components of the Deadline release version.
     * @remark Thinkbox reserves the right to revoke patch versions of Deadline and suceeed them with a new patch version.
     * For this reason, using this method may fail if Thinkbox revokes the specific patch version of the Deadline
     * installer in the event of a critical issue such as a security vulnerability.
     *
     * Use at your own risk.
     */
    static exact(scope, id, versionComponents) {
        class ExactVersion extends VersionQueryBase {
            constructor() {
                super(scope, id);
                const installerBucket = aws_s3_1.Bucket.fromBucketName(this, 'ThinkboxInstallers', ExactVersion.INSTALLER_BUCKET);
                const { majorVersion, minorVersion, releaseVersion, patchVersion } = versionComponents;
                this.majorVersion = majorVersion;
                this.minorVersion = minorVersion;
                this.releaseVersion = releaseVersion;
                const fullVersionString = this.fullVersionString(patchVersion);
                const objectKey = `Deadline/${fullVersionString}/Linux/DeadlineRepository-${fullVersionString}-linux-x64-installer.run`;
                this.linuxInstallers = {
                    patchVersion,
                    repository: {
                        s3Bucket: installerBucket,
                        objectKey,
                    },
                };
            }
        }
        return new ExactVersion();
    }
    /**
     * Specify Version from a fully-qualified Deadline release version string.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @param versionString A fully qualified version string (e.g. 10.1.9.2).
     */
    static exactString(scope, id, versionString) {
        return VersionQuery.exact(scope, id, Version.parse(versionString));
    }
}
exports.VersionQuery = VersionQuery;
//# sourceMappingURL=data:application/json;base64,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