"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 *
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 */
class UsageBasedLicense {
    /**
     *
     */
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
/**
 * Constant used to signify unlimited overage.
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL) (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html ) in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, allows no ingress;
 * when a Deadline Worker requires access to licensing, then the RFDK constructs will grant that worker’s security group
 * ingress on TCP port 17004 as well as other ports as required by the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *    License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *    Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *    Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 */
class UsageBasedLicensing extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets: (_a = props.vpcSubnets) !== null && _a !== void 0 ? _a : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: (_b = props.desiredCount) !== null && _b !== void 0 ? _b : 1,
            maxCapacity: (_c = props.desiredCount) !== null && _c !== void 0 ? _c : 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = (_e = (_d = props.logGroupProps) === null || _d === void 0 ? void 0 : _d.logGroupPrefix) !== null && _e !== void 0 ? _e : UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: props.desiredCount,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        this.connections.allowToDefaultPort(props.renderQueue);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required.
     *
     * @param workerFleet - worker fleet.
     * @param licenses - UBL licenses.
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                this.connections.allowFrom(workerFleet, port);
            });
        });
        this.connections.allowFrom(workerFleet, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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