"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const version_1 = require("./version");
/**
 * Class for interacting with the Deadline stage directory.
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 */
class Stage {
    /**
     * Constructs a Stage instance.
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path.
     *
     * @param manifestPath The path to the manifest JSON file.
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory.
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        // Do minimum supported deadline version check
        const stagedVersion = version_1.Version.parse(version);
        if (stagedVersion.isLessThan(version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            throw new TypeError(`Staged Deadline Version (${version}) is less than the minimum supported version (${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()})`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     */
    getVersion(scope, id) {
        return version_1.VersionQuery.exactString(scope, id, this.manifest.version);
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage.
     *
     * @param scope The scope for the {@link DockerImageAsset}.
     * @param id The construct ID of the {@link DockerImageAsset}.
     * @param recipeName The name of the recipe.
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
}
exports.Stage = Stage;
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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