"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 */
class Repository extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        super(scope, id);
        if (props.database && ((_a = props.backupOptions) === null || _a === void 0 ? void 0 : _a.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_b = props.removalPolicy) === null || _b === void 0 ? void 0 : _b.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem and Database components of the repository
        this.fileSystem = (_d = props.fileSystem) !== null && _d !== void 0 ? _d : new core_2.MountableEfs(this, {
            filesystem: new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
            }),
        });
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                this.node.addWarning(`The parameter databaseAuditLogging only has an effect when the Repository is creating its own database. 
        Please ensure that the Database provided is configured correctly.`);
            }
        }
        else {
            const databaseAuditLogging = (_h = props.databaseAuditLogging) !== null && _h !== void 0 ? _h : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_j = props.documentDbInstanceCount) !== null && _j !== void 0 ? _j : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_k = props.vpcSubnets) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                },
                instances,
                backup: {
                    retention: (_m = (_l = props.backupOptions) === null || _l === void 0 ? void 0 : _l.databaseRetention) !== null && _m !== void 0 ? _m : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_p = (_o = props.removalPolicy) === null || _o === void 0 ? void 0 : _o.database) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_q = props.vpcSubnets) !== null && _q !== void 0 ? _q : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _a, _b;
        const hostMountPoint = (_a = props.containerInstances.filesystemMountPoint) !== null && _a !== void 0 ? _a : '/mnt/repo';
        const containerMountPoint = (_b = props.containers.filesystemMountPoint) !== null && _b !== void 0 ? _b : `/opt/Thinkbox/DeadlineRepository${this.version.majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: url_1.pathToFileURL(containerMountPoint).toString(),
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _a;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tags.of(this.installerGroup).add(tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        var _a;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        if (!((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.repository)) {
            throw new Error('Version given to Repository must provide a Linux Repository installer.');
        }
        const linuxVersionString = version.linuxFullVersionString();
        if (!linuxVersionString) {
            throw new Error('Version given to Repository must provide a full Linux version string.');
        }
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                linuxVersionString,
            ],
        });
    }
}
exports.Repository = Repository;
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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