"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * - An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *    bash/
     *      script-one.sh
     *      script-two.sh
     *    powershell
     *      script-one.ps1
     *      script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}.
     * @param id The construct id for the created {@link ScriptAsset}.
     * @param scriptParams The parameters that are used to compute the conventional path to the script file.
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script.
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
//# sourceMappingURL=data:application/json;base64,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