"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.convert = exports.generate = exports.X509CertificateConverter = exports.X509CertificateGenerator = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto_1 = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const kms_1 = require("../lib/kms");
const secrets_manager_1 = require("../lib/secrets-manager");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class X509Common extends custom_resource_1.DynamoBackedCustomResource {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.secretsManagerClient = secretsManagerClient;
    }
    async doDelete(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const resources = await resourceTable.query(physicalId);
        for (const [key, resource] of Object.entries(resources)) {
            console.log(`Deleting resource for '${key}'`);
            const arn = resource.ARN;
            try {
                const secret = secrets_manager_1.Secret.fromArn(arn, this.secretsManagerClient);
                await secret.delete();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We legit do not have the required permission to delete the secret (very unlikely)
                //  b) The Secret has already been deleted (much more likely; so we continue)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Secret ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async secretsPermissionsCheck(tags) {
        if (!this.debugMode) {
            return;
        }
        const secretName = crypto_1.randomBytes(16).toString('hex');
        const secret = await secrets_manager_1.Secret.create({
            name: secretName,
            client: this.secretsManagerClient,
            description: 'Permissions check',
            data: 'Test data',
            tags,
        });
        if (!secret) {
            throw new Error('Failed to create secret during permission test.');
        }
        await secret.putValue('Test data 2');
        // await secret.getValue(); // We don't give this permissions to the Lambda
        await secret.delete(true);
    }
    /**
     * Helper for creating a Secret and storing its ARN in the database.
     * CustomResources must be idempotent, and it is theoretically possible that Cfn
     * may invoke our lambda more than once for the same operation. If this happens,
     * then we may already have a Secret ARN stored in the database; if we do,
     * then we must reuse that ARN to avoid resource leakage.
     *
     * It's theoretically possible that this function may race with another invocation
     * of itself, but that is so unlikely. Handling it would complicate this function
     * substantially, so we do not implement for that case.
     * @param args
     */
    async createAndStoreSecret(args) {
        let secretArn;
        const existingItem = await args.database.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: args.purpose,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            secretArn = existingItem.ARN;
            const secret = secrets_manager_1.Secret.fromArn(secretArn, this.secretsManagerClient);
            await secret.putValue(args.data);
        }
        else {
            const secret = await secrets_manager_1.Secret.create({
                name: args.name,
                client: this.secretsManagerClient,
                description: args.description,
                data: args.data,
                tags: args.tags,
                encryptionKey: args.encryptionKey,
            });
            if (!secret || !secret.arn) {
                throw Error('Could not create Secret');
            }
            secretArn = secret.arn;
            await args.database.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: args.purpose,
                attributes: {
                    ARN: secretArn,
                },
                allow_overwrite: false,
            });
        }
        return secretArn;
    }
}
class X509CertificateGenerator extends X509Common {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient, secretsManagerClient);
    }
    validateInput(data) {
        return types_1.implementsIX509CertificateGenerate(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const subject = new x509_certs_1.DistinguishedName(resourceProperties.DistinguishedName);
        const passphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Passphrase, this.secretsManagerClient).getValue();
        let signingCert;
        if (resourceProperties.SigningCertificate) {
            const signCert = resourceProperties.SigningCertificate;
            const cert = await secrets_manager_1.Secret.fromArn(signCert.Cert, this.secretsManagerClient).getValue();
            const key = await secrets_manager_1.Secret.fromArn(signCert.Key, this.secretsManagerClient).getValue();
            const pass = await secrets_manager_1.Secret.fromArn(signCert.Passphrase, this.secretsManagerClient).getValue();
            const certChain = signCert.CertChain.length > 0
                ? await secrets_manager_1.Secret.fromArn(signCert.CertChain, this.secretsManagerClient).getValue()
                : '';
            signingCert = new x509_certs_1.Certificate(cert, key, pass, certChain);
        }
        const newCert = await x509_certs_1.Certificate.fromGenerated(subject, passphrase, signingCert);
        const now = new Date(Date.now());
        // timeSuffix = "<year>-<month>-<day>-<time since epoch>" -- to disambiguate secrets
        // in case we do an update on the same day as a create (both old & new exist at the same time)
        const timeSuffix = `${now.getFullYear()}-${now.getMonth()}-${now.getDate()}-${now.getTime()}`;
        const kmsKey = resourceProperties.Secret.EncryptionKey ? kms_1.Key.fromArn(resourceProperties.Secret.EncryptionKey) : undefined;
        const returnArns = {};
        const certComponents = [
            {
                key: 'Cert',
                purpose: 'Certificate',
                data: newCert.cert,
            },
            {
                key: 'Key',
                purpose: 'Private Key',
                data: newCert.key,
            },
            {
                key: 'CertChain',
                purpose: 'Certificate Chain',
                data: newCert.certChain,
            },
        ];
        for (const component of certComponents) {
            if (component.data) {
                const data = component.data;
                const purpose = component.purpose;
                const name = secrets_manager_1.sanitizeSecretName(`${resourceProperties.Secret.NamePrefix}-X.509-${purpose}-${timeSuffix}`);
                const arn = await this.createAndStoreSecret({
                    database: resourceTable,
                    name,
                    physicalId,
                    purpose,
                    data,
                    description: `X.509 ${component.purpose} for ${resourceProperties.Secret.Description}`,
                    tags: resourceProperties.Secret.Tags,
                    encryptionKey: kmsKey,
                });
                returnArns[component.key] = arn;
            }
            else {
                // Case for CertChain being empty. We cannot just skip it due to constraints put on us by CDK's CustomResource
                returnArns[component.key] = '';
            }
        }
        return returnArns;
    }
}
exports.X509CertificateGenerator = X509CertificateGenerator;
class X509CertificateConverter extends X509Common {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient, secretsManagerClient);
    }
    validateInput(data) {
        return types_1.implementsIX509CertificateEncodePkcs12(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const cert = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Cert, this.secretsManagerClient).getValue();
        const certChain = resourceProperties.Certificate.CertChain.length > 0
            ? await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.CertChain, this.secretsManagerClient).getValue()
            : '';
        const key = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Key, this.secretsManagerClient).getValue();
        const sourcePassphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Passphrase, this.secretsManagerClient).getValue();
        const sourceCert = new x509_certs_1.Certificate(cert, key, sourcePassphrase, certChain);
        const passphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Passphrase, this.secretsManagerClient).getValue();
        const newPkcs12 = await sourceCert.toPkcs12(passphrase);
        const now = new Date(Date.now());
        // timeSuffix = "<year>-<month>-<day>-<time since epoch>" -- to disambiguate secrets
        // in case we do an update on the same day as a create (both old & new exist at the same time)
        const timeSuffix = `${now.getFullYear()}-${now.getMonth()}-${now.getDate()}-${now.getTime()}`;
        const secretProps = resourceProperties.Secret;
        const kmsKey = secretProps.EncryptionKey
            ? kms_1.Key.fromArn(secretProps.EncryptionKey)
            : undefined;
        const returnArns = {};
        const pkcs12Props = {
            data: newPkcs12,
            database: resourceTable,
            description: `X.509 PKCS #12 Certificate for ${secretProps.Description}`,
            encryptionKey: kmsKey,
            name: secrets_manager_1.sanitizeSecretName(`${secretProps.NamePrefix}-X.509-CertificatePKCS12-${timeSuffix}`),
            physicalId,
            purpose: 'CertificatePKCS12',
            tags: secretProps.Tags,
        };
        returnArns.Cert = await this.createAndStoreSecret(pkcs12Props);
        return returnArns;
    }
}
exports.X509CertificateConverter = X509CertificateConverter;
/**
 * The handler used to generate an X.509 certificate and then store it in SecretsManager
 */
async function generate(event, context) {
    const handler = new X509CertificateGenerator(new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.generate = generate;
/**
 * The handler used to convert an X.509 certificate to PKCS #12 and store that in SecretsManager
 */
async function convert(event, context) {
    const handler = new X509CertificateConverter(new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.convert = convert;
//# sourceMappingURL=data:application/json;base64,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