"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const handler_1 = require("../handler");
const secretArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('readCertificateData', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const certData = 'BEGIN CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readCertificateData'](secretArn);
        // THEN
        expect(data).toStrictEqual(certData);
    });
    test('not a certificate', async () => {
        // GIVEN
        const certData = 'NOT A CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readCertificateData'](secretArn)).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
    test('binary data', async () => {
        // GIVEN
        const certData = Buffer.from('BEGIN CERTIFICATE', 'utf-8');
        const secretContents = {
            SecretBinary: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readCertificateData'](secretArn)).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
});
describe('readLoginCredentials', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const loginData = {
            username: 'testuser',
            password: 'testpassword',
        };
        const secretContents = {
            SecretString: JSON.stringify(loginData),
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readLoginCredentials'](secretArn);
        // THEN
        expect(data).toStrictEqual(loginData);
    });
    test('binary data', async () => {
        // GIVEN
        const loginData = Buffer.from('some binary data', 'utf-8');
        const secretContents = {
            SecretBinary: loginData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readLoginCredentials'](secretArn)).rejects.toThrowError(/must be a JSON encoded string/);
    });
    test.each([
        ['}{', /Failed to parse JSON in MongoDB login credentials/],
        [
            JSON.stringify({
                password: 'testpassword',
            }),
            /is missing: username/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
            }),
            /is missing: password/,
        ],
    ])('failed: %p to get %p', async (data, expected) => {
        // GIVEN
        const secretContents = {
            SecretString: data,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readLoginCredentials'](secretArn)).rejects.toThrowError(expected);
    });
});
describe('mongoLogin', () => {
    const consoleLog = console.log;
    beforeEach(() => {
        // tslint:disable-next-line: no-empty
        console.log = () => { };
    });
    afterEach(() => {
        console.log = consoleLog;
    });
    test('mongoLogin', async () => {
        // GIVEN
        async function stringSuccessRequestMock(value) {
            return value;
        }
        const mockReadCert = jest.fn((request) => stringSuccessRequestMock(request));
        const mockReadLogin = jest.fn((request) => successRequestMock(request, { username: 'test', password: 'pass' }));
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // tslint:disable-next-line: no-string-literal
        handler['readCertificateData'] = mockReadCert;
        // tslint:disable-next-line: no-string-literal
        handler['readLoginCredentials'] = mockReadLogin;
        const mockDriver = {
            MongoClient: {
                connect: jest.fn(),
            },
        };
        const loginOptions = {
            Hostname: 'testhostname',
            Port: '27017',
            Credentials: 'some credentials',
            CaCertificate: 'cert arn',
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['mongoLogin'](mockDriver, loginOptions);
        // THEN
        expect(mockReadCert.mock.calls.length).toBe(1);
        expect(mockReadCert.mock.calls[0][0]).toStrictEqual(loginOptions.CaCertificate);
        expect(mockReadLogin.mock.calls.length).toBe(1);
        expect(mockReadLogin.mock.calls[0][0]).toStrictEqual(loginOptions.Credentials);
        expect(mockDriver.MongoClient.connect.mock.calls.length).toBe(1);
        expect(mockDriver.MongoClient.connect.mock.calls[0][0]).toStrictEqual('mongodb://testhostname:27017');
        expect(mockDriver.MongoClient.connect.mock.calls[0][1]).toStrictEqual({
            tls: true,
            tlsInsecure: false,
            tlsCAFile: '/tmp/ca.crt',
            auth: {
                user: 'test',
                password: 'pass',
            },
            useUnifiedTopology: true,
        });
    });
});
describe('readPasswordAuthUserInfo', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const userData = {
            username: 'testuser',
            password: 'testpassword',
            roles: [{ role: 'readWrite', db: 'somedb' }],
        };
        const secretContents = {
            SecretString: JSON.stringify(userData),
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readPasswordAuthUserInfo'](secretArn);
        // THEN
        expect(data).toStrictEqual(userData);
    });
    test('binary data', async () => {
        // GIVEN
        const loginData = Buffer.from('Some binary data', 'utf-8');
        const secretContents = {
            SecretBinary: loginData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readPasswordAuthUserInfo'](secretArn)).rejects.toThrowError(/must be a JSON encoded string/);
    });
    test.each([
        ['}{', /Failed to parse JSON for password-auth user Secret/],
        [
            JSON.stringify({
                password: 'testpassword',
                roles: [{ role: 'readWrite', db: 'somedb' }],
            }),
            /is missing: username/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
                roles: [{ role: 'readWrite', db: 'somedb' }],
            }),
            /is missing: password/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
                password: 'testpassword',
            }),
            /is missing: roles/,
        ],
    ])('failed: %p to get %p', async (data, expected) => {
        // GIVEN
        const secretContents = {
            SecretString: data,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readPasswordAuthUserInfo'](secretArn)).rejects.toThrowError(expected);
    });
});
describe('userExists', () => {
    test('user exists', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [
                {
                    _id: 'admin.test',
                    user: 'test',
                    db: 'admin',
                },
            ],
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const exists = await handler['userExists'](mockDb, 'test');
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: 'test',
        });
        expect(exists).toStrictEqual(true);
    });
    test('user does not exists', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [],
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const exists = await handler['userExists'](mockDb, 'test');
        // THEN
        expect(exists).toStrictEqual(false);
    });
    test('request failed', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [],
            ok: 0,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['userExists'](mockDb, 'test')).rejects.toThrowError(/MongoDB error checking whether user exists 'test'/);
    });
});
describe('createUser', () => {
    const originalConsoleLog = console.log;
    let consoleLogMock;
    beforeEach(() => {
        consoleLogMock = jest.fn();
        console.log = consoleLogMock;
    });
    afterEach(() => {
        console.log = originalConsoleLog;
    });
    test('create success with password', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            password: 'password',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['createUser'](mockDb, credentials);
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            createUser: credentials.username,
            pwd: credentials.password,
            roles: credentials.roles,
        });
        expect(consoleLogMock.mock.calls.length).toBe(1);
        expect(consoleLogMock.mock.calls[0][0]).toStrictEqual('Creating user: test');
    });
    test('create success no password', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['createUser'](mockDb, credentials);
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            createUser: credentials.username,
            roles: credentials.roles,
        });
        expect(consoleLogMock.mock.calls.length).toBe(1);
        expect(consoleLogMock.mock.calls[0][0]).toStrictEqual('Creating user: test');
    });
    test('request failed', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 0,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            password: 'password',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['createUser'](mockDb, credentials)).rejects.toThrowError(/MongoDB error when adding user 'test'/);
    });
});
describe('createPasswordAuthUser', () => {
    const originalConsoleLog = console.log;
    let consoleLogMock;
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
        consoleLogMock = jest.fn();
        console.log = consoleLogMock;
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
        console.log = originalConsoleLog;
    });
    test.each([
        [
            [], true,
        ],
        [
            [
                {
                    _id: 'admin.test',
                    user: 'test',
                    db: 'admin',
                },
            ],
            false,
        ],
    ])('userExists %p gives %p', async (userExists, expected) => {
        // GIVEN
        const userData = {
            username: 'testuser',
            password: 'testpassword',
            roles: [{ role: 'readWrite', db: 'somedb' }],
        };
        const secretContents = {
            SecretString: JSON.stringify(userData),
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const userExistsResponse = {
            users: userExists,
            ok: 1,
        };
        const addUserResponse = {
            ok: 1,
        };
        async function commandMock(request) {
            if ('createUser' in request) {
                return addUserResponse;
            }
            return userExistsResponse;
        }
        const mockDb = {
            command: jest.fn((request) => commandMock(request)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const result = await handler['createPasswordAuthUser'](mockDb, secretArn);
        // THEN
        expect(result).toStrictEqual(expected);
        expect(mockDb.command.mock.calls.length).toBe(expected ? 2 : 1);
        // Check args of userExits DB query.
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: userData.username,
        });
        if (expected) {
            // Check args of createUser DB query.
            expect(mockDb.command.mock.calls[1][0]).toStrictEqual({
                createUser: userData.username,
                pwd: userData.password,
                roles: userData.roles,
            });
            expect(consoleLogMock.mock.calls.length).toBe(1);
            expect(consoleLogMock.mock.calls[0][0]).toStrictEqual(`Creating user: ${userData.username}`);
        }
    });
});
describe('createX509AuthUser', () => {
    const originalConsoleLog = console.log;
    let consoleLogMock;
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
        consoleLogMock = jest.fn();
        console.log = consoleLogMock;
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
        console.log = originalConsoleLog;
    });
    test.each([
        [
            [], true,
        ],
        [
            [
                {
                    _id: '$external.CN=myName,OU=myOrgUnit,O=myOrg',
                    user: 'CN=myName,OU=myOrgUnit,O=myOrg',
                    db: '$external',
                },
            ],
            false,
        ],
    ])('userExists %p gives %p', async (userExists, expected) => {
        // GIVEN
        const username = 'CN=TestUser,O=TestOrg,OU=TestOrgUnit';
        const userExistsResponse = {
            users: userExists,
            ok: 1,
        };
        const addUserResponse = {
            ok: 1,
        };
        async function commandMock(request) {
            if ('createUser' in request) {
                return addUserResponse;
            }
            return userExistsResponse;
        }
        const mockDb = {
            command: jest.fn((request) => commandMock(request)),
        };
        async function stringSuccessRequestMock(value) {
            return value;
        }
        async function rfc2253(_arg) {
            return username;
        }
        const mockReadCert = jest.fn((request) => stringSuccessRequestMock(request));
        const mockRfc2253 = jest.fn((arg) => rfc2253(arg));
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // tslint:disable-next-line: no-string-literal
        handler['readCertificateData'] = mockReadCert;
        // tslint:disable-next-line: no-string-literal
        handler['retrieveRfc2253Subject'] = mockRfc2253;
        const userData = {
            certificate: secretArn,
            roles: [{ role: 'readWrite', db: 'somedb' }],
        };
        const userToCreate = {
            Certificate: userData.certificate,
            Roles: JSON.stringify(userData.roles),
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const result = await handler['createX509AuthUser'](mockDb, userToCreate);
        // THEN
        expect(result).toStrictEqual(expected);
        expect(mockDb.command.mock.calls.length).toBe(expected ? 2 : 1);
        // Check args of userExits DB query.
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: username,
        });
        if (expected) {
            // Check args of createUser DB query.
            expect(mockDb.command.mock.calls[1][0]).toStrictEqual({
                createUser: username,
                roles: userData.roles,
            });
            expect(consoleLogMock.mock.calls.length).toBe(1);
            expect(consoleLogMock.mock.calls[0][0]).toStrictEqual(`Creating user: ${username}`);
        }
    });
});
describe('doCreate', () => {
    const originalConsoleLog = console.log;
    let consoleLogMock;
    let mockedHandler;
    let mockMongoClient;
    beforeEach(() => {
        consoleLogMock = jest.fn();
        console.log = consoleLogMock;
        mockMongoClient = {
            db: jest.fn(),
            close: jest.fn(),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // tslint:disable-next-line: no-string-literal
        handler['installMongoDbDriver'] = jest.fn();
        async function returnMockMongoClient(_v1, _v2) {
            return mockMongoClient;
        }
        // tslint:disable-next-line: no-string-literal
        handler['mongoLogin'] = jest.fn((a, b) => returnMockMongoClient(a, b));
        mockedHandler = handler;
    });
    afterEach(() => {
        console.log = originalConsoleLog;
    });
    test('create password auth user', async () => {
        // GIVEN
        async function returnTrue(_v1, _v2) {
            return true;
        }
        const mockCreatePwAuthUser = jest.fn((a, b) => returnTrue(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createPasswordAuthUser'] = mockCreatePwAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            PasswordAuthUsers: ['arn1', 'arn2'],
        };
        // WHEN
        const result = await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(result).toBeUndefined();
        expect(mockCreatePwAuthUser.mock.calls.length).toBe(2);
        expect(mockCreatePwAuthUser.mock.calls[0][1]).toBe('arn1');
        expect(mockCreatePwAuthUser.mock.calls[1][1]).toBe('arn2');
    });
    test('log when cannot create password auth user', async () => {
        // GIVEN
        async function returnFalse(_v1, _v2) {
            return false;
        }
        const mockCreatePwAuthUser = jest.fn((a, b) => returnFalse(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createPasswordAuthUser'] = mockCreatePwAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            PasswordAuthUsers: ['arn1'],
        };
        // WHEN
        await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/No action performed for this user./);
    });
    test('create x509 auth user', async () => {
        // GIVEN
        async function returnTrue(_v1, _v2) {
            return true;
        }
        const mockCreateAuthUser = jest.fn((a, b) => returnTrue(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createX509AuthUser'] = mockCreateAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            X509AuthUsers: [
                {
                    Certificate: 'some arn1',
                    Roles: 'json encoded role',
                },
                {
                    Certificate: 'some arn2',
                    Roles: 'json encoded role',
                },
            ],
        };
        // WHEN
        const result = await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(result).toBeUndefined();
        expect(mockCreateAuthUser.mock.calls.length).toBe(2);
        expect(mockCreateAuthUser.mock.calls[0][1]).toStrictEqual(properties.X509AuthUsers[0]);
        expect(mockCreateAuthUser.mock.calls[1][1]).toStrictEqual(properties.X509AuthUsers[1]);
    });
    test('log when cannot create x509 auth user', async () => {
        // GIVEN
        async function returnFalse(_v1, _v2) {
            return false;
        }
        const mockCreateAuthUser = jest.fn((a, b) => returnFalse(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createX509AuthUser'] = mockCreateAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            X509AuthUsers: [
                {
                    Certificate: 'some arn',
                    Roles: 'json encoded role',
                },
            ],
        };
        // WHEN
        await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/No action performed for this user./);
    });
});
//# sourceMappingURL=data:application/json;base64,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