"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
// import { fake, spy } from 'sinon';
const index_1 = require("../index");
let attachSpy;
let completeSpy;
const originalConsoleLog = console.log;
const originalConsoleError = console.error;
async function successRequestMock(request) {
    return { ...request };
}
async function errorRequestMock() {
    const error = new Error('Mock error message');
    error.code = 'MockRequestException';
    throw error;
}
beforeEach(() => {
    aws_sdk_mock_1.setSDKInstance(AWS);
    console.log = jest.fn(() => { });
    console.error = jest.fn(() => { });
});
afterEach(() => {
    aws_sdk_mock_1.restore('EC2');
    aws_sdk_mock_1.restore('AutoScaling');
    console.log = originalConsoleLog;
    console.error = originalConsoleError;
});
test('ignores test notification', async () => {
    // GIVEN
    const event = {
        Records: [
            {
                Sns: {
                    Message: JSON.stringify({
                        Event: 'autoscaling:TEST_NOTIFICATION',
                    }),
                },
            },
        ],
    };
    attachSpy = jest.fn((request) => successRequestMock(request));
    completeSpy = jest.fn((request) => successRequestMock(request));
    aws_sdk_mock_1.mock('EC2', 'attachNetworkInterface', attachSpy);
    aws_sdk_mock_1.mock('AutoScaling', 'completeLifecycleAction', completeSpy);
    // WHEN
    await index_1.handler(event);
    // THEN
    expect(attachSpy).not.toHaveBeenCalled();
    expect(completeSpy).not.toHaveBeenCalled();
});
test('processes all correct records', async () => {
    // GIVEN
    const event = {
        Records: [
            {
                Sns: {
                    Message: JSON.stringify({
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        AutoScalingGroupName: 'ASG-Name-1',
                        LifecycleHookName: 'Hook-Name-1',
                        EC2InstanceId: 'i-0000000000',
                        LifecycleActionToken: 'Action-Token-1',
                        NotificationMetadata: JSON.stringify({
                            eniId: 'eni-000000000',
                        }),
                    }),
                },
            },
            {
                Sns: {
                    Message: JSON.stringify({
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        AutoScalingGroupName: 'ASG-Name-2',
                        LifecycleHookName: 'Hook-Name-2',
                        EC2InstanceId: 'i-1111111111',
                        LifecycleActionToken: 'Action-Token-2',
                        NotificationMetadata: JSON.stringify({
                            eniId: 'eni-1111111111',
                        }),
                    }),
                },
            },
        ],
    };
    attachSpy = jest.fn((request) => successRequestMock(request));
    completeSpy = jest.fn((request) => successRequestMock(request));
    aws_sdk_mock_1.mock('EC2', 'attachNetworkInterface', attachSpy);
    aws_sdk_mock_1.mock('AutoScaling', 'completeLifecycleAction', completeSpy);
    // WHEN
    await index_1.handler(event);
    // THEN
    expect(attachSpy).toHaveBeenCalledTimes(2);
    expect(completeSpy).toHaveBeenCalledTimes(2);
    expect(attachSpy.mock.calls[0][0]).toStrictEqual({
        DeviceIndex: 1,
        InstanceId: 'i-0000000000',
        NetworkInterfaceId: 'eni-000000000',
    });
    expect(attachSpy.mock.calls[1][0]).toStrictEqual({
        DeviceIndex: 1,
        InstanceId: 'i-1111111111',
        NetworkInterfaceId: 'eni-1111111111',
    });
    expect(completeSpy.mock.calls[0][0]).toStrictEqual({
        AutoScalingGroupName: 'ASG-Name-1',
        LifecycleHookName: 'Hook-Name-1',
        InstanceId: 'i-0000000000',
        LifecycleActionToken: 'Action-Token-1',
        LifecycleActionResult: 'CONTINUE',
    });
    expect(completeSpy.mock.calls[1][0]).toStrictEqual({
        AutoScalingGroupName: 'ASG-Name-2',
        LifecycleHookName: 'Hook-Name-2',
        InstanceId: 'i-1111111111',
        LifecycleActionToken: 'Action-Token-2',
        LifecycleActionResult: 'CONTINUE',
    });
});
test('abandons launch when attach fails', async () => {
    // GIVEN
    const event = {
        Records: [
            {
                Sns: {
                    Message: JSON.stringify({
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        AutoScalingGroupName: 'ASG-Name-1',
                        LifecycleHookName: 'Hook-Name-1',
                        EC2InstanceId: 'i-0000000000',
                        LifecycleActionToken: 'Action-Token-1',
                        NotificationMetadata: JSON.stringify({
                            eniId: 'eni-000000000',
                        }),
                    }),
                },
            },
        ],
    };
    attachSpy = jest.fn(() => errorRequestMock());
    completeSpy = jest.fn((request) => successRequestMock(request));
    aws_sdk_mock_1.mock('EC2', 'attachNetworkInterface', attachSpy);
    aws_sdk_mock_1.mock('AutoScaling', 'completeLifecycleAction', completeSpy);
    // WHEN
    await index_1.handler(event);
    // THEN
    expect(completeSpy.mock.calls[0][0]).toStrictEqual({
        AutoScalingGroupName: 'ASG-Name-1',
        LifecycleHookName: 'Hook-Name-1',
        InstanceId: 'i-0000000000',
        LifecycleActionToken: 'Action-Token-1',
        LifecycleActionResult: 'ABANDON',
    });
});
test('continues when complete lifecycle errors', async () => {
    // GIVEN
    const event = {
        Records: [
            {
                Sns: {
                    Message: JSON.stringify({
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        AutoScalingGroupName: 'ASG-Name-1',
                        LifecycleHookName: 'Hook-Name-1',
                        EC2InstanceId: 'i-0000000000',
                        LifecycleActionToken: 'Action-Token-1',
                        NotificationMetadata: JSON.stringify({
                            eniId: 'eni-000000000',
                        }),
                    }),
                },
            },
            {
                Sns: {
                    Message: JSON.stringify({
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        AutoScalingGroupName: 'ASG-Name-1',
                        LifecycleHookName: 'Hook-Name-1',
                        EC2InstanceId: 'i-0000000000',
                        LifecycleActionToken: 'Action-Token-1',
                        NotificationMetadata: JSON.stringify({
                            eniId: 'eni-000000000',
                        }),
                    }),
                },
            },
        ],
    };
    attachSpy = jest.fn((request) => successRequestMock(request));
    completeSpy = jest.fn(() => errorRequestMock());
    aws_sdk_mock_1.mock('EC2', 'attachNetworkInterface', attachSpy);
    aws_sdk_mock_1.mock('AutoScaling', 'completeLifecycleAction', completeSpy);
    // THEN
    // eslint-disable-next-line: no-floating-promises
    await expect(index_1.handler(event)).resolves.not.toThrow();
    expect(console.error).toHaveBeenCalledTimes(4); // 4 = each of the two records printing two error messages
});
//# sourceMappingURL=data:application/json;base64,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