"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Skill = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: MIT-0
 */
const path = require("path");
const ask = require("@aws-cdk/alexa-ask");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
const customResources = require("@aws-cdk/custom-resources");
const ALEXA_SERVICE_PRINCIPAL = 'alexa-appkit.amazon.com';
const BACKEND_LAMBDA_PERMISSION_ACTION = 'lambda:InvokeFunction';
;
class SkillBase extends cdk.Resource {
}
;
;
/**
 * Defines an Alexa Skill.
 *
 * @stability stable
 * @resource Alexa::ASK::Skill
 */
class Skill extends SkillBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        // Validate that SSM SecureString was not supplied--Alexa::ASK::Skill does not support SSM SecureString references.
        const resolvedClientSecret = cdk.Tokenization.resolve(props.lwaClientSecret, {
            scope,
            resolver: new cdk.DefaultTokenResolver(new cdk.StringConcat()),
        });
        const resolvedRefreshToken = cdk.Tokenization.resolve(props.lwaRefreshToken, {
            scope,
            resolver: new cdk.DefaultTokenResolver(new cdk.StringConcat()),
        });
        if (resolvedClientSecret.includes('ssm-secure')) {
            throw new Error('Invalid prop: lwaClientSecret; SSM SecureString is not supported. Use Secrets Manager secret instead.');
        }
        if (resolvedRefreshToken.includes('ssm-secure')) {
            throw new Error('Invalid prop: lwaRefreshToken; SSM SecureString is not supported. Use Secrets Manager secret instead.');
        }
        super(scope, id);
        // Role giving CfnSkill resource read-only access to skill package asset in S3.
        const askResourceRole = new iam.Role(this, 'AskResourceRole', {
            assumedBy: new iam.ServicePrincipal(ALEXA_SERVICE_PRINCIPAL),
        });
        // Skill package S3 asset.
        const skillPackageAsset = new assets.Asset(this, 'SkillPackageAsset', {
            path: props.skillPackagePath,
            readers: [askResourceRole],
        });
        // Alexa Skill with override that injects the endpoint Lambda Function in the skill manifest.
        const resource = new ask.CfnSkill(this, 'Resource', {
            vendorId: props.alexaVendorId,
            skillPackage: {
                s3Bucket: skillPackageAsset.s3BucketName,
                s3Key: skillPackageAsset.s3ObjectKey,
                s3BucketRole: askResourceRole.roleArn,
                ...props.endpointLambdaFunction && {
                    overrides: {
                        manifest: {
                            apis: {
                                custom: {
                                    endpoint: {
                                        uri: (_b = props.endpointLambdaFunction) === null || _b === void 0 ? void 0 : _b.functionArn,
                                    },
                                },
                            },
                        },
                    },
                },
            },
            authenticationConfiguration: {
                clientId: props.lwaClientId,
                clientSecret: props.lwaClientSecret.toString(),
                refreshToken: props.lwaRefreshToken.toString(),
            },
        });
        // Set resource skillId to Alexa Skill resource Skill ID.
        this.skillId = resource.ref;
        // This section is only necessary if a Lambda Function was supplied in the props.
        if (props.endpointLambdaFunction) {
            // Create placeholder Lambda Permission to allow Alexa Skill to pass endpoint validation.
            // Permission will be replaced with another containing event source validation after Alexa Skill is created.
            const initialLambdaPermission = new lambda.CfnPermission(this, 'InitialLambdaPermission', {
                functionName: props.endpointLambdaFunction.functionArn,
                principal: ALEXA_SERVICE_PRINCIPAL,
                action: BACKEND_LAMBDA_PERMISSION_ACTION,
            });
            // Skill must be created after the initial Lambda Permission resource is in place to prevent endpoint validation errors.
            resource.addDependsOn(initialLambdaPermission);
            // Lambda Function that retrieves the StatementId of the initial Lambda Permission for use by other custom resources.
            const getPermissionStatementIdFunction = new lambda.Function(this, 'GetLambdaPermissionStatementIdFunction', {
                runtime: lambda.Runtime.PYTHON_3_8,
                handler: 'index.lambda_handler',
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/get-lambda-permission-statement-id-handler')),
                initialPolicy: [
                    new iam.PolicyStatement({
                        actions: ['lambda:GetPolicy'],
                        resources: [props.endpointLambdaFunction.functionArn],
                    }),
                ],
            });
            // Custom resource for managing lifecycle of GetLambdaPermissionStatementIdFunction Lambda Function.
            const getPermissionStatementIdCustomResource = new cdk.CustomResource(this, 'GetLambdaPermissionStatementIdCustomResource', {
                serviceToken: new customResources.Provider(this, 'Provider', { onEventHandler: getPermissionStatementIdFunction }).serviceToken,
                properties: {
                    lambda_function_arn: props.endpointLambdaFunction.functionArn,
                    service_principal_to_match: ALEXA_SERVICE_PRINCIPAL,
                    action_to_match: BACKEND_LAMBDA_PERMISSION_ACTION,
                },
            });
            // Custom resource code must run after the initial Lambda Permission resource is in place.
            getPermissionStatementIdCustomResource.node.addDependency(initialLambdaPermission);
            // Get custom resource result for use by other custom resources.
            const permissionStatementId = getPermissionStatementIdCustomResource.getAttString('statement_id');
            // Policy for AwsCustomResource resources.
            const awsCustomResourcePolicy = customResources.AwsCustomResourcePolicy.fromStatements([
                new iam.PolicyStatement({
                    actions: [
                        'lambda:RemovePermission',
                        'lambda:AddPermission',
                    ],
                    resources: [props.endpointLambdaFunction.functionArn],
                }),
            ]);
            // SDK call to be used for RemovePermissionCustomResource.
            const removePermissionStatementSdkCall = {
                service: 'Lambda',
                action: 'removePermission',
                parameters: {
                    FunctionName: props.endpointLambdaFunction.functionArn,
                    StatementId: permissionStatementId,
                },
                ignoreErrorCodesMatching: 'ResourceNotFoundException',
                physicalResourceId: customResources.PhysicalResourceId.of(`RemovePermission-${this.skillId}`),
            };
            const removePermissionCustomResource = new customResources.AwsCustomResource(this, 'RemovePermissionCustomResource', {
                policy: awsCustomResourcePolicy,
                onCreate: removePermissionStatementSdkCall,
                onUpdate: removePermissionStatementSdkCall,
                onDelete: removePermissionStatementSdkCall,
            });
            // RemovePermissionCustomResource code must run after the Alexa Skill has been created to ensure the intial Lambda Permission is in place upon Alexa Skill creation.
            removePermissionCustomResource.node.addDependency(resource);
            // SDK call to be used for AddPermissionCustomResource.
            const addPermissionStatementSdkCall = {
                service: 'Lambda',
                action: 'addPermission',
                parameters: {
                    FunctionName: props.endpointLambdaFunction.functionArn,
                    StatementId: permissionStatementId,
                    Principal: ALEXA_SERVICE_PRINCIPAL,
                    Action: BACKEND_LAMBDA_PERMISSION_ACTION,
                    EventSourceToken: this.skillId,
                },
                physicalResourceId: customResources.PhysicalResourceId.of(`AddPermission-${this.skillId}`),
            };
            const addPermissionCustomResource = new customResources.AwsCustomResource(this, 'AddPermissionCustomResource', {
                policy: awsCustomResourcePolicy,
                onCreate: addPermissionStatementSdkCall,
                onUpdate: addPermissionStatementSdkCall,
            });
            // AddPermissionCustomResource code must run after RemovePermissionCustomResource code has run to prevent attempts to create Permission with redundant StatementIds.
            addPermissionCustomResource.node.addDependency(removePermissionCustomResource);
        }
    }
    /**
     * Reference an existing Skill, defined outside of the CDK code, by Skill ID.
     *
     * @stability stable
     */
    static fromSkillId(scope, id, skillId) {
        class Import extends SkillBase {
            constructor() {
                super(...arguments);
                this.skillId = skillId;
            }
        }
        return new Import(scope, id);
    }
}
exports.Skill = Skill;
_a = JSII_RTTI_SYMBOL_1;
Skill[_a] = { fqn: "cdk-alexa-skill.Skill", version: "1.0.2" };
;
//# sourceMappingURL=data:application/json;base64,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