#! Python
# -*- coding: utf-8 -*-
"""
	@file CleanUpReports.py
	@brief File is module container for class definition CCleanUpReports.
	
	@copyright: Copyright (C) 2022 Freelance Rocket Science, All rights reserved.

	@author  Colin Helms, colinhelms@outlook.com, [CCH]

    @details Provides minimal methods to decimate spaces and commas in text files, 
        writes a clean Excel file.
    
    @remark Change History
        Fri Mar 28 2022 [CCH] File created, GitHub repository GMAT-Automation.
        Tue Apr 26 2022 [CCH] Version 0.2a1, Buildable package, locally deployable.
    
    @bug https://github.com/a093130/GMAT-Automation/issues
"""
import re
import logging
import traceback
from pathlib import Path
from gmatautomation import reduce_report as rr

class CCleanUpReports:
    """ Base Class.  Designed for specialization through overload of the extend function.
    """
    def __init__(self, **args):
        super().__init__(**args)
        self.filelist = []
        return

    def do_batch(self, batchfile, **args):
        """ Open the master 'batchfile'.
            This file should contain a line by line list of paths to report files generated by GMAT.
            
            The base extend() function is called to produce an Excel file from the GMAT text output file.
            Each Excel filename written is appended to the List instance variable for retrieval.
        """

        assert not hasattr(super(), 'do_batch')
        """ MRO delegation chain stops here. """

        regecr = re.compile('\s')

        try:    
            with open(batchfile) as f:
                for filepath in f:                
                    """ Iterate through report files named in batch. """
                    logging.info('Reducing file: {0}'.format(filepath))

                    filepath = regecr.sub('',filepath)
                    """Get rid of newline following path string."""

                    rpt = Path(filepath)

                    if rpt.exists() & rpt.is_file():
                        self.extend(rpt)
                        """ Do not assume any return type. """                       
            return

        except OSError as e:
            logging.error("OS error: %s for filename %s", e.strerror, e.filename)
            print('OS error: ', e.strerror,' in CleanUpReports do_batch() for filename ', e.filename)

        except Exception as e:
            lines = traceback.format_exc().splitlines()
            logging.error("Exception: %s\n%s\n%s\n%s", e.__doc__, lines[0], lines[1], lines[-1])
            print('Exception in CleanUpReports do_batch(): ', e.__doc__, '\n', lines[0],'\n',lines[1], '\n', lines[-1])


    def extend(self, rpt):
        """ This base version of extend() provides basic functionality which simply
            decimates spaces and commas, then writes a "cleaned up" Excel file.  It is
            suitable for cleaning up the GMAT ReportFile.

            Child classes should use extend() for specialization, such as additional
            formatting.
        """
        try:
            nospc = rr.decimate_spaces(rpt)
            reduced = rr.decimate_commas(nospc)
            xlfile = rr.csv_to_xlsx(reduced)

            nospc = Path(nospc)
            if nospc.exists():
                nospc.unlink()

            reduced = Path(reduced)
            if reduced.exists():
                reduced.unlink()

            xlfile = Path(xlfile)
            self.filelist.append(xlfile)

            logging.info('CleanUpReports extend() completed for filename: %s.', xlfile)
            print ('CleanUpReports extend() completed  for filename:', xlfile.name)

            return

        except OSError as e:
            logging.error("OS error: %s for filename %s", e.strerror, e.filename)
            print('OS error: ', e.strerror,' in CleanUpReports extend() for filename ', e.filename)

        except Exception as e:
            lines = traceback.format_exc().splitlines()
            logging.error("Exception: %s\n%s\n%s\n%s", e.__doc__, lines[0], lines[1], lines[-1])
            print('Exception in CleanUpReports extend(): ', e.__doc__, '\n', lines[0], '\n', lines[1],'\n', lines[-1])


if __name__ == "__main__":
    __spec__ = None
    """ Necessary tweak to get Spyder IPython to execute this code. 
    See:
    https://stackoverflow.com/questions/45720153/
    python-multiprocessing-error-attributeerror-module-main-has-no-attribute
    """
    import platform
    import getpass
    import logging
    import traceback
    from PyQt5.QtWidgets import(QApplication, QFileDialog)
    from gmatautomation import CGmatParticulars

    logging.basicConfig(
            filename='./CleanUpReports.log',
            level=logging.INFO,
            format='%(asctime)s %(filename)s \n %(message)s', 
            datefmt='%d%B%Y_%H:%M:%S')

    logging.info("!!!!!!!!!! CleanUpReports Execution Started.!!!!!!!!!! ")
    
    host_attr = platform.uname()
    logging.info('User Id: %s\nNetwork Node: %s\nSystem: %s, %s, \nProcessor: %s', \
                 getpass.getuser(), \
                 host_attr.node, \
                 host_attr.system, \
                 host_attr.version, \
                 host_attr.processor)
    
    gmat_paths = CGmatParticulars()
    o_path = gmat_paths.get_output_path()
    """ o_path is an instance of Path that locates the GMAT output directory. """

    reports = CCleanUpReports()

    qApp = QApplication([])

    """ Batch Processing of .txt Report Files. """
    msg = 'Open ReportFile.batch file naming the GMAT Report Files to be processed.'
    fbatch = QFileDialog().getOpenFileName(None, msg, o_path, filter='Batch files(*.batch)')
    fbatch = Path(fbatch[0])

    try:        
        reports.do_batch(fbatch)

        logging.info('CleanUpReports complete with batch file {0}'.format(fbatch.name))
        print('CleanUpReports complete with batch file: ', fbatch.name, '.\nFilelist follows:')

        for fp in reports.filelist:
            print(fp)

    except OSError as e:
        logging.error("OS error: %s for filename %s", e.strerror, e.filename)
        print('OS error: ', e.strerror,' in CleanUpReports Test Case for filename ', e.filename)

    except Exception as e:
        lines = traceback.format_exc().splitlines()
        logging.error("Exception: %s\n%s\n%s\n%s", e.__doc__, lines[0], lines[1], lines[-1])
        print('Exception in CleanUpReports Test Case: ', e.__doc__, '\n', lines[0], '\n', lines[1],'\n', lines[-1])

    finally:
        qApp.quit()
