"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    /**
     * Specified statistic is greater than or equal to the threshold
     */
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    /**
     * Specified statistic is strictly greater than the threshold
     */
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    /**
     * Specified statistic is strictly less than the threshold
     */
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    /**
     * Specified statistic is less than or equal to the threshold.
     */
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    /**
     * Specified statistic is lower than or greater than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '>=',
};
/**
 * Specify how missing data points are treated during alarm evaluation
 */
var TreatMissingData;
(function (TreatMissingData) {
    /**
     * Missing data points are treated as breaching the threshold
     */
    TreatMissingData["BREACHING"] = "breaching";
    /**
     * Missing data points are treated as being within the threshold
     */
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    /**
     * The current alarm state is maintained
     */
    TreatMissingData["IGNORE"] = "ignore";
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric
 */
class Alarm extends alarm_base_1.AlarmBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // tslint:disable-next-line:max-line-length
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo)
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: mathExprHasSubmetrics(expr) ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
//# sourceMappingURL=data:application/json;base64,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