"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cleanup_1 = require("./cleanup");
const clobber_1 = require("./clobber");
const deps_1 = require("./deps");
const file_1 = require("./file");
const github_1 = require("./github");
const gitpod_1 = require("./gitpod");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const readme_1 = require("./readme");
const tasks_1 = require("./tasks/tasks");
const util_1 = require("./util");
const vscode_1 = require("./vscode");
/**
 * (experimental) Base project.
 *
 * @experimental
 */
class Project {
    /**
     * @experimental
     */
    constructor(options) {
        var _b, _c, _d, _e;
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.newProject = render_options_1.resolveNewProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this.projectType = (_b = options.projectType) !== null && _b !== void 0 ? _b : ProjectType.UNKNOWN;
        if (this.parent && options.outdir && path.isAbsolute(options.outdir)) {
            throw new Error('"outdir" must be a relative path');
        }
        let outdir;
        if (options.parent) {
            if (!options.outdir) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            outdir = path.join(options.parent.outdir, options.outdir);
        }
        else {
            outdir = (_c = options.outdir) !== null && _c !== void 0 ? _c : '.';
        }
        this.outdir = path.resolve(outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        (_d = this.parent) === null || _d === void 0 ? void 0 : _d._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitignore = new ignore_file_1.IgnoreFile(this, '.gitignore');
        this.gitignore.exclude('node_modules/'); // created by running `npx projen`
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        this.deps = new deps_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        // we only allow these global services to be used in root projects
        this.github = !this.parent ? new github_1.GitHub(this, options) : undefined;
        this.vscode = !this.parent ? new vscode_1.VsCode(this) : undefined;
        this.gitpod = options.gitpod ? new gitpod_1.Gitpod(this) : undefined;
        this.devContainer = options.devContainer ? new vscode_1.DevContainer(this) : undefined;
        if ((_e = options.clobber) !== null && _e !== void 0 ? _e : true) {
            new clobber_1.Clobber(this);
        }
        new readme_1.SampleReadme(this, options.readme);
    }
    /**
     * (experimental) Returns all the components within this project.
     *
     * @experimental
     */
    get components() {
        return [...this._components];
    }
    /**
     * (experimental) All files in this project.
     *
     * @experimental
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components.filter(isFile).sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * (experimental) Adds a new task to this project.
     *
     * This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add.
     * @param props Task properties.
     * @experimental
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * (experimental) Finds a file at the specified relative path within this project and all its subprojects.
     *
     * @param filePath The file path.
     * @returns a `FileBase` or undefined if there is no file in that path
     * @experimental
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath) ? filePath : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * (deprecated) Finds a json file by name.
     *
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (experimental) Finds an object file (like JsonFile, YamlFile, etc.) by name.
     *
     * @param filePath The file path.
     * @experimental
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (deprecated) Prints a "tip" message during synthesis.
     *
     * @param message The message.
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * (experimental) Exclude the matching files from pre-synth cleanup.
     *
     * Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match.
     * @experimental
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * (experimental) Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     *
     * @experimental
     */
    synth() {
        const outdir = this.outdir;
        this.logger.info('Synthesizing project...');
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + '/**');
        }
        // delete all generated files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        this.logger.info('Synthesis complete');
    }
    /**
     * (experimental) Called before all components are synthesized.
     *
     * @experimental
     */
    preSynthesize() { }
    /**
     * (experimental) Called after all components are synthesized.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.17.75" };
/**
 * (experimental) The name of the default task (the task executed when `projen` is run without arguments).
 *
 * Normally
 * this task should synthesize the project files.
 *
 * @experimental
 */
Project.DEFAULT_TASK = 'default';
/**
 * (experimental) Which type of project this is.
 *
 * @experimental
 */
var ProjectType;
(function (ProjectType) {
    ProjectType["UNKNOWN"] = "unknown";
    ProjectType["LIB"] = "lib";
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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